/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: xloop.c,v 2.48 2002/07/15 04:30:28 cmalek Exp $
** Program: dowl
**  Module: xloop.c
**  Author: mazer
** Descrip: x-startup and event-loop support
**
** Revision History (most recent last)
**
** Thu Jan 28 16:34:56 1993 mazer
**  renamed xloop<XXXXX>() etc to dloop<XXXXX>() to avoid
**  name conflicts with other libraries.. (The filename is
**  still xloop.c for historical reasons :-)
**
** Wed Feb 24 18:03:38 1993 mazer
**  got rid of busy function .. never really used..
**
** Fri Feb 26 13:52:21 1993 mazer
**  cleaned up function names and made all the actions private and
**  into true x-actions procs..
**
** Mon Dec 13 11:10:04 1993 mazer
**  added dloop_sync();
**
** Sun Jan 23 00:46:37 1994 mazer
**  commented out the Traverse/WidgetName/WidgetClass stuff.
**  define XDEBUG_TOOLS to get it back..
**
** Sun Jan 23 00:49:33 1994 mazer
**   deleted the -con option -- never used it!
**  
** Sun Jan 23 00:59:58 1994 unknown-user
**  globalized the TopLevelContext..
**
** Thu Nov  3 12:41:17 1994 mazer
**  split out dloop stuff in dloop.c and got rid of alt stuff..
**
** Wed Feb 22 13:47:33 1995 mazer
**  added code to append $DOWLDIR to XAPPRESDIR automatically
**  whenever startx() get's called. This should resource file
**  localization..
**
*****************************************************************/

#include "xdphyslib.h"

Widget TopLevel = NULL;
XtAppContext TopLevelContext = NULL;

static int (*orig_XErrorHandler) () = NULL;

static int xdphys_XErrorHandler(Display * dpy, XErrorEvent * e_event);
static void DetachAction(Widget, XEvent *, String *, Cardinal *);
static void Traverse_aux(FILE *, Widget, char *, char *);
static void TraverseAction(Widget, XEvent *, String *, Cardinal *);
static void WidgetName_aux(Widget);
static void WidgetNameAction(Widget, XEvent *, String *, Cardinal *);
static void WidgetClass_aux(Widget);
static void install_default_actions(void);

static int xdphys_XErrorHandler(Display * dpy, XErrorEvent * e_event)
{
	/*
	 * I think orig_XErrorHandler() is guaranteed to always exit(), so
	 * this functions will NEVER return -- so the return(0) is just
	 * to shut up gcc/lint
	 */
	(*orig_XErrorHandler) (dpy, e_event);
	return (0);
}

static int detach_ok = 1;

void forbid_detach(void)
{
	detach_ok = 0;
}

void allow_detach(void)
{
	detach_ok = 1;
}

static void DetachAction(Widget w, XEvent * event, String * params,
			 Cardinal * num_params)
{
	static int detached = 0;

	if (detach_ok) {
		if (!detached) {
			if (fork())
				exit(0);
			else
				detached = 1;
		} else {
			beep(0);
		}
	}
}

static void Traverse_aux(FILE * fp, Widget w, char *nprefix, char *cprefix)
{
	WidgetList wl;
	Cardinal wnum;
	CoreWidgetClass z;
	char *n, *c, *wname, *wclass;
	int i;

	n = XtName(w);
	wname = (char *) malloc(strlen(n) + strlen(nprefix) + 10);
	sprintf(wname, "%s.%s", nprefix, n);

	z = (CoreWidgetClass) XtClass(w);
	c = z->core_class.class_name;
	wclass = (char *) malloc(strlen(c) + strlen(cprefix) + 10);
	sprintf(wclass, "%s.%s", cprefix, c);

	fprintf(fp, "%s\n", wname);
	fprintf(fp, "%s\n", wclass);

	if (XtIsComposite(w)) {
		XtVaGetValues(w, XtNnumChildren, &wnum, XtNchildren, &wl,
			      NULL);
		for (i = 0; i < wnum; i++)
			if (wl[i] != NULL)
				Traverse_aux(fp, wl[i], wname, wclass);
	}
	if (XtIsWidget(w) && w->core.num_popups > 0) {
		for (i = 0; i < w->core.num_popups; i++)
			Traverse_aux(fp, w->core.popup_list[i], wname,
				     wclass);
	}
	free(wname);
	free(wclass);
}

static void TraverseAction(Widget w, XEvent * event, String * params,
			   Cardinal * num_params)
{
	FILE *fp;

	if (*num_params && strcmp(params[0], "top") == 0) {
		w = TopLevel;
	}
	if ((fp = fopen2("/tmp/traverse", "w")) == NULL) {
		XtAppWarning(TopLevelContext, "Can't open traverse file");
	} else {
		Traverse_aux(fp, w, "", "");
		fclose(fp);
	}
}

static void WidgetName_aux(Widget w)
{
	if (w != TopLevel) {
		WidgetName_aux(XtParent(w));
		fprintf(stderr, ".%s", XtName(w));
	} else {
		fprintf(stderr, "%s", XtName(w));
	}
}

static void WidgetNameAction(Widget w, XEvent * event, String * params,
			     Cardinal * num_params)
{
	WidgetName_aux(w);
	fprintf(stderr, "\n");
}


static void WidgetClass_aux(Widget w)
{
	CoreWidgetClass z;

	z = (CoreWidgetClass) XtClass(w);
	if (w != TopLevel) {
		WidgetClass_aux(XtParent(w));
		fprintf(stderr, ".%s", z->core_class.class_name);
	} else {
		fprintf(stderr, "%s", z->core_class.class_name);
	}
}

static void WidgetClassAction(Widget w, XEvent * event, String * params,
			      Cardinal * num_params)
{
	WidgetClass_aux(w);
	fprintf(stderr, "\n");
}

static void install_default_actions(void)
{
	static XtActionsRec actionTable[] = {
		{"detach", DetachAction},
		{"traverse", TraverseAction},
		{"widget-name", WidgetNameAction},
		{"widget-class", WidgetClassAction},
	};
	XtAppAddActions(TopLevelContext, actionTable,
			XtNumber(actionTable));
}

void startx(int *acp, char **av, char *xName, char *xClass)
{
	char *argzero;
	int arg, i;

	if (TopLevel != NULL) {
		fprintf(stderr,
			"startx: fatal error -- X initialized twice\n");
		exit(1);
	}

	if (xName == NULL)
		xName = av[0];
	if (xClass == NULL)
		xClass = "XDphys";

	/* NOTE: this little kludge with "argzero" sets av[0] to
	 * be the xName of the application for the duration of
	 * XtVaAppInitialize(). This allows for the proper resources
	 * to be retreived from the server-database. It's then set
	 * back to the real av[0]
	 */
	if (xName != av[0]) {
		argzero = av[0];
		av[0] = xName;
	} else {
		argzero = NULL;
	}

	TopLevel = (Widget)
	    XtVaAppInitialize(&TopLevelContext,	/* app context */
			      xClass,	/* app class */
			      /* app option table (fallbacks?) */
			      (XrmOptionDescRec *) NULL, (Cardinal) 0,	/* number of app options in table */
			      acp,	/* ptr to number of args on cmd line */
			      av,	/* actual command line arguments */
			      (String *) NULL,	/* opt. "fallback resources" table */
			      (String *) NULL);	/* opt. va_list of XtN<options> */

	/* this is so we can trap X-errors my putting
	 * a breakpoint at xdphys_XErrorHandler() and running with
	 * -sync when problems occur..
	 */
	orig_XErrorHandler = XSetErrorHandler(xdphys_XErrorHandler);

	if (argzero != NULL)
		av[0] = argzero;

#define isarg(pat) (strncmp(av[arg], pat, strlen(av[arg])) == 0)

	for (arg = 1; arg < *acp; arg++) {
		if (isarg("-debug")) {
			debugflag = True;
			fprintf(stderr, "Debug Mode\n");
			*av[arg] = 0;
		} else if (isarg("-v")) {
#ifdef BUILDAT
			fprintf(stderr, "%s: %s\n", av[0], BUILDAT);
#else
			fprintf(stderr, "%s: no version info\n", av[0]);
#endif
			exit(0);
		} else if (isarg("-pid")) {
			fprintf(stdout, "pid %d\n", getpid());
			*av[arg] = 0;
		} else if (isarg("-cd") && av[arg + 1] != NULL) {
			if (chdir(av[arg + 1]) < 0) {
				perror(av[arg + 1]);
				exit(1);
			} else {
				char path[MAXPATHLEN];

				getcwd(path, MAXPATHLEN);
				fprintf(stderr, "Current Dir: %s\n", path);

				*av[arg] = 0;
				*av[++arg] = 0;
			}
		}
	}
#undef isarg

	arg = 1;
	while (av[arg] != NULL) {
		if (*av[arg] == 0) {
			for (i = arg + 1; av[i] != NULL; i++)
				av[i - 1] = av[i];
			av[i - 1] = NULL;
			*acp -= 1;
		} else {
			arg++;
		}
	}
	install_default_actions();
}

#ifdef DLOOP_IN_XLOOP

void dloop(void)
{
	XEvent event;
	static int first_time = 0;

	if (TopLevel == NULL) {
		fprintf(stderr, "dloop called before startx()\n");
		exit(1);
	}

	if (first_time != 0) {
		fprintf(stderr, "dloop: event loop entered twice!\n");
		exit(1);
	} else {
		first_time = 1;
	}

	for (;;) {
		XtAppNextEvent(TopLevelContext, &event);
		XtDispatchEvent(&event);
	}
}

static void (*alt_empty) () = NULL;
static void (*alt_once) () = NULL;

int dloop_empty(void)
{
	register int n;

	if (TopLevel == NULL)
		return (0);

	if (alt_empty != NULL)
		(*alt_empty) ();

	for (n = 0; XtAppPending(TopLevelContext); n++) {
		XtAppProcessEvent(TopLevelContext, XtIMAll);
	}
	return (n);
}

int dloop_once(void)
{
	if (TopLevel == NULL)
		return (0);

	if (alt_once != NULL)
		(*alt_once) ();

	if (XtAppPending(TopLevelContext)) {
		XtAppProcessEvent(TopLevelContext, XtIMAll);
		return (1);
	}
	return (0);
}

int dloop_sync(void)
{
	if (TopLevel == NULL)
		return (0);

	XFlush(XtDisplay(TopLevel));
	XSync(XtDisplay(TopLevel), False);
	return (dloop_empty());
}

void dloop_setalts(void (*empty) (), (*once) ())
{
	alt_empty = empty;
	alt_once = once;
}

#endif				/* DLOOP_IN_XLOOP */

/*
** install accelerators from src from w on down
**   - src is the widget containing the buttons, toggles, etc
**     that you want to have keyboard accelerators for..
**   - dest is the widget where you want the accel's to work
**     this functions descends the widget from from w on down
**     taking each widget in the tree as dest..
*/

void accel_from(Widget w, Widget src)
{
	int i;
	WidgetList wl;
	Cardinal wnum;

	XtInstallAllAccelerators(w, src);
	if (XtIsComposite(w)) {
		XtVaGetValues(w, XtNnumChildren, &wnum, XtNchildren, &wl,
			      NULL);
		for (i = 0; i < wnum; i++)
			if (wl[i] != NULL)
				accel_from(wl[i], src);
	}
	if (XtIsWidget(w) && w->core.num_popups > 0) {
		for (i = 0; i < w->core.num_popups; i++)
			accel_from(w->core.popup_list[i], src);
	}
}


int ApplyToAllWidgets(Widget w, void (*fn) ())
{
	int i, cnt;
	WidgetList wl;
	Cardinal wnum;

	(*fn) (w);
	cnt = 1;
	if (XtIsComposite(w)) {
		XtVaGetValues(w, XtNnumChildren, &wnum, XtNchildren, &wl,
			      NULL);
		for (i = 0; i < wnum; i++)
			if (wl[i] != NULL) {
				cnt += ApplyToAllWidgets(wl[i], fn);
			}
	}
	if (XtIsWidget(w) && w->core.num_popups > 0) {
		for (i = 0; i < w->core.num_popups; i++) {
			cnt +=
			    ApplyToAllWidgets(w->core.popup_list[i], fn);
		}
	}
	return (cnt);
}
