/*
   **   This file is part of XDowl
   **   Copyright (c) 1994 Jamie Mazer
   **   California Institute of Technology
   **   <mazer@asterix.cns.caltech.edu>
 */

/******************************************************************
**  RCSID: $Id: xcalibur.c,v 1.25 2002/07/15 04:30:26 cmalek Exp $
** Program: calib
**  Module: 
**  Author: bjarthur
** Descrip: 
**
** Revision History (most recent last)
**
** 06NOV96 bjarthur
**  copied from xdowl.c
**
** 97.4 bjarthur
**  changed DOWLDIRNEW to XDPHYSDIR
**
** 97.12 bjarthur
**  added support to lock all worksheets during initial portion
**  of each run.  this gives time for the mods to "download" the svars into
**  local variables while insuring that the user doesn't change them while
**  it's doing this.  after download has occurred, each module should unlock
**  the worksheets using the supplied function.  new functions:
**  lock_worksheets(), unlock_worksheets()
**
**  added Comments flag so user could turn off prompting for comments
**
**  changed appRun to setRack to maxAtten instead of 0 after each run
**
** 99.2 cmalek
**  changed .fr search procedure to look first for 
**  $(XDPHYSDIR)/calib/$(HOST).kleft.fr, then for
**  $(XDPHYSDIR)/calib/local.kleft.fr
**
*******************************************************************/

#include "xdphyslib.h"
#include "xcalibur.h"

/*#include "littleowl.bm" */

static void lock_worksheets(void);
static void usage(char *, char *);
static void AboutCB(Widget, XtPointer, XtPointer);
static void savestate(int);
static void SaveStateCB(Widget, XtPointer, XtPointer);
static void Quit(Widget, XtPointer, XtPointer);
static int appRun(int, RunData *, Widget);
static void ModuleStartFn(Widget, RunData *, int);
static void ModuleStartCB(Widget, XtPointer, XtPointer);
static void HelpCB(Widget, XtPointer, XtPointer);
static void ModuleTempCB(Widget, XtPointer, XtPointer);
static void ModuleCloseCB(Widget, XtPointer, XtPointer);
static void ModuleNamedTempCB(Widget, XtPointer, XtPointer);
static void make_close_help(Widget, char *, DockSlot *);
static void QuitAction(Widget, XEvent *, String *, Cardinal *);
static void install_actions(void);
static int calibmain(int, char **, char *, char *);
int main(int, char **);

extern void CombineFrFiles(void);

char *statefile = "~/.xdphysrc/config.xcalibur";

static void usage(char *pname, char *badarg)
{
	fprintf(stderr, "xcalibur -- Version %s \n", XDPHYS_VERSION);

	if (badarg)
		fprintf(stderr, "\n%s: unknown argument \"%s\"\n\n", pname,
			badarg);

	if (badarg)
		exit(1);
}

extern char *statefile;
extern void install_modules(Dock *);
extern void install_xcalibur_plotMethods(void);

static Widget headline = NULL;

static Dock *tooldock = NULL;

static Widget waveshell, lwv, rwv;
static Widget waveshell_out, lwv_out, rwv_out;

#define MAX_PT_PTS 1000
static int lpt_n = 0, rpt_n = 0;
static float lpt_freq[MAX_PT_PTS], rpt_freq[MAX_PT_PTS];
static float lpt_mag[MAX_PT_PTS], rpt_mag[MAX_PT_PTS];
static float lpt_phi[MAX_PT_PTS], rpt_phi[MAX_PT_PTS];

static Field ws_prefs[] = {
	{WS_VSEP, "Stimulus Preferences"},
	{"Reps", "%d", 4, "3"},
	{"Dur (ms)", "%d", 4, "100"},
	{"Delay (ms)", "%d", 4, "5"},
	{"Epoch (ms)", "%d", 4, "110"},
	{"ISI (ms)", "%d", 4, "500"},
	{"Rise (ms)", "%d", 2, "5"},
	{"Fall (ms)", "%d", 2, "5"},
	{"Mic_Adjust_Left (filename)", "%s", 50, ""},
	{"Mic_Adjust_Right (filename)", "%s", 50, ""},
	{"So_Left (mV/Pa)", "%f", 5, ""},
	{"So_Right (mV/Pa)", "%f", 5, ""},
	{"Speaker_Channel (L/R)", "%s", 1, ""},
	{"Reference_Channel (L/R)", "%s", 1, ""},
	{"Comments", "%b", 1, "0"},
	{WS_VSEP},
	{"ana-save", "%b", 1, "0"},
	{"ana-compress", "%b", 1, "0"},
	{"ana-decimate (by)", "%d", 2, "2"},
	{WS_HSEP, "Hardware Configuration"},
	{"Gain_Left (dB)", "%f", 5, "0"},
	{"Gain_Right (dB)", "%f", 5, "0"},
	{"Invert_Left", "%b", 1, "0"},
	{"Invert_Right", "%b", 1, "0"},
	{"daFc (Hz)", "%f", 6, "48000"},
	{"adFc (Hz)", "%f", 6, "48000"},
	{"syncPulse", "%b", 1, "0"},
	{"nicelevel", "%d", 3, "-10"},
#ifndef __tdtproc__
	{"proport.pad (dB)", "%d", 3, "0"},
#ifdef __test__
	{WS_VSEP, "Testing Flags"},
	{"use_atten", "%b", 1, "1"},
	{"use_eq", "%b", 1, "1"},
#endif				/* __test__ */
#endif				/* __tdtproc__ */
	{NULL},
};

static Field StandardParams[] = {
	{"Reps (#)", "%d", 2},
	{"Dur (ms)", "%d", 4},
	{"Rise (ms)", "%d", 2},
	{"Fall (ms)", "%d", 2},
	{"Mic_Adjust_Left", "%s", 50},
	{"Mic_Adjust_Right", "%s", 50},
	{"So_Left", "%f", 5},
	{"So_Right", "%f", 5},
	{"Gain_Left", "%f", 5},
	{"Gain_Right", "%f", 5},
	{"Invert_Left", "%b", 1,},
	{"Invert_Right", "%b", 1,},
	{"Speaker_Channel", "%s", 1},
	{"Reference_Channel", "%s", 1},
	{"daFc (Hz)", "%f", 6},
	{"adFc (Hz)", "%f", 6},
	{"version", "%s", 60},
	{"time", "%s", 60},
	{"timestamp", "%d", 12},
	{"xcalibur.fname", "%s", 32},
	{"ana-save", "%b", 1},
	{"ana-decimate (by)", "%d", 3},
};

static void lock_worksheets(void)
{
	ws_lock(ws_prefs, True);
}

void unlock_worksheets(void)
{
	ws_lock(ws_prefs, False);
}

static char *CopyrightMessage()
{
	static char *Copyright = "\
xdphys: Dichotic Physiology Data Acquisition\n\
Copyright (c) 1993 Jamie Mazer\n\
<mazer@asterix.cns.caltech.edu>\n\
This program is NOT for general distribution\n";
	return (Copyright);
}

#if(0)
void plottrace(xword * ibuf, int a, int b)
{
	static int poplast = 0;
	xword *out;
	int nsamps;

	out = is_getDAbuffer(&nsamps);
	XtWaveSetWaveform(lwv, ibuf + 2 * a, (b - a), 2, -RANGE, RANGE);
	XtWaveSetWaveform(rwv, 1 + ibuf + 2 * a, (b - a), 2, -RANGE,
			  RANGE);
	XtWaveSetWaveform(lwv_out, out + 2 * a, (b - a), 2, -RANGE, RANGE);
	XtWaveSetWaveform(rwv_out, 1 + out + 2 * a, (b - a), 2, -RANGE,
			  RANGE);
}

#endif

void plottrace(void)
{
	xword *in, *out;
	int nsamps;

	in = is_getADbuffer(&nsamps);
	XtWaveSetWaveform(lwv, in, nsamps, 2, -RANGE, RANGE);
	XtWaveSetWaveform(rwv, in + 1, nsamps, 2, -RANGE, RANGE);

	out = is_getDAbuffer(&nsamps);
	XtWaveSetWaveform(lwv_out, out, nsamps, 2, -RANGE, RANGE);
	XtWaveSetWaveform(rwv_out, out + 1, nsamps, 2, -RANGE, RANGE);
}

int clipCheck(xword * buf, int from, int to, float perc)
{
	xword max;

	for (max = (xword) (FRANGE * perc); from < to; from++) {
		if ((buf[from * 2] > max) || (buf[from * 2] < -max))
			return (1);
	}
	return (0);
}

int load_mic_adjust_data(int use_default, float *SoL, float *SoR)
{
	char *filename, fbuf[256];
	FILEDATA *fd;
	int ret_val = 1;
	int ndata;
	float *freqs, *lmags, *rmags, *lphis, *rphis;

	filename = GS("Mic_Adjust_Left");
	if (!strcmp(filename, "") && use_default) {
		strcpy(fbuf, "$(XDPHYSDIR)/calib/$(HOST).kleft.fr");
		filename = envexpand(fbuf);
		if (!probefile(filename)) {
			fprintf(stderr, "xcalibur: %s not found\n",
				filename);
			strcpy(fbuf, "$(XDPHYSDIR)/calib/local.kleft.fr");
			filename = envexpand(fbuf);
			fprintf(stderr, "xcalibur: trying %s \n",
				filename);
		}
	}
	if (strcmp(filename, "")) {
		if (!probefile(filename)) {
			alert("Can't find Mic_Adjust_Left.");
			return (0);
		}
		if ((fd = FD_read(filename, 0)) == NULL) {
			alert("Can't read Mic_Adjust_Left: %s", filename);
			lpt_n = 0;
			ret_val = 0;
		} else {
			if (*FD_GV(fd, "Reference_Channel") == 'L') {
				if (strcmp
				    ("", FD_GV(fd, "Mic_Adjust_Right"))) {
					alert
					    ("%s has a non-blank non-ref-channel Mic_Adjust field.",
					     filename);
					return (0);
				}
			} else if (*FD_GV(fd, "Reference_Channel") == 'R') {
				if (strcmp
				    ("", FD_GV(fd, "Mic_Adjust_Left"))) {
					alert
					    ("%s has a non-blank non-ref-channel Mic_Adjust field.",
					     filename);
					return (0);
				}
			}
			if (strcmp(fd->modname, "fr")) {
				alert
				    ("Mic_Adjust_Left file is not a .fr file.",
				     filename);
				lpt_n = 0;
				ret_val = 0;
			} else {
				ndata = fr_get_ndata(fd);
				freqs = fr_get_freqs(fd);
				lmags = fr_get_lmags(fd);
				rmags = fr_get_rmags(fd);
				lphis = fr_get_lphis(fd);
				rphis = fr_get_rphis(fd);

				for (lpt_n = 0;
				     (lpt_n < ndata)
				     && (lpt_n < MAX_PT_PTS); lpt_n++) {
					lpt_freq[lpt_n] = freqs[lpt_n];
					if (*FD_GV(fd, "Reference_Channel")
					    == 'L') {
						lpt_mag[lpt_n] =
						    rmags[lpt_n] /
						    lmags[lpt_n];
						lpt_phi[lpt_n] =
						    rphis[lpt_n] -
						    lphis[lpt_n];
					} else
					    if (*FD_GV
						(fd,
						 "Reference_Channel") ==
						'R') {
						lpt_mag[lpt_n] =
						    lmags[lpt_n] /
						    rmags[lpt_n];
						lpt_phi[lpt_n] =
						    lphis[lpt_n] -
						    rphis[lpt_n];
					}
				}
				if (*FD_GV(fd, "Reference_Channel") == 'L') {
					if (SoL)
						(*SoL) =
						    FD_GF(fd, "So_Left");
				} else if (*FD_GV(fd, "Reference_Channel")
					   == 'R') {
					if (SoL)
						(*SoL) =
						    FD_GF(fd, "So_Right");
				}
				unwrap(lpt_phi, lpt_n, M_PI);

				if (lpt_n == 0) {
					alert("Mic_Adjust_Left is empty.");
					ret_val = 0;
				}
				if (lpt_n < ndata) {
					alert
					    ("Mic_Adjust_Left is too large to fit in alloc'd memory!");
					ret_val = 0;
				}
				FD_free(fd);
				fprintf(stderr, "xcalibur: using %s \n",
					filename);
			}
		}
	} else
		lpt_n = 0;

	filename = GS("Mic_Adjust_Right");
	if (!strcmp(filename, "") && use_default) {
		strcpy(fbuf, "$(XDPHYSDIR)/calib/$(HOST).kright.fr");
		filename = envexpand(fbuf);
		if (!probefile(filename)) {
			fprintf(stderr, "xcalibur: %s not found\n",
				filename);
			strcpy(fbuf, "$(XDPHYSDIR)/calib/local.kright.fr");
			filename = envexpand(fbuf);
			fprintf(stderr, "xcalibur: trying %s \n",
				filename);
		}
	}
	if (strcmp(filename, "")) {
		if (!probefile(filename)) {
			alert("Can't find Mic_Adjust_Right.");
			return (0);
		}
		if ((fd = FD_read(filename, 0)) == NULL) {
			alert("Can't read Mic_Adjust_Right: %s", filename);
			rpt_n = 0;
			ret_val = 0;
		} else {
			if (*FD_GV(fd, "Reference_Channel") == 'L') {
				if (strcmp
				    ("", FD_GV(fd, "Mic_Adjust_Right"))) {
					alert
					    ("%s has a non-blank non-ref-channel Mic_Adjust field.",
					     filename);
					return (0);
				}
			} else if (*FD_GV(fd, "Reference_Channel") == 'R') {
				if (strcmp
				    ("", FD_GV(fd, "Mic_Adjust_Left"))) {
					alert
					    ("%s has a non-blank non-ref-channel Mic_Adjust field.",
					     filename);
					return (0);
				}
			}
			if (strcmp(fd->modname, "fr")) {
				alert
				    ("Mic_Adjust_Right file is not a .fr file.",
				     filename);
				rpt_n = 0;
				ret_val = 0;
			} else {
				ndata = fr_get_ndata(fd);
				freqs = fr_get_freqs(fd);
				lmags = fr_get_lmags(fd);
				rmags = fr_get_rmags(fd);
				lphis = fr_get_lphis(fd);
				rphis = fr_get_rphis(fd);

				for (rpt_n = 0;
				     (rpt_n < ndata)
				     && (rpt_n < MAX_PT_PTS); rpt_n++) {
					rpt_freq[rpt_n] = freqs[rpt_n];
					if (*FD_GV(fd, "Reference_Channel")
					    == 'L') {
						rpt_mag[rpt_n] =
						    rmags[rpt_n] /
						    lmags[rpt_n];
						rpt_phi[rpt_n] =
						    rphis[rpt_n] -
						    lphis[rpt_n];
					} else
					    if (*FD_GV
						(fd,
						 "Reference_Channel") ==
						'R') {
						rpt_mag[rpt_n] =
						    lmags[rpt_n] /
						    rmags[rpt_n];
						rpt_phi[rpt_n] =
						    lphis[rpt_n] -
						    rphis[rpt_n];
					}
				}
				if (*FD_GV(fd, "Reference_Channel") == 'L') {
					if (SoR)
						(*SoR) =
						    FD_GF(fd, "So_Left");
				} else if (*FD_GV(fd, "Reference_Channel")
					   == 'R') {
					if (SoR)
						(*SoR) =
						    FD_GF(fd, "So_Right");
				}
				unwrap(rpt_phi, rpt_n, M_PI);

				if (rpt_n == 0) {
					alert("Mic_Adjust_Left is empty.");
					ret_val = 0;
				}
				if (rpt_n < ndata) {
					alert
					    ("Mic_Adjust_Left is too large to fit in alloc'd memory!");
					ret_val = 0;
				}
				FD_free(fd);
				fprintf(stderr, "xcalibur: using %s \n",
					filename);
			}
		}
	} else
		rpt_n = 0;

	return (ret_val);
}

void query_mic_adjust(float f, char channel, float *mag, float *phi)
{
	assert((channel == 'L') || (channel == 'R'));

	if (channel == 'L') {
		if (lpt_n) {
			if (mag)
				(*mag) =
				    lin_interpolate(f, lpt_n, lpt_freq,
						    lpt_mag);
			if (phi)
				(*phi) =
				    lin_interpolate(f, lpt_n, lpt_freq,
						    lpt_phi);
		} else {
			if (mag)
				(*mag) = 1.0;
			if (phi)
				(*phi) = 0.0;
		}
	} else {
		if (rpt_n) {
			if (mag)
				(*mag) =
				    lin_interpolate(f, rpt_n, rpt_freq,
						    rpt_mag);
			if (phi)
				(*phi) =
				    lin_interpolate(f, rpt_n, rpt_freq,
						    rpt_phi);
		} else {
			if (mag)
				(*mag) = 1.0;
			if (phi)
				(*phi) = 0.0;
		}
	}
}

static void AboutCB(Widget w, XtPointer client_data, XtPointer call_data)
{
	char tmp[64];

	sprintf(tmp, "Version %s", XDPHYS_VERSION);

	pop_box(CopyrightMessage(), tmp, NULL, NULL);
}

static void HelpCB(Widget w, XtPointer name_blah, XtPointer call_data)
{
	char *name = (char *) name_blah;
	char tmp[100], *file;

	if (name != NULL)
		sprintf(tmp, "$(XDPHYSDIR)/doc/xcalibur.%s.doc", name);
	else
		sprintf(tmp, "$(XDPHYSDIR)/doc/xcalibur.doc");
	file = envexpand(tmp);
	if (!probefile(file)) {
		alert("Can't find help file.");
		return;
	}
	help(file, (name == NULL) ? "xcalibur" : name);
}

static void savestate(int bell)
{
	static char tmp[20];
	char *p;

	ws_tosvars(NULL, NULL);
	saveGeometry();
	savevars(statefile, global_svar_table);
	p = GS("Animal");
	sprintf(tmp, "~/.xdphysrc/xd_%s", p);
	savevars(tmp, global_svar_table);
	if (bell)
		beep(0);
}


static void SaveStateCB(Widget w, XtPointer client_data,
			XtPointer call_data)
{
	savestate(1);
}

static void Quit(Widget w, XtPointer client_data, XtPointer call_data)
{
	if (progRunning) {
		beep(0);
	} else {
		SaveStateCB(w, NULL, NULL);
		clockbar_save();
		unlockApp();
		is_shutdown();
		varsdelete(global_svar_table);
		getpref(NULL, NULL);
		if (w)
			XtCloseDisplay(XtDisplay(TopLevel));
		cleanupParms();
		cache_delete_all();
		exit(0);
	}
}

int default_display_fn(char *datafile)
{
	if (getenv("XDPHYS_NO_GRAPH") == NULL)
		FDObj_File_New(datafile, NULL, NULL);
	return (1);
}

static int appRun(int temp_run, RunData * info, Widget headline)
{
	char *datafile, *comments;
#if 0
	char *anafile;
#endif
	FILE *datafp;
	int scan_result;
	int i;
	FILEDATA *fd;
#ifdef __tdtproc__
	int epoch = GI("epoch");
#endif				/* __tdtproc__ */
	int ana_save;

	if (temp_run == TEMP_YES) {
		datafile = (char *) malloc(strlen(info->suffix) + 6);
		sprintf(datafile, "temp.%s", info->suffix);
	} else if (temp_run == TEMP_NAMED) {
		char buf[100];
		sprintf(buf, "%s.???.%s", GS("animal"), info->suffix);
		datafile = fileBox(buf, "w", "Save to ...");
		if (datafile == NULL)
			return (0);
	} else {		/* TEMP_NO */
		char buf[100];
		sprintf(buf, ".%s", info->suffix);
		datafile = fileBox(buf, "w", "Save to ...");
		if (datafile == NULL)
			return (0);
	}

	SS("xcalibur.fname", datafile);

	if (GI("Comments")) {
		if ((temp_run == TEMP_NO) || (temp_run == TEMP_NAMED)) {
			comments = getComments(datafile);
			if (comments == NULL) {
				return (0);
			}
		} else {
			comments = NULL;
		}
	}
	if (probefile(datafile) && (temp_run == TEMP_NO)) {
		if ((fd = FD_read(datafile, 0)) == NULL) {
			alert("FD_read failed.");
			return (0);
		}
	} else {
		if ((datafp = fopen2(findhome(datafile), "w")) == NULL) {
			alert("Can't write datafile, try again");
			return (0);
		}
	}

	ana_save = GI("ana-save");
	if (ana_save) {
		recording(1);
/*
		anafile = (char *) malloc(strlen(datafile) + 10);
		sprintf(anafile, "%s.ana", datafile);
		if ((analogfp = fopen2(findhome(anafile), "w")) == NULL)
			notify("Warning: Can't write analog data, going on..");
*/
	} else {
		recording(0);
/*
		analogfp = NULL;
		anafile = NULL;
*/
	}

#ifdef __tdtproc__
	is_init(GF("daFc"), 0.0, 0.0, epoch, epoch);
	ws_tows(NULL, NULL);
	is_shutdown();
#endif				/* __tdtproc__ */

	fprintf(datafp, ";; xcalibur %s ver %s\n",
		info->suffix, XDPHYS_VERSION);

	/* Comment Block */
	fprintf(datafp, "COMMENTS\n");
	if (debugflag)
		fprintf(stderr, "comments=<%s>\n",
			comments ? comments : "NULL");
	if (GI("Comments"))
		writeComments(comments, datafp);
	fprintf(datafp, "END_COMMENTS\n");

	/* Parameter Block */
	fprintf(datafp, "PARAMS\n");
	ws_write(datafp, info->params, info->nparams, NULL);
	if (info->depvar)
		fprintf(datafp, "depvar=%s\n", info->depvar);
	fprintf(datafp, "; StandardParams\n");
	ws_write(datafp, StandardParams, WS_N(StandardParams), NULL);
	/*fprintf(datafp, "xcalibur.fname=%s\n", datafile); */
	fprintf(datafp, "END_PARAMS\n");

	/* Rasterdata Block.. may or may not end properly */
	fprintf(datafp, "RASTERDATA\n");

	this_run = datafile;
	scan_result = (*info->scan_fn) (datafp, headline);
	this_run = NULL;

	if (datafp) {
		fprintf(datafp, "END_RASTERDATA\n");
		if (scan_result <= 0)
			fprintf(datafp,
				";; NOTE: run aborted or failed!!\n");
		fclose(datafp);
/*
		if (analogfp)
			fclose(analogfp);
*/
	}
	ws_tosvars(NULL, NULL);	/* in case of any changes during run.. */

	if (scan_result > 0) {
		/* Successful run completed, save the file */
		FDObj_File_New(datafile, NULL, NULL);

	} else if (scan_result == 0 && (temp_run != TEMP_YES)) {
		/* Run was aborted, if temp_run, then delete the file;
		 * otherwise, give the user some options..  */
		i = pop_box("Aborted, what do to with partial data file?",
			    "Delete", "Save", "Save and Plot");
		switch (i) {
		case 1:
#if 0
			unlink(datafile);
			if (anafile) {
				unlink(anafile);
				free(anafile);
				anafile = NULL;
			}
#endif
			break;
		case 3:
			FDObj_File_New(datafile, NULL, NULL);
			break;
		case 2:
			break;
		}
	} else {
#if 0
		unlink(datafile);
		if (anafile) {
			unlink(anafile);
			free(anafile);
			anafile = NULL;
		}
#endif
	}

#if 0
	if (anafile) {
		if (GI("ana-compress")) {
			char buf[MAXPATHLEN + 100];
			/* note that compress/gzip will fail if compressed version is bigger
			 * than the original -- so don't be surprised if not all the
			 * ana files get compressed..  */
			fprintf(stderr,
				"Warning: compressing \"%s\" in background",
				anafile);
			sprintf(buf, "exec gzip %s", anafile);
			system_noblock(buf);
		}
		free(anafile);
	}
	free(datafile);
#endif
	return (scan_result);
}

static void ModuleStartFn(Widget w, RunData * info, int temp_run)
{
	if (!progRunning) {
		if (ws_verify(NULL)) {

			lock_worksheets();
			ws_tosvars(NULL, NULL);
			savevars(statefile, global_svar_table);
			ReverseColors(w);
			set_progRunning(True);

			if (info->pre_fn != NULL)
				(*info->pre_fn) ();

			if (info->scan_fn)
				appRun(temp_run, info, headline);
			else
				alert("No scan function available!");

			if (info->post_fn != NULL)
				(*info->post_fn) ();

			unlock_worksheets();
			setRack(0, attMaxAtten, attMaxAtten);
			set_progRunning(False);
			ReverseColors(w);
		}
	} else {
		beep(0);
	}
}

static void ModuleStartCB(Widget w, XtPointer info_blah,
			  XtPointer call_data)
{
	RunData *info = (RunData *) info_blah;

	ModuleStartFn(w, info, TEMP_NO);
}

static void ModuleCloseCB(Widget w, XtPointer slot_blah,
			  XtPointer call_data)
{
	DockSlot *slot = (DockSlot *) slot_blah;

	XtVaSetValues(slot->dock, XtNstate, 0, NULL);
	popdown_dockslot(slot);
}

static void ModuleTempCB(Widget w, XtPointer info_blah,
			 XtPointer call_data)
{
	RunData *info = (RunData *) info_blah;

	ModuleStartFn(w, info, TEMP_YES);
}

static void ModuleNamedTempCB(Widget w, XtPointer info_blah,
			      XtPointer call_data)
{
	RunData *info = (RunData *) info_blah;

	ModuleStartFn(w, info, TEMP_NAMED);
}

void *make_module_info(char *depvar,	/* name of dependent variable (eg "itd") */
		       char *suffix,	/* suffix for file-type i.d. */
		       char *helpfile,	/* search string for help file */
		       int (*pre) (),	/* call before scan (i.e. to LOCK ws) */
		       int (*scan) (),	/* actually collect the data.. */
		       int (*post) (),	/* call after scan (i.e. to UNLOCK ws) */
		       int (*display) (), FieldList params,	/* parameters for writing headers */
		       int nparams)
{				/* number of parameters */
	RunData *info;

	info = (RunData *) malloc(sizeof(RunData));
	info->depvar = depvar;
	info->suffix = suffix;
	info->helpfile = helpfile;
	info->pre_fn = pre;
	info->scan_fn = scan;
	info->post_fn = post;
	info->display_fn = display;
	info->params = params;
	info->nparams = nparams;
	return ((void *) info);
}

static void make_close_help(Widget handle, char *name, DockSlot * slot)
{
	Widget w1, w2;

	w1 = button_new(XtParent(handle), "Close", "Close", ModuleCloseCB,
			slot, handle, NULL);
	accel_from(XtParent(handle), w1);

	w2 = button_new(XtParent(handle), "Help", "Help", HelpCB,
			name, w1, NULL);
	accel_from(XtParent(handle), w2);
}

void make_start_temp(Widget handle, RunData * info, XtCallbackProc StartCB,
		     XtCallbackProc TempCB, DockSlot * slot)
{
	Widget w1, w2, w3, w4, w5;

	w1 = button_new(XtParent(handle), "Close", "Close", ModuleCloseCB,
			slot, handle, NULL);
	accel_from(XtParent(handle), w1);

	w2 = button_new(XtParent(handle), "Help", "Help", HelpCB,
			info->helpfile, w1, NULL);
	accel_from(XtParent(handle), w2);

	w3 = button_new(XtParent(handle), "Start", "Start",
			StartCB ? StartCB : ModuleStartCB, info, w2, NULL);
	accel_from(XtParent(handle), w3);

	w4 = button_new(XtParent(handle), "Temp", "Temp",
			TempCB ? TempCB : ModuleTempCB, info, w3, NULL);
	accel_from(XtParent(handle), w4);

	w5 = button_new(XtParent(handle), "Named-Temp", "Named-Temp",
			ModuleNamedTempCB, info, w4, NULL);
	accel_from(XtParent(handle), w5);
}

static void QuitAction(Widget w, XEvent * event, String * params,
		       Cardinal * num_params)
{
	Quit(NULL, NULL, NULL);
}

static void install_actions(void)
{
	static XtActionsRec actionTable[] = {
		{"quit", QuitAction},
		{"pause", PauseAction},
	};

	XtAppAddActions(XtWidgetToApplicationContext(TopLevel),
			actionTable, XtNumber(actionTable));
}

static int calibmain(int ac, char **av, char *appname, char *classname)
{
	int arg, i;
	Widget mainform, w, dockform;
	Widget h;
	char tmp[64];

	static MENU_ENTRY OtherMenu[] = {
		{"Other Menu", NULL},
		{"Combine fr files", CombineFrFiles, (void *) NULL},
		{NULL}
	};

	/* sigfatal_setup(); */

	progname = av[0];
	makeXDphysrc();
	initsvars();

	if (!loadvars(statefile, global_svar_table))
		fprintf(stderr, "warning: can't find config file \"%s\"\n",
			statefile);

	SS("caltype", "? None ?");

	install_xcalibur_plotMethods();

	startx(&ac, av, appname ? appname : "xcalibur",
	       classname ? classname : "xcalibur");
	install_actions();

#define isarg(pat) (strncmp(av[arg], pat, strlen(av[arg])) == 0)
	for (arg = 1; arg < ac; arg++) {
		if (isarg("-ov")) {
			unlockApp();
		} else if (isarg("-help") || isarg("-usage")) {
			usage(progname, NULL);
			exit(0);
		} else if (*av[arg]) {
			usage(progname, av[arg]);
		}
	}
#undef isarg

	lockApp();

	mainform = form_new(TopLevel, "main", 0);

	w = NULL;
	w = button_new(mainform, "About", "About", AboutCB, (void *) NULL,
		       w, headline);
	w = button_new(mainform, "Help", "Help", HelpCB, (void *) NULL, w,
		       headline);
	w = button_new(mainform, "Quit", "Quit", Quit, NULL, w, headline);
	w = menu_new(mainform, "Other", OtherMenu, w, headline);

	headline = statline_new(mainform, 40);
	XtVaSetValues(headline, XtNfromVert, NULL, XtNfromHoriz, w, NULL);

	waveshell = waveview_new(mainform, &lwv, &rwv, -1, 1);
	XtVaSetValues(waveshell, XtNfromVert, w, NULL);
	waveshell_out = waveview_new(mainform, &lwv_out, &rwv_out, -1, 1);
	XtVaSetValues(waveshell_out, XtNfromVert, waveshell, NULL);

	tooldock = dock_new(mainform, NULL, 20, &dockform);
	XtVaSetValues(dockform, XtNfromVert, w, XtNfromHoriz, waveshell,
		      NULL);

	install_modules(tooldock);

	i = dock_add(tooldock, "Prefs",
		     pop_ws_new(TopLevel, ws_prefs, "Prefs", &h, NULL,
				NULL));
	make_close_help(h, "prefs", &tooldock->slots[i]);

	sprintf(tmp, "Version %s", XDPHYS_VERSION);
	statline_set(headline, "xcalibur -- %s", tmp);

	XtRealizeWidget(TopLevel);

	/* setup some things for iserver and synth lib */
	is_setFcSetFn(xd_setFc);
	is_setAlertFn(alert);
	is_setNotifyFn(notify);
	is_setYesNoFn(pop_box);
	synth_init();

#ifdef DO_COPYRIGHT
	DisplayNotice(CopyrightMessage());
#endif

	return (1);		/* caller should enter xloop */
}

int main(int ac, char **av)
{
	if (calibmain(ac, av, "xcalibur", "XDphys"))
		dloop();
	exit(0);
}
