/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: waveio.c,v 2.55 2002/07/15 04:30:25 cmalek Exp $
** Program: dowl
**  Module: waveio.c
**  Author: mazer
** Descrip: read/write waveforms
**
** Revision History (most recent last)
**
** Fri Nov 18 14:06:07 1994 mazer
**  Added some minimal support for saving calibration information in
**  analog data files. The basic idea is as follows: two key pieces
**  of information are required to recover accurate voltages:
**  the number of adticks per mv and the dc-offset inherent in the
**  ad converters.  These are floating point values, so in order
**  to store them in the header as ints, I multiply by 1000 and
**  store the "1000" in case it ever changes.. therefore,
**            wnfo->tomv
** tomv =    ------------  = the number of mv per tick (1 mv in ad-ticks)
**          winfo->tomv_div
**  and
**          winfo->offset
** offset = ------------  = the the dc offset in ad-ticks
**        winfo->offset_div
**
**  Retrieve these values (tomv, offset) using wave_getcalib().
**  To convert ad-ticks to mv, use the following formula:
**     actual-mv = (adticks - offset) * tomv
**
** Mon Dec 26 12:29:27 1994 mazer
**  deleted fopenwave() --> use fopen2() instead..
**
*******************************************************************/

#include "xdphyslib.h"

#ifdef __linux__
#include <endian.h>
#ifdef __LITTLE_ENDIAN
#define	M_32_SWAP(a) {							\
	unsigned int _tmp = a;						\
	((char *)&a)[0] = ((char *)&_tmp)[3];				\
	((char *)&a)[1] = ((char *)&_tmp)[2];				\
	((char *)&a)[2] = ((char *)&_tmp)[1];				\
	((char *)&a)[3] = ((char *)&_tmp)[0];				\
}
#define	M_16_SWAP(a) {							\
	unsigned int _tmp = a;						\
	((char *)&a)[0] = ((char *)&_tmp)[1];				\
	((char *)&a)[1] = ((char *)&_tmp)[0];				\
}
#endif /*__LITTLE_ENDIAN */
#endif				/* LINUX */


int writewave(FILE * fp, xword * buffer, int fc, int nsamps, int nchans,
	      char *comment, char *program, float tomv, float offset,
	      char *xdphys_filename, int xdphys_rasternum)
{
	WAVEINFO header;
	char pad[HEADER_PAD];
#ifdef __linux__
	int i;
#endif


	header.majik = WAVE_MAJIK;
	header.fc = fc;
	header.nsamps = nsamps;
	header.nchans = nchans;
	header.bits = 8 * sizeof(xword);
	if (tomv == 0.0 && offset == 0.0) {
		header.tomv = header.tomv_div = 0;
		header.offset = header.offset_div = 0;
	} else {
		header.tomv = RND2INT(tomv * 1000.0);
		header.tomv_div = 1000;
		header.offset = RND2INT(offset * 1000.0);
		header.offset_div = 1000;
	}

	if (comment)
		strncpy(header.comment, comment, sizeof(header.comment));
	else
		strncpy(header.comment, "No comment entered.",
			sizeof(header.comment));
	if (program)
		strncpy(header.program, program, sizeof(header.program));
	else
		strncpy(header.program, "?prog?", sizeof(header.program));
	strncpy(header.arch, ARCH, sizeof(header.arch));

	if (xdphys_filename)
		strncpy(header.xdphys_filename, xdphys_filename,
			sizeof(header.xdphys_filename));
	else
		strncpy(header.xdphys_filename, "???",
			sizeof(header.xdphys_filename));
	header.xdphys_rasternum = xdphys_rasternum;


#ifdef __linux__
	/* Deal with endianness: make everything big endian, like the sparcs */
	M_16_SWAP(header.majik);
	M_32_SWAP(header.fc);
	M_32_SWAP(header.nsamps);
	M_32_SWAP(header.nchans);
	M_32_SWAP(header.tomv);
	M_32_SWAP(header.tomv_div);
	M_32_SWAP(header.offset);
	M_32_SWAP(header.offset_div);
	M_32_SWAP(header.xdphys_rasternum);

	for (i = 0; i < (nsamps * nchans); i++)
		M_16_SWAP(buffer[i]);
#endif

	if ((fwrite(&header, sizeof(header), 1, fp) != 1) ||
	    (fwrite(pad, sizeof(pad), 1, fp) != 1) ||
	    (fwrite(buffer, sizeof(xword) * nchans, nsamps, fp) !=
	     nsamps)) {
		return (0);
	}
	return (1);
}

int savewave(char *filename, xword * buffer, int fc, int nsamps,
	     int nchans, char *comment, char *program, float tomv,
	     float offset, char *xdphys_filename, int xdphys_rasternum)
{
	FILE *fp;

	if ((fp = fopen2(filename, "w")) == NULL) {
		alert("Can't open '%s' for write", filename);
		return (0);
	}

	if (!writewave
	    (fp, buffer, fc, nsamps, nchans, comment, program, tomv,
	     offset, xdphys_filename, xdphys_rasternum)) {
		alert("Write to '%s' failed", filename);
		fclose(fp);
		return (0);
	}

	fclose(fp);
	return (1);
}

int appendwave(char *filename, xword * buffer, int fc, int nsamps,
	       int nchans, char *comment, char *program, float tomv,
	       float offset, char *xdphys_filename, int xdphys_rasternum)
{
	FILE *fp;

	if ((fp = fopen2(filename, "a")) == NULL) {
		alert("Can't open '%s' for write", filename);
		return (0);
	}

	if (!writewave
	    (fp, buffer, fc, nsamps, nchans, comment, program, tomv,
	     offset, xdphys_filename, xdphys_rasternum)) {
		alert("Write to '%s' failed", filename);
		fclose(fp);
		return (0);
	}

	fclose(fp);
	return (1);
}

int wave_getcalib(WAVEINFO * winfo, float *tomv, float *offset)
{
	if (winfo->majik == WAVE_MAJIK_0) {
		return (0);
	} else {
		if (tomv)
			*tomv =
			    (float) winfo->tomv / (float) winfo->tomv_div;
		if (offset)
			*offset =
			    (float) winfo->offset /
			    (float) winfo->offset_div;
		return (1);
	}
}

xword *readwave(FILE * fp, WAVEINFO ** winfo_ret)
{
	WAVEINFO *winfo;
	char pad[HEADER_PAD];
	xword *buffer;
#ifdef __linux__
	int i;
#endif

	winfo = (WAVEINFO *) calloc(1, sizeof(WAVEINFO));

	if ((fread(winfo, sizeof(WAVEINFO), 1, fp) != 1) ||
	    (fread(pad, sizeof(pad), 1, fp) != 1)) {
		free(winfo);
		return (NULL);
	}
#ifdef __linux__
	/* Deal with endianness: make everything big endian, like the sparcs */
	M_16_SWAP(winfo->majik);
	M_32_SWAP(winfo->fc);
	M_32_SWAP(winfo->nsamps);
	M_32_SWAP(winfo->nchans);
	M_32_SWAP(winfo->tomv);
	M_32_SWAP(winfo->tomv_div);
	M_32_SWAP(winfo->offset);
	M_32_SWAP(winfo->offset_div);
	M_32_SWAP(winfo->xdphys_rasternum);
#endif

	switch (winfo->majik) {
	case WAVE_MAJIK_0:
	case WAVE_MAJIK:
		buffer =
		    (xword *) calloc(winfo->nsamps * winfo->nchans,
				     sizeof(xword));
		if (fread
		    (buffer, sizeof(xword) * winfo->nchans, winfo->nsamps,
		     fp) != winfo->nsamps) {
			free(winfo);
			free(buffer);
			return (NULL);
		}
		if (winfo->majik == WAVE_MAJIK_0) {
			winfo->tomv = winfo->tomv_div = 0;
			winfo->offset = winfo->offset_div = 0;
		}
		break;
	default:
		free(winfo);
		return (NULL);
		break;
	}

#ifdef __linux__
	for (i = 0; i < (winfo->nsamps * winfo->nchans); i++)
		M_16_SWAP(buffer[i]);
#endif

	if (winfo_ret)
		*winfo_ret = winfo;
	else
		free(winfo);

	return (buffer);
}

xword *loadwave(char *filename, WAVEINFO ** winfo_ret)
{
	xword *buffer;
	FILE *fp;

	if ((fp = fopen2(filename, "r")) == NULL) {
		alert("Can't open '%s' for read", filename);
		return (NULL);
	}
	if ((buffer = readwave(fp, winfo_ret)) == NULL)
		alert("Read of %s failed", filename);
	fclose(fp);
	return (buffer);
}

xword *nthwave(char *filename, WAVEINFO ** winfo_ret, int n,
	       int *toofar_ptr)
{
	xword *buffer;
	FILE *fp;

	*winfo_ret = NULL;
	*toofar_ptr = 0;
	buffer = NULL;

	if ((fp = fopen2(filename, "r")) == NULL) {
		alert("Can't open '%s' for read", filename);
		return (NULL);
	}

	while (n--) {
		if (*winfo_ret != NULL)
			free(*winfo_ret);
		if (buffer != NULL)
			free(buffer);

		if ((buffer = readwave(fp, winfo_ret)) == NULL) {
			fclose(fp);
			if (toofar_ptr != NULL)
				*toofar_ptr = 1;
			return (NULL);
		}
	}

	fclose(fp);
	return (buffer);
}

xword *loadsongedfile(char *filename, WAVEINFO ** winfo_ret)
{
	xword *buffer;
	char line[80];
	int nsamps;
	FILE *fp;
	WAVEINFO *winfo;
#ifdef __linux__
	int i;
#endif

	if ((fp = fopen2(filename, "r")) == NULL) {
		return (NULL);
	}
	fgets(line, sizeof(line), fp);
	if (strcmp(line, "AD_FREQ: 32000 Hz\n") != 0) {
		fclose(fp);
		return (NULL);
	}
	if (fread(&nsamps, sizeof(nsamps), 1, fp) != 1) {
		fclose(fp);
		return (NULL);
	}
#ifdef __linux__
	M_32_SWAP(nsamps);
#endif
	buffer = (xword *) calloc(nsamps, sizeof(xword));
	if (fread(buffer, sizeof(xword), nsamps, fp) != nsamps) {
		fclose(fp);
		return (NULL);
	}
	fclose(fp);

#ifdef __linux__
	for (i = 0; i < nsamps; i++)
		M_16_SWAP(buffer[i]);
#endif

	winfo = (WAVEINFO *) calloc(1, sizeof(WAVEINFO));
	winfo->majik = 0;
	winfo->fc = 32000;
	winfo->nsamps = nsamps;
	winfo->nchans = 1;
	strcpy(winfo->comment, "songed file");
	winfo->bits = 16;
	strcpy(winfo->program, "songed");
	strcpy(winfo->arch, "???");
	*winfo_ret = winfo;
	return (buffer);
}

int iswavefile(char *filename)
{
	int n;
	xword *buffer;
	WAVEINFO *winfo;
	FILE *fp;

	if ((fp = fopen2(filename, "r")) == NULL) {
		return (0);
	}
	for (n = 0; 1; n++) {
		if ((buffer = readwave(fp, &winfo)) == NULL) {
			fclose(fp);
			return (n);
		}
		free(winfo);
		free(buffer);
	}
}
