/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: statfns.c,v 2.62 2002/07/15 04:30:22 cmalek Exp $
** Program: dowl
**  Module: statfns.c
**  Author: mazer
** Descrip: simple descriptive statistics functions
**
** Revision History (most recent last)
**
** 97.2 bjarthur
**  added median, mode, quantile
**
** 97.4 bjarthur
**  added combine_*
**
*******************************************************************/

#include "xdphyslib.h"

static void my_ftest(float, float, float, float, float *, float *);
static float overlap(float, float, float, float);

#define safe_malloc(f) if((f)==NULL) { \
    fprintf(stderr,"ERROR: out of memory.\n"); return(0); }

/* following two defines are used by findPeak and findPeak_at */
#define TOLERANCE		0.1
#define MIN_TOLERANCE   TOLERANCE*(maxgraphy-mingraphy)


static void my_ftest(float var1, float n1, float var2, float n2,
		     float *f, float *prob)
{
	(*f) = var1 / var2;
	(*prob) = betai(0.5 * n2, 0.5 * n1, n2 / (n2 + n1 * (*f)));
}


/*
two-way ANOVA with fixed effects (model I) and proportional replication.
see Hays (1981, p. 358) for algorithmic details, or Zar (1996, ch 10,12) for a
good discussion of issues (including proportional, as opposed to equal,
replication).  see also JAStatA 92(440):1581, JAStatA 91(436):1619, JAStatA
90(432):1388, and Stat+ProbLetters 23(2):183 for discussions on non-parametric
alternatives to ANOVA
*/

void anova2(float ***data, int R, int C, int **n,
	    float *Fr, float *Fc, float *Fi, float *pr, float *pc,
	    float *pi)
{
	float A, B, *D, *d, *G, *g, **H;
	float SSr, SSc, SSi, SSe, SSt;
	float MSr, MSc, MSi, MSe;
	int DFr, DFc, DFi, DFe, DFt;
	int i, j, k, N;

	/* step 1 */
	for (N = 0, j = 0; j < C; j++)
		for (k = 0; k < R; k++)
			N += n[k][j];

	/* step 2 */
	for (A = 0.0, j = 0; j < C; j++)
		for (k = 0; k < R; k++)
			for (i = 0; i < n[k][j]; i++)
				A += (data[k][j][i] * data[k][j][i]);

	/* step 3 */
	assert((H = (float **) malloc(R * sizeof(float *))) != NULL);
	for (k = 0; k < R; k++) {
		assert((H[k] =
			(float *) malloc(C * sizeof(float))) != NULL);
		for (j = 0; j < C; j++) {
			for (H[k][j] = 0.0, i = 0; i < n[k][j]; i++)
				H[k][j] += data[k][j][i];
		}
	}

	/* step 4 */
	for (B = 0.0, j = 0; j < C; j++)
		for (k = 0; k < R; k++)
			B += H[k][j];
	SSt = A - (B * B) / (float) N;

	/* step 5 */
	assert((D = (float *) malloc(R * sizeof(float))) != NULL);
	assert((d = (float *) malloc(R * sizeof(float))) != NULL);
	for (k = 0; k < R; k++) {
		for (D[k] = d[k] = 0.0, j = 0; j < C; j++) {
			D[k] += H[k][j];
			d[k] += (float) n[k][j];
	}}

	/* step 6 */
	for (SSr = 0.0, k = 0; k < R; k++)
		SSr += (D[k] * D[k] / d[k]);
	SSr = SSr - (B * B) / (float) N;

	/* step 7 */
	assert((G = (float *) malloc(C * sizeof(float))) != NULL);
	assert((g = (float *) malloc(C * sizeof(float))) != NULL);
	for (j = 0; j < C; j++) {
		for (G[j] = g[j] = 0.0, k = 0; k < R; k++) {
			G[j] += H[k][j];
			g[j] += (float) n[k][j];
	}}

	/* step 8 */
	for (SSc = 0.0, j = 0; j < C; j++)
		SSc += (G[j] * G[j] / g[j]);
	SSc = SSc - (B * B) / (float) N;

	/* step 9 */
	for (SSe = 0.0, j = 0; j < C; j++)
		for (k = 0; k < R; k++)
			SSe += (H[k][j] * H[k][j] / (float) n[k][j]);
	SSe = A - SSe;

	/* step 10 */
	SSi = SSt - SSr - SSc - SSe;

	/* steps 12-15 */
	DFr = R - 1;
	MSr = SSr / DFr;
	DFc = C - 1;
	MSc = SSc / DFc;
	DFi = (C - 1) * (R - 1);
	MSi = SSi / DFi;
	for (DFe = 0, DFt = -1, j = 0; j < C; j++)
		for (k = 0; k < R; k++) {
			DFe += (n[k][j] - 1);
			DFt += n[k][j];
		}
	MSe = SSe / DFe;

	/* steps 16-18 */
	my_ftest(MSr, (float) DFr, MSe, (float) DFe, Fr, pr);
	my_ftest(MSc, (float) DFc, MSe, (float) DFe, Fc, pc);
	my_ftest(MSi, (float) DFi, MSe, (float) DFe, Fi, pi);

#if(0)
	fprintf(stdout,
		"source  ss       df       ms       f        pval\n");
	fprintf(stdout,
		"------  --       --       --       -        ----\n");
	fprintf(stdout, "rows    %8.1f %8d %8.1f %8.1f %f\n", SSr, DFr,
		MSr, *Fr, *pr);
	fprintf(stdout, "cols    %8.1f %8d %8.1f %8.1f %f\n", SSc, DFc,
		MSc, *Fc, *pc);
	fprintf(stdout, "inter   %8.1f %8d %8.1f %8.1f %f\n", SSi, DFi,
		MSi, *Fi, *pi);
	fprintf(stdout, "err     %8.1f %8d %8.1f\n", SSe, DFe, MSe);
	fprintf(stdout, "tot     %8.1f %8d\n\n", SSt, DFt);
#endif

	for (k = 0; k < R; k++)
		free(H[k]);
	free(H);
	free(D);
}


/* see Siegel and Castellan (1988), "Nonparametric Statistics for the
   Behavioral Sciences", pp. 151-5 for a description of this test.
   implemented here are three different algorithms which compute this
   statistic.  _rand() uses a fixed number of randomly generated
   partitions to estimate the p-val.  _bin() iterates through all
   possible partitions to determine the p-val exactly (if there's no
   roundoff error).  note that this is exponentially slow.  _dp()
   uses a dynamic programming algorithm (thanks to Sam Roweis and
   Erik Winfree of John Hopfield's group) to iterate through all the
   partitions in polynomial time.  it's also subject to roundoff error.
   furthermore, it only works on positive integers, whereas the others
   can deal with floats.  note that all these routines compute the
   one-tailed statistic, that is, the p-val is low if one[] > two[]. */

#if(0)
float permutation_test_rand(float *one, int n1, float *two, int n2, int np)
{
	float *join, diff, diffp;
	int *bits;
	double fraction, total;
	int i, j;
	struct timeval tp1, tp2;
	struct timezone tzp;
	int time;

	gettimeofday(&tp1, &tzp);

	assert((join =
		(float *) malloc((n1 + n2) * sizeof(float))) != NULL);
	assert((bits = (int *) calloc((n1 + n2), sizeof(int))) != NULL);

	/* compute difference of the sums */
	for (diff = 0.0, i = 0; i < n1; i++)
		diff += one[i];
	for (i = 0; i < n2; i++)
		diff -= two[i];

	/* form union of one[] and two[] */
	for (i = j = 0; i < n1; i++, j++)
		join[j] = one[i];
	for (i = 0; i < n2; i++, j++)
		join[j] = two[i];

	/* enumerate some of the possible partitions */
	for (i = 0; i < n1; i++)
		bits[i] = 1;
	for (i = 0, fraction = 0.0; i < np; i++) {
		randomize_array(bits, (n1 + n2));
		for (j = 0, diffp = 0.0; j < (n1 + n2); j++)
			diffp += ((bits[j] == 1) ? join[j] : -join[j]);
		if (diffp >= diff)
			fraction += 1.0;
	}
	total = (double) np;

	gettimeofday(&tp2, &tzp);
	time = (int) ((tp2.tv_sec - tp1.tv_sec) * 1e3L +
		      (tp2.tv_usec - tp1.tv_usec) / 1e3L);
	if (debugflag)
		fprintf(stderr, "%dms\n", time);

	free(join);
	free(bits);
	return ((float) (fraction / total));
}

static int binary_incr(int *bits, int num)
{
	int carry = 1, i = 0;

	while (carry && (i < num)) {
		bits[i] += carry;
		if (bits[i] == 2) {
			bits[i] = 0;
			carry = 1;
		} else
			carry = 0;
		i++;
	}

	if (carry)
		return (1);
	return (0);
}

static int num_one_bits(int *bits, int num)
{
	int i, ret_val;

	for (i = ret_val = 0; i < num; i++)
		ret_val += bits[i];

	return (ret_val);
}

float permutation_test_bin(float *one, int n1, float *two, int n2)
{
	float *join, diff, diffp;
	int *bits;
	double fraction, total;
	int i, j, overflow;
	struct timeval tp1, tp2;
	struct timezone tzp;
	int time;

	gettimeofday(&tp1, &tzp);

	assert((join =
		(float *) malloc((n1 + n2) * sizeof(float))) != NULL);
	assert((bits = (int *) calloc((n1 + n2), sizeof(int))) != NULL);

	/* compute difference of the sums */
	for (diff = 0.0, i = 0; i < n1; i++)
		diff += one[i];
	for (i = 0; i < n2; i++)
		diff -= two[i];

	/* form union of one[] and two[] */
	for (i = j = 0; i < n1; i++, j++)
		join[j] = one[i];
	for (i = 0; i < n2; i++, j++)
		join[j] = two[i];

	fraction = total = 0.0;
	do {
		do {
			overflow = binary_incr(bits, (n1 + n2));
		}
		while ((!overflow)
		       && (num_one_bits(bits, (n1 + n2)) != n1));
		if (overflow)
			continue;
		for (j = 0, diffp = 0.0; j < (n1 + n2); j++)
			diffp += ((bits[j] == 1) ? join[j] : -join[j]);
		if (diffp >= diff)
			fraction += 1.0;
		total += 1.0;
	}
	while (!overflow);

	if (fabs(1.0 - total / bico((n1 + n2), n1)) > 0.01)
		fprintf(stderr,
			"WARNING: %5.3f%% round-off error in permutation_test_bin.\n",
			100.0 * fabs(1.0 - total / bico((n1 + n2), n1)));

	gettimeofday(&tp2, &tzp);
	time = (int) ((tp2.tv_sec - tp1.tv_sec) * 1e3L +
		      (tp2.tv_usec - tp1.tv_usec) / 1e3L);
	if (debugflag)
		fprintf(stderr, "%dms\n", time);

	free(join);
	free(bits);
	return ((float) (fraction / total));
}
#endif

/* this version has been modified to compute either one- or two-tailed
statistics.  for side == 1, the pval is low if one < two;  for side == -1,
pval is low for one > two;  and for side == 0, pval is low for one != two */

float permutation_test_dp(int *one, int n1, int *two, int n2, int side)
{
	int *join, ss, sum;
	int i, j, n, t, p, ll;
	double ***m, fraction, total;
	struct timeval tp1, tp2;
	struct timezone tzp;
	int time;
	float ret_val;

	assert((side == 1) || (side == -1) || (side == 0));

	gettimeofday(&tp1, &tzp);

	/* form union of one[] and two[] */
	assert((join = (int *) malloc((n1 + n2) * sizeof(int))) != NULL);
	for (i = j = 0; i < n1; i++, j++)
		join[j] = one[i];
	for (i = 0; i < n2; i++, j++)
		join[j] = two[i];

	/* compute sizes and alloc storage */
	ll = n1 + n2;
	qsort(join, ll, sizeof(int), intcompare);
	for (i = (n1 + n2 - 1), ss = 0; i >= n2; i--)
		ss += join[i];
	assert((m =
		(double ***) calloc((ll + 1), sizeof(double **))) != NULL);
	for (i = 0; i < (ll + 1); i++) {
		assert((m[i] =
			(double **) calloc((ss + 1),
					   sizeof(double *))) != NULL);
		for (j = 0; j < (ss + 1); j++)
			assert((m[i][j] =
				(double *) calloc((n1 + 1),
						  sizeof(double))) !=
			       NULL);
	}

	/* loop, loop, loop */
	m[0][0][0] = 1.0;
	for (n = 0; n < ll; n++) {
		for (t = 0; t <= ss; t++) {
			for (p = 0; p < n1; p++) {
				if (m[n][t][p] > 0.0) {
					m[n + 1][t][p] += m[n][t][p];
					m[n + 1][t + join[n]][p + 1] +=
					    m[n][t][p];
				}
			}
			m[n + 1][t][n1] += m[n][t][n1];
		}
	}

	/* find pval */
	for (i = sum = 0; i < n1; i++)
		sum += one[i];
	for (t = 0, fraction = total = 0.0; t <= ss; t++) {
		total += m[ll][t][n1];
		if (t > sum)
			fraction += m[ll][t][n1];
		else if (t == sum)
			fraction += (0.5 * m[ll][t][n1]);
	}

	if (fabs(1.0 - total / bico((n1 + n2), n1)) > 0.01)
		fprintf(stderr,
			"WARNING: %5.3f%% round-off error in permutation_test_dp.\n",
			100.0 * fabs(1.0 - total / bico((n1 + n2), n1)));

	gettimeofday(&tp2, &tzp);
	time = (int) ((tp2.tv_sec - tp1.tv_sec) * 1e3L +
		      (tp2.tv_usec - tp1.tv_usec) / 1e3L);
	if (debugflag)
		fprintf(stderr, "%dms\n", time);

	for (i = 0; i < (ll + 1); i++) {
		for (j = 0; j < (ss + 1); j++) {
			free(m[i][j]);
		}
		free(m[i]);
	}
	free(m);
	free(join);

	ret_val = (float) fraction / (float) bico((n1 + n2), n1);
	if (side == -1)
		return (ret_val);
	else if (side == 1)
		return (1.0 - ret_val);
	else if (side == 0)
		return (2.0 * MY_MIN(ret_val, 1.0 - ret_val));
}


/*
** for reference:
**
**      xbar = sum(x) / N-1			mean
**     sigma = sqrt(sum((x - xbar)**2) / N-1)	std deviation
**  stderror = sigma / sqrt(N)			std deviation of mean (stderr)
**
** also, these are NOT the most efficent way to compute these
** values, however, the code is easily read and quite clear. Don't
** expect them to improve, since this shouldn't be a big time sink
**
** 97.4 bjarthur
**  xbar has N, not N-1, in the denominator.  sigma and stderror are
**  correct as stated, assuming you don't know the mean a priori.
**
*/

float mean(float *data, int n)
{
	int i;
	double sum;

	if (n <= 0)
		return (0.0);

	sum = 0.0;
	for (sum = 0.0, i = 0; i < n; i++) {
		sum += data[i];
	}
	sum /= (double) n;
	return ((float) sum);
}

float quantile(float *data, int n, float percent)
{
	int i, j;
	double m, fp, ip;
	float *d;

	if (n <= 0)
		return (0.0);

	d = (float *) malloc(sizeof(float) * n);
	for (i = 0; i < n; i++)
		d[i] = data[i];
	for (i = 0; i < (n - 1); i++) {
		for (j = i; j < n; j++) {
			if (d[i] > d[j]) {
				SWAP(d[i], d[j], float);
			}
	}}

	fp = modf(0.5 + ((double) n) * percent, &ip);
	ip -= 1.0;		/* c is zero-offset */
	if ((fp + ip) <= 0.0) {
		m = d[0];
	} else if ((fp + ip) >= (double) (n - 1)) {
		m = d[n - 1];
	} else {
		m = d[(int) ip] + fp * (d[(int) ip + 1] - d[(int) ip]);
	}

	free(d);
	return ((float) m);
}

float median(float *data, int n)
{
	return (quantile(data, n, 0.5));
}

float stddev(float *data, int n)
{
	int i;
	double m, SS;

	if (n <= 1)
		return (0.0);

	if (n <= 1)
		return ((float) 0.0);
	m = (double) mean(data, n);
	for (SS = 0.0, i = 0; i < n; i++)
		SS += (data[i] - m) * (data[i] - m);
	SS /= (double) (n - 1);
	SS = sqrt(SS);
	return ((float) SS);
}

float stderror(float *data, int n)
{
	int i;
	double m, SS;

	if (n <= 1)
		return (0.0);

	if (n <= 1)
		return ((float) 0.0);
	m = (float) mean(data, n);
	for (SS = 0.0, i = 0; i < n; i++)
		SS += (data[i] - m) * (data[i] - m);
	SS /= (double) (n - 1);
	SS = sqrt(SS) / sqrt((double) n);
	return ((float) SS);
}

float mode(float *data, int n, float bin_size, float step_size)
{
	int i, j, idx1, idx2, max_diff;
	float fi, max, ret_val, *sorted_data;

	/* sort data */
	assert((sorted_data =
		(float *) malloc(n * sizeof(float))) != NULL);
	for (i = 0; i < n; i++)
		sorted_data[i] = data[i];
	for (i = 0; i < n - 1; i++) {
		max = sorted_data[i];
		idx1 = i;
		for (j = i + 1; j < n; j++) {
			if (sorted_data[j] < max) {
				max = sorted_data[j];
				idx1 = j;
			}
		}
		if (idx1 != i) {
			sorted_data[idx1] = sorted_data[i];
			sorted_data[i] = max;
		}
	}
	/* end-sort data */

	idx1 = max_diff = 0;
	for (fi = sorted_data[0]; fi < sorted_data[n - 1]; fi += step_size) {
		while ((sorted_data[idx1] < fi) && (idx1 < n))
			idx1++;
		idx2 = idx1 + 1;
		while ((sorted_data[idx2] < (fi + bin_size)) && (idx2 < n))
			idx2++;
		if ((idx2 - idx1) > max_diff) {
			max_diff = (idx2 - idx1);
			ret_val = fi + 0.5 * bin_size;
		}
	}

	return (ret_val);
}

/* given the stats (mean, stddev, stderr, #) of 2 sets of numbers,
   the raw data points of which are not known, the following 3
   functions compute the stats of the combined set.  theoretically,
   they're great, but in practice they seem to suffer from some
   numerical instability.  i'd be careful about using them
   repeatably. */

void combine_mean(float u1, int n1, float u2, int n2, float *uT)
{
	(*uT) = (n1 * u1 + n2 * u2) / (n1 + n2);
}

void combine_stddev(float u1, float s1, int n1, float u2, float s2, int n2,
		    float *sT)
{
	float uT;

	uT = (n1 * u1 + n2 * u2) / (n1 + n2);
	(*sT) = sqrt(((n1 - 1) * s1 * s1 + n1 * (u1 - uT) * (u1 - uT) +
		      (n2 - 1) * s2 * s2 + n2 * (u2 - uT) * (u2 -
							     uT)) / (n1 +
								     n2 -
								     1));
}

void combine_stderr(float u1, float e1, int n1, float u2, float e2, int n2,
		    float *eT)
{
	float uT;
	float s1, s2, sT;

	s1 = e1 * sqrt(n1);
	s2 = e2 * sqrt(n2);

	uT = (n1 * u1 + n2 * u2) / (n1 + n2);
	sT = sqrt(((n1 - 1) * s1 * s1 + n1 * (u1 - uT) * (u1 - uT) +
		   (n2 - 1) * s2 * s2 + n2 * (u2 - uT) * (u2 - uT)) / (n1 +
								       n2 -
								       1));

	(*eT) = sT / sqrt(n1 + n2);
}

static float overlap(float al, float ar, float bl, float br)
{
	float ret_val = ar - al;

	if (br < ar)
		ret_val -= (MY_MIN((ar - br), (ar - al)));
	if (bl > al)
		ret_val -= (MY_MIN((bl - al), (ar - al)));
	return (ret_val);
}

/* given data pairs (xvals[n],yvals[n]) and spont, this function returns
the pair with the maximum yval as (maxxptr,maxyptr), computes the width
at parms.width_percent, and returns the left and right xval endpoints as
leftptr and rightptr.  the reference floor for width_percent is specified
by parms.max_min and parms.spont_max.  see doc/xdview.parms.doc for
details. */

int findPeak(int n, float *xvals, float *yvals, float spont,
	     float *maxxptr, float *maxyptr, float *minyptr,
	     float *leftptr, float *rightptr, float *sps)
{
	int i;
	int maxx, max2x;
	float maxy, max2y, max3y;
	float maxgraphy;
	float mingraphy;
	float dist_to_zero = 100000.0;
	float halfmax;
	float miny;
	float miny_left = 0.0;
	float miny_right = 0.0;
	float r, l;
	int floor, ceil;
	int midval_percent;
	int area;
	float stepl, stepr;


	if (n < 1)
		return (0);

	floor = lookupParms_int("parms.floor");
	ceil = lookupParms_int("parms.ceil");
	midval_percent = lookupParms_int("parms.width_percent");

#define YLAST yvals[i-1]
#define YCUR yvals[i]
#define YNEXT yvals[i+1]
#define XCUR xvals[i]

	/* find max and min points for the graph */
	maxgraphy = mingraphy = yvals[0];
	for (i = 1; i < n; i++) {
		if (YCUR > maxgraphy)
			maxgraphy = YCUR;
		if (YCUR < mingraphy)
			mingraphy = YCUR;
	}

	/* find maxx,maxy */
	switch (ceil) {
	case (0):		/* use absolute max */
		for (maxy = yvals[0], maxx = 0, i = 1; i < n; i++) {
			if (yvals[i] > maxy) {
				maxy = yvals[i];
				maxx = i;
			}
		}
		break;
	case (1):		/* find the largest local maximum; if there's a tie, choose the */
		/* one closest to zero */
		for (maxx = maxy = -1, i = 1; i < (n - 1); i++) {
			if ((YLAST <= YCUR) && (YNEXT <= YCUR)) {
				/* we have a peak */
				if ((YCUR > maxy)
				    || ((YCUR == maxy)
					&& (fabs(XCUR) < dist_to_zero))) {
					maxx = i;
					maxy = YCUR;
					dist_to_zero = fabs(XCUR);
				}
			}
		}
		if (maxx == -1)
			return (0);	/* didn't find a peak */
		break;
	case (2):		/* find the local maximum closest to zero */
		/* that's within MIN_TOLERANCE of maxgraphy */
		for (i = 1; i < (n - 1); i++) {
			if (((YLAST <= YCUR) && (YNEXT <= YCUR)) &&
			    ((maxgraphy - YCUR) < MIN_TOLERANCE)) {
				/* we have a peak */
				if (fabs(XCUR) < dist_to_zero) {
					maxx = i;
					maxy = YCUR;
					dist_to_zero = fabs(XCUR);
				}
			}
		}
		if (dist_to_zero == 100000.0)	/* didn't find a peak */
			return (0);
	}

#if(0)
	/* find x-value nearest to peakat */
	for (i = 0; i < n; i++) {
		x = abs(peakat - xvals[i]);
		if (i == 0) {
			d = x;
			maxx = i;
		} else if (x < d) {
			d = x;
			maxx = i;
		}
	}
#endif

	/* _if_ sps!=NULL, find 2nd highest local maximum */
	if (sps != NULL) {
		for (max2y = -1, i = 1; i < (n - 1); i++) {
			if (abs(i - maxx) < 2)
				continue;	/* for peaks with 2 pts of same height */
			if ((YLAST <= YCUR) && (YNEXT <= YCUR)
			    && (YCUR > max2y)) {
				max2y = YCUR;
				max2x = i;
			}
		}
		if (max2y == -1)
			max2y = maxy;
		for (max3y = -1, i = 1; i < (n - 1); i++) {
			if (abs(i - maxx) < 2)
				continue;	/* for peaks with 2 pts of same height */
			if (abs(i - max2x) < 2)
				continue;
			if ((YLAST <= YCUR) && (YNEXT <= YCUR)
			    && (YCUR > max3y)) {
				max3y = YCUR;
			}
		}
		if (max3y == -1)
			max3y = max2y;
	}

	for (miny = yvals[0], i = 1; i < n; i++) {
		if (yvals[i] < miny) {
			miny = yvals[i];
		}
	}
	(*minyptr) = miny;

	/* find miny */
	switch (floor) {
	case (0):		/* use 0 */
		miny = 0.0;
		break;
	case (1):		/* use spont */
		miny = spont;
		break;
	case (2):		/* use absolute min */
		break;
	case (3):
		/* use the average of the two local minumums nearest to the main lobe.
		   the minimums must be within MIN_TOLERANCE of mingraphy */

		/* first look at the minimum to the left */
		for (i = maxx; i > 0; i--) {
			if (((YLAST >= YCUR) && (YNEXT >= YCUR)) &&
			    ((YCUR - mingraphy) < MIN_TOLERANCE)) {
				miny_left = YCUR;
				break;
			}
		}

		/* If we didn't find a min, take mingraphy as min */
		if ((i == 1) && (miny_left == 0.0))
			miny_left = mingraphy;

		/* now look at the minimum to the right */
		for (i = maxx; i < (n - 1); i++) {
			if (((YLAST >= YCUR) && (YNEXT >= YCUR)) &&
			    ((YCUR - mingraphy) < MIN_TOLERANCE)) {
				miny_right = YCUR;
				break;
			}
		}

		/* If we didn't find a min, take mingraphy as min */
		if ((i == (n - 1)) && (miny_right == 0.0))
			miny_right = mingraphy;

		/* Take the average of the two vals as min to use for midval calc */
		miny = (miny_right + miny_left) / 2.0;
	}

#undef YLAST
#undef YCUR
#undef YNEXT
#undef XCUR

	if (maxx == 0 || maxx == (n - 1))
		return (0);


	area = lookupParms_int("parms.area");

	if (!area) {
		halfmax =
		    (maxy - miny) * ((float) midval_percent / 100.0) +
		    miny;
#define lin_int(x1,y1,x2,y2,y) ((y-y1)*(x2-x1)/(y2-y1)+x1)
		for (i = maxx - 1; i >= 0; i--) {
			if ((halfmax >= yvals[i])
			    && (halfmax <= yvals[i + 1])) {
				l = lin_int(xvals[i], yvals[i],
					    xvals[i + 1], yvals[i + 1],
					    halfmax);
				break;
			}
		}
		if (i < 0)
			l = MAXFLOAT;

		for (i = maxx + 1; i < n; i++) {
			if (halfmax >= yvals[i] && halfmax <= yvals[i - 1]) {
				r = lin_int(xvals[i], yvals[i],
					    xvals[i - 1], yvals[i - 1],
					    halfmax);
				break;
			}
		}
		if (i == n)
			r = MAXFLOAT;
	}
#undef lin_int
	else {
		l = r = 0.0;
		for (i = 0; i < n; i++) {
			stepl = (xvals[i] - xvals[i - 1]) / 2.0;
			stepr = (xvals[i + 1] - xvals[i]) / 2.0;
			if (i == 0)
				stepl = stepr;
			if (i == (n - 1))
				stepr = stepl;
			if ((yvals[i] - spont) <= 0)
				continue;
			l += ((yvals[i] - spont) * overlap(xvals[i] -
							   stepl,
							   xvals[i] +
							   stepr,
							   -MAXFLOAT,
							   xvals[maxx]));
			r += ((yvals[i] - spont) * overlap(xvals[i] -
							   stepl,
							   xvals[i] +
							   stepr,
							   xvals[maxx],
							   MAXFLOAT));
		}
	}

	if (maxxptr)
		*maxxptr = xvals[maxx];
	if (maxyptr)
		*maxyptr = yvals[maxx];
	if (leftptr)
		*leftptr = l;
	if (rightptr)
		*rightptr = r;
	if (sps)
		*sps =
		    1.0 - (((max2y + max3y) / 2.0) - miny) / (maxy - miny);

	return (1);
}
