/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: ss_user.c,v 2.52 2002/07/15 04:30:22 cmalek Exp $
** Program: dowl
**  Module: ss_user.c
**  Author: mazer
** Descrip: dowl based user control panel for ss package
**
** Revision History (most recent last)
**
** Tue Dec  1 15:20:40 1992 mazer
**  creation date
**
*******************************************************************/

#include "xdphyslib.h"

static Field ws_ss_user[] = {
	{WS_VSEP, "Wave store"},
	{"ss_user.maxwaves", "%d", 2, "5"},
	{"ss_user.decimate (by)", "%d", 2, "1"},
	{"ss_user.pre_ms (ms)", "%f", 4, "0.5"},
	{"ss_user.post_ms (ms)", "%f", 4, "3.0"},
	{"ss_user.rmsthresh", "%f", 4, "3.5"},
	{WS_VSEP, "Inference"},
	{"ss_user.nmodels", "%d", 2, "5"},
	{"ss_user.max_reestimations", "%d", 2, "3"},
	{"ss_user.kdlevel (12)", "%d", 2, "1"},
	{"ss_user.searchlevel (0=T)", "%d", 2, "1"},
	{"ss_user.seplog (01)", "%b", 1, "0"},
	{WS_VSEP, "Debug"},
	{"ss_user.verbose", "%d", 2, "1"},
	{NULL},
};

SsPanel *global_sspanel = NULL;

static void UpdateSSVarsCB(Widget, XtPointer, XtPointer);
static void SetOtherSSVarCB(Widget, XtPointer, XtPointer);
static void DestroyCB(Widget, XtPointer, XtPointer);
static void SaveCB(Widget, XtPointer, XtPointer);
static void LoadCB(Widget, XtPointer, XtPointer);

static void UpdateSSVarsCB(Widget w, XtPointer client_data,
			   XtPointer call_data)
{
	char buf[1000];		/* hardcoding's always best.. */

	ws_tosvars(NULL, NULL);	/* unload screen data into svars */

	/* threshold in "RMS-units" for first pass at collecting events */
	sprintf(buf, "event_RMS_threshold=%s", GS("ss_user.rmsthresh"));
	ss_SetGlobalVar(buf);

	/* # of inital spike models */
	sprintf(buf, "default_nspike_models=%s", GS("ss_user.nmodels"));
	ss_SetGlobalVar(buf);

	/* # interations */
	sprintf(buf, "max_reestimations=%s",
		GS("ss_user.max_reestimations"));
	ss_SetGlobalVar(buf);

	/* number of exact overlaps incorporated into kd-tree --> higher is slower */
	sprintf(buf, "kdlevel=%s", GS("ss_user.kdlevel"));
	ss_SetGlobalVar(buf);

	/* use a fixed level to detect events (0) or classify at point (1) */
	sprintf(buf, "event_search_level=%s", GS("ss_user.searchlevel"));
	ss_SetGlobalVar(buf);

	/* window around (pre+post) an event to use for analysis */
	sprintf(buf, "pre_ms=%s", GS("ss_user.pre_ms"));
	ss_SetGlobalVar(buf);
	sprintf(buf, "post_ms=%s", GS("ss_user.post_ms"));
	ss_SetGlobalVar(buf);

	/* window around (pre+post) an event to use for analysis */
	sprintf(buf, "verbose=%s", GS("ss_user.verbose"));
	ss_SetGlobalVar(buf);
}

static void SetOtherSSVarCB(Widget w, XtPointer client_data,
			    XtPointer call_data)
{
	char *p;

	if ((p = pop_dialog("ss_SetGlobal: ss_varname=value", "")) == NULL) {
		return;
	}
	if (ss_SetGlobalVar(p) == 0)
		notify("Failed to set %s", p);
	free(p);
}

char *ss_user_genfilename(char *buf, char *type)
{
	if (type)
		sprintf(buf, "%s.%s.%s", GS("animal"), GS("unit"), type);
	else
		sprintf(buf, "%s.%s", GS("animal"), GS("unit"));
	return (buf);
}

void ss_user_initialize(void)
{
	int fc;			/* in hz */
	xword *inbuf;
	int insize;

	inbuf = is_getADbuffer(&insize);

	if (is_adFc <= 0 && insize > 0) {
		/* svar adFc is in kHz ... */
		fc = GI("adFc");
		notify("Guessing at %dhz", fc);
	} else {
		/* is_adFc is in Hz ... */
		fc = is_adFc;
	}

	if (GI("ss_user.decimate") > 0) {
		fc /= GI("ss_user.decimate");
	}

	if (global_sspanel != NULL) {
		notify("global_sspanel already exists!");
		return;
	}

	UpdateSSVarsCB(NULL, NULL, NULL);

	global_sspanel =
	    ssp_Build(TopLevel, True,
		      NULL, NULL, NULL, NULL, GI("ss_user.maxwaves"),
		      300, 300, -FRANGE, fc);
	ssp_Popup(global_sspanel);
}

static void DestroyCB(Widget w, XtPointer client_data, XtPointer call_data)
{
	if (global_sspanel != NULL) {
		ssp_Destroy(global_sspanel);
		global_sspanel = NULL;
	}
}

static void SaveCB(Widget w, XtPointer for_class_blah, XtPointer call_data)
{
	int for_class = (int) for_class_blah;

	char buf[MAXPATHLEN], *ssfile;

	if (global_sspanel == NULL) {
		notify("Nothing to save");
		return;
	}
	ss_user_genfilename(buf, for_class ? "ss" : "ss+");
	if ((ssfile = fileBox(buf, "w", "Save to ...")) != NULL) {
		if (for_class) {
			if (ss_WriteStateForClassification(ssfile,
							   ssp_SpikeSet
							   (global_sspanel))
			    == 0) {
				alert
				    ("ss_WriteStateForClassification failed");
			}
		} else {
			if (ss_WriteState(ssfile,
					  ssp_CircularDataBuffer
					  (global_sspanel),
					  ssp_SpikeSet(global_sspanel),
					  ssp_EventList(global_sspanel),
					  ssp_EventTable(global_sspanel))
			    == 0) {
				alert("ss_WriteState failed");
			}
		}
	}
}

int ss_user_loadfile(char *ssfile)
{
	int fc, ssplus;

	SpikeSet *ss = NULL;
	CircularDataBuffer *cdb = NULL;
	EventList *event_list = NULL;
	EventTable *event_table = NULL;

	if (ss_ReadState(ssfile, &cdb, &ss, &event_list, &event_table, &fc)
	    != 0) {
		ssplus = 1;
	} else if (ss_ReadStateForClassification(ssfile, &ss, &fc) != 0) {
		ssplus = 0;
	} else {
		return (-1);
	}

	global_sspanel = ssp_Build(TopLevel, True,
				   ss, event_list, event_table,
				   cdb,
				   cdb ==
				   NULL ? GI("ss_user.maxwaves") : 0, 300,
				   300, -FRANGE, fc);
	return (fc);
}


static void LoadCB(Widget w, XtPointer client_data, XtPointer call_data)
{
	char buf[MAXPATHLEN], *ssfile;
	int fc;

	if (global_sspanel != NULL) {
		notify("Sorry, won't load over existing sspanel");
		return;
	}

	ss_user_genfilename(buf, "ss");
	if ((ssfile = fileBox(buf, "r", "Load (ss or ss+) ...")) != NULL) {
		if ((fc = ss_user_loadfile(ssfile)) < 0) {
			alert("Can't load: \n%s", ssfile);
		}

		if (GI("ss_user.decimate") > 0) {
			fc *= GI("ss_user.decimate");
		}

		if (fc != is_adFc) {
			if (is_inited)
				notify
				    ("Warning:\nss was at %dhz, now at %dhz",
				     fc, is_adFc);
			else {
				notify
				    ("Warning:\nDACQ uninitialized\b(fc should be set to %dhz)",
				     fc);
			}
		}
		free(ssfile);
	}
}

Widget ss_user_newpanel(Widget parent, int popit)
{
	Widget form, psh, ws, w;

	static MENU_ENTRY FileMenu[] = {
		{"File Menu", NULL},
		{"Save ss for class", SaveCB, (void *) 1},
		{"Save ss+ (big)", SaveCB, (void *) 0},
		{WS_VSEP},
		{"Load ss", LoadCB, (void *) NULL},
		{NULL}
	};

	if (parent == NULL) {
		psh = top_new(TopLevel, "ss_user");
		form = form_new(psh, "form", 0);
	} else {
		psh = NULL;
		form = form_new(parent, "form", 0);
	}

	w = NULL;
	w = menu_new(form, "File", FileMenu, w, NULL);
	w = button_new(form, "Kill", "Kill", DestroyCB, NULL, w, NULL);
	w = button_new(form, "ws->ss", "ws->ss", UpdateSSVarsCB, NULL, w,
		       NULL);
	w = button_new(form, "??->ss", "??->ss", SetOtherSSVarCB, NULL, w,
		       NULL);

	ws = ws_new(form, ws_ss_user, NULL, NULL, NULL);
	XtVaSetValues(ws, XtNfromVert, w, XtNfromHoriz, NULL, NULL);

	if (psh != NULL) {
		if (popit)
			popup(psh);
		return (psh);
	} else {
		return (form);
	}
}
