/******************************************************************
** Program: xdphys
** File:    runscan.c
** Author:  cmalek@caltech.edu (Chris Malek)
** RCSid: $Id: runscan.c,v 2.3 2003/09/10 22:34:04 bja Exp $
** Description: 
**
** This file contains appRun() and its attendant functions.  appRun
** is the function that actually performs the experiment, and writes
** all data to the datafile.
**
*******************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include "xdphyslib.h"
#include "xdphys.h"
#include "runscan.h"
#include "autolog.h"

#ifndef PATH_MAX
/* PATH_MAX should be defined by the OS, but if it isn't (e.g. in SunOS 4.x)
   define it here */
#define PATH_MAX 255
#endif /* PATH_MAX */

/* ---------------------------------------------------------------------- */
/* Prototypes                                                             */
/* ---------------------------------------------------------------------- */

/* ---------------------------------------------------------------------- */
/* Global variables                                                       */
/* ---------------------------------------------------------------------- */

Field StandardParams[] = {
	{"file.version", "%s", 60},
	{"ts.fix_Stim", "%s", 60},
	{"ts.fix_bc (%)", "%d", 3},
	{"ts.fix_iid (dB)", "%d", 3},
	{"ts.fix_itd (us)", "%d", 4},
	{"ts.fix_abi (dB)", "%d", 3},
	{"ts.use_dur_delay", "%b", 1},
	{"ts.dur (ms)", "%d", 4},
	{"ts.delay (ms)", "%d", 4},
	{"Animal", "%s", 3},
	{"Unit", "%s", 2},
	{"Pass (#)", "%d", 5},
	{"AP (um)", "%f", 5},
	{"ML (um)", "%f", 5},
	{"DV (um)", "%f", 5},
	{"Side", "%s", 1},
	{"Reps (#)", "%d", 2},
	{"Dur (ms)", "%d", 4},
	{"Delay (ms)", "%d", 4},
	{"Epoch (ms)", "%d", 4},
	{"ISI (ms)", "%d", 4},
	{"Rise (ms)", "%d", 2},
	{"Fall (ms)", "%d", 2},
	{"Spont_Stims", "%b", 1},
	{"daFc (Hz)", "%f", 6},
	{"adFc (Hz)", "%f", 6},
	{"evtFc (Hz)", "%f", 6},
	{"randomize", "%d", 1},
	{"version", "%s", 60},
	{"time", "%s", 60},
	{"timestamp", "%d", 12},	/* new with version 2.24 */
	{"noise.low", "%d", 12},	/* recorded with version 2.26 */
	{"noise.high", "%d", 12},	/* recorded with version 2.26 */
	{"tone.rad_vary", "%d", 12},
	{"stack.rms", "%b", 1},
	{"xdphys.caltype", "%s", 60},
	{"detect.mode", "%s", 10},
	{"xdphys.gsrnmodels", "%d", 2},
	{"xdphys.fname", "%s", 32},
	{"ana.channel1 (on,off)", "%s", 3},
	{"ana.channel2 (on,off)", "%s", 3},
	{"detect.discrim_channel", "%d", 1},
	{"ana.nchans", "%d", 1},
	{"ana-decimate (by)", "%d", 3},
	{"ana.every (nth)", "%d", 2},
	{"stim.save", "%b", 3, "0"},
	{"stim.decimate (by)", "%d", 5, "1"},
	{"stim.every (nth)", "%d", 2, "1"},
	{"prefs.page", "%d", 1},
	{"prefs.page1", "%s", 15, "1"},
	{"prefs.page2", "%s", 15, "2"},
	{"prefs.page3", "%s", 15, "3"},
	{"prefs.page4", "%s", 15, "4"},
	{"prefs.page5", "%s", 15, "5"},
	{"prefs.page6", "%s", 15, "6"},
#ifdef __tdtproc__
	{"tg6extclock", "%b", 0},
	{"tg6extclock_fc", "%f", 6},
	{"use_led (ms)", "%b", 1},
	{"led_dur (ms)", "%d", 4},
	{"led_delay", "%d", 4},
#endif				/* __tdtproc__ */
	{NULL},
};


/* ---------------------------------------------------------------------- */
/* Functions                                                              */
/* ---------------------------------------------------------------------- */
#define TEMPFILE_STUB "/temp."
static char *make_temp_filename(char *suffix) 
{
	char *final_datafile = NULL;
	char *tempstr;

	assert((final_datafile = (char *) calloc(PATH_MAX, sizeof(char))) != NULL);

	tempstr = getcwd(NULL, (PATH_MAX-strlen(TEMPFILE_STUB)-strlen(suffix)));
	if (tempstr == NULL) {
		perror("xdphys(runscan.c)");
	} else {
		strcpy(final_datafile, tempstr);

		strcat(final_datafile, TEMPFILE_STUB);
		strcat(final_datafile, suffix);

		free(tempstr);
	}
	return(final_datafile);
}
#undef TEMPFILE_STUB

/* ------------------------------------------------------------------ */
static char *make_named_temp_filename(char *suffix) 
{
	char *final_datafile = NULL;
	char buf[PATH_MAX];

	sprintf(buf, "%d.%d.0.%s", GI("Pass"), GI("DV"), suffix);
	final_datafile = fileBox(buf, "w", "Save to ...");

	return(final_datafile);
}

/* ------------------------------------------------------------------ */
static char *make_normal_run_filename(char *suffix) 
{
	char *final_datafile = NULL;
	char tempstr[PATH_MAX];
	
	sprintf(tempstr, "%s.%s.%d.%s", GS("animal"),
		next_alphanum_counter(GS("unit"), 0), GI("filenum"), suffix);
	final_datafile = fileBox(tempstr, "w", "Save to ...");

	return(final_datafile);
}

/* ------------------------------------------------------------------ */
static char *make_datafile_name(int run_type, char *suffix) 
{
	char *final_datafile = NULL;

	switch(run_type) {
		case TEMP_YES:
			final_datafile = make_temp_filename(suffix);
			break;
		case TEMP_NAMED:
			final_datafile = make_named_temp_filename(suffix);
			break;
		default:	
			final_datafile = make_normal_run_filename(suffix);
			break;
	}

	return(final_datafile);
}

/* ------------------------------------------------------------------ */
static char *possibly_use_ramdisk(char *filename) 
{
	char *datafile;

	if (GI("detect.use_ramdisk") && GI("detect.ramdisk_mounted")) {
		datafile = "/mnt/xdphys_disk/xdphys_file.tmp";
	} else {
		datafile = filename;
	}

	return(datafile);
}

/* ------------------------------------------------------------------ */
static void possibly_copy_file_from_ramdisk(char *working_filename, 
	char *final_filename) 
{
	char buf[PATH_MAX];

	if (GI("detect.use_ramdisk") && GI("detect.ramdisk_mounted")) {
		sprintf(buf, "exec /bin/cp %s %s", working_filename,
			final_filename);
		system(buf);
		unlink(working_filename);
	}
}

/* ------------------------------------------------------------------ */
static void write_begin_raster(FILE * fp)
{
	fprintf(fp, "RASTERDATA\n");
}

/* ------------------------------------------------------------------ */
static void write_end_raster(FILE * fp)
{
	fprintf(fp, "END_RASTERDATA\n");
}

/* ------------------------------------------------------------------ */
static void write_file_header(FILE * fp, int run_type, char *filename, 
	RunData * info)
{
	float tomv, offset;
	int i;
	char tempstr[32];
	char *comments;
#ifdef __tdtproc__
	int epoch = GI("epoch");
#endif				/* __tdtproc__ */

	SS("xdphys.fname", filename);
	SI("ana.nchans", spiker_getAnalogNchans());
	sprintf(tempstr, "%s", FILE_FORMAT_VERSION);
	SS("file.version", tempstr);

#ifdef __tdtproc__
	/* This is a hack! We do this because the DD1 may choose a slightly
	   different sampling rate than what we tell it to use */

	is_init(GF("daFc"), 0.0, 0.0, epoch, epoch);
	ws_tows(NULL, NULL);
	is_shutdown();
#endif				/* __tdtproc__ */

	comments = possibly_get_comments(run_type, filename);

	fprintf(fp, ";; xdphys %s ver %s\n", info->suffix,
		XDPHYS_VERSION);

	/* Comment Block */
	fprintf(fp, "COMMENTS\n");
	if (debugflag) {
		fprintf(stderr, "comments=<%s>\n", comments ? comments : "NULL");
	}
	if (GI("Comments"))
		writeComments(comments, fp);
	fprintf(fp, "END_COMMENTS\n");

	/* Parameter Block */
	fprintf(fp, "PARAMS\n");
	ws_write(fp, info->params, info->nparams, NULL);
	if (info->depvar) {
		fprintf(fp, "depvar=%s\n", info->depvar);
	}
	fprintf(fp, "; StandardParams\n");
	ws_write(fp, StandardParams, WS_N(StandardParams), NULL);
	fprintf(fp,"use_cache=%d\n",GI("use_cache"));
	for (i = 0; i < is_getADnchans(); i++) {
		sprintf(tempstr, "ana.channel%d", i + 1);
		if (strncmp(GS(tempstr), "on", 2) == 0) {
			get_analog_conversion(&tomv, &offset, i);
			fprintf(fp, "ana-tomv.chan%d=%f\n", i + 1,
				tomv);
			fprintf(fp, "ana-offset.chan%d=%f\n", i + 1,
				offset);
		}
	}

	fprintf(fp, "END_PARAMS\n");
}

/* ------------------------------------------------------------------ */
static void set_recording_icon_state(void)
{
	int i;
	char buf[15];

	recording(0);

	for (i = 0; i < is_getADnchans(); i++) {
		sprintf(buf, "ana.channel%d", i + 1);
		if (strncmp(GS(buf), "on", 2) == 0) {
			recording(1);
			break;
		}
	}

}

/* ------------------------------------------------------------------ */
static char *process_successful_run(int run_type, char *filename,
	char *suffix)
{
	if (run_type == TEMP_NO) {
		/* only update filenum if it's NOT a run_type 12-mar-94 */
		SI("filenum", GI("filenum") + 1);
		ws_tows(NULL, NULL);	
	}

	append_to_log(run_type, filename, suffix, "COMPLETED");

	return(filename);
}

/* ------------------------------------------------------------------ */
static char *process_aborted_run(int run_type, char *filename,
	char *suffix)
{
	int choice;

	choice = pop_box("Aborted, what do to with partial data file?",
			"Delete", "Save", "Save and Plot");

	switch (choice) {
		case 1:
			unlink(filename);
			append_to_log(run_type, filename, suffix, "DELETED");
			break;
		case 3:
			if (run_type == TEMP_NO) {
				SI("filenum", GI("filenum") + 1);
				ws_tows(NULL, NULL);	/* update the filenum field */
			}

			append_to_log(run_type, filename, suffix, "ABORTED & PRESERVED");
			return(filename);
			break;
		case 2:
			if (run_type == TEMP_NO) {
				SI("filenum", GI("filenum") + 1);
				ws_tows(NULL, NULL);	/* update the filenum field */
			}
			append_to_log(run_type, filename, suffix, "ABORTED & PRESERVED");
			return(NULL);
			break;
	}
}

/* ------------------------------------------------------------------ */
static char *process_runscan_result(int runscan_result, int run_type, 
	char *filename, char *suffix)
{

	if (runscan_result > 0) {
		return(process_successful_run(run_type, filename, suffix));
	} else {
		if (runscan_result == 0 && (run_type != TEMP_YES)) {
			return(process_aborted_run(run_type, filename, suffix));
		} else {
			unlink(filename);
			
		}
	}
	return(NULL);
}

/* ------------------------------------------------------------------ */
static void set_hardware_flags(void)
{
#ifdef __tdtproc__
	if (GI("tg6extclock")) {
		is_tg6_extclock_enable(GI("tg6extclock_fc"));
	} else {
		is_tg6_extclock_disable();
	}
	if (getRaster_getmode() == GSR_ttl)
		is_et1_enable();
	else
		is_et1_disable();

	if (GI("use_led")) {
		is_led_enable();
		is_setLedDur(GI("led_dur")); is_setLedDelay(GI("led_delay"));
	} else
		is_led_disable();

#endif				/* __tdtproc__ */
}

/* ------------------------------------------------------------------ */
static void possibly_compress_file(char *filename)
{
	if (GI("Compress")) {
		char buf[PATH_MAX];
		sprintf(buf, "exec gzip %s", filename);
		system(buf);
	}
}

/* ================================================================== */
/* appRun                                                             */
/* ================================================================== */
char *appRun(int run_type, RunData * info, Widget status)
{
	char *final_datafile = NULL;
	char *working_datafile;
	char *file_to_graph = NULL;
	int scan_result;
	FILE *fp;

	final_datafile = make_datafile_name(run_type, info->suffix);
	if (final_datafile == NULL) {
		alert("Can't determine datafile name, try again");
		return(NULL);
	}
	working_datafile = possibly_use_ramdisk(final_datafile);

	if ((fp = fopen2(findhome(working_datafile), "w")) == NULL) {
		alert("Can't write datafile, try again");
		return (NULL);
	}

	set_recording_icon_state();
	tracer_clear(main_tracer);	

	write_file_header(fp, run_type, working_datafile, info);
	write_begin_raster(fp);

	set_hardware_flags();

	this_run = working_datafile;
	idle_set(dloop_empty);
	scan_result = (*info->scan_fn) (fp, status);
	this_run = NULL;

	write_end_raster(fp);

	if (scan_result <= 0) {
		fprintf(fp, ";; NOTE: run aborted or failed!!\n");
	}
	fclose(fp);

	possibly_copy_file_from_ramdisk(working_datafile, final_datafile);

	ws_tosvars(NULL, NULL);	/* in case of any changes during run.. */

	file_to_graph = process_runscan_result(scan_result, run_type, 
		final_datafile, info->suffix);

	possibly_compress_file(final_datafile);

	return (file_to_graph);
}
