/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: pm_txt.c,v 1.19 2002/07/15 04:30:19 cmalek Exp $
** Program: dowl
**  Module: pm_txt.c
**  Author: bjarthur
** Descrip: xdowl plot method -- plot plain txt files with just cols of data
**
** Revision History (most recent last)
**
** Sun Mar 14 20:37:11 1993 mazer
**  changed to accept the new FILEDATA structure
**
** 97.8 bjarthur
**  creation date.  used to overlay (x,y,e) triple data on xdphys files.
**
*******************************************************************/

#include "xdphyslib.h"
#include "xdphys.h"
#include "plotter.h"

static float *copyVec(float *, int);
static void txt_curveFN(FDO *, FILEDATA *, int, FILE *);
static int txt_plotter(FDO *, FILEDATA *, FDObj_ViewType *, int, FILE *);
static int txt_valid_view(FDO *, int);
static int txt_reader(FILE *, FILEDATA *, int);
static int txt_free(FILEDATA *);
static void fit_menu(FDO *);

void txt_peak_stats(Widget, XtPointer, XtPointer);
/*
void txt_linearity_stats(FDO*, char*,FILE*);
static void txt_LinearityStatsFN(Widget, XtPointer, XtPointer);
*/

static int view_order[][1] = { {PM_CURVE} };
static int nsubviews[] = { 1 };
static int nviews = 1;

typedef struct {
	int n;
	float *x, *y, *e;
} SumData;

#define SUMDATA(data) ((SumData*)(data->sumdata))

int txt_do_plot(FDO * fdo, FILEDATA * fd, int view, int l, FILE * fptr)
{
	if (!txt_valid_view(fdo, view)) {
		pm_type_error("txt", view);
		view = PM_DEFAULT;
		(void) txt_valid_view(fdo, view);
	}

	switch (view) {
	case PM_DEFAULT:
	case PM_CURVE:
		fit_menu(fdo);
		txt_curveFN(fdo, fd, l, fptr);
		break;
	default:
		pm_type_error("txt", view);
		break;
	}

	return (1);
}

static int txt_valid_view(FDO * fdo, int view)
{
	int retval = 0;
	int i, j;

	if (view == PM_DEFAULT) {
		retval = 1;
	} else {
		for (i = 0; i < nviews; i++) {
			for (j = 0; j < nsubviews[i]; j++) {
				if (view_order[i][j] == view) {
					retval = 1;
					fdo->view.lr = i;
					fdo->view.ud = j;
					break;
				}
			}
		}
	}

	return (retval);
}

static int txt_plotter(FDO * fdo, FILEDATA * fd, FDObj_ViewType * view,
		       int l, FILE * fptr)
{
	adjust_index(view->lr, view->ud);

	txt_do_plot(fdo, fd, view_order[view->lr][view->ud], l, fptr);

	return (1);
}

static float *copyVec(float *in, int n)
{
	float *out;

	assert((out = (float *) malloc((n + 1) * sizeof(float))) != NULL);
	memcpy(out + 1, in, n * sizeof(float));
	return (out);
}

static void txt_curveFN(FDO * fdo, FILEDATA * fd, int l, FILE * fptr)
{
	char buf[100], tbuf[100];
	float *px, *py, *pz, *sx, *sy, *sz;
	int pn, sn;
	char *spont;
	float *foox, *fooy, *fooe;
	int i;
	char *ylabel;
	float f1;

	if (fdo->normalize == 1) {
		ylabel = "% max response";
		f1 = SUMDATA(fd)->y[0];
		for (i = 1; i < SUMDATA(fd)->n; i++) {
			if (SUMDATA(fd)->y[i] > f1)
				f1 = SUMDATA(fd)->y[i];
		}
		f1 /= 100.0;
	} else {
		ylabel = " ";
		f1 = 1.0;
	}

	spont = FD_GV(fd, "spont");
	if (spont == NULL)
		FD_plotter_copy(SUMDATA(fd)->x, SUMDATA(fd)->y,
				SUMDATA(fd)->e, SUMDATA(fd)->n, 0, &px,
				&py, &pz, &pn, NULL, NULL, NULL, NULL);
	else {
		foox = copyVec(SUMDATA(fd)->x, SUMDATA(fd)->n);
		foox[0] = (float) SPONT;
		fooy = copyVec(SUMDATA(fd)->y, SUMDATA(fd)->n);
		sscanf(spont, "%f %*f", fooy + 0);
		if (SUMDATA(fd)->e != NULL) {
			fooe = copyVec(SUMDATA(fd)->e, SUMDATA(fd)->n);
			sscanf(spont, "%*f %f", fooe + 0);
		} else
			fooe = NULL;
		FD_plotter_copy(foox, fooy, fooe, 1 + SUMDATA(fd)->n, 1,
				&px, &py, &pz, &pn, &sx, &sy, &sz, &sn);
		free(foox);
		free(fooy);
		if (SUMDATA(fd)->e != NULL)
			free(fooe);
	}

	for (i = 0; i < pn; i++)
		py[i] /= f1;

	if (fptr != NULL) {
		if (fdo->no_text_header != 1) {
			fprintf(fptr, "COMMENTS\n");
			fprintf(fptr, "%%%s, curve view\n",
				strrchr(fd->filename,
					'/') ==
				NULL ? fd->filename : 1 +
				(char *) strrchr(fd->filename, '/'));
			fprintf(fptr, "%%col 1: x[i]\n");
			fprintf(fptr, "%%col 2: y[i]\n");
			if (!(fdo->no_errs))
				fprintf(fptr, "%%col 3: e[i]\n");
			fprintf(fptr, "END_COMMENTS\n");
			fprintf(fptr, "PARAMS\n");
			if (spont != NULL)
				fprintf(fptr, "spont=%s\n", spont);
			fprintf(fptr, "END_PARAMS\n");
			fprintf(fptr, "RASTERDATA\n");
			fprintf(fptr, "nlines=%d\n", SUMDATA(fd)->n);
		}
		if (fdo->no_errs) {
			for (i = 0; i < SUMDATA(fd)->n; i++)
				fprintf(fptr, "%e\t%e\n",
					SUMDATA(fd)->x[i],
					SUMDATA(fd)->y[i]);
		} else {
			for (i = 0; i < SUMDATA(fd)->n; i++)
				fprintf(fptr, "%e\t%e\t%e\n",
					SUMDATA(fd)->x[i],
					SUMDATA(fd)->y[i],
					SUMDATA(fd)->e[i]);
		}
		if (fdo->no_text_header != 1) {
			fprintf(fptr, "END_RASTERDATA\n");
		}
	}
	if (fdo->no_X)
		return;

	sprintf(buf, "%s", shortname(fdo->fds[l]->filename));

	FDObj_Add_Data_All(fdo, px, py, pn);
	FDObj_AddLine(fdo, l, px, py, pz, pn, AtFloat,
		      atQuadLinePlotWidgetClass, AtTypeLINEPOINTS,
		      AtMarkCIRCLE, ConvertColor(fdo->graph,
						 FDObj_Colors[l %
							      FDObj_numColors]),
		      (XtArgVal) buf);
	if (spont != NULL) {
		FDObj_Add_Data_All(fdo, sx, sy, sn);
		FDObj_AddLine(fdo, l, sx, sy, sz, sn, AtFloat,
			      atQuadLinePlotWidgetClass, AtTypeLINEPOINTS,
			      AtMarkCIRCLE, ConvertColor(fdo->graph,
							 FDObj_Colors[l %
								      FDObj_numColors]),
			      (XtArgVal) buf);
	}
	sprintf(tbuf, "@b(%s)",
		fdo->fds[l] ? shortname(fdo->fds[l]->filename) : "???");

	XtVaSetValues(fdo->xaxis, XtNlabel, " ", NULL);
	XtVaSetValues(fdo->yaxis, XtNlabel, ylabel, NULL);
	XtVaSetValues(fdo->graph, XtNtitle, tbuf, XtNshowTitle, True,
		      NULL);
}

static void fit_menu(FDO * fdobj)
{
	if (fdobj->no_X)
		return;

	menubutton_clear(fdobj->fit, &(fdobj->fitpsh));
	XtSetSensitive(fdobj->fit, False);

	menubutton_add(fdobj->fitpsh, "peak stats", txt_peak_stats, fdobj);
	menubutton_add(fdobj->fitpsh, "similarity stats",
		       FD1_similarity_stats, fdobj);
/*menubutton_add(fdobj->fitpsh,"linearity stats", txt_LinearityStatsFN,fdobj);*/

	XtSetSensitive(fdobj->fit, True);
}

static int txt_reader(FILE * fp, FILEDATA * data, int ana)
{
	int nl, act_nl;
	char *p, q[512];
	int err;

	if ((p = skipTo(fp, "nlines=")) == NULL ||
	    sscanf(p, "nlines=%d", &nl) != 1) {
		fclose(fp);
		FD_free(data);
		return (0);
	}

	assert((data->sumdata = malloc(sizeof(SumData))) != NULL);
	assert((SUMDATA(data)->x =
		(float *) malloc(nl * sizeof(float))) != NULL);
	assert((SUMDATA(data)->y =
		(float *) malloc(nl * sizeof(float))) != NULL);
	assert((SUMDATA(data)->e =
		(float *) malloc(nl * sizeof(float))) != NULL);

	act_nl = 0;
	while (!feof(fp)) {
		fgets(q, 512, fp);
		if (!strncmp(q, "END_RASTERDATA", 14))
			break;
		err = sscanf(q, "%f %f %f", &(SUMDATA(data)->x[act_nl]),
			     &(SUMDATA(data)->y[act_nl]),
			     &(SUMDATA(data)->e[act_nl]));
		act_nl++;
	}

	if (act_nl != nl) {
		assert((SUMDATA(data)->x =
			(float *) realloc(SUMDATA(data)->x,
					  act_nl * sizeof(float))) !=
		       NULL);
		assert((SUMDATA(data)->y =
			(float *) realloc(SUMDATA(data)->y,
					  act_nl * sizeof(float))) !=
		       NULL);
		assert((SUMDATA(data)->e =
			(float *) realloc(SUMDATA(data)->e,
					  act_nl * sizeof(float))) !=
		       NULL);
	}

	if (err != 3) {
		free(SUMDATA(data)->e);
		SUMDATA(data)->e = NULL;
	}

	SUMDATA(data)->n = act_nl;
	data->channels = NULL;

	return (1);
}

static int txt_free(FILEDATA * data)
{
	if (SUMDATA(data)) {
		FREE(SUMDATA(data)->x);
		FREE(SUMDATA(data)->y);
		FREE(SUMDATA(data)->e);
	}
	FREE(SUMDATA(data));

	return (1);
}

int pm_txt_init(void)
{
	setFDOvalidviewMethod("txt", txt_valid_view);
	setFDOdoplotMethod("txt", txt_do_plot);
	setFDOplotMethod("txt", txt_plotter);
	setFDreadMethod("txt", txt_reader);
	setFDfreeMethod("txt", txt_free);
	return (1);
}

void txt_peak_stats(Widget w, XtPointer fdobjptr, XtPointer call_data)
{
	FDO *fdobj = (FDO *) fdobjptr;
	int i, *ndata;
	float **x, **y, *spont;
	char **file;

	assert((ndata =
		(int *) malloc(fdobj->nfds * sizeof(int))) != NULL);
	assert((x =
		(float **) malloc(fdobj->nfds * sizeof(float *))) != NULL);
	assert((y =
		(float **) malloc(fdobj->nfds * sizeof(float *))) != NULL);
	assert((spont =
		(float *) malloc(fdobj->nfds * sizeof(float))) != NULL);
	assert((file =
		(char **) malloc(fdobj->nfds * sizeof(char *))) != NULL);

	for (i = 0; i < fdobj->nfds; i++) {
		ndata[i] = SUMDATA(fdobj->fds[i])->n;
		x[i] = SUMDATA(fdobj->fds[i])->x;
		y[i] = SUMDATA(fdobj->fds[i])->y;
		sscanf(FD_GV(fdobj->fds[i], "spont"), "%f %*f", spont + i);
		file[i] = fdobj->fds[i]->filename;
	}

	FD_peak_stats(fdobj->nfds, ndata, x, y, spont, file, fdobj->to_tty,
		      0);

	free(ndata);
	free(x);
	free(y);
	free(spont);
	free(file);
}

#if(0)
void txt_LinearityStatsFN(Widget w, XtPointer fdobjptr,
			  XtPointer call_data)
{
	FDO *fdobj = (FDO *) fdobjptr;
	char *p1, *p2;
	FILE *fptr;

	p1 = pop_dialog("shift?", "");

	if ((p2 = fileBox(".txt", "w", "Save LinearityData to:")) == NULL)
		return;
	if (probefile(p2))
		if (pop_box("File exists!", "Overwrite", "Cancel", NULL) ==
		    2)
			return;
	assert((fptr = fopen(p2, "w")) != NULL);
	fprintf(fptr, ";; xdphys txt ver %s\n", XDPHYS_VERSION);

	txt_linearity_stats(fdobj, p1, fptr);

	fclose(fptr);
}

void txt_linearity_stats(FDO * fdo, char *p, FILE * fptr)
{
	int i, j, k, off;
	float min, max, min2, max2, tmpf;
	float **g_y, *g_x, *e;
	int g_n;
	int reps;
	char *q;
	float qf[100];

	if (fdo->nfds < 2)
		return;

	for (i = 1; i < fdo->nfds; i++) {
		if (strcmp(p, "")) {
			q = strtok(((i == 1) ? p : NULL), ",");
			sscanf(q, "%f", qf + i - 1);
		} else
			qf[i - 1] = 0.0;
	}

	min = SUMDATA(fdo->fds[0])->x[0];
	max = SUMDATA(fdo->fds[0])->x[SUMDATA(fdo->fds[0])->n - 1];
	for (i = 1; i < fdo->nfds; i++) {
		min2 = -qf[i - 1] + SUMDATA(fdo->fds[i])->x[0];
		max2 =
		    -qf[i - 1] +
		    SUMDATA(fdo->fds[i])->x[SUMDATA(fdo->fds[i])->n - 1];
		min = MY_MAX(min, min2);
		max = MY_MIN(max, max2);
	}

	assert((g_y =
		(float **) malloc((fdo->nfds - 1) * sizeof(float *))) !=
	       NULL);
	for (i = 1; i < fdo->nfds; i++) {
		assert((g_y[i - 1] =
			(float *) malloc(SUMDATA(fdo->fds[0])->n *
					 sizeof(float))) != NULL);
		for (k = 0, j = 0; j < SUMDATA(fdo->fds[0])->n; j++) {
			tmpf = SUMDATA(fdo->fds[0])->x[j];
			if ((tmpf < min) || (tmpf > max))
				continue;
			g_y[i - 1][k++] =
			    lin_interpolate(tmpf + qf[i - 1],
					    SUMDATA(fdo->fds[i])->n,
					    SUMDATA(fdo->fds[i])->x,
					    SUMDATA(fdo->fds[i])->y);
		}
	}

	assert((g_x =
		(float *) malloc(SUMDATA(fdo->fds[0])->n *
				 sizeof(float))) != NULL);
	assert((e =
		(float *) malloc(SUMDATA(fdo->fds[0])->n *
				 sizeof(float))) != NULL);
	reps = FD_GI(fdo->fds[0], "Reps");
	for (g_n = 0, off = -1, k = 0, i = 0; i < SUMDATA(fdo->fds[0])->n;
	     i++) {
		tmpf = SUMDATA(fdo->fds[0])->x[i];
		if ((tmpf < min) || (tmpf > max))
			continue;
		if (off == -1)
			off = i;
		e[k] = SUMDATA(fdo->fds[0])->e[i] * sqrt(reps);
		g_x[k++] = SUMDATA(fdo->fds[0])->x[i];
		g_n++;
	}

	FD_linearity_stats(SUMDATA(fdo->fds[0])->x + off,
			   SUMDATA(fdo->fds[0])->y + off, e, g_x, g_y, g_n,
			   fdo->nfds - 1, fdo, fptr);

	for (i = 0; i < fdo->nfds - 1; i++)
		free(g_y[i]);
	free(g_y);
	free(g_x);
	free(e);
}
#endif

int txt_get_n(FILEDATA * data)
{
	return (SUMDATA(data)->n);
}

float *txt_get_x(FILEDATA * data)
{
	return (SUMDATA(data)->x);
}

float *txt_get_y(FILEDATA * data)
{
	return (SUMDATA(data)->y);
}

float *txt_get_e(FILEDATA * data)
{
	return (SUMDATA(data)->e);
}
