/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: pm_itd.c,v 2.69 2003/09/15 18:52:56 bja Exp $
** Program: xdphys
**  Module: pm_itd.c
**  Author: mazer
** Descrip: xdphys plot method -- static itd tuning curve plotter
**
** Revision History (most recent last)
**
** Sat Nov 20 18:45:56 1993 mazer
**  - creation date :: from freq_p.c 
**  - this is to allow for more analysis of static itd tuning
**    curves -- namely to be able to fold them and turn itd into
**    ipd curves for purse tone stim.
**
** 97.3 bjarthur
**   added FitDataToCos courtesy YA
**
*******************************************************************/

#include "xdphyslib.h"
#include "plotter.h"

static int fitLogisticGaussian = 0;

static void fit_menu1(FDO *);
static int itd_plotter(FDO * fdo, FILEDATA *, FDObj_ViewType *, int,
		       FILE *);
static int FitDataToCos(float *, float *, float *, int, int, float **,
			float *, float *, float *, float *, float *, float *, float *, float *,
			float *, int *, float *);
static void FitCosineDerivatives(float, float[], float *, float[], int);
static void FitLogisticGaussianCosineDerivatives(float, float[], float *, float[], int);
static float FitCosPoint(float, float *);
static float EstimateFreq(float *, float *, int);
static int itd_valid_view(FDO *, int);

static int view_order[][3] = {
	{PM_CURVE},
	{PM_MEANPHASE, PM_VS, PM_RAYLEIGH},
	{PM_PERHIST},
	{PM_RASTER, PM_RASTER_RAW},
	{PM_PSTH},
	{PM_ISIH}
};
static int nsubviews[] = { 1, 3, 1, 2, 1, 1 };
static int nviews = 6;

int itd_do_plot(FDO * fdo, FILEDATA * fd, int view, int l, FILE * fptr)
{
	float period;
	int nbins;
	static int last_view = PM_NONE;

	if (!itd_valid_view(fdo, view)) {
		pm_type_error("itd", view);
		view = PM_DEFAULT;
		(void) itd_valid_view(fdo, view);
	}

	FD1_contra_ipsi(fd);

	FD_perhist_compute_period_nbins(fd, &period, &nbins);

	switch (view) {
	case PM_DEFAULT:
	case PM_CURVE:
		fit_menu1(fdo);
		FD1_curveFN(fdo, fd, l, fptr);
		break;
	case PM_MEANPHASE:
		FD1_perhist1FN(fdo, fd, l, fptr, period, nbins, 0);
		break;
	case PM_VS:
		FD1_perhist1FN(fdo, fd, l, fptr, period, nbins, 1);
		break;
	case PM_RAYLEIGH:
		FD1_perhist1FN(fdo, fd, l, fptr, period, nbins, 2);
		break;
	case PM_PERHIST:{
			if (last_view != PM_PERHIST)
				fdo->vsm_data = VSM_ALL_CODE;
			FD1_perhist_stats_menu(fdo);
			FD_vsm_menu(fdo, 1);
			FD1_perhist2FN(fdo, fd, l, fptr, period, nbins);
		}
		break;
	case PM_RASTER:
		FD1_rasterFN(fdo, fd, l, 0, fptr);
		break;
	case PM_RASTER_RAW:
		FD1_rasterFN(fdo, fd, l, 1, fptr);
		break;
	case PM_PSTH:
		FD1_psthFN(fdo, fd, l, fptr);
		break;
	case PM_ISIH:
		FD1_isih_stats_menu(fdo);
		FD1_isihFN(fdo, fd, l, fptr);
		break;
	default:
		pm_type_error("itd", view);
		break;
	}

	last_view = view;
	return (1);
}

static int itd_valid_view(FDO * fdo, int view)
{
	int retval = 0;
	int i, j;

	if (view == PM_DEFAULT) {
		retval = 1;
	} else {
		for (i = 0; i < nviews; i++) {
			for (j = 0; j < nsubviews[i]; j++) {
				if (view_order[i][j] == view) {
					retval = 1;
					fdo->view.lr = i;
					fdo->view.ud = j;
					break;
				}
			}
		}
	}

	return (retval);
}

static int itd_plotter(FDO * fdo, FILEDATA * fd, FDObj_ViewType * view,
		       int l, FILE * fptr)
{
	syn_spec ss;
	fd_syn_spec_parse(FD_GV(fdo->fds[l], "itd.Stim"), 0, &ss);

	adjust_index(view->lr, view->ud);

	if (l == 0) {
		if (!((ss.class == SC_TONE) ||
		      ((ss.class == SC_STACK)
		       && (ss.parms.stack.num_freqs == 1)))) {
			if (view->lr == 1) {
				view->lr = 3;
				view->ud = 0;
			} else if (view->lr == 2) {
				view->lr = 0;
				view->ud = 0;
			}
		}
	} else {
		if (!((ss.class == SC_TONE) ||
		      ((ss.class == SC_STACK)
		       && (ss.parms.stack.num_freqs == 1))))
			if ((view->lr == 1) || (view->lr == 2)) {
				notify
				    ("some curves missing due to different stim types");
				return (0);
			}
	}

	if (!fdo->no_X) {
		if (nsubviews[view->lr] > 1) {
			XtSetSensitive(fdo->up, True);
			XtSetSensitive(fdo->down, True);
		}
	}

	itd_do_plot(fdo, fd, view_order[view->lr][view->ud], l, fptr);

	return (1);
}


static void Heart(Widget w, XtPointer fdoptr, XtPointer call_data)
{
	FDO *fdo = (FDO *) fdoptr;

	int i, count, size;
	char *buf, buf2[128], *tmp;
	float *yfit, G, H, A, B, C, D, E, F, chi;
	int big;
	float *px, *py, *pz;
	int pn;
	float ndata;
	float *depvals;
	float cd, period, msr;
	syn_spec ss;
	char *foo;
	float *means, *stderrs;



	assert((buf = (char *) malloc(1024 * sizeof(char))) != NULL);
	size = 1024;

	if (fitLogisticGaussian) {
		tmp =
		    "equation:  G + H * logistic[ A + B * cos(C*t+D) * exp(-0.5*((t-E)/F)^2) ]\n";
		strcpy(buf, tmp);
		count = strlen(tmp);
		tmp =
		    "units:  G, H (spikes);  A, B (none);  C (rad/usec);  D (rad);  E, F (usec)\n";
		strcat(buf, tmp);
		count += strlen(tmp);
		tmp = "units:  freq, stim (Hz);  CD, period (us)\n\n";
		strcat(buf, tmp);
		count += strlen(tmp);
		tmp =
		    "file                 G       H       A       B      C        D       E        F         chi    |   freq   stim   CD      period   MSR\n";
		strcat(buf, tmp);
		count += strlen(tmp);
		tmp =
		    "----                 -       -       -       -      -        -       -        -         ---    |   ----   ----   --      ------   ---\n";
		strcat(buf, tmp);
		count += strlen(tmp);
	} else {
		tmp = "equation:  A + B * cos(C * t + D)\n";
		strcpy(buf, tmp);
		count = strlen(tmp);
		tmp =
		    "units:  A, B (spikes);  C (rad/msec);  D (rad)\n";
		strcat(buf, tmp);
		count += strlen(tmp);
		tmp = "units:  freq, stim (Hz);  CD, period (us)\n\n";
		strcat(buf, tmp);
		count += strlen(tmp);
		tmp =
		    "file                 A     B     C        D       chi    |  freq   stim   CD      period   MSR\n";
		strcat(buf, tmp);
		count += strlen(tmp);
		tmp =
		    "----                 -     -     -        -       ---    |  ----   ----   --      ------   ---\n";
		strcat(buf, tmp);
		count += strlen(tmp);
	}

	for (i = 0; i < fdo->nfds; i++) {
		ndata = FD1_get_ndata(fdo->fds[i]);
		depvals = FD1_get_depvals(fdo->fds[i]);
		foo = FD_GV(fdo->fds[i], "itd.Stim");
		fd_syn_spec_parse(foo, 0, &ss);
		means = FD1_get_means(fdo->fds[i]);
		stderrs = FD1_get_stderrs(fdo->fds[i]);
		FitDataToCos(depvals + 1, means + 1, stderrs + 1, ndata - 1,
				     (ss.class == SC_TONE) ? (int) (ss.parms.tone.freq) : 0, &yfit,
				     &G, &H, &A, &B, &C, &D, &E, &F, &chi,
				     &big,&msr);
		cd = -D / C;
		period = 2.0 * M_PI / C;
		while (cd > period)
			cd -= period;
		while (cd < -period)
			cd += period;

		if (fitLogisticGaussian)
#ifdef __linux__
			count +=
			    sprintf(buf2,
				    "%-13s   %5.1f    %5.1f  %5.1f   %5.1f    %5.3f  %5.1f     %5.1f   %5.3f   %5.1f%1s |  %5d  %5d  %6.1f  %6.1f  %5.2f\n",
				    strrchr(fdo->fds[i]->filename,
					    '/') ==
				    NULL ? fdo->fds[i]->filename : 1 +
				    (char *) strrchr(fdo->fds[i]->filename,
						     '/'), G, H, A, B, C,
				    D, E, F, chi, big ? "*" : " ",
				    (int) (C * 1.0e6 / (2.0 * M_PI)),
				    (ss.class == SC_TONE) ? (int) (ss.parms.tone.freq) : 0,
            cd,period,msr);
#else
			count +=
			    strlen(sprintf
				   (buf2,
				    "%-13s   %5.1f    %5.1f  %5.1f   %5.1f    %5.3f  %5.1f     %5.1f   %5.3f   %5.1f%1s |  %5d  %5d  %6.1f  %6.1f  %5.2f\n",
				    strrchr(fdo->fds[i]->filename,
					    '/') ==
				    NULL ? fdo->fds[i]->filename : 1 +
				    (char *) strrchr(fdo->fds[i]->filename,
						     '/'), G, H, A, B, C,
				    D, E, F, chi, big ? "*" : " ",
				    (int) (C * 1.0e6 / (2.0 * M_PI)),
				    (ss.class == SC_TONE) ? (int) (ss.parms.tone.freq) : 0,
            cd,period,msr));
#endif
		else
#ifdef __linux__
			count +=
			    sprintf(buf2,
				    "%-13s   %5.1f  %5.1f   %5.3f  %5.1f   %5.1f%1s |  %5d  %5d  %6.1f  %6.1f  %6.1f\n",
				    strrchr(fdo->fds[i]->filename,
					    '/') ==
				    NULL ? fdo->fds[i]->filename : 1 +
				    (char *) strrchr(fdo->fds[i]->filename,
						     '/'), A, B, C,
				    D, chi, big ? "*" : " ",
				    (int) (C * 1.0e6 / (2.0 * M_PI)),
				    (ss.class == SC_TONE) ? (int) (ss.parms.tone.freq) : 0,
            cd,period,msr);
#else
			count +=
			    strlen(sprintf
				   (buf2,
				    "%-13s   %5.1f  %5.1f   %5.1f  %5.1f   %5.3f  %5.1f   %5.1f%1s |  %5d  %5d  %6.1f  %6.1f  %6.1f\n",
				    strrchr(fdo->fds[i]->filename,
					    '/') ==
				    NULL ? fdo->fds[i]->filename : 1 +
				    (char *) strrchr(fdo->fds[i]->filename,
						     '/'), d2, B, C,
				    D, chi, big ? "*" : " ",
				    (int) (C * 1.0e6 / (2.0 * M_PI)),
				    (ss.class == SC_TONE) ? (int) (ss.parms.tone.freq) : 0,
            cd,period,msr));
#endif
		if (fdo->to_tty) {
			printf("%s", buf2);
			continue;
		}
		strcat(buf, buf2);
		if ((size - count) < 512) {
			assert((buf =
				(char *) realloc(buf,
						 (size +
						  1024) * sizeof(char))) !=
			       NULL);
			size += 1024;
		}
		FD_plotter_copy(depvals + 1, yfit, NULL, ndata - 1,
					0, &px, &py, &pz, &pn, NULL, NULL,
					NULL, NULL);
		FD_plotter_copy(depvals + 1, yfit, NULL, ndata - 1, 0,
		     &px, &py, &pz, &pn, NULL, NULL, NULL, NULL);

		if (!fdo->no_X) {
			FDObj_Add_Data_All(fdo, px, py, pn);
			FDObj_AddLine(fdo, i, px, py, pz, pn, AtFloat,
				      atQuadLinePlotWidgetClass,
				      AtTypeLINEPOINTS, AtMarkCIRCLE,
				      ConvertColor(fdo->graph,
						   FDObj_Colors[(i+1) % FDObj_numColors]),
				      (XtArgVal) "fit");
		}
	}

	tmp =
	    "\n(*) indicates chi is big, and fitted parameters are questionable";
	strcat(buf, tmp);
	count += strlen(tmp);

	if (!fdo->to_tty) {
		if (fitLogisticGaussian)
			pop_text("Logistic(Cosine*Gaussian)", buf,
				 strlen(buf) + 1, False);
		else
			pop_text("Cosine", buf, strlen(buf) + 1, False);
	}
}

void Cosine(Widget w, XtPointer fdoptr, XtPointer call_data)
{
	FDO *fdo = (FDO *) fdoptr;

	fitLogisticGaussian = 0;
	Heart(w, fdo, call_data);
}

void LogisticCosineGaussian(Widget w, XtPointer fdoptr, XtPointer call_data)
{
	FDO *fdo = (FDO *) fdoptr;
	fitLogisticGaussian = 1;
	Heart(w, fdo, call_data);
}

static void fit_menu1(FDO * fdobj)
{
	if (fdobj->no_X)
		return;

	menubutton_clear(fdobj->fit, &(fdobj->fitpsh));
	XtSetSensitive(fdobj->fit, False);

	menubutton_add(fdobj->fitpsh, "peak stats", FD1_peak_stats, fdobj);
	menubutton_add(fdobj->fitpsh, "similarity stats",
		       FD1_similarity_stats, fdobj);
	menubutton_add(fdobj->fitpsh, "cosine", Cosine, fdobj);
	menubutton_add(fdobj->fitpsh, "logistic(cosine*gaussian)", LogisticCosineGaussian,
		       fdobj);
	menubutton_add(fdobj->fitpsh, "prob distribution",
		       FD1_prob_distribution, fdobj);
	XtSetSensitive(fdobj->fit, True);
}

int pm_itd_init(void)
{
	setFDOvalidviewMethod("itd", itd_valid_view);
	setFDOdoplotMethod("itd", itd_do_plot);
	setFDOplotMethod("itd", itd_plotter);
	setFDreadMethod("itd", FD1_reader);
	setFDfreeMethod("itd", FD1_free);
	return (1);
}



/************** from yuda *********************/


#define CHI2_STOP_STEP (0.01)

static int FitDataToCos(float *fAxis,	/* x[] of curve to be fit */
			float *fData,	/* y[] of curve to be fit */
			float *fSig,	/* err[] of curve to be fit */
			int ndata, int nFreq, float **yfit,	/* y[] of resulting fit */
			float *G, float *H, float *A, float *B, float *C,
			float *D, float *E, float *F, float *chi, int *big, float *msr)
{
	float *x, *y, *sig;
	int i, j, *ia, ma;
	float *a, **covar, **alpha, alamda = -1, chisq = 10, oldchi = 1000;
	float x0, y0, y1;
	void (*GeneralFit) (float, float[], float *, float[], int);
	GeneralFit = fitLogisticGaussian ? FitLogisticGaussianCosineDerivatives : FitCosineDerivatives;
	ma = fitLogisticGaussian ? 8 : 4;

	alpha = matrix(1, ma, 1, ma);
	covar = matrix(1, ma, 1, ma);
	ia = ivector(1, ma);
	a = vector(1, ma);
	x = vector(1, ndata);
	y = vector(1, ndata);
	sig = vector(1, ndata);
	for (i = 0; i < ndata; i++) {
		sig[i + 1] = (fSig[i]>0.1)?fSig[i]:0.1;
		x[i + 1] = fAxis[i];
		y[i + 1] = fData[i]; }

	for (x0 = x[1], y0 = y1 = y[1], i = 2; i <= ndata; i++) {
		if (y[i] > y0) {
			x0 = x[i], y0 = y[i]; }
		if (y[i] < y1) {
			y1 = y[i]; } }

 	a[1] = y1+(y0-y1)/2;
 	a[2] = (y0-y1)/2;
 	a[3] = (2.0 * M_PI) * ((nFreq!=0)?nFreq:EstimateFreq(x, y, ndata)) / 1000000;  /* rad/ms */
 	a[4] = -a[3] * x0;
	if(fitLogisticGaussian) {
    a[1] = 0.0;
    a[2] = 1.0;
    a[5] = x0;
    a[6] = 200.0;
    a[7] = 0.0;
    a[8] = y0; }

	if(fitLogisticGaussian) {
  	alamda = -1;
  	for (i=1; i<=ma; i++) { ia[i]=1; }  ia[5]=ia[6]=0;
    do {
	  	oldchi = chisq;
	  	mrqmin(x, y, sig, ndata, a, ia, ma, covar, alpha, &chisq, GeneralFit, &alamda); }
	  while ((((oldchi-chisq)>CHI2_STOP_STEP) || ((oldchi-chisq)<-CHI2_STOP_STEP) || (oldchi==chisq)) && (alamda<MAXFLOAT)); }

	alamda = -1;
	for (i=1; i<=ma; i++) { ia[i]=1; }
  do {
		oldchi = chisq;
		mrqmin(x, y, sig, ndata, a, ia, ma, covar, alpha, &chisq, GeneralFit, &alamda); }
	while ((((oldchi-chisq)>CHI2_STOP_STEP) || ((oldchi-chisq)<-CHI2_STOP_STEP) || (oldchi==chisq)) && (alamda<MAXFLOAT));

	alamda = 0;
	mrqmin(x, y, sig, ndata, a, ia, ma, covar, alpha, &chisq, GeneralFit, &alamda);

	if (a[2] < 0) {
		a[2] = -a[2];
		a[4] += M_PI;
		if (a[4] < 0) {
			while (a[4] < M_PI) {
				a[4] += (2.0 * M_PI); } }
		if (a[4] > 0) {
			while (a[4] > M_PI) {
				a[4] -= (2.0 * M_PI); } } }

	assert(((*yfit) = (float *) malloc((ndata) * sizeof(float))) != NULL);
	for (i = 1; i <= ndata; i++)
		(*yfit)[i - 1] = FitCosPoint(x[i], a);
	(*A) = a[1];
	(*B) = a[2];
	(*C) = (a[3]<0)?(-a[3]):a[3];
	(*D) = a[4];
	if (fitLogisticGaussian) {
		(*E) = a[5];
		(*F) = a[6];
	  (*G) = a[7];
	  (*H) = a[8]; }
	(*chi) = chisq;
	(*big) = (chisq > ((ndata - ma) + (float) sqrt(2.0 * (ndata - ma)))) ? 1 : 0;
  (*msr) = FitCosPoint(-(*D)/(*C),a)/((FitCosPoint((2.0*M_PI-(*D))/(*C),a)+FitCosPoint((-2.0*M_PI-(*D))/(*C),a))/2);

  /*
  the mean of the chi2 distribution is the degrees of freedom, i.e.,
  ndata - ma (ma is the number of parameters).  the standard deviation
  of the chi2 distribution is sqrt(2 * mean). thus, i give a warning
  when chi2 exceeds the mean + std.  YA
  */

	free_matrix(covar, 1, ma, 1, ma);
	free_matrix(alpha, 1, ma, 1, ma);
	free_ivector(ia, 1, ma);
	free_vector(a, 1, ma);
	free_vector(x, 1, ndata);
	free_vector(y, 1, ndata);
	free_vector(sig, 1, ndata);
	fflush(stderr);
	return 1;
}

void FitCosineDerivatives(float x, float a[], float *y, float dyda[], int na)
{
	float phi = a[3] * x + a[4];
	float C = cos(phi);
	*y = a[1] + a[2] * C;
	dyda[1] = 1;
	dyda[2] = C;
	dyda[4] = -a[2] * sin(phi);
	dyda[3] = x * dyda[4];
}

/*
Y = G + H / (1 + exp[-A - B * cos(Cx+D) * exp(-0.5*((x-E)/F)^2)])
dY/dG = 1
dY/dH = 1 / {1 + exp[]}
dY/dA = H * exp[] / {1 + exp[]}^2
dY/dB = H * exp[] * cos(Cx+D) * exp(-0.5*((x-E)/F)^2) / {1 + exp[]}^2
dY/dC = -H * exp[] * B * exp(-0.5*((x-E)/F)^2) * sin(Cx+D) * x / {1 + exp[]}^2
dY/dD = -H * exp[] * B * exp(-0.5*((x-E)/F)^2) * sin(Cx+D) / {1 + exp[]}^2
dY/dE = H * exp[] * B * cos(Cx+D) * exp(-0.5*((x-E)/F)^2) * (x-E)/F^2 / {1 + exp[]}^2
dY/dF = H * exp[] * B * cos(Cx+D) * exp(-0.5*((x-E)/F)^2) * (x-E)^2/F^3 / {1 + exp[]}^2
*/

void FitLogisticGaussianCosineDerivatives(float x, float a[], float *y, float dyda[], int na)
{
  float cos_ef = cos(a[3]*x+a[4]);
  float sin_ef = sin(a[3]*x+a[4]);
  float exp_par = exp(-0.5*((x-a[5])/a[6])*((x-a[5])/a[6]));
  float exp_br = exp(-a[1] - a[2] * cos_ef * exp_par);

	*y = a[7] + a[8] / (1.0+exp_br);
	dyda[7] = 1.0;
	dyda[8] = 1.0 / (1.0+exp_br);
	dyda[1] = a[8] * exp_br / (1.0+exp_br) / (1.0+exp_br);
	dyda[2] = a[8] * exp_br * cos_ef * exp_par / (1.0+exp_br) / (1.0+exp_br);
	dyda[3] = -a[8] * exp_br * a[2] * exp_par * sin_ef * x / (1.0+exp_br) / (1.0+exp_br);
	dyda[4] = -a[8] * exp_br * a[2] * exp_par * sin_ef / (1.0+exp_br) / (1.0+exp_br);
	dyda[5] = a[8] * exp_br * a[2] * cos_ef * exp_par * (x-a[5])/a[6]/a[6] / (1.0+exp_br) / (1.0+exp_br);
	dyda[6] = a[8] * exp_br * a[2] * cos_ef * exp_par * (x-a[5])*(x-a[5])/a[6]/a[6]/a[6] / (1.0+exp_br) / (1.0+exp_br);
}

static float FitCosPoint(float x, float *a)
{
	float fFit, tmp;

	fFit = a[2] * cos(a[3]*x+a[4]);
	if (fitLogisticGaussian) {
    tmp=(x-a[5])/a[6];
		fFit *= exp(-0.5*tmp*tmp); }
	fFit += a[1];
	if (fitLogisticGaussian) {
    fFit = a[7]+a[8]/(1.0+exp(-fFit)); }
	return fFit;
}

static float EstimateFreq(float *x, float *y, int ndata)
{
	int nFirstMin, nLastMin, nFirstMax, nLastMax;
	int nMaxs, nMins;
	int ni, i, cond, last;
	float fFreq, fAvgPeriod;
	float fAvg, yi, xi, xmax, ymax, xspan, xup, xdn;

	ymax = y[0], xmax = x[0];
	for (i = 2; i < ndata; i++) {
		xi = x[i], yi = y[i];
		if (yi > ymax)
			ymax = yi, xmax = xi;
	}

	fAvg = 0;
	xspan = 300;
	xup = xmax + xspan;
	xdn = xmax - xspan;
	for (ni = 0, i = 2; i < ndata; i++) {
		xi = x[i], yi = y[i];
		if (xi < xdn || xi > xup)
			continue;
		fAvg += yi;
		ni++;
	}
	if (ni > 1)
		fAvg /= (float) (ni - 1);
	else
		return 7000;

	nMaxs = 0;
	for (last = 0, i = 2; i < ndata; i++) {
		xi = x[i];
		if (xi < xdn || xi > xup)
			continue;
		yi = y[i];
		cond = yi >= fAvg;
		if ((yi >= y[i - 1]) && (yi >= y[i + 1])) {
		  if (last == i - 1) {
			  last = i;
		  	continue; }
			last = i;
			if (cond) {
				nMaxs++;
        if(nMaxs==1) { nFirstMax=i; }
        nLastMax=i; } } }

	nMins = 0;
	for (last = 0, i = 2; i < ndata; i++) {
		xi = x[i];
		if (xi < xdn || xi > xup)
			continue;
		yi = y[i];
		cond = yi <= fAvg;
		if ((yi <= y[i - 1]) && (yi <= y[i + 1])) {
		  if (last == i - 1) {
        last=i;
			  continue; }
			last = i;
			if (cond) {
				nMins++;
        if(nMins==1) { nFirstMin=i; }
        nLastMin=i; } } }

	fAvgPeriod = 0;
	if (nMaxs > 1)
		fAvgPeriod += (x[nLastMax] - x[nFirstMax]) / (nMaxs - 1);
	if (nMins > 1)
		fAvgPeriod += (x[nLastMin] - x[nFirstMin]) / (nMins - 1);
	if ((nMins > 1) && (nMaxs > 1))
		fAvgPeriod /= 2;
	if ((nMins > 1) || (nMaxs > 1)) {
		return 1000000 / fAvgPeriod;
	}
	return 7000;
}
