/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: pm_int.c,v 1.17 2002/07/15 04:30:18 cmalek Exp $
** Program: dowl
**  Module: pm_fr.c
**  Author: bjarthur
** Descrip: xdowl plot method -- static iid tuning curve plotter
**
** Revision History (most recent last)
**
** 10NOV96 bjarthur
**  copied from pm_iid.c
**
** NOV96 bjarthur
**   copied from pm_fr.c
**
*******************************************************************/

#include "xdphyslib.h"
#include "xcalibur.h"
#include "plotter.h"

static void ComputeMagRmsError(Widget, XtPointer, XtPointer);
static void ComputePhiRmsError(Widget, XtPointer, XtPointer);
static void funcs(float, float *, int);
static void ComputeRmsError(FDO *, int);
static void int_FN(FDO *, FILEDATA *, int, int, FILE *);
static void fit_menu1(FDO *);
static void fit_menu2(FDO *);
static int int_plotter(FDO *, FILEDATA *, FDObj_ViewType *, int, FILE *);
static int int_valid_view(FDO *, int);
static int int_reader(FILE *, FILEDATA *, int);
static int int_free(FILEDATA *);

typedef struct {
	int natts;
	float *atts;
	float *mags, *phis, *dists;
	float *mag_sds, *phi_sds;
} SumData;

#define SUMDATA(data) ((SumData*)(data->sumdata))

static int view_order[][1] = {
	{PM_MAG},
	{PM_PHASE},
	{PM_DISTORTION}
};
static int nsubviews[] = { 1, 1, 1 };
static int nviews = 3;

int int_do_plot(FDO * fdo, FILEDATA * fd, int view, int l, FILE * fptr)
{
	if (!int_valid_view(fdo, view)) {
		pm_type_error("int", view);
		view = PM_DEFAULT;
		(void) int_valid_view(fdo, view);
	}

	switch (view) {
	case PM_DEFAULT:
	case PM_MAG:
		fit_menu1(fdo);
		int_FN(fdo, fd, l, 1, fptr);
		break;
	case PM_PHASE:
		fit_menu2(fdo);
		int_FN(fdo, fd, l, 0, fptr);
		break;
	case PM_DISTORTION:
		int_FN(fdo, fd, l, 2, fptr);
		break;
	default:
		pm_type_error("int", view);
		break;
	}

	return (1);
}

static int int_valid_view(FDO * fdo, int view)
{
	int retval = 0;
	int i, j;

	if (view == PM_DEFAULT) {
		retval = 1;
	} else {
		for (i = 0; i < nviews; i++) {
			for (j = 0; j < nsubviews[i]; j++) {
				if (view_order[i][j] == view) {
					retval = 1;
					fdo->view.lr = i;
					fdo->view.ud = j;
					break;
				}
			}
		}
	}

	return (retval);
}

static int int_plotter(FDO * fdo, FILEDATA * fd, FDObj_ViewType * view,
		       int l, FILE * fptr)
{
	adjust_index(view->lr, view->ud);

	int_do_plot(fdo, fd, view_order[view->lr][view->ud], l, fptr);

	return (1);
}

static void ComputeMagRmsError(Widget w, XtPointer fdoptr,
			       XtPointer client_data)
{
	FDO *fdo = (FDO *) fdoptr;

	ComputeRmsError(fdo, 1);
}

static void ComputePhiRmsError(Widget w, XtPointer fdoptr,
			       XtPointer client_data)
{
	FDO *fdo = (FDO *) fdoptr;

	ComputeRmsError(fdo, 0);
}

static void funcs(float x, float *afunc, int ma)
{
	assert(ma == 2);

	afunc[1] = 1.0;
	afunc[2] = x;
}

static void ComputeRmsError(FDO * fdo, int mag)
{
	char *buf, buf2[128], *tmp;
	int l, j, size, count, ia[] = { 1, 0 };
	float rms_err, ftmp;
	float *x, *y, a[2], *covar[3], chisq;
	float freq;

	assert((buf = (char *) malloc(1024 * sizeof(char))) != NULL);
	size = 1024;

	for (j = 0; j < 3; j++)
		assert((covar[j] =
			(float *) malloc(3 * sizeof(float))) != NULL);

	if (mag)
		tmp = "file            intercept     chi       dB\n";
	else
		tmp = "file            intercept     chi       us\n";
	strcpy(buf, tmp);
	count = strlen(tmp);
	tmp = "----            ---------     ---       --\n";
	strcat(buf, tmp);
	count += strlen(tmp);

	for (l = 0; l < fdo->nfds; l++) {
		freq = (float) FD_GI(fdo->fds[l], "int.freq");
		if (mag) {
			a[0] = 1.0;
			a[1] = -1.0;
			lfit(-1 + SUMDATA(fdo->fds[l])->atts,
			     -1 + SUMDATA(fdo->fds[l])->mags,
			     -1 + SUMDATA(fdo->fds[l])->mag_sds,
			     SUMDATA(fdo->fds[l])->natts, -1 + a, -1 + ia,
			     2, covar, &chisq, funcs);
		} else {
			assert((x =
				malloc(SUMDATA(fdo->fds[l])->natts *
				       sizeof(float))) != NULL);
			assert((y =
				malloc(SUMDATA(fdo->fds[l])->natts *
				       sizeof(float))) != NULL);
			for (j = 0; j < SUMDATA(fdo->fds[l])->natts; j++) {
				x[j] =
				    1.0e6 * SUMDATA(fdo->fds[l])->phis[j] /
				    (2.0 * M_PI * freq);
				y[j] =
				    1.0e6 *
				    SUMDATA(fdo->fds[l])->phi_sds[j] /
				    (2.0 * M_PI * freq);
			}
			a[0] = 1.0;
			a[1] = 0.0;
			lfit(-1 + SUMDATA(fdo->fds[l])->atts, -1 + x,
			     -1 + y, SUMDATA(fdo->fds[l])->natts, -1 + a,
			     -1 + ia, 2, covar, &chisq, funcs);
			free(x);
			free(y);
		}

		for (j = 0, rms_err = 0.0; j < SUMDATA(fdo->fds[l])->natts;
		     j++) {
			if (mag) {
				ftmp =
				    a[0] - SUMDATA(fdo->fds[l])->atts[j];
				ftmp =
				    (ftmp - SUMDATA(fdo->fds[l])->mags[j]);
			} else {
				ftmp = a[0];
				ftmp =
				    (ftmp -
				     1.0e6 *
				     SUMDATA(fdo->fds[l])->phis[j] / (2.0 *
								      M_PI
								      *
								      freq));
			}
			rms_err += (ftmp * ftmp);
		}
		rms_err = sqrt(rms_err / j);

		assert((x = (float *) malloc(2 * sizeof(float))) != NULL);
		assert((y = (float *) malloc(2 * sizeof(float))) != NULL);
		x[0] = SUMDATA(fdo->fds[l])->atts[0];
		x[1] =
		    SUMDATA(fdo->fds[l])->atts[-1 +
					       SUMDATA(fdo->fds[l])->
					       natts];
		if (mag) {
			y[0] = a[0] - x[0];
			y[1] = a[0] - x[1];
		} else {
			y[0] = a[0];
			y[1] = a[0];
		}
		if (!fdo->no_X) {
			FDObj_Add_Data_All(fdo, x, y, 2);
			FDObj_AddLine(fdo, l, x, y, NULL, 2, AtFloat,
				      atQuadLinePlotWidgetClass,
				      AtTypeLINEPOINTS, AtMarkCIRCLE,
				      ConvertColor(fdo->graph,
						   FDObj_Colors[l %
								FDObj_numColors]),
				      (XtArgVal) "fit");
		}
#ifdef __linux__
		count += sprintf(buf2,
				 "%-13s   %5.1f(%5.2f)  %6.1f    %5.2f\n",
				 strrchr(fdo->fds[l]->filename,
					 '/') ==
				 NULL ? fdo->fds[l]->filename : 1 +
				 (char *) strrchr(fdo->fds[l]->filename,
						  '/'), a[0],
				 sqrt(covar[1][1]), chisq, rms_err);
#else
		count += strlen(sprintf(buf2,
					"%-13s   %5.1f(%5.2f)  %6.1f    %5.2f\n",
					strrchr(fdo->fds[l]->filename,
						'/') ==
					NULL ? fdo->fds[l]->filename : 1 +
					(char *) strrchr(fdo->fds[l]->
							 filename, '/'),
					a[0], sqrt(covar[1][1]), chisq,
					rms_err));
#endif
		strcat(buf, buf2);
		if ((size - count) < 512) {
			assert((buf =
				(char *) realloc(buf,
						 (size +
						  1024) * sizeof(char))) !=
			       NULL);
			size += 1024;
		}
	}

	if (!fdo->no_X) {
		if (mag)
			pop_text("mag-rms-error", buf, strlen(buf) + 1,
				 False);
		else
			pop_text("phi-rms-error", buf, strlen(buf) + 1,
				 False);
	}

	for (j = 0; j < 3; j++)
		free(covar[j]);
}

static void int_FN(FDO * fdo, FILEDATA * fd, int l, int mag, FILE * fptr)
{
	char *ylabel, *xlabel, *ti, lbuf[100];
	int i;
	float *x, *y;
	float freq;

	freq = (float) FD_GI(fd, "int.freq");

	COPY_ARRAY(SUMDATA(fd)->atts, x, SUMDATA(fd)->natts, float);
	switch (mag) {
	case (1):		/* mags */
		COPY_ARRAY(SUMDATA(fd)->mags, y, SUMDATA(fd)->natts,
			   float);
		break;
	case (0):		/* phis */
		COPY_ARRAY(SUMDATA(fd)->phis, y, SUMDATA(fd)->natts,
			   float);
		break;
	case (2):		/* dists */
		COPY_ARRAY(SUMDATA(fd)->dists, y, SUMDATA(fd)->natts,
			   float);
		break;
	}

	if (mag == 0) {
		for (i = 0; i < SUMDATA(fd)->natts; i++)
			y[i] = 1.0e6 * y[i] / (2.0 * M_PI * freq);
	}

	if (fptr != NULL) {
		fprintf(fptr, "COMMENTS\n");
		fprintf(fptr, "%%text dump of %s\n",
			strrchr(fd->filename, '/') == NULL ? fd->filename :
			1 + (char *) strrchr(fd->filename, '/'));
		fprintf(fptr, "%%col 1: frequency (Hz)\n");
		if (mag)
			fprintf(fptr, "%%col 2: intensity (dB SPL)\n");
		else
			fprintf(fptr, "%%col 2: phase (us)\n");
		fprintf(fptr, "END_COMMENTS\n");
		fprintf(fptr, "PARAMS\n");
		fprintf(fptr, "END_PARAMS\n");
		fprintf(fptr, "RASTERDATA\n");
		fprintf(fptr, "nlines=%d\n", SUMDATA(fd)->natts);
		for (i = 0; i < SUMDATA(fd)->natts; i++)
			fprintf(fptr, "%e\t%e\n", x[i], y[i]);
		fprintf(fptr, "END_RASTERDATA\n");
	}

	if (fdo->no_X)
		return;

	FDObj_Add_Data_All(fdo, x, y, SUMDATA(fd)->natts);
	sprintf(lbuf, "%5d Hz", (int) freq);
	FDObj_AddLine(fdo, l, x, y, NULL, SUMDATA(fd)->natts, AtFloat,
		      atQuadLinePlotWidgetClass, AtTypeLINEPOINTS,
		      AtMarkCIRCLE, ConvertColor(fdo->graph,
						 FDObj_Colors[l %
							      FDObj_numColors]),
		      (XtArgVal) lbuf);

	switch (mag) {
	case (1):		/* mags */
		ylabel = "absolute intensity (dB)";
		break;
	case (0):		/* phis */
		ylabel = "phase (us)";
		break;
	case (2):		/* dists */
		ylabel = "distortion (%)";
		break;
	}
	xlabel = "attenuation (dB)";
	ti = "";
	XtVaSetValues(fdo->xaxis, XtNlabel, xlabel, NULL);
	XtVaSetValues(fdo->yaxis, XtNlabel, ylabel, NULL);
	XtVaSetValues(fdo->graph, XtNtitle, FDObj_Title(fdo, ti, l),
		      XtNshowTitle, True, NULL);
}

static void fit_menu1(FDO * fdo)
{
	menubutton_clear(fdo->fit, &(fdo->fitpsh));
	XtSetSensitive(fdo->fit, False);

	menubutton_add(fdo->fitpsh, "rms-error", ComputeMagRmsError, fdo);
	XtSetSensitive(fdo->fit, True);
}

static void fit_menu2(FDO * fdo)
{
	menubutton_clear(fdo->fit, &(fdo->fitpsh));
	XtSetSensitive(fdo->fit, False);

	menubutton_add(fdo->fitpsh, "rms-error", ComputePhiRmsError, fdo);
	XtSetSensitive(fdo->fit, True);
}

static int int_reader(FILE * fptr, FILEDATA * fd, int ana)
{
	int i, tmp, r1;
	float *p1;
	char *p, buf[1024];
	int *stimarray_att;
	float att, mag, phi, dist, mag_sd, phi_sd;

	fd->rawdata = NULL;
	fd->channels = NULL;
	assert((fd->sumdata = malloc(sizeof(SumData))) != NULL);

	if ((p = skipTo(fptr, "nrasters=")) == NULL
	    || sscanf(p, "nrasters=%d", &(SUMDATA(fd)->natts)) != 1) {
		if (debugflag)
			fprintf(stderr,
				"FD_read: %s missing nrasters= line\n",
				fd->filename);
		fclose(fptr);
		FD_free(fd);
		return (0);
	}

	assert((SUMDATA(fd)->atts = (float *) malloc(SUMDATA(fd)->natts *
						     sizeof(float))) !=
	       NULL);
	assert((SUMDATA(fd)->mags =
		(float *) malloc(SUMDATA(fd)->natts * sizeof(float))) !=
	       NULL);
	assert((SUMDATA(fd)->phis =
		(float *) malloc(SUMDATA(fd)->natts * sizeof(float))) !=
	       NULL);
	assert((SUMDATA(fd)->dists =
		(float *) malloc(SUMDATA(fd)->natts * sizeof(float))) !=
	       NULL);
	assert((SUMDATA(fd)->mag_sds =
		(float *) malloc(SUMDATA(fd)->natts * sizeof(float))) !=
	       NULL);
	assert((SUMDATA(fd)->phi_sds =
		(float *) malloc(SUMDATA(fd)->natts * sizeof(float))) !=
	       NULL);

	if ((stimarray_att =
	     StimArray_Gen(&tmp, 0, FD_GV(fd, "int.att_range"), 1, 1,
			   0)) == NULL)
		return (0);
	assert(tmp == SUMDATA(fd)->natts);
	qsort(stimarray_att, tmp, sizeof(int), intcompare);
	for (i = 0; i < SUMDATA(fd)->natts; i++)
		SUMDATA(fd)->atts[i] = stimarray_att[i];
	free(stimarray_att);

	for (i = 0; i < SUMDATA(fd)->natts; i++) {
		fgets(buf, sizeof(buf), fptr);
		sscanf(buf, "%f %f %f %f %f %f", &att, &mag, &phi, &dist,
		       &mag_sd, &phi_sd);
		p1 = (float *) bsearch(&att, SUMDATA(fd)->atts,
				       SUMDATA(fd)->natts, sizeof(float),
				       floatcompare);
		r1 = p1 - SUMDATA(fd)->atts;
		SUMDATA(fd)->mags[r1] = mag;
		SUMDATA(fd)->phis[r1] = phi;
		SUMDATA(fd)->dists[r1] = dist;
		SUMDATA(fd)->mag_sds[r1] = mag_sd;
		SUMDATA(fd)->phi_sds[r1] = phi_sd;
	}

	unwrap(SUMDATA(fd)->phis, SUMDATA(fd)->natts, M_PI);

	return (1);
}

static int int_free(FILEDATA * fd)
{
	FREE(SUMDATA(fd)->atts);
	FREE(SUMDATA(fd)->mags);
	FREE(SUMDATA(fd)->phis);
	FREE(SUMDATA(fd)->dists);
	FREE(SUMDATA(fd)->mag_sds);
	FREE(SUMDATA(fd)->phi_sds);

	FREE(SUMDATA(fd));

	return (1);
}

int pm_int_init(void)
{
	setFDOvalidviewMethod("int", int_valid_view);
	setFDOdoplotMethod("int", int_do_plot);
	setFDOplotMethod("int", int_plotter);
	setFDreadMethod("int", int_reader);
	setFDfreeMethod("int", int_free);
	return (1);
}
