/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: pm_iid.c,v 2.63 2003/09/15 18:52:55 bja Exp $
** Program: xdphys
**  Module: pm_iid.c
**  Author: mazer
** Descrip: xdphys plot method -- static iid tuning curve plotter
**
** Revision History (most recent last)
**
** Tue Nov  1 17:17:40 1994 mazer
**  - creation date :: from pm_itd.c 
**  - this is just to support the "contraipsi" preference feature
**    do tha IID curves can be plotted relative to c>i and i>c
**    instead of l>r and r>l
**
*******************************************************************/

#include "xdphyslib.h"
#include "xdphys.h"
#include "plotter.h"

static int iid_plotter(FDO*,FILEDATA*,FDObj_ViewType*,int,FILE*);
static void fit_menu(FDO*);
static int iid_valid_view(FDO *, int);
static int FitDataToTwoLogistic(float*, float*, float*, int, int, float**,
      float*, float*, float*, float*, float*, float*, float*, float*, int*, float*);
static void FitTwoLogisticDerivatives(float, float[], float*, float[], int);
static float FitTwoLogisticPointGolden(float);
static float FitTwoLogisticPoint(float, float*);

static int view_order[][3] = {
	{PM_CURVE},
	{PM_MEANPHASE, PM_VS, PM_RAYLEIGH},
	{PM_PERHIST},
	{PM_RASTER, PM_RASTER_RAW},
	{PM_PSTH},
	{PM_ISIH}
};
static int nsubviews[] = { 1, 3, 1, 2, 1, 1 };
static int nviews = 6;

float *G_a;

int iid_do_plot(FDO * fdo, FILEDATA * fd, int view, int l, FILE * fptr)
{
	float period;
	int nbins;
	static int last_view = PM_NONE;

	if (!iid_valid_view(fdo, view)) {
		pm_type_error("iid", view);
		view = PM_DEFAULT;
		(void) iid_valid_view(fdo, view);
	}

	FD1_contra_ipsi(fd);

	FD_perhist_compute_period_nbins(fd, &period, &nbins);

	switch (view) {
	case PM_DEFAULT:
	case PM_CURVE:
		fit_menu(fdo);
		FD1_curveFN(fdo, fd, l, fptr);
		break;
	case PM_MEANPHASE:
		FD1_perhist1FN(fdo, fd, l, fptr, period, nbins, 0);
		break;
	case PM_VS:
		FD1_perhist1FN(fdo, fd, l, fptr, period, nbins, 1);
		break;
	case PM_RAYLEIGH:
		FD1_perhist1FN(fdo, fd, l, fptr, period, nbins, 2);
		break;
	case PM_PERHIST:{
			if (last_view != PM_PERHIST)
				fdo->vsm_data = VSM_ALL_CODE;
			FD1_perhist_stats_menu(fdo);
			FD_vsm_menu(fdo, 1);
			FD1_perhist2FN(fdo, fd, l, fptr, period, nbins);
		}
		break;
	case PM_RASTER:
		FD1_rasterFN(fdo, fd, l, 0, fptr);
		break;
	case PM_RASTER_RAW:
		FD1_rasterFN(fdo, fd, l, 1, fptr);
		break;
	case PM_PSTH:
		FD1_psthFN(fdo, fd, l, fptr);
		break;
	case PM_ISIH:
		FD1_isih_stats_menu(fdo);
		FD1_isihFN(fdo, fd, l, fptr);
		break;
	default:
		pm_type_error("iid", view);
		break;
	}

	last_view = view;
	return (1);
}

static int iid_valid_view(FDO * fdo, int view)
{
	int retval = 0;
	int i, j;

	if (view == PM_DEFAULT) {
		retval = 1;
	} else {
		for (i = 0; i < nviews; i++) {
			for (j = 0; j < nsubviews[i]; j++) {
				if (view_order[i][j] == view) {
					retval = 1;
					fdo->view.lr = i;
					fdo->view.ud = j;
					break;
				}
			}
		}
	}

	return (retval);
}


static int iid_plotter(FDO *fdo, FILEDATA *fd, FDObj_ViewType *view,
			int l, FILE *fptr)
{
  syn_spec ss;
  fd_syn_spec_parse(FD_GV(fdo->fds[l], "iid.Stim"),0,&ss);

  adjust_index(view->lr,view->ud);

  if(l==0) {
    if(!((ss.class == SC_TONE) || 
          ((ss.class == SC_STACK) && (ss.parms.stack.num_freqs == 1)))) {
	  	if(view->lr == 1) {
	  		view->lr = 3;
	  		view->ud = 0; }
	  	else if(view->lr == 2) {
	  		view->lr = 0;
	  		view->ud = 0; } } }
  else {
    if(!((ss.class == SC_TONE) || 
          ((ss.class == SC_STACK) && (ss.parms.stack.num_freqs == 1))))
	  	if((view->lr == 1) || (view->lr == 2)) {
        notify("some curves missing due to different stim types");
        return(0); } }

  if(!fdo->no_X) {
  if(nsubviews[view->lr]>1) {
    XtSetSensitive(fdo->up, True);
    XtSetSensitive(fdo->down, True); } }

  iid_do_plot(fdo, fd, view_order[view->lr][view->ud], l, fptr);

  return(1);
}

void TwoLogistic(Widget w, XtPointer fdoptr, XtPointer call_data)
{
	FDO *fdo = (FDO *) fdoptr;

	int i, count, size;
	char *buf, buf2[128], *tmp;
	float *yfit, A, B, C, D, E, F, G, chi, bIID;
	int big;
	float *px, *py, *pz;
	int pn;
	float ndata;
	float *depvals;
	syn_spec ss;
	char *foo;
	float *means, *stderrs;

	assert((buf = (char *) malloc(1024 * sizeof(char))) != NULL);
	size = 1024;

	tmp = "equation:  A - B*logistic(C*(x-D)) - E*logistic(-F*(x-G))\n";
	strcpy(buf, tmp);
	count = strlen(tmp);
	tmp = "units:  A,B,E (spikes/stim); C,F (1/dB); D,G (dB IID)\n";
	strcat(buf, tmp);
	count += strlen(tmp);
	count += strlen(tmp);
	tmp = "file               A      B      C      D      E      F      G          chi  |   bIID\n";
	strcat(buf, tmp);
	count += strlen(tmp);
	tmp = "----               -      -      -      -      -      -      -          ---  |   ----\n";
	strcat(buf, tmp);
	count += strlen(tmp);

	for (i = 0; i < fdo->nfds; i++) {
		ndata = FD1_get_ndata(fdo->fds[i]);
		depvals = FD1_get_depvals(fdo->fds[i]);
		foo = FD_GV(fdo->fds[i], "itd.Stim");
		fd_syn_spec_parse(foo, 0, &ss);
		means = FD1_get_means(fdo->fds[i]);
		stderrs = FD1_get_stderrs(fdo->fds[i]);
		FitDataToTwoLogistic(depvals + 1, means + 1, stderrs + 1, ndata - 1,
			     (ss.class == SC_TONE) ? (int) (ss.parms.tone.freq) : 0,
        		&yfit, &A, &B, &C, &D, &E, &F, &G, &chi, &big, &bIID);

#ifdef __linux__
			count += sprintf(buf2, "%-13s  %5.1f  %5.1f  %5.1f  %5.3f  %5.1f  %5.1f  %5.3f   %5.1f%1s   %5.1f\n",
			    strrchr(fdo->fds[i]->filename, '/') == NULL ?
					 fdo->fds[i]->filename :
			1 + (char *) strrchr(fdo->fds[i]->filename, '/'),
			  A, B, C, D, E, F, G, chi, big ? "*" : " ",bIID);
#else
			count += strlen(sprintf(buf2, "%-13s  %5.1f  %5.1f  %5.3f  %5.1f  %5.1f  %5.3f  %5.1f   %5.1f%1s   %5.1f\n",
			    strrchr(fdo->fds[i]->filename, '/') == NULL ?
						fdo->fds[i]->filename :
			1 + (char *) strrchr(fdo->fds[i]->filename, '/'),
			  A, B, C, D, E, F, G, chi, big ? "*" : " "),bIID);
#endif
		if (fdo->to_tty) {
			printf("%s", buf2);
			continue;
		}
		strcat(buf, buf2);
		if ((size - count) < 512) {
			assert((buf = (char *) realloc(buf, (size + 1024) * sizeof(char))) != NULL);
			size += 1024;
		}
		FD_plotter_copy(depvals + 1, yfit, NULL, ndata - 1, 0,
		     &px, &py, &pz, &pn, NULL, NULL, NULL, NULL);

		if (!fdo->no_X) {
			FDObj_Add_Data_All(fdo, px, py, pn);
			FDObj_AddLine(fdo, i, px, py, pz, pn, AtFloat,
			     atQuadLinePlotWidgetClass, AtTypeLINEPOINTS,
				   AtMarkCIRCLE, ConvertColor(fdo->graph,
							      FDObj_Colors[(i+1) % FDObj_numColors]), (XtArgVal) "fit");
		}
	}

	tmp = "\n(*) indicates chi is big, and fitted parameters are questionable";
	strcat(buf, tmp);
	count += strlen(tmp);

	if (!fdo->to_tty) {
			pop_text("TwoLogistic", buf, strlen(buf) + 1, False);
	}
}


static void fit_menu(FDO *fdobj)
{
  if(fdobj->no_X) return;

  menubutton_clear(fdobj->fit,&(fdobj->fitpsh));
  XtSetSensitive(fdobj->fit, False);

  menubutton_add(fdobj->fitpsh,"peak stats", FD1_peak_stats,fdobj);
  menubutton_add(fdobj->fitpsh,"similarity stats", FD1_similarity_stats,fdobj);
  /*menubutton_add(fdobj->fitpsh,"linearity stats", LinearityStatsFN,fdobj);*/
  menubutton_add(fdobj->fitpsh,"combine",CombineDataFN,fdobj);
  menubutton_add(fdobj->fitpsh,"two logistic",TwoLogistic,fdobj);
  menubutton_add(fdobj->fitpsh,"prob distribution", FD1_prob_distribution,fdobj);
  XtSetSensitive(fdobj->fit, True);
}

int pm_iid_init(void)
{
  setFDOvalidviewMethod("iid", iid_valid_view);
  setFDOdoplotMethod("iid", iid_do_plot);
  setFDOplotMethod("iid", iid_plotter);
  setFDreadMethod("iid", FD1_reader);
  setFDfreeMethod("iid", FD1_free);
  return(1);
}



/************** from yuda *********************/


#define CHI2_STOP_STEP (0.01)

static int FitDataToTwoLogistic(
  float* fAxis,  /* x[] of curve to be fit */
  float* fData,  /* y[] of curve to be fit */
  float* fSig,   /* err[] of curve to be fit */
  int ndata,
  int nFreq,
  float **yfit,   /* y[] of resulting fit */
  /* parameters of resulting fit */
  float *A,
  float *B, float *C, float *D,
  float *E, float *F, float *G,
  float *chi, int *big, float *bIID)
{
	float *x, *y, *sig;
	int i, i0, *ia, ma;
	float *a, **covar, **alpha, alamda = -1, chisq=10, oldchi=1000;
	float x0, y0, y1, y2, s, s1, s2;
	ma = 7;

	alpha = matrix(1, ma, 1, ma);
	covar = matrix(1, ma, 1, ma);
	ia = ivector(1, ma);
	a = vector(1, ma);
	G_a = vector(1, ma);
	x = vector(1, ndata);
	y = vector(1, ndata);
	sig = vector(1, ndata);
	for (i = 0; i < ndata; i++) {
		sig[i + 1] = (fSig[i]>0.1)?fSig[i]:0.1;
		x[i + 1] = fAxis[i];
		y[i + 1] = fData[i];
	}

	for (x0 = x[1], y0 = y[1], i = 2; i <= ndata; i++) {
		if (y[i] > y0) {
			x0 = x[i], y0 = y[i], i0 = i; } }
	for (y1 = y[i0], s1=0, i = i0; i <= ndata; i++) {
		if (y[i] < y1) {
      y1 = y[i]; }
		s=-(y[i]-y[i+1])/(x[i]-x[i+1]);
		if ((i<ndata-1) && (s > s1)) {
      s1 = s; } }
	for (y2 = y[i0], s2=0, i = i0; i >= 1; i--) {
		if (y[i] < y2) {
      y2 = y[i]; }
		s=(y[i]-y[i-1])/(x[i]-x[i-1]);
		if ((i>2) && (s > s2)) {
      s2 = s; } }
	a[1] = y0;
  a[2] = y0-y1;
  a[5] = y0-y2;
	a[3] = 0.2/s1;
	a[6] = 0.2/s2;
	a[4] = x0+4/a[3];
	a[7] = x0-4/a[6];

	alamda = -1.0;
	for (i = 1; i <= ma; i++) { ia[i] = 1; }
  do {
		oldchi = chisq;
	  mrqmin(x,y,sig,ndata,a,ia,ma,covar,alpha,&chisq,FitTwoLogisticDerivatives,&alamda); }
	while ((((oldchi-chisq)>CHI2_STOP_STEP) || ((oldchi-chisq)<-CHI2_STOP_STEP) || (oldchi==chisq)) && (alamda<MAXFLOAT));

	alamda = 0.0;
	mrqmin(x,y,sig,ndata,a,ia,ma,covar,alpha,&chisq,FitTwoLogisticDerivatives,&alamda);

	assert(((*yfit) = (float *) malloc((ndata) * sizeof(float))) != NULL);
	for (i = 1; i <= ndata; i++) {
		(*yfit)[i - 1] = FitTwoLogisticPoint(x[i], a); }
	(*A) = a[1];
	(*B) = a[2]; (*C) = a[3]; (*D) = a[4];
	(*E) = a[5]; (*F) = a[6]; (*G) = a[7];
	(*chi) = chisq;
	(*big) = (chisq > ((ndata - ma) + (float) sqrt(2.0 * (ndata - ma)))) ? 1 : 0;

  for(i=1; i<=ma; i++) { G_a[i]=a[i]; }
  golden(a[4],a[7]+(a[4]-a[7])/2.0,a[7],FitTwoLogisticPointGolden,0.1,bIID);

/* the mean of the chi2 distribution is the degrees of freedom, i.e.,
   ndata - ma (ma is the number of parameters).  the standard deviation
   of the chi2 distribution is sqrt(2 * mean). thus, i give a warning
   when chi2 exceeds the mean + std.  */

	free_matrix(covar, 1, ma, 1, ma);
	free_matrix(alpha, 1, ma, 1, ma);
	free_ivector(ia, 1, ma);
	free_vector(a, 1, ma);
	free_vector(G_a, 1, ma);
	free_vector(x, 1, ndata);
	free_vector(y, 1, ndata);
	free_vector(sig, 1, ndata);
	fflush(stderr);
	return 1;
}

/*
  y = A - B/(1+exp(-C(x-D))) - E/(1+exp(F(x-G)))

  dy/dA = 1

  dy/dB = -1/(1+exp(-C(x-D)))
  dy/dC = -exp(-C(x-D)) * B(x+D) / (1+exp(-C(x-D)))^2
  dy/dD =  exp(-C(x-D)) * BC / (1+exp(-C(x-D)))^2

  dy/dE = -1/(1+exp(F(x-G)))
  dy/dF =  exp(F(x-G)) * E(x-G) / (1+exp(F(x-G)))^2
  dy/dG = -exp(F(x-G)) * EF / (1+exp(F(x-G)))^2
*/

void FitTwoLogisticDerivatives(float x, float a[], float *y, float dyda[], int na)
{
  *y = FitTwoLogisticPoint(x,a);

  dyda[1] = 1.0;

  dyda[2] = exp(-a[3]*(x-a[4]));
  dyda[3] = -dyda[2] * a[2]*(x-a[4]) / (1.0+dyda[2]) / (1.0+dyda[2]);
  dyda[4] =  dyda[2] * a[2]*a[3] / (1.0+dyda[2]) / (1.0+dyda[2]);
  dyda[2] = -1.0/(1.0+dyda[2]);

  dyda[5] = exp(a[6]*(x-a[7]));
  dyda[6] =  dyda[5] * a[5]*(x-a[7]) / (1.0+dyda[5]) / (1.0+dyda[5]);
  dyda[7] = -dyda[5] * a[5]*a[6] / (1.0+dyda[5]) / (1.0+dyda[5]);
  dyda[5] = -1.0/(1.0+dyda[5]);
}

static float FitTwoLogisticPointGolden(float x) {
  return(-FitTwoLogisticPoint(x,G_a)); }

static float FitTwoLogisticPoint(float x, float *a)
{
  return(a[1] - a[2]/(1.0+exp(-a[3]*(x-a[4]))) - a[5]/(1.0+exp(a[6]*(x-a[7]))));
}
