/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: pm_gen.c,v 1.11 2002/07/15 04:30:17 cmalek Exp $
** Program: xdphys
**  Module: pm_gen.c
**  Author: bjarthur
**
** Revision History (most recent last)
**
** 99.3 bjarthur
**  created from pm_rover
**
*******************************************************************/

#include "xdphyslib.h"
#include "xdphys.h"
#include "plotter.h"
#include "mod_gen_util.h"

static void gen_rasterFN(FDO *, FILEDATA *, int, FILE *);
static int gen_plotter(FDO *, FILEDATA *, FDObj_ViewType *, int, FILE *);
static int gen_valid_view(FDO *, int);
static void gen_curvesFN(FDO *, FILEDATA *, int, FILE *);
static void gen_rasterFN(FDO *, FILEDATA *, int, FILE *);
static int parsestring(char *, float *, float *, float *, float *, char *,
		       float *, float *);
static float *sort_stim(syn_spec *, int);
static int search_stim(char *, syn_spec *, int);
static int gen_summarize(FILEDATA *, int);
static void gen_freeSumData(FILEDATA *);
static int gen_free(FILEDATA *);

typedef struct {
	int nabis, nitds, niids, nbcs, nstims, nmonos, ntwo_snds;
	float *abis, *itds, *iids, *bcs, *f_stims, *monos, *two_snds;
	syn_spec *ss_stims;
	float *******means, *******stderrs, *******stddevs;
	float s_mean, s_stderr, s_stddev;
	int ********nspikes, *******n, *s_nspikes, s_n;
	char ********depstrs;
} SumData;

#define SUMDATA(data) ((SumData*)(data->sumdata))

static int view_order[][1] = {
	{PM_CURVE},
	{PM_RASTER},
	{PM_PSTH},
	{PM_ISIH}
};
static int nsubviews[] = { 1, 1, 1, 1 };
static int nviews = 4;

static int G_only_pure_tones;

int gen_do_plot(FDO * fdo, FILEDATA * fd, int view, int l, FILE * fptr)
{
	if (!gen_valid_view(fdo, view)) {
		pm_type_error("gen", view);
		view = PM_DEFAULT;
		(void) gen_valid_view(fdo, view);
	}

	switch (view) {
	case PM_DEFAULT:
	case PM_CURVE:
		gen_curvesFN(fdo, fd, l, fptr);
		break;
	case PM_RASTER:
		gen_rasterFN(fdo, fd, l, fptr);
		break;
	case PM_PSTH:
		FD1_psthFN(fdo, fd, l, fptr);
		break;
	case PM_ISIH:
		FD1_isih_stats_menu(fdo);
		FD1_isihFN(fdo, fd, l, fptr);
		break;
	default:
		pm_type_error("gen", view);
		break;
	}

	return (1);
}

static int gen_valid_view(FDO * fdo, int view)
{
	int retval = 0;
	int i, j;

	if (view == PM_DEFAULT) {
		retval = 1;
	} else {
		for (i = 0; i < nviews; i++) {
			for (j = 0; j < nsubviews[i]; j++) {
				if (view_order[i][j] == view) {
					retval = 1;
					fdo->view.lr = i;
					fdo->view.ud = j;
					break;
				}
			}
		}
	}

	return (retval);
}

static int gen_plotter(FDO * fdo, FILEDATA * fd, FDObj_ViewType * view,
		       int l, FILE * fptr)
{
	adjust_index(view->lr, view->ud);

	gen_do_plot(fdo, fd, view_order[view->lr][view->ud], l, fptr);

	return (1);
}

static void gen_curvesFN(FDO * fdo, FILEDATA * fd, int l, FILE * fptr)
{
	int i, j, k, ll, m, o, p;
	int ni, nj, nk, nl, nm, no, np;
	char *ti, buf[100], tmps[100];
	float f1;
	char *xlabel, *ylabel;
	float *px, *py, *pz;
	int pn;
	float *sx, *sy, *sz;
	int sn;
	float *tmpx, *tmpy, *tmpz;
	int max, type;
	char monoc;

	f1 = gen_summarize(fd, fdo->sum_code[l]);
	if (fdo->normalize == 1) {
		ylabel = "% max response";
		f1 = SUMDATA(fd)->means[0][0][0][0][0][0][0];
		for (i = 0; i < SUMDATA(fd)->nabis; i++) {
			for (j = 0; j < SUMDATA(fd)->nitds; j++) {
				for (k = 0; k < SUMDATA(fd)->niids; k++) {
					for (ll = 0; ll < SUMDATA(fd)->nbcs; ll++) {
						for (m = 0; m < SUMDATA(fd)->nstims; m++) {
							for (o = 0; m < SUMDATA(fd)-> nmonos; o++) {
								for (p = 0; m < SUMDATA (fd)-> ntwo_snds; p++) {
									if (SUMDATA(fd)->means[i][j][k][ll][m][o][p] > f1)
										f1 = SUMDATA(fd)->means[i][j][k][ll][m][o][p];
								}
							}
						}
					}
				}
			}
		}
		f1 /= 100.0;
	} else {
		ylabel = "nspikes";
		f1 = 1.0;
	}

	max = SUMDATA(fd)->nabis;
	type = 1;
	if (max < SUMDATA(fd)->nitds) {
		max = SUMDATA(fd)->nitds;
		type = 2;
	}
	if (max < SUMDATA(fd)->niids) {
		max = SUMDATA(fd)->niids;
		type = 3;
	}
	if (max < SUMDATA(fd)->nbcs) {
		max = SUMDATA(fd)->nbcs;
		type = 4;
	}
	if ((max < SUMDATA(fd)->nstims) && G_only_pure_tones) {
		max = SUMDATA(fd)->nstims;
		type = 5;
	}

	switch (type) {
	case (1):
		xlabel = "abi (dB)";
		nm = SUMDATA(fd)->nabis;
		tmpx = SUMDATA(fd)->abis;
		ni = SUMDATA(fd)->nitds;
		nj = SUMDATA(fd)->niids;
		nk = SUMDATA(fd)->nbcs;
		nl = SUMDATA(fd)->nstims;
		no = SUMDATA(fd)->nmonos;
		np = SUMDATA(fd)->ntwo_snds;
		break;
	case (2):
		xlabel = "itd (us)";
		nm = SUMDATA(fd)->nitds;
		tmpx = SUMDATA(fd)->itds;
		ni = SUMDATA(fd)->nabis;
		nj = SUMDATA(fd)->niids;
		nk = SUMDATA(fd)->nbcs;
		nl = SUMDATA(fd)->nstims;
		no = SUMDATA(fd)->nmonos;
		np = SUMDATA(fd)->ntwo_snds;
		break;
	case (3):
		xlabel = "iid (dB)";
		nm = SUMDATA(fd)->niids;
		tmpx = SUMDATA(fd)->iids;
		ni = SUMDATA(fd)->nabis;
		nj = SUMDATA(fd)->nitds;
		nk = SUMDATA(fd)->nbcs;
		nl = SUMDATA(fd)->nstims;
		no = SUMDATA(fd)->nmonos;
		np = SUMDATA(fd)->ntwo_snds;
		break;
	case (4):
		xlabel = "bc (%)";
		nm = SUMDATA(fd)->nbcs;
		tmpx = SUMDATA(fd)->bcs;
		ni = SUMDATA(fd)->nabis;
		nj = SUMDATA(fd)->nitds;
		nk = SUMDATA(fd)->niids;
		nl = SUMDATA(fd)->nstims;
		no = SUMDATA(fd)->nmonos;
		np = SUMDATA(fd)->ntwo_snds;
		break;
	case (5):
		xlabel = "bf (Hz)";
		nm = SUMDATA(fd)->nstims;
		tmpx = SUMDATA(fd)->f_stims;
		ni = SUMDATA(fd)->nabis;
		nj = SUMDATA(fd)->nitds;
		nk = SUMDATA(fd)->niids;
		nl = SUMDATA(fd)->nbcs;
		no = SUMDATA(fd)->nmonos;
		np = SUMDATA(fd)->ntwo_snds;
		break;
	}

	assert((tmpy = (float *) malloc(nm * sizeof(float))) != NULL);
	assert((tmpz = (float *) malloc(nm * sizeof(float))) != NULL);
	for (i = 0; i < ni; i++) {
		for (j = 0; j < nj; j++) {
			for (k = 0; k < nk; k++) {
				for (ll = 0; ll < nl; ll++) {
					for (o = 0; o < no; o++) {
						for (p = 0; p < np; p++) {
							for (m = 0; m < nm; m++) {
								switch(type)
								{
								case (1):
									tmpy[m] = SUMDATA(fd)->means[m][i][j][k][ll][o][p];
									tmpz[m] = SUMDATA(fd)->stderrs[m][i][j][k][ll][o][p];
									break;
								case (2):
									tmpy[m] = SUMDATA(fd)->means[i][m][j][k][ll][o][p];
									tmpz[m] = SUMDATA(fd)->stderrs[i][m][j][k][ll][o][p];
									break;
								case (3):
									tmpy[m] = SUMDATA(fd)->means[i][j][m][k][ll][o][p];
									tmpz[m] = SUMDATA(fd)->stderrs[i][j][m][k][ll][o][p];
									break;
								case (4):
									tmpy[m] = SUMDATA(fd)->means[i][j][k][m][ll][o][p];
									tmpz[m] = SUMDATA(fd)->stderrs[i][j][k][m][ll][o][p];
									break;
								case (5):
									tmpy[m] = SUMDATA(fd)->means[i][j][k][ll][m][o][p];
									tmpz[m] = SUMDATA(fd)->stderrs[i][j][k][ll][m][o][p];
									break;
								}
							}

							strcpy(buf, "");
							if (SUMDATA(fd)-> abis != tmpx) {
								sprintf(tmps, "%d;", (int) (SUMDATA (fd)->
								      abis [i]));
								strcat(buf, tmps);
							}
							if (SUMDATA(fd)-> itds != tmpx) {
								sprintf(tmps, "%d;", (int) (SUMDATA
								      (fd)-> itds[(type < 2) ? i : j]));
								strcat(buf, tmps);
							}
							if (SUMDATA(fd)-> iids != tmpx) {
								sprintf(tmps, "%d;", (int) (SUMDATA (fd)->
								      iids[(type < 3) ? j : k]));
								strcat(buf, tmps);
							}
							if (SUMDATA(fd)-> bcs != tmpx) {
								sprintf(tmps, "%d;", (int) (SUMDATA (fd)->
								      bcs[(type < 4) ? k : ll]));
								strcat(buf, tmps);
							}
							if (SUMDATA(fd)-> f_stims != tmpx) {
								sprintf(tmps, "%s;", SUMDATA (fd)-> ss_stims
								     [ll].  descr);
								strcat(buf, tmps);
							}

							switch ((int) (SUMDATA (fd)-> monos[o]))
							{
							case (SYN_LEFT):
								monoc = 'L';
								break;
							case (SYN_RIGHT):
								monoc = 'R';
								break;
							case (SYN_BOTH):
								monoc = 'B';
								break;
							}

							sprintf(tmps, "%c;", monoc);
							strcat(buf, tmps);
							sprintf(tmps, "%d", (int) (SUMDATA (fd)-> two_snds
								 [p]));
							strcat(buf, tmps);

							FD_plotter_copy (tmpx, tmpy, !(fdo-> no_errs) ?
							     tmpz : NULL, nm, 0, &px, &py, &pz, &pn,
							     NULL, NULL, NULL, NULL);

							if (fdo->no_X)
								continue;

							FDObj_Add_Data_All (fdo, px, py, pn);
							FDObj_AddLine(fdo, l, px, py, pz, pn, AtFloat,
								      atQuadLinePlotWidgetClass,
								      AtTypeLINEPOINTS,
								      AtMarkCIRCLE,
								      ConvertColor(fdo-> graph, FDObj_Colors[(i * nj * nk * nl * no * np + j * nk * nl * no * np + k * nl * no * np + ll * no * np + o * np + p) % FDObj_numColors]),
								      (XtArgVal) buf);
						}
					}
				}
			}
		}
	}
	free(tmpy);
	free(tmpz);

	assert((sx = (float *) calloc(2, sizeof(float))) != NULL);
	assert((sy = (float *) calloc(2, sizeof(float))) != NULL);
	sx[0] = px[0];
	sx[1] = px[pn - 1];
	sy[0] = sy[1] = SUMDATA(fd)->s_mean;
	if (!fdo->no_errs) {
		assert((sz = (float *) calloc(2, sizeof(float))) != NULL);
		sz[0] = sz[1] = SUMDATA(fd)->s_stderr;
	} else
		sz = NULL;
	sn = 2;

	if (fdo->no_X)
		return;

	FDObj_Add_Data_All(fdo, sx, sy, sn);
	FDObj_AddLine(fdo, l, sx, sy, sz, sn, AtFloat,
		      atQuadLinePlotWidgetClass, AtTypeLINEPOINTS,
		      AtMarkCIRCLE, ConvertColor(fdo->graph,
						 FDObj_Colors[(ni * nj * nk * nl * no * np) %
							      FDObj_numColors]),
		      (XtArgVal) "spont");

	ti = "curves";
	XtVaSetValues(fdo->graph, XtNtitle, FDObj_Title(fdo, ti, l),
		      XtNshowTitle, True, NULL);
	XtVaSetValues(fdo->xaxis, XtNlabel, xlabel, NULL);
	XtVaSetValues(fdo->yaxis, XtNlabel, ylabel, NULL);
}

static void gen_rasterFN(FDO * fdo, FILEDATA * fd, int l, FILE * fptr)
{
	int i, j, k;
	char *ti;
	SA_SpikeRasterList *srl = NULL;
	char *xlabel, *ylabel;
	float *xdata, *ydata;
	int ndata;
	double xmin, xmax;

	srl = FD_to_SpikeRasterList(FD1_RAWDATA(fdo->fds[l])->nrasters,
				    FD1_RAWDATA(fdo->fds[l])->rastlist,
				    FD_GI(fdo->fds[l], "epoch"),
				    fdo->sum_code[l]);
	ndata = SA_TotalSpikesInSRL(srl);
	assert((xdata = (float *) malloc(sizeof(float) * ndata)) != NULL);
	assert((ydata = (float *) malloc(sizeof(float) * ndata)) != NULL);

	for (k = i = 0; i < srl->nrasters; i++) {
		for (j = 0; j < srl->sr[i]->nspikes; j++) {
			xdata[k] = srl->sr[i]->tm[j];
			ydata[k++] = FD1_RAWDATA(fd)->pres_order[i];
		}
	}

	SA_FreeSpikeRasterList(srl);

	if (fdo->no_X)
		return;

	FDObj_Add_Data_All(fdo, xdata, ydata, ndata);
	FDObj_AddLine(fdo, l, xdata, ydata, NULL, ndata, AtFloat,
		      atQuadLinePlotWidgetClass, AtTypePOINTS, AtMarkVBAR,
		      ConvertColor(fdo->graph,
				   FDObj_Colors[l % FDObj_numColors]),
		      FDObj_Legend(fdo, l));

	xmin = 0.0;
	xmax = (double) FD_GI(fd, "epoch");
	XtVaSetValues(fdo->xaxis, XtNmin, &xmin, XtNmax, &xmax,
		      XtNautoScale, False, NULL);
	xmin = 0.0;
	xmax = (double) srl->nrasters;
	XtVaSetValues(fdo->yaxis, XtNmin, &xmin, XtNmax, &xmax,
		      XtNautoScale, False, NULL);

	ylabel = "raster # (pres order)";
	ti = "rawraster";
	XtVaSetValues(fdo->graph, XtNtitle, FDObj_Title(fdo, ti, l),
		      XtNshowTitle, True, NULL);
	xlabel = "time (ms)";
	XtVaSetValues(fdo->xaxis, XtNlabel, xlabel, NULL);
	XtVaSetValues(fdo->yaxis, XtNlabel, ylabel, NULL);
}

static int parsestring(char *string, float *abi, float *itd, float *iid,
		       float *bc, char *stim, float *mono, float *two_snd)
{
	char i, *p, q[128];
	char monoc;

	assert(strlen(string) < 128);
	strcpy(q, string);

	for (p = strtok(index(q, '<') + 1, ";"), i = 0; p != NULL;
	     p = strtok(NULL, ";"), i++) {
		switch (i) {
		case (0):
			sscanf(p, " %f", abi);
			break;
		case (1):
			sscanf(p, " %f", itd);
			break;
		case (2):
			sscanf(p, " %f", iid);
			break;
		case (3):
			sscanf(p, " %f", bc);
			break;
		case (4):
			strcpy(stim, p + 1);
			break;
		case (5):
			sscanf(p, " %c", &monoc);
			break;
		case (6):
			sscanf(p, " %f", two_snd);
			break;
		}
	}

	if (i == 5) {		/* backwards compatibility */
		stim[strlen(stim) - 1] = '\0';
		return (1);
	} else {
		switch (monoc) {
		case ('L'):
			(*mono) = SYN_LEFT;
			break;
		case ('R'):
			(*mono) = SYN_RIGHT;
			break;
		case ('B'):
			(*mono) = SYN_BOTH;
			break;
		}
		return (0);
	}
}

static float *sort_stim(syn_spec * ssarray, int nssarray)
{
	int i, j, k;
	float min;
	syn_spec tmp;
	float *ret_val;

	assert((ret_val =
		(float *) malloc(nssarray * sizeof(float))) != NULL);

	for (i = 0; i < (nssarray - 1); i++) {
		assert(ssarray[i].class == SC_TONE);
		min = ssarray[i].parms.tone.freq;
		k = i;
		for (j = (i + 1); j < nssarray; j++) {
			if (min > ssarray[j].parms.tone.freq) {
				min = ssarray[j].parms.tone.freq;
				k = j;
			}
		}
		tmp = ssarray[i];
		ssarray[i] = ssarray[k];
		ssarray[k] = tmp;
		ret_val[i] = ssarray[i].parms.tone.freq;
	}

	assert(ssarray[i].class == SC_TONE);
	ret_val[i] = ssarray[i].parms.tone.freq;

	return (ret_val);
}

static int search_stim(char *item, syn_spec * ssarray, int nssarray)
{
	int i;
	syn_spec ss;

	fd_syn_spec_parse(item, 0, &ss);

	for (i = 0; i < nssarray; i++) {
		if (syn_spec_same(&ss, ssarray + i)) {
			syn_spec_free(&ss);
			return (i);
		}
	}

	syn_spec_free(&ss);
	return (-1);
}

#define convert_array(ret_val,data,n,to_type,alloc_flag,free_flag) { \
  int z; /* z must not be used as an index into any of the input args */ \
 \
  if(alloc_flag) \
    assert((ret_val=(to_type*)malloc(n*sizeof(to_type)))!=NULL); \
 \
  for(z=0; z<n; z++) \
    ret_val[z] = (to_type)(data[z]); \
 \
  if(free_flag) \
    free(data); \
}


static int gen_summarize(FILEDATA * fd, int unitid)
{
	int i, j, k, l, m, o, p;
	int r1, r2, r3, r4, r5, r6, r7;
	float *p1, *p2, *p3, *p4, *p6, *p7;
	float abi, iid, itd, bc, mono, two_snd;
	char stim[1024], monoc, tmps[128];
	float ********data, *spont_resp;
	int *******num;
	int num_reps;
	int use_delay;		/* use the pre-stim delay as spontaneous */
	float t_begin;		/* driven begins at in ms (t_ in ticks) */
	float t_end;		/* driven ends at in ms (t_ in ticks) */
	float t_spont_begin;	/* spont begins at in ms (t_ in ticks) */
	float t_spont_end;	/* spont ends at in ms (t_ in ticks) */
	int si, spont_stims;
	int *tmp;
	char *mono_range, *itd_range, *iid_range, *two_snd_range;
	int err;

	gen_freeSumData(fd);
	assert((SUMDATA(fd) =
		(SumData *) calloc(1, sizeof(SumData))) != NULL);

	num_reps = FD_GI(fd, "Reps");

	/* compute nitds, niids, itds, iids, etc... */
	tmp =
	    StimArray_Gen(&(SUMDATA(fd)->nabis), 0, FD_GV(fd, "gen.abi"),
			  1, 0, 0);
	convert_array(SUMDATA(fd)->abis, tmp, SUMDATA(fd)->nabis, float, 1,
		      1);
	qsort(SUMDATA(fd)->abis, SUMDATA(fd)->nabis, sizeof(float),
	      floatcompare);

	assert(strlen(FD_GV(fd, "gen.stim")) < 1024);
	strcpy(stim, FD_GV(fd, "gen.stim"));
	if ((SUMDATA(fd)->ss_stims =
	     gen_SSArray_Gen(&(SUMDATA(fd)->nstims), 0, stim, 1, 0,
			     0)) == NULL)
		return (0);
	if ((G_only_pure_tones =
	     only_this_stim(SUMDATA(fd)->ss_stims, SUMDATA(fd)->nstims,
			    SC_TONE)))
		SUMDATA(fd)->f_stims =
		    sort_stim(SUMDATA(fd)->ss_stims, SUMDATA(fd)->nstims);

	mono_range = FD_GV(fd, "gen.mono");
	itd_range = FD_GV(fd, "gen.itd");
	iid_range = FD_GV(fd, "gen.iid");
	if (((strchr(mono_range, 'l') != NULL)
	     || (strchr(mono_range, 'L') != NULL)
	     || (strchr(mono_range, 'r') != NULL)
	     || (strchr(mono_range, 'R') != NULL))
	    && (strcmp(itd_range, "0") || strcmp(iid_range, "0"))) {
		itd_range = "0";
		iid_range = "0";
	}

	tmp = gen_itd_StimArray_Gen(&(SUMDATA(fd)->nitds), 0, itd_range,
				    1, 0, 0, SUMDATA(fd)->ss_stims,
				    SUMDATA(fd)->nstims);
	convert_array(SUMDATA(fd)->itds, tmp, SUMDATA(fd)->nitds, float, 1,
		      1);
	qsort(SUMDATA(fd)->itds, SUMDATA(fd)->nitds, sizeof(float),
	      floatcompare);

	tmp = StimArray_Gen(&(SUMDATA(fd)->niids), 0, iid_range, 1, 0, 0);
	convert_array(SUMDATA(fd)->iids, tmp, SUMDATA(fd)->niids, float, 1,
		      1);
	qsort(SUMDATA(fd)->iids, SUMDATA(fd)->niids, sizeof(float),
	      floatcompare);

	tmp =
	    StimArray_Gen(&(SUMDATA(fd)->nbcs), 0, FD_GV(fd, "gen.bc"), 1,
			  0, 0);
	convert_array(SUMDATA(fd)->bcs, tmp, SUMDATA(fd)->nbcs, float, 1,
		      1);
	qsort(SUMDATA(fd)->bcs, SUMDATA(fd)->nbcs, sizeof(float),
	      floatcompare);

	tmp =
	    gen_mono_StimArray_Gen(&(SUMDATA(fd)->nmonos), 0, mono_range,
				   1, 0, 0);
	convert_array(SUMDATA(fd)->monos, tmp, SUMDATA(fd)->nmonos, float,
		      1, 1);
	qsort(SUMDATA(fd)->monos, SUMDATA(fd)->nmonos, sizeof(float),
	      floatcompare);

	if ((two_snd_range = FD_GV(fd, "gen.two_snd")) == NULL)
		two_snd_range = FD_GV(fd, "ts.on/off");	/* backwards compatibility */
	tmp =
	    StimArray_Gen(&(SUMDATA(fd)->ntwo_snds), 0, two_snd_range, 1,
			  0, 0);
	convert_array(SUMDATA(fd)->two_snds, tmp, SUMDATA(fd)->ntwo_snds,
		      float, 1, 1);
	qsort(SUMDATA(fd)->two_snds, SUMDATA(fd)->ntwo_snds, sizeof(float),
	      floatcompare);


	/* alloc mem for means, stderrs, & stddevs */
	assert((SUMDATA(fd)->means =
		(float *******) malloc(SUMDATA(fd)->nabis *
				       sizeof(float ******))) != NULL);
	for (i = 0; i < SUMDATA(fd)->nabis; i++) {
		assert((SUMDATA(fd)->means[i] =
			(float ******) malloc(SUMDATA(fd)->nitds *
					      sizeof(float *****))) !=
		       NULL);
		for (j = 0; j < SUMDATA(fd)->nitds; j++) {
			assert((SUMDATA(fd)->means[i][j] =
				(float *****) malloc(SUMDATA(fd)->niids *
						     sizeof(float ****)))
			       != NULL);
			for (k = 0; k < SUMDATA(fd)->niids; k++) {
				assert((SUMDATA(fd)->means[i][j][k] =
					(float ****) malloc(SUMDATA(fd)->
							    nbcs *
							    sizeof(float
								   ***)))
				       != NULL);
				for (l = 0; l < SUMDATA(fd)->nbcs; l++) {
					assert((SUMDATA(fd)->
						means[i][j][k][l] =
						(float ***)
						malloc(SUMDATA(fd)->
						       nstims *
						       sizeof(float **)))
					       != NULL);
					for (o = 0;
					     o < SUMDATA(fd)->nstims;
					     o++) {
						assert((SUMDATA(fd)->
							means[i][j][k][l]
							[o] =
							(float **)
							malloc(SUMDATA
							       (fd)->
							       nmonos *
							       sizeof(float
								      *)))
						       != NULL);
						for (p = 0;
						     p <
						     SUMDATA(fd)->nmonos;
						     p++) {
							assert((SUMDATA
								(fd)->
								means[i][j]
								[k][l][o]
								[p] =
								(float *)
								malloc
								(SUMDATA
								 (fd)->
								 ntwo_snds
								 *
								 sizeof
								 (float)))
							       != NULL);
	}}}}}}

	assert((SUMDATA(fd)->stderrs =
		(float *******) malloc(SUMDATA(fd)->nabis *
				       sizeof(float ******))) != NULL);
	for (i = 0; i < SUMDATA(fd)->nabis; i++) {
		assert((SUMDATA(fd)->stderrs[i] =
			(float ******) malloc(SUMDATA(fd)->nitds *
					      sizeof(float *****))) !=
		       NULL);
		for (j = 0; j < SUMDATA(fd)->nitds; j++) {
			assert((SUMDATA(fd)->stderrs[i][j] =
				(float *****) malloc(SUMDATA(fd)->niids *
						     sizeof(float ****)))
			       != NULL);
			for (k = 0; k < SUMDATA(fd)->niids; k++) {
				assert((SUMDATA(fd)->stderrs[i][j][k] =
					(float ****) malloc(SUMDATA(fd)->
							    nbcs *
							    sizeof(float
								   ***)))
				       != NULL);
				for (l = 0; l < SUMDATA(fd)->nbcs; l++) {
					assert((SUMDATA(fd)->
						stderrs[i][j][k][l] =
						(float ***)
						malloc(SUMDATA(fd)->
						       nstims *
						       sizeof(float **)))
					       != NULL);
					for (o = 0;
					     o < SUMDATA(fd)->nstims;
					     o++) {
						assert((SUMDATA(fd)->
							stderrs[i][j][k][l]
							[o] =
							(float **)
							malloc(SUMDATA
							       (fd)->
							       nmonos *
							       sizeof(float
								      *)))
						       != NULL);
						for (p = 0;
						     p <
						     SUMDATA(fd)->nmonos;
						     p++) {
							assert((SUMDATA
								(fd)->
								stderrs[i]
								[j][k][l]
								[o][p] =
								(float *)
								malloc
								(SUMDATA
								 (fd)->
								 ntwo_snds
								 *
								 sizeof
								 (float)))
							       != NULL);
	}}}}}}

	assert((SUMDATA(fd)->stddevs =
		(float *******) malloc(SUMDATA(fd)->nabis *
				       sizeof(float ******))) != NULL);
	for (i = 0; i < SUMDATA(fd)->nabis; i++) {
		assert((SUMDATA(fd)->stddevs[i] =
			(float ******) malloc(SUMDATA(fd)->nitds *
					      sizeof(float *****))) !=
		       NULL);
		for (j = 0; j < SUMDATA(fd)->nitds; j++) {
			assert((SUMDATA(fd)->stddevs[i][j] =
				(float *****) malloc(SUMDATA(fd)->niids *
						     sizeof(float ****)))
			       != NULL);
			for (k = 0; k < SUMDATA(fd)->niids; k++) {
				assert((SUMDATA(fd)->stddevs[i][j][k] =
					(float ****) malloc(SUMDATA(fd)->
							    nbcs *
							    sizeof(float
								   ***)))
				       != NULL);
				for (l = 0; l < SUMDATA(fd)->nbcs; l++) {
					assert((SUMDATA(fd)->
						stddevs[i][j][k][l] =
						(float ***)
						malloc(SUMDATA(fd)->
						       nstims *
						       sizeof(float **)))
					       != NULL);
					for (o = 0;
					     o < SUMDATA(fd)->nstims;
					     o++) {
						assert((SUMDATA(fd)->
							stddevs[i][j][k][l]
							[o] =
							(float **)
							malloc(SUMDATA
							       (fd)->
							       nmonos *
							       sizeof(float
								      *)))
						       != NULL);
						for (p = 0;
						     p <
						     SUMDATA(fd)->nmonos;
						     p++) {
							assert((SUMDATA
								(fd)->
								stddevs[i]
								[j][k][l]
								[o][p] =
								(float *)
								malloc
								(SUMDATA
								 (fd)->
								 ntwo_snds
								 *
								 sizeof
								 (float)))
							       != NULL);
	}}}}}}

	assert((SUMDATA(fd)->nspikes =
		(int ********) malloc(SUMDATA(fd)->nabis *
				      sizeof(int *******))) != NULL);
	for (i = 0; i < SUMDATA(fd)->nabis; i++) {
		assert((SUMDATA(fd)->nspikes[i] =
			(int *******) malloc(SUMDATA(fd)->nitds *
					     sizeof(int ******))) != NULL);
		for (j = 0; j < SUMDATA(fd)->nitds; j++) {
			assert((SUMDATA(fd)->nspikes[i][j] =
				(int ******) malloc(SUMDATA(fd)->niids *
						    sizeof(int *****))) !=
			       NULL);
			for (k = 0; k < SUMDATA(fd)->niids; k++) {
				assert((SUMDATA(fd)->nspikes[i][j][k] =
					(int *****) malloc(SUMDATA(fd)->
							   nbcs *
							   sizeof(int
								  ****)))
				       != NULL);
				for (l = 0; l < SUMDATA(fd)->nbcs; l++) {
					assert((SUMDATA(fd)->
						nspikes[i][j][k][l] =
						(int ****)
						malloc(SUMDATA(fd)->
						       nstims *
						       sizeof(int ***))) !=
					       NULL);
					for (m = 0;
					     m < SUMDATA(fd)->nstims;
					     m++) {
						assert((SUMDATA(fd)->
							nspikes[i][j][k][l]
							[m] =
							(int ***)
							malloc(SUMDATA
							       (fd)->
							       nmonos *
							       sizeof(int
								      **)))
						       != NULL);
						for (o = 0;
						     o <
						     SUMDATA(fd)->nmonos;
						     o++) {
							assert((SUMDATA
								(fd)->
								nspikes[i]
								[j][k][l]
								[m][o] =
								(int **)
								malloc
								(SUMDATA
								 (fd)->
								 ntwo_snds
								 *
								 sizeof(int
									*)))
							       != NULL);
	}}}}}}

	assert((SUMDATA(fd)->depstrs =
		(char ********) malloc(SUMDATA(fd)->nabis *
				       sizeof(char *******))) != NULL);
	for (i = 0; i < SUMDATA(fd)->nabis; i++) {
		assert((SUMDATA(fd)->depstrs[i] =
			(char *******) malloc(SUMDATA(fd)->nitds *
					      sizeof(char ******))) !=
		       NULL);
		for (j = 0; j < SUMDATA(fd)->nitds; j++) {
			assert((SUMDATA(fd)->depstrs[i][j] =
				(char ******) malloc(SUMDATA(fd)->niids *
						     sizeof(char *****)))
			       != NULL);
			for (k = 0; k < SUMDATA(fd)->niids; k++) {
				assert((SUMDATA(fd)->depstrs[i][j][k] =
					(char *****) malloc(SUMDATA(fd)->
							    nbcs *
							    sizeof(char
								   ****)))
				       != NULL);
				for (l = 0; l < SUMDATA(fd)->nbcs; l++) {
					assert((SUMDATA(fd)->
						depstrs[i][j][k][l] =
						(char ****)
						malloc(SUMDATA(fd)->
						       nstims *
						       sizeof(char ***)))
					       != NULL);
					for (m = 0;
					     m < SUMDATA(fd)->nstims;
					     m++) {
						assert((SUMDATA(fd)->
							depstrs[i][j][k][l]
							[m] =
							(char ***)
							malloc(SUMDATA
							       (fd)->
							       nmonos *
							       sizeof(char
								      **)))
						       != NULL);
						for (o = 0;
						     o <
						     SUMDATA(fd)->nmonos;
						     o++) {
							assert((SUMDATA
								(fd)->
								depstrs[i]
								[j][k][l]
								[m][o] =
								(char **)
								malloc
								(SUMDATA
								 (fd)->
								 ntwo_snds
								 *
								 sizeof
								 (char *)))
							       != NULL);
	}}}}}}


	/* alloc temp storage for raw data */
	assert((data =
		(float ********) malloc(SUMDATA(fd)->nabis *
					sizeof(float *******))) != NULL);
	for (i = 0; i < SUMDATA(fd)->nabis; i++) {
		assert((data[i] =
			(float *******) malloc(SUMDATA(fd)->nitds *
					       sizeof(float ******))) !=
		       NULL);
		for (j = 0; j < SUMDATA(fd)->nitds; j++) {
			assert((data[i][j] =
				(float ******) malloc(SUMDATA(fd)->niids *
						      sizeof(float *****)))
			       != NULL);
			for (k = 0; k < SUMDATA(fd)->niids; k++) {
				assert((data[i][j][k] =
					(float *****) malloc(SUMDATA(fd)->
							     nbcs *
							     sizeof(float
								    ****)))
				       != NULL);
				for (l = 0; l < SUMDATA(fd)->nbcs; l++) {
					assert((data[i][j][k][l] =
						(float ****)
						malloc(SUMDATA(fd)->
						       nstims *
						       sizeof(float ***)))
					       != NULL);
					for (m = 0;
					     m < SUMDATA(fd)->nstims;
					     m++) {
						assert((data[i][j][k][l][m]
							=
							(float ***)
							malloc(SUMDATA
							       (fd)->
							       nmonos *
							       sizeof(float
								      **)))
						       != NULL);
						for (o = 0;
						     o <
						     SUMDATA(fd)->nmonos;
						     o++) {
							assert((data[i][j]
								[k][l][m]
								[o] =
								(float **)
								malloc
								(SUMDATA
								 (fd)->
								 ntwo_snds
								 *
								 sizeof
								 (float
								  *))) !=
							       NULL);
							for (p = 0;
							     p <
							     SUMDATA(fd)->
							     ntwo_snds;
							     p++) {
								assert((data[i][j][k][l][m][o][p] = (float *) malloc(num_reps * sizeof(float))) != NULL);
	}}}}}}}

	assert((num =
		(int *******) calloc(SUMDATA(fd)->nabis,
				     sizeof(int ******))) != NULL);
	for (i = 0; i < SUMDATA(fd)->nabis; i++) {
		assert((num[i] =
			(int ******) calloc(SUMDATA(fd)->nitds,
					    sizeof(int *****))) != NULL);
		for (j = 0; j < SUMDATA(fd)->nitds; j++) {
			assert((num[i][j] =
				(int *****) calloc(SUMDATA(fd)->niids,
						   sizeof(int ****))) !=
			       NULL);
			for (k = 0; k < SUMDATA(fd)->niids; k++) {
				assert((num[i][j][k] =
					(int ****) calloc(SUMDATA(fd)->
							  nbcs,
							  sizeof(int ***)))
				       != NULL);
				for (l = 0; l < SUMDATA(fd)->nbcs; l++) {
					assert((num[i][j][k][l] =
						(int ***)
						calloc(SUMDATA(fd)->nstims,
						       sizeof(int **))) !=
					       NULL);
					for (o = 0;
					     o < SUMDATA(fd)->nstims;
					     o++) {
						assert((num[i][j][k][l][o]
							=
							(int **)
							calloc(SUMDATA
							       (fd)->
							       nmonos,
							       sizeof(int
								      *)))
						       != NULL);
						for (p = 0;
						     p <
						     SUMDATA(fd)->nmonos;
						     p++) {
							assert((num[i][j]
								[k][l][o]
								[p] =
								(int *)
								calloc
								(SUMDATA
								 (fd)->
								 ntwo_snds,
								 sizeof
								 (int))) !=
							       NULL);
	}}}}}}


	FD_calc_params(fd, &spont_stims, &use_delay,
		       &t_begin, &t_end, &t_spont_begin, &t_spont_end);

	if ((!spont_stims) || use_delay)
		assert((spont_resp = (float *) malloc((FD1_RAWDATA(fd)->
						       nrasters) *
						      sizeof(float))) !=
		       NULL);
	else
		assert((spont_resp =
			(float *) malloc(num_reps * sizeof(float))) !=
		       NULL);

	/* sort rasters into array, and compute stats */
	for (i = si = 0; i < FD1_RAWDATA(fd)->nrasters; i++) {

		if ((!spont_stims) || use_delay) {
			if (FD1_RAWDATA(fd)->depints[i] != SPONT) {
				spont_resp[si] =
				    (float) countRaster(t_spont_begin,
							t_spont_end,
							FD1_RAWDATA(fd)->
							rastlist[i],
							unitid);
				spont_resp[si] *=
				    ((t_end - t_begin) / (t_spont_end -
							  t_spont_begin));
				si++;
			} else {
				continue;
			}
		} else if (FD1_RAWDATA(fd)->depints[i] == SPONT) {
			spont_resp[si] =
			    (float) countRaster(t_begin, t_end,
						FD1_RAWDATA(fd)->
						rastlist[i], unitid);
			si++;
			continue;
		}

		err = parsestring(FD1_RAWDATA(fd)->deplines[i],
				  &abi, &itd, &iid, &bc, stim, &mono,
				  &two_snd);

		p1 = (float *) bsearch(&abi, SUMDATA(fd)->abis,
				       SUMDATA(fd)->nabis, sizeof(float),
				       floatcompare);
		r1 = p1 - SUMDATA(fd)->abis;

		p2 = (float *) bsearch(&itd, SUMDATA(fd)->itds,
				       SUMDATA(fd)->nitds, sizeof(float),
				       floatcompare);
		r2 = p2 - SUMDATA(fd)->itds;

		p3 = (float *) bsearch(&iid, SUMDATA(fd)->iids,
				       SUMDATA(fd)->niids, sizeof(float),
				       floatcompare);
		r3 = p3 - SUMDATA(fd)->iids;

		p4 = (float *) bsearch(&bc, SUMDATA(fd)->bcs,
				       SUMDATA(fd)->nbcs, sizeof(float),
				       floatcompare);
		r4 = p4 - SUMDATA(fd)->bcs;

		r5 = search_stim(stim, SUMDATA(fd)->ss_stims,
				 SUMDATA(fd)->nstims);

		if (!err) {
			p6 = (float *) bsearch(&mono, SUMDATA(fd)->monos,
					       SUMDATA(fd)->nmonos,
					       sizeof(float),
					       floatcompare);
			r6 = p6 - SUMDATA(fd)->monos;

			p7 = (float *) bsearch(&two_snd,
					       SUMDATA(fd)->two_snds,
					       SUMDATA(fd)->ntwo_snds,
					       sizeof(float),
					       floatcompare);
			r7 = p7 - SUMDATA(fd)->two_snds;
		} else {
			r6 = r7 = 0;
		}

		data[r1][r2][r3][r4][r5][r6][r7][num[r1][r2][r3][r4][r5][r6][r7]++] =
		    (float) countRaster(t_begin, t_end, FD1_RAWDATA(fd)->rastlist[i],
					unitid);
	}

	SUMDATA(fd)->s_mean = mean(spont_resp, si);
	SUMDATA(fd)->s_stderr = stderror(spont_resp, si);
	SUMDATA(fd)->s_stddev = stddev(spont_resp, si);
	convert_array(SUMDATA(fd)->s_nspikes, spont_resp, si, int, 1, 1);
	SUMDATA(fd)->s_n = si;

	for (i = 0; i < SUMDATA(fd)->nabis; i++) {
		for (j = 0; j < SUMDATA(fd)->nitds; j++) {
			for (k = 0; k < SUMDATA(fd)->niids; k++) {
				for (l = 0; l < SUMDATA(fd)->nbcs; l++) {
					for (m = 0; m < SUMDATA(fd)->nstims; m++) {
						for (o = 0; o < SUMDATA(fd)->nmonos; o++) {
							for (p = 0; p < SUMDATA(fd)-> ntwo_snds; p++) {
								switch ((int) SUMDATA(fd)->monos[o]) {
								case (SYN_LEFT):
									monoc = 'L';
									break;
								case (SYN_RIGHT):
									monoc = 'R';
									break;
								case (SYN_BOTH):
									monoc = 'B';
									break;
								}
								sprintf (tmps, "<%ddB;%dus;%ddB;%d%%;%s;%c;%d>",
								     (int) SUMDATA (fd)-> abis[i],
								     (int) SUMDATA (fd)-> itds[j],
								     (int) SUMDATA (fd)-> iids[k],
								     (int) SUMDATA (fd)-> bcs[l],
								     SUMDATA (fd)-> ss_stims[m].descr,
									 monoc,
								     (int) SUMDATA (fd)-> two_snds[p]);

								SUMDATA(fd)-> means[i][j][k][l][m][o][p] =
								    mean(data [i][j][k][l][m][o][p], 
										num[i][j][k][l][m][o][p]);
								SUMDATA(fd)-> stderrs[i][j][k][l][m][o][p] =
								    stderror(data [i][j][k][l][m][o][p],
								     num[i][j][k][l][m][o][p]);
								SUMDATA(fd)->stddevs[i][j][k][l][m][o][p] =
								    stddev(data[i][j][k][l][m][o][p],
								     num[i][j][k][l][m][o][p]);
								convert_array(SUMDATA(fd)->nspikes[i][j][k][l][m][o][p],
								     data[i][j][k][l][m][o][p],
								     num[i][j][k][l][m][o][p],
								     int, 1, 1);
								assert((SUMDATA(fd)->depstrs[i][j][k][l][m][o][p] = (char *)
									malloc((1 + strlen (tmps)) * sizeof (char)))
								       != NULL);
								strcpy(SUMDATA(fd)-> depstrs[i][j][k][l][m][o][p],
								     tmps);
							}

							FREE(data[i][j][k][l][m][o]);
						}
						FREE(data[i][j][k][l][m]);
					}
					FREE(data[i][j][k][l]);
				}
				FREE(data[i][j][k]);
			}
			FREE(data[i][j]);
		}
		FREE(data[i]);
	}
	FREE(data);

	SUMDATA(fd)->n = num;

	return (1);
}

static void gen_freeSumData(FILEDATA * data)
{
	int i, j, k, l, o, p, m;

	if (SUMDATA(data) != NULL) {
		FREE(SUMDATA(data)->abis);
		FREE(SUMDATA(data)->itds);
		FREE(SUMDATA(data)->iids);
		FREE(SUMDATA(data)->bcs);
		FREE(SUMDATA(data)->f_stims);
		for (o = 0; o < SUMDATA(data)->nstims; o++)
			syn_spec_free(SUMDATA(data)->ss_stims + o);
		FREE(SUMDATA(data)->ss_stims);
		FREE(SUMDATA(data)->monos);
		FREE(SUMDATA(data)->two_snds);
		for (i = 0; i < SUMDATA(data)->nabis; i++) {
			for (j = 0; j < SUMDATA(data)->nitds; j++) {
				for (k = 0; k < SUMDATA(data)->niids; k++) {
					for (l = 0; l < SUMDATA(data)->nbcs; l++) {
						for (o = 0; o < SUMDATA(data)->nstims; o++) {
							for (p = 0; p < SUMDATA (data)-> nmonos; p++) {
								for (m = 0; m < SUMDATA (data)-> ntwo_snds; m++) {
									FREE(SUMDATA(data)->nspikes[i][j][k][l][o][p][m]);
									FREE(SUMDATA(data)->depstrs[i][j][k][l][o][p][m]);
								}
								FREE(SUMDATA(data)->n[i][j][k][l][o][p]);
								FREE(SUMDATA(data)->nspikes[i][j][k][l][o][p]);
								FREE(SUMDATA(data)->depstrs[i][j][k][l][o][p]);
								FREE(SUMDATA(data)->means[i][j][k][l][o][p]);
								FREE(SUMDATA(data)->stderrs[i][j][k][l][o][p]);
								FREE(SUMDATA(data)->stddevs[i][j][k][l][o][p]);
							}
							FREE(SUMDATA(data)->n[i][j][k][l][o]);
							FREE(SUMDATA(data)->nspikes[i][j][k][l][o]);
							FREE(SUMDATA(data)->depstrs[i][j][k][l][o]);
							FREE(SUMDATA(data)->means[i][j][k][l][o]);
							FREE(SUMDATA(data)->stderrs[i][j][k][l][o]);
							FREE(SUMDATA(data)->stddevs[i][j][k][l][o]);
						}
						FREE(SUMDATA(data)->n[i][j][k][l]);
						FREE(SUMDATA(data)->nspikes[i][j][k][l]);
						FREE(SUMDATA(data)->depstrs[i][j][k][l]);
						FREE(SUMDATA(data)->means[i][j][k][l]);
						FREE(SUMDATA(data)->stderrs[i][j][k][l]);
						FREE(SUMDATA(data)->stddevs[i][j][k][l]);
					}
					FREE(SUMDATA(data)->n[i][j][k]);
					FREE(SUMDATA(data)->nspikes[i][j][k]);
					FREE(SUMDATA(data)->depstrs[i][j][k]);
					FREE(SUMDATA(data)->means[i][j][k]);
					FREE(SUMDATA(data)->stderrs[i][j][k]);
					FREE(SUMDATA(data)->stddevs[i][j][k]);
				}
				FREE(SUMDATA(data)->n[i][j]);
				FREE(SUMDATA(data)->nspikes[i][j]);
				FREE(SUMDATA(data)->depstrs[i][j]);
				FREE(SUMDATA(data)->means[i][j]);
				FREE(SUMDATA(data)->stderrs[i][j]);
				FREE(SUMDATA(data)->stddevs[i][j]);
			}
			FREE(SUMDATA(data)->n[i]);
			FREE(SUMDATA(data)->nspikes[i]);
			FREE(SUMDATA(data)->depstrs[i]);
			FREE(SUMDATA(data)->means[i]);
			FREE(SUMDATA(data)->stderrs[i]);
			FREE(SUMDATA(data)->stddevs[i]);
		}
		FREE(SUMDATA(data)->n);
		FREE(SUMDATA(data)->nspikes);
		FREE(SUMDATA(data)->depstrs);
		FREE(SUMDATA(data)->means);
		FREE(SUMDATA(data)->stderrs);
		FREE(SUMDATA(data)->stddevs);
		FREE(SUMDATA(data)->s_nspikes);
		FREE(SUMDATA(data));
	}
}

static int gen_free(FILEDATA * data)
{
	FD1_freeRawData(data);
	gen_freeSumData(data);

	return (1);
}

int pm_gen_init(void)
{
	setFDOvalidviewMethod("gen", gen_valid_view);
	setFDOdoplotMethod("gen", gen_do_plot);
	setFDOplotMethod("gen", gen_plotter);
	setFDreadMethod("gen", FD1_reader);
	setFDfreeMethod("gen", gen_free);
	return (1);
}

void gen_get_ndata(FILEDATA * fd, int *nabis, int *nitds, int *niids,
		   int *nbcs, int *nstims, int *nmonos, int *ntwo_snds)
{
	(*nabis) = SUMDATA(fd)->nabis;
	(*nitds) = SUMDATA(fd)->nitds;
	(*niids) = SUMDATA(fd)->niids;
	(*nbcs) = SUMDATA(fd)->nbcs;
	(*nstims) = SUMDATA(fd)->nstims;
	(*nmonos) = SUMDATA(fd)->nmonos;
	(*ntwo_snds) = SUMDATA(fd)->ntwo_snds;
}

int ********gen_get_nspikes(FILEDATA * fd)
{
	return (SUMDATA(fd)->nspikes);
}

int *******gen_get_n(FILEDATA * fd)
{
	return (SUMDATA(fd)->n);
}

int *gen_get_s_nspikes(FILEDATA * fd)
{
	return (SUMDATA(fd)->s_nspikes);
}

int gen_get_s_n(FILEDATA * fd)
{
	return (SUMDATA(fd)->s_n);
}

char ********gen_get_depstrs(FILEDATA * fd)
{
	return (SUMDATA(fd)->depstrs);
}
