/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: pm_fiid.c,v 2.56 2002/07/15 04:30:17 cmalek Exp $
** Program: xdphys
**  Module: pm_fiid.c
**  Author: mazer
** Descrip: xdphys plot method -- fiid tuning curves
**
** Revision History (most recent last)
**
** 97.2 bjarthur
**  copied from pm_mono.c
**
*******************************************************************/

#include "xdphyslib.h"
#include "xdphys.h"
#include "plotter.h"

static int fiid_plotter(FDO *, FILEDATA *, FDObj_ViewType *, int, FILE *);
static int fiid_valid_view(FDO *, int);

static int view_order[][3] = {
	{PM_CURVE},
	{PM_MEANPHASE, PM_VS, PM_RAYLEIGH},
	{PM_PERHIST},
	{PM_RASTER, PM_RASTER_RAW},
	{PM_PSTH},
	{PM_ISIH}
};
static int nsubviews[] = { 1, 3, 1, 2, 1, 1 };
static int nviews = 6;

int fiid_do_plot(FDO * fdo, FILEDATA * fd, int view, int l, FILE * fptr)
{
	float period;
	int nbins;
	static int last_view = PM_NONE;

	if (!fiid_valid_view(fdo, view)) {
		pm_type_error("fiid", view);
		view = PM_DEFAULT;
		(void) fiid_valid_view(fdo, view);
	}

	FD_perhist_compute_period_nbins(fd, &period, &nbins);

	switch (view) {
	case PM_DEFAULT:
	case PM_CURVE:
		FD1_curveFN(fdo, fd, l, fptr);
		break;
	case PM_MEANPHASE:
		FD1_perhist1FN(fdo, fd, l, fptr, period, nbins, 0);
		break;
	case PM_VS:
		FD1_perhist1FN(fdo, fd, l, fptr, period, nbins, 1);
		break;
	case PM_RAYLEIGH:
		FD1_perhist1FN(fdo, fd, l, fptr, period, nbins, 2);
		break;
	case PM_PERHIST:
		if (last_view != PM_PERHIST)
			fdo->vsm_data = VSM_ALL_CODE;
		FD1_perhist_stats_menu(fdo);
		FD_vsm_menu(fdo, 1);
		FD1_perhist2FN(fdo, fd, l, fptr, period, nbins);
		break;
	case PM_RASTER:
		FD1_rasterFN(fdo, fd, l, 0, fptr);
		break;
	case PM_RASTER_RAW:
		FD1_rasterFN(fdo, fd, l, 1, fptr);
		break;
	case PM_PSTH:
		FD1_psthFN(fdo, fd, l, fptr);
		break;
	case PM_ISIH:
		FD1_isihFN(fdo, fd, l, fptr);
		break;
	default:
		pm_type_error("fiid", view);
		break;
	}

	last_view = view;
	return (1);
}

static int fiid_valid_view(FDO * fdo, int view)
{
	int retval = 0;
	int i, j;

	if (view == PM_DEFAULT) {
		retval = 1;
	} else {
		for (i = 0; i < nviews; i++) {
			for (j = 0; j < nsubviews[i]; j++) {
				if (view_order[i][j] == view) {
					retval = 1;
					fdo->view.lr = i;
					fdo->view.ud = j;
					break;
				}
			}
		}
	}

	return (retval);
}


static int fiid_plotter(FDO * fdo, FILEDATA * fd, FDObj_ViewType * view,
			int l, FILE * fptr)
{
	syn_spec ss;
	fd_syn_spec_parse(FD_GV(fdo->fds[l], "fiid.Stim"), 0, &ss);

	adjust_index(view->lr, view->ud);

	if (l == 0) {
		if (!((ss.class == SC_TONE) ||
		      ((ss.class == SC_STACK)
		       && (ss.parms.stack.num_freqs == 1)))) {
			if (view->lr == 1) {
				view->lr = 3;
				view->ud = 0;
			} else if (view->lr == 2) {
				view->lr = 0;
				view->ud = 0;
			}
		}
	} else {
		if (!((ss.class == SC_TONE) ||
		      ((ss.class == SC_STACK)
		       && (ss.parms.stack.num_freqs == 1))))
			if ((view->lr == 1) || (view->lr == 2)) {
				notify
				    ("some curves missing due to different stim types");
				return (0);
			}
	}

	if (!fdo->no_X) {
		if (nsubviews[view->lr] > 1) {
			XtSetSensitive(fdo->up, True);
			XtSetSensitive(fdo->down, True);
		}
	}

	fiid_do_plot(fdo, fd, view_order[view->lr][view->ud], l, fptr);

	return (1);
}

int pm_fiid_init(void)
{
	setFDOvalidviewMethod("fiid", fiid_valid_view);
	setFDOdoplotMethod("fiid", fiid_do_plot);
	setFDOplotMethod("fiid", fiid_plotter);
	setFDreadMethod("fiid", FD1_reader);
	setFDfreeMethod("fiid", FD1_free);
	return (1);
}
