/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: pm_cf.c,v 1.20 2002/07/15 04:30:16 cmalek Exp $
** Program: dowl
**  Module: pm_cf.c
**  Author: bjarthur
** Descrip: 
**
** Revision History (most recent last)
**
*******************************************************************/

#include "xdphyslib.h"
#include "xdphys.h"
#include "plotter.h"

#define PM_CF_FOO -1000

static void cf_curveFN(FDO *, FILEDATA *, int, FILE *);
static void cf_rasterFN(FDO *, FILEDATA *, int, FILE *);
static void cf_sort(FILEDATA *);
static int parse_deplines(char *, int *);
static void cf_summarize(FILEDATA * fd, int unitid);
static int cf_free(FILEDATA * data);
static int cf_plotter(FDO *, FILEDATA *, FDObj_ViewType *, int, FILE *);
static int cf_valid_view(FDO *, int);

typedef struct {
	int unit;		/* unit id or "mask" */
	int ndata;		/* number of summary data points */
	float *depvals;		/* depvar values (int) for matching means */
	float *ceils;		/* lowest value which had significant resp */
	float *floors;		/* highest value which did not have sig resp */
} SumData;

#define SUMDATA(data) ((SumData*)(data->sumdata))

char dis[50][6];
float mps[50];
float vss[50];
float rays[50];

static int view_order[][3] = {
	{PM_CURVE},
	{PM_MEANPHASE, PM_VS, PM_RAYLEIGH},
	{PM_PERHIST},
	{PM_RASTER},
	{PM_PSTH},
	{PM_ISIH}
};
static int nsubviews[] = { 1, 3, 1, 1, 1, 1 };
static int nviews = 6;

int cf_do_plot(FDO * fdo, FILEDATA * fd, int view, int l, FILE * fptr)
{
	static int last_view = PM_NONE;

	if (!cf_valid_view(fdo, view)) {
		pm_type_error("cf", view);
		view = PM_DEFAULT;
		(void) cf_valid_view(fdo, view);
	}

	switch (view) {
	case PM_DEFAULT:
	case PM_CURVE:
		cf_curveFN(fdo, fd, l, fptr);
		break;
	case PM_MEANPHASE:
		FD1_perhist1FN(fdo, fd, l, fptr, 0.0, PERHIST_DEF_NBINS,
			       0);
		break;
	case PM_VS:
		FD1_perhist1FN(fdo, fd, l, fptr, 0.0, PERHIST_DEF_NBINS,
			       1);
		break;
	case PM_RAYLEIGH:
		FD1_perhist1FN(fdo, fd, l, fptr, 0.0, PERHIST_DEF_NBINS,
			       2);
		break;
	case PM_PERHIST:{
			int *stimarray, nstims;

			if (last_view != PM_PERHIST) {
				stimarray = StimArray_Gen(&nstims, 0,
							  FD_GV(fdo->
								fds[fdo->
								    nfds -
								    1],
								"cf.range"),
							  1, 1, 0);
				fdo->vsm_data = stimarray[0];
			}

			FD1_perhist_stats_menu(fdo);
			FD_vsm_menu(fdo, 0);
			FD1_perhist2FN(fdo, fd, l, fptr, 0.0,
				       PERHIST_DEF_NBINS);
		}
		break;
	case PM_RASTER:
		cf_rasterFN(fdo, fd, l, fptr);
		break;
	case PM_PSTH:
		FD1_psthFN(fdo, fd, l, fptr);
		break;
	case PM_ISIH:
		FD1_isihFN(fdo, fd, l, fptr);
		break;
	default:
		pm_type_error("cf", view);
		break;
	}

	last_view = view;
	return (1);
}

static int cf_valid_view(FDO * fdo, int view)
{
	int retval = 0;
	int i, j;

	if (view == PM_DEFAULT) {
		retval = 1;
	} else {
		for (i = 0; i < nviews; i++) {
			for (j = 0; j < nsubviews[i]; j++) {
				if (view_order[i][j] == view) {
					retval = 1;
					fdo->view.lr = i;
					fdo->view.ud = j;
					break;
				}
			}
		}
	}

	return (retval);
}

static int cf_plotter(FDO * fdo, FILEDATA * fd, FDObj_ViewType * view,
		      int l, FILE * fptr)
{

	adjust_index(view->lr, view->ud);

	if (!fdo->no_X) {
		if (nsubviews[view->lr] > 1) {
			XtSetSensitive(fdo->up, True);
			XtSetSensitive(fdo->down, True);
		}
	}

	cf_do_plot(fdo, fd, view_order[view->lr][view->ud], l, fptr);

	return (1);
}


/* ------------------------------------------------------------------------
   cf_curveFN
   ------------------------------------------------------------------------ */
static void cf_curveFN(FDO * fdo, FILEDATA * fd, int l, FILE * fptr)
{
	int i, *ci, cj, *fi, fj, last;
	char *ti;
	char *xlabel, *ylabel;
	float **xc, **yc, **xf, **yf;

	cf_summarize(fd, fdo->sum_code[l]);

	i = cj = 0;

	while (i < SUMDATA(fd)->ndata) {
		while ((SUMDATA(fd)->ceils[i] == PM_CF_FOO)
		       && (i < SUMDATA(fd)->ndata))
			i++;
		last = i;
		while ((SUMDATA(fd)->ceils[i] != PM_CF_FOO)
		       && (i < SUMDATA(fd)->ndata))
			i++;

		if (last == i)
			continue;

		if (cj == 0) {
			assert((xc =
				(float **) malloc(sizeof(float *))) !=
			       NULL);
			assert((yc =
				(float **) malloc(sizeof(float *))) !=
			       NULL);
			assert((ci = (int *) malloc(sizeof(int))) != NULL);
		} else {
			assert((xc =
				(float **) realloc(xc,
						   (cj +
						    1) *
						   sizeof(float *))) !=
			       NULL);
			assert((yc =
				(float **) realloc(yc,
						   (cj +
						    1) *
						   sizeof(float *))) !=
			       NULL);
			assert((ci =
				(int *) realloc(ci,
						(cj + 1) * sizeof(int))) !=
			       NULL);
		}

		COPY_ARRAY((SUMDATA(fd)->depvals + last), xc[cj],
			   (i - last), float);
		COPY_ARRAY((SUMDATA(fd)->ceils + last), yc[cj], (i - last),
			   float);
		ci[cj] = i - last;
		cj++;
	}

	i = fj = 0;
	while (i < SUMDATA(fd)->ndata) {

		while ((SUMDATA(fd)->floors[i] == PM_CF_FOO)
		       && (i < SUMDATA(fd)->ndata))
			i++;
		last = i;
		while ((SUMDATA(fd)->floors[i] != PM_CF_FOO)
		       && (i < SUMDATA(fd)->ndata))
			i++;

		if (last == i)
			continue;

		if (fj == 0) {
			assert((xf =
				(float **) malloc(sizeof(float *))) !=
			       NULL);
			assert((yf =
				(float **) malloc(sizeof(float *))) !=
			       NULL);
			assert((fi = (int *) malloc(sizeof(int))) != NULL);
		} else {
			assert((xf =
				(float **) realloc(xf,
						   (fj +
						    1) *
						   sizeof(float *))) !=
			       NULL);
			assert((yf =
				(float **) realloc(yf,
						   (fj +
						    1) *
						   sizeof(float *))) !=
			       NULL);
			assert((fi =
				(int *) realloc(fi,
						(fj + 1) * sizeof(int))) !=
			       NULL);
		}

		COPY_ARRAY((SUMDATA(fd)->depvals + last), xf[fj],
			   (i - last), float);
		COPY_ARRAY((SUMDATA(fd)->floors + last), yf[fj],
			   (i - last), float);
		fi[fj] = i - last;
		fj++;
	}

	if (fptr != NULL) {
		if (fdo->no_text_header != 1) {
			fprintf(fptr, "COMMENTS\n");
			fprintf(fptr, "%%text dump of %s, cf curve\n",
				strrchr(fd->filename,
					'/') ==
				NULL ? fd->filename : 1 +
				(char *) strrchr(fd->filename, '/'));
			fprintf(fptr, "%%col 1: frequency (Hz)\n");

			fprintf(fptr,
				"%%col 2: ceiling intensity (dB SPL)\n");
			fprintf(fptr,
				"%%col 3: floor intensity (dB SPL)\n");

			fprintf(fptr, "END_COMMENTS\n");
			fprintf(fptr, "PARAMS\n");
			fprintf(fptr, "END_PARAMS\n");
			fprintf(fptr, "RASTERDATA\n");
			fprintf(fptr, "nlines=%d\n", SUMDATA(fd)->ndata);
		}
		for (i = 0; i < SUMDATA(fd)->ndata; i++)
			fprintf(fptr, "%e\t%e\t%e\n",
				SUMDATA(fd)->depvals[i],
				SUMDATA(fd)->ceils[i],
				SUMDATA(fd)->floors[i]);

		if (fdo->no_text_header != 1) {
			fprintf(fptr, "END_RASTERDATA\n");
		}
	}
	if (fdo->no_X)
		return;

	if (cj != 0) {
		for (i = 0; i < cj; i++) {
			FDObj_Add_Data_All(fdo, xc[i], yc[i], ci[i]);
			FDObj_AddLine(fdo, l, xc[i], yc[i], NULL, ci[i],
				      AtFloat, atQuadLinePlotWidgetClass,
				      AtTypeLINEPOINTS, AtMarkCIRCLE,
				      ConvertColor(fdo->graph, "blue"),
				      FDObj_Legend(fdo, l));
		}
		free(xc);
		free(yc);
		free(ci);
	}
	if (fj != 0) {
		for (i = 0; i < fj; i++) {
			FDObj_Add_Data_All(fdo, xf[i], yf[i], fi[i]);
			FDObj_AddLine(fdo, l, xf[i], yf[i], NULL, fi[i],
				      AtFloat, atQuadLinePlotWidgetClass,
				      AtTypeLINEPOINTS, AtMarkCIRCLE,
				      ConvertColor(fdo->graph, "orange"),
				      FDObj_Legend(fdo, l));
		}
		free(xf);
		free(yf);
		free(fi);
	}
	ti = "curve";
	XtVaSetValues(fdo->graph, XtNtitle, FDObj_Title(fdo, ti, l),
		      XtNshowTitle, True, NULL);
	xlabel = "freq (Hz)";
	ylabel = "intensity (dB)";
	XtVaSetValues(fdo->xaxis, XtNlabel, xlabel, NULL);
	XtVaSetValues(fdo->yaxis, XtNlabel, ylabel, NULL);
}

/* ------------------------------------------------------------------------
   cf_rasterFN
   ------------------------------------------------------------------------ */
static void cf_rasterFN(FDO * fdo, FILEDATA * fd, int l, FILE * fptr)
{
	int i, j, k;
	char *ti;
	SA_SpikeRasterList *srl = NULL;
	char *xlabel, *ylabel;
	float *xdata, *ydata;
	int ndata;
	double xmin, xmax;

	srl = FD_to_SpikeRasterList(FD1_RAWDATA(fd)->nrasters,
				    FD1_RAWDATA(fd)->rastlist, FD_GI(fd,
								     "epoch"),
				    fdo->sum_code[l]);
	ndata = SA_TotalSpikesInSRL(srl);
	assert((xdata = (float *) malloc(sizeof(float) * ndata)) != NULL);
	assert((ydata = (float *) malloc(sizeof(float) * ndata)) != NULL);

	for (k = i = 0; i < srl->nrasters; i++) {
		for (j = 0; j < srl->sr[i]->nspikes; j++) {
			xdata[k] = srl->sr[i]->tm[j];
			ydata[k++] = FD1_RAWDATA(fd)->pres_order[i];
		}
	}

	SA_FreeSpikeRasterList(srl);

	if (fdo->no_X)
		return;

	FDObj_Add_Data_All(fdo, xdata, ydata, ndata);
	FDObj_AddLine(fdo, l, xdata, ydata, NULL, ndata, AtFloat,
		      atQuadLinePlotWidgetClass, AtTypePOINTS, AtMarkVBAR,
		      ConvertColor(fdo->graph,
				   FDObj_Colors[l % FDObj_numColors]),
		      FDObj_Legend(fdo, l));

	xmin = 0.0;
	xmax = (double) FD_GI(fd, "epoch");
	XtVaSetValues(fdo->xaxis, XtNmin, &xmin, XtNmax, &xmax,
		      XtNautoScale, False, NULL);
	xmin = 0.0;
	xmax = (double) srl->nrasters;
	XtVaSetValues(fdo->yaxis, XtNmin, &xmin, XtNmax, &xmax,
		      XtNautoScale, False, NULL);

	ylabel = "raster # (pres order)";
	ti = "rawraster";
	XtVaSetValues(fdo->graph, XtNtitle, FDObj_Title(fdo, ti, l),
		      XtNshowTitle, True, NULL);
	xlabel = "time (ms)";
	XtVaSetValues(fdo->xaxis, XtNlabel, xlabel, NULL);
	XtVaSetValues(fdo->yaxis, XtNlabel, ylabel, NULL);
}


static void cf_sort(FILEDATA * data)
{
	int i, j, k1, k2;
	int last, min, tmp;
	int idx;

	k1 = 0;
	while ((k1 < FD1_RAWDATA(data)->nrasters)) {

		k2 = k1;
		last = FD1_RAWDATA(data)->depints[k1];
		while ((k2 < FD1_RAWDATA(data)->nrasters) &&
		       (FD1_RAWDATA(data)->depints[k2] == last)) {
			k2++;
		}

		for (i = k1; i < (k2 - 1); i++) {
			idx = i;
			parse_deplines(FD1_RAWDATA(data)->deplines[i],
				       &min);
			for (j = i + 1; j < k2; j++) {
				parse_deplines(FD1_RAWDATA(data)->
					       deplines[j], &tmp);
				if (tmp < min) {
					idx = j;
					min = tmp;
				}
			}
			if (idx != i) {
				SWAP(FD1_RAWDATA(data)->pres_order[i],
				     FD1_RAWDATA(data)->pres_order[idx],
				     int);
				SWAP(FD1_RAWDATA(data)->rastlist[i],
				     FD1_RAWDATA(data)->rastlist[idx],
				     spike_t *);
				SWAP(FD1_RAWDATA(data)->deplines[i],
				     FD1_RAWDATA(data)->deplines[idx],
				     char *);
				SWAP(FD1_RAWDATA(data)->depints[i],
				     FD1_RAWDATA(data)->depints[idx], int);
			}
		}

		k1 = k2;
	}
}


static int parse_deplines(char *depline, int *iter)
{
	int foo;

	sscanf(depline, "depvar=%d", &foo);
	if (foo == SPONT) {
		sscanf(depline, "depvar=%*d <SPONT, iter #%d>", iter);
		return (SPONT);
	} else {
		sscanf(depline, "depvar=%*d <%d dB, iter #%d>", &foo,
		       iter);
		return (foo);
	}
}


/* ------------------------------------------------------------------------
   cf_summarize

	   summarize() is a little tricky because there are two options for
	   spontaneous trials.  if Spont_Stims=1, then one spont trial is
	   present for every repetition and the spont rate is calculated
	   from those trials.  if Spont_Stims=0, then
	   independent spont trials are not present, and the spont rate is
	   calculated from the Delay portion of all the trials *
   ------------------------------------------------------------------------ */
static void cf_summarize(FILEDATA * fd, int unitid)
{
	int nreps;		/* number of reps (ideally) */
	float t_begin;		/* driven begins at in ms (t_ in ticks) */
	float t_end;		/* driven ends at in ms (t_ in ticks) */
	float t_spont_begin;	/* spont begins at in ms (t_ in ticks) */
	float t_spont_end;	/* spont ends at in ms (t_ in ticks) */
	int use_delay;		/* use the pre-stim delay as spontaneous */
	int last;
	int last_db;
	int i, k;
	int spont_stims;
	int tmp;
	float **sr, *foo;
	int *nsr, nits, it, nstims, nfoo;
	float pval, high, low, res;
	int *stimarray;
	int iceil, ifloor, imiddle;
	int iter;

	cf_sort(fd);

	if (SUMDATA(fd) != NULL) {
		FREE(SUMDATA(fd)->depvals);
		FREE(SUMDATA(fd)->ceils);
		FREE(SUMDATA(fd)->floors);
		FREE(SUMDATA(fd));
	}

	tmp = (1 + FD1_RAWDATA(fd)->nrasters);
	assert((SUMDATA(fd) = (SumData *) malloc(sizeof(SumData))) !=
	       NULL);
	assert((SUMDATA(fd)->depvals =
		(float *) malloc(tmp * sizeof(float))) != NULL);
	assert((SUMDATA(fd)->ceils = (float *) malloc(tmp * sizeof(float)))
	       != NULL);
	assert((SUMDATA(fd)->floors =
		(float *) malloc(tmp * sizeof(float))) != NULL);

	/* pre-calculate all analysis parameters */

	nreps = FD_GI(fd, "reps");
	pval = FD_GF(fd, "cf.p-val");
	high = (float) FD_GI(fd, "cf.High");
	low = (float) FD_GI(fd, "cf.Low");
	res = (float) FD_GI(fd, "cf.Resolution");

	FD_calc_params(fd, &spont_stims, &use_delay,
		       &t_begin, &t_end, &t_spont_begin, &t_spont_end);

	stimarray = StimArray_Gen(&nstims, FD_GI(fd, "randomize"),
				  FD_GV(fd, "cf.Range"), nreps, 1,
				  spont_stims);
	free(stimarray);

	/* now compute SRs */

	nits = 2 + (int) ceil(log((high - low) / res) / log(2.0));
	assert((nsr = (int *) calloc(nits, sizeof(int))) != NULL);
	assert((sr = (float **) malloc(nits * sizeof(float *))) != NULL);
	if (spont_stims && (!use_delay)) {
		for (i = 0; i < nits; i++)
			assert((sr[i] =
				(float *) malloc(nreps * sizeof(float))) !=
			       NULL);
		i = 0;
		while (i < FD1_RAWDATA(fd)->nrasters
		       && FD1_RAWDATA(fd)->depints[i] == SPONT) {
			sr[i / nreps][i % nreps] =
			    (float) countRaster(t_begin, t_end,
						FD1_RAWDATA(fd)->
						rastlist[i], unitid);
			nsr[i / nreps]++;
			i++;
		}
	} else {
		for (i = 0; i < nits; i++)
			assert((sr[i] =
				(float *) malloc(nreps * nstims *
						 sizeof(float))) != NULL);
		i = 0;
		while (i < FD1_RAWDATA(fd)->nrasters) {
			parse_deplines(FD1_RAWDATA(fd)->deplines[i],
				       &iter);
			sr[iter][nsr[iter]++] =
			    (float) countRaster(t_spont_begin, t_spont_end,
						FD1_RAWDATA(fd)->rastlist
						[i], unitid);
			i++;
		}
	}

	/* now summarize the data */

	assert((foo = (float *) malloc(nreps * sizeof(float))) != NULL);
	i = k = 0;
	while (i < FD1_RAWDATA(fd)->nrasters) {	/* depints */
		if (FD1_RAWDATA(fd)->depints[i] == SPONT) {
			i++;
			continue;
		}

		it = 0;
		iceil = ifloor = imiddle = PM_CF_FOO;
		last = FD1_RAWDATA(fd)->depints[i];
		while ((i < FD1_RAWDATA(fd)->nrasters) && (FD1_RAWDATA(fd)->depints[i] == last)) {	/* intensities */
			nfoo = 0;
			last_db =
			    parse_deplines(FD1_RAWDATA(fd)->deplines[i],
					   &iter);

			while ((i < FD1_RAWDATA(fd)->nrasters) &&	/* reps */
			       (FD1_RAWDATA(fd)->depints[i] == last) &&
			       (parse_deplines(FD1_RAWDATA(fd)->deplines
					       [i], &iter) == last_db)) {
				foo[nfoo++] =
				    (float) countRaster(t_begin, t_end,
							FD1_RAWDATA(fd)->
							rastlist[i],
							unitid);
				i++;
			}

			if (nfoo != nreps)
				continue;

			if (iceil == PM_CF_FOO)
				iceil =
				    parse_deplines(FD1_RAWDATA(fd)->
						   deplines[i - 1], &iter);
			else if (ifloor == PM_CF_FOO)
				ifloor =
				    parse_deplines(FD1_RAWDATA(fd)->
						   deplines[i - 1], &iter);
			else if (imiddle == PM_CF_FOO)
				imiddle =
				    parse_deplines(FD1_RAWDATA(fd)->
						   deplines[i - 1], &iter);
			else {
				tmp =
				    parse_deplines(FD1_RAWDATA(fd)->
						   deplines[i - 1], &iter);
				if (tmp > imiddle)
					ifloor = imiddle;
				else
					iceil = imiddle;
				imiddle = tmp;
			}

			it++;
		}

		if (imiddle != PM_CF_FOO) {
			if (FD_compare_intensities(foo, nfoo, sr[it - 1],
						   nsr[it - 1], pval))
				iceil = imiddle;
			else
				ifloor = imiddle;
		} else if (ifloor != PM_CF_FOO) {
			iceil = ifloor;
			ifloor = PM_CF_FOO;
		} else {
			ifloor = iceil;
			iceil = PM_CF_FOO;
		}

		SUMDATA(fd)->depvals[k] = last;
		SUMDATA(fd)->ceils[k] = iceil;
		SUMDATA(fd)->floors[k] = ifloor;

		k++;
	}

	SUMDATA(fd)->ndata = k;
	SUMDATA(fd)->unit = unitid;

	for (i = 0; i < nits; i++)
		free(sr[i]);
	free(nsr);
	free(sr);
}

/* ------------------------------------------------------------------------
   cf_free
   ------------------------------------------------------------------------ */
int cf_free(FILEDATA * data)
{
	FD1_freeRawData(data);

	FREE(SUMDATA(data)->depvals);
	FREE(SUMDATA(data)->ceils);
	FREE(SUMDATA(data)->floors);

	FREE(FD1_RAWDATA(data));
	FREE(SUMDATA(data));

	return (1);
}

/* note that by using FD1_reader() here, there might be some cases in which
it reports data light, when in fact the data is not data light in the sense
that data light is used elsewhere (ie only 8 reps when there should be 10).
these cases occurr when the user has not set High to be above threshold and
Low to be below threshold.  this results in a mismatch between nrasters and
actual_nrasters */

int pm_cf_init(void)
{
	setFDOvalidviewMethod("cf", cf_valid_view);
	setFDOdoplotMethod("cf", cf_do_plot);
	setFDOplotMethod("cf", cf_plotter);
	setFDreadMethod("cf", FD1_reader);
	setFDfreeMethod("cf", cf_free);
	return (1);
}
