/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: mod_gen.c,v 1.31 2003/10/10 17:24:59 bjorn Exp $
** Program: dowl
**  Module: mod_gen.c
**  Author: arthur
** Descrip: xdowl application module
**
** Revision History (most recent last)
**
** 99.03 bjarthur
**   creation date.  a new 'generic' or 'general' module which can
**   vary an arbitrary number of parameters.  this module largely
**   adds the functionality of being able to interleave stimuli.
*******************************************************************/

#include "xdphyslib.h"
#include "xdphys.h"
#include "mod_gen_util.h"

#ifndef PATH_MAX
/* PATH_MAX should be defined by the OS, but if it isn't (e.g. in SunOS 4.x)
   define it here */
#define PATH_MAX 255
#endif /* PATH_MAX */

#define NUM_PAGES 10
#define MAX_NUM 100
#define MAX_LEN 128

static Field ws_table[] = {
	{"gen.abi (dB)", "%s", 80, "20"},
	{"gen.itd (us)", "%s", 80, "0"},
	{"gen.iid (dB)", "%s", 80, "0"},
	{"gen.bc (%)", "%s", 80, "0"},
	{"gen.stim", "%s", 80, "0"},
	{"gen.mono (L,B,R)", "%s", 5, "B"},
	{"gen.two_snd", "%s", 3, "0"},
	{WS_VSEP},
	{"gen.sep_files", "%s", 3, "0"},	/* not saved to datafile. */
	{NULL},
};

static char *G_datafile[MAX_NUM];
static int G_ndatafiles;
static char G_sep_files[10];

static int **gen_StimArray_Gen(syn_spec **, char *, char *, char *, char *,
			       char *, char *, char *, int *, int, int,
			       int);
static float calc_max_itd(int **, int);
static int bc_not_100(int **, int);
static int nstims_is_one(int **, syn_spec *, int);
static int runScan(FILE *, Widget);
static void Gen_ShiftCB(Widget, XtPointer, XtPointer);
Widget setupGen(DockSlot *);

static int **gen_StimArray_Gen(syn_spec ** ret_ssarray,
			       char *range_abi, char *range_itd,
			       char *range_iid, char *range_bc,
			       char *range_stim, char *range_mono,
			       char *range_two_snd, int *ntrials, int reps,
			       int spont, int rand)
{
	int *stimarray_iid, *stimarray_itd, *stimarray_abi, *stimarray_bc;
	int *stimarray_mono, *stimarray_two_snd;
	int **stimarray, **ret_stimarray;
	int i, j, k, l, m, n, o, p;
	int niids, nitds, nitds_tmp, nabis, nbcs, nstims, nmonos,
	    ntwo_snds;
	syn_spec *ssarray, *ssarray_;

	m = 0;

	if ((stimarray_abi = StimArray_Gen(&nabis, 0, range_abi, 1, 0, 0)) == NULL)
		return (NULL);
	if ((stimarray_iid = StimArray_Gen(&niids, 0, range_iid, 1, 0, 0)) == NULL)
		return (NULL);
	if ((stimarray_bc = StimArray_Gen(&nbcs, 0, range_bc, 1, 0, 0)) == NULL)
		return (NULL);
	if ((stimarray_mono = gen_mono_StimArray_Gen(&nmonos, 0, range_mono, 1, 0,
				    0)) == NULL)
		return (NULL);
	if ((stimarray_two_snd =
	     StimArray_Gen(&ntwo_snds, 0, range_two_snd, 1, 0, 0)) == NULL)
		return (NULL);
	if ((ssarray_ = gen_SSArray_Gen(&nstims, 0, range_stim, 1, 0, 0)) == NULL)
		return (NULL);
	if ((stimarray_itd =
	     gen_itd_StimArray_Gen(&nitds, 0, range_itd, 1, 0, 0, ssarray_,
				   nstims)) == NULL)
		return (NULL);

	assert((stimarray = (int **) malloc(6 * sizeof(int *))) != NULL);
	/* These arrays will be slightly too big if some of stimarray_mono 
	 * is not == SYN_BOTH, i.e. if we have some  monaural stimuli planned.  
	 * If nnon_bin is the number of non-BIN * entries in stimarray_mono, 
	 * it will be off by nnon_bin*nitds.
	 *
	 * m, however will be right, and m is used to compute ntrials, below */

	assert((stimarray[0] = (int *) malloc((spont +
			niids * nitds * nabis * nbcs * nstims * nmonos *
			ntwo_snds) * sizeof(int))) != NULL);
	assert((stimarray[1] = (int *) malloc((spont +
			niids * nitds * nabis * nbcs * nstims * nmonos *
			ntwo_snds) * sizeof(int))) != NULL);
	assert((stimarray[2] = (int *) malloc((spont +
			niids * nitds * nabis * nbcs * nstims * nmonos *
			ntwo_snds) * sizeof(int))) != NULL);
	assert((stimarray[3] = (int *) malloc((spont +
			niids * nitds * nabis * nbcs * nstims * nmonos *
			ntwo_snds) * sizeof(int))) != NULL);
	assert((stimarray[4] = (int *) malloc((spont +
			niids * nitds * nabis * nbcs * nstims * nmonos *
			ntwo_snds) * sizeof(int))) != NULL);
	assert((stimarray[5] = (int *) malloc((spont +
			niids * nitds * nabis * nbcs * nstims * nmonos *
			ntwo_snds) * sizeof(int))) != NULL);
	assert((ssarray = (syn_spec *) malloc((spont +
			niids * nitds * nabis * nbcs * nstims * nmonos *
			ntwo_snds) * sizeof(syn_spec))) != NULL);

	for (o = 0; (o < nmonos) && progRunning; o++) {
		/* if this is a monaural bunch of stims, 
		 * set ITD=0; this means that nitds=1 */
		if (stimarray_mono[o] == SYN_BOTH)
			nitds_tmp = nitds;
		else
			nitds_tmp = 1;
		for (i = 0; (i < nabis) && progRunning; i++) {
			for (j = 0; (j < nitds_tmp) && progRunning; j++) {
				for (k = 0; (k < niids) && progRunning; k++) {
					for (n = 0; (n < nbcs) && progRunning; n++) {
						for (p = 0; (p < ntwo_snds) && progRunning; p++) {
							for (l = 0; (l < nstims) && progRunning; l++) {
								stimarray[0][m] = stimarray_abi[i];
								/* if this is a monaural bunch of stims, 
								 * set ITD=0 */
								if (stimarray_mono[o] == SYN_BOTH)
									stimarray[1][m] = stimarray_itd[j];
								else
									stimarray[1][m] = 0;
								stimarray[2][m] = stimarray_iid[k];
								stimarray[3][m] = stimarray_bc[n];
								stimarray[4][m] = stimarray_mono[o];
								stimarray[5][m] = stimarray_two_snd[p];
								ssarray[m] = ssarray_[l]; 
								m = m + 1;
							}
						}
					}
				}
			}
		}
	}

	if (spont) {
		stimarray[0][m] = SPONT;
		stimarray[1][m] = SPONT;
		stimarray[2][m] = SPONT;
		stimarray[3][m] = SPONT;
		stimarray[4][m] = SPONT;
		stimarray[5][m] = SPONT;
		ssarray[m++].class = SC_INVALID;
	}

	for (i = 0; i < (m - 1); i++) {
		for (j = i + 1; j < m; j++) {
			if ((stimarray[0][i] == stimarray[0][j]) &&
			    (stimarray[1][i] == stimarray[1][j]) &&
			    (stimarray[2][i] == stimarray[2][j]) &&
			    (stimarray[3][i] == stimarray[3][j]) &&
			    (stimarray[4][i] == stimarray[4][j]) &&
			    (stimarray[5][i] == stimarray[5][j]) &&
			    syn_spec_same(ssarray + i, ssarray + j)) {
				alert("Duplicates in stimulus array");
				free(stimarray[0]);
				free(stimarray[1]);
				free(stimarray[2]);
				free(stimarray[3]);
				free(stimarray[4]);
				free(stimarray[5]);
				free(ssarray);
				free(stimarray);
				free(stimarray_abi);
				free(stimarray_itd);
				free(stimarray_iid);
				free(stimarray_bc);
				free(stimarray_mono);
				free(stimarray_two_snd);
				free(ssarray_);
				return (NULL);
			}
		}
	}

	(*ntrials) = m;
	assert((ret_stimarray = (int **) malloc(6 * sizeof(int *))) != NULL);
	assert((ret_stimarray[0] =
		(int *) malloc(reps * (*ntrials) * sizeof(int))) != NULL);
	assert((ret_stimarray[1] =
		(int *) malloc(reps * (*ntrials) * sizeof(int))) != NULL);
	assert((ret_stimarray[2] =
		(int *) malloc(reps * (*ntrials) * sizeof(int))) != NULL);
	assert((ret_stimarray[3] =
		(int *) malloc(reps * (*ntrials) * sizeof(int))) != NULL);
	assert((ret_stimarray[4] =
		(int *) malloc(reps * (*ntrials) * sizeof(int))) != NULL);
	assert((ret_stimarray[5] =
		(int *) malloc(reps * (*ntrials) * sizeof(int))) != NULL);
	assert(((*ret_ssarray) =
		(syn_spec *) malloc(reps * (*ntrials) *
				    sizeof(syn_spec))) != NULL);
	for (i = 0, k = 0; i < reps; i++) {
		if (rand) {
			for (m = 0; m < ((*ntrials) - 1); m++) {
				j = 1 + (random() % ((*ntrials) - 1 - m));
				SWAP(stimarray[0][m], stimarray[0][m + j], int);
				SWAP(stimarray[1][m], stimarray[1][m + j], int);
				SWAP(stimarray[2][m], stimarray[2][m + j], int);
				SWAP(stimarray[3][m], stimarray[3][m + j], int);
				SWAP(stimarray[4][m], stimarray[4][m + j], int);
				SWAP(stimarray[5][m], stimarray[5][m + j], int);
				SWAP(ssarray[m], ssarray[m + j], syn_spec);
			}
		}
		for (j = 0; j < (*ntrials); j++) {
			ret_stimarray[0][k] = stimarray[0][j];
			ret_stimarray[1][k] = stimarray[1][j];
			ret_stimarray[2][k] = stimarray[2][j];
			ret_stimarray[3][k] = stimarray[3][j];
			ret_stimarray[4][k] = stimarray[4][j];
			ret_stimarray[5][k] = stimarray[5][j];
			(*ret_ssarray)[k++] = ssarray[j];
		}
	}

	(*ntrials) *= reps;

	free(stimarray[0]);
	free(stimarray[1]);
	free(stimarray[2]);
	free(stimarray[3]);
	free(stimarray[4]);
	free(stimarray[5]);
	free(ssarray);
	free(stimarray);
	free(stimarray_abi);
	free(stimarray_itd);
	free(stimarray_iid);
	free(stimarray_bc);
	free(stimarray_mono);
	free(stimarray_two_snd);
	free(ssarray_);

	return (ret_stimarray);
}

static float calc_max_itd(int **stimarray, int nstims)
{
	int i;
	float maxitd;

	for (i = 0, maxitd = -1; i < nstims; i++) {
		if (stimarray[1][i] == SPONT)
			continue;
		if (abs(stimarray[1][i]) > maxitd)
			maxitd = abs(stimarray[1][i]);
	}
	maxitd = RND2INT(1.0e-6 * maxitd);

	return (maxitd);
}

static int bc_not_100(int **stimarray, int nstims)
{
	int i;

	for (i = 0; i < nstims; i++) {
		if (stimarray[3][i] == SPONT)
			continue;
		if (stimarray[3][i] != 100)
			return (1);
	}

	return (0);
}

static int nstims_is_one(int **stimarray, syn_spec * ssarray, int nstims)
{
	int i;

	if (!strcmp(G_sep_files, "abi")) {
		for (i = 0; i < nstims - 1; i++) {
			if (stimarray[0][i] == SPONT)
				continue;
			if (stimarray[0][i] != stimarray[0][i + 1])
				return (0);
		}
	} else if (!strcmp(G_sep_files, "itd")) {
		for (i = 0; i < nstims - 1; i++) {
			if (stimarray[1][i] == SPONT)
				continue;
			if (stimarray[1][i] != stimarray[1][i + 1])
				return (0);
		}
	} else if (!strcmp(G_sep_files, "iid")) {
		for (i = 0; i < nstims - 1; i++) {
			if (stimarray[2][i] == SPONT)
				continue;
			if (stimarray[2][i] != stimarray[2][i + 1])
				return (0);
		}
	} else if (!strcmp(G_sep_files, "bc")) {
		for (i = 0; i < nstims - 1; i++) {
			if (stimarray[3][i] == SPONT)
				continue;
			if (stimarray[3][i] != stimarray[3][i + 1])
				return (0);
		}
	} else if (!strcmp(G_sep_files, "bf")) {
		for (i = 0; i < nstims - 1; i++) {
			if (stimarray[3][i] == SPONT)
				continue;
			if (!syn_spec_same(ssarray + i, ssarray + i + 1))
				return (0);
		}
	}

	return (1);
}

static int runScan(FILE * datafp, Widget status)
{
	int i;
	int iid, fiid;		/* current iid in db */
	int abi, fabi;		/* abi for the run */
	int itd, fitd;		/* constant itd in us */
	int bc, fbc;
	int mono;		/* 3=bin, 1=left mono, 2 =right mono */
	int two_snd;
	int delay;		/* stimulus latency in ms */
	int dur;		/* stim durationin ms */
	int epoch;		/* recording epoch in ms */
	int isi;		/* inter-stimulus interval */
	int rise, fall;		/* rise and fall times in ms */
	int nstims, reps;	/* number & reps of stimuli to present */
	int **stimarray;	/* sequence of itds to present */
	spike_t *data;		/* pointer to raster data */
	float latten, ratten;	/* digital attenuator settings */
	float lspl, rspl;	/* desired dbspl levels */
	float max_lspl, max_rspl;	/* max dbspl levels */
	char *abi_range, *itd_range, *iid_range, *bc_range,
	    stim_range[1024];
	char *mono_range, *two_snd_range;
	int maxitd;		/* in ms */
	int use_ts_dur;		/* Boolean: use or not use separate dur/delay for
				   second sound in twosound */
	int ts_dur;		/* Duration for second sound in twosound */
	int ts_delay;		/* Delay for second sound in twosound */
	void *isitime;		/* handle for isi_functions() */
	char depstr[256];
	syn_spec ss, ssf, ss2;	/* full specification for sound */
	char *fstimspec;
	float *avePMk;
	int err;
	char msg[1024];
	int rand, spont;
	float daFc, evtFc;
	/*char *datafile; */
	int ana_every;
	int stim_every;
	double timestamp;
	int retval;
	int outsize, da_channels;
	xword *outbuf;
	syn_spec *ssarray;
	char monoc;

	/* calculate parameters */
	ws_lock(ws_table, True);

	abi_range = GS("gen.abi");
	itd_range = GS("gen.itd");
	iid_range = GS("gen.iid");
	bc_range = GS("gen.bc");
	strcpy(stim_range, GS("gen.stim"));
	mono_range = GS("gen.mono");
	two_snd_range = GS("gen.two_snd");
	strcpy(G_sep_files, GS("gen.sep_files"));
	reps = GI("reps");
	dur = GI("dur");
	delay = GI("delay");
	epoch = GI("epoch");
	rise = GI("rise");
	fall = GI("fall");
	isi = GI("isi");

	if (strchr(two_snd_range, '1') != NULL) {
		fabi = GI("ts.fix_abi");
		fitd = GI("ts.fix_itd");
		fiid = GI("ts.fix_iid");
		fbc = GI("ts.fix_bc");
		fstimspec = GS("ts.fix_stim");
		use_ts_dur = GI("ts.use_dur_delay");
		ts_dur = GI("ts.dur");
		ts_delay = GI("ts.delay");
	}
	rand = GI("randomize");
	spont = GI("Spont_Stims");
	evtFc = GF("evtFc");
	daFc = GF("daFc");

	ana_every = GI("ana.every");
	stim_every = GI("stim.every");
	ws_lock(ws_table, False);
	unlock_most_worksheets();
	/* no svars should be accessed beyond this point */

	if (!strncmp(stim_range, "file=", 5)) {
		char *p;
		FILE *fp;
		int l = filesize(stim_range + 5);
		assert(l < 1024);
		if (l && (fp = fopen2(stim_range + 5, "r")) != NULL) {
			fread(stim_range, l, sizeof(char), fp);
			if ((p = rindex(stim_range, '\n')) != NULL)
				(*p) = '\000';
			fclose(fp);
		} else {
			alert("gen.stim: no such file\n");
			return (-1);
		}
	}
	if (strchr(two_snd_range, '1') != NULL) {
		fd_syn_spec_parse(fstimspec, 0, &ssf);
		if (ssf.class == SC_INVALID) {
			alert("Stim: \"%s\" illegal syn_class\n%s",
			      fstimspec, synthesize_help());
			return (-1);
		}
	}
	if ((stimarray = gen_StimArray_Gen(&ssarray, abi_range, itd_range, 
			iid_range, bc_range, stim_range, mono_range, two_snd_range,
			&nstims, reps, spont, rand)) == NULL)
		return (-1);

	if (nstims == 0) {
		alert("nstims is zero");
		return (-1);
	}
	if (!strcmp(G_sep_files, "bf")
	    && !only_this_stim(ssarray, nstims, SC_TONE)) {
		alert
		    ("if sep_files = 'bf' then stim can be only pure tones");
		return (-1);
	}
	if (strcmp(G_sep_files, "0")
	    && nstims_is_one(stimarray, ssarray, nstims)) {
		alert("nstims == 1 after separating files");
		return (-1);
	}
	if (bc_not_100(stimarray, nstims) &&
	    (!only_this_stim(ssarray, nstims, SC_NOISE)
	     || strcasecmp(mono_range, "b"))) {
		alert("for bc < 100, only binaural noise can be used");
		return (-1);
	}
	maxitd = calc_max_itd(stimarray, nstims);

	fprintf(datafp, "nrasters=%d\n", nstims);

	syn_calibrate = 1;

	is_setNice(GI("nicelevel"));
	is_setSyncPulse(GI("syncPulse"));

	is_init(daFc, 0.0, evtFc, dur + delay + maxitd, epoch);

	da_channels = is_getDAnchans();
	outbuf = is_getDAbuffer(&outsize);

	/* check to see if att can handle all iid's */

	if (strchr(two_snd_range, '1') == NULL) {
		for (i = 0, err = 0; (i < (nstims / reps)) && progRunning;
		     i++) {
			if (ssarray[i].class == SC_INVALID)
				continue;

			abi = stimarray[0][i];
			itd = stimarray[1][i];
			iid = stimarray[2][i];
			bc = stimarray[3][i];

			synthesize(ssarray + i, NULL, outbuf, outsize,
				   da_channels, delay, dur, epoch, itd, bc,
				   rise, fall, SYN_BOTH);
			cal_FigureSPL(ssarray + i, 0, 0, &max_lspl,
				      &max_rspl);

			lspl = (float) abi - (0.5 * (float) iid);
			rspl = (float) abi + (0.5 * (float) iid);
			if (lspl < (max_lspl - attMaxAtten))
				err |= CAL_LEFT_TOO_SOFT;
			if (rspl < (max_rspl - attMaxAtten))
				err |= CAL_RIGHT_TOO_SOFT;
			if (lspl > max_lspl)
				err |= CAL_LEFT_TOO_LOUD;
			if (rspl > max_rspl)
				err |= CAL_RIGHT_TOO_LOUD;
		}

		if (err != 0) {
			sprintf(msg,
				"For some stimuli, requested dB SPL is\n");
			if (err & CAL_LEFT_TOO_LOUD)
				strcat(msg, "too loud in LEFT channel\n");
			if (err & CAL_LEFT_TOO_SOFT)
				strcat(msg, "too soft in LEFT channel\n");
			if (err & CAL_RIGHT_TOO_LOUD)
				strcat(msg, "too loud in RIGHT channel\n");
			if (err & CAL_RIGHT_TOO_SOFT)
				strcat(msg, "too soft in RIGHT channel\n");
			alert(msg);
			return (-1);
		}
	}
	/* All Set: synth and run and run and run ... */

	isitime = isi_start(0);
	for (i = 0; i < nstims && progRunning; i++) {
		set_trial_vars(i, nstims, reps, iid);
		perc_done();

		if (ssarray[i].class == SC_INVALID) {	/* SPONT */
			setRack(0, attMaxAtten, attMaxAtten);
			is_clearOutput(SYN_BOTH);
		} else {
			abi = stimarray[0][i];
			itd = stimarray[1][i];
			iid = stimarray[2][i];
			bc = stimarray[3][i];
			mono = stimarray[4][i];
			two_snd = stimarray[5][i];
			ss = ssarray[i];
			if (!two_snd) {
				synthesize(&ss, NULL, outbuf, outsize,
					   da_channels, delay, dur, epoch,
					   itd, bc, rise, fall, SYN_BOTH);
				lspl = (float) abi - (0.5 * (float) iid);
				rspl = (float) abi + (0.5 * (float) iid);
				if (!cal_SafeFigureAtten(&ss, lspl, rspl, &latten, &ratten)) {
					progRunning = False;
					break;
				}
				if (mono == SYN_RIGHT) {
					is_clearOutput(SYN_LEFT);
					setRack(0, attMaxAtten, ratten);
				} else if (mono == SYN_LEFT) {
					is_clearOutput(SYN_RIGHT);
					setRack(0, latten, attMaxAtten);
				} else {
					setRack(0, latten, ratten);
				}
			} else {
				if (use_ts_dur) {
					two_sound_synthesize(&ss2, NULL, outbuf, outsize,
							     da_channels, 1, epoch, iid - fiid,
							     abi - fabi, &ssf, ts_delay, ts_dur, fitd,
							     fbc, rise, fall, SYN_BOTH, &ss,
							     delay, dur, itd, bc, rise,
							     fall, SYN_BOTH);
				} else {
					two_sound_synthesize(&ss2, NULL,
							     outbuf, outsize, da_channels, 1, epoch,
							     iid - fiid, abi - fabi, &ssf, delay,
							     dur, fitd, fbc, rise, fall,
							     SYN_BOTH, &ss, delay, dur, itd, bc, rise,
							     fall, SYN_BOTH);
				}
				lspl = (float) fabi - (0.5 * (float) fiid);
				rspl = (float) fabi + (0.5 * (float) fiid);
				if (!cal_SafeFigureAtten(&ss2, lspl, rspl, &latten, &ratten)) {
					progRunning = False;
					break;
				}
				if (mono == SYN_RIGHT) {
					is_clearOutput(SYN_LEFT);
					setRack(0, attMaxAtten, ratten);
				} else if (mono == SYN_LEFT) {
					is_clearOutput(SYN_RIGHT);
					setRack(0, latten, attMaxAtten);
				} else {
					setRack(0, latten, ratten);
				}
			}
		}

		switch (mono) {
			case (SYN_LEFT):
				monoc = 'L';
				break;
			case (SYN_RIGHT):
				monoc = 'R';
				break;
			case (SYN_BOTH):
				monoc = 'B';
				break;
		}

		if (ssarray[i].class == SC_INVALID)
			statline_set(status, "this is: SPONT");
		else if (ssarray[i].class == SC_CLICK)
			statline_set(status,
				     "this is: ?? dB; % 4d us; % 3d dB; % 3d %%; %s; %c; %d",
				     itd, iid, bc, ssarray[i].descr, monoc,
				     two_snd);
		else
			statline_set(status,
				     "this is: % 3d dB; % 4d us; % 3d dB; % 3d %%; %s; %c; %d",
				     abi, itd, iid, bc, ssarray[i].descr,
				     monoc, two_snd);

		set_led(3, 1);

		retval = is_sample(X_TRIG | X_EVT | X_DA, epoch, &timestamp);
		if (!(single_step(retval))) {
			alert("is_sample failed: can't continue run");
			set_progRunning(False);
		} else
			MaybePause();

		set_led(3, 0);

		view_input();
		view_output();

		data = getRaster(epoch, &avePMk);

		if (ssarray[i].class == SC_INVALID)
			sprintf(depstr, "depvar=%d <SPONT>\n", SPONT);
		else
			sprintf(depstr, "depvar=%d <%d; %d; %d; %d; %s; %c; %d>\n",
				i, abi, itd, iid, bc, ssarray[i].descr, monoc, two_snd);

		if (avePMk != NULL)
			printAvePMks(avePMk, ((i + 1) % (nstims / reps) ==
				      0) ? ((i + 1) / (nstims / reps)) : 0, nstims / reps);

		write_data_to_xdphysfile(datafp, data, depstr, avePMk,!(i % ana_every),!(i % stim_every), mono);

		isi_stop(isitime);
		isitime = isi_start(isi);
		update_displays(data, 0);

		dloop_empty();

		free(data);
	}

	free(stimarray[0]);
	free(stimarray[1]);
	free(stimarray[2]);
	free(stimarray[3]);
	free(stimarray[4]);
	free(stimarray[5]);
	free(stimarray);
	free(ssarray);


	return (progRunning ? 1 : 0);
}

static int gen_post_fn(char *datafile)
{
	char sys_call[512], *tmp_file, p[MAX_LEN], *tmp;
	FILE *fptr;
	struct timeval *time1;
	float time2;

	if (strcmp(G_sep_files, "0")) {
		assert((tmp_file =
			(char *) calloc(PATH_MAX, sizeof(char))) != NULL);
		getcwd(tmp_file, PATH_MAX);
		assert(strlen(tmp_file) < (PATH_MAX - 13));
		strcat(tmp_file, "/mod_gen.tmp");
		if ((tmp = getenv("XDPHYS_FASTSHELL")) == NULL)
			sprintf(sys_call,
				"mod_gen.gawk -v sep_files=%s %s %s > %s",
				G_sep_files, basename(datafile),
				basename(datafile), tmp_file);
		else
			sprintf(sys_call,
				"%s \"mod_gen.gawk -v sep_files=%s %s %s > %s \" < /dev/null",
				tmp, G_sep_files, datafile, datafile,
				tmp_file);
		time1 = ustime_now();
		system(sys_call);
		time2 = ustime_elapsed(time1);
		printf("%d sec\n", (int) (time2 / 1e6));
		fptr = fopen(tmp_file, "r");
		G_ndatafiles = 0;
		while (!feof(fptr) && G_ndatafiles < MAX_NUM) {
			assert((G_datafile[G_ndatafiles] =
				(char *) malloc(MAX_LEN *
						sizeof(char *))) != NULL);
			if (fgets(p, MAX_LEN, fptr) == NULL) {
				free(G_datafile[G_ndatafiles]);
				break;
			}
			assert(strlen(p) < MAX_LEN);
			printf("%s", p);
			strcpy(G_datafile[G_ndatafiles++], strtok(p, ";"));
		}
		fclose(fptr);
		unlink(tmp_file);
		free(tmp_file);
		if (G_ndatafiles == 1) {
			unlink(findhome(datafile));
		}
		assert(G_ndatafiles < MAX_NUM);
	} else {
		G_ndatafiles = 1;
		assert((G_datafile[0] =
			(char *) malloc(MAX_LEN * sizeof(char *))) !=
		       NULL);
		strcpy(G_datafile[0], datafile);
	}
	return (1);
}

static int gen_display_fn(char *datafile)
{
	FILEDATA *fd;
	FDO *fdo;
	Widget top;
	int i;
	FDObj_ViewType def_view = { 0, 0 };

	top = top_new(TopLevel, "xdphys");
	fdo = FDObj_New(NULL, top, &def_view, 0);
	fdo->destroy_old = 0;
	for (i = 0; i < G_ndatafiles; i++) {
		if ((fd = FD_read(G_datafile[i], 0)) != NULL)
			FDObj_Add(fdo, fd);
		free(G_datafile[i]);
	}

	assert(fdo != NULL);

	FDObj_Update(fdo, NULL);
	popup(top);
	XtAddCallback(fdo->form, XtNdestroyCallback, ClosePShellCB, top);

	return (1);
}

static void Gen_ShiftCB(Widget w, XtPointer foo_shift, XtPointer call_data)
{
	int i;
	char tmp[32], *tmp2;
	int shift = (int) foo_shift;

	ws_tosvars(NULL, NULL);

	i = GI("gen.i");

	sprintf(tmp, "gen.abi%d", i);
	SS(tmp, GS("gen.abi"));
	sprintf(tmp, "gen.iid%d", i);
	SS(tmp, GS("gen.iid"));
	sprintf(tmp, "gen.itd%d", i);
	SS(tmp, GS("gen.itd"));
	sprintf(tmp, "gen.bc%d", i);
	SS(tmp, GS("gen.bc"));
	sprintf(tmp, "gen.stim%d", i);
	SS(tmp, GS("gen.stim"));
	sprintf(tmp, "gen.mono%d", i);
	SS(tmp, GS("gen.mono"));
	sprintf(tmp, "gen.two_snd%d", i);
	SS(tmp, GS("gen.two_snd"));
	sprintf(tmp, "gen.sep_files%d", i);
	SS(tmp, GS("gen.sep_files"));

	i += shift;
	if (i < 0)
		i = NUM_PAGES - 1;
	if (i > NUM_PAGES - 1)
		i = 0;

	sprintf(tmp, "gen.abi%d", i);
	SS("gen.abi", ((tmp2 = GS(tmp)) == NULL) ? "20" : tmp2);
	sprintf(tmp, "gen.iid%d", i);
	SS("gen.iid", ((tmp2 = GS(tmp)) == NULL) ? "0" : tmp2);
	sprintf(tmp, "gen.itd%d", i);
	SS("gen.itd", ((tmp2 = GS(tmp)) == NULL) ? "0" : tmp2);
	sprintf(tmp, "gen.bc%d", i);
	SS("gen.bc", ((tmp2 = GS(tmp)) == NULL) ? "100" : tmp2);
	sprintf(tmp, "gen.stim%d", i);
	SS("gen.stim", ((tmp2 = GS(tmp)) == NULL) ? "bb" : tmp2);
	sprintf(tmp, "gen.mono%d", i);
	SS("gen.mono", ((tmp2 = GS(tmp)) == NULL) ? "B" : tmp2);
	sprintf(tmp, "gen.two_snd%d", i);
	SS("gen.two_snd", ((tmp2 = GS(tmp)) == NULL) ? "0" : tmp2);
	sprintf(tmp, "gen.sep_files%d", i);
	SS("gen.sep_files", ((tmp2 = GS(tmp)) == NULL) ? "0" : tmp2);

	SI("gen.i", i);

	ws_tows(NULL, NULL);
}

Widget setupGen(DockSlot * slot)
{
	Widget ws, h, w1, w2, w3;

	ws = pop_ws_new(TopLevel, ws_table, "gen", &h, NULL, NULL);
	w1 = make_start_temp(h,
			     make_module_info("gen", "gen", "gen", NULL,
					      runScan, gen_post_fn,
					      gen_display_fn, ws_table,
					      7 /*WS_N(ws_table) */ ),
			     NULL, NULL, slot);

	w2 = button_new(XtParent(h), "left", "<-", Gen_ShiftCB, -1, w1,
			NULL);
	accel_from(XtParent(h), w2);

	w3 = button_new(XtParent(h), "left", "->", Gen_ShiftCB, 1, w2,
			NULL);
	accel_from(XtParent(h), w3);

	return (ws);
}
