/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: mod_fr.c,v 1.16 2002/07/15 04:30:12 cmalek Exp $
** Program: dowl
**  Module: xbk.c -> mod_fr.c
**  Author: mazer
** Descrip: calculate freq dependent open-circuit sensistivity
**          for new microphones
**
** Revision History (most recent last)
**  
** Sat Apr 18 19:09:31 1992 mazer
**  creation date :: derrived from xocs.c
**
** Tue Jun 16 15:42:37 1992 mazer
**  changed calls to pow(10,x) to db_to_gain() calls .. fixes
**  off by 10db problem..
**
** Wed Sep 29 01:02:50 1993 mazer
**  major changes..
**   -- sx calculation: calculates ocs values (replaces xocs)
**   -- probe tube compensation mode: reads a probe tube file
**	containing freq dependent amp (db) shifts cause by probe
**      on tube
**
** NOV96 bjarthur
**  incorporated into calib
**
** 97.1 bjarthur
**   changed so that mean,stddev was calculated on dB scale.
**
** 97.4 bjarthur
**   changes stddev to stderror
**
*******************************************************************/

#include "xdphyslib.h"
#include "xcalibur.h"

static int runScan(FILE *, Widget);
Widget setupFr(DockSlot * slot);

static Field ws_table[] = {
	{"fr.range (Hz)", "%s", 30, "1000:13000:500"},
	{"fr.atten (dB)", "%d", 3, "30"},
	{"fr.ambient", "%b", 1, "0"},
	{NULL}
};

static int runScan(FILE * outfile, Widget headline)
{
	int freq;		/* current freq in Hz */
	char *range;		/* range of frequences */
	int *stimarray, nstims;	/* stimulus array derived from range */
	int reps;		/* reps to take at each freq */
	int msdur, tdur;	/* sample parameters */
	int msrise, trise;
	int msfall, tfall;
	int msdelay, tdelay;
	int msepoch, tepoch;
	xword *ibuf;		/* analog input buffer */
	int count;		/* num of input samples */
	int atten;		/* attenuator setting */
	float lgain, rgain;	/* chan a/b gains as mult. factor */
	float linv, rinv;	/* chan a/b invert flag */
	float *lmags, *rmags;	/* measured l/r mv-rms amplitudes */
	float *lphis, *rphis;	/* measured l/r phase values */
	float *ldists, *rdists;	/* measured l/r harmonic distortion */
	float lmag, rmag;	/* mean values for amplitudes -- in mv-rms */
	float lphi, rphi;	/* mean phases */
	float ldist, rdist;	/* mean harmonic distortions */
	float lmagsdv, rmagsdv;	/* standard deviations */
	float lphisdv, rphisdv;
	float SoL, SoR, fooL, fooR;	/* B&K open circuit sens'y (OCS) in mv/Pa */
	int daFc, adFc;
	int ambient;		/* measure ambient SPL(w) */
	float adj_mag, adj_phi;
	float ldbspl, rdbspl;
	syn_spec ss;
	int i, j;
	void *isitime;		/* handle for isi_functions() */
	int isi;		/* inter-stimulus interval */
	char *spk_chan, *ref_chan;
	char *datafile;
	char depstr[256];
	double timestamp;
	int status;
	int outsize, da_channels;
	xword *outbuf;
	int ana_save;
	char *anafile;
	FILE *analogfp;


	/* calculate parameters */
	ws_lock(ws_table, True);

	daFc = GI("daFc");
	adFc = GI("adFc");

	msdur = GI("Dur");
	tdur = (int) ((float) msdur * ((float) daFc) / 1000.0);
	msrise = GI("Rise");
	trise = (int) ((float) msrise * ((float) daFc) / 1000.0);
	msfall = GI("Fall");
	tfall = (int) ((float) msfall * ((float) daFc) / 1000.0);
	msdelay = GI("Delay");
	tdelay = (int) ((float) msdelay * ((float) daFc) / 1000.0);
	msepoch = GI("Epoch");
	tepoch = (int) ((float) msepoch * ((float) daFc) / 1000.0);

	range = GS("fr.range");
	reps = GI("Reps");
	isi = GI("isi");
	atten = GI("fr.atten");

	SoL = GF("So_Left");
	SoR = GF("So_Right");
	spk_chan = GS("Speaker_Channel");
	ref_chan = GS("Reference_Channel");

	ambient = GI("fr.ambient");

	lgain = pow(10.0, GF("Gain_Left") / 20.0);
	rgain = pow(10.0, GF("Gain_Right") / 20.0);
	linv = GI("Invert_Left");
	rinv = GI("Invert_Right");

	ana_save = GI("ana-save");
	if (ana_save) {
		recording(1);
		datafile = GS("xcalibur.fname");
		anafile = (char *) malloc(strlen(datafile) + 10);
		sprintf(anafile, "%s.ana", datafile);
		if ((analogfp = fopen2(findhome(anafile), "w")) == NULL)
			notify
			    ("Warning: Can't write analog data, going on..");
	}

	ws_lock(ws_table, False);
	unlock_worksheets();
	/* no svars should be accessed beyond this point */


	fooL = SoL;
	fooR = SoR;
	if (!load_mic_adjust_data(0, &fooL, &fooR))
		return (0);
	if ((SoL != fooL) || (SoR != fooR)) {
		alert
		    ("So[L,R] in Prefs do not match those in .fr files\nspecified in Mic_Adjust[Left,Right].");
		return (0);
	}

	if ((SoL == 0.0) && (SoR == 0.0)) {
		alert("One of So_Left or So_Right must be non-zero.");
		return (0);
	}

	if ((msdelay + msdur > msepoch) || (msrise + msfall > msdur)) {
		alert
		    ("Something wrong with either Dur, Delay, Epoch, Rise or Fall.");
		return (0);
	}

	if ((*spk_chan != 'L') && (*spk_chan != 'R')) {
		alert("Speaker_Channel must either be 'L' or 'R'.");
		return (0);
	}

	if ((*ref_chan != 'L') && (*ref_chan != 'R')) {
		alert("Reference_Channel must either be 'L' or 'R'.");
		return (0);
	}

	if (((*ref_chan == 'L') && (SoL == 0.0)) ||
	    ((*ref_chan == 'R') && (SoR == 0.0))) {
		alert("Reference_Channel has So of 0.0.");
		return (0);
	}

	if ((stimarray =
	     StimArray_Gen(&nstims, 0, range, 1, 1, 0)) == NULL)
		return (0);

	syn_calibrate = 0;

	lmags = floatVec(reps);
	rmags = floatVec(reps);
	ldists = floatVec(reps);
	rdists = floatVec(reps);
	lphis = floatVec(reps);
	rphis = floatVec(reps);

	is_setNice(GI("nicelevel"));
	is_setSyncPulse(GI("syncPulse"));
	is_init(daFc, adFc, 0.0, msepoch, msepoch);

	da_channels = is_getDAnchans();
	outbuf = is_getDAbuffer(&outsize);

	set_progRunning(True);

	statline_set(headline, "Running fr...");

	fprintf(outfile, ";  1: freq (Hz)\n");
	fprintf(outfile, ";  2: left_mag (mV)\n");
	fprintf(outfile, ";  3: left_phase (rad)\n");
	fprintf(outfile, ";  4: left_distortion (100*f2_mag/f1_mag)\n");
	fprintf(outfile, ";  5: right_mag (mV)\n");
	fprintf(outfile, ";  6: right_phase (rad)\n");
	fprintf(outfile, ";  7: right_distortion (100*f2_mag/f1_mag)\n");
	fprintf(outfile, ";  8: left_mag_stderror (mV)\n");
	fprintf(outfile, ";  9: left_phase_stderror (rad)\n");
	fprintf(outfile, ";  10: right_mag_stderror (mV)\n");
	fprintf(outfile, ";  11: right_phase_stderror(rad)\n");

	fprintf(outfile, "nrasters=%d\n", nstims);

	isitime = isi_start(0);

	for (j = 0; progRunning && j < nstims; j++) {
		percdone((float) j / (float) nstims);
		freq = stimarray[j];
		for (i = 0; i < reps; i++) {
			lmags[i] = rmags[i] = 0.0;
			ldists[i] = rdists[i] = 0.0;
			lphis[i] = rphis[i] = 0.0;
		}

		fd_syn_spec_parse(NULL, freq, &ss);
		is_clearOutput(SYN_BOTH);
		if (!ambient) {
			if (*spk_chan == 'L') {
				synthesize(&ss, NULL, outbuf, outsize,
					   da_channels, msdelay, msdur,
					   msepoch, 0, 100, msrise, msfall,
					   SYN_LEFT);
				is_clearOutput(SYN_RIGHT);
				setRack(0, (float) atten, attMaxAtten);
			} else {
				synthesize(&ss, NULL, outbuf, outsize,
					   da_channels, msdelay, msdur,
					   msepoch, 0, 100, msrise, msfall,
					   SYN_RIGHT);
				is_clearOutput(SYN_LEFT);
				setRack(0, attMaxAtten, (float) atten);
		}} else
			setRack(0, attMaxAtten, attMaxAtten);

		for (i = 0; progRunning && i < reps; i++) {
			set_led(3, 1);

			status =
			    is_sample(X_TRIG | X_AD | X_DA, msepoch,
				      &timestamp);
			if (!(single_step(status))) {
				alert("is_sample failed");
				set_progRunning(False);
			}

			set_led(3, 0);

			view_input();
			view_output();

			ibuf = is_getADbuffer(&count);

			isi_stop(isitime);
			isitime = isi_start(isi);

			dloop_empty();

			plottrace();

			if (ana_save) {
				sprintf(depstr, "depvar=%d\n", freq);
				write_buffers(analogfp, depstr, datafile,
					     i);
			}

			if (clipCheck(ibuf, 0, tepoch, 0.99)) {
				set_progRunning(False);
				alert("Clip on Reference Input");
				break;
			}
			if (clipCheck(ibuf + 1, 0, tepoch, 0.99)) {
				set_progRunning(False);
				alert("Clip on Test Input");
				break;
			}

			dsp_fitsin(ibuf + 2 * (tdelay + trise),
				   tdur - trise - tfall, 2, is_adFc,
				   1.0 * (float) freq, &lmags[i], NULL);
			dsp_fitsin(ibuf + 2 * tdelay, tdur, 2, is_adFc,
				   1.0 * (float) freq, NULL, &lphis[i]);
			dsp_fitsin(ibuf + 2 * (tdelay + trise),
				   tdur - trise - tfall, 2, is_adFc,
				   2.0 * (float) freq, &ldists[i], NULL);

			ldists[i] /= lmags[i];	/* A(F1)/A(F0) */

			if (!ad_to_mv
			    (lmags[i], lmags + i, NULL, AddaRelative)) {
				alert
				    ("Can't convert adticks-rms to mv-rms");
				set_progRunning(False);
				return (0);
			}

			lmags[i] /= lgain;
			query_mic_adjust((float) freq, 'L', &adj_mag,
					 &adj_phi);
			lmags[i] /= adj_mag;
			lphis[i] -= adj_phi;
			if (linv)
				lphis[i] += M_PI;

			lmags[i] /= ROOT2;	/* Amp to RMS */

			dsp_fitsin(ibuf + 1 + 2 * (tdelay + trise),
				   tdur - trise - tfall, 2, is_adFc,
				   1.0 * (float) freq, &rmags[i], NULL);
			dsp_fitsin(ibuf + 1 + 2 * tdelay, tdur, 2, is_adFc,
				   1.0 * (float) freq, NULL, &rphis[i]);
			dsp_fitsin(ibuf + 1 + 2 * (tdelay + trise),
				   tdur - trise - tfall, 2, is_adFc,
				   2.0 * (float) freq, &rdists[i], NULL);

			rdists[i] /= rmags[i];	/* A(F1)/A(F0) */

			if (!ad_to_mv
			    (rmags[i], rmags + i, NULL, AddaRelative)) {
				alert
				    ("Can't convert adticks-rms to mv-rms");
				set_progRunning(False);
				return (0);
			}

			rmags[i] /= rgain;
			query_mic_adjust((float) freq, 'R', &adj_mag,
					 &adj_phi);
			rmags[i] /= adj_mag;
			rphis[i] -= adj_phi;
			if (rinv)
				rphis[i] += M_PI;

			rmags[i] /= ROOT2;
		}		/* Amp to RMS */

		unwrap(lphis, reps, M_PI);
		lmag = mean(lmags, reps);
		lmagsdv = stderror(lmags, reps);
		lphi = mean(lphis, reps);
		lphisdv = stderror(lphis, reps);
		ldist = 100.0 * mean(ldists, reps);

		unwrap(rphis, reps, M_PI);
		rmag = mean(rmags, reps);
		rmagsdv = stderror(rmags, reps);
		rphi = mean(rphis, reps);
		rphisdv = stderror(rphis, reps);
		rdist = 100.0 * mean(rdists, reps);

		if ((SoL != 0.0) && (SoR != 0.0)) {
			ldbspl = 20.0 * log10(lmag / SoL) + 94.0;
			rdbspl = 20.0 * log10(rmag / SoR) + 94.0;
			statline_set(status1,
				     "%5dHz,  %.1fdB,  %.1f%%hd   LEFT",
				     freq, ldbspl, ldist);
			statline_set(status2,
				     "%5dHz,  %.1fdB,  %.1f%%hd   RIGHT",
				     freq, rdbspl, rdist);
		} else if (SoL != 0.0) {
			ldbspl = 20.0 * log10(lmag / SoL) + 94.0;
			statline_set(status1,
				     "%5dHz,  %.1fdB,  %.1f%%hd   LEFT",
				     freq, ldbspl, ldist);
			statline_set(status2,
				     "%5dHz,  %.1fmV,  %.1f%%hd   RIGHT",
				     freq, rmag, rdist);
		} else if (SoR != 0.0) {
			rdbspl = 20.0 * log10(rmag / SoR) + 94.0;
			statline_set(status1,
				     "%5dHz,  %.1fmV,  %.1f%%hd   LEFT",
				     freq, lmag, ldist);
			statline_set(status2,
				     "%5dHz,  %.1fdB,  %.1f%%hd   RIGHT",
				     freq, rdbspl, rdist);
		}

		fprintf(outfile,
			"%5d\t%e\t%e\t%e\t%e\t%e\t%e\t%e\t%e\t%e\t%e\n",
			freq, lmag, lphi, ldist, rmag, rphi, rdist,
			lmagsdv, lphisdv, rmagsdv, rphisdv);

		fflush(outfile);
		dloop_empty();
		MaybePause();
	}

	free(lmags);
	free(rmags);
	free(ldists);
	free(rdists);
	free(lphis);
	free(rphis);

	is_shutdown();

	if (progRunning) {
		statline_set(headline, "Done");
		set_progRunning(False);
		return (1);
	} else {
		statline_set(headline, "Halted");
		return (0);
	}

	if (ana_save)
		fclose(analogfp);
}

Widget setupFr(DockSlot * slot)
{
	Widget ws, h;

	ws = pop_ws_new(TopLevel, ws_table, "fr", &h, NULL, NULL);
	make_start_temp(h,
			make_module_info("fr", "fr", "fr", NULL, runScan,
					 NULL, default_display_fn,
					 ws_table, WS_N(ws_table)), NULL,
			NULL, slot);
	return (ws);
}
