/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: mod_fiid.c,v 2.69 2002/07/15 04:30:12 cmalek Exp $
** Program: dowl
**  Module: mod_fiid.c
**  Author: mazer
** Descrip: monaural response module
**
** Revision History (most recent last)
**
** Sun Mar  8 02:12:22 1992 mazer
**  creation date -- derived from mono.c
**
** Tue Nov  3 10:19:38 1992 mazer
**  moved standard params (Animal, UnitPos etc) into xdowl.c
**
** 97.4 bjarthur
**  added .Side (L,B,R) field
**
** 98.09 bjarthur
**  removed B option from .Side flag so that fd1.c could be used instead
**  of fd2.c.  deleted fd2.c.  see comments in mod_abi for further details
**
*******************************************************************/

#include "xdphyslib.h"
#include "xdphys.h"

/*
** note: fixed < 0 means fix left, fixed > 0 means fix right
*/

static Field ws_table[] = {
	{"fiid.Range (dB)", "%s", 40, "0:50:5"},
	{WS_VSEP},
	{"fiid.Stim", "%s", 30, "0"},
	{"fiid.itd (us)", "%d", 4, "0"},
	{WS_VSEP},
	{"fiid.Fixed (dB)", "%d", 4, "20"},
	{"fiid.Side (L,R)", "%s", 1, "B"},
	{NULL},
};

static int runScan(FILE *, Widget);
Widget setupFiid(DockSlot *);

static int runScan(FILE * datafp, Widget status)
{
	int fspl;		/* fixed level in dbspl */
	int itd;		/* constant itd in us */
	int delay;		/* stimulus latency in ms */
	int dur;		/* stim durationin ms */
	int epoch;		/* recording epoch in ms */
	int isi;		/* inter-stimulus interval */
	int rise, fall;		/* rise and fall times in ms */
	int nstims, reps;	/* number & reps of stimuli to present */
	int *stimarray;		/* sequence of itds to present */
	spike_t *data;		/* pointer to raster data */
	float latten, ratten;	/* digital attenuator settings */
	float lspl, rspl;	/* desired dbspl levels */
	float max_lspl, max_rspl;	/* max dbspl levels */
	char *range;		/* range of stimuli -- dbspl */
	int maxitd;		/* in ms */
	void *isitime;		/* handle for isi_functions() */
	char depstr[256];
	int i;
	float *avePMk;
	syn_spec ss;
	char *stimspec;
	int err;
	char msg[1024];
	char *side;
	int rand, spont_stims;
	float daFc, evtFc;
	int ana_every;
	int stim_every;
	double timestamp;
	int retval;
	int outsize, da_channels;
	xword *outbuf;

	/* calculate parameters */
	ws_lock(ws_table, True);

	stimspec = GS("fiid.Stim");
	range = GS("fiid.Range");
	fspl = GI("fiid.Fixed");
	reps = GI("reps");
	dur = GI("dur");
	delay = GI("delay");
	itd = GI("fiid.itd");
	maxitd = RND2INT(1.e-6 * ((itd > 0) ? itd : -itd));
	epoch = GI("epoch");
	rise = GI("rise");
	fall = GI("fall");
	isi = GI("isi");
	side = GS("fiid.Side");

	rand = GI("randomize");
	spont_stims = GI("Spont_Stims");
	daFc = GF("daFc");
	evtFc = GF("evtFc");

	ana_every = GI("ana.every");
	stim_every = GI("stim.every");

	ws_lock(ws_table, False);
	unlock_most_worksheets();
	/* no svars should be accessed beyond this point */

	fd_syn_spec_parse(stimspec, 0, &ss);
	if (ss.class == SC_INVALID) {
		alert("Stim: \"%s\" illegal syn_class\n%s",
		      stimspec, synthesize_help());
		return (-1);
	}
	syn_calibrate = 1;

	if (strcmp(side, "L") && strcmp(side, "R")) {
		alert("Side parameter must be either L or R.");
		return (0);
	}
	if ((stimarray = StimArray_Gen(&nstims, rand, range, reps, 1,
				       spont_stims)) == NULL) {
		return (-1);
	}

	is_setNice(GI("nicelevel"));
	is_setSyncPulse(GI("syncPulse"));

	is_init(daFc, 0.0, evtFc, dur + delay + maxitd, epoch);

	da_channels = is_getDAnchans();
	outbuf = is_getDAbuffer(&outsize);

	/* check to see if att can handle all SPL's */

	synthesize(&ss, NULL, outbuf, outsize, da_channels, delay, dur,
		   epoch, 0, 100, rise, fall, SYN_BOTH);
	cal_FigureSPL(&ss, 0, 0, &max_lspl, &max_rspl);

	for (i = 0, err = 0; (i < nstims) && progRunning; i++) {
		if (stimarray[i] == SPONT)
			continue;
		if (!strcmp(side, "L")) {
			lspl = stimarray[i];
			rspl = fspl;
		} else {
			lspl = fspl;
			rspl = stimarray[i];
		}
		if (lspl < (max_lspl - attMaxAtten))
			err |= CAL_LEFT_TOO_SOFT;
		if (rspl < (max_rspl - attMaxAtten))
			err |= CAL_RIGHT_TOO_SOFT;
		if (lspl > max_lspl)
			err |= CAL_LEFT_TOO_LOUD;
		if (rspl > max_rspl)
			err |= CAL_RIGHT_TOO_LOUD;
	}

	if (err != 0) {
		sprintf(msg, "For some stimuli, requested dB SPL is\n");
		if (err & CAL_LEFT_TOO_LOUD)
			strcat(msg, "too loud in LEFT channel\n");
		if (err & CAL_LEFT_TOO_SOFT)
			strcat(msg, "too soft in LEFT channel\n");
		if (err & CAL_RIGHT_TOO_LOUD)
			strcat(msg, "too loud in RIGHT channel\n");
		if (err & CAL_RIGHT_TOO_SOFT)
			strcat(msg, "too soft in RIGHT channel\n");
		alert(msg);
		return (-1);
	}
	/* All Set: synth and run and run and run ... */

	fprintf(datafp, "nrasters=%d\n", nstims);

	isitime = isi_start(0);
	for (i = 0; i < nstims && progRunning; i++) {
		set_trial_vars(i, nstims, reps, stimarray[i]);
		perc_done();

		if (stimarray[i] == SPONT) {
			setRack(0, attMaxAtten, attMaxAtten);
			is_clearOutput(SYN_BOTH);
		} else {
			synthesize(&ss, NULL, outbuf, outsize, da_channels,
				   delay, dur, epoch, itd, 100, rise, fall,
				   SYN_BOTH);
			if (!strcmp(side, "L")) {
				lspl = stimarray[i];
				rspl = fspl;
			} else {
				lspl = fspl;
				rspl = stimarray[i];
			}
			if (!cal_SafeFigureAtten
			    (&ss, lspl, rspl, &latten, &ratten)) {
				progRunning = False;
				break;
			}
			setRack(0, latten, ratten);
		}

		if (stimarray[i] == SPONT)
			statline_set(status, "this is: SPONT");
		else
			statline_set(status, "this is: L=%ddB, R=%ddB",
				     (int) lspl, (int) rspl);

		set_led(3, 1);

		retval =
		    is_sample(X_TRIG | X_EVT | X_DA, epoch, &timestamp);
		if (!(single_step(retval))) {
			alert("is_sample failed: can't continue run");
			set_progRunning(False);
		} else
			MaybePause();

		set_led(3, 0);

		view_input();
		view_output();

		data = getRaster(epoch, &avePMk);

		if (stimarray[i] == SPONT)
			sprintf(depstr, "depvar=%d <SPONT>\n", SPONT);
		else
			sprintf(depstr, "depvar=%d <%ddB>\n",
				(!strcmp(side, "L")) ? (int) lspl : (int)
				rspl,
				(!strcmp(side, "L")) ? (int) lspl : (int)
				rspl);

		if (avePMk != NULL)
			printAvePMks(avePMk,
				     ((i + 1) % (nstims / reps) ==
				      0) ? ((i + 1) / (nstims / reps)) : 0,
				     nstims / reps);

		write_data_to_xdphysfile(datafp, data, depstr, avePMk,!(i % ana_every),!(i % stim_every), SYN_BOTH);

		update_displays(data, 0);
		isi_stop(isitime);
		isitime = isi_start(isi);

		dloop_empty();

		free(data);
	}

	free(stimarray);
	return (progRunning ? 1 : 0);
}

Widget setupFiid(DockSlot * slot)
{
	Widget ws, h;

	ws = pop_ws_new(TopLevel, ws_table, "fiid", &h, NULL, NULL);
	make_start_temp(h,
			make_module_info("intensity (dB)", "fiid", "fiid",
					 NULL, runScan, NULL,
					 default_display_fn, ws_table,
					 WS_N(ws_table)), NULL, NULL,
			slot);
	return (ws);
}
