/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: mod_cf.c,v 1.22 2002/07/15 04:30:11 cmalek Exp $
** Program: dowl
**  Module: mod_freq.c
**  Author: mazer
** Descrip: xdowl application module -- BF tuning curve
**
** Revision History (most recent last)
**
** 97.4 bjarthur    <--- creation date
**  copied from mod_bf.c
**
*******************************************************************/

#include "xdphyslib.h"
#include "xdphys.h"

static Field ws_table[] = {
	{"cf.Range (Hz)", "%s", 30, "1000:12000:500"},
	{"cf.Resolution (dB)", "%d", 2, "2"},
	{"cf.High (dB)", "%d", 3, "40"},
	{"cf.Low (dB)", "%d", 3, "-20"},
	{"cf.p-val", "%f", 5, "0.05"},
	{WS_VSEP},
	{"cf.itd (us)", "%d", 4, "0"},
	{"cf.iid (dB)", "%d", 3, "0"},
	{WS_VSEP},
	{"cf.mono (L,B,R)", "%s", 8, "B"},
	{NULL},
};

static int runScan(FILE *, Widget);
Widget setupCf(DockSlot *);

static int runScan(FILE * datafp, Widget status)
{
	int freq;		/* current freq in Hz */
	int itd;		/* constant (best) itd in us */
	float iid;		/* iid value (or 0 for monoaural) */
	int delay;		/* stimulus latency in ms */
	int dur;		/* stim dur & max DAC duration in ms */
	int epoch;		/* recording epoch in ms */
	int isi;		/* inter-stimulus interval */
	int rise, fall;		/* rise and fall times in ms */
	int nstims, nits, reps;	/* number & reps of stimuli to present */
	int *stimarray, *index;	/* sequence of freqs to present */
	spike_t *data;		/* pointer to raster data */
	float latten, ratten;	/* digital attenuator settings */
	float lspl, rspl;	/* desired dbspl levels */
	char *range;		/* range of stimuli -- Hz */
	int maxitd;		/* in ms */
	char *monoflagstr;	/* l,r,b or -1,0,1 */
	int monoflag;		/* 0=bin, -1=left mono, +1 =right mono */
	void *isitime;		/* handle for isi_functions() */
	char depstr[256];
	int i, j, idx;
	float *avePMk;
	syn_spec ss;
	char tmp[64];
	int *invalid;
	float *high_vec, *low_vec, high, low, res;
	int nstims_per_rep;
	float pval;
	int spont_stims, use_delay, err;
	float t_begin, t_end, t_spont_begin, t_spont_end;
	float **intensities;
	float abi;
	char msg[1024];
	float max_lspl, max_rspl;	/* max dbspl levels */
	int rand;
	float daFc, evtFc;
	int ana_every;
	int stim_every;
	double timestamp;
	int retval;
	int outsize, da_channels;
	xword *outbuf;


	/* calculate parameters */
	ws_lock(ws_table, True);

	reps = GI("reps");
	dur = GI("dur");
	delay = GI("delay");
	epoch = GI("epoch");
	rise = GI("rise");
	fall = GI("fall");
	isi = GI("isi");
	evtFc = GF("evtFc");
	daFc = GF("daFc");
	ana_every = GI("ana.every");
	stim_every = GI("stim.every");
	rand = GI("randomize");

	range = GS("cf.Range");
	monoflagstr = GS("cf.mono");
	iid = GF("cf.iid");
	itd = GI("cf.itd");
	high = (float) GI("cf.High");
	low = (float) GI("cf.Low");
	res = (float) GI("cf.Resolution");
	pval = GF("cf.p-val");

	ws_lock(ws_table, False);
	unlock_most_worksheets();

	/* no svars should be accessed beyond this point */

	FD_calc_params(NULL, &spont_stims, &use_delay,
		       &t_begin, &t_end, &t_spont_begin, &t_spont_end);

	/* determine monoflag.. */
	if (strncasecmp(monoflagstr, "l", 1) == 0)
		monoflag = SYN_LEFT;	/* left-mono */
	else if (strncasecmp(monoflagstr, "r", 1) == 0)
		monoflag = SYN_RIGHT;	/* right-mono */
	else if (strncasecmp(monoflagstr, "b", 1) == 0)
		monoflag = SYN_BOTH;	/* binaural */
	else {
		alert("\"mono\" must be L, R or B");
		return (-1);
	}

	if (monoflag != SYN_BOTH && iid != 0.0) {
		notify("Warning: MONO curve -> using IID=0");
		iid = 0.0;
	}
	maxitd = RND2INT(1.e-6 * ((itd > 0) ? itd : -itd));


	syn_calibrate = 1;

	is_setNice(GI("nicelevel"));
	is_setSyncPulse(GI("syncPulse"));

	is_init(daFc, 0.0, evtFc, dur + delay + maxitd, epoch);

	da_channels = is_getDAnchans();
	outbuf = is_getDAbuffer(&outsize);

	/* gen stimarray and alloc memory */

	if ((stimarray = StimArray_Gen(&nstims, rand, range, reps,
				       1, spont_stims)) == NULL)
		return (-1);
	if ((index = StimArray_Gen(&nstims_per_rep, 0, range, 1, 1, 1)) == NULL)
		return (-1);
	qsort(index, nstims_per_rep, sizeof(int), intcompare);
	assert((intensities =
		(float **) malloc(nstims_per_rep * sizeof(float *))) != NULL);
	assert((high_vec =
		(float *) malloc(nstims_per_rep * sizeof(float))) != NULL);
	assert((low_vec =
		(float *) malloc(nstims_per_rep * sizeof(float))) != NULL);
	assert((invalid =
		(int *) calloc(nstims_per_rep, sizeof(int))) != NULL);
	for (i = 0; i < nstims_per_rep; i++) {
		if ((i == 0) && (!spont_stims))
			assert((intensities[i] =
				(float *) malloc(nstims *
						 sizeof(float))) != NULL);
		else
			assert((intensities[i] =
				(float *) malloc(reps * sizeof(float))) !=
			       NULL);
		high_vec[i] = high;
		low_vec[i] = low;
	}

	/* check to see if att can handle all abi's */

	for (i = 0, err = 0; (i < nstims_per_rep) && progRunning; i++) {
		if (stimarray[i] == SPONT)
			continue;
		freq = (float) stimarray[i];
		sprintf(tmp, "tone=%d", freq);
		fd_syn_spec_parse(tmp, 0, &ss);
		synthesize(&ss, NULL, outbuf, outsize, da_channels, delay,
			   dur, epoch, itd, 100, rise, fall, SYN_BOTH);
		cal_FigureSPL(&ss, 0, 0, &max_lspl, &max_rspl);

		lspl = high - (0.5 * iid);
		rspl = high + (0.5 * iid);
		if (lspl < (max_lspl - attMaxAtten))
			err |= CAL_LEFT_TOO_SOFT;
		if (rspl < (max_rspl - attMaxAtten))
			err |= CAL_RIGHT_TOO_SOFT;
		if (lspl > max_lspl)
			err |= CAL_LEFT_TOO_LOUD;
		if (rspl > max_rspl)
			err |= CAL_RIGHT_TOO_LOUD;

		lspl = low - (0.5 * iid);
		rspl = low + (0.5 * iid);
		if (lspl < (max_lspl - attMaxAtten))
			err |= CAL_LEFT_TOO_SOFT;
		if (rspl < (max_rspl - attMaxAtten))
			err |= CAL_RIGHT_TOO_SOFT;
		if (lspl > max_lspl)
			err |= CAL_LEFT_TOO_LOUD;
		if (rspl > max_rspl)
			err |= CAL_RIGHT_TOO_LOUD;
	}

	if (err != 0) {
		sprintf(msg, "For some stimuli, requested dB SPL is\n");
		if (err & CAL_LEFT_TOO_LOUD)
			strcat(msg, "too loud in LEFT channel\n");
		if (err & CAL_LEFT_TOO_SOFT)
			strcat(msg, "too soft in LEFT channel\n");
		if (err & CAL_RIGHT_TOO_LOUD)
			strcat(msg, "too loud in RIGHT channel\n");
		if (err & CAL_RIGHT_TOO_SOFT)
			strcat(msg, "too soft in RIGHT channel\n");
		alert(msg);
		return (-1);
	}
	nits = 2 + (int) ceil(log((double) (high - low) / res) / log(2.0));
	fprintf(datafp, "nrasters=%d\n", nstims * nits);

	/* All Set: synth and run and synth and run ... */

	err = 0;
	isitime = isi_start(0);
	for (j = 0; j < nits && progRunning; j++) {
		for (i = 0; i < nstims && progRunning; i++) {
			freq = stimarray[i];

			if (j == 0) {
				abi = high;
				lspl = abi - (0.5 * iid);
				rspl = abi + (0.5 * iid);
			} else if (j == 1) {
				idx = (int *) bsearch(&freq, index, nstims_per_rep,
						    sizeof(int), intcompare) - index;
				if (invalid[idx])
					continue;
				abi = low;
				lspl = abi - (0.5 * iid);
				rspl = abi + (0.5 * iid);
			} else {
				idx = (int *) bsearch(&freq, index, nstims_per_rep, sizeof(int),
						    intcompare) - index;
				if (invalid[idx])
					continue;
				abi = (high_vec[idx] + low_vec[idx]) / 2.0;
				lspl = abi - (0.5 * iid);
				rspl = abi + (0.5 * iid);
			}

			set_trial_vars(i + j * nstims, nstims * nits, reps * nits, freq);
			perc_done();

			if (freq == SPONT) {
				setRack(0, attMaxAtten, attMaxAtten);
				is_clearOutput(SYN_BOTH);
			} else {
				sprintf(tmp, "tone=%d", freq);
				fd_syn_spec_parse(tmp, 0, &ss);
				synthesize(&ss, NULL, outbuf, outsize,
					   da_channels, delay, dur, epoch,
					   itd, 100, rise, fall, SYN_BOTH);
				if (!cal_SafeFigureAtten(&ss, lspl, rspl, &latten, &ratten)) {
					progRunning = False;
					break;
				}
				if (monoflag == SYN_RIGHT) {
					is_clearOutput(SYN_RIGHT);
					setRack(0, latten, attMaxAtten);
				} else if (monoflag == SYN_LEFT) {
					is_clearOutput(SYN_LEFT);
					setRack(0, attMaxAtten, ratten);
				} else
					setRack(0, latten, ratten);
			}

			if (freq == SPONT)
				statline_set(status, "this is: SPONT");
			else
				statline_set(status, "this freq: %dHz", freq);

			set_led(3, 1);

			retval = is_sample(X_TRIG | X_EVT | X_DA, epoch, &timestamp);
			if (!(single_step(retval))) {
				alert ("is_sample failed: can't continue run");
				set_progRunning(False);
			} else
				MaybePause();

			set_led(3, 0);

			view_input();
			view_output();

			data = getRaster(epoch, &avePMk);

			idx = (int *) bsearch(&freq, index, nstims_per_rep,
					    sizeof(int), intcompare) - index;
			intensities[idx][i / (nstims / reps)] =
			    (float) countRaster(t_begin, t_end, data,
						lookupParms_int("parms.spikemask"));
			if (!spont_stims)
				intensities[0][i] =
				    (float) countRaster(t_spont_begin, t_spont_end, data, 
							lookupParms_int("parms.spikemask"));

			if (freq == SPONT)
				sprintf(depstr, "depvar=%d <SPONT, iter#%d>\n", freq, j);
			else
				sprintf(depstr, "depvar=%d <%d dB, iter #%d>\n",
					freq, (int) abi, j);

			if (avePMk != NULL)
				printAvePMks(avePMk, ((i + 1) % (nstims / reps) ==
					      0) ? ((i + 1) / (nstims / reps)) : 0, nstims / reps);

			write_data_to_xdphysfile(datafp, data, depstr, avePMk,
				!(i % ana_every), !(i % stim_every), monoflag);

			update_displays(data, 0);
			isi_stop(isitime);
			isitime = isi_start(isi);

			dloop_empty();
			free(data);
		}

		if (!progRunning)
			continue;
		for (err = 0, i = 1; i < nstims_per_rep; i++) {	/* start w/ 1 to skip spont */
			if (invalid[i])
				continue;
			if (j == 0) {
				if (!FD_compare_intensities
				    (intensities[i], reps, intensities[0],
				     (spont_stims ? reps : nstims),
				     pval)) {
					invalid[i] = 1;
					err = 1;
				}
			} else if (j == 1) {
				if (FD_compare_intensities
				    (intensities[i], reps, intensities[0],
				     (spont_stims ? reps : nstims),
				     pval)) {
					invalid[i] = 1;
					err = 1;
				}
			} else {
				if (FD_compare_intensities
				    (intensities[i], reps, intensities[0],
				     (spont_stims ? reps : nstims), pval))
					high_vec[i] =
					    (high_vec[i] + low_vec[i]) / 2;
				else
					low_vec[i] =
					    (high_vec[i] + low_vec[i]) / 2;
			}
		}

/*
   fprintf(stderr,"\nhigh = ");
   for(i=1; i<nstims_per_rep; i++)
   fprintf(stderr,"%5.1f, ",high_vec[i]);
   fprintf(stderr,"\nlow = ");
   for(i=1; i<nstims_per_rep; i++)
   fprintf(stderr,"%5.1f, ",low_vec[i]);
   fprintf(stderr,"\ninvalid = ");
   for(i=1; i<nstims_per_rep; i++)
   fprintf(stderr,"%5.1f, ",invalid[i]);
   fprintf(stderr,"\n\n");
 */

		if ((j == 0) && err)
			notify
			    ("For some frequencies, High is not above threshold.");
		if ((j == 1) && err)
			notify
			    ("For some frequencies, Low is not below threshold.");
	}

	free(stimarray);
	free(high_vec);
	free(low_vec);
	for (i = 0; i < nstims_per_rep; i++)
		free(intensities[i]);
	free(intensities);

	return (progRunning ? 1 : 0);
}

Widget setupCf(DockSlot * slot)
{
	Widget ws, h;

	ws = pop_ws_new(TopLevel, ws_table, "cf", &h, NULL, NULL);
	make_start_temp(h,
			make_module_info("cf (Hz)", "cf", "cf", NULL,
					 runScan, NULL, default_display_fn,
					 ws_table, WS_N(ws_table)), NULL,
			NULL, slot);
	return (ws);
}
