/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: mod_bw.c,v 1.16 2002/07/15 04:30:11 cmalek Exp $
** Program: dowl
**  Module: xbkbw.c
**  Author: mazer/bjarthur
** Descrip: calculate freq dependent open-circuit sensistivity
**          for new microphones
**
** Revision History (most recent last)
**  
** August 1996 bjarthur
**   dup'd from xbkint.c and modified to check output intensity 
**   with multiple freqs.  perhaps it's not so wise to assume intensity
**   doesn't change w/ bandwidth.  needs an ear.dbspl file.  see
**   make_calib_file script.
**
** NOV96 bjarthur
**  copied from xbkbw.c
**
** 97.1 bjarthur
**   changed so that mean,stddev was calculated on dB scale.
**
** 97.4 bjarthur
**   changed stddev to stderror
**
*******************************************************************/

#include "xdphyslib.h"
#include "xcalibur.h"
#include "synthlib.h"

static Field ws_table[] = {
	{"bw.bw_range (Hz)", "%s", 20, "1:5000:@15"},
	{"bw.cf (Hz)", "%d", 5, "7000"},
	{"bw.SPL (dB)", "%d", 4, "50"},
	{"bw.check (TS,WN)", "%s", 2, "WN"},
	{"bw.check_file", "%s", 30, ""},
	{NULL}
};

static int runScan(FILE *, Widget);
Widget setupInt(DockSlot * slot);

static int runScan(FILE * outfile, Widget headline)
{
	float bw, cf;		/* current freq in Hz */
	float spl;		/* current freq in Hz */
	char *range;		/* range of intensities */
	int *stimarray, nstims;	/* stimulus array derived from range */
	int reps;		/* reps to take at each freq */
	int msdur, tdur;	/* sample parameters */
	int msrise, trise;
	int msfall, tfall;
	int msdelay, tdelay;
	int msepoch, tepoch;
	xword *ibuf;		/* analog input buffer */
	int count;		/* num of input samples; measure over [a,b] */
	int ref_off;		/* offset values (0 or 1) */
	float gain;		/* chan a/b gains as mult. factor */
	float *rmss;		/* measured l/r mv-rms amplitudes */
	float rms, rms_sd;	/* mean values for amplitudes -- in mv-rms */
	float So;		/* B&K open circuit sens'y (OCS) in mv/Pa */
	int daFc, adFc;
	char tmp[128];
	float atten;
	syn_spec ss;
	int i, j;
	int min_bw;
	char *check, *check_file;
	int ad_channels;
	void *isitime;		/* handle for isi_functions() */
	int isi;		/* inter-stimulus interval */
	char *spk_chan, *ref_chan;
	double timestamp;
	int status;
	int outsize, da_channels;
	xword *outbuf;

	ad_channels = is_getADnchans();

	/* calculate parameters */
	daFc = GI("daFc");
	adFc = GI("adFc");

	msdur = GI("Dur");
	tdur = (int) ((float) msdur * ((float) daFc) / 1000.0);
	msrise = GI("Rise");
	trise = (int) ((float) msrise * ((float) daFc) / 1000.0);
	msfall = GI("Fall");
	tfall = (int) ((float) msfall * ((float) daFc) / 1000.0);
	msdelay = GI("Delay");
	tdelay = (int) ((float) msdelay * ((float) daFc) / 1000.0);
	msepoch = GI("Epoch");
	tepoch = (int) ((float) msepoch * ((float) daFc) / 1000.0);

	range = GS("bw.bw_range");
	cf = (float) GI("bw.cf");
	spl = (float) GI("bw.SPL");
	reps = GI("reps");
	isi = GI("isi");

	spk_chan = GS("Speaker_Channel");
	ref_chan = GS("Reference_Channel");

	check = GS("bw.check");
	check_file = GS("bw.check_file");

	if ((*ref_chan) == 'L') {
		ref_off = 0;
		So = GF("So_Left");
		gain = pow(10.0, GF("Gain_Left") / 20.0);
	} else if ((*ref_chan) == 'R') {
		ref_off = 1;
		So = GF("So_Right");
		gain = pow(10.0, GF("Gain_Right") / 20.0);
	}

	ws_lock(ws_table, False);
	unlock_worksheets();
	/* no svars should be accessed beyond this point */


	if ((msdelay + msdur > msepoch) || (msrise + msfall > msdur)) {
		alert
		    ("Something wrong with either Dur, Delay, Epoch, Rise or Fall.");
		return (0);
	}

	if ((*spk_chan != 'L') && (*spk_chan != 'R')) {
		alert("Speaker_Channel must either be 'L' or 'R'.");
		return (0);
	}

	if ((*ref_chan != 'L') && (*ref_chan != 'R')) {
		alert("Reference_Channel must either be 'L' or 'R'.");
		return (0);
	}

	if (strcmp(check, "TS") && strcmp(check, "WN")) {
		alert("check must either be either 'TS' or 'WN'.");
		return (0);
	}

	if (!load_calib_data(check_file, CAL_RELOAD_YES))
		return (0);
	syn_calibrate = 1;

	if (So == 0.0) {
		alert("Reference_Channel has So of 0.0.");
		return (0);
	}

	if ((stimarray = StimArray_Gen(&nstims, 0, range,
				       1, 0, 0)) == NULL)
		return (0);

	rmss = floatVec(reps);

	is_setNice(GI("nicelevel"));
	is_setSyncPulse(GI("syncPulse"));
	is_init(daFc, adFc, 0.0, msepoch, msepoch);

	da_channels = is_getDAnchans();
	outbuf = is_getDAbuffer(&outsize);

	set_progRunning(True);

	fd_syn_spec_parse("hp", 0, &ss);
	synthesize(&ss, NULL, outbuf, outsize, da_channels, msdelay, msdur,
		   msepoch, 0, 100, msrise, msfall, SYN_LEFT);
	min_bw = (int) ceil(ss.parms.noise.res);
	for (j = 0; j < nstims; j++) {
		if (stimarray[j] < min_bw) {
			alert
			    ("BW.range specifies a bandwidth that is smaller than\nthe minimum resolvable frequency of %d Hz.",
			     min_bw);
			return (0);
		}
	}

	statline_set(headline, "Running bw...");

	fprintf(outfile, "; 1: bandwidth (Hz)\n");
	fprintf(outfile, "; 2: mag (dB)\n");
	fprintf(outfile, "; 3: mag_stderror (dB)\n");
	fprintf(outfile, "nrasters=%d\n", nstims);

	syn_calibrate = 1;

	isitime = isi_start(0);

	for (j = 0; (j < nstims) && progRunning; j++) {
		percdone((float) j / (float) nstims);
		bw = (float) stimarray[j];
		is_clearOutput(SYN_BOTH);
		if (!strcmp(check, "WN"))
			sprintf(tmp, "CF=%d,BW=%d", (int) cf, (int) bw);
		else if (!strcmp(check, "TS"))
			sprintf(tmp, "STACK=%d:%d:%d",
				(int) (cf - 0.5 * bw),
				(int) (cf + 0.5 * bw), stimarray[0]);
		fd_syn_spec_parse(tmp, 0, &ss);

		if (*spk_chan == 'L') {
			synthesize(&ss, NULL, outbuf, outsize, da_channels,
				   msdelay, msdur, msepoch, 0, 100, msrise,
				   msfall, SYN_LEFT);
			is_clearOutput(SYN_RIGHT);
			cal_SafeFigureAtten(&ss, spl, 0.0, &atten, NULL);
			setRack(0, atten, attMaxAtten);
		} else if (*spk_chan == 'R') {
			synthesize(&ss, NULL, outbuf, outsize, da_channels,
				   msdelay, msdur, msepoch, 0, 100, msrise,
				   msfall, SYN_RIGHT);
			is_clearOutput(SYN_LEFT);
			cal_SafeFigureAtten(&ss, 0.0, spl, NULL, &atten);
			setRack(0, attMaxAtten, atten);
		}

		for (i = 0; progRunning && i < reps; i++) {
			set_led(3, 1);

			status =
			    is_sample(X_TRIG | X_AD | X_DA, msepoch,
				      &timestamp);
			if (!(single_step(status))) {
				alert("is_sample failed");
				set_progRunning(False);
			}

			set_led(3, 0);

			view_input();
			view_output();

			ibuf = is_getADbuffer(&count);
			isi_stop(isitime);
			isitime = isi_start(isi);
			dloop_empty();

			if (clipCheck(ibuf + ref_off, 0, tepoch, 0.99)) {
				set_progRunning(False);
				alert("Clip on Reference Input");
				break;
			}

			plottrace();

			rmss[i] =
			    dsp_calcrms(ibuf + ref_off +
					2 * (tdelay + trise),
					tdur - trise - tfall, ad_channels,
					1);

			if (*ref_chan == 'L') {
				if (!ad_to_mv
				    (rmss[i], rmss + i, NULL,
				     AddaRelative)) {
					alert
					    ("Can't convert adticks-rms to mv-rms");
					set_progRunning(False);
					break;
				}
			} else if (*ref_chan == 'R') {
				if (!ad_to_mv
				    (rmss[i], NULL, rmss + i,
				     AddaRelative)) {
					alert
					    ("Can't convert adticks-rms to mv-rms");
					set_progRunning(False);
					break;
				}
			}

			rmss[i] /= gain;
			rmss[i] = 20.0 * log10(rmss[i] / So) + 94.0;
		}

		rms = mean(rmss, reps);
		rms_sd = stderror(rmss, reps);

		statline_set(status1, "CF=%d,  BW=%d,  %3.2fdB", (int) cf,
			     (int) bw, rms);

		fprintf(outfile, "%5d\t%e\t%e\n", stimarray[j], rms,
			rms_sd);

		fflush(outfile);
		dloop_empty();
		MaybePause();
	}

	free(rmss);

	is_shutdown();

	if (progRunning) {
		statline_set(headline, "Done");
		set_progRunning(False);
		return (1);
	} else {
		statline_set(headline, "Halted");
		return (0);
	}
}

Widget setupBw(DockSlot * slot)
{
	Widget ws, h;

	ws = pop_ws_new(TopLevel, ws_table, "bw", &h, NULL, NULL);
	make_start_temp(h,
			make_module_info("bw", "bw", "bw", NULL, runScan,
					 NULL, default_display_fn,
					 ws_table, WS_N(ws_table)), NULL,
			NULL, slot);
	return (ws);
}
