/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: mod_bja.c,v 2.7 2002/07/15 04:30:11 cmalek Exp $
** Program: dowl
**  Module: mod_bja.c
**  Author: arthur
** Descrip: xdowl application module
**
** Revision History (most recent last)
**
** 00.05 bjarthur
**  something just for me.  to counter all those naysayers with
**  a control expt.
*******************************************************************/

#include "xdphyslib.h"
#include "xdphys.h"

static Field ws_table[] = {
	{"bja.abi (dB)", "%s", 60, "20"},
	{"bja.itd (us)", "%s", 60, "0"},
	{"bja.iid (dB)", "%s", 60, "0"},
	{"bja.stim", "%s", 60, "bb"},
	{"bja.range (Hz)", "%s", 60, "64,128,256,512,1024,2048,4096"},
	{NULL},
};

void bja_synthesize(int, syn_spec *, xword *,
		    int, int, int, int, int, int, int, int, int, int);
static int runScan(FILE *, Widget);
Widget setupGen(DockSlot *);

void bja_synthesize(int stim, syn_spec * ss, xword * outbuf,
		    int outsize, int da_channels, int delay, int dur,
		    int epoch, int itd, int bc, int rise, int fall, int lr)
{
	float *fwave, tmp, phi;
	int i, j;
	float lb, la, rb, ra;

	assert((fwave =
		(float *) malloc(da_channels * epoch * (is_daFc / 1000) *
				 sizeof(float))) != NULL);

	synthesize(ss, fwave, NULL, outsize, da_channels,
		   delay, dur, epoch, itd, bc, rise, fall, SYN_BOTH);

	phi = floor(getRad() / (2.0 * M_PI) * outsize);
	phi = (phi / is_daFc) * stim * M_PI;

	lb = la = rb = ra = 0.0;
	for (i = 0, j = 0; i < outsize; i += da_channels, j++) {
		tmp = cos((((float) j) / is_daFc) * stim * M_PI + phi);

		lb += (fwave[i] * fwave[i]);
		fwave[i] *= tmp;
		la += (fwave[i] * fwave[i]);

		rb += (fwave[i + 1] * fwave[i + 1]);
		fwave[i + 1] *= tmp;
		ra += (fwave[i + 1] * fwave[i + 1]);
	}

	ss->lrms *= sqrt(la / lb);
	ss->rrms *= sqrt(ra / rb);

	syn_transfer_to_outbuf(fwave, outbuf, epoch * (is_daFc / 1000),
			       da_channels);
	syn_transfer_to_outbuf(fwave + 1, outbuf + 1,
			       epoch * (is_daFc / 1000), da_channels);

	free(fwave);
}

static int runScan(FILE * datafp, Widget status)
{
	int i;
	int iid;		/* current iid in db */
	int abi;		/* abi for the run */
	int itd;		/* constant itd in us */
	int bc;
	int delay;		/* stimulus latency in ms */
	int dur;		/* stim durationin ms */
	int epoch;		/* recording epoch in ms */
	int isi;		/* inter-stimulus interval */
	int rise, fall;		/* rise and fall times in ms */
	int nstims, reps;	/* number & reps of stimuli to present */
	int *stimarray;		/* sequence of itds to present */
	spike_t *data;		/* pointer to raster data */
	float latten, ratten;	/* digital attenuator settings */
	float lspl, rspl;	/* desired dbspl levels */
	float max_lspl, max_rspl;	/* max dbspl levels */
	void *isitime;		/* handle for isi_functions() */
	char depstr[256];
	syn_spec ss;		/* full specification for sound */
	float *avePMk;
	int err;
	char msg[1024];
	int rand, spont;
	float daFc, evtFc;
	int ana_every;
	int stim_every;
	double timestamp;
	int retval;
	int outsize, da_channels;
	xword *outbuf;
	char *stim, *range;


	/* calculate parameters */
	ws_lock(ws_table, True);

	dur = GI("dur");
	reps = GI("reps");
	delay = GI("delay");
	epoch = GI("epoch");
	rise = GI("rise");
	fall = GI("fall");
	isi = GI("isi");
	spont = GI("Spont_Stims");
	evtFc = GF("evtFc");
	daFc = GF("daFc");
	ana_every = GI("ana.every");
	stim_every = GI("stim.every");
	rand = GI("randomize");

	abi = GI("bja.abi");
	itd = GI("bja.itd");
	iid = GI("bja.iid");
	stim = GS("bja.stim");
	range = GS("bja.range");

	ws_lock(ws_table, False);
	unlock_most_worksheets();
	/* no svars should be accessed beyond this point */

	bc = 100;

	if ((stimarray =
	     StimArray_Gen(&nstims, rand, range, reps, 0, spont)) == NULL)
		return (-1);

	if (nstims == 0) {
		alert("nstims is zero");
		return (-1);
	}
	fprintf(datafp, "nrasters=%d\n", nstims);

	syn_calibrate = 1;

	is_setNice(GI("nicelevel"));
	is_setSyncPulse(GI("syncPulse"));

	is_init(daFc, 0.0, evtFc, dur + delay + itd / 1000, epoch);

	da_channels = is_getDAnchans();
	outbuf = is_getDAbuffer(&outsize);

	syn_spec_parse(stim, 0, &ss);

	/* check to see if att can handle all iid's */

	for (i = 0, err = 0; (i < (nstims / reps)) && progRunning; i++) {
		if (stimarray[i] == SPONT)
			continue;

		bja_synthesize(stimarray[i], &ss, outbuf, outsize,
			       da_channels, delay, dur, epoch, itd, bc,
			       rise, fall, SYN_BOTH);
		cal_FigureSPL(&ss, 0, 0, &max_lspl, &max_rspl);

		lspl = (float) abi - (0.5 * (float) iid);
		rspl = (float) abi + (0.5 * (float) iid);
		if (lspl < (max_lspl - attMaxAtten))
			err |= CAL_LEFT_TOO_SOFT;
		if (rspl < (max_rspl - attMaxAtten))
			err |= CAL_RIGHT_TOO_SOFT;
		if (lspl > max_lspl)
			err |= CAL_LEFT_TOO_LOUD;
		if (rspl > max_rspl)
			err |= CAL_RIGHT_TOO_LOUD;
	}

	if (err != 0) {
		sprintf(msg, "For some stimuli, requested dB SPL is\n");
		if (err & CAL_LEFT_TOO_LOUD)
			strcat(msg, "too loud in LEFT channel\n");
		if (err & CAL_LEFT_TOO_SOFT)
			strcat(msg, "too soft in LEFT channel\n");
		if (err & CAL_RIGHT_TOO_LOUD)
			strcat(msg, "too loud in RIGHT channel\n");
		if (err & CAL_RIGHT_TOO_SOFT)
			strcat(msg, "too soft in RIGHT channel\n");
		alert(msg);
		return (-1);
	}
	/* All Set: synth and run and run and run ... */

	isitime = isi_start(0);
	for (i = 0; i < nstims && progRunning; i++) {
		set_trial_vars(i, nstims, reps, stimarray[i]);
		perc_done();

		if (stimarray[i] == SPONT) {	/* SPONT */
			setRack(0, attMaxAtten, attMaxAtten);
			is_clearOutput(SYN_BOTH);
		} else {
			bja_synthesize(stimarray[i], &ss, outbuf, outsize,
				       da_channels, delay, dur, epoch, itd,
				       bc, rise, fall, SYN_BOTH);
			lspl = (float) abi - (0.5 * (float) iid);
			rspl = (float) abi + (0.5 * (float) iid);
			if (!cal_SafeFigureAtten
			    (&ss, lspl, rspl, &latten, &ratten)) {
				progRunning = False;
				break;
			}
			setRack(0, latten, ratten);
		}

		if (stimarray[i] == SPONT)
			statline_set(status, "this is: SPONT");
		else
			statline_set(status,
				     "this is: % 3d Hz", stimarray[i]);

		set_led(3, 1);

		retval = is_sample(X_TRIG | X_EVT | X_DA, epoch, &timestamp);
		if (!(single_step(retval))) {
			alert("is_sample failed: can't continue run");
			set_progRunning(False);
		} else
			MaybePause();

		set_led(3, 0);

		view_input();
		view_output();

		data = getRaster(epoch, &avePMk);

		if (stimarray[i] == SPONT)
			sprintf(depstr, "depvar=%d <SPONT>\n", SPONT);
		else
			sprintf(depstr, "depvar=%d\n", stimarray[i]);

		if (avePMk != NULL)
			printAvePMks(avePMk,
				     ((i + 1) % (nstims / reps) ==
				      0) ? ((i + 1) / (nstims / reps)) : 0,
				     nstims / reps);

		write_data_to_xdphysfile(datafp, data, depstr, avePMk,!(i % ana_every),
			!(i % stim_every), SYN_BOTH);

		isi_stop(isitime);
		isitime = isi_start(isi);
		update_displays(data, 0);

		dloop_empty();

		free(data);
	}

	free(stimarray);

	return (progRunning ? 1 : 0);
}


Widget setupBja(DockSlot * slot)
{
	Widget ws, h, w1;

	ws = pop_ws_new(TopLevel, ws_table, "bja", &h, NULL, NULL);
	w1 = make_start_temp(h,
			     make_module_info("bja", "bja", "bja", NULL,
					      runScan, NULL,
					      default_display_fn, ws_table,
					      WS_N(ws_table)), NULL, NULL,
			     slot);

	return (ws);
}
