/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: mod_beats.c,v 1.22 2002/07/15 04:30:10 cmalek Exp $
** Program: dowl
**  Module: mod_beats.c
**  Author: mazer
** Descrip: xdowl application module -- DITD tuning curve
**
** Revision History (most recent last)
**
** Sun Nov 14 18:54:20 1993 mazer
**  creation date :: derived from itd.c
**  dynamic itd ("binaural-beat") module
**
** Aug 96 bjarthur
**  changed name to mod_beats from mod_ditd.  thought it more descriptive.
**
*******************************************************************/

#include "xdphyslib.h"
#include "xdphys.h"

static Field ws_table[] = {
	{"beats.Carrier (Hz)", "%d", 5, "7000"},
	{"beats.Signal (Hz)", "%d", 3, "2"},
	{WS_VSEP},
	{"beats.iid (dB)", "%d", 3, "0"},
	{"beats.abi (dB)", "%d", 3, "40"},
	{NULL},
};

static void beats_synthesize(syn_spec *, float *, int, int, int, int, int,
			     int, int);
static int runScan(FILE *, Widget);
Widget setupBeats(DockSlot *);

static void beats_synthesize(syn_spec * ss, float *fwave, int ms_delay,
			     int ms_dur, int ms_epoch, int us_itd,
			     int ms_rise, int ms_fall, int lr)
{
	static float *tmp = NULL;
	static syn_spec ossl = { SC_INVALID }, ossr = {
	SC_INVALID};

	int tk_epoch, tk_delay, tk_dur, tk_rise, tk_fall;
	int outsize, da_channels;
	xword *outbuf;
	int side;

	assert((lr == SYN_LEFT) || (lr == SYN_RIGHT));

	tk_dur = ms_dur * is_daFc / 1000.0;
	tk_delay = ms_delay * is_daFc / 1000.0;
	tk_epoch = ms_epoch * is_daFc / 1000.0;
	tk_rise = ms_rise * is_daFc / 1000.0;
	tk_fall = ms_fall * is_daFc / 1000.0;

	da_channels = is_getDAnchans();
	outbuf = is_getDAbuffer(&outsize);

	if (tmp == NULL)
		assert((tmp =
			(float *) malloc(da_channels * tk_epoch *
					 sizeof(float))) != NULL);

	if ((!syn_spec_same(ss, &ossl)) && (!syn_spec_same(ss, &ossr))) {
		assert(cur_repnum == 0);
		synthesize(ss, tmp, NULL, outsize, da_channels, ms_delay,
			   ms_dur, ms_epoch, us_itd, 100, ms_rise, ms_fall,
			   lr);
		if (lr == SYN_LEFT) {
			syn_spec_copy(ss, &ossl);
			side = 0;
		} else {
			syn_spec_copy(ss, &ossr);
			side = 1;
		}
	}

	else if (syn_spec_same(ss, &ossl)) {
		syn_spec_copy(&ossl, ss);
		side = 0;
	} else if (syn_spec_same(ss, &ossr)) {
		syn_spec_copy(&ossr, ss);
		side = 1;
	}

	if (lr == SYN_LEFT)
		syn_transfer_to_outbuf(tmp + side, outbuf, tk_epoch,
				       da_channels);
	else
		syn_transfer_to_outbuf(tmp + side, outbuf + 1, tk_epoch,
				       da_channels);
}

static int runScan(FILE * datafp, Widget status)
{
	int i;
	int delay;		/* stimulus latency in ms */
	int dur;		/* stim durationin ms */
	int epoch;		/* recording epoch in ms */
	int isi;		/* inter-stimulus interval */
	int rise, fall;		/* rise and fall times in ms */
	int nreps;		/* number reps to present */
	int *data;		/* pointer to raster data */
	float latten, ratten;	/* digital attenuator settings */
	float lspl, rspl;	/* desired dbspl levels */
	int high, low;		/* freq (hz) */
	int spont;
	void *isitime;		/* handle for isi_functions() */
	char depstr[256];
	float *avePMk;
	syn_spec hss, lss;
	long seed = -1;
	int signal, carrier;
	int spont_stims;
	float daFc, evtFc;
	int ana_every;
	int stim_every;
	double timestamp;
	int retval;


	/* calculate parameters */
	ws_lock(ws_table, True);

	dur = GI("dur");
	nreps = GI("reps");
	delay = GI("delay");
	epoch = GI("epoch");
	rise = GI("rise");
	fall = GI("fall");
	isi = GI("isi");
	spont_stims = GI("Spont_Stims");
	evtFc = GF("evtFc");
	daFc = GF("daFc");
	ana_every = GI("ana.every");
	stim_every = GI("ana.every");

	carrier = GI("beats.Carrier");
	signal = GI("beats.Signal");
	lspl = GF("beats.abi") - (0.5 * GF("beats.iid"));
	rspl = GF("beats.abi") + (0.5 * GF("beats.iid"));

	ws_lock(ws_table, False);
	unlock_most_worksheets();

	/* no svars should be accessed beyond this point */

	high = carrier + 0.5 * signal;
	low = carrier - 0.5 * signal;

	syn_calibrate = 1;

	if (spont_stims) {
		/* spont is carefully chosen here to not be == 0.  if it were,
		   then the stimulus generated would have a non zero initial
		   phase.  this is a hack to mimic rad_vary = 0 */
		spont = 1 + (int) floor((double) (ran1(&seed) * nreps));
		nreps++;
	}
#ifndef __linux__
	if (epoch > 5300) {
		alert("Epoch is limited to 5300ms by the SunOS kernel.");
		return (0);
	}
#endif

	if ((dur % signal) != 0) {
		alert
		    ("Dur (in Prefs) and Signal don't give a whole number of periods.");
		return (0);
	}
	if (signal < 1) {
		alert("Signal must be positive.");
		return (0);
	}
	if (dur < 1000) {
		if (pop_box
		    ("Dur is less than 1000ms.", "Continue?", "Abort",
		     NULL) == 2)
			return (0);
	}
	fd_syn_spec_parse(NULL, high, &hss);
	fd_syn_spec_parse(NULL, low, &lss);

	fprintf(datafp, "nrasters=%d\n", nreps);

	is_setNice(GI("nicelevel"));
	is_setSyncPulse(GI("syncPulse"));

	is_init(daFc, 0.0, evtFc, dur + delay, epoch);

	isitime = isi_start(0);
	for (i = 0; i < nreps && progRunning; i++) {
		set_trial_vars(i, nreps, nreps, i);
		perc_done();

		if (i == spont) {
			statline_set(status, "this is: SPONT");
			setRack(0, attMaxAtten, attMaxAtten);
			is_clearOutput(SYN_BOTH);
		} else {
			if (i % 2) {
				beats_synthesize(&hss, NULL, delay, dur, epoch, 0, rise, fall,
						 SYN_LEFT);
				beats_synthesize(&lss, NULL, delay, dur, epoch, 0, rise, fall,
						 SYN_RIGHT);
				statline_set(status, "this is: L=%dHz,  R=%dHz", high, low);
			} else {
				beats_synthesize(&lss, NULL, delay, dur, epoch, 0, rise, fall,
						 SYN_LEFT);
				beats_synthesize(&hss, NULL, delay, dur, epoch, 0, rise, fall,
						 SYN_RIGHT);
				statline_set(status, "this is: L=%dHz,  R=%dHz", low, high);
			}

			if (!cal_SafeFigureAtten
			    (&hss, lspl, 0.0, &latten, NULL))
				return (-1);
			if (!cal_SafeFigureAtten
			    (&lss, 0.0, rspl, NULL, &ratten))
				return (-1);

			setRack(0, latten, ratten);

		}

		set_led(3, 1);

		retval =
		    is_sample(X_TRIG | X_EVT | X_DA, epoch, &timestamp);
		if (!(single_step(retval))) {
			alert("is_sample failed: can't continue run");
			set_progRunning(False);
		} else
			MaybePause();

		set_led(3, 0);

		view_input();
		view_output();

		data = getRaster(epoch, &avePMk);

		if (i == spont)
			sprintf(depstr, "depvar=%d <SPONT>\n", SPONT);
		else
			sprintf(depstr, "depvar=%d (%c higher)\n", i % 2,
				(i % 2) ? 'L' : 'R');

		if (avePMk != NULL)
			printAvePMks(avePMk, i + 1, 1);

		write_data_to_xdphysfile(datafp, data, depstr, avePMk,!(i % ana_every),
			!(i % stim_every), SYN_BOTH);

		isi_stop(isitime);
		isitime = isi_start(isi);
		update_displays(data, 0);

		dloop_empty();

		free(data);
	}

	return (progRunning ? 1 : 0);
}

Widget setupBeats(DockSlot * slot)
{
	Widget ws, h;

	ws = pop_ws_new(TopLevel, ws_table, "beats", &h, NULL, NULL);
	make_start_temp(h,
			make_module_info("itd", "beats", "beats", NULL,
					 runScan, NULL, default_display_fn,
					 ws_table, WS_N(ws_table)), NULL,
			NULL, slot);
	return (ws);
}
