/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: mod_bam.c,v 1.20 2002/07/15 04:30:10 cmalek Exp $
** Program: dowl
**  Module: mod_bam.c
**  Author: bjarthur
** Descrip: xdowl application module -- binaural amplitude modulation
**
** Revision History (most recent last)
**
** 97.3 bjarthur (creation)
**  copied from mod_beats.c
**
*******************************************************************/

#include "xdphyslib.h"
#include "xdphys.h"

static Field ws_table[] = {
	{"bam.Num_Periods (#)", "%d", 3, "10"},
	{"bam.Depth (dB)", "%d", 2, "40"},
	{WS_VSEP},
	{"bam.Stim", "%s", 40, "0"},
	{"bam.iid (dB)", "%d", 3, "0"},
	{"bam.itd (us)", "%d", 4, "0"},
	{"bam.abi (dB)", "%d", 3, "40"},
	{NULL},
};

static void bam_synthesize(syn_spec *, float *, int, int, int, int, int,
			   int, int, float, float, int);
static int runScan(FILE *, Widget);
Widget setupBam(DockSlot *);

static void bam_synthesize(syn_spec * ss, float *fwave, int ms_delay,
			   int ms_dur, int ms_epoch, int us_itd,
			   int ms_rise, int ms_fall, int lr, float num_per,
			   float depth, int side)
{
	static float *tmp = NULL;
	static syn_spec oss = { SC_INVALID };

	int tk_epoch, tk_delay, tk_dur, tk_rise, tk_fall;
	int i, j, outsize, da_channels;
	float am_freq;
	xword *outbuf;
	int othside;

	assert(lr == SYN_BOTH);

	tk_dur = ms_dur * is_daFc / 1000.0;
	tk_delay = ms_delay * is_daFc / 1000.0;
	tk_epoch = ms_epoch * is_daFc / 1000.0;
	tk_rise = ms_rise * is_daFc / 1000.0;
	tk_fall = ms_fall * is_daFc / 1000.0;

	da_channels = is_getDAnchans();
	outbuf = is_getDAbuffer(&outsize);

	if (tmp == NULL)
		assert((tmp = (float *) malloc(da_channels * tk_epoch *
					 sizeof(float))) != NULL);

	if (!syn_spec_same(ss, &oss)) {
		synthesize(ss, tmp, NULL, outsize, da_channels, ms_delay,
			   ms_dur, ms_epoch, us_itd, 100, ms_rise, ms_fall,
			   lr);
		syn_spec_copy(ss, &oss);

		/* zero out one channel during rise/fall times */
		for (i = 0, j = tk_delay * da_channels; i < tk_rise;
		     i++, j += da_channels)
			tmp[j + 1] = 0.0;
		for (i = 0, j = (tk_delay + tk_dur - tk_fall) * da_channels;
		     i < tk_fall; i++, j += da_channels)
			tmp[j + 1] = 0.0;

		/* AM */
		am_freq = num_per * 2.0 * M_PI / (float) (tk_dur - tk_rise - tk_fall);
		for (i = 0, j = (tk_delay + tk_rise) * da_channels;
		     i < (tk_dur - tk_rise - tk_fall); i++, j += da_channels) {
			tmp[j] *= (float) pow(10.0, (-(depth / 2.0) +
					 (depth / 2.0) * cos(am_freq * (double) i)) / 20.0);
			tmp[j + 1] *= (float) pow(10.0, (-(depth / 2.0) +
					 (depth / 2.0) * cos(am_freq * (double) i + M_PI)) / 20.0);
	}}

	syn_spec_copy(&oss, ss);

	othside = (side == 1) ? 0 : 1;

	syn_transfer_to_outbuf(tmp + side, outbuf, tk_epoch, da_channels);
	syn_transfer_to_outbuf(tmp + othside, outbuf + 1, tk_epoch,
			       da_channels);
}

static int runScan(FILE * datafp, Widget status)
{
	int i;
	int delay;		/* stimulus latency in ms */
	int dur;		/* stim durationin ms */
	int epoch;		/* recording epoch in ms */
	int isi;		/* inter-stimulus interval */
	int rise, fall;		/* rise and fall times in ms */
	int nreps;		/* number reps to present and current rep */
	int *data;		/* pointer to raster data */
	float latten, ratten;	/* digital attenuator settings */
	float lspl, rspl;	/* desired dbspl levels */
	int itd;
	int spont;
	void *isitime;		/* handle for isi_functions() */
	char depstr[256];
	float *avePMk;
	syn_spec ss;
	long seed = -1;
	int num_per, depth;
	char *stimspec;
	int spont_stims;
	float daFc, evtFc;
	int ana_every;
	int stim_every;
	double timestamp;
	int retval;


	/* calculate parameters */
	ws_lock(ws_table, True);

	dur = GI("dur");
	nreps = GI("reps");
	delay = GI("delay");
	epoch = GI("epoch");
	rise = GI("rise");
	fall = GI("fall");
	isi = GI("isi");
	spont_stims = GI("Spont_Stims");
	evtFc = GF("evtFc");
	daFc = GF("daFc");
	ana_every = GI("ana.every");
	stim_every = GI("stim.every");

	num_per = GI("bam.Num_Periods");
	depth = GI("bam.Depth");
	stimspec = GS("bam.Stim");
	lspl = GF("bam.abi") - (0.5 * GF("bam.iid"));
	rspl = GF("bam.abi") + (0.5 * GF("bam.iid"));
	itd = GI("bam.itd");

	ws_lock(ws_table, False);
	unlock_most_worksheets();
	/* no svars should be accessed beyond this point */

	syn_calibrate = 1;

	if (depth < 0.0) {
		alert("Depth must not be negative.");
		return (0);
	}
	if (dur < 1000) {
		if (pop_box ("Dur is less than 1000ms.", "Continue?", "Abort",
		     NULL) == 2)
			return (0);
	}
	if (spont_stims) {
		/* spont is carefully chosen here to not be == 0.  if it were,
		   then the stimulus generated would have a non zero initial
		   phase.  this is a hack to mimic rad_vary = 0 */
		spont = 1 + (int) floor((double) (ran1(&seed) * nreps));
		nreps++;
	}
#ifndef __linux__
	if (epoch > 5300) {
		alert("Epoch is limited to 5300ms by the SunOS kernel.");
		return (0);
	}
#endif

	fd_syn_spec_parse(stimspec, 0, &ss);

	fprintf(datafp, "nrasters=%d\n", nreps);

	idle_set(dloop_empty);
	is_setNice(GI("nicelevel"));
	is_setSyncPulse(GI("syncPulse"));

	is_init(daFc, 0.0, evtFc, dur + delay + itd, epoch);

	for (i = 0; i < nreps && progRunning; i++) {
		set_trial_vars(i, nreps, nreps, i);
		perc_done();

		if (i == spont) {
			statline_set(status, "this is: SPONT");
			setRack(0, attMaxAtten, attMaxAtten);
			is_clearOutput(SYN_BOTH);
		} else {
			bam_synthesize(&ss, NULL, delay, dur, epoch, itd,
				       rise, fall, SYN_BOTH, num_per,
				       depth, i % 2);

			if (!cal_SafeFigureAtten
			    (&ss, lspl, rspl, &latten, &ratten))
				return (-1);

			statline_set(status, "this is: %c starts loud",
				     (i % 2) ? 'R' : 'L');
			setRack(0, latten, ratten);
		}

		set_led(3, 1);

		retval = is_sample(X_TRIG | X_EVT | X_DA, epoch, &timestamp);
		if (!(single_step(retval))) {
			alert("is_sample failed: can't continue run");
			set_progRunning(False);
		} else
			MaybePause();

		set_led(3, 0);

		view_input();
		view_output();

		data = getRaster(epoch, &avePMk);

		if (i == spont)
			sprintf(depstr, "depvar=%d <SPONT>\n", SPONT);
		else
			sprintf(depstr, "depvar=%d (%c starts loud)\n",
				i % 2, (i % 2) ? 'R' : 'L');

		if (avePMk != NULL)
			printAvePMks(avePMk, i + 1, 1);

		write_data_to_xdphysfile(datafp, data, depstr, avePMk,!(i % ana_every), !(i % stim_every), SYN_BOTH);

		update_displays(data, 0);
		isi_stop(isitime);
		isitime = isi_start(isi);

		dloop_empty();

		free(data);
	}

	return (progRunning ? 1 : 0);
}

Widget setupBam(DockSlot * slot)
{
	Widget ws, h;

	ws = pop_ws_new(TopLevel, ws_table, "bam", &h, NULL, NULL);
	make_start_temp(h,
			make_module_info("iid", "bam", "bam", NULL,
					 runScan, NULL, default_display_fn,
					 ws_table, WS_N(ws_table)), NULL,
			NULL, slot);
	return (ws);
}
