/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/


/******************************************************************
**  RCSID: $Id: fd.c,v 1.42 2003/09/10 22:34:02 bja Exp $
** Program: xdphys
**  Module: fd.c
**  Author: mazer, bjarthur
** Descrip: reads xdphys datafiles
**
** Revision History (most recent last)
**
** Sat Feb 22 16:34:53 1992 mazer
**  - cleaned up the plotter code -- changed terminology for plot
**    methods: they're now properly called methods.
**  - closing an overlay plot will not reset the overlay counter
**    if you're on to another type of plot, etc. This was really a
**    change to xstuff:PShellDestroyedCB(), not here.
**
** Sat Jul  4 01:32:29 1992 mazer
**  changed meaning of:
**  --> analysis.begin-data <--
**		now relative to "Delay", so begin=0 means start
**		counting at beginning of stimulus, eg:
**		     begin-counting = Delay + analsysis.Delay
**  --> analysis.end-data <--
**		now relative to "Dur", so end=5 means end
**		counting 5ms after stim end, eg:
**		     end-counting = Delay + Durr + analsysis.Delay
**      negative values do what you would expect.
**
** Sat Nov  7 00:00:07 1992 mazer
**  analysis.begin_spont & analysis.end_spont implemented.
**    these refer to a time-window starting at spont-begin an
**    running for spont-end ms.
**  if analysis.use_delay is set, then spike rates are calc'd as:
**     (driven / driven_time * 1000) - (spont / spont_time * 1000)
**  to yield spikes/sec above spont (spont = 0).
**
** Thu Nov 12 12:40:51 1992 mazer
**  added filedata_read() and friends
**
** Tue Mar  2 22:51:14 1993 mazer
**  changed spikemask usage to match event==mask, instead of
**  the old event&mask != 0
**
** Mon Mar  8 01:11:54 1993 mazer
**  completing work on filedata_read and friends..
**
** Fri Mar 19 12:41:09 1993 mazer
**  readfile_summarize() can be called several times on a single
**  FILEDATA structure in order to analyze several units or models
**  contained in a a single datafile, w/o going back to disk..
**
** Wed Mar 24 22:00:41 1993 mazer
**  this is rather complicated now -- but in general: if a datafile
**  contains mulitple units, then if the spikeMask == ALL, then
**  each unit (with data) in the file will be plotted, if the
**  spikeMask == MU, then all the unit data will be summed -- excpet
**  for the "TTL/99" channel - in both cases, this must be explicitly
**  requested by selecting spikeMask == 99
**  
** Sat Mar 27 21:46:08 1993 mazer
**  added cache for plotfile() stuff -- keeps ten most recently
**  used (MRU) FILEDATA structures in memory -- used modification
**  dates to check for a stale cache..
**
** Thu May  6 13:41:03 1993 mazer
**  took out the plot_trim_stddev since it's not implemented anyway..
**
** Fri May 14 11:20:42 1993 mazer
**  added DumpTable() support
**
** Fri Jun 11 09:47:42 1993 mazer
**  added refcnt field.. to the FILEDATA structure..
**  filedata_read() sets refcnt = 1;
**  filedata_cache() increments the count:
**    -- when originally added to the cache
**    -- whenever returned
**  filedata_free() decrments the count && free's when count==0
**
** Sat Nov 20 18:54:55 1993 mazer
**  modifed register_plotMethod() to somewhat more gracefully
**  handle multiple registrations of a single suffix/method pair
**
** Mon Nov 22 11:56:01 1993 mazer
**  changed register_plotMethod to a private function and defined
**  to accesser fns: getPlotfileMethod() and setPlotfileMethod()
**
** Wed Dec  1 12:56:50 1993 mazer
**  changed filedata_XXX() functions to FD_XXX()
**
** Thu Feb  3 14:47:09 1994 mazer
**  added analysis.dur-relative parameter to switch back and
**  for between absolute time windows and duration locked time
**  windows -- in both case t0 is the start of the stimulus; i.e.
**  this only effect end-data.
**
** Tue Nov  1 16:33:36 1994 mazer
**  added trashed field to FILEDATA structure to allow methods to
**   mark data as contaminated
**  changed FD_cache to FD_check_cache
**
** 11NOV96 bjarthur
**   modified FD_read to do calib files
**   made doPlotfileMethod_new to jive with new ways of doing things,
**     but kept old one around to xdplot could still compile (hope
**     to get rid of it someday)
**
** 96.12 bjarthur
**   cache stuff was never used, so I commented it out.
**   got rid of plotfile stuff, no turning back now
**   modified FILEDATA struct so that raw&sum data are specified
**     in pm_*.c files
**   moved FD_to_SpikeRasterList here and changed it to be modular
**     (ie not rely on knowing SA data structures).  not sure why
**     this wasn't done before
**   moved all FDObj_View_*FN()s here so in essence, this file is
**     sort of like a default pm_*.c file, and the Method() functions
**     serve as a means to specify alternates.
**   changed FD_read so that the 1st line of a file HAD TO BE
**     ";; progname ver $REVISION ...".  this makes things quicker
**
** 97.1 bjarthur
**   modified FD_rasterFN() so as to be tolerant to light data.
**     (added SumData->n to facilitate this)
**   modified FD_rasterFN() to force xaxis to epoch, not autoscale
**
** 97.2 bjarthur
**   moved FD_*() to fd1.c and renamed them FD1_*()
**
** 97.4 bjarthur
**   added FD_periodogram(), FD_inverse_pres_order()
**   moved FD_howmany_units() here, instead of duplicating it
**     in fd1.c, fd2.c, pm_rover.2, etc...
**   generalized FD_calc_params so if NULL is passed in as FILEDATA*,
**     then the current params are used.
** 98.5 bjarthur
**   added FD_linearity_stats
**
** 99.6 bjarthur
**   removed FD_linearity_stats
**
*******************************************************************/

#include "xdphyslib.h"

struct methodHolder {
	char *skey;		/* search key/file suffix */
	int (*method) ();	/* function for this suffix */
	struct methodHolder *next;
};

float **G_y, *G_x;
int G_n;

static void *accessMethod(struct methodHolder **, char *, int (*)());
static FILEDATA *FD_new(void);

static struct methodHolder *FDOvalidviewRoot = NULL;
static struct methodHolder *FDOdoplotRoot = NULL;
static struct methodHolder *FDOplotRoot = NULL;
static struct methodHolder *FDreadRoot = NULL;
static struct methodHolder *FDfreeRoot = NULL;

/*   if (method == NULL), then retrieve the requested method
**   otherwise, associate method with indicated suffix */

static void *accessMethod(struct methodHolder **root, char *suffix,
			  int (*method) ())
{
	struct methodHolder *p;

	if (suffix == NULL)
		return (NULL);

	if (method != NULL) {
		for (p = (*root); p != NULL; p = p->next) {
			if (strcmp(suffix, p->skey) == 0) {
				if (method != (void *) p->method) {
					fprintf(stderr,
						"accessPlotfileMethod: overriding plot method `%s`\n",
						suffix);
					p->method = method;
				} else {
					fprintf(stderr,
						"accessPlotfileMethod: duplicate plot method `%s`\n",
						suffix);
				}
				return ((void *) method);
			}
		}
		p = (struct methodHolder *) calloc(1,
						   sizeof(struct
							  methodHolder));
		p->skey = strsave(suffix);
		p->method = method;
		p->next = (*root);
		(*root) = p;
		if (debugflag)
			fprintf(stderr,
				"accessFDOplotMethod: method-'%s' at 0x%x\n",
				suffix, (unsigned int) method);
	} else {
		for (p = (*root); p != NULL; p = p->next) {
			if (strcmp(suffix, p->skey) == 0) {
				return ((void *) p->method);
			}
		}
	}
	return (NULL);
}

void *getFDOvalidviewMethod(char *suffix)
{
	return (accessMethod(&FDOvalidviewRoot, suffix, NULL));
}

void *setFDOvalidviewMethod(char *suffix, int (*method) ())
{
	return (accessMethod(&FDOvalidviewRoot, suffix, method));
}

int doFDOvalidviewMethod(void *fdo, FILEDATA * fd, int view)
{
	int (*method) ();

	if (fd->modname != NULL) {
		if ((method = getFDOvalidviewMethod(fd->modname)) != NULL) {
			return ((*method) ((FDO *) fdo, view));
		}
	}
	return (-1);
}

void *getFDOdoplotMethod(char *suffix)
{
	return (accessMethod(&FDOdoplotRoot, suffix, NULL));
}

void *setFDOdoplotMethod(char *suffix, int (*method) ())
{
	return (accessMethod(&FDOdoplotRoot, suffix, method));
}

int doFDOdoplotMethod(void *fdo, FILEDATA * fd, int view,
		      int l, FILE * fptr)
{
	int (*method) ();

	if (fd->modname != NULL) {
		if ((method = getFDOdoplotMethod(fd->modname)) != NULL) {
			(*method) ((FDO *) fdo, fd, view, l, fptr);
			return (1);
		}
	}
	return (-1);
}

void *getFDOplotMethod(char *suffix)
{
	return (accessMethod(&FDOplotRoot, suffix, NULL));
}

void *setFDOplotMethod(char *suffix, int (*method) ())
{
	return (accessMethod(&FDOplotRoot, suffix, method));
}

int doFDOplotMethod(void *fdo, FILEDATA * fd, void *index,
		    int l, FILE * fptr)
{
	int (*method) ();

	if (fd->modname != NULL) {
		if ((method = getFDOplotMethod(fd->modname)) != NULL) {
			return ((*method)
				((FDO *) fdo, fd, (FDObj_ViewType *) index,
				 l, fptr));
		}
	}
	return (-1);
}

void *getFDreadMethod(char *suffix)
{
	return (accessMethod(&FDreadRoot, suffix, NULL));
}

void *setFDreadMethod(char *suffix, int (*method) ())
{
	return (accessMethod(&FDreadRoot, suffix, method));
}

int doFDreadMethod(void *fptr, FILEDATA * fd, int ana)
{
	int (*method) ();

	if (fd->modname != NULL) {
		if ((method = getFDreadMethod(fd->modname)) != NULL) {
			return ((*method) ((FILE *) fptr, fd, ana));
		}
	}
	return (0);
}

void *getFDfreeMethod(char *suffix)
{
	return (accessMethod(&FDfreeRoot, suffix, NULL));
}

void *setFDfreeMethod(char *suffix, int (*method) ())
{
	return (accessMethod(&FDfreeRoot, suffix, method));
}

int doFDfreeMethod(FILEDATA * fd)
{
	int (*method) ();

	if (fd->modname != NULL) {
		if ((method = getFDfreeMethod(fd->modname)) != NULL) {
			return ((*method) (fd));
		}
	}
	return (0);
}

static FILEDATA *FD_new(void)
{
	FILEDATA *data = (FILEDATA *) calloc(1, sizeof(FILEDATA));

	data->filename = NULL;
	data->progname = NULL;
	data->modname = NULL;
	data->params = NULL;
	data->comments = NULL;
	data->ver_num = 0.0;

	data->rawdata = NULL;
	data->sumdata = NULL;

	data->trashed = 0;

	return (data);
}

void FD_free(FILEDATA * data)
{

	if (data) {
		doFDfreeMethod(data);
		FREE(data->filename);
		FREE(data->progname);
		FREE(data->modname);
		varsdelete(data->params);
		FREE(data->comments);
		FREE(data->channels);
		free(data);
	}
}

char *skipTo(FILE * fp, char *blockName)
{
	static char linebuf[1024];
	int l;

	l = strlen(blockName);
	while (fgets(linebuf, sizeof(linebuf), fp) != NULL) {
		if (strncmp(linebuf, blockName, l) == 0)
			return (linebuf);
	}
	return (NULL);
}

void FD_calc_params(FILEDATA * fd, int *spont_stims, int *use_delay, float *t_begin, float *t_end,	/* in milliseconds */
		    float *t_spont_begin, float *t_spont_end)
{				/* in milliseconds */
	float begin;		/* driven begins at in ms */
	float end;		/* driven ends at in ms  */
	float spont_begin;	/* spont begins at in ms */
	float spont_end;	/* spont ends at in ms  */
	int epoch;		/* "epoch" in ms */
	int delay;
	int dur, dur_rel;


	(*spont_stims) = (fd == NULL) ? GI("Spont_Stims") :
	    FD_GI(fd, "Spont_Stims");
	epoch = (fd == NULL) ? GI("Epoch") : FD_GI(fd, "Epoch");
	delay = (fd == NULL) ? GI("Delay") : FD_GI(fd, "Delay");
	dur = (fd == NULL) ? GI("Dur") : FD_GI(fd, "Dur");
	dur_rel = lookupParms_int("parms.dur_relative");
	(*use_delay) = lookupParms_int("parms.use_delay");

	begin = lookupParms_float("parms.begin_data");
	end = lookupParms_float("parms.end_data");
	if (dur_rel) {
		begin += delay;
		end += (delay + dur);
	}

	if (begin < 0) {
		begin = 0.0;
		fprintf(stderr,
			"FD_calc_params: begin_data < 0, set to %4.2fms\n",
			begin);
	} else if (begin > ((float) epoch)) {
		begin = (float) epoch;
		fprintf(stderr,
			"FD_calc_params: begin_data < epoch, set to %4.2fms\n",
			begin);
	}

	if (end <= begin) {
		end = (float) epoch;
		fprintf(stderr,
			"FD_calc_params: end_data <= begin_data, set to %4.2fms\n",
			end);
	} else if (end > ((float) epoch)) {
		end = (float) epoch;
		fprintf(stderr,
			"FD_calc_params: end_data > epoch, set to %4.2fms\n",
			end);
	}

	if (!(*spont_stims) || (*use_delay)) {
		spont_begin = lookupParms_float("parms.begin_spont");
		spont_end = lookupParms_float("parms.end_spont");
		if (dur_rel) {
			spont_end += (float) delay;
		}
		if (spont_begin < 0) {
			spont_begin = 0.0;
			fprintf(stderr,
				"FD_calc_params: begin_spont < 0, using %4.2fms\n",
				spont_begin);
		}
		if (spont_end > (float) delay) {
			fprintf(stderr,
				"FD_calc_params: end_spont > Delay, using anyway\n");
		}
		if ((spont_end - spont_begin) < (end - begin) / 2.0)
			fprintf(stderr,
				"FD_calc_params: spont_time is short re. driven_time\n");
	}

	*t_begin = begin;
	*t_end = end;
	*t_spont_begin = spont_begin;
	*t_spont_end = spont_end;
}

FILEDATA *FD_read(char *datafile, int ana)
{
	FILE *fp;
	FILEDATA *data;
	char *p;
	char *comments;
	float fversion;
	char *modname;
	char *progname;
	char tmp[64];
	int fc;
	char *format;
	WAVEINFO *waveinfo;
	xword *wave;

	/* analog files */
	if (!strcmp(datafile + strlen(datafile) - 4, ".ana")) {
		if ((fp = fopen2(findhome(datafile), "rb")) == NULL)
			return (NULL);
		if ((wave = readwave(fp, &waveinfo)) != NULL) {
			free(wave);
			free(waveinfo);
			rewind(fp);
			if ((data = FD_new()) == NULL) {
				perror("FD_read");
				return (NULL);
			}
			data->filename = strsave(datafile);
			data->progname = strsave("???");
			data->modname = strsave("ana");
			data->params = NULL;
			data->comments = NULL;
			data->channels = NULL;
			data->params = varsnew(0, -1);
			data->ver_num = 0;
		} else {
			FREE(waveinfo);
			fclose(fp);
			return (NULL);
		}
	} else {		/* all other files */

		if ((fp = fopen2(findhome(datafile), "r")) == NULL)
			return (NULL);

		if ((p = skipTo(fp, "")) == NULL) {
			fclose(fp);
			return (NULL);
		}

		modname = (char *) calloc(32, sizeof(char));
		progname = (char *) calloc(32, sizeof(char));
		if (sscanf(p, ";; %s %s ver %s", progname, modname, tmp) !=
		    3) {
			free(modname);
			free(progname);
			fclose(fp);
			return (NULL);
		}

		if (strncmp(tmp, "$R", 2) == 0) {
			format = ";; %*s %*s ver $%*s %f$";
		} else {
			format = ";; %*s %*s ver %f";
		}

		if (sscanf(p, format, &fversion) != 1) {
			free(modname);
			free(progname);
			fclose(fp);
			return (NULL);
		}

		if (skipTo(fp, "COMMENTS") == NULL) {
			fclose(fp);
			return (NULL);
		}

		if ((comments = readComments(fp, datafile)) == NULL) {
			fclose(fp);
			return (NULL);
		}

		if (skipTo(fp, "PARAMS") == NULL) {
			fclose(fp);
			return (NULL);
		}

		if ((data = FD_new()) == NULL) {
			perror("FD_read");
			return (NULL);
		}

		data->filename = strsave(datafile);
		data->progname = progname;
		data->modname = modname;
		data->params = varsnew(0, -1);
		data->comments = comments;
		data->ver_num = fversion;

		if (!fp_loadvars(fp, data->params)) {
			fclose(fp);
			FD_free(data);
			return (NULL);
		}

		/* deal with backwards compatability issues for the svars 
		   we just read in */
		if (((data->ver_num < 2.45) || (data->ver_num == 2.51))
		    && !(strncmp(data->progname, "dowl", 4))) {
			fc = FD_GI(data, "adfc");
			FD_SI(data, "adfc", fc * 1000);
			fc = FD_GI(data, "dafc");
			FD_SI(data, "dafc", fc * 1000);
			fc = FD_GI(data, "evtfc");
			FD_SI(data, "evtfc", fc * 1000);
			FD_SI(data, "file.version", 1);
		} else {
			/* If ver_num is not in the range above, and file.version
			 * is not set, this is a 1st generation xdphys file */
			if (FD_GI(data, "file.version") == 0)
				FD_SI(data, "file.version", 2);
		}
	}

	if (!doFDreadMethod(fp, data, ana))
		return (NULL);

	fclose(fp);
	return (data);
}

SA_SpikeRasterList *FD_to_SpikeRasterList(int nrasters,
					  spike_t ** rastlist, int epoch,
					  int code)
{
	int i, j, nspikes;
	float *tmp;
	SA_SpikeRasterList *srl;

	SA_InitSpikeRasterList(&srl, nrasters, epoch);

	for (i = 0; i < srl->nrasters; i++) {
		if ((rastlist[i] == NULL) || (rastlist[i][0] == 0))
			continue;
		tmp =
		    (float *) calloc(N_SPIKES(rastlist[i]), sizeof(float));
		nspikes = 0;
		for (j = 0; j < N_SPIKES(rastlist[i]); j++) {
			if (matchEvent(SPIKE_CHAN(rastlist[i], j), code)) {
				tmp[nspikes] = SPIKE_MSEC(rastlist[i], j);
				nspikes++;
			}
		}
		SA_SetSpikeRasterListElt(srl, i, epoch, tmp, nspikes);
		free(tmp);
	}

	return (srl);
}

static float table(int mpn, float vector_strength)
{
	float tabley[24][44] = {
		{1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 0.894,
		 0.865, 0.833,
		 0.800, 0.764, 0.729, 0.693, 0.656, 0.620, 0.584, 0.547,
		 0.511, 0.475,
		 0.440, 0.405, 0.371, 0.338, 0.305, 0.274, 0.244, 0.215,
		 0.189, 0.165,
		 0.144, 0.126, 0.109, 0.095, 0.082, 0.070, 0.059, 0.049,
		 0.040, 0.033,
		 0.026, 0.020, 0.015, 0.011},	/* n=5 */

		{1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 0.896, 0.867,
		 0.836, 0.801,
		 0.765, 0.727, 0.687, 0.645, 0.603, 0.560, 0.518, 0.477,
		 0.437, 0.399,
		 0.362, 0.327, 0.294, 0.262, 0.233, 0.205, 0.179, 0.155,
		 0.133, 0.113,
		 0.096, 0.080, 0.067, 0.055, 0.045, 0.037, 0.030, 0.024,
		 0.019, 0.014,
		 0.010, 0.008, 0.006, 0.004},	/* n=6 */

		{1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 0.880, 0.845,
		 0.808, 0.768,
		 0.727, 0.684, 0.640, 0.595, 0.550, 0.506, 0.462, 0.419,
		 0.378, 0.339,
		 0.301, 0.266, 0.234, 0.204, 0.177, 0.152, 0.130, 0.110,
		 0.092, 0.076,
		 0.063, 0.051, 0.041, 0.032, 0.025, 0.020, 0.015, 0.012,
		 0.009, 0.006,
		 0.004, 0.003, 0.002, 0.001},	/* n=7 */

		{1.000, 1.000, 1.000, 1.000, 1.000, 0.897, 0.863, 0.824,
		 0.783, 0.739,
		 0.692, 0.645, 0.597, 0.549, 0.501, 0.455, 0.410, 0.367,
		 0.326, 0.287,
		 0.251, 0.218, 0.187, 0.160, 0.135, 0.114, 0.095, 0.078,
		 0.064, 0.051,
		 0.041, 0.032, 0.025, 0.019, 0.014, 0.011, 0.008, 0.006,
		 0.004, 0.003,
		 0.002, 0.001, 0.001, 0.001},	/* n=8 */

		{1.000, 1.000, 1.000, 1.000, 1.000, 0.884, 0.846, 0.803,
		 0.757, 0.709,
		 0.659, 0.609, 0.558, 0.507, 0.457, 0.409, 0.364, 0.321,
		 0.280, 0.243,
		 0.209, 0.178, 0.150, 0.125, 0.104, 0.085, 0.069, 0.055,
		 0.044, 0.034,
		 0.027, 0.020, 0.015, 0.011, 0.008, 0.006, 0.004, 0.003,
		 0.002, 0.001,
		 0.001, 0.001, 0.001, 0.001},	/* n=9 */

		{1.000, 1.000, 1.000, 1.000, 1.000, 0.872, 0.829, 0.783,
		 0.733, 0.681,
		 0.628, 0.574, 0.520, 0.468, 0.417, 0.369, 0.323, 0.280,
		 0.241, 0.205,
		 0.173, 0.145, 0.120, 0.098, 0.079, 0.063, 0.050, 0.039,
		 0.030, 0.023,
		 0.017, 0.013, 0.009, 0.007, 0.005, 0.003, 0.002, 0.001,
		 0.001, 0.001,
		 0.001, 0.001, 0.001, 0.001},	/* n=10 */

		{1.000, 1.000, 1.000, 1.000, 1.000, 0.859, 0.813, 0.763,
		 0.710, 0.654,
		 0.598, 0.541, 0.486, 0.432, 0.380, 0.332, 0.287, 0.245,
		 0.208, 0.174,
		 0.144, 0.118, 0.096, 0.077, 0.061, 0.047, 0.037, 0.028,
		 0.021, 0.015,
		 0.011, 0.008, 0.006, 0.004, 0.003, 0.002, 0.001, 0.001,
		 0.001, 0.001,
		 0.001, 0.001, 0.001, 0.001},	/* n=11 */

		{1.000, 1.000, 1.000, 1.000, 0.891, 0.847, 0.797, 0.744,
		 0.687, 0.628,
		 0.569, 0.511, 0.454, 0.399, 0.347, 0.299, 0.255, 0.214,
		 0.179, 0.147,
		 0.120, 0.096, 0.077, 0.060, 0.046, 0.035, 0.027, 0.020,
		 0.014, 0.010,
		 0.007, 0.005, 0.003, 0.002, 0.001, 0.001, 0.001, 0.001,
		 0.001, 0.001,
		 0.001, 0.001, 0.001, 0.001},	/* n=12 */

		{1.000, 1.000, 1.000, 1.000, 0.882, 0.835, 0.782, 0.725,
		 0.665, 0.603,
		 0.542, 0.482, 0.423, 0.368, 0.316, 0.269, 0.226, 0.188,
		 0.154, 0.125,
		 0.100, 0.079, 0.061, 0.047, 0.035, 0.026, 0.019, 0.014,
		 0.010, 0.007,
		 0.005, 0.003, 0.002, 0.001, 0.001, 0.001, 0.001, 0.001,
		 0.001, 0.001,
		 0.001, 0.001, 0.001, 0.001},	/* n=13 */

		{1.000, 1.000, 1.000, 1.000, 0.873, 0.823, 0.766, 0.706,
		 0.643, 0.580,
		 0.516, 0.454, 0.395, 0.340, 0.289, 0.242, 0.201, 0.164,
		 0.132, 0.105,
		 0.083, 0.064, 0.049, 0.037, 0.027, 0.020, 0.014, 0.010,
		 0.007, 0.005,
		 0.003, 0.002, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001,
		 0.001, 0.001,
		 0.001, 0.001, 0.001, 0.001},	/* n=14 */

		{1.000, 1.000, 1.000, 1.000, 0.865, 0.811, 0.752, 0.688,
		 0.623, 0.557,
		 0.492, 0.429, 0.369, 0.314, 0.263, 0.218, 0.178, 0.143,
		 0.114, 0.089,
		 0.069, 0.052, 0.039, 0.029, 0.021, 0.015, 0.010, 0.007,
		 0.005, 0.003,
		 0.002, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001,
		 0.001, 0.001,
		 0.001, 0.001, 0.001, 0.001},	/* n=15 */

		{1.000, 1.000, 1.000, 1.000, 0.856, 0.799, 0.737, 0.671,
		 0.603, 0.535,
		 0.468, 0.404, 0.345, 0.290, 0.241, 0.196, 0.158, 0.125,
		 0.098, 0.075,
		 0.057, 0.043, 0.031, 0.022, 0.016, 0.011, 0.007, 0.005,
		 0.003, 0.002,
		 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001,
		 0.001, 0.001,
		 0.001, 0.001, 0.001, 0.001},	/* n=16 */

		{1.000, 1.000, 1.000, 1.000, 0.848, 0.788, 0.723, 0.654,
		 0.583, 0.513,
		 0.446, 0.381, 0.322, 0.267, 0.219, 0.177, 0.140, 0.110,
		 0.084, 0.064,
		 0.048, 0.035, 0.025, 0.018, 0.012, 0.008, 0.005, 0.004,
		 0.002, 0.001,
		 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001,
		 0.001, 0.001,
		 0.001, 0.001, 0.001, 0.001},	/* n=17 */

		{1.000, 1.000, 1.000, 0.894, 0.839, 0.777, 0.708, 0.637,
		 0.565, 0.493,
		 0.424, 0.360, 0.300, 0.247, 0.200, 0.159, 0.125, 0.096,
		 0.073, 0.054,
		 0.039, 0.028, 0.020, 0.014, 0.009, 0.006, 0.004, 0.003,
		 0.002, 0.001,
		 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001,
		 0.001, 0.001,
		 0.001, 0.001, 0.001, 0.001},	/* n=18 */

		{1.000, 1.000, 1.000, 0.888, 0.831, 0.765, 0.695, 0.621,
		 0.546, 0.474,
		 0.404, 0.339, 0.280, 0.228, 0.182, 0.143, 0.111, 0.084,
		 0.063, 0.046,
		 0.033, 0.023, 0.016, 0.011, 0.007, 0.005, 0.003, 0.002,
		 0.001, 0.001,
		 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001,
		 0.001, 0.001,
		 0.001, 0.001, 0.001, 0.001},	/* n=19 */

		{1.000, 1.000, 1.000, 0.882, 0.822, 0.754, 0.681, 0.605,
		 0.529, 0.455,
		 0.385, 0.320, 0.262, 0.210, 0.166, 0.129, 0.098, 0.073,
		 0.054, 0.039,
		 0.027, 0.019, 0.013, 0.008, 0.005, 0.003, 0.002, 0.001,
		 0.001, 0.001,
		 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001,
		 0.001, 0.001,
		 0.001, 0.001, 0.001, 0.001},	/* n=20 */

		{1.000, 1.000, 1.000, 0.877, 0.814, 0.744, 0.668, 0.590,
		 0.512, 0.437,
		 0.366, 0.302, 0.244, 0.194, 0.151, 0.116, 0.087, 0.064,
		 0.046, 0.033,
		 0.023, 0.015, 0.010, 0.007, 0.005, 0.003, 0.002, 0.001,
		 0.001, 0.001,
		 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001,
		 0.001, 0.001,
		 0.001, 0.001, 0.001, 0.001},	/* n=21 */

		{1.000, 1.000, 1.000, 0.871, 0.806, 0.733, 0.655, 0.575,
		 0.495, 0.420,
		 0.349, 0.285, 0.228, 0.179, 0.138, 0.104, 0.077, 0.056,
		 0.040, 0.028,
		 0.019, 0.013, 0.008, 0.005, 0.003, 0.002, 0.001, 0.001,
		 0.001, 0.001,
		 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001,
		 0.001, 0.001,
		 0.001, 0.001, 0.001, 0.001},	/* n=22 */

		{1.000, 1.000, 1.000, 0.866, 0.798, 0.722, 0.642, 0.560,
		 0.480, 0.403,
		 0.332, 0.269, 0.213, 0.165, 0.126, 0.094, 0.069, 0.049,
		 0.034, 0.023,
		 0.016, 0.010, 0.006, 0.004, 0.002, 0.001, 0.001, 0.001,
		 0.001, 0.001,
		 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001,
		 0.001, 0.001,
		 0.001, 0.001, 0.001, 0.001},	/* n=23 */

		{1.000, 1.000, 1.000, 0.860, 0.790, 0.712, 0.629, 0.546,
		 0.464, 0.387,
		 0.316, 0.253, 0.199, 0.153, 0.115, 0.085, 0.061, 0.043,
		 0.030, 0.020,
		 0.013, 0.008, 0.005, 0.003, 0.002, 0.001, 0.001, 0.001,
		 0.001, 0.001,
		 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001,
		 0.001, 0.001,
		 0.001, 0.001, 0.001, 0.001},	/* n=24 */

		{1.000, 1.000, 1.000, 0.855, 0.782, 0.702, 0.617, 0.532,
		 0.449, 0.372,
		 0.301, 0.239, 0.186, 0.141, 0.105, 0.076, 0.054, 0.038,
		 0.025, 0.017,
		 0.011, 0.007, 0.004, 0.002, 0.001, 0.001, 0.001, 0.001,
		 0.001, 0.001,
		 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001,
		 0.001, 0.001,
		 0.001, 0.001, 0.001, 0.001},	/* n=25 */

		{1.000, 1.000, 0.899, 0.828, 0.744, 0.653, 0.559, 0.468,
		 0.382, 0.304,
		 0.236, 0.179, 0.132, 0.095, 0.066, 0.045, 0.030, 0.019,
		 0.012, 0.007,
		 0.004, 0.002, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001,
		 0.001, 0.001,
		 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001,
		 0.001, 0.001,
		 0.001, 0.001, 0.001, 0.001},	/* n=30 */

		{1.000, 1.000, 0.867, 0.776, 0.673, 0.565, 0.459, 0.361,
		 0.275, 0.203,
		 0.144, 0.099, 0.066, 0.042, 0.026, 0.016, 0.009, 0.005,
		 0.003, 0.001,
		 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001,
		 0.001, 0.001,
		 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001,
		 0.001, 0.001,
		 0.001, 0.001, 0.001, 0.001},	/* n=40 */

		{1.000, 1.000, 0.837, 0.728, 0.609, 0.489, 0.377, 0.279,
		 0.199, 0.135,
		 0.088, 0.055, 0.033, 0.019, 0.010, 0.005, 0.003, 0.001,
		 0.001, 0.001,
		 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001,
		 0.001, 0.001,
		 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001,
		 0.001, 0.001,
		 0.001, 0.001, 0.001, 0.001},	/* n=50 */
	};

	float tablex[44] = {
		0.02, 0.04, 0.06, 0.08, 0.10, 0.12, 0.14, 0.16, 0.18, 0.20,
		0.22, 0.24, 0.26, 0.28, 0.30, 0.32, 0.34, 0.36, 0.38, 0.40,
		0.42, 0.44, 0.46, 0.48, 0.50, 0.52, 0.54, 0.56, 0.58, 0.60,
		0.62, 0.64, 0.66, 0.68, 0.70, 0.72, 0.74, 0.76, 0.78, 0.80,
		0.82, 0.84, 0.86, 0.88
	};

	int idx;

	if (mpn < 5)
		return (1.0);
	else if (mpn > 17)
		return (1.0);
	else {
		if (mpn < 25)
			idx = mpn - 5;
		else if ((mpn > 24) && (mpn < 28))
			idx = 20;	/* n=25 */
		else if ((mpn > 27) && (mpn < 35))
			idx = 21;	/* n=30 */
		else if ((mpn > 34) && (mpn < 45))
			idx = 22;	/* n=40 */
		else if (mpn > 44)
			idx = 23;	/* n=50 */
		return (lin_interpolate
			(vector_strength, 44, tablex, tabley[idx]));
	}
}

void FD_perhist(int nrasters, spike_t ** rastlist, float *delay, float dur,
		float period, float *phase, int nbins, float **depvars,
		float **means, float **stderrs, float **stddevs, int **n,
		float *mean_phase, float *vector_strength, float *rayleigh)
{
	int i, j;
	double mpx, mpy, mpn;
	float time, phi;
	float **tmp;

	if (nbins != 0) {
		assert((depvars != NULL) && (means != NULL)
		       && (stderrs != NULL) && (stddevs != NULL)
		       && (n != NULL));
		assert(((*depvars) =
			(float *) calloc(nbins, sizeof(float))) != NULL);
		assert(((*means) =
			(float *) calloc(nbins, sizeof(float))) != NULL);
		assert(((*stderrs) =
			(float *) calloc(nbins, sizeof(float))) != NULL);
		assert(((*stddevs) =
			(float *) calloc(nbins, sizeof(float))) != NULL);
		assert(((*n) =
			(int *) calloc(nbins, sizeof(int))) != NULL);
		assert((tmp =
			(float **) calloc(nbins,
					  sizeof(float *))) != NULL);
		for (i = 0; i < nbins; i++)
			assert((tmp[i] =
				(float *) calloc(nrasters,
						 sizeof(float))) != NULL);
	}

	mpx = mpy = mpn = 0.0;
	for (i = 0; i < nrasters; i++) {
		for (j = 0; j < N_SPIKES(rastlist[i]); j++) {
			time = SPIKE_MSEC(rastlist[i], j);
			if (time < delay[i])
				continue;
			if (time > delay[i] + dur)
				continue;
			phi =
			    ((time - delay[i]) / period) * 2.0 * M_PI +
			    phase[i];
			if (nbins != 0)
				tmp[(int)
				    floor(fmod(phi, 2.0 * M_PI) /
					  (2.0 * M_PI) * nbins)][i] += 1.0;
			mpx += cos((float) phi);
			mpy += sin((float) phi);
			mpn += 1.0;
	}}

	if (mpn != 0.0) {
		if (vector_strength)
			(*vector_strength) =
			    (float) (sqrt(mpx * mpx + mpy * mpy) / mpn);
		if (mean_phase) {
			if (mpx != 0.0)
				(*mean_phase) = (float) atan2(mpy, mpx);
			else
				(*mean_phase) =
				    (mpy >
				     0.0) ? (M_PI / 2.0) : (-M_PI / 2);
		}
		if (rayleigh)
			(*rayleigh) = (mpn > 50.0) ?
			    exp(-(*vector_strength) * (*vector_strength) *
				mpn) : table((int) mpn,
					     (*vector_strength));
	} else {
		if (vector_strength)
			(*vector_strength) = 0.0;
		if (mean_phase)
			(*mean_phase) = 0.0;
		if (rayleigh)
			(*rayleigh) = 1.0;
	}

	/* see Batschelet (1981) for general info, or Stephens (1969)
	   for a more detailed explanation as well as this equation in
	   specific (p. 284).  as far as I can tell, Stephens is off
	   by a factor of 2 to be self-consistent and consistent with
	   Batschelet.  note, however, that the Rayleigh test is
	   parametric, and hence is not as general as would be nice.
	   a better alternative might be Rao's spacing test, as
	   described in Batschelet (1981, p. 66).  our data is largely
	   uni-modal, though, so the assumption of a von Mises
	   distribution by the Rayleigh test is probably ok.  */

	if (rayleigh)
		if ((*rayleigh) == 0.0)
			(*rayleigh) = MINFLOAT;
	/* this is so a log scale can be used */

	if (nbins != 0) {
		for (i = 0; i < nbins; i++) {
			(*depvars)[i] =
			    ((float) i / (float) nbins) * 2.0 * M_PI;
			(*means)[i] =
			    mean(tmp[i], nrasters) / (dur / period);
			(*stderrs)[i] =
			    stderror(tmp[i], nrasters) / (dur / period);
			(*stddevs)[i] =
			    stddev(tmp[i], nrasters) / (dur / period);
			(*n)[i] = RND2INT(nrasters * (dur / period));
			free(tmp[i]);
		}
		free(tmp);
	}
}

/*
this is a bit tricky.  normally, pres_order[i] tells you in what
order the ith rasterlist entry was presented.  but sometimes you want
to know which rasterlist entry was the ith one presented.  this
function takes pres_order[] and computes the inverse.  so
inv_pres_order[i] tells you which rasterlist entry was the ith one
presented.
*/
void FD_inverse_pres_order(int *pres_order, int **inv_pres_order,
			   int nrasters)
{
	int i;

	assert(((*inv_pres_order) =
		(int *) calloc(nrasters, sizeof(int))) != NULL);

	for (i = 0; i < nrasters; i++)
		(*inv_pres_order)[pres_order[i]] = i;
}

void FD_howmany_units(int **vector, int nrasters, spike_t ** rastlist)
{
	int *r, i, j, k;
	int extend;

	if ((*vector) == NULL) {
		(*vector) = (int *) calloc(1, sizeof(int));
		(*vector)[0] = 0;
	}

	for (i = 0; i < nrasters; i++) {
		if (rastlist[i]) {
			r = rastlist[i];
			for (j = 0; j < N_SPIKES(r); j++) {
				extend = 1;
				if ((*vector)[0] != 0) {
					for (k = 0;
					     (*vector)[0] == 0
					     || k < (*vector)[0]; k++) {
						if ((*vector)[k + 1] ==
						    SPIKE_CHAN(r, j)) {
							extend = 0;
							break;
						}
					}
				}
				if (extend) {
					(*vector) =
					    (int *) realloc(*vector,
							    sizeof(int) *
							    ((*vector)[0] +
							     2));
					(*vector)[0] += 1;
					(*vector)[(*vector)[0]] =
					    SPIKE_CHAN(r, j);
				}
		}}
	}
	qsort((*vector) + 1, (*vector)[0], sizeof(int), intcompare);
}

/* this is only used by _cf.c */
int FD_compare_intensities(float *one, int num1, float *two, int num2,
			   float crit)
{
	float pval;
	int i;
	int *first, *second;

	assert((first = (int *) calloc(num1, sizeof(int))) != NULL);
	assert((second = (int *) calloc(num2, sizeof(int))) != NULL);
	for (i = 0; i < num1; i++)
		first[i] = (int) one[i];
	for (i = 0; i < num2; i++)
		second[i] = (int) two[i];

	pval = permutation_test_dp(first, num1, second, num2, -1);

	free(first);
	free(second);

	if (pval < crit)
		return (1);
	return (0);
}

void FD_plotter_copy(float *x, float *y, float *z, int n, int spont,
		     float **px, float **py, float **pz, int *pn,
		     float **sx, float **sy, float **sz, int *sn)
{
	if (x != NULL) {
		assert(((*px) =
			(float *) calloc(n - spont,
					 sizeof(float))) != NULL);
		memcpy((*px), x + spont, (n - spont) * sizeof(float));
	} else
		(*px) = NULL;

	assert(((*py) =
		(float *) calloc(n - spont, sizeof(float))) != NULL);
	memcpy((*py), y + spont, (n - spont) * sizeof(float));

	if (z != NULL) {
		assert(((*pz) =
			(float *) calloc(n - spont,
					 sizeof(float))) != NULL);
		memcpy((*pz), z + spont, (n - spont) * sizeof(float));
	} else
		(*pz) = NULL;

	(*pn) = n - spont;

	if (spont) {
		assert(((*sx) =
			(float *) calloc(2, sizeof(float))) != NULL);
		assert(((*sy) =
			(float *) calloc(2, sizeof(float))) != NULL);
		(*sx)[0] = (*px)[0];
		(*sx)[1] = (*px)[(*pn) - 1];
		(*sy)[0] = (*sy)[1] = y[0];
		if (z != NULL) {
			assert(((*sz) =
				(float *) calloc(2,
						 sizeof(float))) != NULL);
			(*sz)[0] = (*sz)[1] = z[0];
		} else
			(*sz) = NULL;
		(*sn) = 2;
	}
}

void FD_combine_data(float *x1, float *y1, int n1,
		     float *x2, float *y2, int n2,
		     float **xr, float **yr, int *nr)
{
	int i, j;

	assert((n1 != 0) || (n2 != 0));

	(*nr) = n1 + n2;
	assert(((*xr) = (float *) calloc((*nr), sizeof(float))) != NULL);
	assert(((*yr) = (float *) calloc((*nr), sizeof(float))) != NULL);

	for (i = j = 0; i < n1; i++, j++)
		(*xr)[j] = x1[i];
	for (i = 0; i < n2; i++, j++)
		(*xr)[j] = x2[i];

	qsort(*xr, *nr, sizeof(float), floatcompare);

	/* delete duplicates */
	for (i = 0; i < ((*nr) - 1); i++) {
		if ((*xr)[i] == (*xr)[i + 1]) {
			for (j = i + 1; j < ((*nr) - 1); j++)
				(*xr)[j] = (*xr)[j + 1];
			(*nr)--;
			i--;
		}
	}

	for (i = 0; i < (*nr); i++) {
		(*yr)[i] = 0.0;
		if (n1 != 0)
			(*yr)[i] += lin_interpolate((*xr)[i], n1, x1, y1);
		if (n2 != 0)
			(*yr)[i] += lin_interpolate((*xr)[i], n2, x2, y2);
	}
}

void FD_peak_stats(int nfds, int *ndata, float **x, float **y,
		   float *spont, char **file, int to_tty, int itd)
{
	int i, count, size;
	float max, min, left, right, best, sps;
	char *buf, buf2[128], *tmp, *fmt;

	size = 1024;
	assert((buf = (char *) calloc(size, sizeof(char))) != NULL);

	tmp = (!itd) ?
	    "file             best    max     min     width   center  left    right\n"
	    :
	    "file             best    max     min     width   center  left    right   sps  mod\n";
	strcpy(buf, tmp);
	count = strlen(tmp);
	tmp = (!itd) ?
	    "----             ----    ---     ---     -----   ------  ----    -----\n"
	    :
	    "----             ----    ---     ---     -----   ------  ----    -----   ---  ---\n";
	strcat(buf, tmp);
	count += strlen(tmp);

	for (i = 0; i < nfds; i++) {
		if (!findPeak
		    (ndata[i], x[i], y[i], spont[i], &best, &max, &min,
		     &left, &right, itd ? &sps : NULL))
			fprintf(stderr,
				"WARNING: findPeak() return 0 on %s.\n",
				strrchr(file[i],
					'/') ==
				NULL ? file[i] : 1 +
				(char *) strrchr(file[i], '/'));
		if ((left + right) > 1.0e6)
			fmt = (!itd) ?
			    "%-13s  %6.1e  %6.1e  %6.1e  %6.1e  %6.1e  %6.1e  %6.1e\n"
			    :
			    "%-13s  %6.1e  %6.1e  %6.1e  %6.1e  %6.1e  %6.1e  %6.1e  %5.3f\n";
		else
			fmt = (!itd) ?
			    "%-13s  %6.1f  %6.1f  %6.1f  %6.1f  %6.1f  %6.1f  %6.1f\n"
			    :
			    "%-13s  %6.1f  %6.1f  %6.1f  %6.1f  %6.1f  %6.1f  %6.1f  %5.3f  %5.3f\n";
#ifdef __linux__
		count += sprintf(buf2, fmt,
				 strrchr(file[i],
					 '/') ==
				 NULL ? file[i] : 1 +
				 (char *) strrchr(file[i], '/'), best, max,
				 min, (right - left), (left + right) / 2.0,
				 left, right, sps, (max - min) / max);
#else
		count += strlen(sprintf(buf2, fmt,
					strrchr(file[i],
						'/') ==
					NULL ? file[i] : 1 +
					(char *) strrchr(file[i], '/'),
					best, max, min, (right - left),
					(left + right) / 2.0, left, right,
					sps, (max - min) / max));
#endif
		if (to_tty)
			printf("%s", buf2);
		strcat(buf, buf2);
		if ((size - count) < 512) {
			assert((buf =
				(char *) realloc(buf,
						 (size +
						  1024) * sizeof(char))) !=
			       NULL);
			size += 1024;
		}
	}

	if (!to_tty)
		pop_text("peak_stats", buf, strlen(buf) + 1, False);
}

void funcs1(double x, double *afunc, int ma)
{
	int i;

	for (i = 0; i < ma; i++)
		afunc[i + 1] =
		    (double) lin_interpolate((float) x, G_n, G_x, G_y[i]);
}

void funcs2(double x, double a[], double *yfit, double dyda[], int ma)
{
	int i;
	double tmp1, tmp2, tmp3 = log(10.0);

	(*yfit) = 0.0;
	for (i = 0; i < ma; i++) {
		tmp1 =
		    (double) lin_interpolate((float) x, G_n, G_x, G_y[i]);
		tmp2 = pow(10.0, a[i + 1]) * tmp1;
		(*yfit) += tmp2;
		dyda[i + 1] = tmp2 * tmp3;
	}
}

void FD_perhist_compute_period_nbins(FILEDATA * fd, float *period,
				     int *nbins)
{
	syn_spec ss;
	int fc;
	char tmp[128];

	if (!strcmp(fd->modname, "freq") || !strcmp(fd->modname, "bf")) {
		(*period) = 0.0;
		(*nbins) = PERHIST_DEF_NBINS;
		return;
	}

	sprintf(tmp, "%s.stim", fd->modname);
	fd_syn_spec_parse(FD_GV(fd, tmp), 0, &ss);

	fc = (int) FD_GF(fd, "evtFc");
	if (((fd->ver_num < 2.45) || (fd->ver_num == 2.51))
	    && !(strncmp(fd->progname, "dowl", 4)))
		fc *= 1000;

	if (ss.class == SC_TONE) {
		(*period) = 1000.0 / ss.parms.tone.freq;
		(*nbins) =
		    fc / greatest_common_factor(fc, ss.parms.tone.freq);
	} else {
		if ((ss.class == SC_STACK)
		    && (ss.parms.stack.num_freqs == 1)) {
			(*period) = 1000.0 / ss.parms.stack.freqs[0];
			(*nbins) =
			    fc / greatest_common_factor(fc,
							ss.parms.stack.
							freqs[0]);
		}
	}
}

void FD_perhist1FN(void *fdobjptr, FILEDATA * fd, int spont_stims,
		   int nrasters, int *pres_order, spike_t ** rastlist,
		   int *depints, float **depvals, float **means, int *n,
		   float *mean_phase, float *vector_strength,
		   float *rayleigh, float period, int nbins, int type)
{
	float *phase, *delay;
	int i, j, rad_vary, nreps;
	int tmp;

	rad_vary = FD_GI(fd, "tone.rad_vary");
	if (rad_vary)
		nreps = FD_GI(fd, "Reps");

	assert((phase = (float *) calloc(nrasters, sizeof(float))) !=
	       NULL);
	assert((delay = (float *) calloc(nrasters, sizeof(float))) !=
	       NULL);
	assert(((*depvals) = (float *) calloc(nrasters, sizeof(float))) !=
	       NULL);
	assert(((*means) = (float *) calloc(nrasters, sizeof(float))) !=
	       NULL);

	i = (*n) = 0;
	if ((period == 0.0) && (spont_stims)) {
		while ((i < nrasters) && (SPONT == depints[i]))
			i++;
	}

	if ((period != 0.0) && (!spont_stims))
		(*n)++;

	while (i < nrasters) {
		tmp = depints[i];
		j = i;

		while ((j < nrasters) && (tmp == depints[j])) {
			phase[j] = rad_vary ? (float) (getRadWithArgs
						       (pres_order[j] /
							(nrasters / nreps),
							nreps)) : 0.0;
			delay[j] = (float) FD_GI(fd, "Delay");
			j++;
		}

		FD_perhist((j - i), rastlist + i, delay + i,
			   (float) FD_GI(fd, "Dur"),
			   (period ==
			    0.0) ? 1000.0 / (float) depints[i] : period,
			   phase + i, 0, NULL, NULL, NULL, NULL, NULL,
			   mean_phase, vector_strength, rayleigh);

		switch (type) {
		case (PERHIST_MP):
			(*means)[(*n)] = (*mean_phase);
			break;
		case (PERHIST_VS):
			(*means)[(*n)] = (*vector_strength);
			break;
		case (PERHIST_RT):
			(*means)[(*n)] = (*rayleigh);
			break;
		}
		(*depvals)[(*n)] = depints[i];
		(*n)++;
		i = j;
	}

	if (period != 0.0) {
		if (!spont_stims) {
			for (i = 0; i < nrasters; i++)
				delay[i] = 0.0;
			FD_perhist(nrasters, rastlist,
				   delay, (float) FD_GI(fd, "Delay"),
				   period, phase, 0, NULL, NULL, NULL,
				   NULL, NULL, mean_phase, vector_strength,
				   rayleigh);
			switch (type) {
			case (PERHIST_MP):
				(*means)[0] = (*mean_phase);
				break;
			case (PERHIST_VS):
				(*means)[0] = (*vector_strength);
				break;
			case (PERHIST_RT):
				(*means)[0] = (*rayleigh);
				break;
			}
		}
	}

	if (type == PERHIST_MP) {
		if (period == 0.0)
			unwrap((*means), (*n), M_PI);
		else
			unwrap((*means) + 1, (*n) - 1, M_PI);
		for (i = 0; i < (*n); i++)
			(*means)[i] /= (2.0 * M_PI);
	}

	free(phase);
	free(delay);
}

static int *get_stimarray(FILEDATA *, int *);
static void next(Widget, XtPointer, XtPointer);
static void prev(Widget, XtPointer, XtPointer);
static void jump(Widget, XtPointer, XtPointer);
static void all(Widget, XtPointer, XtPointer);

static int G_all;

void FD_vsm_menu(void *fdobjptr, int all_flag)
{
	FDO *fdo = (FDO *) fdobjptr;

	if (fdo->no_X)
		return;

	G_all = all_flag;

	menubutton_clear(fdo->vsm, &(fdo->vsmpsh));
	XtSetSensitive(fdo->vsm, False);

	menubutton_add(fdo->vsmpsh, "next", next, fdo);
	menubutton_add(fdo->vsmpsh, "prev", prev, fdo);
	if (G_all)
		menubutton_add(fdo->vsmpsh, "all", all, fdo);
	menubutton_add(fdo->vsmpsh, "jump", jump, fdo);
	XtSetSensitive(fdo->vsm, True);
}

static int *get_stimarray(FILEDATA * fd, int *nstims)
{
	int start, stop, step;
	char tmp[128], *tmp2;
	int *stimarray;

	sprintf(tmp, "%s.range", fd->modname);
	if ((tmp2 = FD_GV(fd, tmp)) != NULL) {
		stimarray =
		    StimArray_Gen(nstims, 0, FD_GV(fd, tmp), 1, 1, 0);
	} else {
		sprintf(tmp, "%s.start", fd->modname);
		start = FD_GI(fd, tmp);
		sprintf(tmp, "%s.stop", fd->modname);
		stop = FD_GI(fd, tmp);
		sprintf(tmp, "%s.step", fd->modname);
		step = FD_GI(fd, tmp);
		sprintf(tmp, "%d:%d:%d", start, stop, step);
		stimarray = StimArray_Gen(nstims, 0, tmp, 1, 1, 0);
	}

	return (stimarray);
}

static void next(Widget w, XtPointer fdobjptr, XtPointer call_data)
{
	FDO *fdo = (FDO *) fdobjptr;
	int nstims, *stimarray, *result;

	stimarray = get_stimarray(fdo->fds[fdo->nfds - 1], &nstims);

	if (fdo->vsm_data == VSM_ALL_CODE)
		fdo->vsm_data = (*stimarray);
	else {
		result =
		    (int *) bsearch(&(fdo->vsm_data), stimarray, nstims,
				    sizeof(int), intcompare);
		if (result == NULL) {
			fdo->vsm_data =
			    G_all ? VSM_ALL_CODE : (*stimarray);
		} else {
			result++;
			if (result == (stimarray + nstims)) {
				fdo->vsm_data =
				    G_all ? VSM_ALL_CODE : (*stimarray);
			} else
				fdo->vsm_data = (*result);
		}
	}

	FDObj_Update(fdo, NULL);
	free(stimarray);
}

static void prev(Widget w, XtPointer fdobjptr, XtPointer call_data)
{
	FDO *fdo = (FDO *) fdobjptr;
	int nstims, *stimarray, *result;

	stimarray = get_stimarray(fdo->fds[fdo->nfds - 1], &nstims);

	if (fdo->vsm_data == VSM_ALL_CODE)
		fdo->vsm_data = (*(stimarray + nstims - 1));
	else {
		result =
		    (int *) bsearch(&(fdo->vsm_data), stimarray, nstims,
				    sizeof(int), intcompare);
		if (result == NULL) {
			fdo->vsm_data =
			    G_all ? VSM_ALL_CODE : (*stimarray);
		} else {
			result--;
			if (result < stimarray) {
				fdo->vsm_data =
				    G_all ? VSM_ALL_CODE
				    : (*(stimarray + nstims - 1));
			} else
				fdo->vsm_data = (*result);
		}
	}

	FDObj_Update(fdo, NULL);
	free(stimarray);
}

static void jump(Widget w, XtPointer fdobjptr, XtPointer call_data)
{
	FDO *fdo = (FDO *) fdobjptr;
	char *tmp;

	if ((tmp = pop_dialog("jump to?", "")) == NULL)
		return;
	if (!strcmp(tmp, "all")) {
		if (G_all)
			fdo->vsm_data = VSM_ALL_CODE;
		else {
			alert("'all' doesn't make sense for .bf files.");
			return;
		}
	} else
		sscanf(tmp, "%d", &(fdo->vsm_data));

	FDObj_Update(fdo, NULL);
}

static void all(Widget w, XtPointer fdobjptr, XtPointer call_data)
{
	FDO *fdo = (FDO *) fdobjptr;

	fdo->vsm_data = VSM_ALL_CODE;

	FDObj_Update(fdo, NULL);
}

void FD_perhist2FN(void *fdobjptr, FILEDATA * fd,
		   int nrasters, int *pres_order, spike_t ** rastlist,
		   int *depints, float **depvals, float **means,
		   float **stderrs, float **stddevs, int **n,
		   float *mean_phase, float *vector_strength,
		   float *rayleigh, float period, int nbins)
{
/* fdo->no_errs=1 here means it's a true histogram, ie spikes are simply summed
   and no division by # trials or # cycles occurrs.  fdo->no_errs=0 here means
   that a division by # cycles occurrs.  note that this is different from
   the old perhist, which divides by # trials */

	FDO *fdo = (FDO *) fdobjptr;
	float *delay, *phase;
	int i, rad_vary, nreps;
	int spont_stims;
	int from, to, depint;
	int nstims, *stimarray;

	rad_vary = FD_GI(fd, "tone.rad_vary");
	if (rad_vary)
		nreps = FD_GI(fd, "Reps");
	spont_stims = FD_GI(fd, "Spont_Stims");

	assert((phase =
		(float *) calloc(nrasters, sizeof(float))) != NULL);
	assert((delay =
		(float *) calloc(nrasters, sizeof(float))) != NULL);

	depint = fdo->vsm_data;

	for (i = 0; i < nrasters; i++) {
		phase[i] = rad_vary ?
		    (float) (getRadWithArgs(pres_order[i] /
					    (nrasters / nreps),
					    nreps)) : 0.0;
		delay[i] = (float) FD_GI(fd, "Delay");
	}

	/* check to make sure that vsm_data is in range, if not reset to default */
	from = 0;
	while ((from < nrasters) && (depints[from] != depint))
		from++;

	if (from == nrasters) {
		if (strcmp(fdo->fds[fdo->nfds - 1]->modname, "bf") &&
		    strcmp(fdo->fds[fdo->nfds - 1]->modname, "freq"))
			fdo->vsm_data = depint = VSM_ALL_CODE;
		else {
			stimarray =
			    get_stimarray(fdo->fds[fdo->nfds - 1],
					  &nstims);
			fdo->vsm_data = depint = (*stimarray);
			free(stimarray);
		}
	}

	/* find subset of rasters which are for depint = vsm_data */
	if (depint == VSM_ALL_CODE) {
		from = 0;
		to = nrasters;
		if (spont_stims)
			while (depints[from] == SPONT)
				from++;
	} else {
		from = 0;
		while ((from < nrasters) && (depints[from] != depint))
			from++;
		assert(from != nrasters);
		to = from;
		while ((to < nrasters) && (depints[to] == depint))
			to++;
	}

	FD_perhist((to - from), rastlist + from,
		   delay + from, (float) FD_GI(fd, "Dur"),
		   (period == 0.0) ? 1000.0 / (float) depint : period,
		   phase + from, nbins, depvals, means, stderrs, stddevs,
		   n, mean_phase, vector_strength, rayleigh);

	for (i = 0; i < nbins; i++)
		(*depvals)[i] /= (2.0 * M_PI);

	if (fdo->no_errs)
		for (i = 0; i < nbins; i++)
			(*means)[i] *= ((to - from) * FD_GI(fd, "Dur") /
					((period ==
					  0.0) ? 1000.0 /
					 (float) depint : period));

	(*mean_phase) /= (2.0 * M_PI);

	free(phase);
	free(delay);
}

spike_t **FD_get_rastlist(FILEDATA * fd)
{
	if (!strcmp(fd->modname, "abi") || !strcmp(fd->modname, "bf") ||
	    !strcmp(fd->modname, "freq") || !strcmp(fd->modname, "bc") ||
	    !strcmp(fd->modname, "bam") || !strcmp(fd->modname, "beats") ||
	    !strcmp(fd->modname, "cf") || !strcmp(fd->modname, "fiid") ||
	    !strcmp(fd->modname, "iid") || !strcmp(fd->modname, "itd") ||
	    !strcmp(fd->modname, "rover") || !strcmp(fd->modname, "sps") ||
	    !strcmp(fd->modname, "nop") || !strcmp(fd->modname, "gen") ||
	    !strcmp(fd->modname, "bja"))
		return (FD1_RAWDATA(fd)->rastlist);
	else
		return (NULL);
}

int FD_get_nrasters(FILEDATA * fd)
{
	if (!strcmp(fd->modname, "abi") || !strcmp(fd->modname, "bf") ||
	    !strcmp(fd->modname, "freq") || !strcmp(fd->modname, "bc") ||
	    !strcmp(fd->modname, "bam") || !strcmp(fd->modname, "beats") ||
	    !strcmp(fd->modname, "cf") || !strcmp(fd->modname, "fiid") ||
	    !strcmp(fd->modname, "iid") || !strcmp(fd->modname, "itd") ||
	    !strcmp(fd->modname, "rover") || !strcmp(fd->modname, "sps") ||
	    !strcmp(fd->modname, "nop") || !strcmp(fd->modname, "gen") ||
	    !strcmp(fd->modname, "bja"))
		return (FD1_RAWDATA(fd)->nrasters);
	else
		return (0);
}

int *FD_get_pres_order(FILEDATA * fd)
{
	if (!strcmp(fd->modname, "abi") || !strcmp(fd->modname, "bf") ||
	    !strcmp(fd->modname, "freq") || !strcmp(fd->modname, "bc") ||
	    !strcmp(fd->modname, "bam") || !strcmp(fd->modname, "beats") ||
	    !strcmp(fd->modname, "cf") || !strcmp(fd->modname, "fiid") ||
	    !strcmp(fd->modname, "iid") || !strcmp(fd->modname, "itd") ||
	    !strcmp(fd->modname, "rover") || !strcmp(fd->modname, "sps") ||
	    !strcmp(fd->modname, "nop") || !strcmp(fd->modname, "gen") ||
	    !strcmp(fd->modname, "bja"))
		return (FD1_RAWDATA(fd)->pres_order);
	else
		return (NULL);
}
