/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: dsp.c,v 2.53 2002/07/15 04:30:03 cmalek Exp $
** Program: dowl
**  Module: dsp.c
**  Author: mazer
** Descrip: dsp algorithms -- mostly four1() and twofft() wrappers
**
** Revision History (most recent last)
**
** Tue Jun 22 18:51:47 1993 mazer
**  - changed FFTplot() to use realft()
**  - deleted FFTnoise()
**  - replaced FFTfilter() with one that uses realft() instead of
**    four1() .. and it's now called dsp_bandpass()
**  - deleted freq2bin() and bin2freq()
**  - deleted real_rms()
**  - added normalize argument to CORplot()
**  - chaned sono() to use realft() instead of fft_net()
**  - read_table() gone...
**
** Thu Jun 24 10:16:30 1993 mazer
**  function name changes:
**     FFTfilter -> dsp_bandpass
**     FFTplot -> dsp_spectrumPlot
**     sono -> dsp_sonoPlot
**     fitSinLI -> dsp_fitsin
**  
** Mon Sep  6 20:08:02 1993 mazer
**  dsp_spectrumPlot() --> removed cutoff and mag args, use
**  dsp_phasePlot() for phase now..
**  
** Wed Sep  8 23:31:17 1993 mazer
**  added skip parameter to dsp_fitsin() and added new function
**  dsp_calcrms();
**
** Tue Jan 11 21:29:37 1994 mazer
**  reaped the busted sonogram code..
**
** Fri Jan 21 01:16:26 1994 mazer
**  added working sonogram code:
**    dsp_compute_sonogram() -- generates a sonogram
**    dsp_pgm_sonogram() -- writes a pgm file
**
** Mon Jan 24 23:36:06 1994 mazer
**  added dsp_gaussianfilter() -- this applies a gaussian
**  filter of (mean,sigma) in the frequency domain
**
** Thu Feb  3 01:22:45 1994 mazer
**  added a number on dsp_XXkernel() functions:
**	    bp     (rectangular)
**	   softbp  (rect+cos edges)
**	   gauss   (gaussian)
**  and added nbitsIs() -- an integer wrapper for pow(2.0, N)
**
** Wed Jun 22 11:07:30 1994 mazer
**  changed all functions to be of the form:
**     f(buf, nsamps, skip) instead of having some like that and
**  some that took a (start,stop) pair:
**    dsp_spectrumPlot
**    dsp_psdPlot
**    dsp_phasePlot
**    dsp_calc
**    dsp_fit
**
** 98.11 bjarthur
**  deleted all the stuff we don't use
**
*******************************************************************/

#include "xdphyslib.h"

#define MAG(r, i) 	sqrt((r) * (r) + (i) * (i))
#define PHI(r, i) 	atan2((double)(i), (double) (r))

double dsp_calcrms(xword * xbuf, int nsamps, int skip, int stripDC)
{
	double sum, npts, dc;
	int i;

	if (stripDC) {
		for (dc = 0.0, i = 0; i < nsamps; i++) {
			dc += xbuf[i * skip];
		}
		dc /= (double) i;
	} else {
		dc = 0.0;
	}

	for (npts = sum = 0.0, i = 0; i < nsamps; i++) {
		sum +=
		    ((double) xbuf[i * skip] -
		     dc) * ((double) xbuf[i * skip] - dc);
		npts += 1.0;
	}
	return (sqrt(sum / npts));
}


float dsp_fitsin(xword * xbuf, int nsamps, int skip, int fc, float f,
		 float *amp, float *phi)
{
	int i, k, idx, navg, period;
	float v, real, imag;
	int end;

	if (f < 0.0) {
		float lv;
		int zeros;

		zeros = 0;
		navg = 1;
		lv = 0;
		for (i = 0; i < nsamps; i++) {
			idx = i * skip;
			if (navg) {
				for (v = 0.0, k = -skip * navg;
				     k <= skip * navg; k += skip)
					v += xbuf[idx + k];
				v /= (skip * navg) + 1;
			} else {
				v = xbuf[idx];
				if (i != 0) {
					if (lv <= 0 && v > 0)
						zeros++;
				}
				lv = v;
			}
			f = ((float) zeros) / (((float) (nsamps)) / fc);
		}
	}
	/* compute an integral number of periods less than the number of
	 * samples :
	 */
	period = (int) floor((float) fc / f);
	end = (int) floor((double) nsamps / (double) period) * period;

	real = imag = 0.0;
	for (i = 0; i < end; i++) {
		idx = i * skip;
		real += sin(f * i * 2.0 * M_PI / (float) fc) * xbuf[idx];
		imag += cos(f * i * 2.0 * M_PI / (float) fc) * xbuf[idx];
	}
	real /= (float) i;
	imag /= (float) i;

	if (amp != NULL)
		*amp = 2.0 * MAG(real, imag);	/* the 2.0 is b/c <sin^2(t)> = 0.5 */
	if (phi != NULL)
		*phi = PHI(real, imag);	/* no need for 2.0 here b/c PHI is ratio */
	return (f);
}

void dsp_stripDC(xword * xbuf, int nsamps, int skip)
{
	register double s, t;
	register int i;

	for (s = 0.0, i = 0; i < nsamps; i++) {
		s += xbuf[i * skip];
	}
	s /= (double) nsamps;
	for (i = 0; i < nsamps; i++) {
		t = xbuf[i * skip] - s;
		xbuf[i * skip] = (xword) (t > 0 ? t + 0.5 : t - 0.5);
	}
}


/* --------------------------------------------------------------
 * dsp_highpass_xword
 * 
 * xbuf is the interlaced ad buffer from tdtproc
 * returns filtered waveform in xbuf
   -------------------------------------------------------------- */
void dsp_highpass_xword(xword * xbuf, int nsamps, int fc, int nchans,
			float low)
{
	float *fbuf, *retval;
	int i, j;

	assert((fbuf = (float *) calloc(nsamps, sizeof(float))) != NULL);
	for (i = 0, j = 0; j < nsamps; j++, i += nchans)
		fbuf[j] = (float) xbuf[i];
	retval = dsp_highpass(&fbuf, nsamps, fc, low);
	for (i = 0, j = 0; j < nsamps; j++, i += nchans)
		xbuf[i] = (xword) retval[j];
	free(fbuf);
	free(retval);
}


/* --------------------------------------------------------------
 * dsp_highpass
 * 
 * returns filtered waveform (doesn't affect passed in buffer)
   -------------------------------------------------------------- */
float *dsp_highpass(float **fbuf, int nsamps, int fc, float low)
/*
**   float *fbuf;		pointer to the data buffer
**   int nsamps;		number of samples in fbuf
**   int fc;		    sampling frequency for waveform in fbuf
**   float low;			low end cutoff for filter in Hz
**   int window;		apply cosine shaped window to waveform?
*/
{
	int i, filter_length;
	float f, delta;		/* freq of current bin */
	float *workbuf;

	/* find next largest power of two */
	next2(nsamps, filter_length);
	assert((workbuf =
		(float *) calloc(filter_length, sizeof(float))) != NULL);
	memcpy(workbuf, *fbuf, nsamps * sizeof(float));

	/* zero out the tail of the signal */
	i = nsamps;
	while (i < filter_length)
		workbuf[i++] = 0.0;

	/* convert to the frequency domain */
	realft(workbuf - 1, filter_length, 1);

	workbuf[0] = 0.0;	/* real part of first (dc) component */
	workbuf[1] = 0.0;	/* imaginary part of first (dc) component */
	delta = filter_length * (1.0 / fc);

	/* zero out the frequencies we don't want */
	for (i = 1; i < filter_length / 2; i++) {
		f = i / delta;
		if (f < low) {
			/* samples are take up two entries in the array: i = real, 
			 * i+1 = imaginary.  We have to zero them both. */
			workbuf[(i * 2) + 0] = 0.0;
			workbuf[(i * 2) + 1] = 0.0;
		} else {
			break;
		}
	}

	/* convert back to the time domain */
	realft(workbuf - 1, filter_length, -1);

	/* multiply result by 2/filter_length) */
	norm_ifft(workbuf, filter_length, nsamps);

	return (workbuf);
}


/* normalize the first <limit> samples of the time domain signal x(t)
** generated by the IFFT
*/
void norm_ifft(float *x, int n2, int limit)
{
	register float norm = 2.0 / (float) n2;
	register int i;

	if (limit <= 0)
		limit = n2;
	for (i = 0; i < limit; i++)
		x[i] *= norm;
}

/***********************************************************
*******  unused code 
***********************************************************/
#if(0)

#define INT(f) ((int)((f) < 0 ? ((f) - 0.5) : ((f) + 0.5)))

static int allocate_sonogram(float ***, int, int);
static float *genKernel(float *, int);
#if(0)
static void make_quick_graph(char *, char *, char *, int, float *, float *,
			     int);
#endif
static void DestructionCB(Widget, XtPointer, XtPointer);

/* amp,phase -> real,imag */
void ap2ri(float a, float p, float *r, float *i)
{
	*r = a * cos(p);
	*i = a * sin(p);
}

/* real,imag -> amp,phase */
void ri2ap(float r, float i, float *a, float *p)
{
	*a = sqrt((r * r) + (i * i));
	*p = atan2(i, r);
}

void *dsp_spectrumPlot(xword * xbuf, int nsamps, int skip, int fc,
		       char *titlestr)
{
	int i, j, l;
	int n2, n;
	float *data, *plotdata;
	float f, delta;
	float x;
	char *title;
	float parseval;

	l = nsamps;
	next2(l, n2);
	n = n2 / 2;
	data = (float *) calloc(n2 + 2, sizeof(float));	/* +2 to peel out last comp. */
	plotdata = (float *) calloc(n2 + 2, sizeof(float));

	for (i = 0; i < l; i++) {
#if(0)				/* window the data */
		c = 0.5 * (1.0 - cos(i * 2.0 * M_PI / (l - 1.0)));
		data[i] = xbuf[i * skip] * c;
#else
		data[i] = xbuf[i * skip];
#endif
	}

	while (i < n2)
		data[i++] = 0.0;

	realft(data - 1, n2, 1);
	data[n2] = data[1];	/* peel out last comp. */
	data[n2 + 1] = 0.0;
	data[1] = 0.0;

	delta = n2 * (1.0 / fc);
	parseval = 0.0;
	for (j = 0, i = 0; i < 1 + (n2 / 2); i++) {
		f = i / delta;
		x = MAG(data[i * 2], data[i * 2 + 1]);
		parseval += (x * x);
		plotdata[j++] =
		    20.0 * log10((x == 0.0) ? MINFLOAT : x *
				 (2.0 / nsamps));
		plotdata[j++] = f;
	}

	parseval /= (float) (1 + n2 / 2);
	parseval = (float) sqrt((double) parseval / (double) nsamps);

	fprintf(stderr, "rms (Parseval) = %f\n", parseval);	/*Parseval's theorem */

	title = (char *) malloc(strlen(titlestr) + 40);
	sprintf(title, "\"%s\" (Spectrum, %.1fHz/bin)",
		titlestr, (double) fc / (double) (nsamps));

	{
		int npts = j / 2;
		float *xxx = plotdata + 1;
		float *yyy = plotdata;

		atgraph_quick(title, "Freq (Hz)", "(dB ticks re 1)",
			      1, NULL, &npts, &xxx, &yyy, NULL, 2);
	}

	free(title);
	free(data);
	free(plotdata);
	return (NULL);
}

void *dsp_phasePlot(xword * xbuf, int nsamps, int skip, int fc,
		    char *titlestr, int unwrap, int in_us)

/*
**   xword *xbuf;		pointer to the data buffer
**   int fc;			sampling frequency
**   char *titlestr;		title string
**   int unwrap;		unwrap phase?
**   int in_us;			convert rad to us?
*/
{
	int i, j, l;
	int n2, n;
	int *d;
	float *data, *freqs, *p;
	float f, delta, pmin;
	char *title;
	float wrap_cutoff = M_PI;

	l = nsamps;
	next2(l, n2);
	n = n2 / 2;

	data = (float *) calloc(n2, sizeof(float));

	p = (float *) calloc(n2 / 2, sizeof(float));
	d = (int *) calloc(n2 / 2, sizeof(int));
	freqs = (float *) calloc(n2 / 2, sizeof(float));

	for (i = 0; i < l; i++) {
#if(0)				/* window the data */
		c = 0.5 * (1.0 - cos(i * 2.0 * M_PI / (l - 1.0)));
		data[i] = xbuf[i * skip] * c;
#else
		data[i] = xbuf[i * skip];
#endif
	}

	while (i < n2)
		data[i++] = 0.0;

	realft(data - 1, n2, 1);

	delta = n2 * (1.0 / fc);
	for (i = 0; i < n; i++) {
		f = i / delta;
		p[i] = PHI(data[i * 2], data[i * 2 + 1]);
		freqs[i] = f;
	}
	if (unwrap) {
		for (pmin = p[0], i = 1; i < n; i++) {
			if (p[i] < pmin)
				pmin = p[i];
		}
		for (i = 1; i < n; i++) {
			p[i] = fmod((p[i] - pmin), 2.0 * M_PI) + pmin;
			d[i] = 0;
		}

		for (j = 0, i = 1; i < n; j++, i++) {
			if (p[j] - p[i] > wrap_cutoff) {
				d[i] = 1;
			}
		}
		for (j = i = 0; i < n; i++) {
			if (d[i] != 0)
				j++;
			p[i] += wrap_cutoff * (float) j;
		}
	}

	for (i = 0; in_us && i < n; i++) {
		p[i] = 1.0e6 * p[i] / (2.0 * M_PI) / freqs[i];
	}

	title = (char *) malloc(strlen(titlestr) + 40);
	sprintf(title, "\"%s\" (Phase plot,%.1fhz/b)", titlestr,
		(double) fc / (double) (nsamps));

	{
		int npts = n - 1;
		float *xxx = freqs + 1;
		float *yyy = p + 1;

		atgraph_quick(title, "Freq (Hz)",
			      in_us ? "Phase (us)" : "Phase (rad)",
			      1, NULL, &npts, &xxx, &yyy, NULL, 1);
	}

	free(title);
	free(data);
	free(p);
	free(d);
	free(freqs);
	return (NULL);
}

void *dsp_psdPlot(xword * xbuf, int nsamps, int skip, int fc, char *title)
{
#define WINDOW(j,a,b) (1.0-fabs((((j)-1)-(a))*(b)))	/* Bartlett */
	int mm, m44, m43, m4, kk, joffn, joff, j2, j;
	float w, facp, facm, *w1, *w2;
	float sumw = 0.0;
	float den = 0.0;

	int ovrlap = 1;
	float *p, *pp, a0;
	int m, k, idx;
	int n2;

	next2(nsamps, n2);
	if (ovrlap) {
		m = n2 / 4;
		k = 0.5 * (n2 / m - 1);
	} else {
		m = n2 / 4;
		k = 0.25 * (n2 / m);
	}
	pp = (float *) calloc(sizeof(float), m);
	p = pp - 1;
	idx = 0;

	mm = m + m;
	m43 = (m4 = mm + mm) + 3;
	m44 = m43 + 1;
	w1 = vector(1, m4);
	w2 = vector(1, m);
	facm = m;
	facp = 1.0 / m;
	for (j = 1; j <= mm; j++)
		sumw += SQR(WINDOW(j, facm, facp));
	for (j = 1; j <= m; j++)
		p[j] = 0.0;
	if (ovrlap)
		for (j = 1; j <= m; j++)
			w2[j] = (idx < nsamps) ? xbuf[idx++ * skip] : 0;
	for (kk = 1; kk <= k; kk++) {
		for (joff = -1; joff <= 0; joff++) {
			if (ovrlap) {
				for (j = 1; j <= m; j++)
					w1[joff + j + j] = w2[j];
				for (j = 1; j <= m; j++)
					w2[j] =
					    (idx <
					     nsamps) ? xbuf[idx++ *
							    skip] : 0.0;
				joffn = joff + mm;
				for (j = 1; j <= m; j++)
					w1[joffn + j + j] = w2[j];
			} else {
				for (j = joff + 2; j <= m4; j += 2)
					w2[j] =
					    (idx <
					     nsamps) ? xbuf[idx++ *
							    skip] : 0.0;
			}
		}
		for (j = 1; j <= mm; j++) {
			j2 = j + j;
			w = WINDOW(j, facm, facp);
			w1[j2] *= w;
			w1[j2 - 1] *= w;
		}
		four1(w1, mm, 1);
		p[1] += (SQR(w1[1]) + SQR(w1[2]));
		for (j = 2; j <= m; j++) {
			j2 = j + j;
			p[j] += (SQR(w1[j2]) + SQR(w1[j2 - 1])
				 + SQR(w1[m44 - j2]) + SQR(w1[m43 - j2]));
		}
		den += sumw;
	}
	den *= m4;
	for (j = 1; j <= m; j++) {
		p[j] /= den;
		if (j == 1 || p[j] > a0)
			a0 = p[j];
	}
	free_vector(w2, 1, m);
	free_vector(w1, 1, m4);

	p = (float *) calloc(sizeof(float), m);

	w = ((float) fc / 2.0) / (float) m;
	for (j = 0; j < m; j++) {
		p[j] = w * j;
		pp[j] = 20.0 * log10(pp[j] / a0);
	}

	atgraph_quick(title ? title : "psd",
		      "Freq (Hz)", "dB", 1, NULL, &m, &p, &pp, NULL, 1);

	free(pp);
	free(p);
	return (NULL);
#undef WINDOW
}

int dsp_correl(xword * xbuf1, int n1, int skip1, xword * xbuf2, int n2,
	       int skip2, float us_maxtime, int fc, int normalize,
	       float **timesp, float **corrsp)
{
	int n, i, j, l;		/* l = actual number of samples */
	float *data1, *data2;	/* buffer of complex data for fft */
	float *cor;		/* correlation results */
	float *times, *corrs;	/* buffer for time values and subset corrs */
	float c, t, maxcor;

	l = (n1 > n2) ? n1 : n2;
	next2(l, n);

	data1 = (float *) malloc(n * sizeof(float));
	data2 = (float *) malloc(n * sizeof(float));
	cor = (float *) malloc(n * 2 * sizeof(float));

	for (i = 0; i < n1; i++) {	/* hanning window over data segment */
		c = 0.5 * (1.0 - cos(i * 2.0 * M_PI / (l - 1.0)));
		data1[i] = xbuf1[i * skip1] * c;
	}
	while (i < n) {		/* pad rest out with zeros */
		data1[i++] = 0.0;
	}

	for (i = 0; i < n2; i++) {	/* hanning window over data segment */
		c = 0.5 * (1.0 - cos(i * 2.0 * M_PI / (l - 1.0)));
		data2[i] = xbuf2[i * skip2] * c;
	}
	while (i < n) {		/* pad rest out with zeros */
		data2[i++] = 0.0;
	}

	/* data2, data1: puts things in standard klab conv'n (phi<0 :: L lead R) */
	correl(data2 - 1, data1 - 1, n, cor - 1);

	times = (float *) malloc(n * 2 * sizeof(float));
	corrs = (float *) malloc(n * 2 * sizeof(float));

	if (normalize) {
		for (i = 0, maxcor = 0.0; i < n; i++) {
			if ((cor[i] > 0.0) && (cor[i] > maxcor))
				maxcor = cor[i];
			else if ((cor[i] < 0.0) && (-cor[i] > maxcor))
				maxcor = -cor[i];
		}
	} else {
		for (i = 0, maxcor = 0.0; i < n; i++) {
			if (cor[i] >= 0.0)
				maxcor += cor[i];
			else if (cor[i] >= 0.0)
				maxcor -= cor[i];
		}
	}

	for (j = 0, i = n / 2; i < n; i++) {
		t = -((float) n - (float) i) / (float) fc *1e6;	/* in us */
		if (us_maxtime <= 0.0
		    || (us_maxtime > 0 && t > -us_maxtime)) {
			times[j] = t;
			corrs[j] = cor[i] / maxcor;
			j++;
		}
	}
	for (i = 0; i < n / 2; i++) {
		t = (float) i / (float) fc *1e6;	/* in us */
		if (us_maxtime <= 0.0
		    || (us_maxtime > 0 && t < us_maxtime)) {
			times[j] = t;
			corrs[j] = cor[i] / maxcor;
			j++;
		}
	}
	*timesp = times;
	*corrsp = corrs;

	free(data1);
	free(data2);
	free(cor);
	return (j);
}

void *dsp_correlPlot(xword * xbuf1, int n1, int skip1, xword * xbuf2,
		     int n2, int skip2, float us_maxtime, int fc,
		     char *titlestr, int normalize)
{
	int n, i, j, l;		/* l = actual number of samples */
	float *data1, *data2;	/* buffer of complex data for fft */
	float *cor;		/* correlation results */
	float *times, *corrs;	/* buffer for time values and subset corrs */
	float c, t, maxcor;
	char title[100];

	l = (n1 > n2) ? n1 : n2;
	next2(l, n);

	data1 = (float *) malloc(n * sizeof(float));
	data2 = (float *) malloc(n * sizeof(float));
	cor = (float *) malloc(n * 2 * sizeof(float));

	for (i = 0; i < n1; i++) {	/* hanning window over data segment */
		c = 0.5 * (1.0 - cos(i * 2.0 * M_PI / (l - 1.0)));
		data1[i] = xbuf1[i * skip1] * c;
	}
	while (i < n) {		/* pad rest out with zeros */
		data1[i++] = 0.0;
	}

	for (i = 0; i < n2; i++) {	/* hanning window over data segment */
		c = 0.5 * (1.0 - cos(i * 2.0 * M_PI / (l - 1.0)));
		data2[i] = xbuf2[i * skip2] * c;
	}
	while (i < n) {		/* pad rest out with zeros */
		data2[i++] = 0.0;
	}

	/* data2, data1: puts things in standard klab conv'n (phi<0 :: L lead R) */
	correl(data2 - 1, data1 - 1, n, cor - 1);

	times = (float *) malloc(n * 2 * sizeof(float));
	corrs = (float *) malloc(n * 2 * sizeof(float));

	if (normalize) {
		for (i = 0, maxcor = 0.0; i < n; i++) {
			if ((cor[i] > 0.0) && (cor[i] > maxcor))
				maxcor = cor[i];
			else if ((cor[i] < 0.0) && (-cor[i] > maxcor))
				maxcor = -cor[i];
		}
	} else {
		for (i = 0, maxcor = 0.0; i < n; i++) {
			if (cor[i] >= 0.0)
				maxcor += cor[i];
			else if (cor[i] >= 0.0)
				maxcor -= cor[i];
		}
	}

	for (j = 0, i = n / 2; i < n; i++) {
		t = -((float) n - (float) i) / (float) fc *1e6;	/* in us */
		if (us_maxtime <= 0.0
		    || (us_maxtime > 0 && t > -us_maxtime)) {
			times[j] = t;
			corrs[j] = cor[i] / maxcor;
			j++;
		}
	}
	for (i = 0; i < n / 2; i++) {
		t = (float) i / (float) fc *1e6;	/* in us */
		if (us_maxtime <= 0.0
		    || (us_maxtime > 0 && t < us_maxtime)) {
			times[j] = t;
			corrs[j] = cor[i] / maxcor;
			j++;
		}
	}

	if (titlestr)
		strcpy(title, titlestr);
	else
		sprintf(title, "%s-correlation",
			xbuf1 != xbuf2 ? "cross" : "auto");


	atgraph_quick(title, "Time (us)", "Corr",
		      1, NULL, &j, &times, &corrs, NULL, 1);

	free(data1);
	free(data2);
	free(cor);
	free(times);
	free(corrs);
	return (NULL);
}

void dsp_bandpass_xword(xword * xbuf, int nsamps, float low, float high,
			int window, int lr, int outskip, int calibrate)
{
	float *fbuf;
	int i, j;

	fbuf = (float *) malloc(nsamps * sizeof(float));
	assert(fbuf != NULL);
	for (i = 0, j = 0; j < nsamps; i += outskip, j++)
		fbuf[j] = (float) xbuf[i];
	dsp_bandpass(&fbuf, nsamps, low, high, window, lr, calibrate);
	for (i = 0, j = 0; j < nsamps; i += outskip, j++)
		xbuf[i] = (xword) fbuf[j];
	free(fbuf);
}

void dsp_bandpass(float **fbuf, int nsamps, float low, float high,
		  int window, int lr, int calibrate)
/*
**   float *fbuf;		pointer to the data buffer
**   int nsamps;		number of samples
**   float low, high;	filter characteristics in Hz
**   int window;		window input data?
*/
{
	int i, n2, n;
	float f, delta;		/* freq of current bin */
	float c;
	float scale;
	int flag;
	float phi;
	float amplitude, phase;
	float *tmpbuf;

	next2(nsamps, n2);
	n = n2 / 2;
	tmpbuf = (float *) realloc(*fbuf, n2 * sizeof(float));
	assert(tmpbuf != NULL);
	/* free(*fbuf); NO! NO! NO! */
	(*fbuf) = tmpbuf;

	if (window == DSP_WINDOW) {
		for (i = 0; i < nsamps; i++) {
			c = 0.5 * (1.0 -
				   cos(i * 2.0 * M_PI / (nsamps - 1)));
			(*fbuf)[i] *= c;
		}
	} else
		i = nsamps;

	while (i < n2)
		(*fbuf)[i++] = 0.0;

	realft((*fbuf) - 1, n2, 1);

	(*fbuf)[0] = 0.0;	/* mag of first (dc) component */
	(*fbuf)[1] = 0.0;	/* mag of last (??) component */
	flag = 1;
	delta = n2 * (1.0 / is_daFc);
	for (i = 1; i < n2 / 2; i++) {
		f = i / delta;
		if (f < low || f > high) {
			(*fbuf)[(i * 2) + 0] = 0.0;
			(*fbuf)[(i * 2) + 1] = 0.0;
		} else if (calibrate) {
			if (lr & IS_LEFT) {
				EarCal_FigureCal(f, &scale, NULL, &phi,
						 NULL);
			} else if (lr & IS_RIGHT) {
				EarCal_FigureCal(f, NULL, &scale, NULL,
						 &phi);
			}
			ri2ap((*fbuf)[(i * 2) + 0], (*fbuf)[(i * 2) + 1],
			      &amplitude, &phase);
			amplitude *= scale;
			phase -= phi;
			ap2ri(amplitude, phase, &((*fbuf)[(i * 2) + 0]),
			      &((*fbuf)[(i * 2) + 1]));
		}
	}

	realft((*fbuf) - 1, n2, -1);
	norm_ifft((*fbuf), n2, nsamps);
}

void dsp_gaussianfilter(xword * xbuf, int nsamps, int skip, float mean,
			float sigma, int fc, int window, int rescale)
/*
**   xword *xbuf;		pointer to the data buffer 
**   int nsamps, skip;		number of samples and skip interval 
**   float mean, sigma;		mu,sigma of gaussian in Hz 
**   int fc;			sampling freq in Hz 
**   int window;		window input data? 
*/
{
	int i, n2, n;
	float f, delta;		/* freq of current bin */
	float *fbuf, g;		/* fftbuffer, gaussian's current value */
	float postmax, premax, c;

	next2(nsamps, n2);
	n = n2 / 2;
	fbuf = (float *) calloc(n2, sizeof(float));

	for (i = 0; i < nsamps; i++) {
		if (window == DSP_WINDOW) {
			c = 0.5 * (1.0 -
				   cos(i * 2.0 * M_PI / (nsamps - 1)));
			fbuf[i] = xbuf[i * skip] * c;
		} else {
			fbuf[i] = xbuf[i * skip];
		}
		if (rescale == DSP_RESCALE) {
			if (i == 0 || fbuf[i] > premax
			    || fbuf[i] < -premax)
				premax =
				    (fbuf[i] >= 0) ? fbuf[i] : -fbuf[i];
		}
	}
	while (i < n2)
		fbuf[i++] = 0.0;

	realft(fbuf - 1, n2, 1);

	/* note: First and last components of ft have no phase values!
	 *       They're handled specially here first..
	 */
	fbuf[0] = 0.0;		/* mag of first (dc) component */
	fbuf[1] = 0.0;		/* mag of last (??) component */

	delta = n2 * (1.0 / fc);
	for (i = 1; i < n; i++) {
		f = i / delta;
		g = exp(-((mean - f) / sigma) * ((mean - f) / sigma));
#ifdef OLD_N_SLOW
		ri2ap(fbuf[i * 2 + 0], fbuf[i * 2 + 1], &a, &p);
		a *= g;
		ap2ri(a, p, &(fbuf[i * 2 + 0]), &(fbuf[i * 2 + 1]));
#else
		fbuf[i * 2 + 0] *= g;
		fbuf[i * 2 + 1] *= g;
#endif				/* OLD_N_SLOW */
	}

	realft(fbuf - 1, n2, -1);
	norm_ifft(fbuf, n2, nsamps);

	if (rescale == DSP_RESCALE) {
		for (i = 0; i < nsamps; i++) {
			if (i == 0 || fbuf[i] > postmax
			    || fbuf[i] < -postmax)
				postmax =
				    (fbuf[i] >= 0) ? fbuf[i] : -fbuf[i];
		}
		f = premax / postmax;
	} else {
		f = 1.0;
	}

	for (i = 0; i < nsamps; i++) {
		if (fbuf[i] >= 0.0)
			xbuf[i * skip] = 0.5 + f * fbuf[i];
		else
			xbuf[i * skip] = -0.5 + f * fbuf[i];
	}
	free(fbuf);
}

static int allocate_sonogram(float ***sonogramp, int nslices, int nfreqs)
{
	int i;

	if (*sonogramp) {
		while ((*sonogramp)[i]) {
			free((char *) (*sonogramp)[i]);
			(*sonogramp)[i] = NULL;
		}
		/* Should be initialized to NULL */
		free(*sonogramp);
		*sonogramp = NULL;
	}

	*sonogramp = (float **) malloc((nslices + 1) * sizeof(float *));

	for (i = 0; i < nslices; i++) {
		(*sonogramp)[i] = (float *) malloc(nfreqs * sizeof(float));
		if (!(*sonogramp)[i])
			return (0);
	}

	(*sonogramp)[i] = NULL;
	return (1);
}

int dsp_compute_sonogram(short *xbuf, int xbuf_len, int xbuf_skip,
			 float ***sonogramp, int *nslicesp, int *nfreqsp,
			 int nbits, float timeres, int adfreq)
/*
     short *xbuf;		xbuf buffer
     int xbuf_len;		length of xbuf buffer
     int xbuf_skip;		skip's in xbuf
     float ***sonogramp;	pointer to sonogram array (unallocated)
     int *nslicesp;		number of ffts in the sonogram
     int *nfreqsp;		number of frequencies in each fft
     int nbits;			bit resolution of the fft window
     float timeres;		time step for moving window (msecs)
     int adfreq;		rate at which xbuf was sampled
*/
{
	int i, j, k;
	int fft_len, halflen, slice;
	int index_step, nslices, nfreqs;
	float *signal = NULL;
	float *w = NULL;
	float **sonogram = NULL;

	fft_len = (int) pow(2.0, (float) nbits);
	halflen = fft_len / 2;

	/* four1 takes len complex numbers, stored as pairs of floats */
	if (!(signal = (float *) malloc(2 * fft_len * sizeof(float))))
		return (0);

	if (!(w = (float *) malloc(fft_len * sizeof(float))))
		return (0);

	nfreqs = halflen;
	index_step = (int) (timeres * adfreq / 1000.0);
	nslices = (xbuf_len - fft_len) / index_step + 1;

	if (!allocate_sonogram(sonogramp, nslices, nfreqs))
		return (0);

	sonogram = *sonogramp;

	/* hanning window */
	for (i = 0; i < fft_len; i++) {
		w[i] = 0.5 * (1.0 - cos(i * 2.0 * M_PI / (fft_len - 1.0)));
	}

	/* Copy stimulus into the signal array */
	for (i = 0, slice = 0; i < xbuf_len - fft_len;
	     i += index_step, slice++) {
		for (j = i, k = 0; k < fft_len; j++, k++) {
			signal[2 * k] = (float) xbuf[j * xbuf_skip] * w[k];
			signal[2 * k + 1] = 0.0;
		}

		/* Perform an FFT on the signal using the numerical recipies routine */
		four1(signal - 1, fft_len, 1);

		/* Take the square norm of the resulting complex vector and store it. */
		for (j = 0; j < halflen; j++) {
			/* Compute the ampiltude */
			sonogram[slice][j] =
			    sqrt(SQR(signal[2 * j]) +
				 SQR(signal[2 * j + 1]));
		}
	}
	if (signal) {
		free((char *) signal);
		signal = NULL;
	}
	if (w) {
		free((char *) w);
		w = NULL;
	}

	*sonogramp = sonogram;
	*nslicesp = nslices;
	*nfreqsp = nfreqs;
	return (1);
}


int dsp_pgm_sonogram(float **sono, int nslices, int nfreqs, int adfreq,
		     float dbclip, float overlap, int maxgrey,
		     float msmarks, float hzmarks, char *fname)
/*
**   float dbclip, overlap;	db, ms
*/
{
	float fmaxgrey = maxgrey;
	float maxval, minval, minval2, x, maxval2;
	int slice, f, hskip, hcnt;
	FILE *pp;
	unsigned char c;

	if (*fname == '|')
		pp = popen(fname + 1, "w");
	else
		pp = fopen2(fname, "w");
	if (pp == NULL)
		return (0);

	/* calculate: min, second-min and max values */
	for (slice = 0; slice < nslices; slice++) {
		for (f = 0; f < nfreqs; f++) {
			x = sono[slice][f];
			if (slice == 0 && f == 0) {
				minval = maxval = x;
				minval2 = -1.0;
			} else {
				if (x > maxval) {
					maxval = x;
				}
				if (x < minval) {
					minval = x;
				}
				if (x > minval
				    && (minval2 == -1.0 || x < minval2)) {
					minval2 = x;
				}
			}
		}
	}
	if (minval2 == -1.0)
		minval2 = (minval + maxval) / 2.0;

	/* convert to db and find new min/maxvals */
	maxval2 = maxval;
	for (slice = 0; slice < nslices; slice++) {
		for (f = 0; f < nfreqs; f++) {
			if ((x = sono[slice][f]) == 0.0)
				x = minval2;
			if (x != 0.0)
				sono[slice][f] = 20.0 * log10(x / maxval2);
			else
				sono[slice][f] =
				    20.0 * log10(1e-10 / maxval2);
			if ((slice == 0 && f == 0)
			    || sono[slice][f] > maxval) {
				maxval = sono[slice][f];
			}
			if ((slice == 0 && f == 0)
			    || sono[slice][f] < minval) {
				minval = sono[slice][f];
			}
		}
	}

	/* convert grey scale and plot.. */
	fprintf(pp, "P5\n# sonogram\n%d %d\n%d\n",
		nslices, nfreqs + 10, (int) fmaxgrey);
	if (hzmarks > 0.0) {
		hskip =
		    (int) (0.5 + ((float) adfreq / 2.0) / (float) hzmarks);
	} else {
		hskip = 0;
	}
	for (hcnt = 0, f = nfreqs - 1; f >= 0; f--) {
		for (slice = 0; slice < nslices; slice++) {
			if (sono[slice][f] < dbclip)
				c = fmaxgrey;
			else
				c = 0.5 + fmaxgrey -
				    (fmaxgrey * (sono[slice][f] - dbclip) /
				     (maxval - dbclip));
			if (hskip && hcnt == hskip)
				c = maxgrey - c;
			fwrite(&c, sizeof(unsigned char), 1, pp);
		}
		if (hskip) {
			if (hcnt++ >= hskip)
				hcnt = 0;
		}
	}

	if (msmarks > 0.0) {
		/* write margin + tickmarks @ 10ms */
		for (c = fmaxgrey, f = 0; f < 5; f++) {
			for (slice = 0; slice < nslices; slice++) {
				fwrite(&c, sizeof(unsigned char), 1, pp);
			}
		}
		for (f = 0; f < 5; f++) {
			for (x = overlap, slice = 0; slice < nslices;
			     slice++, x += overlap) {
				if (x >= msmarks) {
					c = (unsigned char) 0;
					x = 0.0;
				} else {
					c = fmaxgrey;
				}
				fwrite(&c, sizeof(unsigned char), 1, pp);
			}
		}
	}
	fclose(pp);

	for (slice = 0; slice < nslices; slice++)
		free(sono[slice]);
	free(sono);
	return (1);
}

/* generate kernel from F-domain rep */
static float *genKernel(float *fbuf, int nk)
{
	register int i, j, n;
	register float *k;

	realft(fbuf - 1, nk, -1);
	norm_ifft(fbuf, nk, 0);
	k = floatVec(nk);
	n = nk / 2;
	for (j = 0, i = n; i < nk; i++, j++) {
		k[j] = fbuf[i];
	}
	for (i = 0; i < n; i++, j++) {
		k[j] = fbuf[i];
	}
	free(fbuf);		/* note: free's the old buffer! */
	return (k);
}

float *dsp_lpkernel(int nbits, float corner, int fc)
/*     int nbits;			kernel size = 2^nbits */
{
	float *fbuf, delta, f;
	int i, n, nk;

	nk = (int) pow(2.0, (float) nbits);
	n = nk / 2;
	fbuf = floatVec(nk);
	fbuf[0] = 0.0;		/* mag of first (dc) component */
	fbuf[1] = 0.0;		/* mag of last (??) component */
	delta = nk * (1.0 / fc);
	for (i = 1; i < n; i++) {
		f = i / delta;
		if (f > corner) {
			fbuf[(i * 2) + 0] = 0.0;
			fbuf[(i * 2) + 1] = 0.0;
		} else {
			/* amp = 1, phi = 0 -> 1 + 0i [sqrt(1^2 + 0^2) + atan(0/1)i]
			 * -- this is a unity gain, 0-phase rect filter
			 */
			fbuf[(i * 2) + 0] = 1.0;
			fbuf[(i * 2) + 1] = 0.0;
		}
	}
	return (genKernel(fbuf, nk));
}

float *dsp_bpkernel(int nbits, float locut, float hicut, int fc)
/*     int nbits;			kernel size = 2^nbits */
{
	float *fbuf, delta, f;
	int i, n, nk;

	nk = (int) pow(2.0, (float) nbits);
	n = nk / 2;
	fbuf = floatVec(nk);
	fbuf[0] = 0.0;		/* mag of first (dc) component */
	fbuf[1] = 0.0;		/* mag of last (??) component */
	delta = nk * (1.0 / fc);
	for (i = 1; i < n; i++) {
		f = i / delta;
		if (f < locut || f > hicut) {
			fbuf[(i * 2) + 0] = 0.0;
			fbuf[(i * 2) + 1] = 0.0;
		} else {
			/* amp = 1, phi = 0 -> 1 + 0i [sqrt(1^2 + 0^2) + atan(0/1)i]
			 * -- this is a unity gain, 0-phase rect filter
			 */
			fbuf[(i * 2) + 0] = 1.0;
			fbuf[(i * 2) + 1] = 0.0;
		}
	}
	return (genKernel(fbuf, nk));
}

/* this is a rectangular filter with cos-edges.. */
float *dsp_softbpkernel(int nbits, float locut, float hicut,
			float edgeperiod, int fc)
/*
** int nbits;			kernel size = 2^nbits
** float edgeperiod;		in hz
*/
{
	float *fbuf, delta, f, c0;
	int i, n, nk;
	float locut2 = locut - edgeperiod;
	float hicut2 = hicut + edgeperiod;

	nk = (int) pow(2.0, (float) nbits);
	n = nk / 2;
	fbuf = floatVec(nk);

	fbuf[0] = 0.0;		/* mag of first (dc) component */
	fbuf[1] = 0.0;		/* mag of last (??) component */
	delta = nk * (1.0 / fc);
	c0 = M_PI / 2.0 / edgeperiod;
	for (i = 1; i < n; i++) {
		f = i / delta;
		if (f < locut2 || f > hicut2) {
			fbuf[(i * 2) + 0] = 0.0;
		} else if (f < locut) {
			fbuf[(i * 2) + 0] = cos(c0 * (locut - f));
		} else if (f > hicut) {
			fbuf[(i * 2) + 0] = cos(c0 * (f - hicut));
		} else {
			fbuf[(i * 2) + 0] = 1.0;
		}
		fbuf[(i * 2) + 1] = 0.0;
	}
	return (genKernel(fbuf, nk));
}

float *dsp_gausskernel(int nbits, float mean, float sigma, int fc)
/*     int nbits;			kernel size = 2^nbits */
{
	float *fbuf, delta, f;
	int i, n, nk;

	nk = (int) pow(2.0, (float) nbits);
	n = nk / 2;
	fbuf = floatVec(nk);
	fbuf[0] = 0.0;		/* mag of first (dc) component */
	fbuf[1] = 0.0;		/* mag of last (??) component */
	delta = nk * (1.0 / fc);
	for (i = 1; i < n; i++) {
		f = i / delta;
		fbuf[i * 2 + 0] =
		    exp(-((mean - f) / sigma) * ((mean - f) / sigma));
		fbuf[i * 2 + 1] = 0.0;
	}
	return (genKernel(fbuf, nk));
}

float *dsp_blackmankernel(int M)
{
	float *k;
	int n;

	k = floatVec(M);
	for (n = 0; n < M; n++) {
		k[n] =
		    0.42 - 0.5 * cos(2 * M_PI * n / M) +
		    0.008 * cos(4 * M_PI * n / M);
	}
	return (k);
}

float dsp_convolve(xword * xbuf_in, int nsamps, int skip, float *k,
		   int nbits)
/*     int nbits;			 kernel size = 2^nbits */
{
	register float *xout, *pf;
	register int i, j, n, idx, nk;
	int nk2;
	float xmin, xmax, pa_in;

	nk = (int) pow(2.0, (float) nbits);
	nk2 = nk / 2;
	xout = floatVec(nsamps);	/* should come back 0'd from calloc */

	xmin = xmax = xbuf_in[0];
	for (i = 1; i < nsamps; i++) {
		if (xbuf_in[i] < xmin)
			xmin = xbuf_in[i];
		else if (xbuf_in[i] > xmax)
			xmax = xbuf_in[i];
	}
	pa_in = xmax - xmin;

	for (i = 0; i < nsamps; i++) {
		n = i - nk2;
		idx = n * skip;
		pf = k;
		for (j = 0; j < nk; j++, n++, pf++, idx += skip) {
			if (n >= 0 && n < nsamps) {
				xout[i] += *pf * xbuf_in[idx];
			}
		}
	}

	for (pf = xout, i = 0; i < nsamps; i++, xbuf_in += skip, pf++) {
		if (*pf < -FRANGE) {
			*xbuf_in = -RANGE;
		} else if (*pf > FRANGE) {
			*xbuf_in = RANGE;
		} else if (*pf < 0) {
			*xbuf_in = (xword) (-0.5 + *pf);
		} else {
			*xbuf_in = (xword) (0.5 + *pf);
		}
	}

	xmin = xmax = xout[0];
	for (i = 1; i < nsamps; i++) {
		if (xout[i] < xmin)
			xmin = xout[i];
		else if (xout[i] > xmax)
			xmax = xout[i];
	}
	free(xout);
	return ((xmax - xmin) / pa_in);
}


float *dsp_adjustkernel(int nbits, char *adjfile, int fc)

/*
**   char *adjfile;		adjustment data 
**   int fc;			sampling freq in Hz 
*/
{
	float *fbuf, delta, f;
	int i, n, nk;
	float *fs, *adjs, minval, maxval;
	int npts;

	if ((npts = xyfloat_load(adjfile, &fs, &adjs)) < 1) {
		return (NULL);
	}
	/* convert db power values into requires mult. scale factors */
	floatVec_minmax(adjs, npts, &minval, &maxval);
	for (i = 0; i < npts; i++) {
		adjs[i] = pow(10.0, (minval - adjs[i]) / 20.0);
	}
	floatVec_minmax(fs, npts, &minval, &maxval);

	nk = (int) pow(2.0, (float) nbits);
	n = nk / 2;
	fbuf = floatVec(nk);
	fbuf[0] = 0.0;		/* mag of first (dc) component */
	fbuf[1] = 0.0;		/* mag of last (??) component */
	delta = nk * (1.0 / fc);
	for (i = 1; i < n; i++) {
		f = i / delta;
		if (f < minval || f > maxval) {
			fbuf[(i * 2) + 0] = fbuf[(i * 2) + 1] = 0.0;
		} else {
			fbuf[i * 2 + 0] =
			    lin_interpolate(f, npts, fs, adjs);
			fbuf[i * 2 + 1] = 0;
		}
	}
	free(fs);
	free(adjs);
	return (genKernel(fbuf, nk));
}

int dsp_fftadjust(xword * xbuf, int nsamps, int skip, char *adjfile,
		  int fc)

/*
**   xword *xbuf;		pointer to the data buffer
**   int nsamps, skip;		number of samples and skip interval
**   char *adjfile;		adjustment data
**   int fc;			sampling freq in Hz
*/
{
	int i, n2, n;
	float f, delta, s;
	float *fbuf;

	float *fs, *adjs, minval, maxval;
	int npts;

	if ((npts = xyfloat_load(adjfile, &fs, &adjs)) < 1) {
		return (0);
	}
	/* convert db power values into requires mult. scale factors */
	floatVec_minmax(adjs, npts, &minval, &maxval);
	for (i = 0; i < npts; i++) {
		adjs[i] = pow(10.0, (minval - adjs[i]) / 20.0);
	}
	floatVec_minmax(fs, npts, &minval, &maxval);

	next2(nsamps, n2);
	n = n2 / 2;
	fbuf = (float *) calloc(n2, sizeof(float));

	for (i = 0; i < nsamps; i++) {
		fbuf[i] = xbuf[i * skip];
	}
	while (i < n2)
		fbuf[i++] = 0.0;

	realft(fbuf - 1, n2, 1);

	/* note: First and last components of ft have no phase values!
	 *       They're handled specially here first..
	 */
	fbuf[0] = 0.0;		/* mag of first (dc) component */
	fbuf[1] = 0.0;		/* mag of last (??) component */

	delta = n2 * (1.0 / fc);
	for (i = 1; i < n; i++) {
		f = i / delta;
		if (f < minval || f > maxval) {
			fbuf[(i * 2) + 0] = fbuf[(i * 2) + 1] = 0.0;
		} else {
			s = lin_interpolate(f, npts, fs, adjs);
#ifdef OLD_N_SLOW
			ri2ap(fbuf[i * 2 + 0], fbuf[i * 2 + 1], &a, &p);
			a *= s;
			ap2ri(a, p, &fbuf[(i * 2) + 0],
			      &fbuf[(i * 2) + 1]);
#else
			fbuf[i * 2 + 0] *= s;
			fbuf[i * 2 + 1] *= s;
#endif				/* OLD_N_SLOW */
		}
	}

	realft(fbuf - 1, n2, -1);
	norm_ifft(fbuf, n2, nsamps);

	for (i = 0; i < nsamps; i++) {
		if (fbuf[i] >= 0.0)
			xbuf[i * skip] = fbuf[i] + 0.5;
		else
			xbuf[i * skip] = fbuf[i] - 0.5;
	}
	free(fbuf);
	free(fs);
	free(adjs);
	return (1);
}

void dsp_fftnoise(xword * xbuf, int nsamps, int skip, float locut,
		  float hicut, int fc, float usphase, float rf, float rd)
/*
**   xword *xbuf;		pointer to the data buffer
**   int nsamps, skip;		number of samples and skip interval
**   float locut, hicut;	filter characteristics in Hz
**   int fc;			sampling freq in Hz
**   float usphase;		us phase shift
**   float rf, rd;		ripple "frequency" and depth
*/
{
	int i, n2, n, seed = -1;
	float f, delta;		/* freq of current bin */
	float *fbuf, *fbuf2;
	float postmax, premax = 32000;
	float a, rphase1, rphase2, sphase = usphase / 1.0e6;
	float twopi = 2.0 * M_PI;

	next2(nsamps * 2, n2);
	n = n2 / 2;
	fbuf = (float *) calloc(n2, sizeof(float));
	fbuf2 = (float *) calloc(n2, sizeof(float));

	/* note: First and last components of ft have no phase values!
	 *       They're handled specially here first..
	 */
	fbuf[0] = 0.0;		/* mag of first (dc) component */
	fbuf[1] = 0.0;		/* mag of last (??) component */
	fbuf2[0] = 0.0;		/* mag of first (dc) component */
	fbuf2[1] = 0.0;		/* mag of last (??) component */

	delta = n2 * (1.0 / fc);
	a = 1.0;
	if (hicut == 0.0)
		hicut = locut + delta;
	for (i = 1; i < n; i++) {
		f = i / delta;
		if (f < locut || f > hicut) {
			fbuf[(i * 2) + 0] = fbuf2[(i * 2) + 0] = 0.0;
			fbuf[(i * 2) + 1] = fbuf2[(i * 2) + 1] = 0.0;
		} else {
			if (rf != 0.0)
				a = (1.0 +
				     sin(rf * (float) i)) / (2.0 / rd);
			/* rphase1 from -pi to pi */
			rphase1 = twopi * uniform(&seed) - M_PI;
			/* rphase2 offset by sphase in rad's */
			rphase2 = rphase1 - (sphase * f * twopi);
/*
      if (rphase2 > M_PI) {
	while (rphase2 > M_PI)
	  rphase2 = rphase2 - twopi;
      } else if (rphase2 < -M_PI) {
	while (rphase2 < -M_PI)
	  rphase2 = rphase2 + twopi;
      }
*/
			ap2ri(a, rphase1, &fbuf[(i * 2) + 0],
			      &fbuf[(i * 2) + 1]);
			ap2ri(a, rphase2, &fbuf2[(i * 2) + 0],
			      &fbuf2[(i * 2) + 1]);
		}
	}

	realft(fbuf - 1, n2, -1);
	norm_ifft(fbuf, n2, 0);

	realft(fbuf2 - 1, n2, -1);
	norm_ifft(fbuf2, n2, 0);

	{
		float *cor = (float *) malloc(n2 * 2 * sizeof(float));
		float *times = (float *) malloc(n2 * 2 * sizeof(float));
		float *corrs = (float *) malloc(n2 * 2 * sizeof(float));
		float t;
		int i, j, n = n2;

		correl(fbuf - 1, fbuf2 - 1, n2, cor - 1);

		for (j = 0, i = n / 2; i < n2; i++) {
			t = -((float) n - (float) i) / (float) fc *1e6;	/* in us */
			if (t > -1000.0) {
				times[j] = t;
				corrs[j] = cor[i];
				j++;
			}
		}
		for (i = 0; i < n / 2; i++) {
			t = (float) i / (float) fc *1e6;	/* in us */
			if (t < 1000.0) {
				times[j] = t;
				corrs[j] = cor[i];
				j++;
			}
		}
		atgraph_quick("Xcor", "time", "corr",
			      1, NULL, j, &times, &corrs, NULL, 1);
		free(cor);
		free(times);
		free(corrs);
	}


	fbuf += (nsamps / 2);	/* skip to middle */
	fbuf2 += (nsamps / 2);

	for (i = 0; i < nsamps; i++) {
		if (i == 0 || fbuf[i] > postmax || fbuf[i] < -postmax)
			postmax = (fbuf[i] >= 0) ? fbuf[i] : -fbuf[i];
		if (fbuf2[i] > postmax || fbuf2[i] < -postmax)
			postmax = (fbuf2[i] >= 0) ? fbuf2[i] : -fbuf2[i];
	}
	f = premax / postmax;

	for (i = 0; i < nsamps; i++) {
		if (fbuf[i] >= 0.0)
			xbuf[i * skip] = 0.5 + f * fbuf[i];
		else
			xbuf[i * skip] = -0.5 + f * fbuf[i];
		if (skip > 1) {
			if (fbuf2[i] >= 0.0)
				xbuf[i * skip + 1] = 0.5 + f * fbuf2[i];
			else
				xbuf[i * skip + 1] = -0.5 + f * fbuf2[i];
		}
	}

	fbuf -= (nsamps / 2);	/* skip back to beginning */
	free(fbuf);
	fbuf2 -= (nsamps / 2);
	free(fbuf2);
}

int dsp_zeroCrossings(xword * xbuf, int nsamps, int skip)
{
	int xs = 0;
	int i;

	for (i = 0; i < nsamps; i++) {
		if (xbuf[i * skip] < 0 && xbuf[(i + 1) * skip] >= 0) {
			xs++;
		}
	}
	return (xs);
}

void dsp_ffttone(xword * xbuf, int nsamps, int skip, float tonef, int fc,
		 int lr, int usitd)
/*
**   xword *xbuf;		pointer to the data buffer
**   int nsamps, skip;		number of samples and skip interval
**   float tonef;		filter characteristics in Hz
**   int fc;			sampling freq in Hz
*/
{
	int i, n2, n;
	float delta;		/* freq of current bin */
	float *fbufl, *fbufr;

	next2(nsamps * 2, n2);
	n = n2 / 2;
	fbufl = (float *) calloc(n2, sizeof(float));
	fbufr = (float *) calloc(n2, sizeof(float));

	for (i = 0; i < n2; i++) {
		fbufl[i] = fbufr[i] = 0.0;
	}

	delta = n2 * (1.0 / fc);
	i = delta * tonef;
	ap2ri(1.0, usitd * 1.0e-6 * tonef * 2.0 * M_PI,
	      &fbufl[(i * 2) + 0], &fbufl[(i * 2) + 1]);
	ap2ri(1.0, 0.0, &fbufr[(i * 2) + 0], &fbufr[(i * 2) + 1]);

	if (lr & IS_LEFT)
		realft(fbufl - 1, n2, -1);
	if (lr & IS_RIGHT)
		realft(fbufr - 1, n2, -1);

	fbufl += (nsamps / 2);	/* skip to middle */
	fbufr += (nsamps / 2);	/* skip to middle */

	if (lr & IS_LEFT)
		for (i = 0; i < nsamps; i++) {
			if (fbufl[i] >= 0.0)
				xbuf[i * skip] = FRANGE * fbufl[i] + 0.5;
			else
				xbuf[i * skip] = FRANGE * fbufl[i] - 0.5;
		}
	if (lr & IS_RIGHT)
		for (i = 0; i < nsamps; i++) {
			if (fbufr[i] >= 0.0)
				xbuf[i * skip + 1] =
				    FRANGE * fbufr[i] + 0.5;
			else
				xbuf[i * skip + 1] =
				    FRANGE * fbufr[i] - 0.5;
		}

	fbufl -= (nsamps / 2);	/* skip back to beginning */
	fbufr -= (nsamps / 2);	/* skip back to beginning */
	free(fbufl);
	free(fbufr);
}

float dsp_calcsigma(xword * xbuf, int nsamps, int skip)
{
	register int i, j;
	register double mean, sig;

	mean = 0.0;
	for (j = 0, i = 0; i < nsamps; i++, j += skip)
		mean += xbuf[j];
	mean /= nsamps;

	sig = 0.0;
	for (j = 0, i = 0; i < nsamps; i++, j += skip)
		sig += (xbuf[j] - mean) * (xbuf[j] - mean);
	return (sqrt(sig / (nsamps - 1)));
}



static void DestructionCB(Widget w, XtPointer topptr, XtPointer call_data)
{
	Widget top = (Widget) topptr;

	XtDestroyWidget(top);

}
#endif
