/******************************************************************
** Program: xdphys
** File:    dataio.c
** Author:  cmalek@caltech.edu (Chris Malek)
** RCSid: $Id: dataio.c,v 2.1 2002/07/15 04:30:02 cmalek Exp $
** Description: 
**
** This file contains functions and support for reading and writing
** data from/to xdphys datafiles.
**
*******************************************************************/

#include "xdphyslib.h"
#include "dataio.h"

/* ---------------------------------------------------------------------- */
/* Macros                                                                 */
/* ---------------------------------------------------------------------- */
#ifndef __linux__
#define	M_32_SWAP(a) {							\
	unsigned int _tmp = a;						\
	((char *)&a)[0] = ((char *)&_tmp)[3];				\
	((char *)&a)[1] = ((char *)&_tmp)[2];				\
	((char *)&a)[2] = ((char *)&_tmp)[1];				\
	((char *)&a)[3] = ((char *)&_tmp)[0];				\
}
#define	M_16_SWAP(a) {							\
	unsigned short _tmp = a;						\
	((char *)&a)[0] = ((char *)&_tmp)[1];				\
	((char *)&a)[1] = ((char *)&_tmp)[0];				\
}
#define	M_16_SWAP2(a) {							\
	unsigned short _tmp = a;						\
	((char *)&a)[0] = ((char *)&_tmp)[1];				\
	((char *)&a)[1] = ((char *)&_tmp)[0];				\
}
#else /* __linux__ */
#define	M_32_SWAP(a) 
#define	M_16_SWAP(a)
#endif /* not __linux__ */

/* ---------------------------------------------------------------------- */
/* Prototypes                                                             */
/* ---------------------------------------------------------------------- */
static void write_analog_to_xdphysfile(FILE *);
static void read_single_analog(FILE *, int, xword *);
#ifndef __tdtproc__
static spike_t *useTTL(int, xword *, int, int, int);
#endif
static spike_t *useWindow(int, xword *, int, int, int);
static spike_t *merge_spikelists(int *, int *);
static spike_t *useSS(int, xword *, xword *, int, int, int, float **);
static spike_t *getRaster_window(int);
static spike_t *getRaster_ss(int, float **);
static spike_t *getRaster_ttl(int);
static void write_hex_block (FILE *, xword *, int, int, int);
static void write_raster_to_xdphysfile(FILE *, spike_t *, char *, float *);

/* ---------------------------------------------------------------------- */
/* Global variables                                                       */
/* ---------------------------------------------------------------------- */
int G_detect_curr = 0, G_detect_color[4], G_detect_onoff[4];
int G_detect_x[4], G_detect_y[4];

/* ---------------------------------------------------------------------- */
/* Local Functions                                                        */
/* ---------------------------------------------------------------------- */

static void write_raster_to_xdphysfile(FILE * fp, spike_t * spikelist, 
	char *depstr, float *avePMk)
{
	int i;

	if (depstr != NULL)
		fprintf(fp, depstr);

	if (avePMk != NULL) {
		fprintf(fp, "avepmk=");
		for (i = 1; i <= avePMk[0]; i++) {
			fprintf(fp, "%f ", avePMk[i]);
		}
		fprintf(fp, "\n");
		free(avePMk);
	}
	/*  -- this is the CORRECT way to do it -- */

	fprintf(fp, "nevents=%d\n", N_SPIKES(spikelist));
	for (i = 0; i < N_SPIKES(spikelist); i++)
		fprintf(fp, "%d\t%d\n", SPIKE_TIME(spikelist, i),
			SPIKE_CHAN(spikelist, i));

	fflush(fp);

}

/* ---------------------------------------------------------------------- */

static void read_single_analog(FILE * fp, int nsamps, xword * buf)
{
	int i, tmpi;
	char tmp[32];

	for (i = 0; i < nsamps; i++) {
		fread(tmp + 4, sizeof(char), 4, fp);
		sscanf(tmp, "%x", &tmpi);
		/* Read in in little-endian (i.e. i386) format */
		M_16_SWAP(((short *) &tmpi)[2]);
		buf[i] = (short) tmpi;
		if (((i + 1) % 20) == 0)
			/* deal with the end of line character */
			fread(tmp + 4, sizeof(char), 1, fp);
		strcpy(tmp, "00000000");
	}
}

/* ---------------------------------------------------------------------- */

static spike_t *getRaster_window(int ms_time)
{
	int fc;
	int nsamps;
	int nchans = is_getADnchans();
	int nskip;
	xword *ttlbuf, *workbuf;	/* ttlbuf is channel A */
	spike_t *retval;

	nskip = 2;
#ifndef __tdtproc__
	fc = is_evtFc;
#else				/* __tdtproc__ */
	fc = is_adFc;
#endif				/* __tdtproc__ */

	if (spiker_getTTLbuffer(&nsamps, &ttlbuf) == 0) {
		alert("getRaster_window: no channel marked as 'ttl'");
		return (NULL);
	}

	assert(ttlbuf != NULL);

	/* copy the ttlbuf into a working buffer, so we can safely change its 
	 * contents.  The user may be saving the contents of that channel as an
	 * analog waveform, and we don't want to corrupt it */

	assert((workbuf =
		(xword *) calloc(nchans * nsamps, sizeof(xword))) != NULL);
	memcpy(workbuf, ttlbuf, nchans * nsamps * sizeof(xword));

	if (GI("detect.use_highpass")) {
		/* dsp_highpass_xword modifies workbuf in place */
		dsp_highpass_xword(workbuf, nsamps, fc, nchans,
				   GI("detect.highpass_lowfreq"));
	}

	/* retval contains the detected spikes */
	retval = useWindow(ms_time, workbuf, nsamps, nskip, fc);

	free(workbuf);

	return (retval);
}

/* ------------------------------------------------------------------------
   useSS

   Currently useSS() will gather ss data and ALSO the TTL pulse
   data, storing the TTL pulse data with the eventstamp -1 so
   that data can be reanalysed using conventional methods or
   the spike sorted can be tested again hardware discriminators

   note that useSS does _not_ modify the ttl buffer like useWindow
   and useTTL.  i'm not sure why they do that...
   ------------------------------------------------------------------------ */
static spike_t *useSS(int ms_time,	/* examine from t=0 to t=ms_time */
		      xword * analogbuf,	/* pointer to the input buffer */
		      xword * ttlbuf,	/* pointer to the ttl buffer */
		      int nsamps,	/* number of samples */
		      int nskip,	/* number of samples per frame */
		      int fc,	/* sampling freq */
		      float **avePMk)
{
	SpikeSet *ss;
	Sample *trace;
	EventList *event_list;
	EventTable *event_table;
	SpikeList *spike_list;
	spike_t *all_spikes, *ttl_spikes, *ss_spikes;
	int count, i, n, m, decby;
	float ms;
	char slog[MAXPATHLEN];

	/* new idea: 05-Jun-94
	 *  If there's a spike set, then useSS will call ssp_Classify()
	 *   to get the spike data back, the raster will be merged with the
	 *   results of useTTL, just in case you want to go back to the TTL
	 *   later. The analog trace will get PushData'd.
	 *  If the spike has not been inferred yet, then the trace will
	 *   simply be PushData'd and the useTTL raster returned.
	 */
#ifdef __tdtproc__
	ttl_spikes = getRaster_ttl(0);
#else				/* __tdtproc__ */
	ttl_spikes = useTTL(ms_time, ttlbuf, nsamps, nskip, fc);
#endif				/* __tdtproc__ */


	/* stash a temp copy of the current waveform in an ss compatible format
	 */

	if ((decby = GI("ss_user.decimate")) < 1) {
		decby = 1;
	}
	nsamps /= decby;

	trace = (Sample *) malloc(sizeof(Sample) * nsamps);
	for (i = 0; i < nsamps; i++) {
		trace[i] = (Sample) analogbuf[i * nskip * decby];
	}

	/* build global_sspanel, if it doesn't already exist */
	if (global_sspanel == NULL)
		ss_user_initialize();

	if (!ssp_SpikeSetInferred(global_sspanel)) {

		/* just push it and return TTL data */

		ssp_PushData(global_sspanel, trace, nsamps, NULL, NULL, NULL);
		return (ttl_spikes);
	} else {
		/* give it to the ss library to classify, push and then merge

		 * 0.0  => time in secs of data start (this collection), use auto stamp
		 * NULL => name of binary drift/log/misclustering file for this unit
		 * see dcp/proport.c, as usual..
		 */
		if (ssp_Classify(global_sspanel, trace, nsamps,
				 &event_list, &event_table, &spike_list,
				 (double) 0.0, GI("ss_user.seplog") ?
				 ss_user_genfilename(slog, "sep") : NULL)) {
			ssp_PushData(global_sspanel, trace, nsamps,
				     event_list, event_table, spike_list);
		}
		ss = ssp_SpikeSet(global_sspanel);

		/*
		 ** event_table->nevents is the maximum number of events, but
		 ** the actual events may be less due to the inclusion of the
		 ** noise model in the event_table data. The actual spike array
		 ** returned is realloc'd to the exact size just before it
		 ** gets returned.
		 **
		 ** note: here's how the event_table model numbers (m) should
		 **       currently be parsed (or use the macros provided below):
		 ** if (m < 0)
		 **   unselected model
		 ** else if (m >= 0 && m < ss->nmodels)
		 **   selected model
		 ** else if (m >= ss->nmodels && m < (2 * ss_nmodels) - 1)
		 **   outlier from model # (m - ss_nmodels)
		 ** else
		 **   bad news..
		 */

#define Keeper(ss, mnum)  \
    (mnum >= 0 && mnum < ss->nmodels)
#define Tosser(ss, mnum)  \
    (mnum < 0)
#define Outlier(ss, mnum) \
    (mnum >= ss->nmodels && mnum < ((2 * ss->nmodels) - 1))

		assert((ss_spikes = alloc_raster(event_table->nevents)) != NULL);

		for (count = n = 0; n < event_table->nevents; n++) {
			m = event_table->PrInf[n]->M_mp;	/* most likely model for this event */
			if (Keeper(ss, m)) {
				ms = event_table->tau[n];	/* convert ms to tenths of us */
				SPIKE_TIME(ss_spikes, count) = (spike_t) (ms * 10000.0);
				SPIKE_CHAN(ss_spikes, count) = m;
				count++;
			} else if (Outlier(ss, m)) {
				ms = event_table->tau[n];	/* convert ms to ticks */
				SPIKE_TIME(ss_spikes, count) = (spike_t) (ms * 10000.0);
				SPIKE_CHAN(ss_spikes, count) = OUTLIER_CODE;
				count++;
			}
		}

		(*avePMk) = (float *) malloc(sizeof(float) *
				     (1 + spike_list->nmodels));
		(*avePMk)[0] = spike_list->nmodels;
		for (i = 1; i <= spike_list->nmodels; i++) {
			if (spike_list->nspikes[i - 1] != 0) {
				(*avePMk)[i] = (spike_list->sumPMk[i - 1]) /
				    (float) (spike_list->nspikes[i - 1]);
			} else {
				(*avePMk)[i] = spike_list->sumPMk[i - 1];
			}
		}

		ss_FreeEventList(event_list);
		ss_FreeEventTable(event_table);
		ss_FreeSpikeList(spike_list);
		free(trace);

		assert((ss_spikes = realloc_raster(ss_spikes, count)) != NULL);
		N_SPIKES(ss_spikes) = (spike_t) count;

		all_spikes = merge_spikelists(ttl_spikes, ss_spikes);

		free(ttl_spikes);
		free(ss_spikes);

		return (all_spikes);
	}
}

/* ---------------------------------------------------------------------- */

static spike_t *getRaster_ss(int ms_time, float **avePMk)
{
	int fc;
	int nsamps;
	int nskip;
	xword *ttlbuf;		/* ttlbuf is channel A */
	xword *analogbuf;	/* analogbuf is channel B */

	nskip = 2;
#ifndef __tdtproc__
	fc = is_evtFc;
#else				/* __tdtproc__ */
	fc = is_adFc;
#endif				/* __tdtproc__ */

	if (spiker_getTTLbuffer(&nsamps, &ttlbuf) == 0) {
		alert("getRaster_ss: no channel marked as 'ttl'");
		return (NULL);
	}
	return (useSS(ms_time, analogbuf, ttlbuf, nsamps, nskip, fc, avePMk));
}

/* ------------------------------------------------------------------------
   getRaster_ttl: getRaster

	If we're using TDT equipment, get spikes from the ET1; if we're not
	and we still want TTL spikes, read spikes from the "ttl" buffer, and
	do thresholding.
   ------------------------------------------------------------------------ */
static spike_t *getRaster_ttl(int ms_time)
{
#ifdef __tdtproc__

	spike_t *spikes = NULL;
	unsigned long *et1buf = NULL;
	int nspikes = 0;
	int i;

	et1buf = is_getSpikebuffer(&nspikes);
	assert(et1buf != NULL);

	assert((spikes = alloc_raster(nspikes)) != NULL);

	for (i = 0; i < nspikes; i++) {
		SPIKE_TIME(spikes, i) = (spike_t) et1buf[i] * 10;
		SPIKE_CHAN(spikes, i) = /*0x01 */ TTL_CODE;
	}
	N_SPIKES(spikes) = (spike_t) nspikes;
	return (spikes);

#else				/* __tdtproc__ */
	xword *ttlbuf;		/* ttlbuf is channel A */
	xword *analogbuf;	/* analogbuf is channel B */
	int nsamps;
	int nskip;
	int fc;

	nskip = 2;
	fc = is_evtFc;
	if (spiker_getTTLbuffer(&nsamps, &ttlbuf) == 0) {
		alert("getRaster_ttl: no channel marked as 'ttl'");
		return (NULL);
	}
	return (useTTL(ms_time, ttlbuf, nsamps, nskip, fc));
#endif				/* __tdtproc__ */
}

#ifndef __tdtproc__
/* ------------------------------------------------------------------------
   useTTL

   - locates TTL pulses in an digitized ttl signal, returns a spike raster
   - side effect: modified the ttlbuf to zero/non-zero values
   ------------------------------------------------------------------------ */
static spike_t *useTTL(int ms_time,	/* examine from t=0 to t=ms_time */
		       xword * ttlbuf,	/* pointer to the input buffer */
		       int nsamps,	/* number of samples */
		       int nskip,	/* number of samples per frame */
		       int fc)
{				/* sampling freq */
	register int thresh;	/* threshold (must cross) for TTL pulse */
	register int maxtime;	/* time to sample in sample ticks */
	register int in_pulse;	/* state machine: in or out of TTL pulse */
	register int count;	/* count of spikes */
	register spike_t *spikes;	/* point to spike data */
	register int i, j;	/* index vars */
	register int signum;	/* dichotic room has inverted TTL pulses. */
	int room = 10;		/* available room in growing spike bufer */

	if (debugflag)
		fprintf(stderr, "useTTL(%d ms)\n", ms_time);

	thresh = RANGE * 50 / 100;
	maxtime = fc * ms_time / 1000;
	if (maxtime > nsamps)
		maxtime = nsamps;

	if ((signum = GI("spiker.ttl_signum")) != -1)
		signum = 1;

	count = 0;
	in_pulse = 1;
	spikes = alloc_raster(room);

	for (j = 0, i = 0; i < maxtime; i++, j += nskip) {
		if (!in_pulse && (signum * ttlbuf[j]) > thresh) {
			in_pulse = 1;
			if (count >= room) {
				room += 10;
				spikes = realloc_raster(spikes, room);
			}
			/* spike time in tenths of a microsecond */
			SPIKE_TIME(spikes, count) =
			    (spike_t) ((float) i * 1.0 / fc * 1e7);
			SPIKE_CHAN(spikes, count) = /*0x01 */ TTL_CODE;	/* spike channel */
			ttlbuf[j] = RANGE / 2;
			count++;
		} else if (in_pulse && (signum * ttlbuf[j]) < thresh) {
			in_pulse = 0;
			ttlbuf[j] = 0;
		} else {
			ttlbuf[j] = 0;
		}
	}
	N_SPIKES(spikes) = (spike_t) count;
	return (spikes);
}
#endif				/* __tdtproc__ */

/* ------------------------------------------------------------------------
   useWindow

   - does simple window discrimination on analog trace
   ------------------------------------------------------------------------ */
static spike_t *useWindow(int ms_time,	/* examine from t=0 to t=ms_time */
			  xword * analogbuf,	/* pointer to the analog waveform */
			  int nsamps,	/* number of samples */
			  int nskip,	/* number of samples per frame */
			  int fc)
{				/* sampling freq */
	register int maxtime;	/* time to sample in sample ticks */
	register int in_spike;	/* state machine: in or out of spike */
	register spike_t *spikes;	/* count of spikes, point to spike data */
	register int count;
	register int i, j, k;	/* index vars */
	register int i2, j2, valid;	/* excursion variables */
	int room = 10;		/* available room in growing spike bufer */
	int maxx;
	xword ym, yl, yr;
	float offx;

	if ((maxtime = fc * ms_time / 1000) > nsamps)
		maxtime = nsamps;

	count = 0;
	in_spike = 1;
	spikes = alloc_raster(room);

	for (j = 0, i = 0; i < maxtime; i++, j += nskip) {
		if (!in_spike && (analogbuf[j] > G_detect_y[0])) {
			maxx = -1;
			for (j2 = j + nskip, i2 = i + 1; i2 < maxtime; i2++, j2 += nskip) {
				if (maxx == -1) {
					if ((analogbuf[j2 - nskip] > analogbuf[j2])) {
						maxx = i2 - 1;
					}
				}
				if (analogbuf[j2] <= G_detect_y[0]) {
					break;
				}
			}

			if (maxx != -1) {
				ym = analogbuf[maxx * nskip];
				yl = ym - analogbuf[(maxx - 1) * nskip];
				yr = ym - analogbuf[(maxx + 1) * nskip];
				offx = (float) maxx + (float) (yl - yr) / 
					(float) (yl + yr) / 2.0;

				valid = 1;
				for (k = 1; k < 4; k++) {
					if (G_detect_onoff[k]) {
						if (G_detect_color[k]) {
							if ((lin_interpolate2(analogbuf, nsamps, nskip, offx + G_detect_x[k])) > G_detect_y[k])
								valid = 0;
						} else {
							if ((lin_interpolate2(analogbuf, nsamps, nskip, offx + G_detect_x[k])) < G_detect_y[k])
								valid = 0;
						}
					}
				}
			} else {
				offx = (float) (i2 - 1);
				valid = 0;
			}

			if (valid) {
				if (count > (room - 1)) {
					room += 10;
					spikes =
					    realloc_raster(spikes, room);
				}
				SPIKE_TIME(spikes, count) = (spike_t) (offx * 1.0 / fc * 1.0e7);
				SPIKE_CHAN(spikes, count) = TTL_CODE;
				count++;
			} else {
				if (count > (room - 1)) {
					room += 10;
					spikes = realloc_raster(spikes, room);
				}
				SPIKE_TIME(spikes, count) = (spike_t) ((float) offx * 1.0 / fc *
					       1.0e7);
				SPIKE_CHAN(spikes, count) = OUTLIER_CODE;
				count++;
				while (i2 < maxtime) {
					if (analogbuf[j2] <= G_detect_y[0]) {
						break;
					} else {
						i2++;
						j2 += nskip;
					}
				}
			}
			j = j2 - nskip;	/* skip ahead past end of spike or bad spike */
			i = i2 - 1;
			in_spike = 0;
		} else if (in_spike && (analogbuf[j] <= G_detect_y[0])) {
			in_spike = 0;
		}
	}
	N_SPIKES(spikes) = (spike_t) count;
	return (spikes);
}

/* ---------------------------------------------------------------------- */

static spike_t *merge_spikelists(spike_t * ttl_sl, spike_t * sl2)
{
	spike_t *sl;
	int n, nttl, n2;

	if (debugflag) {
		fprintf(stderr,
			"merge_spikelists: Merging %d + %d spikes\n",
			ttl_sl[0], sl2[0]);
	}
	sl = alloc_raster(N_SPIKES(ttl_sl) + N_SPIKES(sl2));
/*  the preceding line used to be...
   sl = (int *) calloc(2 * (ttl_sl[0] + sl2[0]) + 1, sizeof(int));
 */
	for (nttl = n2 = n = 0; nttl < N_SPIKES(ttl_sl) || n2 < N_SPIKES(sl2); n++) {
		if (nttl < N_SPIKES(ttl_sl) && (n2 >= N_SPIKES(sl2)
			|| SPIKE_TIME(ttl_sl, nttl) < SPIKE_TIME(sl2, n2))) {
			SPIKE_TIME(sl, n) = SPIKE_TIME(ttl_sl, nttl);
			SPIKE_CHAN(sl, n) = TTL_CODE;
			nttl++;
		} else {
			SPIKE_TIME(sl, n) = SPIKE_TIME(sl2, n2);
			SPIKE_CHAN(sl, n) = SPIKE_CHAN(sl2, n2);
			n2++;
		}
	}
	N_SPIKES(sl) = (spike_t) n;
	return (sl);
}

/* ---------------------------------------------------------------------- */

static void write_hex_block (FILE *fp, xword *buffer, int buf_length, 
	int decimate, int skip)
{
	xword tmpword;
	int buf_index;
	int output_index;
	char tmp[32];

	for (buf_index = output_index = 0; output_index < buf_length; 
			output_index++, buf_index += (decimate * skip)) {
		tmpword = buffer[buf_index];
		sprintf(tmp, "%08x", tmpword);
		fprintf(fp, "%s", tmp + 4);
		if (((output_index + 1) % 20) == 0)
			fprintf(fp, "\n");
	}

	if ((output_index % 20) != 0)
		fprintf(fp, "\n");
}

/* ---------------------------------------------------------------------- */
/* write_stimulus_to_xdphysfile                                           */
/*                                                                        */
/* Write the stimulus to the xdphys wavefile in hex format.  Write        */
/* channels based on  the mono flag.                                      */
/*                                                                        */
/* WARNING: I assume here that were only ever going to play two channels. */
/*          This may not always be true.                                  */
/* ---------------------------------------------------------------------- */
static void write_stimulus_to_xdphysfile(FILE * fp, int side)
{
	/* side can be SYN_LEFT, SYN_RIGHT, SYN_BOTH */
	xword *outbuf;
	int outsize;
	int decimate;

	if (GI("stim.save")) {
		decimate = GI("stim.decimate");
		outbuf = is_getDAbuffer(&outsize);

		if (side & SYN_LEFT) {
			fprintf(fp, "STIMULUS\n");
			fprintf(fp, "channel=1\n");
			write_hex_block (fp, outbuf, outsize, decimate, 2);
			fprintf(fp, "END_STIMULUS\n");
		}
		if (side & SYN_RIGHT) {
			fprintf(fp, "STIMULUS\n");
			fprintf(fp, "channel=2\n");
			write_hex_block (fp, outbuf+1, outsize, decimate, 2);
			fprintf(fp, "END_STIMULUS\n");
		}
	}
}

/* ---------------------------------------------------------------------- */
/* write_analog_to_xdphysfile                                             */
/*                                                                        */
/* The difference between write_buffers and write_analog_to_xdphysfile    */
/* is that the former writes a binary file in old .ana format, while the  */
/* latter converts the AD buffers to hexadecimal text format to be written*/ 
/* into the xdphys datafile.                                              */
/* ---------------------------------------------------------------------- */
#define WRITE_THIS_CHANNEL(chan_state) (strncmp(chan_state, "on", 2) == 0) 
static void write_analog_to_xdphysfile(FILE * fp)
{
	xword **anabufs;
	int chan;
	int nsamples;
	int decimate;
	char *chan_state;
	char tmp[32];

	decimate = GI("ana-decimate");
	if (LastVarUndefined || decimate < 1)
		decimate = 1;

	assert(sizeof(xword) == 2);

	if (fp != NULL) {
		spiker_getAnalogBuffers(&nsamples, &anabufs);
		nsamples /= decimate;

		for (chan = 0; chan < is_getADnchans(); chan++) {
			sprintf(tmp, "ana.channel%d", chan + 1);
			chan_state = GS(tmp);
			if (chan_state != NULL) {
				if (WRITE_THIS_CHANNEL(chan_state)) {

					fprintf(fp, "TRACE\n");
					fprintf(fp, "channel=%d\n", chan + 1);

					write_hex_block(fp, anabufs[chan], nsamples, decimate, 
						is_getADnchans());
						

					fprintf(fp, "END_TRACE\n");
				}
			}
		}
		free(anabufs);
	}
}
#undef WRITE_THIS_CHANNEL

/* ---------------------------------------------------------------------- */
/* Public Functions                                                       */
/* ---------------------------------------------------------------------- */

/* ---------------------------------------------------------------------- */
/* write_data_to_xdphysfile                                               */
/*                                                                        */
/* This is the function that the modules should call during their         */
/* runScan functions to write the data record to the xdphys datafile.     */
/* ---------------------------------------------------------------------- */

void write_data_to_xdphysfile(FILE *fp, spike_t *raster, char *depstr, 
	float *avePMk, int write_analog, int write_stimulus, int side)
{

	write_raster_to_xdphysfile(fp, raster, depstr, avePMk);
	if (write_analog)
		write_analog_to_xdphysfile(fp);
	if (write_stimulus)
		write_stimulus_to_xdphysfile(fp, side);
}

/* ---------------------------------------------------------------------- */

void get_analog_conversion(float *tomv, float *offset, int chan)
{
	/* get calibration data for analog channel */

	if (chan == 0) {
		if (ad_to_mv(1.0, tomv, NULL, AddaRelative) == 0)
			(*tomv) = (*offset) = 0.0;
		if (ad_to_mv(2.0, offset, NULL, AddaAbsolute) == 0)
			(*tomv) = (*offset) = 0.0;
	} else {
		if (ad_to_mv(1.0, NULL, tomv, AddaRelative) == 0)
			(*tomv) = (*offset) = 0.0;
		if (ad_to_mv(0.0, NULL, offset, AddaAbsolute) == 0)
			(*tomv) = (*offset) = 0.0;
	}
}

/* ---------------------------------------------------------------------- */
/* write_buffers                                                          */
/*                                                                        */
/* The difference between write_buffers and write_analog_to_xdphysfile    */
/* is that the former writes a binary file in old .ana format, while the  */
/* latter converts the AD buffers to hexadecimal text format to be written*/ 
/* into the xdphys datafile.                                              */
/*                                                                        */
/* write_buffers is now _rarely_ used.                                    */
/* ---------------------------------------------------------------------- */

void write_buffers(FILE *fp, char *depstr, char *filename, int rasternum)
{
	xword *adbuf, *lbuf, *rbuf;
	int nsamps, i, j;
	float tomv, offset;
	int decimate;

	decimate = GI("ana-decimate");
	if (LastVarUndefined || decimate < 1)
		decimate = 1;

	/* this just strips out the trailing newlines.. */
	if (depstr && (i = strlen(depstr)) > 0) {
		if (depstr[i - 1] == '\n')
			depstr[i - 1] = 0;
	}

	if ((fp != NULL) && ((adbuf = is_getADbuffer(&nsamps)) != NULL)) {
		nsamps /= decimate;

		assert((lbuf = (xword *) calloc(nsamps, sizeof(xword))) != NULL);
		assert((rbuf = (xword *) calloc(nsamps, sizeof(xword))) != NULL);
		for (j = i = 0; i < nsamps; i++, j += (2 * decimate)) {
			lbuf[i] = adbuf[j];
			rbuf[i] = adbuf[j + 1];
		}

		get_analog_conversion(&tomv, &offset, 0);
		writewave(fp, lbuf, is_adFc / decimate, nsamps, 1,
			  depstr, "write_buffers()", tomv, offset, filename,
			  rasternum);

		get_analog_conversion(&tomv, &offset, 1);
		writewave(fp, rbuf, is_adFc / decimate, nsamps, 1,
			  depstr, "write_buffers()", tomv, offset, filename,
			  rasternum);

		free(lbuf);
		free(rbuf);
	}
}


/* ---------------------------------------------------------------------- */

int read_analog_from_xdphysfile(FILE * fp, FILEDATA * data, xword *** anabufs,
	       int **chan_ids)
{
	int i;
	int nsamps;
	char tmp[32];
	float adFc;
	float epoch;
	int pos;
	int nchans;
	int decimate;

	adFc = (float) FD_GI(data, "adFc");
	epoch = ((float) FD_GI(data, "epoch")) / 1000.0;

	/* only file format >= 2.1 has ana.nchans, and 
	 * more than 1 analog trace per raster */
	if (FD_GI(data, "file.version") >= 3)
		nchans = FD_GI(data, "ana.nchans");
	else
		nchans = 1;

	assert((*anabufs =
		(xword **) calloc(nchans, sizeof(xword *))) != NULL);
	assert((*chan_ids = (int *) calloc(nchans, sizeof(int))) != NULL);
	for (i = 0; i < nchans; i++)
		assert(((*anabufs)[i] = (xword *) malloc(nsamps *
							 sizeof(xword *)))
		       != NULL);

	strcpy(tmp, "00000000");
	decimate = GI("ana-decimate");
	if (LastVarUndefined || decimate < 1)
		decimate = 1;

	nsamps = (int) (adFc * epoch) / decimate;
	pos = ftell(fp);
	fread(tmp + 4, sizeof(char), 4, fp);
	if ((!strcmp(tmp, "0000depv")) || (!strcmp(tmp, "0000END_"))) {
		/* There are no traces associated with this raster */
		fseek(fp, pos, SEEK_SET);
		for (i = 0; i < nchans; i++) {
			free((*anabufs)[i]);
			free((*anabufs));
			(*anabufs) = NULL;
			return (0);
		}
	} else {
		if (FD_GI(data, "file.version") >= 3) {
			for (i = 0; i < nchans; i++) {
				/* Read the "TRACE" marker */
				fgets(tmp, sizeof(tmp), fp);
				/* Read in the "channel=" line */
				fgets(tmp, sizeof(tmp), fp);
				sscanf(tmp, "channel=%d", &((*chan_ids)[i]));
				read_single_analog(fp, nsamps, (*anabufs)[i]);
				/* Read the "END_TRACE" marker */
				fgets(tmp, sizeof(tmp), fp);
			}
		} else {
			/* Cover backwards compatibility */
			read_single_analog(fp, nsamps, (*anabufs)[0]);
			(*chan_ids)[0] = 1;
		}
	}

	return (nchans);
}

/* ---------------------------------------------------------------------- */

int countRaster(float mintime, float maxtime, spike_t * spikelist,
		int mask)
/*
   **   mintime;                minimum possible timestamp (ms)
   **   maxtime;                maximum possible timestamp (ms)
   **   mask;           event -> spike mask
 */
{
	int i, count;

	mintime *= 10000.0;
	maxtime *= 10000.0;

	for (count = i = 0; i < N_SPIKES(spikelist); i++) {
		if (mask == MU_CODE || SPIKE_CHAN(spikelist, i) == mask)
			if ((SPIKE_TIME(spikelist, i) >= (int) mintime) &&
			    (SPIKE_TIME(spikelist, i) <= (int) maxtime))
				count++;
	}
	return (count);
}

/* ---------------------------------------------------------------------- */

spike_t *mergeRasters(spike_t * spikelist1, spike_t * spikelist2)
{

	spike_t *merged;
	int i1, i2;
	int mcount;

	i1 = 1;
	i2 = 1;

	/* worst case malloc size */
	merged = alloc_raster(N_SPIKES(spikelist1) + N_SPIKES(spikelist2));
	assert(merged != NULL);

	mcount = 0;
	while (i1 < N_SPIKES(spikelist1) && i2 < N_SPIKES(spikelist2)) {
		if (i2 == N_SPIKES(spikelist2) ||
		    (SPIKE_TIME(spikelist1, i1) <
		     SPIKE_TIME(spikelist2, i2))) {
			SPIKE_TIME(merged, mcount) =
			    SPIKE_TIME(spikelist1, i1);
			SPIKE_CHAN(merged, mcount) =
			    SPIKE_CHAN(spikelist1, i1);
			i1++;
		} else {
			if (i1 == N_SPIKES(spikelist2) ||
			    (SPIKE_TIME(spikelist2, i2) <
			     SPIKE_TIME(spikelist1, i1))) {
				SPIKE_TIME(merged, mcount) =
				    SPIKE_TIME(spikelist2, i2);
				SPIKE_CHAN(merged, mcount) =
				    SPIKE_CHAN(spikelist2, i2);
				i2++;
			} else {	/* i1, i2 != 0 and *spikelist1 == *spikelist2 */
				SPIKE_TIME(merged, mcount) =
				    SPIKE_TIME(spikelist1, i1);
				SPIKE_CHAN(merged, mcount) =
				    SPIKE_CHAN(spikelist2,
					       i2) | SPIKE_CHAN(spikelist2,
								i2);
				i1++;
				i2++;
			}
		}
		mcount++;
		assert(mcount <
		       (N_SPIKES(spikelist1) + N_SPIKES(spikelist2)));
	}
	N_SPIKES(merged) = mcount;
	assert((merged = realloc_raster(merged, mcount)) != NULL);

	return (merged);
}


/* ---------------------------------------------------------------------- */

spike_t *readRaster(FILE * fp, FILEDATA * data, float **avepmk)
{
	int i, nspikes, nmodels;
	spike_t *spikelist = NULL;
	spike_t time;
	int fc;
	char buf[1000];
	int gsrss;
	int bin_spikes = 0;

	fc = FD_GI(data, "evtFc");
	gsrss = (!strcmp(FD_GV(data, "detect.mode"), "ss")) ? 1 : 0;
	bin_spikes = GI("raster.bin_spikes");
	(*avepmk) = NULL;

	while (fgets(buf, sizeof(buf), fp) != NULL) {
		if (strncmp(buf, "avepmk=", 7) == 0) {
			nmodels = FD_GI(data, "xdphys.gsrnmodels");
			assert(((*avepmk) = (float *) malloc(nmodels * sizeof(float))) 
					!= NULL);
			for (i = 0; i < nmodels; i++)
				fscanf(fp, "%f", (*avepmk) + i);
		} else if (strncmp(buf, "nevents=", 8) == 0) {
			nspikes = atoi(buf + 8);
			assert((spikelist =
				alloc_raster(nspikes)) != NULL);

			/* Deal with the different versions of xdowl datafiles */
			if (((data->ver_num < 2.45) || (data->ver_num == 2.51)) 
				&& !(strncmp(data->progname, "dowl", 4))) {	/* there was actually never a ver 2.45 */
				for (i = 0; i < nspikes; i++) {
					fscanf(fp, "%d %d\n", &time,
					       &SPIKE_CHAN(spikelist, i));
					/* convert from adticks to tenths of microseconds */
					SPIKE_TIME(spikelist, i) = (int) (((float) time /
						    (float) fc) * 1e7);
					if (bin_spikes)
						SPIKE_TIME(spikelist, i) = SPIKE_TIME(spikelist, i) -
						    (SPIKE_TIME (spikelist, i) % 208);

					if (SPIKE_TIME(spikelist, i) == 0) {
						nspikes--;
						i--;
					}
					if (gsrss) {
						if (SPIKE_CHAN(spikelist, i) == 99)
							SPIKE_CHAN(spikelist, i) = TTL_CODE;
						else if (SPIKE_CHAN(spikelist, i) == 100)
							SPIKE_CHAN(spikelist, i) = OUTLIER_CODE;
						else if (SPIKE_CHAN(spikelist, i) == -1)
							SPIKE_CHAN(spikelist, i) = MU_CODE;
					} else {
						if (SPIKE_CHAN(spikelist, i) == 1)
							SPIKE_CHAN(spikelist, i) = TTL_CODE;
					}
				}
			} else {
				for (i = 0; i < nspikes; i++) {
					fscanf(fp, "%d %d\n", &time, &SPIKE_CHAN(spikelist, i));
					if (bin_spikes)
						time -= (time % 208);
					SPIKE_TIME(spikelist, i) = time;
					if (SPIKE_TIME(spikelist, i) == 0) {
						nspikes--;
						i--;
					}
				}
			}

			N_SPIKES(spikelist) = (spike_t) nspikes;

			return (spikelist);
		}
	}
	return (NULL);
}

/* ------------------------------------------------------------------------
   getRaster

   - try to locate spike events in the analog input buffer.
   - returns a spike raster: [nspikes, ts1, cs1, ts2, cs2 ...]
   where tsN is a timestamp and csN is a event stamp
   - ms_time specified where to STOP looking for events!
   + can have side effects: may modify the analog trace see 
   static/local functions above

   if __tdtproc__ is defined, spikes are detected by a Tucker Davis SD1 
   Spike Discriminator/ET1 Event Timer pair, which returns timestamps
   in microseconds, so we don't have to search the ttlbuffer for spikes.
   ------------------------------------------------------------------------ */
spike_t *getRaster(int ms_time, float **avePMk)
{
	(*avePMk) = NULL;
	switch (getRaster_getmode()) {
	case GSR_window:
		return (getRaster_window(ms_time));
		break;
	case GSR_ss:
		return (getRaster_ss(ms_time, avePMk));
		break;
	case GSR_ttl:
		return (getRaster_ttl(ms_time));
		break;
	}

	return (NULL);
}

/* ---------------------------------------------------------------------- */
int getRaster_getmode(void)
{
	char *mode = GS("detect.mode");

	if (!strcmp(mode, "ttl"))
		return (GSR_ttl);
	else if (!strcmp(mode, "window"))
		return (GSR_window);
	else if (!strcmp(mode, "ss"))
		return (GSR_ss);
	else
		return (GSR_invalid);

}

/* ---------------------------------------------------------------------- */

int getRaster_getnmodels(void)
{
	switch (getRaster_getmode()) {
	case GSR_ttl:
		return (1);
	case GSR_window:
		return (1);
	case GSR_ss:
		{
			SpikeSet *ss = ssp_SpikeSet(global_sspanel);
			if (ss)
				return (ss->nmodels);
			else
				return (0);
		}
	default:
		fprintf(stderr,
			"getRaster_getnmodels: invalid mode -> ttl\n");
		return (1);
	}
}


/* ------------------------------------------------------------------------
   spiker_getTTLbuffer

   Return the TTL buffer -- which buffer is the TTL buffer is determined
   by the "spiker.channelA" and "spiker.channelB" svars

   There will be either 1 or 0 TTL buffers.

   Returns 1 if a TTL buf exists, 0 otherwise.

   ------------------------------------------------------------------------ */
int spiker_getTTLbuffer(int *nsamples, xword ** ttlbuf)
{
	xword *adbuf;
	int discrim_channel;
	char *mode;
	char msg[128];

	assert((adbuf = is_getADbuffer(nsamples)) != NULL);

	mode = GS("detect.mode");

	discrim_channel = GI("detect.discrim_channel");

	if ((discrim_channel < 1) || (discrim_channel > is_getADnchans())) {
		sprintf(msg,
			"spiker_getTTLbuffer: detect.discrim_channel must be >= 1 and <= %d",
			is_getADnchans());
		alert(msg);
		*nsamples = 0;
		*ttlbuf = NULL;
		return (0);
	} else {
		*ttlbuf = adbuf + (discrim_channel - 1);
		return (1);
	}

	return (1);
}

/* ------------------------------------------------------------------------
   spiker_getAnalogNchans

   Return number of channels for which analog data will be saved.

   ------------------------------------------------------------------------ */
int spiker_getAnalogNchans(void)
{
	char tmp[32];
	char *chan_state;
	int i;
	int nchans = 0;

	for (i = 0; i < is_getADnchans(); i++) {
		sprintf(tmp, "ana.channel%d", i + 1);
		chan_state = GS(tmp);
		if (chan_state != NULL)
			if (strncmp(chan_state, "on", 2) == 0)
				nchans++;
	}

	return (nchans);

}

/* ------------------------------------------------------------------------
   spiker_getAnalogBuffers

   Return number of samples per channel in *nsamps
   Return an array of pointers to the analog buffers .

   anabufs should be freed when the calling function is done with it.

   ------------------------------------------------------------------------ */
void spiker_getAnalogBuffers(int *nsamples, xword *** anabufs)
{
	xword *adbuf;
	int i, nchans;

	nchans = is_getADnchans();

	assert((adbuf = is_getADbuffer(nsamples)) != NULL);
	assert((*anabufs = calloc(nchans, sizeof(xword *))) != NULL);

	for (i = 0; i < nchans; i++)
		(*anabufs)[i] = adbuf + i;
}

/* ---------------------------------------------------------------------- */

void unmakePatternVector(int *pattern_vector)
{
	if (pattern_vector)
		free(pattern_vector);
}

/* ---------------------------------------------------------------------- */

int *makePatternVector(char *pattern)
{
	extern char *index(), *strtok(), *strcpy();
	char *p, *copy;
	int i, *vec;

	vec = NULL;
	if (index(pattern, ',') != NULL) {
		copy = strcpy(malloc(strlen(pattern) + 1), pattern);
		for (i = 0, p = strtok(copy, ","); p != NULL;
		     p = strtok(NULL, ",")) {
			if (i == 0)
				vec = (int *) calloc((i + 2), sizeof(int));
			else
				vec =
				    (int *) realloc(vec,
						    (i + 2) * sizeof(int));
			vec[i++] = atoi(p);
		}
		free(copy);
		if (vec)
			vec[0] = i;
	} else if (*pattern == '#') {
		vec = (int *) calloc(2, sizeof(int));
		vec[0] = 1;
		vec[1] = atoi(pattern + 1);
	} else if (sscanf(pattern, "%d", &i) == 1) {
		vec = (int *) calloc(2, sizeof(int));
		vec[0] = 1;
		vec[1] = i;
	} else if (strcasecmp(pattern, "TTL") == 0) {
		vec = (int *) calloc(2, sizeof(int));
		vec[0] = 1;
		vec[1] = TTL_CODE;
	} else if (strcasecmp(pattern, "OUT") == 0) {
		vec = (int *) calloc(2, sizeof(int));
		vec[0] = 1;
		vec[1] = OUTLIER_CODE;
	}
	return (vec);
}


/* ---------------------------------------------------------------------- */

int matchSpike(int code,	/* spiker code for this event */
	       char *pattern,	/* user-specified pattern */
	       int *pattern_vector)
{				/* to speed up vector patterns  */
	extern char *index(), *strtok(), *strcpy();
	char *p, *copy;
	int i;

	if (pattern_vector != NULL) {
		for (i = 0; i < pattern_vector[0]; i++)
			if (code == pattern_vector[i + 1])
				return (1);
		return (0);
	} else if (strcasecmp(pattern, "MU") == 0) {
		return (1);	/* MultiUnit */
	} else if (strcasecmp(pattern, "TTL") == 0) {
		return (code == TTL_CODE);
	} else if (strcasecmp(pattern, "OUT") == 0) {
		return (code == OUTLIER_CODE);
	} else if (index(pattern, ',') != NULL) {
		copy = strcpy(malloc(strlen(pattern) + 1), pattern);
		for (p = strtok(copy, ","); p != NULL;
		     p = strtok(NULL, ",")) {
			if (code == atoi(p)) {
				free(copy);
				return (1);
			}
		}
		free(copy);
		return (0);
	} else if (*pattern == '#') {
		return (code == atoi(pattern + 1));	/* specific spike channel */
	} else if (sscanf(pattern, "%d", &i) == 1) {
		return (code == i);	/* specific spike channel (alt spec) */
	} else {
		fprintf(stderr, "matchSpike: invalid pattern \"%s\"\n",
			pattern);
		return (0);
	}
}

/* ---------------------------------------------------------------------- */

/*
   rep == 0 if we're in the middle of a repetition, otherwise
   (when a integer number of reps have been completed), rep == rep#

   note that this function also introduces a small memory leak
 */

void printAvePMks(float *avepmk, int rep, int stims_per_rep)
{
	static float *sumpmk = NULL;
	int i;

	if (sumpmk == NULL)
		assert((sumpmk =
			(float *) calloc(1 + avepmk[0],
					 sizeof(float))) != NULL);
	else if (sumpmk[0] != avepmk[0]) {
		free(sumpmk);
		assert((sumpmk =
			(float *) calloc(1 + avepmk[0],
					 sizeof(float))) != NULL);
	}
	sumpmk[0] = avepmk[0];

	for (i = 1; i < (1 + avepmk[0]); i++)
		sumpmk[i] += avepmk[i];

	if (rep > 0) {
		fprintf(stderr, "\n<PMk[i]> (rep #%d): ", rep);
		for (i = 1; i < (1 + sumpmk[0]); i++)
			fprintf(stderr, "%f ", sumpmk[i] / stims_per_rep);
		free(sumpmk);
		sumpmk = NULL;
	}
}
