/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/


/******************************************************************
**  RCSID: $Id: addacal.c,v 2.53 2002/07/15 04:29:59 cmalek Exp $
** Program: dowl
**  Module: addacal
**  Author: mazer
** Descrip: proport/656 a/d and d/a calibration support (DC+Range)
**
** Revision History (most recent last)
**
** Mon Feb 15 23:55:08 1993 mazer
**  changed function names: fad2mv()->ad_to_mv() mv2ad()->mv_to_ad()
**  and deleted the old ad2mv(), since it was never used
**
** Tue Feb 16 13:35:41 1993 mazer
**  changed #defines to enum types and if/else's to switches
**
** Mon Feb  7 16:36:18 1994 mazer
**  changed location of proport calibration file
**  1st choice: $DOWLDIR/$HOST.proport.data
**  2nd choice: $DOWLDIR/proport.data
**  else: fail..
**
** Sun Feb 13 12:16:21 1994 mazer
**  now only choice is DOWLDIR/HOST.proport.data -- this
**  is required -- you must setenv HOST or these will fail..
**
** NOV 96 bjarthur
**  changed location to $DOWLDIR/calib/data/$HOST.proport.data
**
** 97.4 bjarthur
**  changed DOWLDIR to XDPHYSDIR
**
** 97.4 cmalek
**  changed calib data dir to $XDPHYSDIR/calib
**  changed $HOST.proport.data to $HOST.addacal.data
**
** 99.2 cmalek
**  changed search path for addacal file to look first for
**  $HOST.addacal.data, then for local.addacal.data
**  removed MASSCOMP related code
**
*******************************************************************/

/*
** calibration data consists of two values (for each gain setting):
**  slope (m) & intercept (b)
**       mv = (ticks - b) / m
**    or
**       ticks = 1/m * mv + b
**  therefore:
**    slope (m) is in units of ticks/mv
**    and the intercept (b) is in units of ticks
**
**  These values where measured semi-automatically using a calibrated
**  dc-signal generator and fitting a linear regression to the data
**  points using "xaddacal".  This will simple system NOT work for the
**  newer AC-COUPLED proports -- the gain's must be measured with a
**  calibrated frequency generator -- like the one on the lock in
**  amp..
*/

/*
** tool-fns to take care of dc offset and variable gain problems
** with the proport 656 converters on the sparc (nop's for masscomp)
**
** bool = ad_to_mv(float adticks, float *lmv, float *rmv, int mode)
** bool = mv_to_da(float mv, xword *ladticks, xword radticks, int mode)
**
** note: right now the data file units are mv, not volts!!
*/

#ifndef __NO_XDPHYSLIB
#include "xdphyslib.h"
#else				/* __NO_XDPHYSLIB */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "systems.h"
#include "addacal.h"
#include "svars.h"
#include "pop.h"
#include "misc.h"
#endif				/* __NO_XDPHYSLIB */

static int data_loaded = 0;
static int ad_first_time = 1;
static int da_first_time = 1;

static int readVoltCal(void);
static int getCalFloat(float *, char *);

static int readVoltCal(void)
{
	char *p, *from;
#ifndef __linux__
	extern char *getenv();
#endif
	char tmp[50];
	char msg[1024];

	if (getenv("HOST") == NULL) {
		alert("Please setenv HOST");
		return (0);
	}

	strcpy(tmp, "$(XDPHYSDIR)/calib/$(HOST).addacal.data");
	from = envexpand(tmp);
	if (!probefile(from)) {
		fprintf(stderr, "xdphys: %s not found\n", from);
		strcpy(tmp, "$(XDPHYSDIR)/calib/local.addacal.data");
		from = envexpand(tmp);
		fprintf(stderr, "xdphys: trying %s\n", from);
		if (!probefile(from)) {
			fprintf(stderr, "xdphys: %s not found\n", from);
			sprintf(msg,
				"Can't find tick to mV conversion file\n"
				"This should be located at:\n%s\n", from);
			alert(msg);
			exit(1);
		}
	}
	fprintf(stderr, "xdphys: using %s\n", from);
	if (debugflag)
		fprintf(stderr,
			"Loading tick/mV conversion factors from:\n%s\n",
			from);

	p = strsave(from);
	if (loadvars(p, global_svar_table)) {
		free(p);
		if (debugflag)
			fprintf(stderr, "addacal-load OK");
		return (1);
	} else {
		free(p);
		if (debugflag)
			fprintf(stderr, "addacal-load FAILED");
		return (0);
	}
}

static int getCalFloat(float *f, char *p)
{
	*f = GF(p);
	if (LastVarUndefined) {
		alert("No addacal data on: %s", p);
		return (0);
	}
	return (1);
}

void adda_force_reload(void)
{
	data_loaded = 0;
	ad_first_time = 1;
	da_first_time = 1;
}

int ad_to_mv(float adticks, float *lmv, float *rmv, AddaMode mode)
{
	/* see notes above to see that these slope and intercept values
	 * actually refer to.
	 */
	static float l_slope, l_intercept;	/* left: slope, intercept (B,A) */
	static float r_slope, r_intercept;	/* right: slope, intercept (B,A) */

	if (!data_loaded) {
		if (!readVoltCal())
			return (0);
		data_loaded = 1;
	}
	if (ad_first_time) {
#ifndef __tdtproc__
		int i;
		i = GI("proport.pad");
		assert(!LastVarUndefined);
		if (i == -20) {
			if (!getCalFloat(&l_slope, "ad.left.slope.-20db"))
				return (0);
			if (!getCalFloat
			    (&l_intercept, "ad.left.intercept.-20db"))
				return (0);
			if (!getCalFloat(&r_slope, "ad.right.slope.-20db"))
				return (0);
			if (!getCalFloat
			    (&r_intercept, "ad.right.intercept.-20db"))
				return (0);
		} else if (i == 0) {
			if (!getCalFloat(&l_slope, "ad.left.slope.0db"))
				return (0);
			if (!getCalFloat
			    (&l_intercept, "ad.left.intercept.0db"))
				return (0);
			if (!getCalFloat(&r_slope, "ad.right.slope.0db"))
				return (0);
			if (!getCalFloat
			    (&r_intercept, "ad.right.intercept.0db"))
				return (0);
		} else {
			alert("Unknown ProPort pad mode: %ddB", i);
			return (0);
		}
#else				/* __tdtproc__ */
		if (!getCalFloat(&l_slope, "ad.left.slope"))
			return (0);
		if (!getCalFloat(&l_intercept, "ad.left.intercept"))
			return (0);
		if (!getCalFloat(&r_slope, "ad.right.slope"))
			return (0);
		if (!getCalFloat(&r_intercept, "ad.right.intercept"))
			return (0);
#endif				/* __tdtproc__ */

		ad_first_time = 0;
	}
	switch (mode) {
	case AddaAbsolute:
		if (lmv != NULL)
			*lmv = (adticks - l_intercept) / l_slope;
		if (rmv != NULL)
			*rmv = (adticks - r_intercept) / r_slope;
		return (1);
	case AddaRelative:
		if (lmv != NULL)
			*lmv = adticks / l_slope;
		if (rmv != NULL)
			*rmv = adticks / r_slope;
		return (1);
	default:
		return (0);
	}
}

int mv_to_da(float mv, xword * ladticks, xword * radticks, int mode)
{
	static float l_slope, l_intercept;	/* left: slope, intercept (B,A) */
	static float r_slope, r_intercept;	/* right: slope, intercept (B,A) */

	if (!data_loaded) {
		if (!readVoltCal())
			return (0);
		data_loaded = 1;
	}
	if (da_first_time) {
		if (!getCalFloat(&l_slope, "da.left.slope"))
			return (0);
		if (!getCalFloat(&l_intercept, "da.left.intercept"))
			return (0);
		if (!getCalFloat(&r_slope, "da.right.slope"))
			return (0);
		if (!getCalFloat(&r_intercept, "da.right.intercept"))
			return (0);
		da_first_time = 0;
	}

	switch (mode) {
	case AddaAbsolute:
		if (ladticks != NULL)
			*ladticks = (xword) ((mv - l_intercept) / l_slope);
		if (radticks != NULL)
			*radticks = (xword) ((mv - r_intercept) / r_slope);
		return (1);
	case AddaRelative:
		if (ladticks != NULL)
			*ladticks = (xword) (mv / l_slope);
		if (radticks != NULL)
			*radticks = (xword) (mv / r_slope);
		return (1);
	default:
		return (0);
	}
}
