/*
 * ** This file is part of XDowl
 * ** Copyright (c) 1994 Jamie Mazer
 * ** California Institute of Technology
 * ** <mazer@asterix.cns.caltech.edu>
 */

/******************************************************************
**  RCSID: $Id: Canvas.c,v 2.48 2002/07/15 04:29:56 cmalek Exp $
** Program: dowl
**  Module: Canvas.c
**  Author: mazer
** Descrip: canvas widget -- true widget for X/ps display lists
**
** Revision History (most recent last)
**
** Sat Jul 21 ??:??:?? 1990 mazer
**  creation date: drawing/canvas widget
**
** Wed Feb 12 03:15:37 1992 mazer
**  removed XtCanvasForceUpdate() and associated properties.
**  widget will always stay updated, I hope..
**
** Sat Feb 15 20:46:49 1992 mazer
**  added CanvasInfo popup statline .. this is VERY dowl dependent!
**
** Sun Mar 15 13:41:49 1992 mazer
**  changed action names to be "more" unique -- they're all
**  prefixed with canvas- now.
**
** Fri May  1 19:45:29 1992 mazer
**  added CanvasVerticalText and friends..
**
** Sun May  3 16:51:15 1992 mazer
**  added xfig/copy support
**  
** Sun Jan 31 21:49:33 1993 mazer
**  moved PrintAction into PostScript.c to avoid dependencies
**  
** Mon Feb  1 20:03:51 1993 mazer
**  - removed statline__() stuff: now ReportAction() should write
**    directly onto the window using a "rubberGC"
**  - removed XtCanvasDeleteAll() and modified XtCanvasClear() to
**    take a second argument (T/F) indicating whether or not to
**    clear the coord-stack
**
** Fri Feb  5 15:50:22 1993 mazer
**  - unionized the canvasObject struct
**  - added CanvasPoints type for fast waveform and cluster plotting
**    displays
**
** Tue Feb 16 13:27:42 1993 mazer
**  - added erase_rubber field to the canvas widget to hold the
**    current state of the rubber banding boxes and lines. This
**    was to prevent ghosting when the canvas is cleared during a
**    drag operation.
**
** Tue Mar 16 19:23:58 1993 mazer
**  added a simple message window for tracking cursor etc..
**
** Tue Apr  6 00:38:28 1993 mazer
**  style stuff for lines is messed up -- the current situation
**  fails for LineSolid, since it's defined to be 0 and xor'ing
**  is signaled by <= 0 style values.. this must be fixed..
**
** Mon Oct 11 14:39:19 1993 mazer
**  trimmed out XtNcolorize and XtNpaletteNN stuff..
**
** Thu Dec  9 22:34:39 1993 mazer
**  removed copy-action since it's covered more generally 
**  by the PostScript module
**
** Sun Mar  6 15:27:03 1994 mazer
**  moved postscript and fig output generators from PostScript.c
**  back here -- to avoid sucking in Wave, AtPlotter etc..
**
*******************************************************************/

#include "xdphyslib.h"

#define Abs(n)		(((n) < 0) ? -(n) : (n))
#define Max(x, y)	(((x) > (y)) ? (x) : (y))
#define Min(x, y)	(((x) < (y)) ? (x) : (y))

static Cursor horiz_arrow, vert_arrow, hv_arrow, cross_hair;
static void Redisplay(CanvasWidget w, XEvent * event, Region region);
static void GeneratePSAction(CanvasWidget w, XEvent * event,
			     String * params, Cardinal * num_params);
static void GenerateFIGAction(Widget w, XEvent * event, String * params,
			      Cardinal * num_params);

static void pushVC(CanvasWidget);
static int popVC(CanvasWidget);
static void InputAction(CanvasWidget, XEvent *, String *, Cardinal *);
static void make_rubberGC(Widget, GC *);
static void RubberBoxAction(CanvasWidget, XEvent *, String *, Cardinal *);
static void RubberXhairAction(CanvasWidget, XEvent *, String *,
			      Cardinal *);
static void SetViewAction(CanvasWidget, XEvent *, String *, Cardinal *);
static void RefreshAction(CanvasWidget, XEvent *, String *, Cardinal *);
static void ClearAction(CanvasWidget, XEvent *, String *, Cardinal *);
static void ReportAction(CanvasWidget, XEvent *, String *, Cardinal *);
static void ToggleLinesAction(CanvasWidget, XEvent *, String *,
			      Cardinal *);
static void Initialize(CanvasWidget, CanvasWidget);
static int xscale(CanvasWidget, float, int);
static int yscale(CanvasWidget, float, int);
static void scaleObject(CanvasWidget, CanvasObject *);
static void drawtext(CanvasWidget, CanvasObject *, int);
static void drawObject(CanvasWidget, CanvasObject *, Region, int);
static void Destroy(CanvasWidget);
static void Resize(CanvasWidget);
static Boolean SetValues(CanvasWidget, CanvasWidget, CanvasWidget);
static CanvasObject *newCanvasObject(CanvasWidget);
static void freeCanvasObject(CanvasObject *);
static char *getafilename(char *, char *);

static XtResource resources[] = {
#define offset(field) XtOffset(CanvasWidget, canvas.field)
	{XtNforeground, XtCForeground, XtRPixel, sizeof(Pixel),
	 offset(foreground_pixel), XtRString, XtDefaultForeground}
	,
	{XtNcallback, XtCCallback, XtRCallback, sizeof(XtCallbackList),
	 offset(input_callback), XtRCallback, (caddr_t) NULL}
	,
	{XtNfont, XtCFont, XtRString, sizeof(String),
	 offset(fontname), XtRString, "6x13"}
	,
	{XtNupdateOnDraw, XtCUpdateOnDraw, XtRBoolean, sizeof(Boolean),
	 offset(update_on_draw), XtRImmediate, (XtPointer) True}
	,
#undef offset
};

static void pushVC(CanvasWidget w)
{
	CanvasCoord *cc = (CanvasCoord *) malloc(sizeof(CanvasCoord));

	cc->xmin = w->canvas.vc_xmin;
	cc->xmax = w->canvas.vc_xmax;
	cc->ymin = w->canvas.vc_ymin;
	cc->ymax = w->canvas.vc_ymax;
	cc->next = w->canvas.vc_stack;
	w->canvas.vc_depth++;
	w->canvas.vc_stack = cc;
}

static int popVC(CanvasWidget w)
{
	CanvasCoord *cc;

	cc = w->canvas.vc_stack;
	if (w->canvas.vc_stack) {
		w->canvas.vc_xmin = cc->xmin;
		w->canvas.vc_xmax = cc->xmax;
		w->canvas.vc_ymin = cc->ymin;
		w->canvas.vc_ymax = cc->ymax;
		w->canvas.vc_stack = cc->next;
		free(cc);
		w->canvas.vc_depth--;
		return (1);
	} else {
		return (0);
	}
}

static void InputAction(CanvasWidget w, XEvent * event, String * params,
			Cardinal * num_params)
/*
 * String *params;    ignored
 * Cardinal *num_params;  ignored
 */
{
	/*
	 *  Note: InputAction (by default) get's called only in response to
	 *    unmodified mouse presses. If you want to catch other events,
	 *    use Action/Translation Table combination!!
	 */
	XtCallCallbacks(w, XtNcallback, (caddr_t) event);
}

static void make_rubberGC(Widget w, GC * gcp)
{
	XGCValues values;

	values.function = GXxor;
	values.foreground = 0xffffffff;
	values.plane_mask =
	    BlackPixelOfScreen(XtScreen(w)) ^
	    WhitePixelOfScreen(XtScreen(w));

	if (*gcp == NULL)
		*gcp = XCreateGC(XtDisplay(w), XtWindow(w),
				 GCFunction | GCPlaneMask | GCForeground,
				 &values);
	else
		XSetState(XtDisplay(w), *gcp, values.foreground,
			  values.background, values.function,
			  values.plane_mask);
}

static void RubberBoxAction(CanvasWidget w, XEvent * event,
			    String * params, Cardinal * num_params)
{
	static int xc, yc;
	static int xl, yl;
	static int x, y, wd, ht;
	static GC gc = NULL;
	static CanvasWidget on = NULL;

	if (*num_params == 0) {
		if (w == on) {
			if (xl >= 0 && w->canvas.erase_rubber)
				XDrawRectangle(XtDisplay(w), XtWindow(w),
					       gc, x, y, wd, ht);
			xl = event->xbutton.x;
			yl = event->xbutton.y;
			if (xc < xl) {
				x = xc;
				wd = xl - xc;
			} else {
				x = xl;
				wd = xc - xl;
			}
			if (yc < yl) {
				y = yc;
				ht = yl - yc;
			} else {
				y = yl;
				ht = yc - yl;
			}
			XDrawRectangle(XtDisplay(w), XtWindow(w), gc, x, y,
				       wd, ht);
			w->canvas.erase_rubber = True;
		}
	} else if (strcasecmp(params[0], "begin") == 0) {
		xc = event->xbutton.x;
		yc = event->xbutton.y;
		make_rubberGC(w, &gc);
		on = w;
		xl = -1;
		if (*num_params > 1 && !strcasecmp(params[1], "horiz"))
			XDefineCursor(XtDisplay(w), XtWindow(w),
				      horiz_arrow);
		else if (*num_params > 1 && !strcasecmp(params[1], "vert"))
			XDefineCursor(XtDisplay(w), XtWindow(w),
				      vert_arrow);
		else
			XDefineCursor(XtDisplay(w), XtWindow(w), hv_arrow);
	} else if (strcasecmp(params[0], "end") == 0) {
		on = NULL;
		if (xl >= 0 && w->canvas.erase_rubber)
			XDrawRectangle(XtDisplay(w), XtWindow(w), gc, x, y,
				       wd, ht);
		w->canvas.erase_rubber = False;
		XUndefineCursor(XtDisplay(w), XtWindow(w));
	}
}

static void RubberXhairAction(CanvasWidget w, XEvent * event,
			      String * params, Cardinal * num_params)
{
	static int lx, ly;
	static float lxf, lyf;
	static char msgbuf[100];
	static GC gc = NULL;

#define DOIT()	\
    XDrawLine(XtDisplay(w), XtWindow(w), gc, lx, 0, lx, w->core.height); \
    XDrawLine(XtDisplay(w), XtWindow(w), gc, 0, ly, w->core.width, ly);

#define SETIT() \
    lx = event->xbutton.x; \
    ly = event->xbutton.y; \
    lxf = (float) lx * (w->canvas.vc_xmax - w->canvas.vc_xmin) / \
      (float) w->core.width + w->canvas.vc_xmin; \
    lyf = (float) (w->core.height - ly) * \
      (w->canvas.vc_ymax - w->canvas.vc_ymin) / (float) w->core.height + \
	w->canvas.vc_ymin; \
    sprintf(msgbuf, "X=%.2f  Y=%.2f", lxf, lyf);

	if (*num_params == 0) {
		if (w->canvas.erase_rubber) {
			DOIT();
		}
		SETIT();
		DOIT();
		XtCanvasSetMsg(w, msgbuf);
		w->canvas.erase_rubber = True;
	} else if (strcasecmp(params[0], "begin") == 0) {
		w->canvas.xhair_set = 0;
		make_rubberGC(w, &gc);
		SETIT();
		DOIT();
		XtCanvasSetMsg(w, msgbuf);
		w->canvas.erase_rubber = True;
	} else if (strcasecmp(params[0], "end") == 0) {
		DOIT();
		XtCanvasSetMsg(w, NULL);
		w->canvas.xhair_set = 1;
		w->canvas.xhair_px = lx;
		w->canvas.xhair_py = ly;
		if (w->canvas.erase_rubber)
			XtCanvasXYtoWC(w, lx, ly, &w->canvas.xhair_x,
				       &w->canvas.xhair_y);
		w->canvas.erase_rubber = False;
	}
#undef SETIT
#undef DOIT
}


static void SetViewAction(CanvasWidget w, XEvent * event, String * params,
			  Cardinal * num_params)
{
	float fx, fy, shift, f;

	if ((*num_params < 1) || (strcasecmp(params[0], "reset") == 0)) {
		popVC(w);
	} else {
		XtCanvasXYtoWC(w, event->xbutton.x, event->xbutton.y, &fx,
			       &fy);
		if (strcasecmp(params[0], "left") == 0) {
			pushVC(w);
			w->canvas.vc_xmin = fx;
		} else if (strcasecmp(params[0], "right") == 0) {
			pushVC(w);
			w->canvas.vc_xmax = fx;
		} else if (strcasecmp(params[0], "top") == 0) {
			pushVC(w);
			w->canvas.vc_ymax = fy;
		} else if (strcasecmp(params[0], "bottom") == 0) {
			pushVC(w);
			w->canvas.vc_ymin = fy;
		} else if (strcasecmp(params[0], "beginhold") == 0) {
			w->canvas.tmp_xmin = w->canvas.vc_xmin;
			w->canvas.tmp_xmax = w->canvas.vc_xmax;
			w->canvas.tmp_ymin = w->canvas.vc_ymin;
			w->canvas.tmp_ymax = w->canvas.vc_ymax;
		} else if (strcasecmp(params[0], "endhold") == 0) {
			pushVC(w);
			if (w->canvas.tmp_xmin < w->canvas.tmp_xmax) {
				w->canvas.vc_xmin = w->canvas.tmp_xmin;
				w->canvas.vc_xmax = w->canvas.tmp_xmax;
			} else if (w->canvas.tmp_xmin > w->canvas.tmp_xmax) {
				w->canvas.vc_xmax = w->canvas.tmp_xmin;
				w->canvas.vc_xmin = w->canvas.tmp_xmax;
			}	/* else
				   * XBell(XtDisplay(w), 0); */
			if (w->canvas.tmp_ymin < w->canvas.tmp_ymax) {
				w->canvas.vc_ymin = w->canvas.tmp_ymin;
				w->canvas.vc_ymax = w->canvas.tmp_ymax;
			} else if (w->canvas.tmp_ymin > w->canvas.tmp_ymax) {
				w->canvas.vc_ymax = w->canvas.tmp_ymin;
				w->canvas.vc_ymin = w->canvas.tmp_ymax;
			}	/* else 
				   * XBell(XtDisplay(w), 0); */
		} else if (strcasecmp(params[0], "xmin") == 0)
			if (*num_params == 2)	/* hold! */
				w->canvas.tmp_xmin = fx;
			else
				w->canvas.vc_xmin = fx;
		else if (strcasecmp(params[0], "xmax") == 0)
			if (*num_params == 2)	/* hold! */
				w->canvas.tmp_xmax = fx;
			else
				w->canvas.vc_xmax = fx;
		else if (strcasecmp(params[0], "ymin") == 0)
			if (*num_params == 2)	/* hold! */
				w->canvas.tmp_ymin = fy;
			else
				w->canvas.vc_ymin = fy;
		else if (strcasecmp(params[0], "ymax") == 0)
			if (*num_params == 2)	/* hold! */
				w->canvas.tmp_ymax = fy;
			else
				w->canvas.vc_ymax = fy;
		else if (strcasecmp(params[0], "shift_left") == 0) {
			shift = 0.75 * (f =
					w->canvas.vc_xmax -
					w->canvas.vc_xmin);
			if (w->canvas.vc_xmin - shift >= w->canvas.wc_xmin) {
				w->canvas.vc_xmin -= shift;
				w->canvas.vc_xmax -= shift;
			} else {
				w->canvas.vc_xmin = w->canvas.wc_xmin;
				w->canvas.vc_xmax = w->canvas.wc_xmin + f;
				XBell(XtDisplay(w), 0);
			}
		} else if (strcasecmp(params[0], "shift_right") == 0) {
			shift = 0.75 * (f =
					w->canvas.vc_xmax -
					w->canvas.vc_xmin);
			if (w->canvas.vc_xmax + shift <= w->canvas.wc_xmax) {
				w->canvas.vc_xmin += shift;
				w->canvas.vc_xmax += shift;
			} else {
				XBell(XtDisplay(w), 0);
				w->canvas.vc_xmax = w->canvas.wc_xmax;
				w->canvas.vc_xmin = w->canvas.wc_xmax - f;
			}
		} else if (strcasecmp(params[0], "shift_up") == 0) {
			shift = 0.75 * (f =
					w->canvas.vc_ymax -
					w->canvas.vc_ymin);
			if (w->canvas.vc_ymin - shift >= w->canvas.wc_ymin) {
				w->canvas.vc_ymin -= shift;
				w->canvas.vc_ymax -= shift;
			} else {
				w->canvas.vc_ymin = w->canvas.wc_ymin;
				w->canvas.vc_ymax = w->canvas.wc_ymin + f;
				XBell(XtDisplay(w), 0);
			}
		} else if (strcasecmp(params[0], "shift_down") == 0) {
			shift = 0.75 * (f =
					w->canvas.vc_ymax -
					w->canvas.vc_ymin);
			if (w->canvas.vc_ymax + shift <= w->canvas.wc_ymax) {
				w->canvas.vc_ymin += shift;
				w->canvas.vc_ymax += shift;
			} else {
				XBell(XtDisplay(w), 0);
				w->canvas.vc_ymax = w->canvas.wc_ymax;
				w->canvas.vc_ymin = w->canvas.wc_ymax - f;
			}
		} else
			XtAppWarning(XtWidgetToApplicationContext(w),
				     "SetViewAction: bad argument list");
	}
}

static void RefreshAction(CanvasWidget w, XEvent * event, String * params,
			  Cardinal * num_params)
/*
 * String *params;    ignored
 * Cardinal *num_params;  ignored
 */
{
	XtCanvasSetVC(w, (float) 0.0, (float) 0.0, (float) 0.0,
		      (float) 0.0);
}

static void ClearAction(CanvasWidget w, XEvent * event, String * params,
			Cardinal * num_params)
/*
 * String *params;    ignored
 * Cardinal *num_params;  ignored
 */
{
	XtCanvasClear(w, False);
}

static void ReportAction(CanvasWidget w, XEvent * event, String * params,
			 Cardinal * num_params)
/*
 * String *params;    ignored
 * Cardinal *num_params;  ignored
 */
{
	float fx, fy, x, y;
	char repbuf[200];

	static float x0, y0;
	static int xc, yc;
	static int xl, yl;
	static GC gc = NULL;

	if (*num_params && !strcasecmp(params[0], "start")) {
		xc = xl = -1;
		yc = yl = -1;
		make_rubberGC(w, &gc);
		XDefineCursor(XtDisplay(w), XtWindow(w), cross_hair);

		XtCanvasXYtoWC(w, event->xbutton.x, event->xbutton.y, &x0,
			       &y0);
		sprintf(repbuf, "[%.2f,%.2f]", x0, y0);
		XtCanvasSetMsg(w, repbuf);
		w->canvas.erase_rubber = True;
		xl = xc = event->xbutton.x;
		yl = yc = event->xbutton.y;
	} else if (*num_params && !strcasecmp(params[0], "tell")) {
		if (w->canvas.erase_rubber) {
			if (xc != xl || yc != yl)
				XDrawLine(XtDisplay(w), XtWindow(w), gc,
					  xl, yl, xc, yc);
		}
		XtCanvasXYtoWC(w, event->xbutton.x, event->xbutton.y, &fx,
			       &fy);
		if (*num_params > 1 && !strcasecmp(params[1], "horiz")) {
			xc = event->xbutton.x;
			yc = event->xbutton.y = yl;
			x = fx - x0;
			y = fy - y0;
			sprintf(repbuf, "X: %.2f :: %.2f = %.2f", x0, fx,
				sqrt(x * x));
			XtCanvasSetMsg(w, repbuf);
		} else if (*num_params > 1
			   && !strcasecmp(params[1], "vert")) {
			xc = event->xbutton.x = xl;
			yc = event->xbutton.y;
			x = fx - x0;
			y = fy - y0;
			sprintf(repbuf, "Y: %.2f :: %.2f = %.2f", y0, fy,
				sqrt(y * y));
			XtCanvasSetMsg(w, repbuf);
		} else {
			xc = event->xbutton.x;
			yc = event->xbutton.y;
			x = fx - x0;
			y = fy - y0;
			sprintf(repbuf, "(%.2f,%.2f)--(%.2f,%.2f) = %.2f",
				x0, y0, fx, fy, sqrt(x * x + y * y));
			XtCanvasSetMsg(w, repbuf);
		}
		XDrawLine(XtDisplay(w), XtWindow(w), gc, xl, yl, xc, yc);
		w->canvas.erase_rubber = True;
	} else if (*num_params && !strcasecmp(params[0], "stop")) {
		XUndefineCursor(XtDisplay(w), XtWindow(w));
		XtCanvasSetMsg(w, NULL);
		if (w->canvas.erase_rubber) {
			if (xc != yl || yc != yl)
				XDrawLine(XtDisplay(w), XtWindow(w), gc,
					  xl, yl, xc, yc);
		}
		w->canvas.erase_rubber = False;
	} else {
		XtAppWarning(XtWidgetToApplicationContext(w),
			     "Canvas Widget: ReportAction(illegal parameter)");
	}
}

static void ToggleLinesAction(CanvasWidget w, XEvent * event,
			      String * params, Cardinal * num_params)
/*
 * String *params;
 * Cardinal *num_params;
 */
{
	w->canvas.mark_lines = !w->canvas.mark_lines;
	XClearWindow(XtDisplay(w), XtWindow(w));
	w->canvas.erase_rubber = False;
	Redisplay(w, NULL, NULL);
}

static XtActionsRec actions[] = {
	{"canvas-input", InputAction},
	{"canvas-rubberxhair", RubberXhairAction},
	{"canvas-rubberbox", RubberBoxAction},
	{"canvas-setview", SetViewAction},
	{"canvas-refresh", RefreshAction},
	{"canvas-clear", ClearAction},
	{"canvas-report", ReportAction},
	{"canvas-marklines", ToggleLinesAction},
	{"canvas-ps", GeneratePSAction},
	{"canvas-fig", GenerateFIGAction},
};

static char default_translations[] = "\
      None<Btn2Down>:	canvas-rubberxhair(begin)\n\
    None<Btn2Motion>: 	canvas-rubberxhair()\n\
	None<Btn2Up>: 	canvas-rubberxhair(end)\n\
\
 <KeyPress>bracketleft:	canvas-setview(left) canvas-refresh()\n\
<KeyPress>bracketright:	canvas-setview(right) canvas-refresh()\n\
       <KeyPress>equal:	canvas-setview(top) canvas-refresh()\n\
   <KeyPress>quoteleft:	canvas-setview(bottom) canvas-refresh()\n\
  <KeyPress>quoteright:	canvas-setview(bottom) canvas-refresh()\n\
\
      Ctrl<Btn1Down>:	canvas-rubberbox(begin,horiz)\
			 canvas-setview(beginhold) canvas-setview(xmin,hold)\n\
	Ctrl<Btn1Up>:	canvas-rubberbox(end)\
			 canvas-setview(xmax,hold) canvas-setview(endhold)\
			  canvas-refresh()\n\
      Ctrl<Btn2Down>:	canvas-rubberbox(begin,both)\
			 canvas-setview(xmin,hold) canvas-setview(ymax,hold)\n\
	Ctrl<Btn2Up>:	canvas-rubberbox(end)\
			 canvas-setview(xmax,hold) canvas-setview(ymin,hold) \
			  canvas-setview(endhold)\
			   canvas-refresh()\n\
      Ctrl<Btn3Down>:	canvas-rubberbox(begin,vert)\
			 canvas-setview(beginhold) canvas-setview(ymax,hold)\n\
	Ctrl<Btn3Up>:	canvas-rubberbox(end)\
			 canvas-setview(ymin,hold) canvas-setview(endhold)\
			  canvas-refresh()\n\
     Ctrl<BtnMotion>:	canvas-rubberbox()\n\
\
  Ctrl<KeyPress>Left:	canvas-setview(shift_left) canvas-refresh()\n\
 Ctrl<KeyPress>Right:	canvas-setview(shift_right) canvas-refresh()\n\
    Ctrl<KeyPress>Up:	canvas-setview(shift_down) canvas-refresh()\n\
  Ctrl<KeyPress>Down:	canvas-setview(shift_up) canvas-refresh()\n\
    <KeyPress>Escape:	canvas-setview(reset) canvas-refresh()\n\
Meta<KeyPress>Escape:	canvas-refresh()\n\
\
       Meta<BtnDown>:	canvas-report(start)\n\
    Meta<Btn1Motion>:	canvas-report(tell, horiz)\n\
    Meta<Btn2Motion>:	canvas-report(tell)\n\
    Meta<Btn3Motion>:	canvas-report(tell, vert)\n\
         Meta<BtnUp>:	canvas-report(stop)\n\
\
     Meta<KeyPress>M:	canvas-marklines()\n\
          <KeyPress>:	canvas-input()\n\
             <BtnUp>:	canvas-input()\n\
           <BtnDown>:	canvas-input()\n\
\
          <Key>Print:	canvas-ps()\n\
      Meta<Key>Print:	canvas-ps(file)\n\
     Shift<Key>Print:	canvas-fig()\n\
          Meta<Key>C:	canvas-fig(file)\n";

static void Initialize(CanvasWidget request, CanvasWidget new)
{
	Font font;
	XGCValues values;
	static int init_cursors = 1;

	values.background = new->core.background_pixel;
	values.foreground = new->canvas.foreground_pixel;

	new->canvas.gc =
	    XtGetGC(new, GCForeground | GCBackground, &values);
	XSetFunction(XtDisplay(new), new->canvas.gc, GXcopy);
	new->canvas.lineList = NULL;

	new->canvas.wc_xmin = new->canvas.wc_xmax = 0.0;
	new->canvas.wc_ymin = new->canvas.wc_ymax = 0.0;

	new->canvas.vc_xmin = new->canvas.vc_xmax = 0.0;
	new->canvas.vc_ymin = new->canvas.vc_ymax = 0.0;
	new->canvas.vc_stack = NULL;
	new->canvas.vc_depth = 0;

	new->canvas.mark_lines = False;
	new->canvas.the_pen = new->canvas.foreground_pixel;
	new->canvas.msgwin = (Window) NULL;

	if ((font =
	     XLoadFont(XtDisplay(new), new->canvas.fontname)) != NULL) {
		XSetFont(XtDisplay(new), new->canvas.gc, font);
		new->canvas.fontinfo =
		    XLoadQueryFont(XtDisplay(new), new->canvas.fontname);
	} else {
		XtAppWarning(XtWidgetToApplicationContext(new),
			     "Can't load specified canvas font.");
	}

	if (init_cursors) {	/* first time only.. */
		horiz_arrow =
		    XCreateFontCursor(XtDisplay(new),
				      XC_sb_h_double_arrow);
		vert_arrow =
		    XCreateFontCursor(XtDisplay(new),
				      XC_sb_v_double_arrow);
		hv_arrow = XCreateFontCursor(XtDisplay(new), XC_sizing);
		cross_hair = XCreateFontCursor(XtDisplay(new), XC_cross);
		init_cursors = 0;
	}
}

#define f2i(x) ((int)(((x) < 0.0) ? (-0.5 + x) : (0.5 + x)))

static int xscale(CanvasWidget w, float fx, int fixp)
{
	float g;

	if (fixp)
		g = fx * w->core.width;
	else
		g = (fx - w->canvas.vc_xmin) /
		    (w->canvas.vc_xmax -
		     w->canvas.vc_xmin) * w->core.width;
	if (g < 0.0)
		g = 0.0;
	else if (g > w->core.width)
		g = w->core.width;
	return (f2i(g));
}

static int yscale(CanvasWidget w, float fy, int fixp)
{
	float g;

	if (fixp)
		g = w->core.height - fy * w->core.height;
	else
		g = w->core.height - (fy - w->canvas.vc_ymin) /
		    (w->canvas.vc_ymax -
		     w->canvas.vc_ymin) * w->core.height;
	if (g < 0.0)
		g = 0.0;
	else if (g > w->core.height)
		g = w->core.height;
	return (f2i(g));
}

static void scaleObject(CanvasWidget w, CanvasObject * p)
{
	register float g;
	register int i;

	if (p->rescale) {
		switch (p->typecode) {
		case CanvasLine:
			p->u.point2.x2 = xscale(w, p->u.point2.fx2,
						p->
						float_or_fix &
						CanvasFloatX);
			p->u.point2.y2 =
			    yscale(w, p->u.point2.fy2,
				   p->float_or_fix & CanvasFloatY);
		case CanvasPoint:
		case CanvasText:
		case CanvasVerticalText:
			p->x1 =
			    xscale(w, p->fx1,
				   p->float_or_fix & CanvasFloatX);
			p->y1 =
			    yscale(w, p->fy1,
				   p->float_or_fix & CanvasFloatY);
			break;
		case CanvasPenColor:
			/* do nothing */
			break;
		case CanvasPoints:
			for (i = 0; i < p->u.points.npoints; i++) {
				p->u.points.ptdata[i].x =
				    xscale(w, p->u.points.fxvals[i],
					   p->float_or_fix & CanvasFloatX);
				p->u.points.ptdata[i].y =
				    yscale(w, p->u.points.fyvals[i],
					   p->float_or_fix & CanvasFloatY);
			}
			break;
		case CanvasRectangle:
		case CanvasFilledRectangle:
		case CanvasCircle:
		case CanvasFilledCircle:
			p->x1 =
			    xscale(w, p->fx1,
				   p->float_or_fix & CanvasFloatX);
			p->y1 =
			    yscale(w, p->fy1,
				   p->float_or_fix & CanvasFloatY);
			if (p->float_or_fix & CanvasFloatWidth)
				g = p->u.point2.fx2 *
				    (float) w->core.width;
			else
				g = p->u.point2.fx2 /
				    (float) (w->canvas.vc_xmax -
					     w->canvas.vc_xmin) *
				    (float) w->core.width;
			p->u.point2.x2 = f2i(g);
			if (p->float_or_fix & CanvasFloatHeight)
				g = p->u.point2.fy2 * w->core.height;
			else
				g = p->u.point2.fy2 /
				    (float) (w->canvas.vc_ymax -
					     w->canvas.vc_ymin) *
				    (float) w->core.height;
			p->u.point2.y2 = f2i(g);
			break;
		default:
			break;
		}
		p->rescale = 0;
	}
}

static void drawtext(CanvasWidget w, CanvasObject * p, int style)
{
	register unsigned int xc, yc;
	char *tmp, *line;
#ifndef index
	extern char *index();
#endif
#ifndef strtok
	extern char *strtok();
#endif

	if (style & CanvasTextRight)
		xc = p->x1 - p->u.text.width;
	else if (style & CanvasTextHCenter)
		xc = p->x1 - (p->u.text.width / 2);
	else			/* CanvasTextLeft */
		xc = p->x1;

	if (style & CanvasTextTop)
		yc = p->y1 + p->u.text.height;
	else if (style & CanvasTextVCenter)
		yc = p->y1 + (p->u.text.height / 2);
	else			/* CanvasTextBottom */
		yc = p->y1;

	if ((line = index(p->u.text.string, '\n')) == NULL) {
		XPS_DrawString(XtDisplay(w), XtWindow(w), w->canvas.gc, xc,
			       yc, p->u.text.string,
			       strlen(p->u.text.string));
	} else {
		tmp = strcpy(malloc(strlen(p->u.text.string) + 1),
			     p->u.text.string);
		for (line = strtok(tmp, "\n"); line != NULL;
		     line = strtok(NULL, "\n")) {
			XPS_DrawString(XtDisplay(w), XtWindow(w),
				       w->canvas.gc, xc, yc, line,
				       strlen(line));
			yc += 1.1 * p->u.text.height;
		}
		free(tmp);
	}
}

static void drawObject(CanvasWidget w, CanvasObject * p, Region region,
		       int xorit)
{
	register unsigned int xc, yc;	/* ok, they're really set before use! */
	register unsigned int wd, ht;	/* ok, they're really set before use! */
	int width, style, i, use_qcol;
	char buf[2];
	Pixel qcol;		/* also set before use! */

	if (!XtIsRealized(w))
		return;

	scaleObject(w, p);

	region = NULL;		/* this is buggy! */

	if (p->typecode != CanvasPenColor) {
		/*
		 * p->style isn't defined for CanvasPenColor objects!!
		 *  --> there's a serious bug here.. <--
		 */
		if ((style = p->style) < 0 || xorit) {
			XSetFunction(XtDisplay(w), w->canvas.gc, GXxor);
			style = LineSolid;
		} else {
			XSetFunction(XtDisplay(w), w->canvas.gc, GXcopy);
		}
	}

	switch (p->typecode) {
	case CanvasPenColor:
		XSetForeground(XtDisplay(w), w->canvas.gc, p->u.pen);
		w->canvas.the_pen = p->u.pen;
		break;
	case CanvasRectangle:
		xc = p->x1 - p->u.point2.x2 / 2;
		yc = p->y1 - p->u.point2.y2 / 2;
		if (!region || XPointInRegion(region, xc, yc)) {
			XSetLineAttributes(XtDisplay(w), w->canvas.gc,
					   p->line_width, style, CapRound,
					   JoinRound);
			XPS_DrawRectangle(XtDisplay(w), XtWindow(w),
					  w->canvas.gc, xc, yc,
					  p->u.point2.x2, p->u.point2.y2);
		}
		break;
	case CanvasFilledRectangle:
		xc = p->x1 - p->u.point2.x2 / 2;
		yc = p->y1 - p->u.point2.y2 / 2;
		if (!region || XPointInRegion(region, xc, yc)) {
			XSetLineAttributes(XtDisplay(w), w->canvas.gc,
					   p->line_width, style, CapRound,
					   JoinRound);
			XPS_FillRectangle(XtDisplay(w), XtWindow(w),
					  w->canvas.gc, xc, yc,
					  p->u.point2.x2, p->u.point2.y2);
		}
		break;
	case CanvasCircle:
		xc = p->x1 - p->u.point2.x2 / 2;
		yc = p->y1 - p->u.point2.y2 / 2;
		if (!region || XPointInRegion(region, xc, yc)) {
			XSetLineAttributes(XtDisplay(w), w->canvas.gc,
					   p->line_width, style, CapRound,
					   JoinRound);
			XPS_DrawArc(XtDisplay(w), XtWindow(w),
				    w->canvas.gc, xc, yc, p->u.point2.x2,
				    p->u.point2.y2, 0, 360 * 64);
		}
		break;
	case CanvasFilledCircle:
		xc = p->x1 - p->u.point2.x2 / 2;
		yc = p->y1 - p->u.point2.y2 / 2;
		if (!region || XPointInRegion(region, xc, yc)) {
			XSetLineAttributes(XtDisplay(w), w->canvas.gc,
					   p->line_width, style, CapRound,
					   JoinRound);
			XPS_FillArc(XtDisplay(w), XtWindow(w),
				    w->canvas.gc, xc, yc, p->u.point2.x2,
				    p->u.point2.y2, 0, 360 * 64);
		}
		break;
	case CanvasLine:
		if (region) {
			xc = Min(p->x1, p->u.point2.x2);
			yc = Max(p->y1, p->u.point2.y2);
			if ((wd = p->x1 - p->u.point2.x2) == 0)
				wd = 1;
			if ((ht = p->y1 - p->u.point2.y2) == 0)
				ht = 1;
		}
		use_qcol = 0;
		width = use_qcol ? 1 : p->line_width;

		if (!region
		    || XRectInRegion(region, xc, yc, wd,
				     ht) != RectangleOut) {
			if (use_qcol)
				XSetForeground(XtDisplay(w), w->canvas.gc,
					       qcol);
			XSetLineAttributes(XtDisplay(w), w->canvas.gc,
					   width, style, CapRound,
					   JoinRound);
			XPS_DrawLine(XtDisplay(w), XtWindow(w),
				     w->canvas.gc, p->x1, p->y1,
				     p->u.point2.x2, p->u.point2.y2);
			if (use_qcol)
				XSetForeground(XtDisplay(w), w->canvas.gc,
					       w->canvas.the_pen);
		}
		if (w->canvas.mark_lines) {
			if (!region
			    || XPointInRegion(region, p->x1, p->y1)) {
				if (use_qcol)
					XSetForeground(XtDisplay(w),
						       w->canvas.gc, qcol);
				XSetLineAttributes(XtDisplay(w),
						   w->canvas.gc, 0,
						   LineSolid, CapRound,
						   JoinRound);
				XPS_FillRectangle(XtDisplay(w),
						  XtWindow(w),
						  w->canvas.gc, p->x1 - 1,
						  p->y1 - 1, 3, 3);
				if (use_qcol)
					XSetForeground(XtDisplay(w),
						       w->canvas.gc,
						       w->canvas.the_pen);
			}
		}
		break;
	case CanvasPoint:
		if (!region || XPointInRegion(region, p->x1, p->y1)) {
			XSetLineAttributes(XtDisplay(w), w->canvas.gc,
					   0, LineSolid, CapRound,
					   JoinRound);
			XPS_DrawPoint(XtDisplay(w), XtWindow(w),
				      w->canvas.gc, p->x1, p->y1);
		}
		break;
	case CanvasPoints:
		if (p->u.points.connect) {
			XSetLineAttributes(XtDisplay(w), w->canvas.gc,
					   0, LineSolid, CapRound,
					   JoinRound);
			XPS_DrawLines(XtDisplay(w), XtWindow(w),
				      w->canvas.gc, p->u.points.ptdata,
				      p->u.points.npoints,
				      CoordModeOrigin);
		} else {
			for (i = 0; i < p->u.points.npoints; i++) {
				if (!region || XPointInRegion(region,
							      p->u.points.
							      ptdata[i].x,
							      p->u.points.
							      ptdata[i].
							      y)) {
					XPS_DrawPoint(XtDisplay(w),
						      XtWindow(w),
						      w->canvas.gc,
						      p->u.points.
						      ptdata[i].x,
						      p->u.points.
						      ptdata[i].y);
				}
			}
		}
		break;
	case CanvasText:
		drawtext(w, p, style);
		break;
	case CanvasVerticalText:
		if (style & CanvasTextRight)
			xc = p->x1 - p->u.text.width;
		else if (style & CanvasTextHCenter)
			xc = p->x1 - (p->u.text.width / 2);
		else		/* CanvasTextLeft */
			xc = p->x1;

		if (style & CanvasTextTop)
			yc = p->y1 +
			    p->u.text.height * strlen(p->u.text.string);
		else if (style & CanvasTextVCenter)
			yc = p->y1 -
			    (p->u.text.height * strlen(p->u.text.string) /
			     2);
		else		/* CanvasTextBottom */
			yc = p->y1;

		buf[1] = 0;
		for (i = 0; p->u.text.string[i]; i++) {
			buf[0] = p->u.text.string[i];
			XPS_DrawString(XtDisplay(w), XtWindow(w),
				       w->canvas.gc, xc, yc, buf,
				       strlen(buf));
			yc += p->u.text.height;
		}
		break;
	}
}

static void Redisplay(CanvasWidget w, XEvent * event, Region region)
{
	CanvasObject *p;

	if (!XtIsRealized(w))
		return;

	if (w->canvas.lineList) {
		XSetForeground(XtDisplay(w), w->canvas.gc,
			       w->canvas.foreground_pixel);
		for (p = w->canvas.lineList; p->next != NULL; p = p->next);
		while (p) {
			drawObject(w, p, region, False);
			p = p->prev;
		}
		if (w->canvas.vc_depth != 0) {
			char buf[10];
			sprintf(buf, "%d", w->canvas.vc_depth);
			XPS_DrawString(XtDisplay(w), XtWindow(w),
				       w->canvas.gc, 0, w->core.height,
				       buf, strlen(buf));
		}
	}
}

static void Destroy(CanvasWidget w)
{
	register CanvasObject *p, *q;
	static void freeCanvasObject();

	while (popVC(w));
	p = w->canvas.lineList;
	while (p) {
		q = p->next;
		freeCanvasObject(p);
		p = q;
	}
	XFreeFont(XtDisplay(w), w->canvas.fontinfo);
	XtReleaseGC(w, w->canvas.gc);
}

static void Resize(CanvasWidget w)
{
	CanvasObject *p;

	for (p = w->canvas.lineList; p; p = p->next) {
		p->rescale = 1;
	}
}

static Boolean SetValues(CanvasWidget current, CanvasWidget request,
			 CanvasWidget new)
{
	return (False);
}

CanvasClassRec canvasClassRec = {
	{			/* core fields */
	 /* superclass   */ (WidgetClass) & widgetClassRec,
	 /* class_name   */ "Canvas",
	 /* widget_size    */ sizeof(CanvasRec),
	 /* class_initialize   */ NULL,
	 /* class_part_initialize  */ NULL,
	 /* class_inited   */ False,
	 /* initialize   */ Initialize,
	 /* initialize_hook    */ NULL,
	 /* realize      */ XtInheritRealize,
	 /* actions      */ actions,
	 /* num_actions    */ XtNumber(actions),
	 /* resources    */ resources,
	 /* num_resources    */ XtNumber(resources),
	 /* xrm_class    */ NULLQUARK,
	 /* compress_motion    */ True,
	 /* compress_exposure  */ True,
	 /* compress_enterleave  */ True,
	 /* visible_interest   */ False,
	 /* destroy      */ Destroy,
	 /* resize     */ Resize,
	 /* expose     */ Redisplay,
	 /* set_values   */ SetValues,
	 /* set_values_hook    */ NULL,
	 /* set_values_almost  */ XtInheritSetValuesAlmost,
	 /* get_values_hook    */ NULL,
	 /* accept_focus   */ NULL,
	 /* version      */ XtVersion,
	 /* callback_private   */ NULL,
	 /* tm_table     */ default_translations,
	 /* query_geometry   */ XtInheritQueryGeometry,
	 /* display_accelerator  */ XtInheritDisplayAccelerator,
	 /* extension    */ NULL
	 }
	,
	{			/* canvas fields */
	 /* empty      */ 0
	 }
};

WidgetClass canvasWidgetClass = (WidgetClass) & canvasClassRec;

static CanvasObject *newCanvasObject(CanvasWidget w)
{
	CanvasObject *p = (CanvasObject *) malloc(sizeof(CanvasObject));

	/* now insert at the head of the line list */
	p->prev = NULL;
	if (w->canvas.lineList)
		w->canvas.lineList->prev = p;
	p->next = w->canvas.lineList;
	w->canvas.lineList = p;
	return (p);
}

static void freeCanvasObject(CanvasObject * p)
{
	switch (p->typecode) {
	case CanvasText:
	case CanvasVerticalText:
		if (p->u.text.string)
			free(p->u.text.string);
		break;
	case CanvasPoints:
		if (p->u.points.fxvals)
			free(p->u.points.fxvals);
		if (p->u.points.fyvals)
			free(p->u.points.fyvals);
		if (p->u.points.ptdata)
			free(p->u.points.ptdata);
		break;
	default:
		break;
	}
	free(p);
}

CanvasObject *XtCanvasAddPoint(CanvasWidget w, float x1, float y1,
			       int float_or_fix)
{
	CanvasObject *p = newCanvasObject(w);

	p->fx1 = x1;
	p->fy1 = y1;

	p->float_or_fix = float_or_fix;
	p->typecode = CanvasPoint;
	p->line_width = 0;
	p->style = 0;
	p->rescale = 1;

	if (w->canvas.update_on_draw)
		drawObject(w, p, NULL, False);
	return (p);
}

CanvasObject *XtCanvasAddLine(CanvasWidget w, float x1, float y1, float x2,
			      float y2, int lwidth, int lstyle,
			      int float_or_fix)
{
	CanvasObject *p = newCanvasObject(w);

	p->fx1 = x1;
	p->fy1 = y1;
	p->u.point2.fx2 = x2;
	p->u.point2.fy2 = y2;

	p->float_or_fix = float_or_fix;
	p->line_width = lwidth;
	p->style = lstyle;
	p->typecode = CanvasLine;
	p->rescale = 1;

	if (w->canvas.update_on_draw)
		drawObject(w, p, NULL, False);
	return (p);
}

CanvasObject *XtCanvasAddPoints(CanvasWidget w, CanvasData dtype, int n,
				int connect, void *x, int xskip,
				float xoffset, void *y, int yskip,
				float yoffset, int float_or_fix)
{
	CanvasObject *p = newCanvasObject(w);
	register int i;

	int *x_as_int, *y_as_int;
	short *x_as_short, *y_as_short;
	float *x_as_float, *y_as_float;

	p->u.points.fxvals = (float *) malloc(n * sizeof(float));
	p->u.points.fyvals = (float *) malloc(n * sizeof(float));
	p->u.points.ptdata = (XPoint *) malloc(n * sizeof(XPoint));
	p->u.points.npoints = n;
	p->u.points.connect = connect;

	switch (dtype) {
	case CanvasDataFloat:
		x_as_float = (float *) x;
		y_as_float = (float *) y;
		for (i = 0; i < n; i++) {
			if (x)
				p->u.points.fxvals[i] =
				    x_as_float[i * xskip] + xoffset;
			else
				p->u.points.fxvals[i] =
				    (i * xskip) + xoffset;
			if (y)
				p->u.points.fyvals[i] =
				    y_as_float[i * yskip] + yoffset;
			else
				p->u.points.fyvals[i] =
				    (i * yskip) + yoffset;
		}
		break;
	case CanvasDataInt:
		x_as_int = (int *) x;
		y_as_int = (int *) y;
		for (i = 0; i < n; i++) {
			if (x)
				p->u.points.fxvals[i] =
				    x_as_int[i * xskip] + xoffset;
			else
				p->u.points.fxvals[i] =
				    (i * xskip) + xoffset;
			if (y)
				p->u.points.fyvals[i] =
				    y_as_int[i * yskip] + yoffset;
			else
				p->u.points.fyvals[i] =
				    (i * yskip) + yoffset;
		}
		break;
	case CanvasDataShort:
		x_as_short = (short *) x;
		y_as_short = (short *) y;
		for (i = 0; i < n; i++) {
			if (x)
				p->u.points.fxvals[i] =
				    x_as_short[i * xskip] + xoffset;
			else
				p->u.points.fxvals[i] =
				    (i * xskip) + xoffset;
			if (y)
				p->u.points.fyvals[i] =
				    y_as_short[i * yskip] + yoffset;
			else
				p->u.points.fyvals[i] =
				    (i * yskip) + yoffset;
		}
		break;
	}

	p->float_or_fix = float_or_fix;
	p->typecode = CanvasPoints;
	p->line_width = 0;
	p->style = 0;
	p->rescale = 1;

	if (w->canvas.update_on_draw)
		drawObject(w, p, NULL, False);
	return (p);
}

CanvasObject *XtCanvasAddText(CanvasWidget w, float x1, float y1,
			      char *text, int style, int float_or_fix)
{
	CanvasObject *p = newCanvasObject(w);

	p->fx1 = x1;
	p->fy1 = y1;
	p->u.text.string = (char *) malloc(strlen(text) + 1);
	strcpy(p->u.text.string, text);
	p->u.text.width =
	    XTextWidth(w->canvas.fontinfo, text, strlen(text));
	p->u.text.height = w->canvas.fontinfo->ascent;

	p->float_or_fix = float_or_fix;
	p->typecode = CanvasText;
	p->style = style;
	p->rescale = 1;
	if (w->canvas.update_on_draw)
		drawObject(w, p, NULL, False);
	return (p);
}

CanvasObject *XtCanvasAddVerticalText(CanvasWidget w, float x1, float y1,
				      char *text, int style,
				      int float_or_fix)
{
	CanvasObject *p = newCanvasObject(w);

	p->fx1 = x1;
	p->fy1 = y1;
	p->u.text.string = (char *) malloc(strlen(text) + 1);
	strcpy(p->u.text.string, text);
	p->u.text.width = XTextWidth(w->canvas.fontinfo, "X", 1);
	p->u.text.height =
	    w->canvas.fontinfo->ascent + w->canvas.fontinfo->descent + 1;

	p->float_or_fix = float_or_fix;
	p->typecode = CanvasVerticalText;
	p->style = style;
	p->rescale = 1;

	if (w->canvas.update_on_draw)
		drawObject(w, p, NULL, False);
	return (p);
}

CanvasObject *XtCanvasAddRectangle(CanvasWidget w, float x1, float y1,
				   float width, float height, int lwidth,
				   int lstyle, int float_or_fix)
{
	CanvasObject *p = newCanvasObject(w);

	p->fx1 = x1;
	p->fy1 = y1;
	p->u.point2.fx2 = width;
	p->u.point2.fy2 = height;

	p->float_or_fix = float_or_fix;
	p->typecode = CanvasRectangle;
	p->line_width = lwidth;
	p->style = lstyle;
	p->rescale = 1;

	if (w->canvas.update_on_draw)
		drawObject(w, p, NULL, False);
	return (p);
}

CanvasObject *XtCanvasAddFilledRectangle(CanvasWidget w, float x1,
					 float y1, float width,
					 float height, int lwidth,
					 int lstyle, int float_or_fix)
{

	CanvasObject *p = newCanvasObject(w);

	p->fx1 = x1;
	p->fy1 = y1;
	p->u.point2.fx2 = width;
	p->u.point2.fy2 = height;

	p->float_or_fix = float_or_fix;
	p->typecode = CanvasFilledRectangle;
	p->line_width = lwidth;
	p->style = lstyle;
	p->rescale = 1;

	if (w->canvas.update_on_draw)
		drawObject(w, p, NULL, False);
	return (p);
}

CanvasObject *XtCanvasAddCircle(CanvasWidget w, float x1, float y1,
				float width, float height, int lwidth,
				int lstyle, int float_or_fix)
{
	CanvasObject *p = newCanvasObject(w);

	p->fx1 = x1;
	p->fy1 = y1;
	p->u.point2.fx2 = width;
	p->u.point2.fy2 = height;

	p->float_or_fix = float_or_fix;
	p->typecode = CanvasCircle;
	p->line_width = lwidth;
	p->style = lstyle;
	p->rescale = 1;

	if (w->canvas.update_on_draw)
		drawObject(w, p, NULL, False);
	return (p);
}

CanvasObject *XtCanvasAddFilledCircle(CanvasWidget w, float x1, float y1,
				      float width, float height,
				      int lwidth, int lstyle,
				      int float_or_fix)
{
	CanvasObject *p = newCanvasObject(w);

	p->fx1 = x1;
	p->fy1 = y1;
	p->u.point2.fx2 = width;
	p->u.point2.fy2 = height;

	p->float_or_fix = float_or_fix;
	p->typecode = CanvasFilledCircle;
	p->line_width = lwidth;
	p->style = lstyle;
	p->rescale = 1;

	if (w->canvas.update_on_draw)
		drawObject(w, p, NULL, False);
	return (p);
}

CanvasObject *XtCanvasPenColor(CanvasWidget w, Pixel pen)
{
	CanvasObject *p = newCanvasObject(w);

	p->u.pen = pen;

	p->typecode = CanvasPenColor;
	p->rescale = 0;

	if (w->canvas.update_on_draw)
		drawObject(w, p, NULL, False);
	return (p);
}


int XtCanvasDeleteObject(CanvasWidget w, CanvasObject * objectid)
{
	CanvasObject *p;
	Pixel saved_fg;

	for (p = w->canvas.lineList; p != NULL; p = p->next) {
		if (p == objectid) {
			if (p == w->canvas.lineList) {
				w->canvas.lineList = p->next;
				if (p->next)
					p->next->prev = NULL;
			} else {
				if (p->prev)
					p->prev->next = p->next;
				if (p->next)
					p->next->prev = p->prev;
			}
			if (p->typecode != CanvasPenColor) {	/* this is NOT perfect! */
				if (p->style >= 0) {	/* regular object style */
					saved_fg =
					    w->canvas.foreground_pixel;
					XSetForeground(XtDisplay(w),
						       w->canvas.gc,
						       w->core.
						       background_pixel);
					drawObject(w, p, NULL, False);
					XSetForeground(XtDisplay(w),
						       w->canvas.gc,
						       saved_fg);
				} else {	/* XOR type object */
					drawObject(w, p, NULL, True);
				}
			}
			freeCanvasObject(p);
			return (1);
		}
	}
	return (0);
}

void XtCanvasClear(CanvasWidget w, int clear_coords)
{
	CanvasObject *p, *q;

	if (clear_coords) {
		while (popVC(w));
		w->canvas.vc_xmin = w->canvas.wc_xmin;
		w->canvas.vc_xmax = w->canvas.wc_xmax;
		w->canvas.vc_ymin = w->canvas.wc_ymin;
		w->canvas.vc_ymax = w->canvas.wc_ymax;
	}
	for (p = w->canvas.lineList; p; p = q) {
		q = p->next;
		freeCanvasObject(p);
	}
	w->canvas.lineList = NULL;
	if (XtIsRealized(w))
		XClearWindow(XtDisplay(w), XtWindow(w));
	w->canvas.erase_rubber = False;
}

void XtCanvasPS(CanvasWidget w, char *destination, char *titlestring)
{
	if (destination != NULL) {
		XPS_SetFileOutput(1);	/* 1=file, 0=printer */
		XPS_SetFilename(destination);
	} else {
		XPS_SetFileOutput(0);	/* 1=file, 0=printer */
	}

	XPS_PreparePS(XtDisplay(w), XtWindow(w), (double) 0.80, 0,
		      titlestring, 0);
	Redisplay(w, NULL, NULL);
	XPS_FinishPS();
}

void XtCanvasSetWC(CanvasWidget w, float xmin, float xmax, float ymin,
		   float ymax)
{
	if ((w = (CanvasWidget) XtCanvasFind(w))) {
		w->canvas.wc_xmin = xmin;
		w->canvas.wc_xmax = xmax;

		w->canvas.wc_ymin = ymin;
		w->canvas.wc_ymax = ymax;

		if (XtIsRealized(w)) {
			XClearWindow(XtDisplay(w), XtWindow(w));
			w->canvas.erase_rubber = False;
			Resize(w);
			Redisplay(w, NULL, NULL);
		}
	} else {
		XtAppWarning(XtWidgetToApplicationContext(w),
			     "XtCanvasSetWC: can't locate canvas");
	}
}

void XtCanvasGetWC(CanvasWidget w, float *xmin, float *xmax, float *ymin,
		   float *ymax)
{
	if ((w = (CanvasWidget) XtCanvasFind(w))) {
		*xmin = w->canvas.wc_xmin;
		*xmax = w->canvas.wc_xmax;

		*ymin = w->canvas.wc_ymin;
		*ymax = w->canvas.wc_ymax;
	} else {
		XtAppWarning(XtWidgetToApplicationContext(w),
			     "XtCanvasGetWC: can't locate canvas");
	}
}

void XtCanvasClearVC(CanvasWidget w)
{
	int i = 0;

	while (popVC(w))
		i++;
	if (i && XtIsRealized(w)) {
		XClearWindow(XtDisplay(w), XtWindow(w));
		w->canvas.erase_rubber = False;
		Resize(w);
		Redisplay(w, NULL, NULL);
	}
}

void XtCanvasUpdate(CanvasWidget w)
{
	if (XtIsRealized(w))
		Redisplay(w, NULL, NULL);
}

void XtCanvasSetVC(CanvasWidget w, float xmin, float xmax, float ymin,
		   float ymax)
{
	if ((w = (CanvasWidget) XtCanvasFind(w))) {
		if (xmin != xmax) {
			w->canvas.vc_xmin = xmin;
			w->canvas.vc_xmax = xmax;
		}
		if (ymin != ymax) {
			w->canvas.vc_ymin = ymin;
			w->canvas.vc_ymax = ymax;
		}
		if (XtIsRealized(w)) {
			XClearWindow(XtDisplay(w), XtWindow(w));
			w->canvas.erase_rubber = False;
			Resize(w);
			Redisplay(w, NULL, NULL);
		}
	} else {
		XtAppWarning(XtWidgetToApplicationContext(w),
			     "XtCanvasSetVC: can't locate canvas");
	}
}

void XtCanvasGetVC(CanvasWidget w, float *xmin, float *xmax, float *ymin,
		   float *ymax)
{
	if ((w = (CanvasWidget) XtCanvasFind(w))) {
		*xmin = w->canvas.vc_xmin;
		*xmax = w->canvas.vc_xmax;

		*ymin = w->canvas.vc_ymin;
		*ymax = w->canvas.vc_ymax;
	} else {
		XtAppWarning(XtWidgetToApplicationContext(w),
			     "XtCanvasGetVC: can't locate canvas");
	}
}

Widget XtCanvasFind(Widget w)
{
	if (XtClass(w) != canvasWidgetClass) {
		XtAppWarning(XtWidgetToApplicationContext(w),
			     "XtFindCanvas: passed a non-canvas");
		return (XtNameToWidget(w, "canvas"));
	} else {
		return (w);
	}
}

void XtCanvasXYtoWC(CanvasWidget w, int x, int y, float *wc_x, float *wc_y)
{
	if ((w = (CanvasWidget) XtCanvasFind(w))) {
		*wc_x =
		    (float) x *(w->canvas.vc_xmax -
				w->canvas.vc_xmin) /
		    (float) w->core.width + w->canvas.vc_xmin;
		*wc_y =
		    (float) (w->core.height - y) * (w->canvas.vc_ymax -
						    w->canvas.vc_ymin) /
		    (float) w->core.height + w->canvas.vc_ymin;
	} else {
		XtAppWarning(XtWidgetToApplicationContext(w),
			     "XtCanvasXYtoWC: can't locate canvas");
	}
}

int XtCanvasUserView(CanvasWidget w)
{
	return (w->canvas.vc_stack != NULL);
}


int XtCanvasXhair(CanvasWidget w, float *xp, float *yp, int *pxp, int *pyp,
		  int clearit)
{
	if (w->canvas.xhair_set) {
		if (xp != NULL)
			*xp = w->canvas.xhair_x;
		if (yp != NULL)
			*yp = w->canvas.xhair_y;
		if (pxp != NULL)
			*pxp = w->canvas.xhair_px;
		if (pyp != NULL)
			*pyp = w->canvas.xhair_py;
		if (clearit)
			w->canvas.xhair_set = 0;
		return (1);
	} else {
		return (0);
	}
}

CanvasObject *XtCanvasNearestObject(CanvasWidget w, int x, int y, int type)
{

	int d, dmin;
	CanvasObject *p, *closest;

	closest = NULL;
	for (p = w->canvas.lineList; p; p = p->next) {
		if (type && type != p->typecode)
			continue;
		d = ((x - p->x1) * (x - p->x1)) +
		    ((y - p->y1) * (y - p->y1));
		if (closest == NULL) {
			closest = p;
			dmin = d;
		} else if (d < dmin) {
			closest = p;
			dmin = d;
		}
	}
	return (closest);
}

void XtCanvasToggleFill(CanvasWidget w, CanvasObject * p)
{
	switch (p->typecode) {
	case CanvasFilledRectangle:
		p->typecode = CanvasRectangle;
		Redisplay(w, NULL, NULL);
		break;
	case CanvasRectangle:
		p->typecode = CanvasFilledRectangle;
		Redisplay(w, NULL, NULL);
		break;
	case CanvasFilledCircle:
		p->typecode = CanvasCircle;
		Redisplay(w, NULL, NULL);
		break;
	case CanvasCircle:
		p->typecode = CanvasFilledCircle;
		Redisplay(w, NULL, NULL);
		break;
	default:
		break;
	}
}

void XtCanvasSetMsg(CanvasWidget w, char *msg)
{
	Dimension wd;

	if (msg) {
		wd = 10 + XTextWidth(w->canvas.fontinfo, msg, strlen(msg));
		if (wd > w->core.width)
			wd = w->core.width;
		if (w->canvas.msgwin == NULL) {
			w->canvas.msgwin =
			    XCreateSimpleWindow(XtDisplay(w), XtWindow(w),
						1, 1, wd,
						w->canvas.fontinfo->
						ascent +
						w->canvas.fontinfo->
						descent + 5, 1, 0, 0);
			/* 0,0 weren't they're initially */
			XSetWindowBackground(XtDisplay(w),
					     w->canvas.msgwin,
					     w->core.background_pixel);
			XMapWindow(XtDisplay(w), w->canvas.msgwin);
			XRaiseWindow(XtDisplay(w), w->canvas.msgwin);
		} else {
			XResizeWindow(XtDisplay(w), w->canvas.msgwin,
				      wd, w->canvas.fontinfo->ascent +
				      w->canvas.fontinfo->descent + 5);
		}
		XClearWindow(XtDisplay(w), w->canvas.msgwin);
		XDrawString(XtDisplay(w), w->canvas.msgwin, w->canvas.gc,
			    3,
			    w->canvas.fontinfo->ascent +
			    w->canvas.fontinfo->descent, msg, strlen(msg));
	} else if (w->canvas.msgwin != NULL) {
		XDestroyWindow(XtDisplay(w), w->canvas.msgwin);
		w->canvas.msgwin = (Window) NULL;
	}
}

static char *getafilename(char *startwith, char *prompt)
{
	char *name, *fileBox();

	if ((name = fileBox(startwith, "a", prompt)) != NULL) {
		if (ishiddenfile(name) &&
		    pop_box("Write to hidden file?", "No", "Yes",
			    NULL) == 1) {
			free(name);
			return (NULL);
		} else {
			return (name);
		}
	} else {
		return (NULL);
	}
}

static void GeneratePSAction(CanvasWidget w, XEvent * event,
			     String * params, Cardinal * num_params)
{
	char *name;

	if (*num_params > 0 && strcasecmp(params[0], "file") == 0) {
		if ((name =
		     getafilename(".ps", "Save PostScript to:")) != NULL) {
			XtCanvasPS(w, name, NULL);
			free(name);
		}
	} else {
		XtCanvasPS(w, NULL, name);
	}
}

static void GenerateFIGAction(Widget w, XEvent * event, String * params,
			      Cardinal * num_params)
{
	char *name;
	WidgetClass class;

	class = XtClass(w);
	if (*num_params > 0 && strcasecmp(params[0], "file") == 0) {
		if ((name = getafilename(".fig", "Save Fig to:")) == NULL) {
			return;
		} else {
			XPS_SetFilename(name);
		}
	} else {
		unlink(findhome("~/.xfig"));
		XPS_SetFilename(findhome("~/.xfig"));
	}
	XtUnmapWidget(w);
	XPS_PrepareFIG(XtDisplay(w), XtWindow(w));
	XtMapWidget(w);
	dloop_empty();
	XPS_FinishFIG();
	free(name);
	XtUnmapWidget(w);
	XtMapWidget(w);
}
