function showtraces (filename, depvar, epoch, samp_rate, title_str, opt)
% USAGE: SHOWTRACES('FILENAME', DEPVAR, EPOCH, SAMP_RATE, 'TITLE');
%     or   SHOWTRACES('FILENAME', DEPVAR, EPOCH, SAMP_RATE, 'TITLE', WHICH);
%
%  SHOWTRACES plots all analog traces from xdphys file FILENAME for which
%  the depvar was DEPVAR.
%
%  Parameters:
%
%     FILENAME:    the filename of either a new xdphys file (a text file which 
%                  contains both analog traces and spike times) or an old 
%                  xdphys 'ana' file (a binary file which contains only the
%                  analog traces)
%     DEPVAR:      the value of the depvar you are interested in, for example:
%                  if it's an ITD file, and you're interested in -30us, use
%                  DEPVAR=-30
%     EPOCH:       length of the epoch in milliseconds (this is almost always
%                  400)
%     SAMP_RATE:   sampling rate for the analog traces: if you are unsure,
%                  look in the xdphys file for 'ana_decimate', and then
%                  compute SAMP_RATE=48/ana_decimate.
%     TITLE:       A title for the topmost graph.
%     WHICH:       (optional) Plot only traces whose indices are contained in
%                  this vector.
%
%
%  Examples:
%
%   o Suppose there is a file, 123.01.01.itd, in the current directory,
%     and we want to look at the traces for itd=0.  The epoch for that
%     trial was 400ms and ana_decimate was 2.  We want to title the graph
%     "123.01.01.itd, 0 us".  The appropriate call for SHOWTRACES is then:
%
%       showtraces('123.01.01.itd',0,400,24,'123.01.01.itd, 0 us');
%
%   o Suppose we have the same situation as above, but we only want to plot
%     the third trace.  The appropriate call for SHOWTRACES is then:
%
%       showtraces('123.01.01.itd',0,400,24,'123.01.01.itd, 0 us',3);
%
%     and if we wanted traces 2 and 3, it would be
%
%       showtraces('123.01.01.itd',0,400,24,'123.01.01.itd, 0 us',[2 3]);
%
%
%  SEE ALSO: READANA

% Read in the data for depvar 'depvar'
d=readana(filename, depvar);
data=d.traces;
data=(data/10);
if (nargin==6),
	data=data(:,opt);
end
trace_len=size(data,1);

time = 1:1:trace_len;
time = time*1/samp_rate;
nreps = size(data,2);
nplots = nreps;
clf;
for i=1:nreps
	ymax = max(data(:,i));
	ymin = min(data(:,i));
	ax(i)=subplot(nplots,1,i); 
	plot(time,data(:,i));
	ylim([ymin ymax]);
	xlim([0 epoch]);
	ylabel(sprintf('Trace %d', i));
	% axis off;
end

subplot(nplots,1,1);
title(title_str);
subplot(nplots,1,nplots); 
xlabel('Time (ms)','FontSize',12);

set(ax,'box','on');
