function data=readana(filename, depvar)
% USAGE: DATA = READANA('FILENAME');
%        DATA = READANA('FILENAME', DEPVAR);
%
%   READANA reads analog traces from an xdphys data file.  It can read
%   both the old style .ana files, and the newer "gen" style files. 
%
%   DATA is an array of structs, where N is the number of different
%   depvars.  Each struct has the following structure:
%
%        data.depint:    depvar 
%        data.traces:    MxN matrix of analog traces for data.depint 
%   
%   Each trace in data.traces M=(samp_freq * epoch/1000) samples long,
%	and is in mV.
%
%   If the parameter DEPVAR is supplied, READANA only reads those traces
%   whose depvar equals DEPVAR (and DATA is 1x1).  If DEPVAR is not supplied, 
%	all traces in the file are read (and DATA is 1xN, N=(# depvars)).
%
%	DATA is sorted in order of increasing depvar.
%
%   NEW FILES:
%   ---------
%
%       If you used the new xdphys with the gen module to save your data,
%       your analog data is stored in the .gen/.itd/.abi/etc. file.  Simply
%       use READANA on that file to extract the analog data.
%
%   COMPRESSED FILES: 
%   ------------------
%
%        READANA can now handle compressed files; just supply the filename 
%        of the compressed file, and READANA will uncompress it temporarily.
%
% SEE ALSO: XDVIEW_CURVE, XDVIEW_GETCAL

% NOTES:
%
% Two kinds of analog files: pre-gen module and post-gen module.
%
%  An old style header struct looks like, for dowl, and pre-gen module xdphys:
%
%   int        magic   ( = 0x5731 )
%   int        fc
%   int        nsamps
%   int        nchans
%   char[100]  comment
%   int        bits
%   char[20]   program
%   char[20]   arch
%   int        tomv
%   int        tomv_div
%   int        offset
%   int        offset_div
%   char[336]  padding
%
%  It is 512 bytes long, and is defined in xdphys/src/waveio.h
%  xdphys is probably compiled to align structs along 4 byte boundries,
%  so types with size < 4 get converted to 4 bytes in the header. This 
%  header preceeds each trace.  Old style ana files are always written in
%  big endian format.
%
%  A post-gen, file version 3 xdphys header looks like:
%
%	the file that ana() writes to stdout looks like this:
%
%	Header:
%		int magic				 ( = ANA_MAJIK (see waveio.h))
%		char adFc[15]            ( = "adFc=%d" in Hz)
%		char epoch[15]           ( = "epoch=%d in ms)
%		char decimate[15]        ( = "ana-decimate=%d", boolean (1/0))
%		char nrasters[15]         ( = "nrasters=%d" )
%		char nchans[15]          ( = "nchans=%d" )
%	Conversion block: 1 per channel:
%		char tomv[30][4]          ( = "ana-tomv=%f"  in mV/tick)
%		char offset[30][4]        ( = "ana-offset=%f" in mV)
%	Trace:  (X nrasters)
%		char depvar[30]            ( = "depvar=%s" )
%		char rasternum[30]         ( = "rasternum=%d" )
%		char chan_id[30]           ( = "channel=%d" )
%		xword DATA               ((adFc*epoch)/1000)/decimate) samples
%
%  This header appears at the beginning of the file only.  New style ana files
%  are written in whatever was the default endianness for the machine on
%  which it was created.
%

if(nargin<2)
  depvar = 'all';
end

retval=dir(filename);
if (size(retval) == [0 1]),
	error(sprintf('readana: %s does not exist', filename));
end

% if the file is compressed, uncompress it
newfname=uncompress(filename);

[t, fid]=file_type(newfname,depvar);

switch t
	case 0,
		[d, depints, rasternums]=read_old_style(fid, depvar);
	case 1,
		[d, depints, rasternums]=read_new_style(fid, depvar);
	case 2,
		[d, depints, rasternums]=read_new_style(fid, 'all');
end
data=make_struct(d,depints,rasternums);

% clean up uncompressed temporary file
if (isempty(strmatch(newfname,filename))),
	[s,w] = unix(['rm ' newfname]);
end
	


% ---------------------------------------------------
function data=make_struct(d,depints,rasternums)

% make_struct turns the matrix D and the vector DEPINTS into a vector
%   of structs DATA.  Each struct in DATA has the following form:
%
%        data.depint:    depvar 
%        data.traces:    MxN matrix of analog traces for data.depint 
%   
% make_struct assumes that d and depints are sorted.

i=0;
rep=1;
last_depint=-9999;
for j=1:length(depints),
	if (depints(j) ~= last_depint)
		i=i+1;
		rep=1;
		data(i).depint=depints(j);
	end
	data(i).traces(:,rep)=d(:,j);
	data(i).rasternums(rep)=rasternums(j);
	rep=rep+1;
	last_depint=depints(j);
end
	
% ---------------------------------------------------
function [newfname]=uncompress(filename);

%  Uncompresses the file into a temporary file, if necessary
tmpdir=getenv('TMPDIR');
if isempty(tmpdir)
	tmpdir='.';
end

newfname=filename;
if ((~isempty(findstr('.Z',filename))) | (~isempty(findstr('.gz',filename)))),
	newfname=sprintf('%s/tmp.xdphysfile',tmpdir);
	disp(['readana: uncompressing to ' newfname ' ...']);
	[s,w] = unix(['zcat ' filename ' > ' newfname]);
end

	

% ---------------------------------------------------
function [data, depints, rasternums]=read_old_style(fid, depvar);

% read_old_style expects that the first magic has been read when it
% begins.

count=1;
trace_num=1;
while (count == 1),
	fc = fread(fid,1,'int');
	nsamps = fread(fid,1,'int');
	nchans = fread(fid,1,'int');
	temp=sscanf(setstr(fread(fid,100,'char')'),'depvar=%d');
	if (strcmp(depvar,'all') | (depvar == temp))
		dp(trace_num)=temp;
		pad = fread(fid,44,'char');
		to_mv = fread(fid,1,'int');
		to_mv_div = fread(fid,1,'int');
		offset = fread(fid,1,'int');
		offset_div = fread(fid,1,'int');
		pad = fread(fid,32,'char')';
		rasternums(trace_num) = fread(fid,1,'int');
		pad = fread(fid,300,'char')';  % was 336

		trace=fread(fid,nsamps*nchans,'short');

		data(:,trace_num) = (trace - (offset/offset_div))*(to_mv/to_mv_div);
		disp(sprintf('Read trace for depvar=%d', dp(trace_num)));
		trace_num = trace_num + 1;
	else
		pad=fread(fid,198+(nsamps*nchans),'short');
	end;
	[magic, count] = fread(fid,1,'int');
end;

% now sort the data and depints array
[depints, i]=sort(dp);
rasternums=rasternums(:,i);
data=data(:,i);

% ---------------------------------------------------
function [data, depints, rasternums]=read_new_style(fid, depvar);

% read_new_style expects that magic has been read when it
% begins.

% read_new_style reads file format 3 xdphys files.

head.adFc=sscanf(setstr(fread(fid,15,'char')'),'adFc=%d');
head.epoch=sscanf(setstr(fread(fid,15,'char')'),'epoch=%d');
head.decimate=sscanf(setstr(fread(fid,15,'char')'),'ana-decimate=%d');
head.nrasters=sscanf(setstr(fread(fid,15,'char')'),'nrasters=%d');
head.nchans=sscanf(setstr(fread(fid,15,'char')'),'nchans=%d');
head.tomv(1:4)=0;
head.offset(1:4)=0;
for i=1:head.nchans,
	channel=sscanf(setstr(fread(fid,30,'char')'),'channel=%d');
	head.tomv(channel)=sscanf(setstr(fread(fid,30,'char')'),'ana-tomv.%s');
	head.offset(channel)=sscanf(setstr(fread(fid,30,'char')'),'ana-offset.%s');
end

disp(sprintf('readana: Reading %d channels.', head.nchans));
disp(sprintf('readana: Reading %d traces/channel.', head.nrasters));

nsamps=((head.adFc*head.epoch)/1000)/head.decimate;

trace_num=1;
max=head.nrasters;
for i = 1:head.nchans,
	for n = 1:max,
		depint=sscanf(setstr(fread(fid,30,'char')'),'depvar=%d');
		rasternum=sscanf(setstr(fread(fid,30,'char')'),'rasternum=%d');
		chan_id=sscanf(setstr(fread(fid,30,'char')'),'chan_id=%d');
		trace=fread(fid,nsamps,'short');
		if (strcmp(depvar,'all') | (depint == depvar))
			depints(trace_num)=depint;
			rasternums(trace_num)=rasternum;
			data(:,i,trace_num) = (trace*head.tomv(chan_id))+head.offset(chan_id);
			disp(sprintf('readana: Read trace for depvar=%d', depints(trace_num)));
			trace_num=trace_num+1;
		end
	end
end


% ---------------------------------------------------
function [t, fid]=file_type(filename,depvar)
%
%  t=0   if file is old style file
%  t=1   if file is new style file
%  t=2   if file is new style file, but we created it via xdview


% First, try to open it as an old style ana file
% Old style ana files were in big-endian format.

[magic, fid]=get_magic(filename, 'b');

disp(' ');
if (bitand(magic,hex2dec('ffff0000')) == hex2dec('57310000')),
	% this is an old style file
	disp(['readana: ' filename ' is an old style file.']);
	t=0;
else
	if (magic == hex2dec('5733')),
		% this is an new style file, written on a big endian machine
		disp(['readana: ' filename ' is output from "xdview -ana", big endian.']);
		t=1;
	else
		% try to reopen it as little endian and check the magic
		fclose(fid);
		[magic, fid]=get_magic(filename, 'l');
		if (magic == hex2dec('5733')),
			disp(['readana: ' filename ' is output from "xdview -ana", little endian.']);
			t=1;
		else
			% see if it's an xdphys text file
			if ((bitand(magic,hex2dec('ffff'))) == hex2dec('3b3b')) % = ';;'
				% this may be an xdphys text file, let's check
				frewind(fid);
				progname = fread(fid,9,'char');
				if ~strcmp(progname, ';; xdphys'),
					disp(['readana: ' filename ' is an xdphys text file.']);
					% it _is_ an xdphys text file, try to extract
					% analog data from it
					fid=generate_ana_file(filename,depvar);
					t=2;
				end
			end
		end
	end
end
disp(' ');


% ---------------------------------------------------
function [magic, fid]=get_magic(filename, opt)

fid=fopen(filename, 'r', opt);
if fid == (-1),
	error(['Unable to open file ', filename]);
end;

[magic,count] = fread(fid,1,'int');

% ---------------------------------------------------
function fid=generate_ana_file(filename,depvar)

tmpdir=getenv('TMPDIR');
if isempty(tmpdir)
	tmpdir='.';
end

[s,w] = unix(['rm -f ' tmpdir '/tmp.anafile']);


if (ischar(depvar)) 
	cmd=sprintf('xdview -ana %s %s > %s/tmp.anafile',depvar,filename,tmpdir);
else
	cmd=sprintf('xdview -ana %d %s > %s/tmp.anafile',depvar,filename,tmpdir);
end
disp('readana: Executing the following command to extract analog data:');
disp(' ');
disp(cmd);

[s,w] = unix(cmd);

if (s == 0)
	[magic, fid]=get_magic([tmpdir '/tmp.anafile'],'n');
else
	error('readana: The xdview command did not complete successfully.');	
end

