function [p,sig,tt,h,wr]=itdscatter2(alpha, varargin);
% USAGE: [P,SIG,TT,H,WR]=ITDSCATTER2(ALPHA, FILENAME1, ... , FILENAMEn)
%
%   ITDSCATTER extracts the spike rates for each trial over a range of depvars 
%   from a set of xdphys data files, FILENAME1 to FILENAMEn, and plots the 
%   spike rates of FILENAME2 through FILENAMEn (trial by trial) against 
%   those from FILENAME1.
%
%   The blue line is the average of NREPS lines fit to the NREPS repetitions
%   of (nspikes_FILENAME1,nspikes_FILENAME1) by the least squares method.  
%   Likewise, the other lines are the average of nreps lines fit to the nreps 
%   repetitions of (nspikes_FILENAM1,nspikes_FILENAMEn) by the least squares 
%   method.
%
%   PARAMETERS:
%
%       ALPHA is the desired level of significance for the t-tests and 
%         the Wilcoxon rank test, and must be between 0 and 1 (a good 
%         value is 0.05).
%
%       FILENAME1, FILENAME2, etc.  are xdphys data files (*.itd, *.abi, etc.)
%
%   RETURN VALUES:
% 
%       P contains the parameters of the equation of the non blue lines:
%
%            Y = P(1)*X + P(2)
%
%       SIG contains the variance estimates for the parameters in P.
%
%       TT contains the t-test results (see below) for the parameters in P.
%
%       H contains the t-test reccommendations (see below).
%
%       WR contains the Wilcoxon rank test results (see below) for the
%          parameters in P.
%
%   VALIDITY TESTING:
%   ----------------
%
%     ITDSCATTER performs both t-tests and Wilcoxon ranks tests on the 
%     line coordinates.  
%
%     T-TESTS:
%
%       For the y-intercept, the null hypothesis is that the y-intercept of 
%       the pink line is from a distribution whose is mean equal to that of 
%       the distribution of the y-intercept of the blue line.
%
%       tt(2,:) shows the probability of observing the y-intercept of the
%       non-blue lines by chance given that the null hypothesis is true. 
%
%       if h(2,:) is 0, the blue and non-blue lines y-intercepts are 
%       from the same distribution, with significance level ALPHA.  
%       If 1, they are not from the same distribution.
%
%       In the second, the null hypothesis is that the slope of the
%       non-blue lines is from a distribution whose mean is equal to that of 
%       the distribution of the slope of the blue line.
%
%       tt(1,:) shows the probability of observing the slope of the non-blue 
%       lines by chance given that the null hypothesis is true. 
%
%       if h(1,:) is 0, the blue and non-blue lines slopes are 
%       from the same distribution, with significance level ALPHA.  
%       If 1, they are not from the same distribution.
%   
%       Small values of tt(1,:) and tt(2,:) cast doubt on the validity 
%       of the null hypotheses.
%
%     WILCOXON RANK TESTS:
% 
%       The Wilcoxon rank test is a non-parametric test which determines
%       the probability that two sets of observations are from the same
%       distribution.  
%
%       wr(2,:) shows the probability that the y-intercept of the non-blue
%       lines is from the same distribution as that of the blue line.
%
%       wr(1,:) shows the probability that the y-intercept of the non-blue
%       lines is from the same distribution as that of the blue line.
%
%       Values of wr(2,:) and wr(1,:) close to zero suggest that the 
%       distributions are not the same.
%
%   Example:
%
%      [p,sig,tt,wr] = itdscatter2(0.05,'123.01.01.itd','123.01.08.itd',
%                                  '123.01.12.itd');
%
% SEE ALSO: ITDSCATTER


[pre_nspikes, pre_nreps]=get_nspikes(char(varargin(1)));

%points(:,1)=reshape(pre_nspikes,1,prod(size(pre_nspikes)))';

% first find the slope/intercepts for pre vs. pre
count=1;
points=[];
for i=1:pre_nreps,
	for j=1:pre_nreps,
		p_pre(:,count)=polyfit(pre_nspikes(:,i),pre_nspikes(:,i),1)';
		points=[points' pre_nspikes(:,i)']';
		count=count+1;
	end;
end;

p(:,1)=mean(p_pre')';
sig(:,1)=std(p_pre')';
h(:,1)=[0 0]';

for k=2:length(varargin),
	[points(:,k),p(:,k),sig(:,k),tt(:,k),h(:,k),wr(:,k)]=trend(char(varargin(k)),pre_nreps,pre_nspikes,p_pre,alpha);
end


x=[min(min(pre_nspikes)) max(max(pre_nspikes))];

symbols={'bo' 'md' 'gs' 'rd' 'mv' 'g^' 'rh'};
lines={'b-' 'm-' 'g-' 'r-' 'm-.' 'g-.' 'r-.'};

% Plot the graph
cla
xlabel(sprintf('impulses/sec (%s)',char(varargin(1))));
ylabel('impulses/sec');
hold on

for i=1:length(varargin),
	plot(points(:,1),points(:,i),char(symbols(i)));
	y(:,i)=polyval(p(:,i),x)';
	plot(x,y(:,i),char(lines(i)));
end	
legend(cat(1,varargin,varargin),2);

p=p(:,2:length(varargin));
sig=sig(:,2:length(varargin));
tt=tt(:,2:length(varargin));
h=h(:,2:length(varargin));
wr=wr(:,2:length(varargin));

% ---------------------------------------------
function [y,nreps]=get_nspikes(filename)

% first, figure out how many reps we have (ugly, I know)
[s,w] = unix(['xdview -curve -text ' filename]);
t = findstr(w,'Reps=');
tt = findstr(w(t(1)+5:end),char(10));
nreps = sscanf(w(t(1)+5:t(1)+5+tt(1)),'%f');

[s,w] = unix(['xdview -nspikes ' filename]);
y = sscanf(w,'%f');
y = reshape(y,nreps,length(y)/nreps)';
% convert to impulses/sec (we're in impulses/100ms)
y = y*10;


% ---------------------------------------------
function [points,p,sig,tt,h,wr]=trend(filename,pre_nreps,pre_nspikes,p_pre,alpha)

[post_nspikes, post_nreps]=get_nspikes(filename);

%points=reshape(post_nspikes,1,prod(size(post_nspikes)))';

count=1;
points=[];
for i=1:pre_nreps,
	for j=1:post_nreps,
		p_post(:,count)=polyfit(pre_nspikes(:,i),post_nspikes(:,j),1)';
		points=[points' post_nspikes(:,j)']';
		count=count+1;
	end;
end;
p=mean(p_post')';
sig=std(p_post')';
	

[h(1),tt(1)]=ttest(p_post(1,:),1,alpha);
[h(2),tt(2)]=ttest(p_post(2,:),0,alpha);
wr(1)=ranksum(p_pre(1,:),p_post(1,:),alpha);
wr(2)=ranksum(p_pre(2,:),p_post(2,:),alpha);
tt=tt';
wr=wr';
h=h';
