/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: xstuff.c,v 2.50 2001/02/19 21:39:51 cmalek Exp $
** Program: dowl
**  Module: xstuff.c
**  Author: mazer
** Descrip: misc routines for X that don't fit elsewhere..
**
** Revision History (most recent last)
**
** Fri Feb 21 16:37:24 1992 mazer
**  no more warping pointer onto popups - the popup should appear
**  beneath the cursor, wherever than may be.. some other junk
**  functions, like centerWidget were reaped..
**
** Sat Feb 22 16:36:42 1992 mazer
**  PShellDestroyedCB() modfied to only nullify the pshell arg IF
**  it's the same one that was destroyed.
**
** Tue Apr 14 13:26:08 1992 mazer
**  changed makePixmap() to be #ifdef mc700 and #else instead of
**  #ifdef sparc.. looks like sparc and mips servers are the same..
**  masscomp is the looser, again.
**
** Wed Feb 24 17:58:54 1993 mazer
**  warp_to_widget() deleted -- this was not a recommended call
**  anyway according to x11 docs..
**
** Sun Mar  6 23:03:42 1994 mazer
**  these nasty popup functions:
**    popup_at(psh, x, y)
**    popup_pointer(psh) <-- new (really was simple popup)
**    popup(psh)       <-- now allows window manager to set pos..
**    xpopup(psh)
**    popdown(psh)
**    popkill(psh)
**
** Sat Apr  9 18:51:08 1994 mazer
**  changed shell_new() to appshell_new() --> this should only be
**  used to make real application shells -- not simple popup
**  windows
**
** Thu May  5 15:08:19 1994 mazer
**  changed appshell_new() to top_new() for new TopLevel window
**  this coincides with a number of cleanup changes in which
**  pop_new() calls all over the place are replaced with top_new()
**  calls where appropriate.
**
** 97.12 bjarthur
**  changed pop_new() to use transientShell
**
*******************************************************************/

#include "xdphyslib.h"

static void place_popup(Widget);

void beep(
	int		urgency)		/* -100 - +100 (min - max) */
{
  if (TopLevel) {
    XBell(XtDisplay(TopLevel), urgency);
    dloop_sync();
  } else {
    fprintf(stderr, "\007");
    fflush(stderr);
  }
}

static void place_popup(Widget pshell)	/* position popup under the cursor */
{
  Position xpos, ypos;
  Dimension w, h;
  Window win;
  int rx, ry, wx, wy;
  unsigned int mask;
  void *geom_data;

  XtVaGetValues(pshell, XtNgeometry, &geom_data, NULL);
  if (geom_data == NULL) {
    if (!XtIsRealized(pshell))
      XtRealizeWidget(pshell);
    XtVaGetValues(pshell, XtNx, &xpos, XtNy, &ypos, NULL);
    if (xpos <= 0 && ypos <= 0) {
      XQueryPointer(XtDisplay(pshell), XtWindow(pshell),
		    &win, &win, &rx, &ry, &wx, &wy, &mask);
      XtVaGetValues(pshell, XtNwidth, &w, XtNheight, &h, NULL);
      xpos = rx - w / 2;
      ypos = ry - h / 2;
      if ((xpos + w) > WidthOfScreen(XtScreen(pshell)))
	xpos = WidthOfScreen(XtScreen(pshell)) - w - 10;
      if (xpos < 0)
	xpos = 0;
      if ((ypos + h) > HeightOfScreen(XtScreen(pshell)))
	ypos = HeightOfScreen(XtScreen(pshell)) - h - 10;
      if (ypos < 0)
	ypos = 0;
      XtVaSetValues(pshell, XtNx, xpos, XtNy, ypos, NULL);
    }
  }
}

void popup_at(Widget pshell, Position xpos, Position ypos)
{
  XtVaSetValues(pshell, XtNx, xpos, XtNy, ypos, NULL);
  XtPopup(pshell, XtGrabNone);
  XRaiseWindow(XtDisplay(pshell), XtWindow(pshell));
}

void popup_pointer(Widget pshell)
{
  place_popup(pshell);
  XtPopup(pshell, XtGrabNone);
  XRaiseWindow(XtDisplay(pshell), XtWindow(pshell));
}

void popup(Widget pshell)		/* use geometry (if avail) else cursorpos */
{
  XtPopup(pshell, XtGrabNone);
  XRaiseWindow(XtDisplay(pshell), XtWindow(pshell));
}

void xpopup(Widget pshell)		/* use geometry (if avail) else cursorpos */
{
  place_popup(pshell);
  XtPopup(pshell, XtGrabNonexclusive);
  XRaiseWindow(XtDisplay(pshell), XtWindow(pshell));
}

void popdown(Widget pshell)
{
  XtPopdown(pshell);
  dloop_sync();
}

void popkill(Widget pshell)
{
  XtDestroyWidget(pshell);
}


void ClosePShellCB(Widget w, 
	XtPointer pshellptr, 
	XtPointer client_data)
{
  Widget pshell = (Widget) pshellptr;

  if (pshell)
    XtDestroyWidget(pshell);
}

void HidePShellCB(
	Widget w, 
	XtPointer pshellptr, 
	XtPointer client_data)
{
  Widget pshell = (Widget) pshellptr;

  if (pshell)
    popdown(pshell);
}

void ExposePShellCB(
	Widget w, 
	XtPointer pshellptr, 
	XtPointer client_data)
{
  Widget pshell = (Widget) pshellptr;

  if (pshell)
    popup(pshell);
}

void PShellDestroyedCB(
	Widget w,
	XtPointer pshellptr, 
	XtPointer client_data)
{
  Widget *pshell = (Widget*) pshellptr;
  if (*pshell == w) {
    *pshell = NULL;
  }
}

Pixmap makePixmap(
	char *data, 
	unsigned int w, 
	unsigned int h)
{
  int scr;
  Display *dpy;
  Pixmap pm;
  extern Widget TopLevel;

  dpy = XtDisplay(TopLevel);
  scr = XDefaultScreen(dpy);
  pm = XCreatePixmapFromBitmapData(dpy,
				   XDefaultRootWindow(dpy),
				   data, w, h, 
#ifdef mc700
				   XWhitePixel(dpy, scr),
				   XBlackPixel(dpy, scr),
#else
				   XBlackPixel(dpy, scr),
				   XWhitePixel(dpy, scr),
#endif
				   1);
  return(pm);
}

int ConvertColor(
	Widget w, 
	char *color_name)
{
  XrmValue from, to;

  from.size = strlen(color_name) + 1;
  from.addr = color_name;

/*
 * This conversion accepts a colorname from rgb.txt, or a #rrrgggbbb
 * rgb color definition, or the special toolkit strings "XtDefaultForeground"
 * and "XtDefaultBackground".
 */

  XtConvert(w, XtRString, (XrmValuePtr) &from, XtRPixel, (XrmValuePtr) &to);
  if (to.addr == NULL) {
      return(-1);
  }

  return( (int) *((Pixel *) to.addr) );
}

void ReverseColors(
	Widget w)
{
  Pixel fg, bg;

  XtVaGetValues(w, XtNforeground, &fg, XtNbackground, &bg, NULL);
  XtVaSetValues(w, XtNforeground, bg, XtNbackground, fg, NULL);
}


Widget pop_new(
	Widget parent, 
	char *name)
{
  return(XtVaCreatePopupShell(name ? name : "PopupShell",
			      transientShellWidgetClass,
			      parent ? parent : TopLevel,
			      XtNallowShellResize, True, NULL));
}

Widget top_new(
	Widget parent, 
	char *name)
{
	return (XtVaCreatePopupShell(name ? name : "PopupShell",
			     applicationShellWidgetClass,
			     parent ? parent : TopLevel,
			     XtNallowShellResize, True, NULL));
}

Widget form_new(
	Widget parent, 
	char *name, 
	int  border)
{
  if (border >= 0) {
    return(XtVaCreateManagedWidget(name ? name : "form",
				   formWidgetClass, parent,
				   XtNborderWidth, border, NULL));
  } else {
    return(XtVaCreateManagedWidget(name ? name : "form",
				   formWidgetClass, parent, NULL));
  }
}

Widget box_new(
	Widget parent, 
	char *name, 
	int border)
{
  if (border >= 0) {
    return(XtVaCreateManagedWidget(name ? name : "box",
				   boxWidgetClass, parent,
				   XtNborderWidth, border, NULL));
  } else {
    return(XtVaCreateManagedWidget(name ? name : "box",
				   boxWidgetClass, parent, NULL));
  }
}

int char_size(
	Widget w, 
	int *maxwidth, 
	int *maxheight)
{
  XFontStruct *fstruct;

  XtVaGetValues(w, XtNfont, &fstruct, NULL);
  if (fstruct) {
    if (maxwidth != NULL)
      *maxwidth = fstruct->max_bounds.width;
    if (maxheight != NULL)
      *maxheight = fstruct->max_bounds.ascent + fstruct->max_bounds.descent;
    return(1);
  } else {
    return(0);
  }
}

void set_led(
	int n, 
	int on)
{
  XKeyboardControl xk;

  if (TopLevel) {
    xk.led_mode = on ? LedModeOn : LedModeOff;
    xk.led = n;
    XChangeKeyboardControl(XtDisplay(TopLevel), KBLed | KBLedMode, &xk);
  }
}

int PercentDisplayWidth(
	int percent, 
	int max_pix)
{
  int k;
  k = percent * DisplayWidth(XtDisplay(TopLevel),
			     XScreenNumberOfScreen(XtScreen(TopLevel))) / 100;
  return((k > max_pix) ? max_pix : k);
}

int PercentDisplayHeight(
	int percent, 
	int max_pix)
{
  int k;
  k = percent * DisplayHeight(XtDisplay(TopLevel),
			     XScreenNumberOfScreen(XtScreen(TopLevel))) / 100;
  return((k > max_pix) ? max_pix : k);
}
