/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: xdview.c,v 1.40 2001/03/27 06:59:29 cmalek Exp $
** Program: xdview
**  Module: xdview.c
**  Author: mazer
** Descrip: xdview main() and argument parsing
**
** Revision History (most recent last)
**
*******************************************************************/

#ifndef __linux__
#define _POSIX_PATH_MAX			255
#endif __linux__

#include "xdphyslib.h"
#include <limits.h>
#include "plotter.h"
#include "pm_gen.h"

#ifdef __linux__

char *optarg;
int optind;

#include "unistd.h"
#include <getopt.h>
#define OptArg optarg
#define OptInd optind

#else /* not  __linux__ */

#include "getopt.h"
#define OptArg OptArg
#define OptInd OptInd

#endif /* __linux__ */


#define DEBUG if (debugflag) fprintf

extern Widget TopLevel;

extern void install_xdphys_plotMethods(void);
extern void install_xcalibur_plotMethods(void);
extern void txt_peak_stats(Widget, XtPointer, XtPointer);
/*extern void txt_linearity_stats(FDO*, char*, FILE*);*/

static XDviewArgs *xdview_parse_cmdline(int, char**, char***, int*);
static void init_cmdline_flags(XDviewArgs*);
static void resolve_plot_type(XDviewArgs*, XDviewPlotType *, int);
static void usage(void);
static void DestructionCB(Widget, XtPointer, XtPointer);
static void sigmoid_stats(char**,int);
static void peak_stats(char**,int);
static void isih_stats(char**,int);
static void nspikes(char**,int);
static void similarity_stats(char**,int);
/*static void linearity_stats(char**,int,XDviewArgs*);*/
static void ana(char**,int,XDviewArgs*);
static void prob_distribution(char**,int);
static void combine_data(char**,int, XDviewArgs *);
static void display_window(char**,int, XDviewArgs *);
static void print_postscript(char **, int , XDviewArgs *);
static void print_text(char **, int , XDviewArgs *);
static void print_params(char **, int , XDviewArgs *);
static void print_rasters(char **, int , XDviewArgs *);
static void Quit(Widget, XtPointer, XtPointer);
static void QuitAction(Widget, XEvent*, String*, Cardinal*);
static void install_actions(Widget);
int main(int, char**);
#ifdef __test__
static void display_cmdline_flags(XDviewArgs *, char **, int);
#endif __test__
static void add_plottype_struct(XDviewPlotType **, int *);
static void init_plottype_struct(XDviewPlotType *);


/*  ------------------------------------------------------------------------ */
#define XDVIEW_VERSION 					"2.0"
#define DEBUG_SWITCH(arg, label, val) DEBUG(stderr, \
	"  xdview: parsed %s switch: %s=%d\n", arg, label, val)

static XDviewArgs *xdview_parse_cmdline(
	int		ac,
	char	**av,
	char	***filenames,
	int		* nfiles)
{
	XDviewArgs	*fdo_flags = NULL;
	XDviewPlotType	*plot_type = NULL;
	int	nplottypes = 0;
	char		c;
	char 		*tmp;
	static struct option longopts[] = {
		{"curve", 0, NULL, 'a'},
		{"errs", 0, NULL, 'A'},
		{"no_errs", 0, NULL, 'b'},
		{"left", 0, NULL, 'B'},
		{"right", 0, NULL, 'c'},
		{"both", 0, NULL, 'C'},
		{"raster", 0, NULL, 'd'},
		{"raw", 0, NULL, 'D'},
		{"presorder", 0, NULL, 'e'},
		{"psth", 0, NULL, 'E'},
		{"isih", 0, NULL, 'f'},
		{"perhist", 0, NULL, 'F'},
		{"rayleigh", 0, NULL, 'g'},
		{"vs", 0, NULL, 'G'},
		{"mp", 0, NULL, 'h'},
		{"no_text_header", 0, NULL, 'H'},
		{"phase", 0, NULL, 'i'},
		{"mag", 0, NULL, 'I'},
		{"distortion", 0, NULL, 'j'},
		{"separate", 0, NULL, 'J'},
		{"difference", 0, NULL, 'k'},
		{"leak", 0, NULL, 'K'},
		{"raw_points", 0, NULL, 'l'},
		{"color", 0, NULL, 'L'},
		{"baw", 0, NULL, 'm'},
		{"grid", 0, NULL, 'M'},
		{"log", 1, NULL, 'n'},
		{"no_grid", 0, NULL, 'N'},
		{"params", 0, NULL, 'o'},
		{"ps", 0, NULL, 'O'},
		{"xfig", 0, NULL, 'p'},
		{"text", 0, NULL, 'P'},
		{"roveritd", 0, NULL, 'q'},
		{"roveriid", 0, NULL, 'Q'},
		{"roversurface", 0, NULL, 'r'},
		{"debug", 0, NULL, 'R'},
		{"version", 0, NULL, 's'},
		{"peak_stats", 0, NULL, 'S'},
		{"combine", 1, NULL, 't'},
		{"out", 1, NULL, 'T'},
		{"help", 0, NULL, 'u'},
		{"isih_stats", 0, NULL, 'U'},
		{"similarity_stats", 0, NULL, 'v'},
		{"nspikes", 0, NULL, 'V'},
		/*{"linearity_stats", 1, NULL, 'w'},*/
		{"prob", 0, NULL, 'W'},
		{"ana", 1, NULL, 'x'},
		{"rasters", 0, NULL, 'X'},
		{"sigmoid_stats", 0, NULL, 'y'},
		{0, 0, 0, 0}};

	int i;

	assert((fdo_flags = (XDviewArgs *) calloc(1, sizeof(XDviewArgs))) != NULL);

	init_cmdline_flags(fdo_flags);

	DEBUG(stderr, "xdview: parsing command line options\n");
	while (1) {

		if ((c = getopt_long_only(ac, av, 
			"aAbBcCdDeEfFgGhiIjJkKlLmMn:NoOpPqQrRsSt:T:uUvVw:WxXy",
			longopts, NULL)) == -1)
			break;

		switch (c) {
			case 'a':
				add_plottype_struct(&plot_type, &nplottypes);
				plot_type[nplottypes-1].plot_type = PM_CURVE;
				break;
			case 'A':
				fdo_flags->no_errs = 0;
				break;
			case 'b':
				fdo_flags->no_errs = 1;
				break;
			case 'B':
				plot_type[nplottypes-1].side = SideLeft;
				break;
			case 'c':
				plot_type[nplottypes-1].side = SideRight;
				break;
			case 'C':
				plot_type[nplottypes-1].side = SideBoth;
				break;
			case 'd':
				add_plottype_struct(&plot_type, &nplottypes);
				plot_type[nplottypes-1].plot_type = PM_RASTER_RAW;
				break;
			case 'D':
				plot_type[nplottypes-1].order = OrderRaw;
				break;
			case 'e':
				plot_type[nplottypes-1].order = OrderPres;
				break;
			case 'E':
				add_plottype_struct(&plot_type, &nplottypes);
				plot_type[nplottypes-1].plot_type = PM_PSTH;
				break;
			case 'f':
				add_plottype_struct(&plot_type, &nplottypes);
				plot_type[nplottypes-1].plot_type = PM_ISIH;
				break;
			case 'F':
				add_plottype_struct(&plot_type, &nplottypes);
				plot_type[nplottypes-1].plot_type = PM_PERHIST;
				break;
			case 'g':
				add_plottype_struct(&plot_type, &nplottypes);
				plot_type[nplottypes-1].plot_type = PM_RAYLEIGH;
				break;
			case 'G':
				add_plottype_struct(&plot_type, &nplottypes);
				plot_type[nplottypes-1].plot_type = PM_VS;
				break;
			case 'h':
				add_plottype_struct(&plot_type, &nplottypes);
				plot_type[nplottypes-1].plot_type = PM_MEANPHASE;
				break;
			case 'H':
				fdo_flags->no_text_header = 1;
				break;
			case 'i':
				add_plottype_struct(&plot_type, &nplottypes);
				plot_type[nplottypes-1].plot_type = PM_PHASE;
				break;
			case 'I':
				add_plottype_struct(&plot_type, &nplottypes);
				plot_type[nplottypes-1].plot_type = PM_MAG;
				break;
			case 'j':
				add_plottype_struct(&plot_type, &nplottypes);
				plot_type[nplottypes-1].plot_type = PM_DISTORTION;
				break;
			case 'J':
				plot_type[nplottypes-1].channel_op = Separate;
				break;
			case 'k':
				plot_type[nplottypes-1].channel_op = Difference;
				break;
			case 'K':
				plot_type[nplottypes-1].leak = 1;
				break;
			case 'l':
				fdo_flags->show_raw_points = 1;
				break;
			case 'L':
				fdo_flags->baw = 0;
				break;
			case 'm':
				fdo_flags->baw = 1;
				break;
			case 'M':
				fdo_flags->no_grid = 0;
				break;
			case 'n':
				tmp = OptArg;
				switch (tmp[0]) {
					case 'x':
						fdo_flags->log_x = 1;
						break;
					case 'y':
						fdo_flags->log_y = 1;
						break;
					case 'b':
						fdo_flags->log_x = 1;
						fdo_flags->log_y = 1;
						break;
					default:
						break; 
				}
				break;
			case 'N':
				fdo_flags->no_grid = 1;
				break;
			case 'o':
				fdo_flags->action = DispParms;
				break;
			case 'O':
				fdo_flags->action = DispPs;
				break;
			case 'p':
				fdo_flags->action = DispXFig;
				fprintf(stderr, "xdview: -xfig is not yet supported\n");
				return(NULL);
				break;
			case 'P':
				fdo_flags->action = DispText;
				break;
			case 'q':
				add_plottype_struct(&plot_type, &nplottypes);
				plot_type[nplottypes-1].plot_type = PM_ROVER_ITD;
				break;
			case 'Q':
				add_plottype_struct(&plot_type, &nplottypes);
				plot_type[nplottypes-1].plot_type = PM_ROVER_IID;
				break;
			case 'r':
				add_plottype_struct(&plot_type, &nplottypes);
				plot_type[nplottypes-1].plot_type = PM_ROVER_SURFACE;
				break;
			case 'R':
				debugflag = 1;
				break;
			case 's':
				fprintf(stderr, "\nxdview Version %s\n", XDVIEW_VERSION);
				return(NULL);
				break;
			case 'S':
				fdo_flags->action = PeakStats;
				break;
			case 't':
				fdo_flags->action = CombineData;
				fdo_flags->other_parms = OptArg;
				break;
			case 'T':
				fdo_flags->out_file = OptArg;
				break;
			case 'u':
				usage();
				return(NULL);
				break;
			case 'U':
				fdo_flags->action = IsihStats;
				break;
			case 'v':
				fdo_flags->action = SimilarityStats;
				break;
			case 'V':
				fdo_flags->action = NSpikes;
				break;
/*
			case 'w':
				fdo_flags->action = LinearityStats;
				fdo_flags->other_parms = OptArg;
				break;
*/
			case 'W':
				fdo_flags->action = ProbDistribution;
				break;
			case 'x':
				fdo_flags->action = Ana;
				fdo_flags->other_parms = OptArg;
				break;
			case 'X':
				fdo_flags->action = DispRasters;
				break;
			case 'y':
				fdo_flags->action = SigmoidStats;
				break;
			case '?':
				fprintf(stderr, "xdview: flag unrecognised\n");
				return(NULL);
				break;
		}
	}

	if (OptInd < ac) {
		assert(((*filenames) = (char **) calloc(ac-OptInd, sizeof(char *))) 
			!= NULL);
		for (i=0; i<ac-OptInd; i++) {
			assert(((*filenames)[i] = (char *) calloc(_POSIX_PATH_MAX, 
				sizeof(char))) != NULL);
			(void) strcpy((*filenames)[i], av[i+OptInd]);
		}
	}
	*nfiles = ac - OptInd;

	if (plot_type == NULL)
		add_plottype_struct(&plot_type, &nplottypes);

	resolve_plot_type(fdo_flags, plot_type, nplottypes);

	if (plot_type != NULL)
		free(plot_type);

	if (fdo_flags->action == DispPs) {
		if (fdo_flags->out_file == NULL) 
			fdo_flags->out_file = "-";
	}

	return(fdo_flags);
		
}
#undef DEBUG_SWITCH

static void add_plottype_struct(XDviewPlotType **plottype, int *nplottypes)
{
	if (*plottype == NULL) {
		assert((*plottype = (XDviewPlotType *) calloc(1, 
			sizeof(XDviewPlotType))) != NULL);
		*nplottypes = 1;
	} else {
		assert((*plottype = (XDviewPlotType *) realloc(*plottype, 
			sizeof(XDviewPlotType)*(*nplottypes + 1))) != NULL);
		(*nplottypes)++;
	}
	init_plottype_struct(&(*plottype)[*nplottypes-1]);
}

static void init_plottype_struct(XDviewPlotType *plottype)
{
	plottype->leak=0;
	plottype->side=SideBoth;
	plottype->order=OrderRaw;
	plottype->channel_op=Separate;
	plottype->plot_type=PM_DEFAULT;
}

static void init_cmdline_flags(XDviewArgs *fdo_flags)
{
	fdo_flags->no_grid=0;
	fdo_flags->normalize=0;
	fdo_flags->baw=0;
	fdo_flags->log_x = 0;
	fdo_flags->log_y = 0;
	fdo_flags->no_errs = 0;
	fdo_flags->show_raw_points = 0;
	fdo_flags->no_text_header = 0;
	fdo_flags->plot_type=NULL;
	fdo_flags->action=DispWindow;
	fdo_flags->other_parms=NULL;
	fdo_flags->out_file=NULL;

}

static void resolve_plot_type(XDviewArgs *fdo_flags, 
	XDviewPlotType *plot_type, int nplottypes)
{
	int i;

	assert((fdo_flags->plot_type = (int *) calloc(nplottypes, sizeof(int)))
		!= NULL);

	fdo_flags->nplottypes = nplottypes;
	for (i=0; i<nplottypes; i++) {
		fdo_flags->plot_type[i] = plot_type[i].plot_type;
		switch (plot_type[i].side) {
			case SideLeft:
				if (fdo_flags->plot_type[i] == PM_PERHIST)
					fdo_flags->plot_type[i] = PM_PERHIST_LEFT;
				break;
			case SideRight:
				if (fdo_flags->plot_type[i] == PM_PERHIST)
					fdo_flags->plot_type[i] = PM_PERHIST_RIGHT;
				break;
			default:
				break;
		}

		if (plot_type[i].order != OrderRaw) {
			if (fdo_flags->plot_type[i] == PM_RASTER_RAW)
				fdo_flags->plot_type[i] = PM_RASTER;
		}

		if (plot_type[i].leak == 1) {
			switch (fdo_flags->plot_type[i]) {
				case PM_MAG:
					fdo_flags->plot_type[i] = PM_LEAK_MAG;
					break;
				case PM_PHASE:
					fdo_flags->plot_type[i] = PM_LEAK_PHASE;
					break;
				case PM_DISTORTION:
					fdo_flags->plot_type[i] = PM_LEAK_DISTORTION;
					break;
				default:
					break;
			}
		}

		if (plot_type[i].channel_op != Separate) {
			switch (fdo_flags->plot_type[i]) {
				case PM_MAG:
					fdo_flags->plot_type[i] = PM_DIFFERENCE_MAG;
					break;
				case PM_PHASE:
					fdo_flags->plot_type[i] = PM_DIFFERENCE_PHASE;
					break;
				default:
					break;
			}
		}
	}
}

static void usage(void) 
{
	fprintf(stderr, "usage: xdview [flags] [filenames]\n");
	fprintf(stderr, "\n");
	fprintf(stderr, "flags are:\n");
	fprintf(stderr, "	-curve, -errs, -no_errs, -left, -right, \n");
	fprintf(stderr, "	-both, -raster, -raw, -presorder, -psth \n");
	fprintf(stderr, "	-isih, -perhist, -rayleigh, -vs, -mp, \n");
	fprintf(stderr, "	-phase, -mag, -distortion, -separate, \n");
	fprintf(stderr, "	-difference, -leak, -color, -baw, -log <x|y|both>\n");
	fprintf(stderr, "	-grid, -no_grid, -params, -ps, -xfig, -text,\n");
	fprintf(stderr, "	-roveritd, -roveriid, -roversurface, -debug, -out\n");
	fprintf(stderr, "	-peak_stats, -isih_stats, -similarity_stats, -combine\n");
	fprintf(stderr, "	-nspikes -prob -ana -sigmoid_stats -no_text_header\n");
	fprintf(stderr, "	-raw_points\n");
}

static void DestructionCB(Widget w, XtPointer topptr, XtPointer call_data)
{
	Widget top = (Widget) topptr;

	XtDestroyWidget(top);
	exit(0);
}

static void change_fdobj_settings(FDO *fdo, XDviewArgs *args)
{
	fdo->no_grid = args->no_grid;
	fdo->baw = args->baw;
	fdo->log_x = args->log_x;
	fdo->log_y = args->log_y;
	fdo->no_errs = args->no_errs;
	fdo->normalize = args->normalize;
	fdo->no_text_header = args->no_text_header;
	fdo->show_raw_points = args->show_raw_points;
}

extern void Sigmoid(Widget, XtPointer, XtPointer);

static void sigmoid_stats(char **flist, int nfiles)
{
	FILEDATA *fd;
	FDO *fdo=NULL;
	int i;
	FDObj_ViewType def_view = {0,0};

	fdo = FDObj_New(NULL, NULL, &def_view, 1);
	fdo->destroy_old = 0;
	for(i=0; i<nfiles; i++) {
		if ((fd = FD_read(flist[i],0)) != NULL)
				FDObj_Add(fdo,fd); }

	assert(fdo!=NULL);

	FDObj_Update(fdo,NULL);
	fdo->to_tty = 1;
	Sigmoid(NULL, fdo, 0);
	FDObj_Close(fdo);
}

static void peak_stats(char **flist, int nfiles)
{
	FILEDATA *fd;
	FDO *fdo=NULL;
	int i;
	FDObj_ViewType def_view = {0,0};

	fdo = FDObj_New(NULL, NULL, &def_view, 1);
	fdo->destroy_old = 0;
	for(i=0; i<nfiles; i++) {
		if ((fd = FD_read(flist[i],0)) != NULL)
				FDObj_Add(fdo,fd); }

	assert(fdo!=NULL);

	FDObj_Update(fdo,NULL);
	fdo->to_tty = 1;
	if(!strcmp(fdo->fds[0]->modname,"txt"))
		txt_peak_stats(NULL, fdo, 0);
	else
		FD1_peak_stats(NULL, fdo, 0);
	FDObj_Close(fdo);
}

static void nspikes(char **flist, int nfiles)
{
	FILEDATA *fd;
	FDO *fdo=NULL;
	int i,j,k,l,m,nn,o,p,q,**nspikes,*n,ndata;
  int nabis,nitds,niids,nbcs,nstims,nmonos,ntwo_snds;
  int ********gen_nspikes,*******gen_n,*gen_s_nspikes,gen_s_n;
	FDObj_ViewType def_view = {0,0};
  char ********gen_depstrs;
  float *depvals;

	fdo = FDObj_New(NULL, NULL, &def_view, 1);
	fdo->destroy_old = 0;
	for (i = 0; i < nfiles; i++) {
		if ((fd = FD_read(flist[i], 0)) != NULL)
			FDObj_Add(fdo, fd);
	}

	assert(fdo != NULL);

	FDObj_Update(fdo, NULL);

	for(i=0; i<nfiles; i++) {
	  if(!strcmp(fdo->fds[0]->modname,"gen")) {
      gen_get_ndata(fdo->fds[i],
            &nabis,&nitds,&niids,&nbcs,&nstims,&nmonos,&ntwo_snds);
      gen_n=gen_get_n(fdo->fds[i]);
      gen_nspikes=gen_get_nspikes(fdo->fds[i]);
      gen_s_n=gen_get_s_n(fdo->fds[i]);
      gen_s_nspikes=gen_get_s_nspikes(fdo->fds[i]);
      gen_depstrs=gen_get_depstrs(fdo->fds[i]);
      fprintf(stdout,"<SPONT> ");
      for(q=0; q<gen_s_n; q++)
        fprintf(stdout,"%d ", gen_s_nspikes[q]);
      fprintf(stdout,"\n");
	    for(j=0; j<nabis; j++) {
	      for(k=0; k<nitds; k++) {
	        for(l=0; l<niids; l++) {
	          for(m=0; m<nbcs; m++) {
	            for(nn=0; nn<nstims; nn++) {
	              for(o=0; o<nmonos; o++) {
	                for(p=0; p<ntwo_snds; p++) {
                    fprintf(stdout,"%s ",gen_depstrs[j][k][l][m][nn][o][p]);
	                  for(q=0; q<gen_n[j][k][l][m][nn][o][p]; q++)
                      fprintf(stdout,"%d ",
                            gen_nspikes[j][k][l][m][nn][o][p][q]);
                    fprintf(stdout,"\n"); } } } } } } } }
    else {
      ndata=FD1_get_ndata(fdo->fds[i]);
      n=FD1_get_n(fdo->fds[i]);
      nspikes=FD1_get_nspikes(fdo->fds[i]);
      depvals=FD1_get_depvals(fdo->fds[i]);
	    for(j=0; j<ndata; j++) {
        fprintf(stdout,"%d ",(int)depvals[j]);
	      for(k=0; k<n[j]; k++)
          fprintf(stdout,"%d ",nspikes[j][k]);
        fprintf(stdout,"\n"); } } }

	FDObj_Close(fdo);
}

static void isih_stats(char **flist, int nfiles)
{
	FILEDATA *fd;
	FDO *fdo=NULL;
	int i;
	FDObj_ViewType def_view = {0,0};

	fdo = FDObj_New(NULL, NULL, &def_view, 1);
	fdo->destroy_old = 0;
	for(i=0; i<nfiles; i++) {
		if ((fd = FD_read(flist[i],0)) != NULL)
				FDObj_Add(fdo,fd); }

	assert(fdo!=NULL);

	FDObj_Update(fdo,NULL);
	fdo->to_tty = 1;
	FD1_isih_stats(NULL, fdo, 0);
	FDObj_Close(fdo);
}

static void similarity_stats(char **flist, int nfiles)
{
	FILEDATA *fd;
	FDO *fdo=NULL;
	int i;
	FDObj_ViewType def_view = {0,0};

	fdo = FDObj_New(NULL, NULL, &def_view, 1);
	fdo->destroy_old = 0;
	for(i=0; i<nfiles; i++) {
		if ((fd = FD_read(flist[i],0)) != NULL)
				FDObj_Add(fdo,fd); }

	assert(fdo!=NULL);

	FDObj_Update(fdo,NULL);
	fdo->to_tty = 1;
	FD1_similarity_stats(NULL, fdo, 0);
	FDObj_Close(fdo);
}

static void prob_distribution(char **flist, int nfiles)
{
	FILEDATA *fd;
	FDO *fdo=NULL;
	Widget top;
	int i;
	FDObj_ViewType def_view = {0,0};

	top = top_new(TopLevel, "xdview");
	fdo = FDObj_New(NULL, top, &def_view, 0);
	fdo->destroy_old = 0;
	for(i=0; i<nfiles; i++) {
		if ((fd = FD_read(flist[i],0)) != NULL)
				FDObj_Add(fdo,fd); }

	assert(fdo!=NULL);

	FDObj_Update(fdo,NULL);
	fdo->to_tty = 1;
	if(strcmp(fdo->fds[0]->modname,"txt"))
		FD1_prob_distribution(NULL, fdo, 0);
	FDObj_Close(fdo);
	XtDestroyWidget(top);
}

#if(0)
static void linearity_stats(char **flist, int nfiles, XDviewArgs *args)
{
	FILEDATA *fd;
	FILE *fptr;
	FDO *fdo=NULL;
	int i;
	FDObj_ViewType def_view = {0,0};

	if(args->out_file!=NULL)
		assert((fptr=fopen(findhome(args->out_file),"w"))!=NULL);
	else
		fptr = NULL;

	fdo = FDObj_New(NULL, NULL, &def_view, 1);
	fdo->destroy_old = 0;
	for(i=0; i<nfiles; i++) {
		if ((fd = FD_read(flist[i],0)) != NULL)
				FDObj_Add(fdo,fd); }

	assert(fdo!=NULL);

	FDObj_Update(fdo,NULL);
	fdo->to_tty = 1;
	if(!strcmp(fdo->fds[0]->modname,"txt"))
		txt_linearity_stats(fdo, args->other_parms,fptr);
	else
		FD1_linearity_stats(fdo, args->other_parms,fptr);
	FDObj_Close(fdo);
}
#endif

/* -------------------------------------------------------------
	the file that ana() writes to stdout looks like this:

	Header:
		int magic				 ( = ANA_MAJIK (see waveio.h))
		char adFc[15]            ( = "adFc=%d" in Hz)
		char epoch[15]           ( = "epoch=%d in ms)
		char decimate[15]        ( = "ana-decimate=%d", boolean (1/0))
		char nrasters[15]         ( = "nrasters=%d" )
		char nchans[15]          ( = "nchans=%d" )
	Conversion block: 1 per channel:
		char tomv[30]          ( = "ana-tomv=%f"  in mV/tick)
		char offset[30]        ( = "ana-offset=%f" in mV)
	Trace:  (X nrasters)
		char depvar[30]            ( = "depvar=%s" )
		char rasternum[30]         ( = "rasternum=%d" )
		char chan_id[30]           ( = "channel=%d" )
		xword DATA               ((adFc*epoch)/1000)/decimate) samples

   ------------------------------------------------------------- */
typedef struct _ana_header {
	int magic;
	char adFc[15];
	char epoch[15];
	char decimate[15];
	char nrasters[15];
	char nchans[15];
} ana_header;

static void ana(char **flist, int nfiles, XDviewArgs * args)
{
	FILEDATA *fd;		/* FILDATA struct for the analog file */
	int i, j;
	int depvar_filter = INT_MIN;	/* if depvar_filter is INT_MIN, depvar has not been 
					   supplied on the command line */
	char tmp[30];		/* print depvar for the current trace into this */
	char tmp2[30];		/* print depvar for the current trace into this */
	int total_rasters;	/* total number of traces in the ana file */
	int nrasters = 0;	/* number of traces actually written out */
	int nbytes;		/* number of bytes in each trace */
	int *depints, *pres_order;

	int nchans;		/* number of analog channels that were recorded for */
	/* each stimulus presentation */

	xword ***ana;		/* array of arrays of analog traces (heheh) */
	int **chan_ids;
	ana_header header;
	float adFc;
	float epoch;
	int decimate;

	assert(args->out_file == NULL);
	assert(args->other_parms != NULL);
	assert(nfiles == 1);

	if (strncmp(args->other_parms, "all", 3))
		sscanf(args->other_parms, "%d", &depvar_filter);


	if ((fd = FD_read(flist[0], 1)) == NULL) {
		perror("xdview: open ana file");
		exit(1);
	}

	adFc = (float) FD_GI(fd, "adFc");
	epoch = ((float) FD_GI(fd, "epoch")) / 1000.0;
	decimate = FD_GI(fd, "ana-decimate");
	if (LastVarUndefined || decimate < 1)
		decimate = 1;
	nbytes = (int) (adFc * epoch) / decimate * sizeof(xword);

	depints = FD1_get_depints(fd);
	pres_order = FD1_get_pres_order(fd);

	nchans = FD1_get_ana_nchans(fd);
	ana = FD1_get_ana(fd);
	chan_ids = FD1_get_chan_ids(fd);
	assert(ana != NULL);
	assert(chan_ids != NULL);

	/* Construct and write the header */
	memset((void *) &header, 0, sizeof(header));
	header.magic = ANA_MAJIK;
	sprintf(header.adFc, "adFc=%d\n", FD_GI(fd, "adFc"));
	sprintf(header.epoch, "epoch=%d\n", FD_GI(fd, "epoch"));
	sprintf(header.decimate, "ana-decimate=%d\n", FD_GI(fd, "ana-decimate"));
	total_rasters = FD1_get_nrasters(fd);
	if (depvar_filter == INT_MIN) {
		for (i = 0; i < total_rasters; i++)
			if (ana[i] != NULL)
				nrasters++;
	} else {
		for (i = 0; i < total_rasters; i++)
			if ((depints[i] == depvar_filter) && (ana[i] != NULL))
				nrasters++;
	}
	sprintf(header.nrasters, "nrasters=%d\n", nrasters);
	sprintf(header.nchans, "nchans=%d\n", nchans);
	if (write(fileno(stdout), (void *) &header, sizeof(ana_header))
	    != sizeof(ana_header)) {
		perror("xdview: ana write header");
		exit(1);
	}

	/* Write the conversion block */

	/* This block assumes that # channels for which "ana.channel#=on"
	 * is equal to ana.nchans.  If it doesn't, something is wrong, and
	 * the assert() should kick in */


	if (FD_GI(fd, "file.version") >= 3) {
		i = 1;
		while (i <= nchans) {
			sprintf(tmp, "ana.channel%d\n", i);
			assert(FD_GV(fd, tmp) != NULL);
			if (strncmp(FD_GV(fd, tmp), "on", 2)==0) {

				memset((void *) tmp, 0, 30);
				sprintf(tmp, "channel=%d", i);
				if (write(fileno(stdout), (void *) tmp, 30) != 30) {
					perror("xdview: ana write channel, conversion block");
					exit(1);
				}

				sprintf(tmp2, "ana-tomv.chan%d", i);
				memset((void *) tmp, 0, 30);
				sprintf(tmp, "ana-tomv=%f\n", FD_GF(fd, tmp2));
				if (write(fileno(stdout), (void *) tmp, 30) != 30) {
					perror("xdview: ana write ana-tomv");
					exit(1);
				}

				sprintf(tmp2, "ana-offset.chan%d", i);
				memset((void *) tmp, 0, 30);
				sprintf(tmp, "ana-offset=%f\n", FD_GF(fd, tmp2));
				if (write(fileno(stdout), (void *) tmp, 30) != 30) {
					perror("xdview: ana write ana-tomv");
					exit(1);
				}

			}
			i++;
		}
	} else {
		/* Ain't backwards compatibility great? */
		memset((void *) tmp, 0, 30);
		sprintf(tmp, "ana-tomv.chan1=%f\n", FD_GF(fd, "ana-tomv"));
		if (write(fileno(stdout), (void *) tmp, 30) != 30) {
			perror("xdview: ana write ana-tomv");
			exit(1);
		}

		memset((void *) tmp, 0, 30);
		sprintf(tmp, "ana-offset.chan1=%f\n", FD_GF(fd, "ana-offset"));
		if (write(fileno(stdout), (void *) tmp, 30) != 30) {
			perror("xdview: ana write ana-tomv");
			exit(1);
		}
	}


	/* Write the traces */
	for (i = 0; i < total_rasters; i++) {
		if (ana[i] != NULL) {
			for (j = 0; j < nchans; j++) {
				assert(ana[i][j] != NULL);
				/* If depvar_filter == INT_MIN, a specific depvar has not been 
				 * supplied on the command line, therefore print traces for all 
				 * depints to stdout, else print traces only for the requested 
				 * depvar */
				if (((depvar_filter == INT_MIN)
				     || (depints[i] == depvar_filter))) {
					memset((void *) tmp, 0, 30);
					sprintf(tmp, "depvar=%d", depints[i]);
					if (write(fileno(stdout), (void *) tmp, 30)
					    != 30) {
						perror("xdview: ana write depstr");
						exit(1);
					}
					memset((void *) tmp, 0, 30);
					sprintf(tmp, "rasternum=%d", pres_order[i]);
					if (write(fileno(stdout), (void *) tmp, 30)
					    != 30) {
						perror
						    ("xdview: ana write rasternumstr");
						exit(1);
					}
					memset((void *) tmp, 0, 30);
					sprintf(tmp, "channel=%d", chan_ids[i][j]);
					if (write(fileno(stdout), (void *) tmp, 30)
					    != 30) {
						perror("xdview: ana write channel");
						exit(1);
					}
					if (write
					    (fileno(stdout), (void *) ana[i][j],
					     nbytes) != nbytes) {
						perror("xdview: ana write trace");
						exit(1);
					}
				}
			}
		}
	}
}

static void combine_data(char **flist, int nfiles, XDviewArgs *args)
{
	FILEDATA *fd;
	FILE *fptr;
	FDO *fdo=NULL;
	int i;
	FDObj_ViewType def_view = {0,0};

	if(args->out_file!=NULL)
		assert((fptr=fopen(findhome(args->out_file),"w"))!=NULL);
	else
		fptr = NULL;

	fdo = FDObj_New(NULL, NULL, &def_view, 1);
	fdo->destroy_old = 0; 
	for(i=0; i<nfiles; i++) {
		if ((fd = FD_read(flist[i],0)) != NULL)
				FDObj_Add(fdo,fd); }

	assert(fdo!=NULL);

	FDObj_Update(fdo,NULL);

	fdo->to_tty = 1;
	fprintf(fptr,";; xdphys txt ver %s\n",XDPHYS_VERSION);
	FD1_combine_data(fdo, args->other_parms, fptr);

	FDObj_Close(fdo);
	if(args->out_file!=NULL)
		fclose(fptr);
}

static void print_postscript(char **flist, int nfiles, XDviewArgs *args)
{
	FILEDATA *fd;
	FDO *fdo=NULL;
	Widget top;
	int i;
	FDObj_ViewType def_view = {0,0};

	if (flist == NULL) {
		fprintf(stderr, "xdview: you must supply a filename with -ps\n");
		return; }
  
	top = top_new(TopLevel, "xdview");
	fdo = FDObj_New(NULL, top, &def_view, 0);
	fdo->destroy_old = 0; 
	for(i=0; i<nfiles; i++) {
		if ((fd = FD_read(flist[i],0)) != NULL)
				FDObj_Add(fdo,fd); }

	assert(fdo!=NULL);

	change_fdobj_settings(fdo, args);

	popup(top);
	dloop_sync();
	mssleep(250);
	popdown(top);

	for (i=0; i<args->nplottypes; i++) {
		if (!doFDOvalidviewMethod(fdo, fd, args->plot_type[i])) {
			fprintf(stderr, "xdview: %s is invalid for %s files\n", 
				pm_names[args->plot_type[i]], extension(flist[0]));
		} else {
			FDObj_Update_ForceView(fdo, NULL, args->plot_type[i]);

			AtPlotterGeneratePostscript(args->out_file,"w",
				 fdo->graph, 0.8, 0); } }

	FDObj_Close(fdo);
	XtDestroyWidget(top);
}

static void print_params(char **flist, int nfiles, XDviewArgs *args)
{
	FILEDATA *fd;
	FDO *fdo=NULL;
	FDObj_ViewType def_view = {0,0};
	int i;

	if (flist == NULL) {
		fprintf(stderr, "xdview: you must supply a filename with -params\n");
		return; }
	if(args->out_file!=NULL) {
		fprintf(stderr, "xdview: -out option incompatible w/ -params option\n");
		return; }
  
	fdo = FDObj_New(NULL, NULL, &def_view, 1);
	fdo->destroy_old = 0; 
	for(i=0; i<nfiles; i++) {
		if ((fd = FD_read(flist[i],0)) != NULL)
				FDObj_Add(fdo,fd); }

	assert(fdo!=NULL);

	fdo->to_tty = 1;
	FDObj_CommentsCB(NULL, fdo, 0);

	FDObj_Close(fdo);
}

static void print_rasters(char **flist, int nfiles, XDviewArgs *args)
{
	FILEDATA *fd;
	FDO *fdo=NULL;
	FDObj_ViewType def_view = {0,0};
	int i;

	if (flist == NULL) {
		fprintf(stderr, "xdview: you must supply a filename with -rasters\n");
		return; }
	if(args->out_file!=NULL) {
		fprintf(stderr, "xdview: -out option incompatible w/ -rasters option\n");
		return; }
	if(nfiles!=1) {
		fprintf(stderr, "xdview: only 1 file allowed w/ -rasters option\n");
		return; }
  
	fdo = FDObj_New(NULL, NULL, &def_view, 1);
	fdo->destroy_old = 0; 
	for(i=0; i<nfiles; i++) {
		if ((fd = FD_read(flist[i],0)) != NULL)
				FDObj_Add(fdo,fd); }

	assert(fdo!=NULL);

	fdo->to_tty = 1;
	FDObj_PrintRasters(NULL, fdo, 0);

	FDObj_Close(fdo);
}

static void print_text(char **flist, int nfiles, XDviewArgs * args)
{
	FILEDATA *fd;
	FDO *fdo = NULL;
	Widget top;
	int i;
	FDObj_ViewType def_view =
	{0, 0};
	FILE *fptr;

	if (flist == NULL) {
		fprintf(stderr, "xdview: you must supply a filename with -text\n");
		return;
	}
	if (args->out_file != NULL)
		assert((fptr = fopen(findhome(args->out_file), "w")) != NULL);
	else
		fptr = stdout;

	fdo = FDObj_New(NULL, top, &def_view, 1);
	fdo->destroy_old = 0;
	for (i = 0; i < nfiles; i++) {
		if ((fd = FD_read(flist[i], 0)) != NULL)
			FDObj_Add(fdo, fd);
	}

	assert(fdo != NULL);

	change_fdobj_settings(fdo, args);

	if (!fdo->no_text_header) 
		fprintf(fptr, ";; xdphys txt ver %s\n", XDPHYS_VERSION);
	for (i = 0; i < args->nplottypes; i++) {
		if (!doFDOvalidviewMethod(fdo, fd, args->plot_type[i])) {
			fprintf(stderr, "xdview: %s is invalid for %s files\n",
				pm_names[args->plot_type[i]], extension(flist[0]));
		} else {
			FDObj_Update_ForceView(fdo, NULL, args->plot_type[i]);
			FDObj_Update(fdo, fptr);
		}
	}

	FDObj_Close(fdo);

	if (args->out_file != NULL)
		fclose(fptr);
}

static void display_window(char **flist, int nfiles, XDviewArgs *args)
{
	FILEDATA *fd;
	FDO *fdo=NULL;
	Widget top;
	Widget form;
	int i;
	FDObj_ViewType def_view = {0,0};

	top = top_new(TopLevel, "xdview");
	form = form_new(top, "xdviewTopForm",1 );

	if (nfiles==0)
		fdo = FDObj_FileBrowser_New("*", form, NULL);
	else if (is_pattern(flist[0]))
		fdo = FDObj_FileBrowser_New(flist[0], form, NULL);
	else {
		fdo = FDObj_New(NULL, form, &def_view, 0);
		fdo->destroy_old = 0; 
		for(i=0; i<nfiles; i++) {
			if ((fd = FD_read(flist[i],0)) != NULL)
				FDObj_Add(fdo,fd); } }

	assert(fdo!=NULL);

	XtRealizeWidget(top);
	dloop_sync();
	mssleep(250);

	change_fdobj_settings(fdo, args);

	if(fdo!=NULL) {
		if (args->plot_type[0] != PM_DEFAULT) 
			FDObj_Update_ForceView(fdo, NULL, args->plot_type[0]);
		else
			FDObj_Update(fdo,NULL);
		fdo->to_tty = 0;
		XtAddCallback(fdo->form, XtNdestroyCallback, DestructionCB, top);
		popup(top);
		dloop(); 
	}
	else
		beep(1);
}

static void Quit(Widget w, XtPointer client_data, XtPointer call_data)
{
	exit(0);
}

static void QuitAction(Widget w, XEvent *event, String *params,
      Cardinal *num_params)
{
  Quit(NULL,NULL,NULL);
}

static void install_actions(Widget TopLevel)
{
	static XtActionsRec actionTable[] = {
		{ "quit",		QuitAction			},
	};

	XtAppAddActions(XtWidgetToApplicationContext(TopLevel),
		actionTable, XtNumber(actionTable));
}

int main(int ac, char **av)
{
	XDviewArgs *args;
	char **files;
	int nfiles;

	progname = av[0];
	initsvars();

	install_xdphys_plotMethods();
	install_xcalibur_plotMethods();
	synth_init();

	args = xdview_parse_cmdline(ac, av, &files, &nfiles);
	if (args == NULL) exit(1);

#ifdef __test__
	display_cmdline_flags(args, files, nfiles);
#endif /* __test__ */

	switch(args->action) {
		case(DispWindow):
	    startx(&ac, av, "xdview", "XDphys");
	    install_actions(TopLevel);
			display_window(files, nfiles, args);
			break;
		case(SigmoidStats):
			sigmoid_stats(files,nfiles);
			break;
		case(PeakStats):
			peak_stats(files,nfiles);
			break;
		case(IsihStats):
			isih_stats(files,nfiles);
			break;
		case(SimilarityStats):
			similarity_stats(files,nfiles);
			break;
/*
		case(LinearityStats):
			linearity_stats(files,nfiles,args);
			break;
*/
		case(Ana):
			ana(files,nfiles,args);
			break;
		case(ProbDistribution):
			prob_distribution(files,nfiles);
			break;
		case(NSpikes):
			nspikes(files,nfiles);
			break;
		case(CombineData):
			combine_data(files, nfiles, args);
			break;
		case(DispPs):
	    startx(&ac, av, "xdview", "XDphys");
	    install_actions(TopLevel);
			print_postscript(files, nfiles, args);
			break;
		case(DispXFig):
			fprintf(stderr, "xdview: -xfig is not yet supported\n");
			break;
		case(DispText):
			print_text(files, nfiles, args);
			break;
		case(DispParms):
			print_params(files, nfiles, args);
			break;
		case(DispRasters):
			print_rasters(files, nfiles, args);
			break;
	}

	return(0);
}

#ifdef __test__
static void display_cmdline_flags(XDviewArgs *fdo_flags, char **files, 
	int nfiles)
{
	int i;

	if (fdo_flags == NULL) {
		fprintf(stderr, "fdo_flags == NULL\n");
		return;
	}

	if ((fdo_flags->plot_type >= 0)  &&
		(fdo_flags->plot_type < PM_NVIEWS))
		fprintf(stderr,"fdo_flags->plot_type=%s\n",pm_names[fdo_flags->plot_type]);
	else {
		switch (fdo_flags->plot_type) {
			case PM_DEFAULT:
				fprintf(stderr,"fdo_flags->plot_type=PM_DEFAULT\n");
				break;
			case PM_NONE:
				fprintf(stderr,"fdo_flags->plot_type=PM_NONE\n");
				break;
			default:
				fprintf(stderr,"fdo_flags->plot_type=UNKNOWN (%d)\n",
					fdo_flags->plot_type);
				break;
		}
	}

	fprintf(stderr,"fdo_flags->grid=%d\n",fdo_flags->no_grid);
	fprintf(stderr,"fdo_flags->y_axis_style=%d\n",fdo_flags->y_axis_style);
	fprintf(stderr,"fdo_flags->color=%d\n",fdo_flags->color);
	fprintf(stderr,"fdo_flags->log_x=%d\n", fdo_flags->log_x);
	fprintf(stderr,"fdo_flags->log_y=%d\n", fdo_flags->log_y);
	switch (fdo_flags->action) {
		case DispWindow:
			fprintf(stderr,"fdo_flags->action=DispWindow\n");
			break;
		case DispPs:
			fprintf(stderr,"fdo_flags->action=DispPs\n");
			break;
		case DispXFig:
			fprintf(stderr,"fdo_flags->action=DispXFig\n");
			break;
		case DispText:
			fprintf(stderr,"fdo_flags->action=DispText\n");
			break;
		case DispParms:
			fprintf(stderr,"fdo_flags->action=DispParms\n");
			break;
		case CombineData:
			fprintf(stderr,"fdo_flags->action=CombineData\n");
			break;
		case PeakStats:
			fprintf(stderr,"fdo_flags->action=PeakStats\n");
			break;
		case IsihStats:
			fprintf(stderr,"fdo_flags->action=IsihStats\n");
			break;
	}
	fprintf(stderr,"--------------------\n");
	if (nfiles == 0) 
		fprintf(stderr,"no files\n");
	else 
		for (i=0; i<nfiles; i++) 
			fprintf(stderr,"file%d: %s\n", i, files[i]);

}
#endif __test__

