/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: xdphys.c,v 1.45 2001/07/09 04:44:41 cmalek Exp $
** Program: xdphys
**  Module: xdphys.c
**  Author: mazer
** Descrip: xdphys actual user interface for data collection
**
** Revision History (most recent last)
**
** Wed Mar  2 22:38:48 1994 mazer
**  Creation date :: was xdowl.c -- now xdowl.c is just a 
**    simple call dowl dowlmain() in this module..
**  The idea is that all the ugly stuff is hidden in this
**    module -- in order to write your own custom application
**    you can essentially start with xdowl.c and add/replace
**    the module and plotmethod lists with your own functions..
**
** 96-7 bjarthur
**  added append_files stuff
**  added HelpCB stuff
**  added Close buttons on mods
**  got rid of slow analog display of entire epoch
** 
** 97.4 bjarthur
**  changed DOWLDIRNEW to XDPHYSDIR
**  now print ;; xdphys ... instead of ;; dowl ... as 1st line of datafile
**
** 97.12 bjarthur
**  added support to lock all worksheets during initial portion
**  of each run.  this gives time for the mods to "download" the svars into
**  local variables while insuring that the user doesn't change them while
**  it's doing this.  after download has occurred, each module should unlock
**  the worksheets using the supplied function.  new functions:
**  lock_worksheets(), unlock_most_worksheets(), unlock_rest_worksheets()
**
**  added Comments flag so user could turn off prompting for comments
**
**  changed appRun to setRack to maxAtten instead of 0 after each run
**
** 98.01 bjarthur
**  added Calc RMS feature.
**
** 99.01 bjarthur
**  added trace.decay variable.  now the histograms in the rig are exponentially
**  decayed in time, instead of cumulative, like they were previously, and like
**  the histograms for all the mods still are.  so only recent neural data is
**  displayed, and fewer clears will be needed
**
** 99.02 bjarthur
**  re-added the soft window discriminator, from the rcs files.  somehow, this
**  was deleted; turns out it might be quite useful.  modified the gui some.
**
** 99.03 bjarthur
**  got rid of append files stuff.  much too complicated to do online.  wrote
**  an awk script to do it posthoc instead.  got rid of ana-separate-files
**  too for the same reason.  use 'cat' posthoc instead
**
*******************************************************************/

#include "xdphyslib.h"
#include "xdphys.h"

static void usage(char*, char*);
static void AboutCB(Widget, XtPointer, XtPointer);

#ifdef SANAL
static void FDOExternCB(Widget, XtPointer, XtPointer);
#endif


static void lock_worksheets(void);
static void savestate(int);
static void SaveStateCB(Widget, XtPointer, XtPointer);
static void Quit(Widget, XtPointer, XtPointer);
static void QuitClockbar(Widget, XtPointer, XtPointer);
static char *datafile_name(char*, char*, int, char*);
static void delimitLogEntry(FILE*);
static char *appRun(int, RunData*, Widget);
static void LogView(Widget w, XtPointer, XtPointer);
static void LogEntry(Widget, XtPointer, XtPointer);
static void PlotCalibData(Widget, XtPointer, XtPointer);
static void ReloadCalibData(Widget, XtPointer, XtPointer);
static void CalcRMS(Widget, XtPointer, XtPointer);
static void ModuleStartFn(Widget, XtPointer, XtPointer);
static void ModuleStartCB(Widget, XtPointer, XtPointer);
static void HelpCB(Widget, XtPointer, XtPointer);
static void ModuleTempCB(Widget, XtPointer, XtPointer);
static void ModuleNamedTempCB(Widget, XtPointer, XtPointer);
static Widget make_close_help(Widget, XtPointer, XtPointer);
static void NewUnit(Widget, XtPointer, XtPointer);
static void PrevFile(Widget w, XtPointer, XtPointer);
static void NextFile(Widget w, XtPointer, XtPointer);
static int  SaveTraceWaveform(char*);
static void SaveTraceCB(Widget, XtPointer, XtPointer);
static void ChangeAnimal(Widget, XtPointer, XtPointer);
static void NewPass(Widget, XtPointer, XtPointer);
static void NewUnitAction(Widget, XEvent*, String*, Cardinal*);
static void QuitAction(Widget, XEvent*, String*, Cardinal*);
static void ToDockAction(Widget, XEvent*, String*, Cardinal*);
static void install_actions(void);
static int  xdphysmain(int, char**, char*, char*);
static void  WriteFileHeader(FILE *, RunData *, char*);
static void  WriteBeginRaster(FILE *);
static void  WriteEndRaster(FILE *);
static void NumberButtonCB(Widget, XtPointer, XtPointer);
static void loadPrefs(int);
static void savePrefs(int);

#if 0
static double timestamp(void);
#endif

int main(int, char**);

extern void install_xcalibur_plotMethods(void);

char *statefile = "~/.xdphysrc/config.xdphys";

static void usage(char *pname, char *badarg)
{
	char version[32];
	sprintf(version, "%s", XDPHYS_VERSION);
	fprintf(stderr, "xdphys -- Version %s\n", version);
	if (badarg)
		fprintf(stderr, "\n%s: unknown argument \"%s\"\n\n", pname,
			badarg);

	if (badarg)
		exit(1);
}

#ifndef MOD_MENU
extern void install_modules(Dock*);
#else /* MOD_MENU */
extern void module_menu(MENU_ENTRY **);
#endif /* MOD_MENU */

extern void install_xdphys_plotMethods(void);

static Widget headline = NULL;

static Dock    *tooldock = NULL;
static TRACER  *main_tracer = NULL;

static Field ws_twosound[] = {
  { "ts.fix_abi (dB)",   "%d", 3,  "40"  },
  { "ts.fix_itd (us)",   "%d", 4,  "0" },
  { "ts.fix_iid (dB)",   "%d", 3,  "0" },
  { "ts.fix_bc (%)",     "%d", 3,  "100" },
  { "ts.fix_Stim",       "%s", 60, "0" },
  { "ts.use_dur_delay",  "%b", 1, "0" },
  { "ts.dur (ms)",       "%d", 4, "0" },
  { "ts.delay (ms)",     "%d", 4, "0" },
/*
  { "ts.rms",            "%b", 1,  "1" },
*/
  { NULL },
};

static Field ws_prefs[] = {
  { WS_VSEP, "Stimulus Parameters" },
  { "ISI (ms)",			"%d",	4, "1000"	},
  { "Reps",		    	"%d",	3, "5"		},
  { "Delay (ms)",		"%d",	4, "100" 	},
  { "Dur (ms)",			"%d",	4, "100" 	},
  { "Epoch (ms)",		"%d",	4, "400" 	},
  { "Rise (ms)",		"%d",	3, "5" 	},
  { "Fall (ms)",		"%d",	3, "5" 	},
  { "Spont_Stims",	"%b",	1, "1" 	},
  { "Randomize",		"%b",	1, "1"		},
  { "noise.low (Hz)",		"%d",	5, "500" 	},
  { "noise.high (Hz)",		"%d",	5, "12000"	},
  { "tone.rad_vary",		"%b",	1, "1"	},
  { "stack.rms",		"%b",	1, "0"	},
  { "cal_file",		"%s",	15, "ear.cal" 	},
  { "use_cache",		"%b",	1, "1"	},
  { WS_VSEP, "Analog Save Parameters" },
  { "ana.channel1 (on,off)","%s",	3, "off"	},
  { "ana.channel2 (on,off)","%s",	3, "on"	},
  { "ana-decimate (by)", "%d", 3, "1" },
  { "ana.every (nth)",   "%d", 2, "1" },
  { WS_HSEP, "DA/AD Stuff" },
  { "daFc (Hz)",		"%f",	6, "48000"	},
  { "adFc (Hz)",		"%f",	6, "48000"	},
  { "syncPulse",		"%b",	1, "1"	},
#ifdef sparc
  { "nicelevel",		"%d",   3, "-10"	},
  { "proport.pad (dB)", 	"%d",   3, "0"	},
  { "evtFc (Hz)",		"%f",	6, "48000"	},
  { "spiker.ttl_signum (1,-1)", "%d",	2, "1"	},
#endif /* not sparc .. mc700 or sgi */
#ifdef __tdtproc__ 
  { "evtFc (kHz)",		"%f",	6, "1000"	},
  { "maxspikes",		"%d",	5, "1000"	},
  { "raster.bin_spikes","%b", 1, "0" },
#endif
  { WS_VSEP, "Discrimination Settings" },
  { "detect.mode (ttl,window,ss)", "%s",	6, "ttl"	},
  { "detect.discrim_channel", "%d",	1, "1"	},
  { "detect.tracer_channel", "%d",	1, "1"	},
  { "detect.use_highpass", "%b",	1, "1"	},
  { "detect.highpass_lowfreq", "%d",	5, "100"	},
  { "detect.jump", "%d",	4, "100"	},
  { WS_VSEP, "Running Parameters" },
  { "Comments",			"%b",	1, "0" 	},
  { "Compress",     "%b", 1, "0" },
  { WS_VSEP, "Rig Controls" },
  { "rig.auto_shutoff",		"%d",	4, "500" 	},
  { "rig.itd++",		"%d",	4, "30" 	},
  { "rig.iid++",		"%d",	4, "5" 	},
  { "rig.freq++",		"%d",	4, "250" 	},
  { WS_HSEP, "Trace Displays" },
  { "msPre",			"%f",	4, "0.5"	},
  { "msPost",			"%f",	4, "1"	},
  { "maxTraces",		"%d",	2, "50"	},
  { "autoscale",		"%b",	1, "1"	},
  { WS_VSEP, "Histogram Windows" },
  { "trace.psth_bin (ms-wd)",	"%f",  	6, "1.0" },
  { "trace.isih_bin (ms-wd)",	"%f",  	6, "0.25" },
  { "trace.isih_max (ms)",	"%f",  	6, "20" },
  { "trace.decay (#trials)",	"%d",  	4, "60" },
#ifdef __tdtproc__
  { WS_VSEP, "LED Parameters" },
  { "use_led",			"%b",	1, "0"	},
  { "led_delay (ms)",	"%d",	4, "100"	},
  { "led_dur (ms)",		"%d",	4, "100"	},
#endif /* __tdtproc__ */
#ifdef __test__
  { WS_VSEP, "Testing Flags" },
  { "use_atten",		"%b",	1, "1"	},
  { "use_eq",		"%b",	1, "1"	},
#endif __test__
  { NULL },
};

static Field ws_expinfo[] = {
  { WS_VSEP, "This Animal.." },
  { "Animal",			"%s",	5, "666"	},
  { "Unit",			"%s",	2, "00"	},
  { "Filenum",			"%d",	2, "00"	},
  { WS_VSEP, "This Pass.." },
  { "Pass",			"%d",	5, "1"	},
  { "AP (mm)",			"%f",	5, "0.0"	},
  { "ML (mm)",			"%f",	6, "0.0"	},
  { "DV (um)",			"%d",	5, "00000"	},
  { "Side (L/R)",		"%s",	1, "R"	},
  { NULL },
};


static Field StandardParams[] = {
  { "file.version",      "%s", 60 },
  { "ts.fix_Stim",       "%s", 60 },
  { "ts.fix_bc (%)",     "%d", 3 },
  { "ts.fix_iid (dB)",   "%d", 3 },
  { "ts.fix_itd (us)",   "%d", 4 },
  { "ts.fix_abi (dB)",   "%d", 3 },
  { "ts.use_dur_delay",  "%b", 1 },
  { "ts.dur (ms)",            "%d", 4 },
  { "ts.delay (ms)",          "%d", 4 },
  { "Animal",		"%s",	3 },
  { "Unit",		"%s",	2 },
  { "Pass (#)",		"%d",	5 },
  { "AP (um)",		"%f",	5 },
  { "ML (um)",		"%f",	5 },
  { "DV (um)",		"%f",	5 },
  { "Side",		"%s",	1 },
  { "Reps (#)",		"%d",	2 },
  { "Dur (ms)",		"%d",	4 },
  { "Delay (ms)",	"%d",	4 },
  { "Epoch (ms)",	"%d",	4 },
  { "ISI (ms)",		"%d",	4 },
  { "Rise (ms)",	"%d",	2 },
  { "Fall (ms)",	"%d",	2 },
  { "Spont_Stims",	"%b",	1 },
  { "daFc (Hz)",	"%f",	6 },
  { "adFc (Hz)",	"%f",	6 },
  { "evtFc (Hz)",	"%f",	6 },
  { "randomize",	"%d",	1 },
  { "version", 		"%s",   60 },
  { "time",		"%s",   60 },
  { "timestamp",	"%d",   12 }, /* new with version 2.24 */
  { "noise.low",	"%d",	12 }, /* recorded with version 2.26 */
  { "noise.high",	"%d",	12 }, /* recorded with version 2.26 */
  { "tone.rad_vary",	"%d",	12 },
  { "stack.rms",		"%b",	1	},
  { "xdphys.caltype",	"%s",	60 },
  { "detect.mode",	"%s",	10 },
  { "xdphys.gsrnmodels",	"%d",	2 },
  { "xdphys.fname",	"%s",	32 },
  { "ana.channel1 (on,off)","%s",	3 },
  { "ana.channel2 (on,off)","%s",	3 },
  { "detect.discrim_channel", "%d",	1 },
  { "ana.nchans","%d",	1 },
  { "ana-decimate (by)", "%d", 3 },
  { "ana.every (nth)",   "%d", 2 },
  { "prefs.page",   "%d", 1 },
#ifdef __tdtproc__
  { "use_led (ms)",   "%b", 1 },
  { "led_dur (ms)",   "%d", 4 },
  { "led_delay",   "%d", 4 },
#endif /* __tdtproc__ */
  { NULL },
};

static void lock_worksheets(void)
{
	ws_lock(ws_prefs,True);
	ws_lock(ws_expinfo,True);
}

void unlock_most_worksheets(void)
{
	ws_lock(ws_prefs,False);
	ws_lock(ws_expinfo,False);
}


static char *CopyrightMessage()
{
  static char *Copyright = "\
xdphys: Dichotic Physiology Data Acquisition\n\
Copyright (c) 1993-2000 Jamie Mazer, Ben Arthur, Chris Malek\n\
<lab@etho.caltech.edu>\n\
This program is NOT for general distribution\n";
  return(Copyright);
}

static void AboutCB(Widget w, XtPointer client_data, XtPointer call_data)
{
  char tmp[64];

  sprintf(tmp, "Version %s", XDPHYS_VERSION);
  pop_box(CopyrightMessage(), tmp, NULL, NULL);
}

#if SANAL
static void FDOExternCB(Widget w, XtPointer opt_wild_blah, XtPointer call_data)
{
  char *opt_wild = (char*)opt_wild_blah;
  char p[MAXPATHLEN + 100];
  
  if (opt_wild == NULL)
    sprintf(p,"exec xdview '%s.%s.*.*' &",
	    GS("animal"), next_alphanum_counter(GS("unit"), 0));
  else
    sprintf(p, "exec xdview '%s' &", opt_wild);
  system_noblock(p);
}
#endif /* SANAL */

static void savestate(int bell)
{
  static char tmp[20];
  char *p;

  ws_tosvars(NULL,NULL);
  saveGeometry();
  savevars(statefile, global_svar_table);
  p = GS("Animal");
  sprintf(tmp, "~/.xdphysrc/xd_%s", p);
  savevars(tmp, global_svar_table);
  if (bell)
    beep(0);
}


static void SaveStateCB(Widget w, XtPointer client_data, XtPointer call_data)
{
  savestate(1);
}

static void Quit(Widget w, XtPointer client_data, XtPointer call_data)
{
  if (progRunning) {
    beep(0); }
  else {
  	savePrefs(GI("prefs.page"));
    SaveStateCB(w, NULL, NULL);
    clockbar_save();
    unlockApp();
    is_shutdown();
    varsdelete(global_svar_table);
    getpref(NULL, NULL);
    XtCloseDisplay(XtDisplay(TopLevel));
    cleanupParms();
    cache_delete_all();
    exit(0); }
}

static void QuitClockbar(Widget w, XtPointer client_data, XtPointer call_data)
{
  if (progRunning) {
    beep(0); }
  else {
    SaveStateCB(w, NULL, NULL);
    clockbar_save();
    system_noblock("exec xclockbar &");
    unlockApp();
    is_shutdown();
    varsdelete(global_svar_table);
    getpref(NULL, NULL);
    if (w)
      XtCloseDisplay(XtDisplay(TopLevel));
    cleanupParms();
    cache_delete_all();
    exit(0); }
}

static char *datafile_name(char *animal, char *unit, int filenum, char *suffix)
     /* unit SHOULD be of strlen 2, i.e. from next_alphanum_counter */
{
  char buf[MAXPATHLEN];

  sprintf(buf, "%s.%s.%02d.%s", animal, unit, filenum, suffix);
  return(strsave(buf));
}

void perc_done(void)
{
	float f;
	time_t elapsed, remaining;
	static time_t start_time;
	extern time_t time();

	if (cur_trialnum < 0 || cur_trialnum == cur_ntrials) {
		statline_set(status1, "");
		statline_set(status2, "");
		percdone(0.0);
	} else {
		f = (float) cur_trialnum / (float) cur_ntrials;
		percdone(f);
		if (cur_trialnum == 0) {
			start_time = time(NULL);
		} else {
			elapsed = time(NULL) - start_time;
			remaining = elapsed / f - elapsed;
			if (status2) {
				if (remaining / 60)
					statline_set(status2,
						     "%d:%02d left. r%d of %d %20s",
						     remaining / 60,
						     remaining % 60,
						     cur_repnum + 1,
						     cur_nreps, " ");
				else
					statline_set(status2,
						     "0:%02d left. r%d of %d %20s",
						     remaining % 60,
						     cur_repnum + 1,
						     cur_nreps, " ");
			}

		}
	}
}

int update_displays(spike_t *spikelist, int decay_flag)
{
  int from, to, i;
  static void *previous_us_time = NULL;
  int ms_elapsed_time = -1;

  /* Delay and Dur are in milliseconds */
#ifndef __tdtproc__
  from = (i = GI("Delay")) / 1000.0 * is_evtFc;
  to = (i + GI("Dur")) / 1000.0 * is_evtFc;
#else /* __tdtproc__ */
  from = (i = GI("Delay")) / 1000.0 * is_adFc;
  to = (i + GI("Dur")) / 1000.0 * is_adFc;
#endif /* __tdtproc__ */

  if (previous_us_time != NULL)
    ms_elapsed_time = (int) (0.5 + ustime_elapsed(previous_us_time) / 1000.0);
  previous_us_time = ustime_now();

  tracer_update(main_tracer, spikelist, NULL, 0, from, to - 1, decay_flag);

  if (ms_elapsed_time >= 0) {
    statline_set(headline, "isi = %d ms", ms_elapsed_time);
  }
  return(ms_elapsed_time);
}

static void delimitLogEntry(FILE *fp)
{
  int d;

  for (d = 70; d >= 0; d--) {
    fputs(d ? "-" : "-\n", fp);
  }
}

#if(0)
static int append_files(char *datafile, int append)
{
	FILE *fp, *fporig, *fpapp;
	int nrasters_orig, nrasters_app, Reps_orig, Reps_app;
	char buf[1000], Range_orig[128], Range_app[128];

	assert((append == APPEND_REPS) || (append == APPEND_RANGE));
	strcpy(buf, strcat(findhome(datafile), ".orig"));
	rename(findhome(datafile), buf);
	if ((fp = fopen2(findhome(datafile), "w")) == NULL) {
		alert("Can't append datafile.");
		return (0);
	}
	if ((fporig = fopen2(strcat(findhome(datafile), ".orig"), "r")) == NULL) {
		alert("Can't append datafile.");
		return (0);
	}
	if ((fpapp = fopen2(strcat(findhome(datafile), ".app"), "r")) == NULL) {
		alert("Can't append datafile");
		return (0);
	}

	while (strncmp(fgets(buf, sizeof(buf), fporig), "END_COMMENTS", 12))
		fprintf(fp, "%s", buf);
	fprintf(fp, ";;This file has been appended.  Old params, now changed:\n");

	if (append == APPEND_RANGE) {
		while (strstr(buf, "StandardParams") == NULL) {
			if (strstr(buf, "Range=") != NULL)
				fprintf(fp, ";;  %s", buf);
			fgets(buf, sizeof(buf), fporig);
		}
	} else {
		while (strncmp(fgets(buf, sizeof(buf), fporig), "Reps=", 5));
		fprintf(fp, ";;  %s", buf);
	}

	while (strncmp(fgets(buf, sizeof(buf), fporig), "time=", 5));
	fprintf(fp, ";;  %s", buf);
	while (strncmp(fgets(buf, sizeof(buf), fporig), "timestamp=", 10));
	fprintf(fp, ";;  %s", buf);
	while (strncmp(fgets(buf, sizeof(buf), fporig), "nrasters=", 9));
	fprintf(fp, ";;  %s", buf);

	while (strncmp(fgets(buf, sizeof(buf), fpapp), "PARAMS", 6));
	fprintf(fp, "END_COMMENTS\n%s", buf);

	if (append == APPEND_RANGE) {
		rewind(fporig);
		while (strncmp(fgets(buf, sizeof(buf), fporig), "PARAMS", 6));
		fgets(buf, sizeof(buf), fpapp);
		while (strstr(buf, "StandardParams") == NULL) {
			if (strstr(buf, "Range=") == NULL)
				fprintf(fp, "%s", buf);
			else {
				sscanf(1 + strchr(buf, '='), "%s", Range_orig);
				while (strstr
				       (fgets(buf, sizeof(buf), fporig),
					"Range=") == NULL);
				sscanf(1 + strchr(buf, '='), "%s", Range_app);
				fprintf(fp, "Range=%s\n",
					StimArray_Combine(Range_orig, Range_app));
/*
        fwrite(buf,sizeof(char),-1+strlen(buf),fp);
        sscanf(1+strchr(buf,'='),"%s",Range_orig);
        while(strstr(fgets(buf,sizeof(buf),fporig),"Range=")==NULL);
        sscanf(1+strchr(buf,'='),"%s",Range_app);
        if(StimArray_Equivalent(Range_orig,Range_app))
          fprintf(fp,"\n");
        else
          fprintf(fp,",%s",1+strchr(buf,'='));
*/
			}
			fgets(buf, sizeof(buf), fpapp);
		}
		fprintf(fp, "%s", buf);
	}
	/* to get StandardParams line */
	while (strncmp(fgets(buf, sizeof(buf), fpapp), "Reps=", 5)) {
		fprintf(fp, "%s", buf);
	}

	if (append == APPEND_REPS) {
		sscanf(buf, "Reps=%d", &Reps_app);
		rewind(fporig);
		while (strncmp(fgets(buf, sizeof(buf), fporig), "PARAMS", 6));
		while (strncmp(fgets(buf, sizeof(buf), fporig), "Reps=", 5));
		sscanf(buf, "Reps=%d", &Reps_orig);
		fprintf(fp, "Reps=%d\n", Reps_orig + Reps_app);
	} else {
		fprintf(fp, "%s", buf);
	}

	while (strncmp(fgets(buf, sizeof(buf), fpapp), "nrasters=", 9)) {
		fprintf(fp, "%s", buf);
	}
	sscanf(buf, "nrasters=%d", &nrasters_app);
	while (strncmp(fgets(buf, sizeof(buf), fporig), "nrasters=", 9));
	sscanf(buf, "nrasters=%d", &nrasters_orig);
	fprintf(fp, "nrasters=%d\n", nrasters_orig + nrasters_app);

	while (strncmp(fgets(buf, sizeof(buf), fporig), "END_RASTERDATA", 14)) {
		fprintf(fp, "%s", buf);
	}
	while (strncmp(fgets(buf, sizeof(buf), fpapp), "END_RASTERDATA", 14)) {
		fprintf(fp, "%s", buf);
	}
	fprintf(fp, "%s", buf);	/* to get END_RASTERDATA */

	fclose(fp);
	fclose(fporig);
	fclose(fpapp);

	unlink(strcat(findhome(datafile), ".orig"));
	unlink(strcat(findhome(datafile), ".app"));

	return (1);
}
#endif

int default_display_fn(char *datafile)
{
	if (getenv("XDPHYS_NO_GRAPH")==NULL)
		FDObj_File_New(datafile, NULL, NULL); 
  return(1);
}

static char *appRun(int temp_run, RunData *info, Widget status)
{
	char *datafile, *cwd,*comments, *p;
	FILE *datafp;
	FILE *logfp;
	int scan_result;
	int i;
	int filenum;
	char *ret_val=NULL;
	char buf[100];

	if (temp_run == TEMP_YES) {
		assert((datafile=(char*)calloc(128,sizeof(char)))!=NULL);
		assert(strlen(cwd = getcwd(NULL,128))<100);
		strcpy(datafile,cwd);
		free(cwd);
		strcat(datafile, "/temp.");
		strcat(datafile, info->suffix);
	} else if (temp_run == TEMP_NAMED) {
		sprintf(buf, "%d.%d.0.%s", GI("Pass"), GI("DV"), info->suffix);
		datafile = fileBox(buf, "w", "Save to ...");
		if (datafile == NULL)
			return(NULL);
	} else {  /* TEMP_NO */
		filenum = GI("filenum");
		p = datafile_name(GS("animal"), next_alphanum_counter(GS("unit"), 0),
			filenum, info->suffix);
		datafile = fileBox(p, "w", "Save to ...");
		free(p);
		if (datafile == NULL)
			return(NULL);
	}

	SS("xdphys.fname",datafile);
	SI("ana.nchans",spiker_getAnalogNchans());

	if(GI("Comments")) {
		if ((temp_run==TEMP_NO) || (temp_run==TEMP_NAMED)) {
			comments = getComments(datafile);
			if(comments == NULL) {
				return(NULL); } }
		else {
			comments = NULL; } }

#if(0)
	if(probefile(datafile) && (temp_run==TEMP_NO)) {
		if((fd=FD_read(datafile,0))==NULL) {
			alert("FD_read failed.");
			return(NULL); }
		mis1=ws_check_params(fd,info->params,info->nparams,msg);
		mis2=ws_check_params(fd,StandardParams,WS_N(StandardParams),msg);
		if((mis1==MISMATCH_YES) || (mis2==MISMATCH_YES)) {
			alert(msg);
			return(NULL); }
		if((mis1==MISMATCH_RANGE) && (mis2==MISMATCH_REPS)) {
			strcpy(msg,"Both Range and Reps are different, when only one can be.\n");
			strcat(msg,"When appending files, the Range and Reps parameters must\n");
			strcat(msg,"be set such that each datapoint retains the same total\n");
			strcat(msg,"number of repetitions.");
			alert(msg);
			return(NULL); }
		if(mis1==MISMATCH_RANGE)
			append=APPEND_RANGE;
		else
			append=APPEND_REPS;

		if ((datafp = fopen2(strcat(findhome(datafile),".app"), "w")) == NULL) {
			alert("Can't write datafile, try again");
			return(NULL); } }
	else {
#endif
		if ((datafp = fopen2(findhome(datafile), "w")) == NULL) {
			alert("Can't write datafile, try again");
			return(NULL); }
#if(0)
		append=APPEND_NO; }
#endif

	recording(0);
	for (i=0; i<is_getADnchans(); i++) {
		sprintf(buf, "ana.channel%d",i+1);
		if (strncmp(GS(buf),"on",2)==0) {
			recording(1);
			break;
		}
	}

	sprintf(buf, "%s", FILE_FORMAT_VERSION);
	SS("file.version",buf);

	WriteFileHeader(datafp, info, comments);
	WriteBeginRaster(datafp);

	tracer_clear(main_tracer);	/* clear hists and raster before run. */

	/* ----------------------------------------------------------- */
	/* Do the trials                                               */
	/* ----------------------------------------------------------- */


	this_run = datafile;
	idle_set(dloop_empty);
	scan_result = (*info->scan_fn)(datafp, status);
	this_run = NULL;
	/* ----------------------------------------------------------- */

	if (datafp) {
		WriteEndRaster(datafp);
		if (scan_result <= 0)
			fprintf(datafp, ";; NOTE: run aborted or failed!!\n");
		fclose(datafp);
	}

	ws_tosvars(NULL,NULL);		/* in case of any changes during run.. */

	if (scan_result > 0) {
		/* Successful run completed, save the file */
		if ((temp_run==TEMP_NO) /*&& (append==APPEND_NO)*/) {
			/* only update filenum if it's NOT a temp_run 12-mar-94 */
			SI("filenum", filenum + 1);
			ws_tows(NULL,NULL);			/* update the filenum field */
		}

		if ((logfp = fopen2(AUTOLOG, "a")) != NULL) {
			delimitLogEntry(logfp);
			fprintf(logfp, "%s\n** file \"%s\" [type=%s] COMPLETED\n",
				GS("time"), basename(datafile), info->suffix);
			if(GI("Comments") && comments)
				fputs(comments, logfp);
			fclose(logfp); 
		}

    ret_val = datafile;

	} else if (scan_result == 0 && (temp_run!=TEMP_YES)) {
		/* Run was aborted, if temp_run, then delete the file;
		 * otherwise, give the user some options..  */
		i = pop_box("Aborted, what do to with partial data file?",
			"Delete", "Save", "Save and Plot");
		switch (i) { 
			case 1:
				unlink(datafile);

				if ((logfp = fopen2(AUTOLOG, "a")) != NULL) {
					delimitLogEntry(logfp);
					fprintf(logfp, "%s\n** file \"%s\" [type=%s] DELETED\n",
						GS("time"), basename(datafile), info->suffix);
					fclose(logfp); 
				}
				break;
			case 3:
				if (temp_run==TEMP_NO) {
					SI("filenum", filenum + 1);
					ws_tows(NULL,NULL);		/* update the filenum field */
				}

				if ((logfp = fopen2(AUTOLOG, "a")) != NULL) {
					delimitLogEntry(logfp);
					fprintf(logfp, "%s\n** file \"%s\" [type=%s] ABORTED & PRESERVED\n", GS("time"), basename(datafile), info->suffix);
					if (GI("Comments") && comments)
						fputs(comments, logfp);
					fclose(logfp); 
				}

				ret_val = datafile;
				break;
			case 2:
				if (temp_run==TEMP_NO) {
					SI("filenum", filenum + 1);
					ws_tows(NULL,NULL);		/* update the filenum field */
				}
				if ((logfp = fopen2(AUTOLOG, "a")) != NULL) {
					delimitLogEntry(logfp);
					fprintf(logfp, "%s\n** file \"%s\" [type=%s] ABORTED & PRESERVED\n", GS("time"), basename(datafile), info->suffix);
					if (GI("Comments") && comments)
						fputs(comments, logfp);
					fclose(logfp);
				}
				break; 
		} 
	} else 
			unlink(datafile);

	if (GI("Compress")) {
		char buf[MAXPATHLEN + 100];
		/*fprintf(stderr, "Warning: compressing \"%s\" in background", datafile);*/
		sprintf(buf, "exec gzip %s", datafile);
		system(buf); 
	}

	return(ret_val);
}

static void LogView(Widget w, XtPointer client_data, XtPointer call_data)
{
  if (probefile(AUTOLOG))
    pop_file_at("LogView", AUTOLOG, 1);
  else
    notify("Can't open \"%s\"", AUTOLOG);
}

static void LogEntry(Widget w, XtPointer client_data, XtPointer call_data)
{
  char logbuf[1000];
  char *p;
  FILE *logfp;

  sprintf(logbuf, "%s :: LOG ENTRY\n", GS("time"));
  if ((p = pop_text("Log Entry", logbuf, sizeof(logbuf) - 1, 1)) != NULL) {
    if (logbuf[strlen(logbuf) - 1] != '\n')
      strcat(logbuf, "\n");
    if ((logfp = fopen2(AUTOLOG, "a")) != NULL) {
      delimitLogEntry(logfp);
      fputs(p, logfp);
      fclose(logfp);
    } else {
      alert("Can't open logfile: %s", AUTOLOG);
    }
  }
}

static void PlotCalibData(Widget w, XtPointer client_data, XtPointer call_data)
{
	plot_calib_data();
}

static void ReloadCalibData(Widget w, XtPointer client_data,XtPointer call_data)
{
	load_calib_data(GS("cal_file"), CAL_RELOAD_YES);
}

/*-----------------------------------------------------------------------
 * CalcRMS
	makes calculation on the analog channel pointed to by the 
	detect.discrim_channel svar
  -----------------------------------------------------------------------*/
static void CalcRMS(Widget w, XtPointer client_data, XtPointer call_data)
{
	xword **anabufs;
	xword *ibuf;
	int count;
	int epoch;
	double rms;
	int i;
	char txtbuf[5000], tmp[500];
	float mv;
	double timestamp;
	int nchans = is_getDAnchans();
	int nsamps;

	ws_tosvars(NULL, NULL);
	epoch = GI("epoch");

	is_init(0.0, GF("adFc"), 0.0, epoch, epoch);

	if (!is_sample(X_AD, epoch, &timestamp)) {
		alert("is_sample failed");
		return;
	}

	spiker_getAnalogBuffers(&nsamps, &anabufs);
	ibuf = anabufs[(GI("detect.discrim_channel")-1)];
	free(anabufs);

	dsp_stripDC(ibuf, count, nchans);

	for (rms = 0.0, i = 0; i < count; i++)
		rms += (ibuf[i * nchans] * ibuf[i * nchans]);
	rms = sqrt((rms / (double) count));

	if ((GI("detect.discrim_channel")-1))
		ad_to_mv((float) rms, NULL, &mv, AddaRelative);
	else
		ad_to_mv((float) rms, &mv, NULL, AddaRelative);
	sprintf(tmp, "RMS Voltage on analog input line = %duV\n",
		(int) (mv * 1000.0));
	strcpy(txtbuf, tmp);

	pop_text("RMS", txtbuf, strlen(txtbuf), False);
}

static void WriteBeginRaster(
	FILE	*datafp)
{
  fprintf(datafp, "RASTERDATA\n");
}

static void WriteEndRaster(
	FILE	*datafp)
{
    fprintf(datafp, "END_RASTERDATA\n");
}

static void WriteFileHeader(FILE *datafp, RunData *info,char *comments)
{
	float tomv, offset;
	int i;
	char tmp[32];

#ifdef __tdtproc__
	/* This is a hack! */
	int epoch = GI("epoch");
	is_init(GF("daFc"), 0.0, 0.0, epoch, epoch);
	ws_tows(NULL, NULL);
	is_shutdown();

#endif				/* __tdtproc__ */

	fprintf(datafp, ";; xdphys %s ver %s\n", info->suffix, XDPHYS_VERSION);

	/* Comment Block */
	fprintf(datafp, "COMMENTS\n");
	if (debugflag) {
		fprintf(stderr, "comments=<%s>\n", comments ? comments : "NULL");
	}
	if (GI("Comments"))
		writeComments(comments, datafp);
	fprintf(datafp, "END_COMMENTS\n");

	/* Parameter Block */
	fprintf(datafp, "PARAMS\n");
	ws_write(datafp, info->params, info->nparams, NULL);
	if (info->depvar)
		fprintf(datafp, "depvar=%s\n", info->depvar);
	fprintf(datafp, "; StandardParams\n");
	ws_write(datafp, StandardParams, WS_N(StandardParams), NULL);

	for (i = 0; i < is_getADnchans(); i++) {
		sprintf(tmp,"ana.channel%d",i+1);
		if (strncmp(GS(tmp),"on",2)==0) {
			get_analog_conversion(&tomv, &offset, i);
			fprintf(datafp, "ana-tomv.chan%d=%f\n", i+1, tomv);
			fprintf(datafp, "ana-offset.chan%d=%f\n", i+1, offset);
		}
	}

	fprintf(datafp, "END_PARAMS\n");
}

static void ModuleStartFn(Widget w, XtPointer info_blah,
     XtPointer temp_run_blah)
{
  RunData *info = (RunData*)info_blah;
  int temp_run = (int)temp_run_blah;
  char *app_result;

	if (!progRunning) {
		if (ws_verify(NULL)) {
			if (load_calib_data(GS("cal_file"), CAL_RELOAD_NO)) {

				lock_worksheets();
				ws_tosvars(NULL,NULL);
				savevars(statefile, global_svar_table);
				ReverseColors(w);
				set_progRunning(True);
  
				if (info->pre_fn != NULL)
					(*info->pre_fn)();
  
				if (info->scan_fn)
					app_result = appRun(temp_run, info, status1);
				else
					alert("No scan function available!");

				is_shutdown();
				fflush(stderr);

				/*	perc_done(-1, -1); */
  
				if (app_result && (info->post_fn != NULL))
					(*info->post_fn)(app_result);

				if (app_result && (info->display_fn != NULL))
					(*info->display_fn)(app_result);

				fflush(stderr);
  
        if(app_result==NULL)
          unlock_most_worksheets();
  
				setRack(0, attMaxAtten, attMaxAtten);
				set_progRunning(False);
				ReverseColors(w);
        FREE(app_result); } } }
	else {
		beep(0); }
}

static void ModuleStartCB(Widget w, XtPointer info_blah, XtPointer call_data)
{
  RunData *info = (RunData*)info_blah;

  ModuleStartFn(w, info, TEMP_NO);
}

static void ModuleCloseCB(Widget w, XtPointer slot_blah, XtPointer call_data)
{
  DockSlot *slot = (DockSlot*)slot_blah;

#ifndef MOD_MENU
  XtVaSetValues(slot->dock, XtNstate, 0, NULL);
  popdown_dockslot(slot);
#endif /* MOD_MENU*/
}

static void NumberButtonCB(Widget w, XtPointer num_holder, XtPointer call_data)
{
	int cur_num = GI("prefs.page");
	int num = (int) num_holder;

	if (cur_num != num) {
		savePrefs(cur_num);
		loadPrefs(num);
	}
	SI("prefs.page",num);

}


static void loadPrefs(int num)
{
	char fname[40];
	SVAR_TABLE *tmp_svar_table;

	sprintf(fname, "~/.xdphysrc/xdphys.prefs.%d", num);
	tmp_svar_table = varsnew(0, 100);
	if (!loadvars(fname, tmp_svar_table)) {
		/*savePrefs(num);*/
		return;
	}

	ws_tows(NULL, tmp_svar_table);
	ws_tosvars(NULL, NULL);
	varsdelete(tmp_svar_table);
}

static void savePrefs(int num)
{
	char fname[40];

	sprintf(fname, "~/.xdphysrc/xdphys.prefs.%d", num);
	ws_writeconfig(fname, ws_prefs, sizeof(ws_prefs) / sizeof(Field), NULL);
}

static void HelpCB(Widget w, XtPointer name_blah, XtPointer call_data)
{
  char *name = (char*)name_blah;

  char tmp[100],*file;

  if(name!=NULL)
    sprintf(tmp,"$(XDPHYSDIR)/doc/xdphys.%s.doc",name);
  else
    sprintf(tmp,"$(XDPHYSDIR)/doc/xdphys.doc");
  file = envexpand(tmp);
  if (!probefile(file)) {
    alert("Can't find help file.");
    return; }
  help(file,(name==NULL) ? "xdphys" : name);
}

static void ModuleTempCB(Widget w, XtPointer info_blah, XtPointer call_data)
{
  RunData *info = (RunData*)info_blah;

  ModuleStartFn(w, info, (XtPointer)TEMP_YES);
}

static void ModuleNamedTempCB(Widget w, XtPointer info_blah,
      XtPointer call_data)
{
  RunData *info = (RunData*)info_blah;

  ModuleStartFn(w, info, (XtPointer)TEMP_NAMED);
}

void *make_module_info(
     char *depvar,		/* name of dependent variable (eg "itd") */
     char *suffix,		/* suffix for file-type i.d. */
     char *helpfile,	/* search string for help file */
     int (*pre)(),		/* call before scan (i.e. to LOCK ws) */
     int (*scan)(),		/* actually collect the data.. */
     int (*post)(),		/* call after scan (i.e. to UNLOCK ws) */
     int (*display)(),
     FieldList params,		/* parameters for writing headers */
     int nparams)		/* number of parameters */
{
  RunData *info;

  info = (RunData *) malloc(sizeof(RunData));
  info->depvar = depvar;
  info->suffix = suffix;
  info->helpfile = helpfile;
  info->pre_fn = pre;
  info->scan_fn = scan;
  info->post_fn = post;
  info->display_fn = display;
  info->params = params;
  info->nparams = nparams;
  return((void *)info);
}

static Widget make_close_help(Widget handle, XtPointer name_blah,
      XtPointer slot_blah)
{
  char *name = (char*)name_blah;
  DockSlot *slot = (DockSlot*)slot_blah;
  Widget w1, w2;

  w1 = button_new(XtParent(handle),"Close","Close",
      ModuleCloseCB, slot,handle,NULL);
  accel_from(XtParent(handle), w1);

  w2 = button_new(XtParent(handle),"Help","Help", HelpCB,
        name,w1,NULL);
  accel_from(XtParent(handle), w2);

  return(w2);
}

static void make_numbered_buttons(Widget handle)
{
	Widget w1, w2, w3;

	w1=toggle_new(XtParent(handle), "1", "1", handle, NULL,
		   NULL, NumberButtonCB, 1, True);
	w2=toggle_new(XtParent(handle), "2", "2", w1, NULL,
		   w1, NumberButtonCB, 2, False);
	w3=toggle_new(XtParent(handle), "3", "3", w2, NULL,
		   w2, NumberButtonCB, 3, False);
	/*
	w1 = button_new(XtParent(handle), "1", "1", NumberButtonCB, 1, handle, NULL);
	accel_from(XtParent(handle), w1);

	w2 = button_new(XtParent(handle), "2", "2", NumberButtonCB, 2, w1, NULL);
	accel_from(XtParent(handle), w2);

	w3 = button_new(XtParent(handle), "3", "3", NumberButtonCB, 3, w2, NULL);
	accel_from(XtParent(handle), w3);
	*/
}


Widget make_start_temp(Widget handle, RunData *info, XtCallbackProc StartCB,
      XtCallbackProc TempCB, DockSlot *slot)
{
  Widget w1, w2, w3, w4, w5;

  w1 = button_new(XtParent(handle),"Close","Close", ModuleCloseCB,
                  slot,handle,NULL);
  accel_from(XtParent(handle), w1);

  w2 = button_new(XtParent(handle),"Help","Help", HelpCB,
                  info->helpfile,w1,NULL);
  accel_from(XtParent(handle), w2);

  w3 = button_new(XtParent(handle), "Start", "Start",
		  StartCB ? StartCB : ModuleStartCB,
		  info, w2, NULL);
  accel_from(XtParent(handle), w3);

  w4 = button_new(XtParent(handle), "Temp", "Temp",
		  TempCB ? TempCB : ModuleTempCB,
		  info, w3, NULL);
  accel_from(XtParent(handle), w4);

  w5 = button_new(XtParent(handle), "Named-Temp", "Named-Temp",
		  ModuleNamedTempCB, info, w4, NULL);
  accel_from(XtParent(handle), w5);

  return(w5);
}

static void NewUnit(Widget w, XtPointer client_data, XtPointer call_data)
{
  char *p;
  char *next_unit;

  if (progRunning) {
    beep(0);
    return;
  }
  next_unit = next_alphanum_counter(GS("unit"), 1);
  if ((p = pop_dialog("This will be unit:", next_unit)) == NULL)
    return;
  next_unit = next_alphanum_counter(p, 0);
  free(p);

  if ((p = pop_dialog("New Electrode Depth:", GS("DV"))) == NULL)
    return;

  SS("unit", next_unit);
  SI("DV", atoi(p));
  SI("filenum", 0);
  free(p);
  ws_tows(NULL,NULL);
  p = datafile_name(GS("animal"), next_alphanum_counter(GS("unit"), 0),
		    GI("filenum"), "....");
  notify("Unit-%s File-%02d at %dum: %s",
	 next_alphanum_counter(GS("unit"), 0),
	 GI("filenum"), GI("DV"), p);
  free(p);
  p = (char *)malloc(100);
  sprintf(p, "%s.%s.*.*", GS("animal"), next_alphanum_counter(GS("unit"), 0));
  ws_tows(NULL,NULL);
  free(p);
  clockbar_start_userClk();
}

/* backup a single datafile number (over-write last usually) */
static void PrevFile(Widget w, XtPointer client_data, XtPointer call_data)
{
  int next_filenum;

  if (progRunning) {
    beep(0);
    return;
  }

  next_filenum = GI("filenum") - 1;
  if (next_filenum < 0) {
    notify("That would make next file negative!");
  } else {
    SI("filenum", next_filenum);
    ws_tows(NULL,NULL);
  }
}

/* advance a single datafile number */
static void NextFile(Widget w, XtPointer client_data, XtPointer call_data)
{
  int next_filenum;

  if (progRunning) {
    beep(0);
    return;
  }

  next_filenum = GI("filenum") + 1;
  SI("filenum", next_filenum);
  ws_tows(NULL,NULL);
}

static int SaveTraceWaveform(char *ucom)
{
  xword *ibuf;
  int count;
  char fname[MAXPATHLEN];

  ibuf = is_getADbuffer(&count);
  if (ibuf) {
    sprintf(fname, "%s.%s.typ",
	    GS("animal"), next_alphanum_counter(GS("unit"), 0));
    if (!appendwave(fname, ibuf, is_adFc, count, 2, ucom, progname, 0.0, 0.0,
					NULL, 0))
      return(0);
    return(1);
  } else {
    return(0);
  }
}

static void SaveTraceCB(Widget w, XtPointer client_data, XtPointer call_data)
{
  char *com, *p;
  int count;
  xword *ibuf;

  if ((ibuf = is_getADbuffer(&count)) == NULL) {
    beep(0);
    return;
  }

  if (this_run) {
    com = malloc(strlen(this_run) + 20);
    sprintf(com, "%s #%d", basename(this_run), cur_depvar);
    if ((p = pop_dialog("Short Comment:", com)) != NULL) {
      if (SaveTraceWaveform(p))
	beep(0);
    }
    free(com);
    free(p);
  } else {
    if ((p = pop_dialog("Short Comment:", "")) != NULL) {
      if (SaveTraceWaveform(p))
	beep(0);
    }
    free(p);
  }
}

static void ChangeAnimal(Widget w, XtPointer animalid_blah, XtPointer call_data)
{
  char *animalid = (char*)animalid_blah;
  char *file, buf[20], *p;

  if (animalid != NULL) {
    sprintf(buf, "~/.xdphysrc/xd_%s", animalid);
    if (!loadvars(buf, global_svar_table))
      fprintf(stderr, "warning: can't load owl data for \"%s\"\007\n", animalid); }
  else if ((file = fileBoxMatch(findhome("~/.xdphysrc/"), "r",
				 "Expinfo from..", "xd_*", GF_NODIRS))) {
    if (!loadvars(file, global_svar_table)) {
      alert("Can't load \"%s\"", file); }
    else {
      ws_tows(NULL,NULL);
      p = (char *)malloc(100);
      sprintf(p, "%s.%s.*.*", GS("animal"), GS("unit"));
      free(p); } }
}

static void NewPass(Widget w, XtPointer client_data, XtPointer call_data)
{
  int this_pass;
  char *p, buf[100], *side, lr[100];
  float ml, ap;
  FILE *logfp;

  if (progRunning) {
    beep(0);
    return;
  }
  sprintf(buf, "%g,%g,%s", GF("ML"), GF("AP"), GS("Side"));
  if ((p = pop_dialog("New Electrode Position (ML,AP,l/r):", buf)) != NULL) {
    strcpy(lr, "");
    if (sscanf(p, "%f,%f,%s", &ml, &ap, lr) >= 2) {
      if (*lr && (*lr == 'l' || *lr == 'L' || *lr == 'r' || *lr == 'R'))
	side = lr;
      else
	side = GS("Side");
      if ((*side == 'l' || *side == 'L') && ml > 0.0) {
	alert("ML > 0 for LEFT hemisphere\nDidn't set new pass\nPlease fix.");
      } else if ((*side == 'r' || *side == 'R') && ml < 0.0) {
	alert("ML < 0 for RIGHT hemisphere\nDidn't set new pass\nPlease fix.");
      } else {
	this_pass = GI("pass") + 1;
	SI("pass", this_pass);
	SF("ML", ml);
	SF("AP", ap);
	SI("DV", 0);
	SS("Side", side);
	ws_tows(NULL,NULL);
	if ((logfp = fopen2(PASSLOG, "a")) != NULL) {
	  if (pop_box("Is this new a electrode?", "No", "Yes", NULL) == 2)
	    fprintf(logfp, "\n");
	  fprintf(logfp, "pass %d : %f ml %f ap %s\n",
		  GI("pass"), GF("ML"), GF("ap"), GS("side"));
	  fclose(logfp);
	} else {
	  alert("Can't open %s for logging", PASSLOG);
	}
	notify("Pass-%02d at %.2fml %.2fap (%s)",
	       GI("pass"), GF("ML"), GF("ap"), GS("side"));
      }
    } else {
      beep(0);
    }
    free(p);
  }
}

/*
static void updateSSButts(void)
{
  int i;

  if(!strcmp(GS("detect.mode"),"ss"))
    for(i=0; i<7; i++)
      XtSetSensitive(main_tracer->ssbutts[i], True);
  else
    for(i=0; i<7; i++)
      XtSetSensitive(main_tracer->ssbutts[i], False);
}

static void SetGsrCB(Widget w, XtPointer new_mode_blah, XtPointer call_data)
{
  GSRMode new_mode = (GSRMode)new_mode_blah;
  Boolean s;

  XtVaGetValues(w, XtNstate, &s, NULL);
  if (s)
    getRaster_setmode(new_mode);
  else
    getRaster_setmode(GSR_invalid);

  updateSSButts();
}
*/

static void NewUnitAction(Widget w, XEvent *event, String *params,
      Cardinal *num_params)
{
  NewUnit(NULL,NULL,NULL);
}

static void QuitAction(Widget w, XEvent *event, String *params,
      Cardinal *num_params)
{
  Quit(NULL,NULL,NULL);
}

static void ToDockAction(Widget w, XEvent *event, String *params,
      Cardinal *num_params)
{
  static int dockpos = -1;
  Widget d;

  if (*num_params == 1 && strncasecmp(params[0], "prev", 4) == 0) {
    d = dock_nth(tooldock, dockpos - 1, &dockpos);
  } else {
    d = dock_nth(tooldock, dockpos + 1, &dockpos);
  }
  XWarpPointer(XtDisplay(d), None, XtWindow(d), 0, 0, 0, 0, 0, 0);
}

static void install_actions(void)
{
  static XtActionsRec actionTable[] = {
    { "quit",         QuitAction         },
    { "pause",        PauseAction        },
    { "newunit",      NewUnitAction      },
/*  { "cleartracer",  ClearTracerAction  }, */
    { "todock",       ToDockAction       },
   };

  XtAppAddActions(XtWidgetToApplicationContext(TopLevel),
		  actionTable, XtNumber(actionTable));
}

static int xdphysmain(int ac, char **av, char *appname, char *classname)
{
  int arg, i;
  Widget mainform, w, dockform, h;
/*
  Widget sd_form, sd_button, wform;
*/
  char	tmp[64];
#ifdef MOD_MENU
  MENU_ENTRY *mod_menu = NULL;
#endif /* MOD_MENU */

  char *ssfile = NULL;

  static MENU_ENTRY FileMenu[] = {
    { "File Menu",   NULL },
    { "Quit w/ clk", QuitClockbar,    (void *)NULL },
    { MENU_SEP },
    { "Save State",	 SaveStateCB,     (void *)NULL },
    { "Save Trace",	 SaveTraceCB,     (void *)NULL },
    { MENU_SEP },
    { "Log View",	   LogView,         (void *)NULL },
    { "Log Entry",   LogEntry,        (void *)NULL },
    { "Plot Cal",    PlotCalibData,   (void *)NULL },
    { "Reload Cal",  ReloadCalibData, (void *)NULL },
    { "Calc RMS",    CalcRMS,         (void *)NULL },
    { MENU_SEP },
    { "About",       AboutCB,         (void *)NULL },
    { NULL }
  };

  static MENU_ENTRY SetMenu[] = {
    { "Set Menu", NULL },
    { "debugflag",	MENU_NOCB,	(void *)&debugflag 		},
#ifndef __linux__
    { "single_step",	MENU_NOCB,	(void *)&single_step_mode	},
#endif
    { "view_output",	view_output_init,(void *)NULL			},
    { "view_input",	view_input_init,(void *)NULL			},
#if(0)
#ifndef __linux__
    { MENU_SEP },
    { "sync_pulse",	MENU_NOCB,	(void *)&is_syncPulse		},
#endif /* __linux__ */
#endif
    { NULL }
  };

  static MENU_ENTRY NewMenu[] = {
    { "New Menu", NULL },
    { "Prev File",	PrevFile,	(void *)NULL 			},
    { "Next File",	NextFile,	(void *)NULL 			},
    { "Unit",		NewUnit,	(void *)NULL 			},
    { "Pass",		NewPass,	(void *)NULL			},
    { "Animal",		ChangeAnimal,	(void *)NULL			},
    { NULL }
  };

  static MENU_ENTRY ForkMenu[] = {
    { "Fork Menu", NULL },
    { "xdview",		FDOExternCB,	(void *)"*"			},
    { NULL }
  };

  /* sigfatal_setup();*/

  progname = av[0];
  makeXDphysrc();
  initsvars();

  if (!loadvars(statefile, global_svar_table))
    fprintf(stderr, "warning: can't find config file \"%s\"\n", statefile);

  SS("caltype", "? None ?");

  install_xdphys_plotMethods();
  install_xcalibur_plotMethods();

  startx(&ac, av, appname ? appname : "xdphys", classname ? classname : "XDphys");
  install_actions();

#define isarg(pat) (strncmp(av[arg], pat, strlen(av[arg])) == 0)
  for (arg = 1; arg < ac; arg++) {
    if (isarg("-ov")) {
      unlockApp();
    } else if (isarg("-loadss")) {
      ssfile = av[++arg];
    } else if (isarg("-animal")) {
      if (av[++arg] != NULL)
	ChangeAnimal(NULL, av[arg], NULL);
      else {
	fprintf(stderr, "%s: -animal must be followed by animal-id\n", progname);
	exit(1);
      }
    } else if (isarg("-help") || isarg("-usage")) {
      usage(progname, NULL);
      exit(0);
    } else if (*av[arg]) {
      usage(progname, av[arg]);
    }
  }
#undef isarg
  
  lockApp();

  mainform = form_new(TopLevel, "main", 0);

  headline = statline_new(mainform, 40);
  XtVaSetValues(headline, XtNfromVert, NULL, NULL);

  w = NULL;

#ifdef mc700
  w = button_new(menubar, "Run-Rig", "Run-Rig", (XtCallbackProc) RunRig, 
      0, w, NULL);
#endif
  w = button_new(mainform,"quit","Quit", Quit, NULL, w, headline);
  w = button_new(mainform,"help","Help", HelpCB, NULL, w, headline);
  w = menu_new(mainform, "File", FileMenu, w, headline);
  w = menu_new(mainform, "Set", SetMenu, w, headline);
  w = menu_new(mainform, "New", NewMenu, w, headline);
  w = menu_new(mainform, "Fork Off", ForkMenu, w, headline);

#ifdef MOD_MENU
  module_menu(&mod_menu);
  w = menu_new(mainform, "Modules", mod_menu, w, headline);
#endif /* MOD_MENU */

  main_tracer = tracer_new(mainform, 0, 400, 500, NULL);
      /* 2nd arg = 1 turns the analog waveform on.  I (bjarthur) turned
         this off Sep 96 b/c it takes ALOT of time.  Of the 300 ms that
         update_displays() takes on a Sun Sparcstation IPX, it took well
         over 90%.  I asked around and nobody uses it.  but perhaps a
         better way to change things
         would be to add an onscreen on/off button, but it would require
         substantially more work */

  XtVaSetValues(main_tracer->form, XtNfromVert, w,
		XtNfromHoriz, NULL, NULL);

  tooldock = dock_new(mainform, NULL, 50, &dockform);
  XtVaSetValues(dockform, XtNfromVert, w,
		XtNfromHoriz, main_tracer->form, NULL);

#ifndef MOD_MENU
  install_modules(tooldock);
#endif /* __MOD_MENU */

  i = dock_add(tooldock, "TwoSound",
	   pop_ws_new(TopLevel, ws_twosound, "TwoSound", &h, NULL, NULL));
  make_close_help(h,"twosound",&tooldock->slots[i]);

  i = dock_add(tooldock, "Prefs",
	   pop_ws_new(TopLevel, ws_prefs, "Prefs", &h, NULL, NULL));
  h = make_close_help(h,"prefs",&tooldock->slots[i]);
  make_numbered_buttons(h);

  SI("prefs.page",1);
  loadPrefs(1);

  i = dock_add(tooldock, "Unit,Pos",
	   pop_ws_new(TopLevel, ws_expinfo, "Unit,Pos", &h, NULL, NULL));
  make_close_help(h,"unitpos",&tooldock->slots[i]);

  dock_add(tooldock, "SpikeSort", ss_user_newpanel(NULL, 0));

  XtVaSetValues(w = clockbar_build(mainform, 0, 0, NULL),
		XtNfromHoriz, 	NULL,
		XtNfromVert, 	main_tracer->form,
		NULL);

  sprintf(tmp, "Version %s", XDPHYS_VERSION);
  statline_set(headline, "xdphys -- %s", tmp);

  tracer_add_discrim(main_tracer);

  XtRealizeWidget(TopLevel);

  tracer_clear(main_tracer);

  /* setup some things for iserver & synth lib */
  is_setFcSetFn(xd_setFc);
  is_setAlertFn(alert);
  is_setNotifyFn(notify);
  is_setYesNoFn(pop_box);
	synth_init();

  if (ssfile != NULL) {
    int fc;
    if ((fc = ss_user_loadfile(ssfile)) < 0) {
      fprintf(stderr, "Can't load: %s\n", ssfile);
      exit(1);
    } else {
      notify("DACQ fc should be %dhz", fc);
    }
  }

#ifdef DO_COPYRIGHT
  DisplayNotice(CopyrightMessage());
#endif

  return(1);		/* caller should enter xloop */
}

int main(int ac, char **av)
{
  if (xdphysmain(ac, av, "xdphys", "XDphys"))
    dloop();
  exit(0);
}


#if 0
static double timestamp(void)
{
	struct timeval	time;
	struct timezone	zone = { 0, 0 };
	double timestamp;

	if (gettimeofday(&time, &zone) == -1) {
		perror("gettimeofday");
		exit(1);
	}

	timestamp = (double) time.tv_sec + ((double) time.tv_usec)/1000000.0;

	return(timestamp);
}

#endif
