/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: ws.c,v 2.61 2001/04/12 04:33:28 cmalek Exp $
** Program: dowl
**  Module: ws.c
**  Author: mazer
** Descrip: worksheet-like interface for X
**
** Revision History (most recent last)
**
** Sat Feb 29 21:12:33 1992 mazer
**  - added wsfield_new() single field ability
**  - width of text boxes are sized to fit field lengths
**
** Mon Nov  9 18:00:01 1992 mazer
**   added ws_help field
**
** Wed Jan 27 15:48:55 1993 mazer
**  fixed to StepGeneric() to prevent stepping with arrow
**  buttons when ws is locked
**
** Tue Mar 16 11:21:12 1993 mazer
**  added ws_lock(table, Boolean) and delete ws_lockall() and
**  ws_unlockall().. If you pass ws_lock(NULL, Bool) then you
**  get the same functionality of the old ws_[un]lockall() fn's
**
** Thu Oct  7 20:10:28 1993 mazer
**  reaped the CHANGE_BUTTONS code..
**
** Wed Jan 19 19:54:37 1994 mazer
**  removed ws_help and wsi_lockable -- and added the
**  wsi_lit field to help control the accelerator bug..
**
** 97.7 bjarthur
**  changed ftab to global_ftab.  changed the functions to reference
**    this as a default if no other was specified.
**
** 97.11 bjarthur
**  fixed bug in addtabtab.  used to cause xdphys to crash when parms were
**   viewed
**
** 97.12 bjarthur
**  fixed bug in the popuping up of dock slots such that the now popup
**  in the position they were last.  required the addition of a saveGeometry()
**  call in popdown_dockslot as well as a change from top_new() to pop_new()
**  to pop_ws_new()
**
** 97.12 bjarthur
**  fixed bug in ToggleAction() which caused some changes to ws_table
**  values to be erased in certain conditions
**
*******************************************************************/

#include "xdphyslib.h"
#include "fd.h"

#define MAXFIELDLISTS 30

static FieldList	global_ftab[MAXFIELDS];
static FieldList	*ftabtab[MAXFIELDLISTS];

static XtTranslations	TextTrans = NULL;
static String _TextTrans =
"<Key>Down: 		ws-move(next)\n\
 <Key>Up: 			ws-move(prev)\n\
 <Key>Tab:			ws-move(next)\n\
 <Key>Return:		ws-move(next)\n\
 <Key>KP_Enter: ws-move(next)\n\
 <Enter>:       ws-light(1)\n\
 <Leave>:       ws-light(0)\n\
";

static XtTranslations	BoolTrans = NULL;
static String _BoolTrans =
"<Key>Down:			ws-move(next)\n\
 <Key>Up:				ws-move(prev)\n\
 <Key>Tab:			ws-move(next)\n\
 <Key>Return:		ws-move(next)\n\
 <Key>KP_Enter: ws-move(next)\n\
 <Key>:					ws-toggle()\n\
 <Enter>:       ws-light(1)\n\
 <Leave>:       ws-light(0)\n\
";

static void jump_to_nth(FieldList);
static int verifyField(FieldList);
static Field *widget2field(Widget);
static Field *verifyWidget(Widget);
static void MoveChangeAction(Widget, XEvent*, String*, Cardinal*);
static void LightAction(Widget, XEvent*, String*, Cardinal*);
static void ToggleAction(Widget, XEvent*, String*, Cardinal*);
static char *noprefix(char*);
static void field_new(Widget, Field*, int, int);
static char *gensvar(char*);
static void ws_find_max_length(FieldList, int, int *, int *);

static void jump_to_nth(FieldList f)
{
  Dimension wd, ht;
  
  if (f != NULL) {
    XtVaGetValues(f->wsi_text, XtNwidth, &wd, XtNheight, &ht, NULL);
    XWarpPointer(XtDisplay(f->wsi_text), None, 
		 XtWindow(f->wsi_text), 0, 0, 0, 0, wd - 1, ht / 2);
  }
}

static int verifyField(FieldList f)	/* verify field contains valid entry */
{
  char buf[255];
  int valid;

  switch (f->ws_type[1])
    {
    case 's':
      valid = 1;
      break;
    case 'b':
      valid = ((*f->wsi_value == '0') ||
	       (*f->wsi_value == '1'));
      break;
    default:
      valid = (sscanf(f->wsi_value, f->ws_type, buf) == 1);
      break;
    }
  if (valid) {
    SS(f->wsi_svar, f->wsi_value);
  }
  return(valid);
}

static Field *widget2field(Widget w)
{
  int i,j;

  for (j = 0; ftabtab[j] != NULL; j++) {
  	for (i = 0; ftabtab[j][i] != NULL; i++)
			if ((ftabtab[j][i]->wsi_text == w) || (ftabtab[j][i]->wsi_label == w))
				return(ftabtab[j][i]); }
	return(NULL);
}

/* verify text in widget, w is text widget */
static Field *verifyWidget(Widget w)
{
	Field *f;

  if ((f = widget2field(w)) != NULL) {
    if (verifyField(f))
      return(f);
    else
      return(NULL);
  }
  XtAppError(XtWidgetToApplicationContext(w),
	     "verifyWidget: seem to lost a widget");
  return(NULL);
}

static void MoveChangeAction(Widget w, XEvent *event, String *params,
      Cardinal *num_params)
{
	Field *f;

  if ((f = verifyWidget(w)) == NULL) {
    beep(0);
  } else {
    if (*num_params == 1 && *params[0] == 'n')
      jump_to_nth(f->wsi_next);
    else /* if (*num_params == 1 && *params[0] == 'p') */
      jump_to_nth(f->wsi_prev);
  }
}

static void LightAction(Widget w, XEvent *event, String *params,
      Cardinal *num_params)
{
	Field *f;

  if (*num_params > 0 && *params[0] == '1') {
    f = widget2field(w);
    if (! f->wsi_lit) {
      ReverseColors(w);
      f->wsi_lit = 1;
    }
  } else {
    f = widget2field(w);
    if (f->wsi_lit) {
      ReverseColors(w);
      f->wsi_lit = 0;
    }
  }
}

static void ToggleAction(Widget w, XEvent *event, String *params,
      Cardinal *num_params)
{
  int j;
	Field *f;

  if ((f=widget2field(w))!=NULL) {
    if (f->ws_type[1] != 'b') {
      XtAppError(XtWidgetToApplicationContext(w),
		 "ToggleAction: not a boolean");
    } else {
      ws_tosvars(NULL,NULL);
      j = atoi(f->wsi_value);
      sprintf(f->wsi_value, "%d", !j);
      beep(0);			/* async, so start it now.. */
      verifyField(f);		/* this will call setvar().. */
      ws_tows(NULL,NULL);		/* this makes the display update.. */
    }
  }
}

static XtActionsRec actionTable[] = {
  { "ws-move", 		MoveChangeAction	},
  { "ws-toggle",	ToggleAction		},
  { "ws-light",		LightAction		},
};

static char *noprefix(char *prompt)
{
  char *p;

  if ((p = rindex(prompt, '.')) == NULL)
    return(prompt);
  else
    return(p + 1);
}

static void field_new(
     Widget parent,
     Field *f,
     int label_len,		/* label description length (0 for auto) */
     int slot_len)		/* entry slot length (0 for semi-auto) */
{
  char *labelstr, *labelname, *p;
  int char_width,char_height;

  if (label_len) {
    labelstr = (char *)malloc(label_len + 1);
    labelname = (char *)malloc(label_len + 1);
    sprintf(labelstr, "%*s", label_len, noprefix(f->ws_prompt));
  } else {
    p = noprefix(f->ws_prompt);
    labelstr = (char *)malloc(strlen(p) + 1);
    labelname = (char *)malloc(strlen(p) + 1);
    sprintf(labelstr, "%s", p);
  }
  sscanf(labelstr, "%s", labelname);
  f->wsi_label = label_new(parent, labelname, NULL, NULL, 0);
  label_set(f->wsi_label, labelstr);
  XtVaSetValues(f->wsi_label,
		XtNvertDistance, 2,
		XtNhorizDistance, 0, NULL);

#ifdef X11R4
  f->wsi_text =
    XtVaCreateManagedWidget(labelname, asciiTextWidgetClass, parent,
			    XtNborderWidth, 	1,
			    XtNstring, 		f->wsi_value,
			    XtNlength, 		f->ws_length,
			    XtNeditType, 	XawtextEdit,
			    XtNfromHoriz, 	f->wsi_label,
			    XtNvertDistance,	2,
			    XtNhorizDistance,	0,
			    NULL);
#else /* X11R4 */
  f->wsi_text =
    XtVaCreateManagedWidget(labelname, asciiTextWidgetClass, parent,
			    XtNborderWidth, 	1,
			    XtNstring, 		f->wsi_value,
			    XtNlength, 		f->ws_length,
			    XtNeditType,	XawtextEdit,
			    XtNfromHoriz, 	f->wsi_label,
			    XtNvertDistance,	2,
			    XtNhorizDistance,	0,
			    XtNuseStringInPlace,True,
			    NULL);
  char_size(f->wsi_text, &char_width, &char_height);
  if (slot_len == 0)
    slot_len = f->ws_length + 2;
  if (slot_len < 10)
    slot_len = 10;
  XtVaSetValues(f->wsi_text,
		XtNwidth, char_width * (slot_len + 2), NULL);
  if(slot_len < f->ws_length) {
    XtVaSetValues(f->wsi_text,
	  	XtNheight, (Dimension)floor(char_height * 2.5),
      XtNscrollHorizontal, XawtextScrollAlways, NULL); }
#endif /* X11R4 */

  f->wsi_typelabel =
    label_new(parent, "typelabel", f->wsi_text,  NULL, 0);
  p = (char*) malloc(strlen(f->ws_type) + 5);
  sprintf(p, "{%s}", f->ws_type + 1);
  XtVaSetValues(f->wsi_typelabel, XtNlabel, p,
		XtNvertDistance, 2, XtNhorizDistance, 0, NULL);
  free(p);

  free(labelstr);
  free(labelname);
  if (strncmp(f->ws_type, "%b", 2) == 0)
    XtOverrideTranslations(f->wsi_text, BoolTrans);
  else
    XtOverrideTranslations(f->wsi_text, TextTrans);
  XawTextSetInsertionPoint(f->wsi_text,
			   (XawTextPosition) strlen(f->wsi_value));
}

static char *gensvar(char *prompt)
{
  char *p, *q;

  p = strsave(prompt);
  if ((q = index(p, ' ')) != NULL)
    *q = 0;
  return(p);
}

static void addtabtab(FieldList *ftab)
{
	int i;

	i=0;
	while((ftabtab[i]!=ftab) && (ftabtab[i]!=NULL) && (i<MAXFIELDLISTS))
		i++;
	assert(i<MAXFIELDLISTS);
	if(ftabtab[i]==NULL)
		ftabtab[i]=ftab;
}


Widget ws_new(Widget parent, FieldList table, char *title,
			FieldList *ftab, SVAR_TABLE *svar_table)
{
	char *q;
	Widget superform, horiz_form, next_horiz, orig_vert_form, vert_form, form, w;
	int i, j, llen, flen, offset;

	if (ftab == NULL)
		ftab = global_ftab;
	if (svar_table == NULL)
		svar_table = global_svar_table;

	if (TextTrans == NULL) {
		if (ftab == global_ftab)
			for (j = 0; j < MAXFIELDS; j++)
				ftab[j] = NULL;
		for (j = 0; j < MAXFIELDLISTS; j++)
			ftabtab[j] = NULL;
		XtAppAddActions(XtWidgetToApplicationContext(TopLevel),
				actionTable, XtNumber(actionTable));
		BoolTrans = XtParseTranslationTable(_BoolTrans);
		TextTrans = XtParseTranslationTable(_TextTrans);
	}

	addtabtab(ftab);

	/* Window title */
	superform = XtVaCreateManagedWidget(title ? title : "ws",
					    formWidgetClass, parent, NULL);
	w = title ? label_new(superform, title, NULL, NULL, 1) : NULL;

	for (offset = 0; ftab[offset]; offset++) {
		if (offset == MAXFIELDS) {
			fprintf(stderr, "ws_new: too many slots\n");
			exit(1);
		}
	}

	ws_find_max_length(table, 0, &llen, &flen);

	/* Main form, underneath window title */
	form = XtVaCreateManagedWidget("wsform", formWidgetClass, superform,
				       XtNfromVert, w, NULL);
	horiz_form = NULL;
	vert_form = form;
	orig_vert_form = w;
	next_horiz = form;
	w = NULL;
	for (j = 0, i = offset; table[j].ws_prompt != NULL; j++) {
		if (*table[j].ws_prompt && !ISSEP(table[j].ws_prompt)) {
			/* Regular Field */
			if ((i + 1) == MAXFIELDS) {
				fprintf(stderr, "ws_new: too many slots\n");
				exit(1);
			}
			ftab[i] = &table[j];
			ftab[i]->wsi_value = (char *) malloc(ftab[i]->ws_length + 1);
			ftab[i]->wsi_svar = gensvar(ftab[i]->ws_prompt);
			if ((q = getvar(ftab[i]->wsi_svar, svar_table)) != NULL)
				strncpy(ftab[i]->wsi_value, q, ftab[i]->ws_length);
			else if (ftab[i]->ws_default != NULL)
				strncpy(ftab[i]->wsi_value, ftab[i]->ws_default,
					ftab[i]->ws_length);
			ftab[i]->wsi_value[ftab[i]->ws_length] = 0;
			field_new(form, ftab[i], llen, flen);
			XtVaSetValues(ftab[i]->wsi_label, XtNfromVert, w, NULL);
			XtVaSetValues(ftab[i]->wsi_text, XtNfromVert, w, NULL);
			XtVaSetValues(ftab[i]->wsi_typelabel, XtNfromVert, w, NULL);
			w = ftab[i]->wsi_text;
			if (i != offset) {
				ftab[i]->wsi_prev = ftab[i - 1];
				ftab[i - 1]->wsi_next = ftab[i];
			}
			ftab[i]->wsi_lit = 0;
			i++;
		} else {
			if (ISVSEP(table[j].ws_prompt)) {
				/* Separator */
				if (j != 0) {
					/* if this is not the first entry in the table ... */
					form = XtVaCreateManagedWidget("wsform",
								       formWidgetClass,
								       superform,
								       XtNfromVert,
								       vert_form,
								       NULL);
					if (horiz_form != NULL)
						XtVaSetValues(form, XtNfromHoriz,
							      horiz_form, NULL);
					vert_form = form;
				}

				if (table[j].ws_type == NULL) {
					w = NULL;
				} else {
					w =
					    label_new(form, table[j].ws_type, NULL,
						      NULL, 1);
				}
			} else {
				if (ISHSEP(table[j].ws_prompt)) {
					if (j != 0) {
						/* if this is not the first entry in the table ... */
						horiz_form = next_horiz;
						form =
						    XtVaCreateManagedWidget("wsform",
									    formWidgetClass,
									    superform,
									    XtNfromVert,
									    orig_vert_form,
									    XtNfromHoriz,
									    horiz_form,
									    NULL);
						next_horiz = form;
						vert_form = form;
					}

					ws_find_max_length(table, j + 1, &llen,
							   &flen);

					if (table[j].ws_type == NULL) {
						w = NULL;
					} else {
						w =
						    label_new(form, table[j].ws_type,
							      NULL, NULL, 1);
					}
				}
			}
		}
	}
	ftab[i] = NULL;
	ftab[i - 1]->wsi_next = ftab[offset];
	ftab[offset]->wsi_prev = ftab[i - 1];
	ws_tosvars(ftab, svar_table);
	return (superform);
}

/* ----------------------------------------------------------------
   ws_find_max_length:

   Search a FieldList and return the length of the longest
   field, and the index of that field in the table.  Stop if either
   the end of the FieldList or a WS_HSEP is hit.
   ---------------------------------------------------------------- */
static void ws_find_max_length(FieldList table, int start, int *llen, 
	int *flen)
{
	int i, tmp;

	assert(table != NULL);
	assert(flen != NULL);
	assert(llen != NULL);

	*flen=0;
	*llen=0;
	for (i = start; table[i].ws_prompt != NULL; i++) {
		if (ISHSEP(table[i].ws_prompt))
			break;
		if (*table[i].ws_prompt) {
			if ((tmp = strlen(noprefix(table[i].ws_prompt))) > *llen)
				*llen = tmp;
			if (table[i].ws_length > *flen)
				*flen = table[i].ws_length;
		}
	}
	*flen = MY_MIN(WS_MAX_DISP_LENGTH, *flen);
}

int ws_verify(FieldList *ftab)
{
  int i;

	if(ftab==NULL)
		ftab = global_ftab;

  for (i = 0; ftab[i] != NULL; i++)
    if (!verifyField(ftab[i])) {
      alert("\"%s\" is invalid (type=%s), please fix",
	    			ftab[i]->ws_prompt, ftab[i]->ws_type);
      jump_to_nth(ftab[i]);
      return(0);
    }
  return(1);
}

void ws_write(FILE *fp, FieldList table, int tsize, SVAR_TABLE *svar_table)
{
	int i;
	char *q;

	if (svar_table == NULL)
		svar_table = global_svar_table;

	for (i = 0; i < tsize; i++) {
		if (table[i].ws_prompt != NULL && *table[i].ws_prompt) {
			if (strcmp(table[i].ws_prompt, WS_VSEP) == 0)
				continue;
			if (strcmp(table[i].ws_prompt, WS_HSEP) == 0)
				continue;
			if (table[i].wsi_svar == NULL)
				table[i].wsi_svar = gensvar(table[i].ws_prompt);

			if ((q = getvar(table[i].wsi_svar, svar_table)) != NULL) {
				if (!strncmp(q, "file=", 5)) {
					char *p, *r;
					FILE *fp2;
					int l = filesize(q + 5);
					if (l && (fp2 = fopen2(q + 5, "r")) != NULL) {
						r = calloc(l + 3, sizeof(char));
						fread(r, l, sizeof(char), fp2);
						if ((p = rindex(r, '\n')) != NULL)
							(*p) = '\000';
						fprintf(fp, "%s=%s\n", table[i].wsi_svar, r);
						free(r);
						fclose(fp2);
					} else {
						fprintf(fp, "%s=no such file\n", table[i].wsi_svar);
					}
				} else {
					fprintf(fp, "%s=%s\n", table[i].wsi_svar, q);
				}
			}

			else if (table[i].ws_default != NULL)
				fprintf(fp, "%s=%s\n", table[i].wsi_svar,
					table[i].ws_default);
			else
				fprintf(fp, "%s=?xyzzy?\n", table[i].wsi_svar);
		}
	}
}


#if(0)
int ws_check_params(FILEDATA *fd, FieldList table, int tsize, char *msg)
{
  int i,mis;
  char *q, buf[128];

  mis=MISMATCH_NO;
  for (i = 0; i < tsize; i++) {
    if (table[i].ws_prompt != NULL && *table[i].ws_prompt) {
      if(!strcmp(table[i].ws_prompt,"time") ||
            !strcmp(table[i].ws_prompt,"timestamp"))
        continue;
      if ((strcmp(table[i].ws_prompt, WS_VSEP) == 0) ||
          (strncmp(table[i].ws_prompt, "xdphys.fname", 10) == 0))
        continue;
      if (table[i].wsi_svar == NULL)
        table[i].wsi_svar = gensvar(table[i].ws_prompt);
      if ((q = GS(table[i].wsi_svar)) == NULL)
        if ((q = table[i].ws_default) == NULL)
          q = "?xyzzy?";
      if(strcmp(q,FD_GV(fd,table[i].wsi_svar))) {
        if(strstr(table[i].wsi_svar,"Range") && (mis!=MISMATCH_YES)) {
          if(!StimArray_Equivalent(q,FD_GV(fd,table[i].wsi_svar)))
            if(StimArray_Overlap(q,FD_GV(fd,table[i].wsi_svar))) {
              mis=MISMATCH_YES;
              if(msg!='\0') strcat(msg,"\n");
              sprintf(buf,"'%s' parameter is not equivalent and overlapping.\n",
                    table[i].wsi_svar);
              strcat(msg,buf);
              strcat(msg,"   Ranges must be exactly the same, or completely\n");
              strcat(msg,"   different when appending datafiles."); }
            else if(!StimArray_SameShift(q,FD_GV(fd,table[i].wsi_svar))) {
              mis=MISMATCH_YES;
              if(msg!='\0') strcat(msg,"\n");
              strcat(msg,"   Ranges must be shifted by the same valued.\n"); }
            else {
              mis=MISMATCH_RANGE; } }
        else if(!strcmp(table[i].wsi_svar,"Reps") && (mis!=MISMATCH_YES)) {
          mis=MISMATCH_REPS; }
        else {
          mis=MISMATCH_YES;
          if(msg!='\0') strcat(msg,"\n");
          sprintf(buf,"Mismatch in %s parameter",table[i].wsi_svar);
          strcat(msg,buf); } } } }

  return(mis);
}
#endif

void ws_writeconfig(char *fname, FieldList table, int tsize,
			SVAR_TABLE *svar_table)
{
  FILE *fp;

  if ((fp = fopen2(findhome(fname), "w")) == NULL) {
    fprintf(stderr, "ws_writeconfig: can't write %s\n", fname);
    perror("ws_writeconfig");
  } else {
    fprintf(fp, "# xdphys database file: do not edit by hand\n");
    ws_write(fp, table, tsize, svar_table);
    fclose(fp);
  }
}

void ws_preload(FieldList table, int tsize)
{
  int i;
  for (i = 0; i < tsize; i++) {
    if (table[i].ws_prompt && table[i].ws_length == WS_PRELOAD) {
      SS(table[i].ws_prompt, table[i].ws_default);
    }
  }
}

void ws_tosvars(FieldList *ftab, SVAR_TABLE *svar_table) /* WORKSHEET --> SVARS*/
{
  int i;

	if(ftab==NULL)
		ftab = global_ftab;
	if(svar_table==NULL)
		svar_table = global_svar_table;

  for (i = 0; ftab[i] != NULL; i++)
    setvar(ftab[i]->wsi_svar, ftab[i]->wsi_value, svar_table);
}

void ws_tows(FieldList *ftab, SVAR_TABLE *svar_table)	/* SVARS --> WORKSHEET */
{
  int i;
  char *q;
  XawTextBlock tb;

	if(ftab==NULL)
		ftab = global_ftab;
	if(svar_table==NULL)
		svar_table = global_svar_table;

  for (i = 0; ftab[i] != NULL; i++) {
    if ((q = getvar(ftab[i]->wsi_svar,svar_table)) != NULL) {
      tb.firstPos = 0;
      tb.length = strlen(q);
      tb.ptr = q;
      XawTextReplace(ftab[i]->wsi_text, 0, 299, &tb);
    }
  }
}

Widget pop_ws_new(Widget parent, FieldList table, char *title, Widget *handle,
			FieldList *ftab, SVAR_TABLE *svar_table)
{
  Widget pshell, ws, w, form;

  pshell = pop_new(parent, title);
  form = XtVaCreateManagedWidget(title ? title : "popws",
				 formWidgetClass, pshell, NULL);
  if (title != NULL) {
    w = label_new(form, title, NULL, NULL, 0);
    ReverseColors(w); }

  if (handle != NULL)
    *handle = w;

  ws = ws_new(form, table, NULL, ftab, svar_table);

  XtVaSetValues(ws, XtNfromVert, w, XtNfromHoriz, NULL, NULL);

  return(pshell);
}

void ws_lock(FieldList table, int lockit)
{
  int j;

  if (table) {
    for (j = 0; table[j].ws_prompt != NULL; j++) {
      if (table[j].wsi_text != NULL) {
				XtSetSensitive(table[j].wsi_text, lockit ? False : True);
      }
    }
  } else {
    for (j = 0; global_ftab[j] != NULL; j++) {
      if (global_ftab[j]->wsi_text != NULL) {
				XtSetSensitive(global_ftab[j]->wsi_text, lockit ? False : True);
      }
    }
  }
}
