/*
 * ** This file is part of XDowl
 * ** Copyright (c) 1994 Jamie Mazer
 * ** California Institute of Technology
 * ** <mazer@asterix.cns.caltech.edu>
 */

/******************************************************************
**  RCSID: $Id: waver.c,v 2.58 2000/06/20 19:25:07 cmalek Exp $
** Program: dowl
**  Module: waver.c
**  Author: mazer
** Descrip: plot waveforms on screen
**
** Revision History (most recent last)
**
** Mon Sep 27 17:22:03 1993 mazer
**  added waveview_new() pseudo-widget.. replaces zillions of
**  versions of build_viewer() calls all over the place.
**
** Wed Oct 13 00:28:54 1993 mazer
**  deleted plotWave(), plotWave2(), setCursor(), clrCursor(),
**    popwaver_new()
**  changed popfilewaver_new() to handle nthwave-type files
**  changed waver_new() to canvaswaver_new()
**
*******************************************************************/

#include "xdphyslib.h"

static Pixel rreedd, green, black, blue, purple, yellow, gray;

static void init_pens(Widget);
static void CleanupCB(Widget, struct WavefileViewData *, caddr_t);
static void ExportXoCB(Widget, struct WavefileViewData *, caddr_t);
static void NextCB(Widget, struct WavefileViewData *, caddr_t);
static void NthCB(Widget, struct WavefileViewData *, caddr_t);
static void PrevCB(Widget, struct WavefileViewData *, caddr_t);

static void init_pens(Widget w)
{
	static int first_time = 1;

	if (first_time) {
		black = ConvertColor(w, "black");
		gray = ConvertColor(w, "gray");
		rreedd = ConvertColor(w, "red");
		green = ConvertColor(w, "darkgreen");
		blue = ConvertColor(w, "royal blue");
		purple = ConvertColor(w, "purple");
		yellow = ConvertColor(w, "gold");
		first_time = 0;
	}
}

Widget canvaswaver_new(Widget parent, Dimension width, Dimension height)
{
	return (XtVaCreateManagedWidget("waver", canvasWidgetClass, parent,
					XtNwidth, width,
					XtNheight, height,
					XtNupdateOnDraw, False,
					NULL));
}

/* ------------------------------------------------------------------------
 * plotUnitPerievent
 * 
 * The Waver widget plots data in adticks; it was too much of a pain
 * to change it to use microseconds.  Dynraster is in milliseconds,
 * why not this?
 * 
 * Also, to convert milliseconds to adticks, it uses is_adFc; probably
 * this should be passed in as an argument.
 * ------------------------------------------------------------------------ */
void plotUnitPerievent(
			      Widget w,
			      xword * buffer,
			      int start,	/* in adticks */
			      int stop,		/* in adticks */
			      int pre,	/* in adticks */
			      int post,		/* in adticks */
			      int max_traces,
			      spike_t * spikelist,
			      int mask,
			      int lines,
			      int autoscale)
{
	register int i, j, tmin, tmax;
	register int event, begin, end;		/* these are in adticks */
	register float x;
	xword *event_ptr;
	float fpre, fpost;
	int nevents;
	char buf[20];
	extern int debugflag;
	float foo, offx = 0;
	float xword_max, tmp;
	int gsr_window_flag;

	gsr_window_flag = (getRaster_getmode() == GSR_window) ? 1 : 0;

	w = XtCanvasFind(w);
	XtCanvasClear(w, False);

	XtCanvasSetWC(w, (float) -pre, (float) post, -FRANGE, FRANGE);
	init_pens(w);

	tmax = -RANGE;
	tmin = RANGE;

	if (gsr_window_flag) {
		fpre = (float) pre;
		fpost = (float) post;

		if (sizeof(xword) == sizeof(short))
			 xword_max = MAXSHORT;
		else
			xword_max = MAXINT;

		if (G_detect_y[0] > tmax)
			tmax = G_detect_y[0];
		if (G_detect_y[0] < tmin)
			tmin = G_detect_y[0];
	}

	nevents = 0;

	if (buffer != NULL && spikelist != NULL) {
		for (i = 0; i < N_SPIKES(spikelist); i++) {
			if ((SPIKE_CHAN(spikelist, i) == mask) ||
			    (SPIKE_CHAN(spikelist, i) == OUTLIER_CODE)) {
				nevents++;
				if (max_traces == 0 || nevents < max_traces) {
					if (debugflag)
						printf("%c.", (char) ('A' + nevents));
					foo = SPIKE_SEC(spikelist, i) * (float) is_adFc;
					event = (int) floor((double) foo + 0.5);
					offx = foo - event;

					begin = event - pre;
					if (begin < start)
						begin = start;
					end = event + post;
					if (end > stop)
						end = stop;
					event_ptr = &buffer[begin * 2];

					if (SPIKE_CHAN(spikelist, i) == mask)
						XtCanvasPenColor(w, black);
					else
						XtCanvasPenColor(w, gray);

					XtCanvasAddPoints(w, sizeof(xword) == sizeof(short) ?
							  CanvasDataShort : CanvasDataInt, end - begin, 1, NULL, 1,
							   (float) (begin - event) - offx, event_ptr, 2,
							  0.0, CanvasFix);

					for (j = end - begin; j > 0; j--, event_ptr += 2) {
						if (*event_ptr > tmax)
							tmax = *event_ptr;
						if (*event_ptr < tmin)
							tmin = *event_ptr;
					}
				}
			}
		}

		if (debugflag)
			printf("\n");

		if (gsr_window_flag) {
			for (i = 1; i < 4; i++) {
				if (G_detect_onoff[i]) {
					tmp = G_detect_y[i];
					if (tmp > tmax)
						tmax = tmp;
					if (tmp < tmin)
						tmin = tmp;
				}
			}
			for (i = 1; i < 4; i++) {
				if (G_detect_onoff[i]) {
					if (G_detect_color[i]) {
						XtCanvasPenColor(w, rreedd);
					} else {
						XtCanvasPenColor(w, green);
					}
					XtCanvasAddFilledCircle(w,
						      G_detect_x[i], G_detect_y[i],
								(fpre + fpost) / 60.0, (tmax - tmin) / 20.0,
						0, LineSolid, CanvasFix);
				}
			}
		}
		XtCanvasPenColor(w, blue);
		XtCanvasAddLine(w, -fpre, G_detect_y[0], fpost, G_detect_y[0], 0,
				LineOnOffDash, CanvasFix);

		if (!XtCanvasUserView(w)) {
			if (autoscale && spikelist && N_SPIKES(spikelist) > 0) {
				float del = tmax - tmin;
				tmax += (0.05 * del);
				tmin -= (0.05 * del);
				if (nevents > max_traces)
					sprintf(buf, "%d++", nevents);
				else
					sprintf(buf, "%d", nevents);
				XtCanvasPenColor(w, blue);
				XtCanvasAddText(w, 0.0, 0.0, buf, CanvasTextRight, CanvasFloatY);

				XtCanvasPenColor(w, blue);
				XtCanvasAddLine(w, (float) 0.0, tmin, (float) 0.0, tmax,
						0, LineSolid, CanvasFix);

				XtCanvasSetVC(w, (float) (-pre), (float) post, (float) tmin, (float) tmax);
				if (!gsr_window_flag) {
					XtCanvasPenColor(w, blue);
					XtCanvasAddLine(w, (float) -pre, (float) tmin, (float) -pre,
							(float) (tmin + ((tmax - tmin) / FRANGE * (tmax - tmin))),
						8, LineSolid, CanvasFix);
				}
			} else {
				XtCanvasSetVC(w, (float) -pre, (float) post, -FRANGE, FRANGE);
			}
		}
	}
	XtCanvasUpdate(w);
}


Widget waveview_new(Widget parent, Widget * l, Widget * r, int fc_hz, int link)
/*     int fc_hz;     -1 for don't know/care */
{
	Widget form, psh;

	if (parent == NULL)
		psh = parent = pop_new(TopLevel, "WaveView");
	else
		psh = NULL;

	form = form_new(parent, "waveview", 0);

	if (l != NULL)
		*l = XtVaCreateManagedWidget("Left", waveWidgetClass, form,
					     XtNfc, fc_hz,
					   XtNwidth, 500, XtNheight, 100,
					     XtNborderWidth, 0,
					     NULL);
	if (l != NULL)
		*r = XtVaCreateManagedWidget("Right", waveWidgetClass, form,
					     XtNfc, fc_hz,
					     XtNwidth, 500,
					     XtNheight, 100,
					     XtNborderWidth, 0,
					     XtNfromVert, l ? *l : NULL,
					     XtNvertDistance, l ? 2 : 0,
					     NULL);
	if (l && r && link)
		XtWaveLink(*l, *r, 1);
	return (psh ? psh : form);
}

Widget popfilewaver_new(Widget parent, char *filename, int n, Dimension width,
		  Dimension height, Widget * lp, Widget * rp, char **com)
{
	Widget pshell, form, lwave, rwave, w;
	xword *buffer;
	WAVEINFO *winfo;
	int toofar, i;
	char *p;

	if (n < 0) {
		i = 1;
		while ((w = popfilewaver_new(parent, filename, i,
			     width, height, NULL, NULL, NULL)) != NULL) {
			popup(w);
			i++;
		}
		return (w);
	} else {
		if ((buffer = nthwave(filename, &winfo, n, &toofar)) != NULL) {
			p = (char *) malloc(strlen(filename) + 20);
			sprintf(p, "%s #%d", filename, n);
			pshell = pop_new(parent, p);
			free(p);

			form = XtVaCreateManagedWidget("waver", formWidgetClass, pshell, NULL);
			w = button_new(form, "Close", "Close", ClosePShellCB, pshell, NULL, NULL);
			if (winfo->comment)
				label_new(form, winfo->comment, w, NULL, 0);
			lwave = XtVaCreateManagedWidget("left", waveWidgetClass, form,
						    XtNuserCursors, True,
				  XtNwidth, width, XtNheight, height / 2,
							XtNfromVert, w,
							XtNfc, winfo->fc,
						      XtNfromHoriz, NULL,
							NULL);
			XtWaveSetWaveform(lwave, buffer, winfo->nsamps, winfo->nchans,
					  -RANGE, RANGE);
			if (winfo->nchans >= 2) {
				rwave = XtVaCreateManagedWidget("right", waveWidgetClass, form,
						    XtNuserCursors, True,
				  XtNwidth, width, XtNheight, height / 2,
						      XtNfromVert, lwave,
							XtNfc, winfo->fc,
						      XtNfromHoriz, NULL,
								NULL);
				XtWaveSetWaveform(rwave, buffer + 1, winfo->nsamps, winfo->nchans,
						  -RANGE, RANGE);
			} else {
				rwave = NULL;
			}
			if (lp)
				*lp = lwave;
			if (rp)
				*rp = rwave;
			if (com)
				*com = strcpy(malloc(strlen(winfo->comment) + 1), winfo->comment);
			free(buffer);
			free(winfo);
			return (pshell);
		} else {
			if (lp)
				*lp = NULL;
			if (rp)
				*rp = NULL;
			if (com)
				*com = NULL;
			return (NULL);
		}
	}
}

static void CleanupCB(Widget w, struct WavefileViewData *data,
		      caddr_t call_data)
{
	if (data && data->filename)
		free(data->filename);
	if (data)
		free(data);
}

static void ExportXoCB(Widget w, struct WavefileViewData *data,
		       caddr_t call_data)
{
	char buf[1000];
	sprintf(buf, "exec xo -noio %s#%d -det", data->filename, data->cpos);
	system(buf);
}

static void NextCB(Widget w, struct WavefileViewData *data, caddr_t call_data)
{
	xword *buffer;
	WAVEINFO *winfo;
	int toofar;
	char foo[10];

	if ((buffer = nthwave(data->filename,
			      &winfo, data->cpos + 1, &toofar)) != NULL) {
		data->cpos++;
		XtWaveSetWaveform(data->left, buffer, winfo->nsamps, winfo->nchans,
				  -RANGE, RANGE);
		if (winfo->nchans > 1)
			XtWaveSetWaveform(data->right, buffer + 1, winfo->nsamps, winfo->nchans,
					  -RANGE, RANGE);
		else
			XtWaveSetWaveform(data->right, NULL, 0, 0, -RANGE, RANGE);
		sprintf(foo, "%03d", data->cpos);
		label_set(data->nlabel, foo);
		label_set(data->clabel, winfo->comment);
		free(buffer);
		free(winfo);
	} else {
		beep(0);
	}
}

static void NthCB(Widget w, struct WavefileViewData *data, caddr_t call_data)
{
	xword *buffer;
	WAVEINFO *winfo;
	int toofar, n;
	char foo[10], *p;

	if ((p = pop_dialog("Trace number?", "")) != NULL) {
		if (sscanf(p, "%d", &n) == 1) {
			if ((buffer = nthwave(data->filename,
					  &winfo, n, &toofar)) != NULL) {
				data->cpos = n;
				XtWaveSetWaveform(data->left, buffer, winfo->nsamps, winfo->nchans,
						  -RANGE, RANGE);
				if (winfo->nchans > 1)
					XtWaveSetWaveform(data->right, buffer + 1,
							  winfo->nsamps, winfo->nchans, -RANGE, RANGE);
				else
					XtWaveSetWaveform(data->right, NULL, 0, 0, -RANGE, RANGE);
				sprintf(foo, "%03d", data->cpos);
				label_set(data->nlabel, foo);
				label_set(data->clabel, winfo->comment);
				free(buffer);
				free(winfo);
			}
		}
		free(p);
	}
}

static void PrevCB(Widget w, struct WavefileViewData *data, caddr_t call_data)
{
	xword *buffer;
	WAVEINFO *winfo;
	int toofar;
	char foo[10];

	if (data->cpos > 1) {
		if ((buffer = nthwave(data->filename,
			     &winfo, data->cpos - 1, &toofar)) != NULL) {
			data->cpos--;
			XtWaveSetWaveform(data->left, buffer, winfo->nsamps, winfo->nchans,
					  -RANGE, RANGE);
			if (winfo->nchans > 1)
				XtWaveSetWaveform(data->right, buffer + 1, winfo->nsamps,
					   winfo->nchans, -RANGE, RANGE);
			else
				XtWaveSetWaveform(data->right, NULL, 0, 0, -RANGE, RANGE);
			sprintf(foo, "%03d", data->cpos);
			label_set(data->nlabel, foo);
			label_set(data->clabel, winfo->comment);
			free(buffer);
			free(winfo);
		} else {
			beep(0);
		}
	} else {
		beep(0);
	}
}

Widget popfilewaver2_new(Widget parent, char *filename, Dimension width,
			 Dimension height)
{
	Widget pshell, form, w, x;
	xword *buffer;
	WAVEINFO *winfo;
	int toofar, n;
	struct WavefileViewData *data;
	char buf[20];

	if ((n = iswavefile(filename)) <= 0)
		return (NULL);
	if ((buffer = nthwave(filename, &winfo, 1, &toofar)) == NULL) {
		return (NULL);
	}
	if (parent == NULL) {
		pshell = pop_new(parent, "popfilewaver");
		parent = pshell;
	} else {
		pshell = NULL;
	}

	data = (struct WavefileViewData *) malloc(sizeof(struct WavefileViewData));

	form = XtVaCreateManagedWidget("waver", formWidgetClass, parent, NULL);
	XtAddCallback(form, XtNdestroyCallback, CleanupCB, data);

	if (pshell)
		w = x = button_new(form, "Close", "Close", ClosePShellCB, pshell, NULL, NULL);
	else
		w = x = NULL;
	x = button_new(form, "->xo", "->xo", ExportXoCB, data, x, NULL);
	if (w == NULL)
		w = x;
	x = button_new(form, "<<", "<<", PrevCB, data, x, NULL);
	x = button_new(form, "N", "N", NthCB, data, x, NULL);
	x = button_new(form, ">>", ">>", NextCB, data, x, NULL);

	sprintf(buf, "of %d", n);
	x = label_new(form, filename, NULL, w, 0);	/* used to be ... w, NULL, 0 */
	data->nlabel = label_new(form, "001", x, w, 0);
	label_new(form, buf, data->nlabel, w, 0);

	w = data->clabel = label_new(form, winfo->comment, NULL, x, 0);

	data->cpos = 1;
	data->filename = strcpy(malloc(strlen(filename) + 1), filename);
	data->left = XtVaCreateManagedWidget("left", waveWidgetClass, form,
					     XtNuserCursors, True,
				  XtNwidth, width, XtNheight, height / 2,
					     XtNfromVert, w,
					     XtNfc, winfo->fc,
					     XtNfromHoriz, NULL,
					     XtNuserCursors, True,
					     NULL);
	XtWaveSetWaveform(data->left, buffer, winfo->nsamps, winfo->nchans,
			  -RANGE, RANGE);
	data->right = XtVaCreateManagedWidget("right", waveWidgetClass, form,
					      XtNuserCursors, True,
				  XtNwidth, width, XtNheight, height / 2,
					      XtNfromVert, data->left,
					      XtNfc, winfo->fc,
					      XtNfromHoriz, NULL,
					      XtNuserCursors, True,
					      NULL);
	if (winfo->nchans > 1) {
		XtWaveSetWaveform(data->right, buffer + 1, winfo->nsamps, winfo->nchans,
				  -RANGE, RANGE);
	} else {
		XtWaveSetWaveform(data->right, NULL, 0, 0, -RANGE, RANGE);
	}


	XtWaveLink(data->left, data->right, 1);
	free(buffer);
	free(winfo);
	return (pshell ? pshell : form);
}
