/*
   **   This file is part of XDowl
   **   Copyright (c) 1994 Jamie Mazer
   **   California Institute of Technology
   **   <mazer@asterix.cns.caltech.edu>
 */

/******************************************************************
**  RCSID: $Id: unittrace.c,v 2.65 2001/04/02 03:02:30 cmalek Exp $
** Program: dowl
**  Module: unittrace.c
**  Author: mazer
** Descrip: "real-time" unit activity display
**
** Revision History (most recent last)
**
** Tue Mar 17 00:30:00 1992 mazer
**  modified for setting window discrim parameters
**  now uses WaveWidget to keep fast trace+perievent
**   simultaneously on screen
**  
** Fri Mar 20 01:54:31 1992 mazer
**  Added functional on-line histogram'ing for ISI and PSTH's
**
** Sun May 17 21:57:05 1992 mazer
**  removed histogramming feature untill new histogram tool is done
**
** Tue Nov  3 10:18:52 1992 mazer
**  made it easier to have multiple instantiations of traceunits,
**  though there's still a "main_trace" for the discriminator
**
** Wed Nov  4 17:52:34 1992 mazer
**  -- moved SaveTraceAction to xdowl:SaveTraceCB()
**  -- only one window can set the discrim parameters (main_trace)
**
** Wed Feb 10 13:07:27 1993 mazer
**  moved discrim buttons (upper,lower,time) out so they can
**  be placed on the main xdowl form
**
** Tue Mar  9 18:23:43 1993 mazer
**  TraceUnit2 and traceunit_displayfile() stuff removed.. this
**  must be re-written!
**
** Sun Mar 14 14:44:33 1993 mazer
**  this module has now been complete recoded to support stand
**  alone trace displays .. TraceUnit() is dead, long live the
**  tracers!
**
** Sun Jan 23 19:24:38 1994 mazer
**  added option to tracer_new() so application without the
**  sorter can eliminate the sorter/mu-only buttons
**
** Tue Nov  1 19:00:31 1994 mazer
**  changed tracer_update() to subtract dc-offset from waveforms
**
** 99.02 bjarthur
**  re-added the soft window discriminator, from the rcs files.  somehow, this
**  was deleted; turns out it might be quite useful.  modified the gui some.
**
*******************************************************************/

#include "xdphyslib.h"

static void setmask(TRACER *, int);
static void ClearCB(Widget, XtPointer, XtPointer);
static void IncMaskCB(Widget, XtPointer, XtPointer);
static void DecMaskCB(Widget, XtPointer, XtPointer);
static void TTLMaskCB(Widget, XtPointer, XtPointer);
static void OUTMaskCB(Widget, XtPointer, XtPointer);
static void MUMaskCB(Widget, XtPointer, XtPointer);
static void CloneCB(Widget, XtPointer, XtPointer);
static void DestroyTracer(TRACER *);
static void CloseitCB(Widget, XtPointer, XtPointer);
static void SettingsCB(Widget, XtPointer, XtPointer);
static void tracer_update_histograms(TRACER *, spike_t *, int);
static int calc_sigmas(int, xword *, float, int *);

static void SetCurrAction(Widget w, XEvent * event, String * params,
			  Cardinal * num_params)
{
	int gsr_window_flag;
	assert((*num_params) == 1);

	gsr_window_flag = (getRaster_getmode() == GSR_window) ? 1 : 0;

	if (gsr_window_flag) {
		if (!strcmp(params[0], "thresh")) {
			G_detect_curr = 0;
		} else if (!strcmp(params[0], "one")) {
			G_detect_curr = 1;
		} else if (!strcmp(params[0], "two")) {
			G_detect_curr = 2;
		} else if (!strcmp(params[0], "three")) {
			G_detect_curr = 3;
		}
	}
}

static void ToggleColorAction(Widget w, XEvent * event, String * params,
			      Cardinal * num_params)
{
	int gsr_window_flag;
	assert((*num_params) == 0);

	gsr_window_flag = (getRaster_getmode() == GSR_window) ? 1 : 0;

	if (gsr_window_flag) {
		if (G_detect_color[G_detect_curr])
			G_detect_color[G_detect_curr] = 0;
		else
			G_detect_color[G_detect_curr] = 1;
	}
}

static void ToggleOnOffAction(Widget w, XEvent * event, String * params,
			      Cardinal * num_params)
{
	int gsr_window_flag;
	assert((*num_params) == 0);

	gsr_window_flag = (getRaster_getmode() == GSR_window) ? 1 : 0;
	if (gsr_window_flag) {
		assert((*num_params) == 0);

		if (G_detect_onoff[G_detect_curr])
			G_detect_onoff[G_detect_curr] = 0;
		else
			G_detect_onoff[G_detect_curr] = 1;
	}
}

static void ResetAction(Widget w, XEvent * event, String * params,
			Cardinal * num_params)
{
	int gsr_window_flag;
	int i;
	assert((*num_params) == 0);

	gsr_window_flag = (getRaster_getmode() == GSR_window) ? 1 : 0;

	if (gsr_window_flag) {
		if (G_detect_curr == 0) {
			xword *ibuf;
			int count, i, j; 
			int nchans = is_getDAnchans();
			float mean;

			if (is_inited) {
				if (!spiker_getTTLbuffer(&count, &ibuf)) {
					alert("ResetAction: no ttl buffer found");
					return;
				}

				for (j = 0, i = 0, mean = 0.0; j < count; j++, i += nchans)
					mean += ibuf[i];
				mean /= count;
				G_detect_y[0] = mean;
			} else {
				G_detect_y[0] = 0;
			}
		} else {
			G_detect_onoff[G_detect_curr] = 0;
			G_detect_color[G_detect_curr] = 0;
			G_detect_x[G_detect_curr] = 0;
			G_detect_y[G_detect_curr] = G_detect_y[0];
		}
	} else {
		for (i = 0; i < 4; i++) {
			G_detect_onoff[i] = 0;
			G_detect_color[i] = 1;
			G_detect_x[i] = 0;
			G_detect_y[i] = 0;
		}
	}
}

static void ResetAllAction(Widget w, XEvent * event, String * params,
			   Cardinal * num_params)
{
	int gsr_window_flag;
	xword *ibuf;
	int count, i, j, nchans = is_getDAnchans();
	float mean;

	assert((*num_params) == 0);

	gsr_window_flag = (getRaster_getmode() == GSR_window) ? 1 : 0;
	if (gsr_window_flag) {

		if (is_inited) {
			if (!spiker_getTTLbuffer(&count, &ibuf)) {
				alert("ResetAction: no ttl buffer found");
				return;
			}

			for (j = 0, i = 0, mean = 0.0; j < count; j++, i += nchans)
				mean += ibuf[i];
			mean /= count;
			G_detect_y[0] = mean;
		} else 
			G_detect_y[0] = 0;

		for (i = 1; i < 4; i++) {
			G_detect_onoff[i] = 0;
			G_detect_color[i] = 0;
			G_detect_x[i] = 0;
			G_detect_y[i] = G_detect_y[0];
		}

		G_detect_curr = 0;
	} else {
		for (i = 0; i < 4; i++) {
			G_detect_onoff[i] = 0;
			G_detect_color[i] = 1;
			G_detect_x[i] = 0;
			G_detect_y[i] = 0;
		}
	}
}

static void AdjustAction(Widget w, XEvent * event, String * params,
			 Cardinal * num_params)
{
	int i, jump_size;
	static int key_accel = 1;
	static Time last_timestamp = 0;
	int gsr_window_flag;

	assert((*num_params) == 1);

	gsr_window_flag = (getRaster_getmode() == GSR_window) ? 1 : 0;
	if (gsr_window_flag) {
		jump_size = GI("detect.jump");

		if (event->type == KeyPress) {
			if (last_timestamp && (event->xkey.time - last_timestamp) < 200)
				key_accel++;
			else
				key_accel = 1;
			last_timestamp = event->xkey.time;
		} else {
			key_accel = 1;
		}

		if (!strcasecmp(params[0], "up")) {
			i = G_detect_y[G_detect_curr] + (jump_size * key_accel);
			if (i > RANGE) {
				i = RANGE;
			}
			G_detect_y[G_detect_curr] = i;
		} else if (!strcasecmp(params[0], "down")) {
			i = G_detect_y[G_detect_curr] - (jump_size * key_accel);
			if (i < -RANGE) {
				i = -RANGE;
			}
			G_detect_y[G_detect_curr] = i;
		} else if (!strcasecmp(params[0], "left")) {
			G_detect_x[G_detect_curr]--;
		} else if (!strcasecmp(params[0], "right")) {
			G_detect_x[G_detect_curr]++;
		}
	}
}

static void install_discrim_actions(Widget w)
{
	static XtActionsRec actions[] =
	{
		{"discrim-setcurr", SetCurrAction},
		{"discrim-togglecolor", ToggleColorAction},
		{"discrim-toggleonoff", ToggleOnOffAction},
		{"discrim-reset", ResetAction},
		{"discrim-resetall", ResetAllAction},
		{"discrim-adjust", AdjustAction},
	};

	static String bindings = "#override \n\
	      :<KeyPress>t:	discrim-setcurr(thresh) \n\
	      :<KeyPress>1:	discrim-setcurr(one) \n\
	      :<KeyPress>2:	discrim-setcurr(two) \n\
	      :<KeyPress>3:	discrim-setcurr(three) \n\
	      :<KeyPress>c:	discrim-togglecolor() \n\
	      :<KeyPress>o:	discrim-toggleonoff() \n\
	      :<KeyPress>r:	discrim-reset() \n\
	      :<KeyPress>a:	discrim-resetall() \n\
	      :<KeyPress>h:	discrim-adjust(left) \n\
	      :<KeyPress>l:	discrim-adjust(right) \n\
	      :<KeyPress>j:	discrim-adjust(down) \n\
	      :<KeyPress>k:	discrim-adjust(up) \n";

/*
   static String bindings = "\
   <Key>KP_7:   discrim-adjust(lower,up)\n\
   <Key>KP_1:   discrim-adjust(lower,down)\n\
   <Key>KP_9:   discrim-adjust(upper,up)\n\
   <Key>KP_3:   discrim-adjust(upper,down)\n\
   <Key>Up:     discrim-adjust(window,up)\n\
   <Key>Down:   discrim-adjust(window,down)\n\
   <Key>Left:   discrim-adjust(time,shrink)\n\
   <Key>Right:  discrim-adjust(time,grow)\n\
   <Key>KP_5:   discrim-adjust(lower,zero)\n\
   <Key>t:              discrim-set(time)\n\
   <Key>u:              discrim-set(upper)\n";
 */

	XtAppAddActions(XtWidgetToApplicationContext(w), actions, XtNumber(actions));
	XtOverrideTranslations(w, XtParseTranslationTable(bindings));
}

void tracer_add_discrim(TRACER * t)
{
/*
   t->discrim = toggle_new(t->scopeform, "vu disc",
   t->ac_coup, NULL, NULL, NULL, NULL, False);
   toggle_set(t->discrim, False);
 */
	install_discrim_actions(t->scope);
}

static void setmask(TRACER * t, int i)
{
	char buf[20];

	if (t->mask_button) {
		t->mask = i;
		switch (t->mask) {
		case TTL_CODE:
			XtVaSetValues(t->mask_button, XtNlabel, "TTL", NULL);
			break;
		case OUTLIER_CODE:
			XtVaSetValues(t->mask_button, XtNlabel, "OUT", NULL);
			break;
		case MU_CODE:
			XtVaSetValues(t->mask_button, XtNlabel, "MU", NULL);
			break;
		default:
			sprintf(buf, "#%02d", t->mask);
			XtVaSetValues(t->mask_button, XtNlabel, buf, NULL);
		}
	}
}

static void ClearCB(Widget w, XtPointer t_blah, XtPointer call_data)
{
	TRACER *t = (TRACER *) t_blah;

	athisto_clear(t->psth);
	athisto_clear(t->isih);
	athisto_update(t->psth);
	athisto_update(t->isih);
	dynraster_clear(t->raster);
}

#if(0)				/* doesn't work; would be involved to fix */
void ClearTracerAction(Widget w, XEvent * event, String * params,
		       Cardinal * num_params)
{
	ClearCB(NULL, NULL, NULL);
}

#endif

static void IncMaskCB(Widget w, XtPointer t_blah, XtPointer call_data)
{
	TRACER *t = (TRACER *) t_blah;

	setmask(t, t->mask + 1);
	ClearCB(w, t, call_data);
}

static void DecMaskCB(Widget w, XtPointer t_blah, XtPointer call_data)
{
	TRACER *t = (TRACER *) t_blah;

	if (t->mask > MU_CODE) {
		setmask(t, t->mask - 1);
		ClearCB(w, t, call_data);
	}
}

static void TTLMaskCB(Widget w, XtPointer t_blah, XtPointer call_data)
{
	TRACER *t = (TRACER *) t_blah;

	setmask(t, TTL_CODE);
	ClearCB(w, t, call_data);
}

static void OUTMaskCB(Widget w, XtPointer t_blah, XtPointer call_data)
{
	TRACER *t = (TRACER *) t_blah;

	setmask(t, OUTLIER_CODE);
	ClearCB(w, t, call_data);
}

static void MUMaskCB(Widget w, XtPointer t_blah, XtPointer call_data)
{
	TRACER *t = (TRACER *) t_blah;

	setmask(t, MU_CODE);
	ClearCB(w, t, call_data);
}

static void CloneCB(Widget w, XtPointer t_blah, XtPointer call_data)
{
	TRACER *t = (TRACER *) t_blah;
	TRACER *t2;		/* get it? */

	/* clone button only avail with ssbuttons == True anyway, so clone
	 *  should always assume ssbuttons to be true..
	 */
	t2 = tracer_new(NULL, 0, t->wd, t->ht, t);
	setmask(t2, t->mask);
/*
   t2->ms_pre = t->ms_pre;
   t2->ms_post = t->ms_post;
   toggle_set(t2->autoscale, toggle_get(t->autoscale));
 */
	toggle_set(t2->active, toggle_get(t->active));
}

static void DestroyTracer(TRACER * t)
{
	if (t->prev)
		t->prev->next = t->next;
	if (t->next)
		t->next->prev = t->prev;
	XtPopdown(XtParent(t->form));
	XtDestroyWidget(XtParent(t->form));
	free(t);
}

static void CloseitCB(Widget w, XtPointer t_blah, XtPointer call_data)
{
	TRACER *t = (TRACER *) t_blah;

	if (t->prev == NULL && t->next == NULL) {
		alert("Can't kill last tracer");
	} else {
		if (progRunning)
			t->killme = 1;
		else
			DestroyTracer(t);
	}
}


static void SettingsCB(Widget w, XtPointer t_blah, XtPointer call_data)
{
	TRACER *t = (TRACER *) t_blah;

	notify("Mask=%d, pre=%dms, post=%dms", t->mask, GI("mspre"), GI("mspost"));
}

TRACER *tracer_new(Widget parent, int anatrace, int width, int height,
		   TRACER * last)
{
	int popit, splitby;
	Widget x, hform;
	TRACER *t;
	Cardinal asdf = 0;

	t = (TRACER *) malloc(sizeof(TRACER));

	if (parent == NULL) {
		popit = 1;
		parent = top_new(TopLevel, "tracer");
	} else {
		popit = 0;
	}

	t->mask = -1;

	t->killme = 0;
	if (anatrace) {
		t->wd = width;
		t->ht = (int) (0.5 + (0.66 * height));
		splitby = 4;
	} else {
		t->wd = width;
		t->ht = height;
		splitby = 3;
	}

	if ((t->prev = last) != NULL) {
		t->next = last->next;
		if (t->next)
			t->next->prev = t;
		last->next = t;
	} else {
		t->next = NULL;
	}

	t->form = form_new(parent, "tracer", 0);

	x = NULL;
	if (popit)
		x = button_new(t->form, "Close", "Close", CloseitCB, t, NULL, NULL);

	t->active =
	    x = toggle_new(t->form, "On", "On", x, NULL, NULL, NULL, NULL, True);
	x = button_new(t->form, "Clear", "Clear", ClearCB, t, x, NULL);

	t->mask_button = t->ssbutts[0] =
	    x = button_new(t->form, "mask", "mask", MUMaskCB, t, x, NULL);
	XtVaSetValues(t->mask_button, XtNresizable, True, NULL);
	setmask(t, -1);
	x = t->ssbutts[1] = button_new(t->form, "++", "++", IncMaskCB, t, x, NULL);
	x = t->ssbutts[2] = button_new(t->form, "--", "--", DecMaskCB, t, x, NULL);
	x = t->ssbutts[3] = button_new(t->form, "->TTL", "->TTL", TTLMaskCB, t, x, NULL);
	x = t->ssbutts[4] = button_new(t->form, "->OUT", "->OUT", OUTMaskCB, t, x, NULL);
	x = t->ssbutts[5] = button_new(t->form, "->?", "->?", SettingsCB, t, x, NULL);
	x = t->ssbutts[6] = button_new(t->form, "Clone", "Clone", CloneCB, t, x, NULL);

	x = t->active;

	t->paned = XtVaCreateManagedWidget("paned", panedWidgetClass, t->form,
				      XtNwidth, width, XtNheight, height,
					XtNorientation, XtorientVertical,
					   XtNfromVert, x, NULL);

	hform = form_new(t->paned, "histograms", -1);
	t->psth = athisto_new(hform, NULL, NULL, ath_linear, NULL, ath_linear,
			      100, 1.0);
	XtVaSetValues(t->psth->plotter,
		      XtNfromHoriz, NULL,
		      XtNwidth, width / 2,
		      XtNheight, height / splitby,
		      NULL);
	t->isih = athisto_new(hform, NULL, NULL, ath_linear, NULL, ath_linear,
			      100, 0.20);
	XtVaSetValues(t->isih->plotter,
		      XtNfromHoriz, t->psth->plotter,
		      XtNwidth, width / 2,
		      XtNheight, height / splitby,
		      XtNhorizDistance, 0,
		      NULL);

	t->raster = dynraster_new(t->paned, width, height / splitby, 20);

	t->scope = canvaswaver_new(t->paned, width, height / splitby);
	XtVaSetValues(t->scope, XtNfromVert, t->raster, NULL);
	ResetAllAction(t->scope, NULL, NULL, &asdf);

	if (anatrace) {
		t->wave = XtVaCreateManagedWidget("anatrace", waveWidgetClass, t->paned,
						  XtNwidth, width,
					     XtNheight, height / splitby,
						  NULL);
	} else {
		t->wave = NULL;
	}

	if (popit) {
		Position xpos, ypos;

		getGeometry(parent, &xpos, &ypos);
		if (xpos != 0 && ypos != 0) {
			popup_at(parent, xpos, ypos);
		} else {
			popup(parent);
		}
	}
	return (t);
}

void tracer_clear(TRACER * t)
{
	ClearCB(NULL, t, NULL);
	if (t->next)
		tracer_clear(t->next);
}

static void tracer_update_histograms(TRACER * t, spike_t * spikelist,
				     int decay_flag)
{
	int i, n;
	int update_isih, update_psth;
	int psth_nbins, isih_nbins;
	float psth_bwidth, isih_bwidth;
	float ms_time, last_time;

	psth_bwidth = GF("trace.psth_bin");	/* psth bin width */
	isih_bwidth = GF("trace.isih_bin");	/* isih bin width */

	psth_nbins = 0.5 + GI("epoch") / psth_bwidth;
	isih_nbins = 0.5 + (GI("trace.isih_max") / isih_bwidth);

	/* mask = (!strcmp(GS("detect.mode"),"ss")) ? t->mask : MU_CODE; */

	update_psth = update_isih = 0;
	last_time = -1.0;

	if (decay_flag) {
		update_psth = update_isih = 1;
		athisto_decay(t->psth, GI("trace.decay"));
		athisto_decay(t->isih, GI("trace.decay"));
	}
	if (spikelist != NULL)
		for (n = i = 0; i < N_SPIKES(spikelist); i++) {
			if (matchEvent(SPIKE_CHAN(spikelist, i), t->mask)) {
				n++;
				update_psth = 1;
				ms_time = SPIKE_MSEC(spikelist, i);
				athisto_add(t->psth, ms_time, psth_nbins, psth_bwidth);
				if (last_time >= 0.0) {
					if (athisto_add(t->isih, ms_time - last_time, isih_nbins, isih_bwidth))
						update_isih = 1;
				}
				last_time = ms_time;
			}
		}
	if (update_psth)
		athisto_update(t->psth);
	if (update_isih)
		athisto_update(t->isih);
}

static int calc_sigmas(int nsamps, xword * data, float n, int *meanoffset)
{
	register int i, k;
	register double mean = 0.0, sig = 0.0;
	extern double sqrt();

	for (i = 0, k = 2 * nsamps; i < k; i += 2)
		mean += data[i];
	mean /= nsamps;

	for (i = 0, k = 2 * nsamps; i < k; i += 2)
		sig += (data[i] - mean) * (data[i] - mean);
	sig /= (float) (nsamps - 1);
	if (meanoffset != NULL) {
		if (mean > 0) {
			*meanoffset = (int) (mean + 0.5);
		} else {
			*meanoffset = (int) (mean - 0.5);
		}
	}
	return ((int) (0.5 + n * sqrt((double) sig)));
}

void tracer_update( TRACER * t,
			  spike_t * spikelist,
			  xword * anabuf,
			  int nsamps,
			  int c1,	/* from adtick */
			  int c2,	/* to adtick */
			  int decay_flag)
{
	int pre, post;
	int fc;
	int max;
	int mask;
	xword **anabufs;

	if (anabuf == NULL) {
		spiker_getAnalogBuffers(&nsamps, &anabufs);
		anabuf = anabufs[GI("detect.tracer_channel")-1];
		free(anabufs);
	}

	if (t->killme) {
		/* this little hack to to avoid closing window in the middle
		 * of an update event .. delete's are postponed while running
		 * until a safe time to kill the windows off..
		 */
		if (t->next)
			tracer_update(t->next, spikelist, anabuf, nsamps, c1, c2, decay_flag);
		DestroyTracer(t);
	} else {
		if (toggle_get(t->active)) {
			if (t->wave != NULL) {
				int mean, sigma;

				sigma = calc_sigmas(nsamps, anabuf, (float) 3.0, &mean);
				sigma -= mean;
				mean = 0;

#ifndef __tdtproc__
				fc = is_evtFc;
#else				/* __tdtproc__ */
				fc = is_adFc;
#endif				/* __tdtproc__ */
				if (c1 != c2)
					XtVaSetValues(t->wave,
					  XtNcursor1, c1, XtNcursor2, c2,
						      XtNhorizCursor1, sigma, XtNhorizCursor2, -sigma,
						      XtNshowHorizCursor1, True, XtNshowHorizCursor2, True,
						      XtNfc, fc, NULL);
				else
					XtVaSetValues(t->wave,
						      XtNhorizCursor1, sigma, XtNhorizCursor2, -sigma,
						      XtNshowHorizCursor1, True, XtNshowHorizCursor2, True,
						      XtNfc, fc, NULL);

				XtWaveSetWaveformOffset(t->wave,
				 anabuf, nsamps, 2, -RANGE, RANGE, mean);
			}
			if (t->psth && t->isih)
				tracer_update_histograms(t, spikelist, decay_flag);

			pre = RND2INT(GF("mspre") * (float) is_adFc / 1000.0);
			post = RND2INT(GF("mspost") * (float) is_adFc / 1000.0);

			if (pre < 0)
				pre = 1 * is_adFc / 1000;
			if (post < 0)
				post = 2 * is_adFc / 1000;

			max = (int) (1000.0 * (float) nsamps / (float) is_adFc);
/** WARNING: this assumes that adFc is the same as evtFc..  */

			/* mask = (!strcmp(GS("detect.mode"),"ss")) ? t->mask : MU_CODE; */
			dynraster_add(t->raster, max, spikelist, t->mask, 0, 0);

			plotUnitPerievent(t->scope, anabuf, 0, nsamps, pre, post,
					  GI("maxTraces"), spikelist, t->mask, 1, GI("autoscale"));
		}
		if (t->next)
			tracer_update(t->next, spikelist, anabuf, nsamps, c1, c2, decay_flag);
	}
}
