/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: svars.c,v 2.56 2001/03/27 06:59:27 cmalek Exp $
** Program: dowl
**  Module: svars.c
**  Author: mazer
** Descrip: hash table interface for run-time variables
**
** Revision History (most recent last)
**
** Fri Jun  5 15:20:01 1992 mazer
**  "version" --> get current version number
**  "time"    --> get current time string
**  "DOWLDIR" --> current dowl-lib directory name (from environment)
**  "LIBDIR"  --> current dowl-lib directory name (same as "DOWLDIR")
**  "foo"     --> get value from svar named "foo"
**  "$foo"    --> get value from file "./.foo"
**  "!foo"    --> override specials and get value from svar "foo"
**                use this to explicitly ask for something like "time"
**		  when parsing a datafile.
**
** Tue Jun 16 01:42:31 1992 mazer
**  extensive changes to allow for a dynamic hash tables (as a step
**  towards doing away with the shadowing garbage).
**
** Fri Sep 18 10:12:02 1992 mazer
**   merging changes dhb made for monkey vision applications
**  
** Mon Nov  2 15:23:15 1992 mazer
**  -- shadow'd hash tables are now gone -- this was really UGLY
**  -- changed default_hash_table to more accurate global_hash_table
**  -- added read/write hooks:
**      Read hook called before read takes place, write called
**      after write takes place. During hook execution, hooks are
**      disabled so that hook routines can call getvar/setvar safely
**  
** Sun Dec  6 11:41:58 1992 mazer
**  Set things up so there's a SVAR* and a separate SVAR_TABLE*.
**  This means allows for hash tables of differing sizes, so save
**  space when someone gets around to taking advantage of this..
**
** Wed Feb  3 23:43:50 1993 mazer
**  varsnew() and new_hash_table() now take a size argument
**
** Sat Feb 27 11:03:59 1993 mazer
**  added getvar_match() - lookups svars based on simple regular
**  expressions. Usage: getvar_match(pattern, htable, &n) can be
**  called repeatedly until in returns NULL, indicating no more
**  matches -- &n is used as an internal pointer to the last found
**  match. Initialize to (-1) and leave it alone..
**
** Mon Mar 15 18:26:17 1993 mazer
**  added another special var:
**  "timestamp"    --> get current time (in secs past
**				 some magic date in 1970..)
**
** Thu Sep 16 20:30:12 1993 mazer
**  removed LIBDIR and DOWLDIR stuff -- use getenv instead!
**
** Mon Nov 22 15:01:48 1993 mazer
**  added getvar_bool() and GB()
**
** Mon Nov 22 15:20:09 1993 mazer
**  added getprefs() and as_int(), as_float(), as_bool()
**
** Thu Feb 24 00:21:17 1994 mazer
**  added another variable-value class:
**    "<foo" reads input from file foo, if possible
**
** 97.4 bjarthur
**  added vars2ascii
**
** 99.2 bjarthur
**  86'd xdphys.gsrmode.  what a hack that was.  nmodels and caltype
**  need to go too.
**
*******************************************************************/

#include "xdphyslib.h"

/*
** hash table interface for a database of variables
**
** char *getvar_match("pattern", hashtab, &int);
**
** char *getvar("name", hashtab);			-- aka GS()
** int   getvar_int("name", hashtab);			-- aka GI()
** float getvar_float("name", hashtab);			-- aka GF()
**
** char *setvar("name", "value", hashtab);		-- aka SS()
** int   setvar_int("name", int j, hashtab);		-- aka SI()
** float setvar_float("name", float x, hashtab);	-- aka SF()
**
** int loadvars("filename", hashtab);
** int savevars("filename", hashtab);
**
** NOTES
**     ^<name> -- just don't save in state files..
**     $<name> -- means look in file <name> for the value
**     version -- returns the version string for current program
**     time    -- returns a timestamp string (human readable)
**
** ABOUT STORAGE
**   setvar() makes an internal strcpy and saves a COPY of the value
**   getvar() returns NULL or a point to it's internal COPY of the
**      value of the svar. This is good until the next time that save
**	gets set .. then you're screwed .. so don't old on to it!
**   -> This should NEVER be passed to free().
**
**
** Fri Jun 11 01:17:12 1993 mazer
**  svar_post_changed_hook: gets called AFTER a variable has been
**  changed.
*/

/*
** at some point, INITIAL_HSIZE should become dynamic by incorporation into
** the hashtable structure itself.
*/
#define INITIAL_HSIZE	691
#define AVAIL		(-1)

void (*svar_read_hook)() = NULL;
void (*svar_write_hook)() = NULL;
void (*svar_changed_hook)() = NULL;
void (*svar_post_changed_hook)() = NULL;

int LastVarUndefined = 0;
SVAR_TABLE *global_svar_table = NULL;

static int hash_string(char*);
static SVAR *sv_lookup(char*, SVAR_TABLE*);
static char *upcase(char*);
static void varssort(SVAR_TABLE*);  /* not initially static */

SVAR_TABLE *new_hash_table(SVAR_TABLE *clonefrom, int size)
{
  SVAR_TABLE *new_tab;
  int i;

  new_tab = (SVAR_TABLE *) malloc(sizeof(SVAR_TABLE));
  if (size < 0)
    new_tab->svt_size = INITIAL_HSIZE;
  else
    new_tab->svt_size = size;
  new_tab->svt_head = NULL;
  new_tab->svt_table = (SVAR *) malloc(new_tab->svt_size * sizeof(SVAR));
  for (i = 0; i < new_tab->svt_size; i++) {
    if (clonefrom == NULL || clonefrom->svt_table[i].sv_hashval == AVAIL) {
      new_tab->svt_table[i].sv_hashval = AVAIL;
      new_tab->svt_table[i].sv_name = NULL;
      new_tab->svt_table[i].sv_value = NULL;
    } else {
      new_tab->svt_table[i].sv_hashval =
	clonefrom->svt_table[i].sv_hashval;
      new_tab->svt_table[i].sv_name =
	strsave(clonefrom->svt_table[i].sv_name);
      new_tab->svt_table[i].sv_value =
	strsave(clonefrom->svt_table[i].sv_value);
    }
  }
  return(new_tab);
}

void initsvars(void)
{
  if (global_svar_table == NULL)
    global_svar_table = new_hash_table(NULL, -1);
}

static int hash_string(char *p)     /* my own ugly design! */
{
  register int i, hash;
#ifndef __linux__
  extern char toupper();
#endif __linux__

  for (i = strlen(p), hash = 0; *p; i--, p++)
    if (islower(*p))
      hash += (i * toupper(*p));
    else
      hash += (i * *p);
  return(hash >> 2);
}

void sv_stats(SVAR_TABLE *hashtab)
{
  register int hashed_name, start, i, j;
  int nprobes, nitems, done, worst, n;
  char *name;

  nprobes = 0;
  nitems = 0;
  worst = 0;
  for (j = 0; j < hashtab->svt_size; j++) {
    if (hashtab->svt_table[j].sv_hashval != AVAIL) {
      hashed_name = hash_string(name = hashtab->svt_table[j].sv_name);
      nitems++;

      start = hashed_name % hashtab->svt_size;
      done = 0;
      n = 0;
      for (i = start; !done && i < hashtab->svt_size; i++) {
	nprobes++; n++;
	if (hashtab->svt_table[i].sv_name != NULL &&
	    hashtab->svt_table[i].sv_hashval == hashed_name &&
	    strcasecmp(hashtab->svt_table[i].sv_name, name) == 0)
	  done = 1;
      }
      for (i = 0; !done && i < start; i++) {
	nprobes++; n++;
	if (hashtab->svt_table[i].sv_name != NULL &&
	    hashtab->svt_table[i].sv_hashval == hashed_name &&
	    strcasecmp(hashtab->svt_table[i].sv_name, name) == 0)
	  done = 1;
      }
      if (n > worst)
	worst = n;
    }
  }
  printf("%d items (%d max): avg search = %.2f probes (%d worst)\n",
	 nitems, hashtab->svt_size, (float) nprobes / (float) nitems, worst);
}

static SVAR *sv_lookup(char *name, SVAR_TABLE *hashtab)
{
  register int hashed_name, start, i;

  if (hashtab == NULL) {
    fprintf(stderr, "sv_lookup: called with uninitialized hashtab\n");
    fprintf(stderr, "           (fyi: looking up \"%s\")\n", name);
    exit(1);
  }

  hashed_name = hash_string(name);
  start = hashed_name % hashtab->svt_size;

  for (i = start; i < hashtab->svt_size; i++)
    if (hashtab->svt_table[i].sv_name != NULL &&
	hashtab->svt_table[i].sv_hashval == hashed_name &&
	strcasecmp(hashtab->svt_table[i].sv_name, name) == 0)
      return(&hashtab->svt_table[i]);
  for (i = 0; i < start; i++)
    if (hashtab->svt_table[i].sv_name != NULL &&
	hashtab->svt_table[i].sv_hashval == hashed_name &&
	strcasecmp(hashtab->svt_table[i].sv_name, name) == 0)
      return(&hashtab->svt_table[i]);
  return(NULL);
}

static char *upcase(char *p)
{
  char *q = strcpy((char *)malloc(strlen(p) + 1), p);
#ifndef __linux__
  extern char toupper();
#endif __linux__

  p = q;
  while (*q) {
    if (islower(*q))
      *q = toupper(*q);
    q++;
  }
  return(p);
}

char *getvar_match(char *pattern, SVAR_TABLE *hashtab, int *next)
{
  char *Upattern, *Uname;

  if (hashtab == NULL) {
    fprintf(stderr, "getvar_match: called with uninitialied hashtab\n");
    exit(1);
  }

  if (*next < 0)
    *next = 0;

  Upattern = upcase(pattern);
  while (*next < hashtab->svt_size) {
    if (hashtab->svt_table[*next].sv_name != NULL) {
      Uname = upcase(hashtab->svt_table[*next].sv_name);
	if (match(Upattern, Uname)) {
	  free(Upattern);
	  free(Uname);
	  return(hashtab->svt_table[*next].sv_value);
	}
      free(Uname);
    }
    (*next)++;
  }
  free(Upattern);
  return(NULL);
}

char *getvar(char *name, SVAR_TABLE *hashtab)
{
	extern time_t time();
	time_t t;
	char *p, *value;
	SVAR *v;
	FILE *fp;
	static char file_value[100];
	static char versionstr[100] = "";

	if (svar_read_hook != NULL) {
		void (*hook) ();

		hook = svar_read_hook;
		svar_read_hook = NULL;
		(*hook) (name, hashtab);
		svar_read_hook = hook;
	}

	LastVarUndefined = 0;

	if (name != NULL && *name == '$') {
		p = (char *) malloc(strlen(name) + 1);
		sprintf(p, ".%s", name + 1);
		fp = fopen2(p, "r");
		free(p);
		if (fp != NULL && fgets(file_value, sizeof(file_value), fp) != NULL) {
			if ((p = rindex(file_value, '\n')) != NULL)
				*p = '\000';
			value = file_value;
		} else {
			LastVarUndefined = 1;
			value = NULL;
		}
		if (fp != NULL)
			fclose(fp);
	} else if (*name == '<') {
		int l = filesize(name + 1);
		if (l && (fp = fopen2(name + 1, "r")) != NULL) {
			value = calloc(l + 1, sizeof(char));
			fread(value, l, sizeof(char), fp);
			fclose(fp);
		} else {
			value = "No file";
		}
	} else if (!strcasecmp(name, "xdphys.caltype")) {
		value = GS("caltype");
	} else if (!strcasecmp(name, "xdphys.gsrnmodels")) {
		static char buf[100];
		int foo;
		foo = getRaster_getnmodels();
		sprintf(buf, "%d", foo);
		value = buf;
	} else if (!strcasecmp(name, "version")) {
		if (*versionstr == 0)
			sprintf(versionstr, "%s Version %s", basename(progname),
				XDPHYS_VERSION);
		value = versionstr;
	} else if (!strcasecmp(name, "timestamp")) {
		static char buf[100];
		t = time((long *) 0);
		sprintf(buf, "%ld", t);
		value = buf;
	} else if (!strcasecmp(name, "time")) {
		static char buf[100];
#ifdef sgi
		extern size_t strftime();
#else
#ifndef __linux__
		extern int strftime();
#endif				/* __linux__ */
#endif
		t = time((long *) 0);
		strftime(buf, sizeof(buf), "%a %h %d %r %Y %Z", localtime(&t));
		value = buf;
	} else if (*name == '!' && (v = sv_lookup(name + 1, hashtab)) != NULL) {
		value = v->sv_value;
	} else if ((v = sv_lookup(name, hashtab)) != NULL) {
		value = v->sv_value;
	} else if (!strcmp(name, "Spont_Stims")) {
		value = "1";
	} else {
		LastVarUndefined = 1;
		value = NULL;
	}

	/* isn't backwords compatibility great! */
	if (value == NULL) {
		if (!strcasecmp(name, "detect.mode"))
			value = getvar("xdphys.gsrmode", hashtab);
		else if (!strcasecmp(name, "xdphys.gsrmode"))
			value = getvar("dowl.gsrmode", hashtab);
	}

	if (LastVarUndefined && debugflag)
		fprintf(stderr, "warning: svars.getvar(%s) = undefined\n", name);
	return (value);
}

int as_int(char *p)
{
	int tmp;

	if (p != NULL)
		sscanf(p, "%d", &tmp);
	else
		tmp = 0;
	return(tmp);
}

float as_float(char *p)
{
	float tmp;

	if (p != NULL)
		sscanf(p, "%f", &tmp);
	else
		tmp = 0.0;
	return(tmp);
}

int as_bool(char *p)
{
#define ISIT(q) (strcasecmp(p,q) == 0)

  if (p &&
      (ISIT("yes") || ISIT("y") ||
       ISIT("true") || ISIT("t") || ISIT("on") ||
       atoi(p))) {
    return(1);
  } else {
    return(0);
  }

#undef IS
}

int getvar_int(char *p, SVAR_TABLE *hashtab)
{
  return(as_int(getvar(p, hashtab)));
}

float getvar_float(char *p, SVAR_TABLE *hashtab)
{
  return(as_float(getvar(p, hashtab)));
}

int getvar_bool(char *p, SVAR_TABLE *hashtab)
{
  return(as_bool(getvar(p, hashtab)));
}

char *setvar(char *name, char *value, SVAR_TABLE *hashtab)
{
  SVAR *v;
  FILE *fp;
  char *p;
  register int hashed_name, start, i;
  int call_post_changed_hook = 0;

  if (svar_changed_hook || svar_post_changed_hook) {
    if (((p = getvar(name, hashtab)) == NULL) || (strcmp(p, value) != 0)) {
      if (svar_changed_hook)
				(*svar_changed_hook)(name, p, value, hashtab);
      if (svar_post_changed_hook)
				call_post_changed_hook = 1;
    }
  }

  if (name && *name == '$') {
    p = (char *) malloc(strlen(name) + 1);
    sprintf(p, ".%s", name + 1);
    if ((fp = fopen2(p, "w")) != NULL) {
      fputs(value, fp);
      fclose(fp);
    }
    free(p);
  } else if ((v = sv_lookup(name, hashtab)) != NULL) {
    if (v->sv_value != NULL)
      free(v->sv_value);
    v->sv_value = strsave(value);
  } else {
    if (hashtab == NULL) {
      fprintf(stderr, "setvar: called with uninitialied hashtab\n");
      exit(1);
    }
    hashed_name = hash_string(name);
    start = hashed_name % hashtab->svt_size;
    for (i = start; i < hashtab->svt_size; i++)
      if (hashtab->svt_table[i].sv_name == NULL)
				break;
    if (i==hashtab->svt_size /*hashtab->svt_table[i].sv_name != NULL*/) {
      for (i = 0; i < start; i++)
				if (hashtab->svt_table[i].sv_name == NULL)
				  break;
      if (hashtab->svt_table[i].sv_name != NULL) {
				fprintf(stderr, "setvar: out of hash space\n");
				exit(1);
      }
    }
    hashtab->svt_table[i].sv_hashval = hashed_name;
    hashtab->svt_table[i].sv_name = strsave(name);
    hashtab->svt_table[i].sv_value = strsave(value);
    hashtab->svt_table[i].sv_next = hashtab->svt_head;
    hashtab->svt_head = &(hashtab->svt_table[i]);
  }

  if (call_post_changed_hook)
    (*svar_post_changed_hook)(name, hashtab);

  if (svar_write_hook != NULL) {
    void (*hook)();

    hook = svar_write_hook;
    svar_write_hook = NULL;
    (*hook)(name, value, hashtab);
    svar_write_hook = hook;
  }

  return(value);
}

int setvar_int(char *name, int value, SVAR_TABLE *hashtab)
{
  static char buf[100];

  sprintf(buf, "%d", value);
  setvar(name, buf, hashtab);
  return(value);
}

float setvar_float(char *name, float value, SVAR_TABLE *hashtab)
{
  static char buf[100];

  sprintf(buf, "%g", value);
  setvar(name, buf, hashtab);
  return(value);
}

int fp_loadvars(
    FILE *fp, 
    SVAR_TABLE *hashtab)
{
  char buf[1000], *p, *q;
  int l;

  while (fgets(buf, sizeof(buf), fp) != NULL) {
    if ((p = index(buf, '#')))
      *p = 0;
    if (strncmp(buf, "END", 3) == 0)
      return(1);
    if ((p = index(buf, '=')) != NULL) {
      *p = '\0';
      if ((q = index(++p, '\n')) != NULL)
				*q = '\0';
      if (strncmp(p, "!BEGIN", 6) != 0)
				setvar(buf, p, hashtab);
      else if (sscanf(p + 7, "%d", &l) ==1 ) {
				p = (char *)malloc(l + 1);
				if (fread(p, sizeof(char), l, fp) < l)
					fprintf(stderr, "warning: short value for %s, not set\n", buf);
				else {
				  p[l] = '\000';
					setvar(buf, p, hashtab); }
				free(p); }
			else {
				fprintf(stderr, "warning: bad value for %s, not set\n", buf);
      }
    }
  }
  return(1);
}

int loadvars(char *fname, SVAR_TABLE *hashtab)
{
  FILE *fp;

  if ((fp = fopen2(findhome(fname), "r")) == NULL) {
    return(0);
  }
  fp_loadvars(fp, hashtab);
  fclose(fp);
  return(1);
}

int fp_savevars(FILE *fp, SVAR_TABLE *hashtab)
{
	register int i;

	for (i = 0; hashtab != NULL && i < hashtab->svt_size; i++)
		if (hashtab->svt_table[i].sv_hashval != AVAIL) {
			if (index(hashtab->svt_table[i].sv_value, '\n') == NULL)
				fprintf(fp, "%s=%s\n",
					hashtab->svt_table[i].sv_name,
					hashtab->svt_table[i].sv_value);
			else {
				fprintf(fp, "%s=!BEGIN %ld\n%s\n",
					hashtab->svt_table[i].sv_name,
					(long) strlen(hashtab->svt_table[i].sv_value),
					hashtab->svt_table[i].sv_value);
			}
		}
	return (1);
}

int savevars(char *fname, SVAR_TABLE *hashtab)
{
  FILE *fp;
  extern int rename();

  if (hashtab == NULL) {
    fprintf(stderr, "savevars: called with uninitialied hashtab\n");
    exit(1);
  }

  fname = strsave(findhome(fname));

  if (probefile(fname)) {
    char backup[MAXPATHLEN];
    sprintf(backup, "%s.bak", fname);
    if ((rename(fname, backup) != 0))
      perror(backup);
  }
    
  if ((fp = fopen2(fname, "w")) == NULL) {
    free(fname);
    return(0);
  }
  fprintf(fp, "# xdphys database file: do not edit by hand\n");
  fprintf(fp, "# %s\n", getvar("time",hashtab));

  fp_savevars(fp,hashtab);

  fprintf(fp, "# end xdphys database file\n");
  fclose(fp);
  free(fname);
  return(1);
}

SVAR *varsfirst(SVAR_TABLE *hashtab)
{
  int i;

  if (hashtab == NULL) {
    fprintf(stderr, "varsfirst: called with uninitialied hashtab\n");
    exit(1);
  }

  for (i = 0; i < hashtab->svt_size; i++) {
    if (hashtab->svt_table[i].sv_name != NULL)
      return(&hashtab->svt_table[i]);
  }
  return(NULL);
}

SVAR *varsnext(SVAR_TABLE *hashtab, SVAR *last)
{
  int i;

  if (hashtab == NULL) {
    fprintf(stderr, "varsnext: called with uninitialied hashtab\n");
    exit(1);
  }

  for (i = last - hashtab->svt_table + 1; i < hashtab->svt_size; i++) {
    if (hashtab->svt_table[i].sv_name != NULL)
      return(&hashtab->svt_table[i]);
  }
  return(NULL);
}

SVAR_TABLE *varsnew(int cloneflag, int size)
{
  return(new_hash_table(cloneflag ? global_svar_table : NULL, size));
}

void varsdelete(SVAR_TABLE *hashtab)
{
  int i;

  if (hashtab == NULL) {
    fprintf(stderr, "varsdelete: called with uninitialied hashtab\n");
    exit(1);
  }

  for (i = 0; i < hashtab->svt_size; i++) {
    if (hashtab->svt_table[i].sv_hashval != AVAIL) {
	free(hashtab->svt_table[i].sv_name);
	free(hashtab->svt_table[i].sv_value);
      }
    }
  free(hashtab->svt_table);
  free(hashtab);
}

static void varssort(SVAR_TABLE *hashtab)
{
  int i, j, t, cnt;
  int *idx;

  if (hashtab == NULL) {
    fprintf(stderr, "varssort: called with uninitialied hashtab\n");
    exit(1);
  }

  for (cnt = i = 0; i < hashtab->svt_size; i++) {
    if (hashtab->svt_table[i].sv_hashval != AVAIL) {
      cnt++;
    }
  }
  if (cnt == 0) {
    hashtab->svt_head = NULL;
    return;
  }
  idx = (int *)malloc(cnt * sizeof(int));
  for (cnt = i = 0; i < hashtab->svt_size; i++) {
    if (hashtab->svt_table[i].sv_hashval != AVAIL) {
      idx[cnt++] = i;
    }
  }
  for (i = 0; i < cnt - 1; i++) {
    for (j = 0; j < cnt - 1; j++) {
      if (strcasecmp(hashtab->svt_table[idx[j]].sv_name,
		 hashtab->svt_table[idx[j+1]].sv_name) > 0) {
	t = idx[j];
	idx[j]=idx[j+1];
	idx[j+1]=t;
      }
    }
  }
  hashtab->svt_head = &hashtab->svt_table[idx[0]];
  for (i = 0; i < cnt - 1; i++)
    hashtab->svt_table[idx[i]].sv_next = &hashtab->svt_table[idx[i+1]];
  hashtab->svt_table[idx[i]].sv_next = NULL;
}

void varsprint(SVAR_TABLE *hashtab)
{
  SVAR *n;

  varssort(hashtab);
  if ((n = hashtab->svt_head) != NULL)
    while (n != NULL) {
      printf("%s=%s\n", n->sv_name, n->sv_value);
      n = n->sv_next;
    }
}


/***************************************************************
 **  This is a simple user-editable preference manager
 **  to handle things not put into worksheets..
 **
 **  It's is hardcoded to read (read-only) ~/.dowlrc/Prefs
 ***************************************************************/

#define PREF_FILE "~/.xdphysrc/Prefs"

char *getpref(char *name, char *def)
{
  int t;
  char *p;
  static time_t last_t = 0;
  static SVAR_TABLE *pref_svar_table = NULL;

  if (name == NULL) {
    if (pref_svar_table) {
      varsdelete(pref_svar_table);
      pref_svar_table = NULL;
    }
    return(NULL);
  } else {
    if (pref_svar_table == NULL)
      pref_svar_table = new_hash_table(NULL, -1);
    t = probefile(PREF_FILE);
    if (t && t != last_t) {
      loadvars(PREF_FILE, pref_svar_table);
      last_t = t;
    }
    p = getvar(name, pref_svar_table);
    return((p != NULL) ? p : def);
  }
}

char *vars2ascii(SVAR_TABLE *params)
{
  SVAR *sv;
  int count,size;
  char *buf,buf2[512];

  assert((buf=(char*)malloc(1024*sizeof(char)))!=NULL);
  size = 1024;

  strcpy(buf,"");  count = 0;
  sv = params->svt_head;
  while(sv) {
    sprintf(buf2, "%s=%s\n", sv->sv_name, sv->sv_value);
    strcat(buf,buf2);  count+=strlen(buf2);
    if((size-count)<512) {
        assert((buf=(char*)realloc(buf,(size+1024)*sizeof(char)))!=NULL);
        size += 1024; }

    sv = sv->sv_next; }

  return(buf);
}
