 /*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: slider.c,v 2.46 1997/05/23 16:55:12 bjarthur Exp $
** Program: dowl
**  Module: slider.c
**  Author: mazer
** Descrip: wrappers around Xaw Scrollbars
**
** Revision History (most recent last)
**
** Sat Feb 15 00:38:33 1992 mazer
**  tried to make scroll bars non-resizable, so they don't get
**  ugly and take up too much space..
** Mon Feb 17 13:19:53 1992 mazer
**  struct _Slider is now an opaque datatype!
** Sat Jul  4 15:21:56 1992 mazer
**  added code to fix the size of the sliders, properly, I think.
**
*******************************************************************/

#include "xdphyslib.h"

#define THUMBSIZE	(float)0.05 /* 5% size'd thumb */

static int nearStop(SliderObj, float);
static void JumpProc(Widget, SliderObj, float*);

static int nearStop(SliderObj s, float perc)
{
  float exact;

  exact = (perc * (s->sl_high - s->sl_low)) + s->sl_low;
  exact /= s->sl_step;
  if (exact < 0.0)
    return(s->sl_step * (int)(-0.5 + exact));
  else
    return(s->sl_step * (int)(0.5 + exact));
}

static void JumpProc(Widget w, SliderObj s, float *percent_ptr)
{
  int value;

  if (s && s->sl_umod) {
    if (s->sl_vert)
      value = nearStop(s, 1.0 - *percent_ptr);
    else
      value = nearStop(s, *percent_ptr);
    slider_set(s, value);
  }
}

void *slider_new(
     Widget parent,
     char *label,			/* text label/title */
     float low, float high, float step,	/* range of values */
     float initval,			/* initial value */
     int vert,				/* verical bar? */
     int filled,			/* filled bar? */
     int umod,				/* user modifiable? */
     int optsize)			/* specify -1 for default */
{
  SliderObj s = (SliderObj) malloc(sizeof(SliderStruct));
  /*
  static String translateTable =
    "<BtnDown>:   StartScroll(Continuous) MoveThumb() NotifyThumb()\n\
     <BtnMotion>: MoveThumb() NotifyThumb()\n\
     <BtnUp>:     NotifyScroll(Proportional) EndScroll()\n";
 */
  Arg args[20];
  int n;
  char *p;

  /*
  static XtTranslations translatedTable = NULL;
  if (translatedTable == NULL) {
    translatedTable = XtParseTranslationTable(translateTable);
  }  
  */

  if (optsize <= 0)
    optsize = 150;

  s->sl_low    = low;
  s->sl_high   = high;
  s->sl_step   = step;
  s->sl_svar   = strsave(label);
  s->sl_vert   = vert;
  s->sl_filled = filled;
  s->sl_umod   = umod;
  if ((p = rindex(label, '.')) != NULL) {
    s->sl_form   = form_new(parent, p + 1, 1);
    s->sl_title  = label_new(s->sl_form, p + 1, NULL, NULL, 0);
  } else {
    s->sl_form   = form_new(parent, label, 1);
    s->sl_title  = label_new(s->sl_form, label, NULL, NULL, 0);    
  }
  s->sl_output = statline_new(s->sl_form, 10);

  n=0;
  if (s->sl_vert) {
	  XtSetArg(args[n], XtNorientation, XtorientVertical); n++;
	  XtSetArg(args[n], XtNwidth, 40); n++;
	  XtSetArg(args[n], XtNheight, optsize); n++;
	  XtSetArg(args[n], XtNfromVert, s->sl_title); n++;
	  XtSetArg(args[n], XtNfromHoriz, NULL); n++;
	  XtSetArg(args[n], XtNhorizDistance, 10); n++;

	  s->sl_sbar = XtCreateManagedWidget("scrollbar", scrollbarWidgetClass, 
		  s->sl_form, args, n);

	  n=0;
	  XtSetArg(args[n], XtNfromVert, s->sl_sbar); n++;
	  XtSetArg(args[n], XtNleft, XawRubber); n++;
	  XtSetArg(args[n], XtNright, XawRubber); n++;
	  XtSetArg(args[n], XtNtop, XawRubber); n++;
	  XtSetArg(args[n], XtNbottom, XawRubber); n++;

	  XtSetValues(s->sl_output, args, n);
  } else {
	  XtSetArg(args[n], XtNorientation, XtorientHorizontal); n++;
	  XtSetArg(args[n], XtNwidth, optsize); n++;
	  XtSetArg(args[n], XtNheight, 20); n++;
	  XtSetArg(args[n], XtNfromVert, s->sl_title); n++;
	  XtSetArg(args[n], XtNfromHoriz, NULL); n++;

	  s->sl_sbar = XtCreateManagedWidget("scrollbar", scrollbarWidgetClass, 
		  s->sl_form, args, n);

	  n=0;
	  XtSetArg(args[n], XtNfromHoriz, s->sl_title); n++;
	  XtSetArg(args[n], XtNleft, XawRubber); n++;
	  XtSetArg(args[n], XtNright, XawRubber); n++;
	  XtSetArg(args[n], XtNtop, XawRubber); n++;
	  XtSetArg(args[n], XtNbottom, XawRubber); n++;

	  XtSetValues(s->sl_output, args, n);
  }

  /*
  XtOverrideTranslations(s->sl_sbar, translatedTable);
  */
  XtAddCallback(s->sl_sbar, XtNjumpProc, (XtCallbackProc) JumpProc, s);
  s->_setCB = NULL;
  slider_set(s, initval);

  return((void *)s);
}

void slider_setCB(SliderObj slider, void (*callbackfn)())
{
  slider->_setCB = callbackfn;
}

float slider_get(SliderObj s)
{
  s->sl_delta = False;
  return(s->sl_value);
}

void slider_set_noCB(
	SliderObj s, 
	float value)
{
  Arg args[20];
  float top, shown;

  if (value < s->sl_low)
    value = s->sl_low;
  else 
	  if (value > s->sl_high)
        value = s->sl_high;
  s->sl_value = value;
  if (s->sl_filled) {
    shown = (value / (s->sl_high - s->sl_low));
    top = 1.0 - shown;
  } else {
	shown = THUMBSIZE;
    if (s->sl_vert) 
	  top = 1.0 - (s->sl_value - s->sl_low)/(s->sl_high - s->sl_low);
	else
      top = (s->sl_value - s->sl_low) / (s->sl_high - s->sl_low);
  }
  if (sizeof(float) > sizeof(XtArgVal)) {
    XtSetArg(args[0], XtNshown, &shown);
    XtSetArg(args[1], XtNtopOfThumb, &top);
  } else {
    XtArgVal *l_top = (XtArgVal *) &top;
    XtArgVal *l_shown = (XtArgVal *) &shown;
    XtSetArg(args[0], XtNshown, *l_shown);
    XtSetArg(args[1], XtNtopOfThumb, *l_top);
  }
  XtSetValues(s->sl_sbar, args, 2);
#ifdef USE_K
  if ((value < 1000.0) && (value > -1000.0))
    statline_set(s->sl_output, "%d", (int)value);
  else
    statline_set(s->sl_output, "%fk", (float)value/1000.0);
#else
  statline_set(s->sl_output, "%d", (int)value);
#endif
    
  if (s->sl_svar != NULL)
    SI(s->sl_svar, value);
  s->sl_delta = True;
}

void slider_set(SliderObj s, float value)
{
  slider_set_noCB(s, value);

  if (s->_setCB != NULL)
    (s->_setCB)(s);
}

int slider_modified(SliderObj s)
{
  return(s->sl_delta);
}

Widget slider_widget(SliderObj s)
{
  return(s->sl_form);
}

void gauge_set(Widget g, float value)
{
  if (value > 1.0) value = 1.0;
  if (value < 0.0) value = 0.0;

  XawScrollbarSetThumb(g, (float) 0.0, value);
}

Widget gauge_new(Widget parent, float initval)
{
  Widget g;

  g = XtVaCreateManagedWidget("gauge", scrollbarWidgetClass, parent,
			      XtNlength,	200,
			      XtNwidth, 	6,
			      XtNorientation,	XtorientVertical,
			      NULL);
  XtAddCallback(g, XtNjumpProc, (XtCallbackProc) JumpProc, NULL);
  XtAddCallback(g, XtNscrollProc, (XtCallbackProc) JumpProc, NULL);
  gauge_set(g, initval);
  return(g);
}
