/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: sleep.c,v 2.45 1998/07/22 18:15:59 bjarthur Exp $
** Program: dowl
**  Module: sleep.c
**  Author: mazer
** Descrip: ms/us sleep functions and isi_sleep functions
**
** Revision History (most recent last)
** 
** Sep 96 bjarthur
**  moved idle.c in here
**
*******************************************************************/

#include "xdphyslib.h"

#define US_PER_SEC 1000000

static void (*idle_function)(void);

static void do_idle(void);

void ussleep(int us)
{
  static struct timeval tbefore, tafter;
  static struct timezone tz;

  gettimeofday(&tbefore, &tz);
  tbefore.tv_usec += us;
  if (tbefore.tv_usec >= US_PER_SEC) {
    tbefore.tv_sec += tbefore.tv_usec / US_PER_SEC;
    tbefore.tv_usec = tbefore.tv_usec % US_PER_SEC;
  }
  do {
    do_idle();
    gettimeofday(&tafter, &tz);
  } while (!((tafter.tv_sec > tbefore.tv_sec) ||
	     (tafter.tv_sec == tbefore.tv_sec &&
	      tafter.tv_usec > tbefore.tv_usec)));
}

void mssleep(int ms)
{
  ussleep(ms * 1000);
}

/*
** isi_ussleep and isi_mssleep functions act much like the above
** ussleep and mssleep functions except, that you can call them once
** with a sleep time (like before you present a stimulus) and then
** call them after presentation with 0 (zero), and then will include
** sleep until the specified time has elapsed from the first call.
**
** This lets you make the isi the true interstimulus interval, instead
** of the sleep time between stimuli. Or at least as close as possible,
** since stimulus generation could exceed the isi, making the sleeping
** unnecessary.
**
** If you call with (us < 0), then return is 1 if isi_done, 0 otherwise..
** do_idle() will be called 1x.. Note: if (us == 0) the return will be
** the number of us actually slept..
**
*/

int isi_ussleep(int us)
{
  static struct timeval tbefore, tafter;
  struct timezone tz;

  if (us > 0) {
    gettimeofday(&tbefore, &tz);
    tbefore.tv_usec += us;
    if (tbefore.tv_usec >= US_PER_SEC) {
      tbefore.tv_sec += tbefore.tv_usec / US_PER_SEC;
      tbefore.tv_usec = tbefore.tv_usec % US_PER_SEC;
    }
  } else if (us < 0) {
    gettimeofday(&tafter, &tz);
    do_idle();
    if (tafter.tv_sec > tbefore.tv_sec)
      return(1);
    if (tafter.tv_sec == tbefore.tv_sec && tafter.tv_usec > tbefore.tv_usec)
      return(1);
    return(0);
  } else while (1) {
    gettimeofday(&tafter, &tz);
    if (tafter.tv_sec > tbefore.tv_sec)
      break;
    if (tafter.tv_sec == tbefore.tv_sec && tafter.tv_usec > tbefore.tv_usec)
      break;
    do_idle();
  }
  return((long) ((tafter.tv_sec - tbefore.tv_sec) * US_PER_SEC)
	 + (tafter.tv_usec - tbefore.tv_usec));
}

int isi_mssleep(int ms)
{
  return(isi_ussleep(ms * 1000));
}


void *isi_start(int ms)
{
  struct timeval *tstart;
  struct timezone tz;

  tstart = (struct timeval *)malloc(sizeof(struct timeval));
  gettimeofday(tstart, &tz);
  tstart->tv_usec += (ms * 1000);
  if (tstart->tv_usec >= US_PER_SEC) {
    tstart->tv_sec += tstart->tv_usec / US_PER_SEC;
    tstart->tv_usec = tstart->tv_usec % US_PER_SEC;
  }
  return((void *)tstart);
}

int isi_done(struct timeval *tstart)
{
  struct timeval tnow;
  struct timezone tz;

  do_idle();
  gettimeofday(&tnow, &tz);
  if (tnow.tv_sec > tstart->tv_sec) {
    free(tstart);
    return(1);
  }
  if (tnow.tv_sec == tstart->tv_sec && tnow.tv_usec > tstart->tv_usec) {
    free(tstart);
    return(1);
  }
  return(0);
}

void isi_stop(struct timeval *tstart)
{
  struct timeval tnow;
  struct timezone tz;

  while (1) {
    do_idle();
    gettimeofday(&tnow, &tz);
    if (tnow.tv_sec > tstart->tv_sec)
      break;
    if (tnow.tv_sec == tstart->tv_sec && tnow.tv_usec > tstart->tv_usec)
      break;
  }
  free(tstart);
}


void *ustime_now(void)
{
  struct timeval *now;
  struct timezone tz;

  now = (struct timeval *)malloc(sizeof(struct timeval));
  gettimeofday(now, &tz);
  return((void *)now);
}

float ustime_elapsed(struct timeval *then)
{
  float diff;
  struct timeval *now;
  struct timezone tz;

  now = (struct timeval *)malloc(sizeof(struct timeval));
  gettimeofday(now, &tz);

  diff =
    ((now->tv_sec - then->tv_sec) * 1.0e6) + (now->tv_usec - then->tv_usec);
  free(then);
  free(now);
  return(diff);
}

void *idle_set(void (*fn)(void))
{
  void (*old_fn)();

  old_fn = idle_function;
  idle_function = fn;
  return((void *)old_fn);
}

static void do_idle(void)
{
  if (idle_function)
    (*idle_function)();
}
