/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: pm_rover.c,v 2.66 2000/12/08 05:21:09 cmalek Exp $
** Program: xdphys
**  Module: pm_rover.c
**  Author: mazer
** Descrip: xdphys plot method -- ROVER one & two snd-src sampling
**
** Revision History (most recent last)
**
** Sun Mar 14 20:37:11 1993 mazer
**  changed to accept the new FILEDATA structure
**
** 96.fall bjarthur
**  basically, completely rewrote
**
*******************************************************************/

#include "xdphyslib.h"
#include "xdphys.h"
#include "plotter.h"

static void rover_itd_curvesFN(FDO*, FILEDATA*,int,FILE*);
static void rover_iid_curvesFN(FDO*, FILEDATA*,int,FILE*);
static void rover_surfaceFN(FDO*, FILEDATA*,int,FILE*);
static void rover_rasterFN(FDO*, FILEDATA*, int, FILE*);
static void rover_text(FDO *, FILEDATA *, FILE *);
static int rover_plotter(FDO*, FILEDATA*, FDObj_ViewType*, int, FILE*);
static int rover_valid_view(FDO *, int);
static int parsestring(char*, float*, float*);
static int rover_summarize(FILEDATA*, int);
static void rover_freeSumData(FILEDATA*);
static int rover_free(FILEDATA*);

typedef struct {
  int           nitds,niids;
  float        *itds,*iids;
  float       **means,**stderrs,**stddevs;
} SumData;

#define SUMDATA(data) ((SumData*)(data->sumdata))

static int view_order[][1] = {
			{PM_ROVER_ITD},
			{PM_ROVER_IID},
			{PM_ROVER_SURFACE},
			{PM_RASTER},
			{PM_PSTH},
			{PM_ISIH}};
static int nsubviews[] = {1,1,1,1,1,1};
static int nviews = 6;

int rover_do_plot(FDO *fdo, FILEDATA *fd, int view, int l, FILE *fptr)
{
  if (!rover_valid_view(fdo, view)) {
		pm_type_error("rover", view);
		view = PM_DEFAULT;
		(void) rover_valid_view(fdo, view); }

  switch(view) {
    case PM_DEFAULT:
    case PM_ROVER_ITD:
      rover_itd_curvesFN(fdo,fd,l,fptr);
      break;
    case PM_ROVER_IID:
      rover_iid_curvesFN(fdo,fd,l,fptr);
      break;
    case PM_ROVER_SURFACE:
      rover_surfaceFN(fdo,fd,l,fptr);
      break;
    case PM_RASTER:
      rover_rasterFN(fdo,fd,l,fptr);
      break;
    case PM_PSTH:
      FD1_psthFN(fdo,fd,l,fptr);
      break;
    case PM_ISIH:
      FD1_isihFN(fdo,fd,l,fptr);
      break;
  	default:
  		pm_type_error("rover", view);
  		break; }

  return(1);
}

static int rover_valid_view(FDO *fdo, int view)
{
	int retval = 0;
	int i,j;

	if (view == PM_DEFAULT) {
			retval = 1;
	} else {
		for (i=0; i<nviews; i++) {
			for (j=0; j<nsubviews[i]; j++) {
				if (view_order[i][j] == view) {
					retval = 1;
					fdo->view.lr = i;
					fdo->view.ud = j;
					break; } } } }

	return(retval);
}

static int rover_plotter(FDO *fdo, FILEDATA *fd, FDObj_ViewType *view,
			int l, FILE *fptr)
{
  adjust_index(view->lr,view->ud);

  rover_do_plot(fdo, fd, view_order[view->lr][view->ud], l, fptr);
  
  return(1);
}

static void rover_itd_curvesFN(FDO *fdo, FILEDATA *fd,int l, FILE *fptr)
{
	int i, j;
	char *ti, buf[100];
	float f1;
	char *xlabel, *ylabel;
	float *px, *py, *pz;
	int pn;

	f1 = rover_summarize(fd, fdo->sum_code[l]);
	if (fdo->normalize == 1) {
		ylabel = "% max response";
		f1 = SUMDATA(fd)->means[0][0];
		for (i=0; i<SUMDATA(fd)->niids; i++) {
			for (j=0; j<SUMDATA(fd)->nitds; j++) {
				if (SUMDATA(fd)->means[i][j] > f1)
					f1 = SUMDATA(fd)->means[i][j]; } }
		f1 /= 100.0; }
	else {
		ylabel = "nspikes";
		f1 = 1.0; }
  
	if (fptr)
		rover_text(fdo,fd,fptr);

  if(fdo->no_X) return;

	for(i=0; i<SUMDATA(fd)->niids; i++) {

		FD_plotter_copy(SUMDATA(fd)->itds, SUMDATA(fd)->means[i], NULL, 
			SUMDATA(fd)->nitds, 0, &px, &py, &pz, &pn, NULL,NULL,NULL,NULL);

		FDObj_Add_Data_All(fdo,px,py,pn);
		sprintf(buf,"%2.0fdB iid",SUMDATA(fd)->iids[i]);
		FDObj_AddLine(fdo, l, px, py, pz, pn, AtFloat, 
			atQuadLinePlotWidgetClass, AtTypeLINEPOINTS, AtMarkCIRCLE, 
			ConvertColor(fdo->graph, FDObj_Colors[i%FDObj_numColors]), 
			(XtArgVal) buf); }

	ti = "itd_curves";
	XtVaSetValues(fdo->graph, XtNtitle, FDObj_Title(fdo, ti, l),
        XtNshowTitle, True, NULL);
	xlabel = "itd (us)";
	XtVaSetValues(fdo->xaxis, XtNlabel, xlabel, NULL);
	XtVaSetValues(fdo->yaxis, XtNlabel, ylabel, NULL);

}

static void rover_iid_curvesFN(FDO *fdo, FILEDATA *fd,int l, FILE *fptr)
{
	int i, j;
	char *ti, buf[100];
	float f1;
	char *xlabel, *ylabel;
	float *px, *py, *pz;
	int pn;
  float *tmp;

	f1 = rover_summarize(fd, fdo->sum_code[l]);
	if (fdo->normalize == 1) {
		ylabel = "% max response";
		f1 = SUMDATA(fd)->means[0][0];
		for (i=0; i<SUMDATA(fd)->niids; i++) {
			for (j=0; j<SUMDATA(fd)->nitds; j++) {
				if (SUMDATA(fd)->means[i][j] > f1)
					f1 = SUMDATA(fd)->means[i][j]; } }
		f1 /= 100.0; }
  else {
		ylabel = "nspikes";
		f1 = 1.0; }

  assert((tmp=(float*)malloc(SUMDATA(fd)->niids*sizeof(float)))!=NULL);

	if (fptr)
		rover_text(fdo,fd,fptr);

  if(fdo->no_X) return;

	for(i=0; i<SUMDATA(fd)->nitds; i++) {

	  for(j=0; j<SUMDATA(fd)->niids; j++)
      tmp[j] = SUMDATA(fd)->means[j][i];

		FD_plotter_copy(SUMDATA(fd)->iids, tmp,
			NULL, SUMDATA(fd)->niids, 0, &px, &py, &pz, &pn, NULL,
			NULL,NULL,NULL);
		FDObj_Add_Data_All(fdo,px,py,pn);
		sprintf(buf,"%4.0fus itd",SUMDATA(fd)->itds[i]);
		FDObj_AddLine(fdo, l, px, py, pz, pn, AtFloat, 
			atQuadLinePlotWidgetClass, AtTypeLINEPOINTS, AtMarkCIRCLE, 
			ConvertColor(fdo->graph, FDObj_Colors[i%FDObj_numColors]), 
			(XtArgVal) buf); }

  free(tmp);

	ti = "iid_curves";
	XtVaSetValues(fdo->graph, XtNtitle, FDObj_Title(fdo, ti, l),
        XtNshowTitle, True, NULL);
	xlabel = "iid (dB)";
	XtVaSetValues(fdo->xaxis, XtNlabel, xlabel, NULL);
	XtVaSetValues(fdo->yaxis, XtNlabel, ylabel, NULL);
}

static void rover_surfaceFN(FDO *fdo, FILEDATA *fd,int l, FILE *fptr)
{
  float high, low;
  int i,j;
  float min,max;
  float *xfdata,*yfdata,*xodata,*yodata;
  int nhigh, nlow;
  char *ti, buf[100];
  char *xlabel, *ylabel;


  rover_summarize(fd, fdo->sum_code[l]);

  high = (float)lookupParms_int("parms.rov_high");
  low = (float)lookupParms_int("parms.rov_low");

  if(low>high) {
    fprintf(stderr,"ERROR: parms.rov_low> parms.rov_high\n");
    fprintf(stderr,"   setting parms.rov_low = parms.rov_high\n");
    low = high; }
  
  min = max = SUMDATA(fd)->means[0][0];
  for (i=0; i<SUMDATA(fd)->niids; i++) {
    for (j=0; j<SUMDATA(fd)->nitds; j++) {
      if (SUMDATA(fd)->means[i][j] > max)
        max = SUMDATA(fd)->means[i][j];
      if (SUMDATA(fd)->means[i][j] < min)
        min = SUMDATA(fd)->means[i][j]; } }

  high = min + (max-min)*(high/100.0);
  low = min + (max-min)*(low/100.0);

  assert((xfdata = (float*)malloc(SUMDATA(fd)->nitds*SUMDATA(fd)->nitds*
        sizeof(float)))!=NULL);
  assert((yfdata = (float*)malloc(SUMDATA(fd)->nitds*SUMDATA(fd)->nitds*
        sizeof(float)))!=NULL);
  assert((xodata = (float*)malloc(SUMDATA(fd)->nitds*SUMDATA(fd)->nitds*
        sizeof(float)))!=NULL);
  assert((yodata = (float*)malloc(SUMDATA(fd)->nitds*SUMDATA(fd)->nitds*
        sizeof(float)))!=NULL);

  nhigh = nlow = 0;
  for(i=0; i<SUMDATA(fd)->niids; i++) {
    for(j=0; j<SUMDATA(fd)->nitds; j++) {
      if(SUMDATA(fd)->means[i][j]>high) {
        xfdata[nhigh] = SUMDATA(fd)->itds[j];
        yfdata[nhigh] = SUMDATA(fd)->iids[i];
        nhigh++; }
      else if(SUMDATA(fd)->means[i][j]>low) {
        xodata[nlow] = SUMDATA(fd)->itds[j];
        yodata[nlow] = SUMDATA(fd)->iids[i];
        nlow++; } } }

  if (fptr)
	  rover_text(fdo,fd,fptr);

  if(fdo->no_X) return;

  assert((xfdata = (float*)realloc(xfdata,nhigh*sizeof(float)))!=NULL);
  assert((yfdata = (float*)realloc(yfdata,nhigh*sizeof(float)))!=NULL);
  assert((xodata = (float*)realloc(xodata,nlow*sizeof(float)))!=NULL);
  assert((yodata = (float*)realloc(yodata,nlow*sizeof(float)))!=NULL);

  FDObj_Add_Data_All(fdo,xfdata,yfdata,nhigh);
  sprintf(buf,">%4.1f%%",high);
  FDObj_AddLine(fdo, l, xfdata, yfdata, NULL, nhigh, AtFloat,
				atQuadLinePlotWidgetClass, AtTypePOINTS, AtMarkCIRCLE,
				ConvertColor(fdo->graph,"red"), (XtArgVal) buf);

  FDObj_Add_Data_All(fdo,xodata,yodata,nlow);
  sprintf(buf,">%4.1f%%",low);
  FDObj_AddLine(fdo, l, xodata, yodata, NULL, nlow, AtFloat,
				atQuadLinePlotWidgetClass, AtTypePOINTS, AtMarkCIRCLE,
				ConvertColor(fdo->graph,"blue"), (XtArgVal) buf);

  ti = "surface";
  XtVaSetValues(fdo->graph, XtNtitle, FDObj_Title(fdo, ti, l),
        XtNshowTitle, True, NULL);
  xlabel = "itd (us)";
  ylabel = "iid (dB)";
  XtVaSetValues(fdo->xaxis, XtNlabel, xlabel, NULL);
  XtVaSetValues(fdo->yaxis, XtNlabel, ylabel, NULL);
}

/* rover_rasterFN(), rover_psthFN(), and rover_isihFN() are mostly just
   unmodified copies of FD_*FN()s, copied here simply to preserve
   modularity. (rover_rasterFN has had the raw parameter removed.) */

static void rover_rasterFN(FDO *fdo, FILEDATA *fd, int l, FILE *fptr)
{
  int i, j, k;
  char *ti;
  SA_SpikeRasterList *srl = NULL;
  char *xlabel, *ylabel;
  float *xdata, *ydata;
  int ndata;
  double xmin, xmax;

  srl = FD_to_SpikeRasterList(FD1_RAWDATA(fdo->fds[l])->nrasters,
        FD1_RAWDATA(fdo->fds[l])->rastlist, FD_GI(fdo->fds[l],"epoch"),
        fdo->sum_code[l]);
  ndata = SA_TotalSpikesInSRL(srl);
  assert((xdata = (float*)malloc(sizeof(float)*ndata))!=NULL);
  assert((ydata = (float*)malloc(sizeof(float)*ndata))!=NULL);

  for (k=i=0; i < srl->nrasters; i++) {
    for (j=0; j < srl->sr[i]->nspikes; j++) {
      xdata[k] = srl->sr[i]->tm[j];
      ydata[k++] = FD1_RAWDATA(fd)->pres_order[i]; } }

  SA_FreeSpikeRasterList(srl);

  if(fdo->no_X) return;

  FDObj_Add_Data_All(fdo,xdata,ydata,ndata);
  FDObj_AddLine(fdo, l, xdata, ydata, NULL, ndata, AtFloat,
        atQuadLinePlotWidgetClass, AtTypePOINTS, AtMarkVBAR,
				ConvertColor(fdo->graph, FDObj_Colors[l%FDObj_numColors]),
				FDObj_Legend(fdo,l));

  xmin = 0.0;   xmax = (double)FD_GI(fd,"epoch");
  XtVaSetValues(fdo->xaxis, XtNmin, &xmin, XtNmax, &xmax,
        XtNautoScale, False, NULL);
  xmin = 0.0;   xmax = (double)srl->nrasters;
  XtVaSetValues(fdo->yaxis, XtNmin, &xmin, XtNmax, &xmax,
       XtNautoScale, False, NULL);

  ylabel = "raster # (pres order)";
  ti = "rawraster";
  XtVaSetValues(fdo->graph, XtNtitle, FDObj_Title(fdo, ti, l),
        XtNshowTitle, True, NULL);
  xlabel = "time (ms)";
  XtVaSetValues(fdo->xaxis, XtNlabel, xlabel, NULL);
  XtVaSetValues(fdo->yaxis, XtNlabel, ylabel, NULL);
}

static int parsestring(char *p, float *itd, float *iid)
{
  if (sscanf(index(p, '<') + 1, "%f us, %f db", itd, iid) != 2)
    if (!strcmp(p,"depvar=-6666 <SPONT>")) {
      (*itd)=(float)SPONT;
      return(1); }
    else
      return(0);
  else
    return(1);
}

static int rover_summarize(FILEDATA *fd, int unitid)
{
  int i,j,r1,r2;
  float *p1,*p2;
  float iid,itd;
  int *sa_itd, *sa_iid;
  float ***data;
  int **num;
  int num_reps;
  int use_delay;                /* use the pre-stim delay as spontaneous */
  float t_begin;                  /* driven begins at in ms (t_ in ticks) */
  float t_end;                    /* driven ends at in ms (t_ in ticks) */
  float t_spont_begin;            /* spont begins at in ms (t_ in ticks) */
  float t_spont_end;              /* spont ends at in ms (t_ in ticks) */
  int spont_stims;

	rover_freeSumData(fd);
	assert((SUMDATA(fd) = (SumData*)malloc(sizeof(SumData)))!=NULL);

  /* compute nitds, niids, itds, & iids */
  if((sa_itd = StimArray_Gen(&(SUMDATA(fd)->nitds),0,
        FD_GV(fd,"rover.itd-range"), 1, 1, 0))==NULL) return(0);
  qsort(sa_itd,SUMDATA(fd)->nitds,sizeof(int),intcompare);
  assert((SUMDATA(fd)->itds=(float*)malloc(SUMDATA(fd)->nitds*
        sizeof(float)))!=NULL);
  for(i=0; i<SUMDATA(fd)->nitds; i++)
    SUMDATA(fd)->itds[i]=sa_itd[i];
  free(sa_itd);

  if((sa_iid = StimArray_Gen(&(SUMDATA(fd)->niids),0,
        FD_GV(fd,"rover.iid-range"), 1, 1, 0))==NULL) return(0);
  qsort(sa_iid,SUMDATA(fd)->niids,sizeof(int),intcompare);
  assert((SUMDATA(fd)->iids=(float*)malloc(SUMDATA(fd)->niids*
        sizeof(float)))!=NULL);

  for(i=0; i<SUMDATA(fd)->niids; i++)
    SUMDATA(fd)->iids[i]=sa_iid[i];
  free(sa_iid);

  /* alloc mem for means, stderrs, & stddevs */
  assert((SUMDATA(fd)->means=(float**)malloc(SUMDATA(fd)->niids*
        sizeof(float*)))!=NULL);
  for(i=0; i<SUMDATA(fd)->niids; i++)
    assert((SUMDATA(fd)->means[i]=(float*)malloc(SUMDATA(fd)->nitds*
          sizeof(float)))!=NULL);
  assert((SUMDATA(fd)->stderrs=(float**)malloc(SUMDATA(fd)->niids*
        sizeof(float*)))!=NULL);
  for(i=0; i<SUMDATA(fd)->niids; i++)
    assert((SUMDATA(fd)->stderrs[i]=(float*)malloc(SUMDATA(fd)->nitds*
          sizeof(float)))!=NULL);
  assert((SUMDATA(fd)->stddevs=(float**)malloc(SUMDATA(fd)->niids*
        sizeof(float*)))!=NULL);
  for(i=0; i<SUMDATA(fd)->niids; i++)
    assert((SUMDATA(fd)->stddevs[i]=(float*)malloc(SUMDATA(fd)->nitds*
          sizeof(float)))!=NULL);

  /* alloc temp storage for raw data */
  num_reps=FD_GI(fd,"Reps");
  assert((data=(float***)malloc(SUMDATA(fd)->niids*sizeof(float**)))!=NULL);
  for(i=0; i<SUMDATA(fd)->niids; i++) {
	  assert((data[i]=(float**)malloc(SUMDATA(fd)->nitds*sizeof(float*)))
		  !=NULL);
	  for(j=0; j<SUMDATA(fd)->nitds; j++) {
		  assert((data[i][j]=(float*)malloc(num_reps*sizeof(float)))!=NULL); 
	  } 
  }

  assert((num=(int**)malloc(SUMDATA(fd)->niids*sizeof(float*)))!=NULL);
  for(i=0; i<SUMDATA(fd)->niids; i++) {
	  assert((num[i]=(int*)calloc(SUMDATA(fd)->nitds,sizeof(float)))!=NULL); 
  }

  FD_calc_params(fd, &spont_stims, &use_delay,
      &t_begin, &t_end, &t_spont_begin, &t_spont_end);
 
  /* sort rasters into array, and compute stats */
  for(i=0; i<FD1_RAWDATA(fd)->nrasters; i++) {

	  if(FD1_RAWDATA(fd)->rastlist[i]==NULL) 
		  continue;
	  parsestring(FD1_RAWDATA(fd)->deplines[i],&itd,&iid);

	  if(itd==(float)SPONT) 
		  continue;

	  p1=(float*)bsearch(&iid,SUMDATA(fd)->iids,SUMDATA(fd)->niids,
		  sizeof(float), floatcompare);
	  r1=p1-SUMDATA(fd)->iids;

	  p2=(float*)bsearch(&itd,SUMDATA(fd)->itds,SUMDATA(fd)->nitds,
		  sizeof(float), floatcompare);
	  r2=p2-SUMDATA(fd)->itds;

	  data[r1][r2][num[r1][r2]++]=(float)countRaster(t_begin, t_end,
          FD1_RAWDATA(fd)->rastlist[i], unitid); 
  }

  for(i=0; i<SUMDATA(fd)->niids; i++) {
	  for(j=0; j<SUMDATA(fd)->nitds; j++) {
		  if(num[i][j]<num_reps)
/*
			  fprintf(stderr, 
				  "; note trial (%dus,%ddB) has %d of %d samples\n",
				  (int)SUMDATA(fd)->itds[j],(int)SUMDATA(fd)->iids[i],j,
				  num_reps);
*/

		  SUMDATA(fd)->means[i][j] = mean(data[i][j],num[i][j]);
		  SUMDATA(fd)->stderrs[i][j] = stderror(data[i][j],num[i][j]);
		  SUMDATA(fd)->stddevs[i][j] = stddev(data[i][j],num[i][j]); 
	  } 
  }

  /* free temp storage */
  for(i=0; i<SUMDATA(fd)->niids; i++) {
    for(j=0; j<SUMDATA(fd)->nitds; j++)
      free(data[i][j]);
    free(data[i]); }
  free(data);
  for(i=0; i<SUMDATA(fd)->niids; i++)
    free(num[i]);
  free(num);

  return(1);
}

static void rover_text(FDO *fdo, FILEDATA *fd, FILE *fptr)
{
	int i,j;

	fprintf(fptr, "COMMENTS\n");
	fprintf(fptr,"%%%s, rover view\n",
		strrchr(fd->filename,'/')==NULL ? fd->filename :
		1+(char*)strrchr(fd->filename,'/'));
	fprintf(fptr,"%%col 1: ITD\n");
	fprintf(fptr,"%%col 2: IID\n");
	fprintf(fptr,"%%col 3: nspikes\n");
	if(!(fdo->no_errs))
		fprintf(fptr,"%%col 4: std error\n");
	fprintf(fptr, "END_COMMENTS\n");
	fprintf(fptr, "PARAMS\n");
	fprintf(fptr,"Reps=%d\n",FD_GI(fd,"Reps"));
/*
	fprintf(fptr,"spont=%e\t%e\n",
		SUMDATA(fd)->means[0][0], SUMDATA(fd)->stderrs[0][0]);
*/
	fprintf(fptr, "END_PARAMS\n");
	fprintf(fptr, "RASTERDATA\n");
	fprintf(fptr, "nlines=%d\n",(SUMDATA(fd)->nitds)*(SUMDATA(fd)->niids));
	if(fdo->no_errs) {
		for(i=0; i<SUMDATA(fd)->niids; i++)
			for(j=0; j<SUMDATA(fd)->nitds; j++)
				fprintf(fptr,"%e\t%e\t%e\n",SUMDATA(fd)->itds[j],
					SUMDATA(fd)->iids[i], SUMDATA(fd)->means[i][j]); 
	}
	else {
		for(i=0; i<SUMDATA(fd)->niids; i++)
			for(j=0; j<SUMDATA(fd)->nitds; j++)
				fprintf(fptr,"%e\t%e\t%e\t%e\n",SUMDATA(fd)->itds[j],
					SUMDATA(fd)->iids[i], SUMDATA(fd)->means[i][j],
					SUMDATA(fd)->stderrs[i][j]); 
	}

	fprintf(fptr, "END_RASTERDATA\n"); 
}

static void rover_freeSumData(FILEDATA *data)
{
	int i;

  if(SUMDATA(data)!=NULL) {
    FREE(SUMDATA(data)->itds);
    FREE(SUMDATA(data)->iids);
    for(i=0; i<SUMDATA(data)->niids; i++) {
      FREE(SUMDATA(data)->means[i]);
      FREE(SUMDATA(data)->stderrs[i]);
      FREE(SUMDATA(data)->stddevs[i]); }
    FREE(SUMDATA(data)->means);
    FREE(SUMDATA(data)->stderrs);
    FREE(SUMDATA(data)->stddevs);

    FREE(SUMDATA(data)); }
}

static int rover_free(FILEDATA *data)
{
	FD1_freeRawData(data);
	rover_freeSumData(data);

  return(1);
}

int pm_rover_init(void)
{
  setFDOvalidviewMethod("rov", rover_valid_view);
  setFDOdoplotMethod("rov", rover_do_plot);
  setFDOplotMethod("rov", rover_plotter);
  setFDreadMethod("rov", FD1_reader);
  setFDfreeMethod("rov", rover_free);
  return(1);
}
