/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: pm_itd.c,v 2.65 2001/02/19 21:39:51 cmalek Exp $
** Program: xdphys
**  Module: pm_itd.c
**  Author: mazer
** Descrip: xdphys plot method -- static itd tuning curve plotter
**
** Revision History (most recent last)
**
** Sat Nov 20 18:45:56 1993 mazer
**  - creation date :: from freq_p.c 
**  - this is to allow for more analysis of static itd tuning
**    curves -- namely to be able to fold them and turn itd into
**    ipd curves for purse tone stim.
**
** 97.3 bjarthur
**   added FitDataToCos courtesy YA
**
*******************************************************************/

#include "xdphyslib.h"
#include "plotter.h"

int fitGaussenv = 0;
int rectify_fit = 1;

static void Heart(Widget, XtPointer, XtPointer);
static void Cosine(Widget, XtPointer, XtPointer);
static void CosineGaussian(Widget, XtPointer, XtPointer);
static void fit_menu1(FDO*);
static int itd_plotter(FDO *fdo, FILEDATA*, FDObj_ViewType*, int, FILE*);
static int FitDataToCos(float*, float*, float*, int, int, float**, float*,
      float*, float*, float*, float*, float*, int*);
static float FitCosPoint(float, float*);
static float EstimateFreq(float*, float*, int);
static float EstimateFall(float*, float*, int);
static int itd_valid_view(FDO *, int);

static int view_order[][3] = {
			{PM_CURVE},
			{PM_MEANPHASE, PM_VS, PM_RAYLEIGH},
			{PM_PERHIST},
			{PM_RASTER, PM_RASTER_RAW},
			{PM_PSTH},
			{PM_ISIH}};
static int nsubviews[] = {1,3,1,2,1,1};
static int nviews = 6;

int itd_do_plot(FDO *fdo, FILEDATA *fd, int view, int l, FILE *fptr)
{
  float period;
  int nbins;
  static int last_view=PM_NONE;

  if (!itd_valid_view(fdo, view)) {
		pm_type_error("itd", view);
		view = PM_DEFAULT;
		(void) itd_valid_view(fdo, view); }

  FD1_contra_ipsi(fd);
 
  FD_perhist_compute_period_nbins(fd, &period, &nbins);

  switch(view) {
    case PM_DEFAULT:
    case PM_CURVE:
      fit_menu1(fdo);
      FD1_curveFN(fdo,fd,l,fptr);
      break;
    case PM_MEANPHASE:
      FD1_perhist1FN(fdo,fd,l,fptr,period,nbins,0);
      break;
    case PM_VS:
      FD1_perhist1FN(fdo,fd,l,fptr,period,nbins,1);
      break;
    case PM_RAYLEIGH:
      FD1_perhist1FN(fdo,fd,l,fptr,period,nbins,2);
      break;
    case PM_PERHIST: {
      if(last_view!=PM_PERHIST)
      	fdo->vsm_data=VSM_ALL_CODE;
      FD1_perhist_stats_menu(fdo);
      FD_vsm_menu(fdo,1);
      FD1_perhist2FN(fdo,fd,l,fptr,period,nbins); }
      break;
    case PM_RASTER:
      FD1_rasterFN(fdo,fd,l,0,fptr);
      break;
    case PM_RASTER_RAW:
      FD1_rasterFN(fdo,fd,l,1,fptr);
      break;
    case PM_PSTH:
      FD1_psthFN(fdo,fd,l,fptr);
      break;
    case PM_ISIH:
      FD1_isih_stats_menu(fdo);
      FD1_isihFN(fdo,fd,l,fptr);
      break; 
		default:
			pm_type_error("itd", view);
			break; }

  last_view = view;
  return(1);
}

static int itd_valid_view(FDO *fdo, int view)
{
	int retval = 0;
	int i,j;

	if (view == PM_DEFAULT) {
			retval = 1;
	} else {
		for (i=0; i<nviews; i++) { for (j=0; j<nsubviews[i]; j++) {
				if (view_order[i][j] == view) {
					retval = 1;
					fdo->view.lr = i;
					fdo->view.ud = j;
					break; } } } }

	return(retval);
}

static int itd_plotter(FDO *fdo, FILEDATA *fd, FDObj_ViewType *view,
			int l, FILE *fptr)
{
  syn_spec ss;
  fd_syn_spec_parse(FD_GV(fdo->fds[l], "itd.Stim"),0,&ss);

  adjust_index(view->lr,view->ud);

  if(l==0) {
    if(!((ss.class == SC_TONE) || 
  			  ((ss.class == SC_STACK) && (ss.parms.stack.num_freqs == 1)))) {
      if(view->lr == 1) {
        view->lr = 3;
        view->ud = 0; }
      else if(view->lr == 2) {
        view->lr = 0;
        view->ud = 0; } } }
  else {
    if(!((ss.class == SC_TONE) ||
          ((ss.class == SC_STACK) && (ss.parms.stack.num_freqs == 1))))
      if((view->lr == 1) || (view->lr == 2)) {
        notify("some curves missing due to different stim types");
        return(0); } }

  if(!fdo->no_X) {
  if(nsubviews[view->lr]>1) {
    XtSetSensitive(fdo->up, True);
    XtSetSensitive(fdo->down, True); } }
 
  itd_do_plot(fdo, fd, view_order[view->lr][view->ud], l, fptr);

  return(1);
}


static void Heart(Widget w, XtPointer fdoptr, XtPointer call_data)
{
	FDO *fdo = (FDO *) fdoptr;

	int i, count, size;
	char *buf, buf2[128], *tmp;
	float *yfit, dc, amp, omega, phi, tau, chi;
	int big;
	float *px, *py, *pz;
	int pn;
	float ndata;
	float *depvals;
	float cd, period;
	syn_spec ss;
	char *foo;
	int SpontStims;
	float *means, *stderrs;



	assert((buf = (char *) malloc(1024 * sizeof(char))) != NULL);
	size = 1024;

	if (fitGaussenv) {
		tmp = "equation:  dc + amp * cos(omega * t + phi) * exp(-t / tau)\n";
		strcpy(buf, tmp);
		count = strlen(tmp);
		tmp = "units:  dc, amp (spikes);  omega (rad/sec);  phi (rad);  tau (sec)\n";
		strcat(buf, tmp);
		count += strlen(tmp);
		tmp = "units:  freq, stim (Hz);  CD, period (us)\n\n";
		strcat(buf, tmp);
		count += strlen(tmp);
		tmp = "file            dc     amp     omega    phi     tau     chi    |  freq   stim   CD      period\n";
		strcat(buf, tmp);
		count += strlen(tmp);
		tmp = "----            --     ---     -----    ---     ---     ---    |  ----   ----   --      ------\n";
		strcat(buf, tmp);
		count += strlen(tmp);
	} else {
		tmp = "equation:  dc + amp * cos(omega * t + phi)\n";
		strcpy(buf, tmp);
		count = strlen(tmp);
		tmp = "units:  dc, amp (spikes);  omega (rad/sec);  phi (rad)\n";
		strcat(buf, tmp);
		count += strlen(tmp);
		tmp = "units:  freq, stim (Hz);  CD, period (us)\n\n";
		strcat(buf, tmp);
		count += strlen(tmp);
		tmp = "file            dc     amp     omega    phi     chi    |  freq   stim   CD      period\n";
		strcat(buf, tmp);
		count += strlen(tmp);
		tmp = "----            --     ---     -----    ---     ---    |  ----   ----   --      ------\n";
		strcat(buf, tmp);
		count += strlen(tmp);
	}

	for (i = 0; i < fdo->nfds; i++) {
		ndata = FD1_get_ndata(fdo->fds[i]);
		depvals = FD1_get_depvals(fdo->fds[i]);
		SpontStims = FD_GI(fdo->fds[i], "Spont_Stims");
		foo = FD_GV(fdo->fds[i], "itd.Stim");
		fd_syn_spec_parse(foo, 0, &ss);
		means = FD1_get_means(fdo->fds[i]);
		stderrs = FD1_get_means(fdo->fds[i]);
		if (SpontStims) {
			FitDataToCos(depvals + 1, means + 1, stderrs + 1, ndata - 1,
				     (ss.class == SC_TONE) ? (int) (ss.parms.tone.freq) : 0,
			&yfit, &dc, &amp, &omega, &phi, &tau, &chi, &big);
		} else {
			FitDataToCos(depvals, means, stderrs, ndata,
				     (ss.class == SC_TONE) ? (int) (ss.parms.tone.freq) : 0,
			&yfit, &dc, &amp, &omega, &phi, &tau, &chi, &big);
		}
		cd = -phi / omega;
		period = 2.0 * M_PI / omega;
		while (cd > period)
			cd -= period;
		while (cd < -period)
			cd += period;

		if (fitGaussenv)
#ifdef __linux__
			count += sprintf(buf2, "%-13s   %5.1f  %5.1f   %5.3f    %5.1f   %5.3f   %5.1f%1s |  %5d  %5d  %6.1f  %6.1f\n",
			    strrchr(fdo->fds[i]->filename, '/') == NULL ?
					 fdo->fds[i]->filename :
			1 + (char *) strrchr(fdo->fds[i]->filename, '/'),
			  dc, amp, omega, phi, tau, chi, big ? "*" : " ",
				    (int) (omega * 1.0e6 / (2.0 * M_PI)),
					 (ss.class == SC_TONE) ? (int) (ss.parms.tone.freq) : 0, cd, period);
#else
			count += strlen(sprintf(buf2, "%-13s   %5.1f  %5.1f   %5.3f    %5.1f   %5.3f   %5.1f%1s |  %5d  %5d  %6.1f  %6.1f\n",
			    strrchr(fdo->fds[i]->filename, '/') == NULL ?
						fdo->fds[i]->filename :
			1 + (char *) strrchr(fdo->fds[i]->filename, '/'),
			  dc, amp, omega, phi, tau, chi, big ? "*" : " ",
				    (int) (omega * 1.0e6 / (2.0 * M_PI)),
						(ss.class == SC_TONE) ? (int) (ss.parms.tone.freq) : 0, cd, period));
#endif
		else
#ifdef __linux__
			count += sprintf(buf2, "%-13s   %5.1f  %5.1f   %5.3f    %5.1f   %5.1f%1s |  %5d  %5d  %6.1f  %6.1f\n",
			    strrchr(fdo->fds[i]->filename, '/') == NULL ?
					 fdo->fds[i]->filename :
			1 + (char *) strrchr(fdo->fds[i]->filename, '/'),
			       dc, amp, omega, phi, chi, big ? "*" : " ",
				    (int) (omega * 1.0e6 / (2.0 * M_PI)),
					 (ss.class == SC_TONE) ? (int) (ss.parms.tone.freq) : 0, cd, period);
#else
			count += strlen(sprintf(buf2, "%-13s   %5.1f  %5.1f   %5.3f    %5.1f   %5.1f%1s |  %5d  %5d  %6.1f  %6.1f\n",
			    strrchr(fdo->fds[i]->filename, '/') == NULL ?
						fdo->fds[i]->filename :
		      1 + (char *) strrchr(fdo->fds[i]->filename, '/'),
			       dc, amp, omega, phi, chi, big ? "*" : " ",
				    (int) (omega * 1.0e6 / (2.0 * M_PI)),
						(ss.class == SC_TONE) ? (int) (ss.parms.tone.freq) : 0, cd, period));
#endif
		strcat(buf, buf2);
		if ((size - count) < 512) {
			assert((buf = (char *) realloc(buf, (size + 1024) * sizeof(char))) != NULL);
			size += 1024;
		}
		if (SpontStims) {
			FD_plotter_copy(depvals + 1, yfit, NULL, ndata - 1, 0,
			     &px, &py, &pz, &pn, NULL, NULL, NULL, NULL);
		} else {
			FD_plotter_copy(depvals, yfit, NULL, ndata, 0,
			     &px, &py, &pz, &pn, NULL, NULL, NULL, NULL);
		}

		if (!fdo->no_X) {
			FDObj_Add_Data_All(fdo, px, py, pn);
			FDObj_AddLine(fdo, i, px, py, pz, pn, AtFloat,
			     atQuadLinePlotWidgetClass, AtTypeLINEPOINTS,
				   AtMarkCIRCLE, ConvertColor(fdo->graph,
							      FDObj_Colors[i % FDObj_numColors]), (XtArgVal) "fit");
		}
	}

	tmp = "\n(*) indicates chi is big, and fitted parameters are questionable";
	strcat(buf, tmp);
	count += strlen(tmp);

	if (!fdo->no_X) {
		if (fitGaussenv)
			pop_text("Cosine w/ Gaussian", buf, strlen(buf) + 1, False);
		else
			pop_text("Cosine", buf, strlen(buf) + 1, False);
	}
}

static void Cosine(Widget w, XtPointer fdoptr, XtPointer call_data)
{
	FDO *fdo = (FDO *) fdoptr;

	fitGaussenv = 0;
	Heart(w, fdo, call_data);
}

static void CosineGaussian(Widget w, XtPointer fdoptr, XtPointer call_data)
{
  FDO *fdo = (FDO *) fdoptr;
/*
i wouldn't trust this code with fitGaussenv = 1.  something's wierd
with the way it does the Gaussian envelope.  specifically, it prints that
it computes exp(-t / tau), but a cursory look at the code makes me think
it computes exp(-t^2 * tau).  i'd go over it with a fine
toothed comb before i used it.  bjarthur 7/97
*/
  fitGaussenv=1;
  Heart(w, fdo, call_data);
}

static void fit_menu1(FDO *fdobj)
{
  if(fdobj->no_X) return;

  menubutton_clear(fdobj->fit,&(fdobj->fitpsh));
  XtSetSensitive(fdobj->fit, False);

  menubutton_add(fdobj->fitpsh,"peak stats", FD1_peak_stats,fdobj);
  menubutton_add(fdobj->fitpsh,"similarity stats", FD1_similarity_stats,fdobj);
  menubutton_add(fdobj->fitpsh,"cosine",Cosine,fdobj);
  menubutton_add(fdobj->fitpsh,"cosine w/ gaussian",CosineGaussian,fdobj);
  menubutton_add(fdobj->fitpsh,"prob distribution", FD1_prob_distribution,fdobj);
  XtSetSensitive(fdobj->fit, True);
}

int pm_itd_init(void)
{
  setFDOvalidviewMethod("itd", itd_valid_view);
  setFDOdoplotMethod("itd", itd_do_plot);
  setFDOplotMethod("itd", itd_plotter);
  setFDreadMethod("itd", FD1_reader);
  setFDfreeMethod("itd", FD1_free);
  return(1);
}



/************** from yuda *********************/


#define CHI2_STOP_STEP (0.5)

void GeneralCosinus(float, float[], float*, float[], int);
void GaussianCosinus(float, float[], float*, float[], int);

static int FitDataToCos(
  float* fAxis,  /* x[] of curve to be fit */
  float* fData,  /* y[] of curve to be fit */
  float* fSig,   /* err[] of curve to be fit */
  int ndata,
  int nFreq,
  float **yfit,   /* y[] of resulting fit */
  float *dc, float *amp, float *omega,  /* parameters of resulting fit */
  float *phi, float *tau,
  float *chi, int *big)
{
	float fError, fFreq, fCD;
	float *x, *y, *sig;
	int i, *ia, ma;
	float *a, **covar, **alpha, alamda = -1, chisq = 10, oldchi = 1000;
	float x0, y0;
	void (*GeneralFit) (float, float[], float *, float[], int);
	GeneralFit = fitGaussenv ? GaussianCosinus : GeneralCosinus;
	ma = fitGaussenv ? 5 : 4;
	/*ndata--; */

	alpha = matrix(1, ma, 1, ma);
	covar = matrix(1, ma, 1, ma);
	ia = ivector(1, ma);
	a = vector(1, ma);
	x = vector(1, ndata);
	y = vector(1, ndata);
	sig = vector(1, ndata);
	fError = 0;
	for (i = 0; i < ndata; i++) {
		sig[i + 1] = fSig[i];
		x[i + 1] = fAxis[i];
		y[i + 1] = fData[i];
		fError += fSig[i] * fSig[i];
	}
	fError /= ndata;

	for (i = 1; i <= ma; i++)
		ia[i] = 1;
	for (x0 = x[1], y0 = y[1], i = 1; i <= ndata; i++)
		if (y[i] > y0) {
			x0 = x[i], y0 = y[i];
		}
	a[1] = y0 / 2;
	a[2] = a[1];
	if (nFreq != 0)
		a[3] = (2.0 * M_PI) * (fFreq = (float) nFreq) / 1000000;
	else
		a[3] = (2.0 * M_PI) * (fFreq = EstimateFreq(x, y, ndata)) / 1000000;
	if (fitGaussenv) {
		if (nFreq != 0) {
			a[5] = 0;
		} else {
			a[5] = EstimateFall(x, y, ndata);
		}
	}
	a[4] = -a[3] * x0;
	alamda = -1;

	while (((oldchi - chisq) > CHI2_STOP_STEP) || ((oldchi - chisq) < -CHI2_STOP_STEP)) {
		oldchi = chisq;
		fFreq = a[3] * 1000000 / (2.0 * M_PI);
		fCD = -a[4] / a[3];
		mrqmin(x, y, sig, ndata, a, ia, ma, covar, alpha, &chisq, GeneralFit, &alamda);
	}
	oldchi = 0;
	mrqmin(x, y, sig, ndata, a, ia, ma, covar, alpha, &chisq, GeneralFit, &oldchi);

	if (a[2] < 0) {
		a[2] = -a[2];
		a[4] += M_PI;
		if (a[4] < 0)
			while (a[4] < M_PI)
				a[4] += (2.0 * M_PI);
		if (a[4] > 0)
			while (a[4] > M_PI)
				a[4] -= (2.0 * M_PI);
	}
	assert(((*yfit) = (float *) malloc((ndata) * sizeof(float))) != NULL);
	for (i = 1; i <= ndata; i++)
		(*yfit)[i - 1] = FitCosPoint(x[i], a);
	(*dc) = a[1];
	(*amp) = a[2];
	(*omega) = a[3];
	(*phi) = a[4];
	if (fitGaussenv)
		(*tau) = a[5];
	(*chi) = chisq;
	(*big) = (chisq > ((ndata - ma) + (float) sqrt(2.0 * (ndata - ma)))) ? 1 : 0;

/* the mean of the chi2 distribution is the degrees of freedom, i.e.,
   ndata - ma (ma is the number of parameters).  the standard deviation
   of the chi2 distribution is sqrt(2 * mean). thus, i give a warning
   when chi2 exceeds the mean + std.  */


	free_matrix(covar, 1, ma, 1, ma);
	free_matrix(alpha, 1, ma, 1, ma);
	free_ivector(ia, 1, ma);
	free_vector(a, 1, ma);
	free_vector(x, 1, ndata);
	free_vector(y, 1, ndata);
	free_vector(sig, 1, ndata);
	fflush(stderr);
	return 1;
}

void GeneralCosinus(float x, float a[], float *y, float dyda[], int na)
{
	float phi = a[3] * x + a[4];
	float C = cos(phi);
	*y = a[1] + a[2] * C;
	if (*y < 0 && rectify_fit) {
		*y = 0;
		dyda[1] = dyda[2] = dyda[3] = dyda[4] = 0;
	} else {
		dyda[1] = 1;
		dyda[2] = C;
		dyda[4] = -a[2] * sin(phi);
		dyda[3] = x * dyda[4];
	}
}

void GaussianCosinus(float x, float a[], float *y, float dyda[], int na)
{
  float phi = a[3] * x + a[4];
  float C = cos(phi);
  float E = exp(-a[5] * phi * phi);
  *y = a[1] + a[2] * E * C;
  if(*y < 0 && rectify_fit)
  {
    *y = 0;
    dyda[1] = dyda[2] = dyda[3] = dyda[4] = dyda[5] = 0;
  }
  else
  {
    dyda[1] = 1;
    dyda[2] = E * C;
    dyda[4] = -a[2] * E * sin(phi);
    dyda[3] = x* dyda[4]; 
    dyda[5] = -phi * phi * a[2] * dyda[2];
  }
} 

static float FitCosPoint(float x, float* a)
{
  float fFit;
  float phi = a[3] * x + a[4];
  fFit = a[2] * cos(phi);
  if(fitGaussenv) fFit *= exp(-a[5] * phi * phi);
  fFit += a[1];
  if(fFit < 0 && rectify_fit) fFit = 0;
  return fFit;
}

void Gaussian(x,a,y,dyda,na)
  float x, a[], *y, dyda[];
  int na;
{
  float phi = x - a[3];
  float E = exp(-a[2] * phi * phi);
  *y = a[1] * E;
  {
    dyda[1] = E;
    dyda[2] = -phi * phi * a[1] * E;
    dyda[3] = 2 * phi * E;
  }
} 

static float EstimateFall(float *x, float *y, int ndata)
{
  return 1.0 / 500.0 / 500.0;
}

static float EstimateFreq(float *x, float *y, int ndata)
{
  int nFirstMin, nLastMin, nFirstMax, nLastMax;
  int nMaxs, nMins;
  int nMaxsReal, nMinsReal;
  int ni, i, cond, last;
  float fFreq, fAvgPeriod;
  float fAvg, yi, xi, xmax, ymax, xspan, xup, xdn;

  ymax = y[0], xmax = x[0];
  for(i = 2; i < ndata; i++)
  {
    xi = x[i], yi = y[i];
    if(yi > ymax) ymax = yi, xmax = xi;
  }

  fAvg = 0;
  xspan = 300;
  xup = xmax + xspan;
  xdn = xmax - xspan;
  for(ni = 0, i = 2; i < ndata; i++)
  {
    xi = x[i], yi = y[i];
    if(xi < xdn || xi > xup) continue;
    fAvg += yi;
    ni++;
  }
  if(ni > 1)
    fAvg /= (float)(ni - 1);
  else
    return 7000;

  nMaxs = 0; nMaxsReal = 0;
  for(last = 0, i = 2; i < ndata; i++)
  {
    xi = x[i];
    if(xi < xdn || xi > xup) continue;
    if(last == i-1) continue;
    yi = y[i];
    cond = yi >= fAvg;
    if((yi > y[i-1]) && (yi > y[i+1])) {last=i;if(cond) nMaxsReal++; nMaxs++;}
    else
      if(i > 2)
       if((yi >= y[i-1]) && (yi > y[i+1])) {last=i;if(cond) nMaxsReal++; nMaxs++;}
    else
      if(i < ndata-1)
       if((yi > y[i-1]) && (yi >= y[i+1])) {last=i;if(cond) nMaxsReal++; nMaxs++;}
  }

  nMins = 0; nMinsReal = 0;
  for(last = 0, i = 2; i < ndata; i++)
  {
    xi = x[i];
    if(xi < xdn || xi > xup) continue;
    if(last == i-1) continue;
    yi = y[i];
    cond = yi <= fAvg;
    if((yi < y[i-1]) && (yi < y[i+1])) {last=i;if(cond) nMinsReal++; nMins++;}
    else
      if(i > 2)
       if((yi <= y[i-1]) && (yi < y[i+1])) {last=i;if(cond) nMinsReal++; nMins++;}
    else
      if(i < ndata-1)
       if((yi < y[i-1]) && (yi <= y[i+1])) {last=i;if(cond) nMinsReal++; nMins++;}
  }

  for(i = 2; i < ndata; i++)
  {
    xi = x[i];
    if(xi < xdn || xi > xup) continue;
    yi = y[i];
    if((yi < y[i-1]) && (yi <= y[i+1])) if(yi < fAvg) break;
  }
  nFirstMin = i;
    
  for(i = ndata-1; i > 1; i--)
  {
    xi = x[i];
    if(xi < xdn || xi > xup) continue;
    yi = y[i];
    if((yi <= y[i-1]) && (yi < y[i+1])) if(yi < fAvg) break;
  }
  nLastMin = i;

  for(i = 2; i < ndata; i++)
  {
    xi = x[i];
    if(xi < xdn || xi > xup) continue;
    yi = y[i];
    if((yi > y[i-1]) && (yi >= y[i+1])) if(yi > fAvg) break;
  }
  nFirstMax = i;
    
  for(i = ndata-1; i > 1; i--)
  {
    xi = x[i];
    if(xi < xdn || xi > xup) continue;
    yi = y[i];
    if((yi >= y[i-1]) && (yi > y[i+1])) if(yi > fAvg) break;
  }
  nLastMax = i;

  nMaxsReal = nMaxs - nMaxsReal;
  nMinsReal = nMins - nMinsReal;

  nMaxs -= nMaxsReal + nMinsReal;
  nMins -= nMaxsReal + nMinsReal;
  fAvgPeriod = 0;
  if(nMaxs > 1)
    fAvgPeriod += (x[nLastMax] - x[nFirstMax]) / (nMaxs - 1);
  if(nMins > 1)
    fAvgPeriod += (x[nLastMin] - x[nFirstMin]) / (nMins - 1);
  if((nMins > 1) && (nMaxs > 1)) fAvgPeriod /= 2;
  if((nMins > 1) || (nMaxs > 1))
  {
    return fFreq = 1000000 / fAvgPeriod;
  }
  return 7000;
}
