/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: pm_fr.c,v 1.18 1999/05/18 19:17:13 bjarthur Exp $
** Program: xdphys
**  Module: pm_fr.c
**  Author: bjarthur
** Descrip: xdphys plot method -- static iid tuning curve plotter
**
** Revision History (most recent last)
**
** 10NOV96 bjarthur
**  copied from pm_iid.c
**
*******************************************************************/

#include "xdphyslib.h"
#include "xcalibur.h"
#include "plotter.h"

static void fr_separateFN(FDO*,FILEDATA*,int,int,FILE*);
static void fr_distortionFN(FDO*, FILEDATA*, int, FILE*);
static void fr_differenceFN(FDO*,FILEDATA*,int,int,FILE*);
static int fr_plotter(FDO*,FILEDATA*,FDObj_ViewType*,int,FILE*);
static int fr_valid_view(FDO *, int);
static int fr_reader(FILE*,FILEDATA*,int);
static int fr_free(FILEDATA*);

typedef struct {
  int           ndata;
  float        *freqs;
  float        *l_mags,*l_phis,*l_dists;
  float        *r_mags,*r_phis,*r_dists;
} SumData;

#define SUMDATA(data) ((SumData*)(data->sumdata))

static int view_order[][5] = {
			{PM_MAG},
			{PM_PHASE},
			{PM_DIFFERENCE_MAG},
			{PM_DIFFERENCE_PHASE},
			{PM_DISTORTION}};
static int nsubviews[] = {1,1,1,1,1};
static int nviews = 5;


int fr_do_plot(FDO *fdo, FILEDATA *fd, int view, int l, FILE *fptr)
{
  if (!fr_valid_view(fdo, view)) {
		pm_type_error("fr", view);
		view = PM_DEFAULT;
		(void) fr_valid_view(fdo, view); }

  switch(view) {
    case PM_DEFAULT:
    case PM_MAG:
      fr_separateFN(fdo,fd,l,1,fptr);
      break;
    case PM_PHASE:
      fr_separateFN(fdo,fd,l,0,fptr);
      break;
    case PM_DIFFERENCE_MAG:
      fr_differenceFN(fdo,fd,l,1,fptr);
      break;
    case PM_DIFFERENCE_PHASE:
      fr_differenceFN(fdo,fd,l,0,fptr);
      break;
    case PM_DISTORTION:
      fr_distortionFN(fdo,fd,l,fptr);
      break; 
  	default:
  		pm_type_error("fr", view);
  		break; }

  return(1);
}

static int fr_valid_view(FDO *fdo, int view)
{
	int retval = 0;
	int i,j;

	if (view == PM_DEFAULT) {
			retval = 1;
	} else {
		for (i=0; i<nviews; i++) {
			for (j=0; j<nsubviews[i]; j++) {
				if (view_order[i][j] == view) {
					retval = 1;
					fdo->view.lr = i;
					fdo->view.ud = j;
					break;
				}
			}
		}
	}

	return(retval);
}

static int fr_plotter(FDO *fdo, FILEDATA *fd, FDObj_ViewType *view,
			int l, FILE *fptr)
{
  adjust_index(view->lr,view->ud);

  fr_do_plot(fdo, fd, view_order[view->lr][view->ud], l, fptr);

  return(1);
}

static void fr_separateFN(FDO *fdo, FILEDATA *fd, int l, int mag, FILE *fptr)
{
  char *ylabel, *xlabel, *ti;
  float SoL, SoR;
  int i;
  float *xl, *yl, *xr, *yr;

  SoL = FD_GF(fd,"So_Left");
  SoR = FD_GF(fd,"So_Right");

  if(SoL!=0.0) {
		COPY_ARRAY(SUMDATA(fd)->freqs,xl,SUMDATA(fd)->ndata,float);
		COPY_ARRAY(mag ? SUMDATA(fd)->l_mags : SUMDATA(fd)->l_phis,
					yl,SUMDATA(fd)->ndata,float);

    if(!mag)
      unwrap(yl,SUMDATA(fd)->ndata,M_PI);

    for(i=0; i<SUMDATA(fd)->ndata; i++) {
      if(mag)
        yl[i]=20.0*log10(yl[i]/SoL)+94.0;
      else
        yl[i]=1.0e6*yl[i]/(2.0*M_PI*xl[i]); } }

  if(SoR!=0.0) {
		COPY_ARRAY(SUMDATA(fd)->freqs,xr,SUMDATA(fd)->ndata,float);
		COPY_ARRAY(mag ? SUMDATA(fd)->r_mags : SUMDATA(fd)->r_phis,
					yr,SUMDATA(fd)->ndata,float);

    if(!mag)
      unwrap(yr,SUMDATA(fd)->ndata,M_PI);

    for(i=0; i<SUMDATA(fd)->ndata; i++) {
      if(mag)
        yr[i]=20.0*log10(yr[i]/SoR)+94.0;
      else
        yr[i]=1.0e6*yr[i]/(2.0*M_PI*xr[i]); } }

  if(fptr!=NULL) {
    fprintf(fptr, "COMMENTS\n");
    fprintf(fptr,"%%text dump of %s, separate view\n",
          strrchr(fd->filename,'/')==NULL ? fd->filename :
          1+(char*)strrchr(fd->filename,'/'));
    fprintf(fptr,"%%col 1: frequency (Hz)\n");
    if(mag)
      fprintf(fptr,"%%col 2: intensity (dB SPL)\n");
    else
      fprintf(fptr,"%%col 2: phase (us)\n");
    fprintf(fptr, "%%left side data precedes right side data IF SoL and SoR aren't both 0.0\n");
    fprintf(fptr, "%%if both sides are not desired, then delete one and\n");
    fprintf(fptr, "%%change nlines= line appropriately\n");
    fprintf(fptr, "END_COMMENTS\n");
    fprintf(fptr, "PARAMS\n");
    fprintf(fptr, "END_PARAMS\n");
    fprintf(fptr, "RASTERDATA\n");
    fprintf(fptr, "nlines=%d\n", ((SoL!=0.0) && (SoR!=0.0)) ?
          2*SUMDATA(fd)->ndata : SUMDATA(fd)->ndata);
    if(SoL!=0.0) {
      for(i=0; i<SUMDATA(fd)->ndata; i++)  /* +=2 */
        fprintf(fptr,"%e\t%e\n",xl[i],yl[i]); }
    if(SoR!=0.0) {
      for(i=0; i<SUMDATA(fd)->ndata; i++)
        fprintf(fptr,"%e\t%e\n",xr[i],yr[i]); }
    fprintf(fptr, "END_RASTERDATA\n"); }

  if(fdo->no_X) return;

  if(SoL!=0.0) {
    FDObj_Add_Data_All(fdo,xl,yl,SUMDATA(fd)->ndata);
    FDObj_AddLine(fdo,l,xl,yl,NULL,SUMDATA(fd)->ndata,AtFloat,
					atQuadLinePlotWidgetClass, AtTypeLINEPOINTS,
					AtMarkCIRCLE, ConvertColor(fdo->graph, "darkgreen"), 
		(XtArgVal) "left"); }

  if(SoR!=0.0) {
    FDObj_Add_Data_All(fdo,xr,yr,SUMDATA(fd)->ndata);
    FDObj_AddLine(fdo,l,xr,yr,NULL,SUMDATA(fd)->ndata,AtFloat,
					atQuadLinePlotWidgetClass, AtTypeLINEPOINTS,
					AtMarkCIRCLE, ConvertColor(fdo->graph, "red"), 
		(XtArgVal) "right"); }

  if(mag) {
    ylabel="absolute intensity (dB)";
    ti = "separate intensity"; }
  else {
    ylabel="phase (us)";
    ti = "separate phase"; }
  xlabel="frequency (Hz)";
  XtVaSetValues(fdo->xaxis, XtNlabel, xlabel, NULL);
  XtVaSetValues(fdo->yaxis, XtNlabel, ylabel, NULL);
  XtVaSetValues(fdo->graph, XtNtitle, FDObj_Title(fdo, ti, l),
        XtNshowTitle, True, NULL);
}

static void fr_distortionFN(FDO *fdo, FILEDATA *fd, int l, FILE *fptr)
{
  char *ylabel, *xlabel, *ti;
  int i;
  float *xl, *yl, *xr, *yr;

	COPY_ARRAY(SUMDATA(fd)->freqs,xl,SUMDATA(fd)->ndata,float);
	COPY_ARRAY(SUMDATA(fd)->l_dists,yl,SUMDATA(fd)->ndata,float);
	COPY_ARRAY(SUMDATA(fd)->freqs,xr,SUMDATA(fd)->ndata,float);
	COPY_ARRAY(SUMDATA(fd)->r_dists,yr,SUMDATA(fd)->ndata,float);

  if(fptr!=NULL) {
    fprintf(fptr, "COMMENTS\n");
    fprintf(fptr,"%%text dump of %s, separate view\n",
          strrchr(fd->filename,'/')==NULL ? fd->filename :
          1+(char*)strrchr(fd->filename,'/'));
    fprintf(fptr,"%%col 1: frequency (Hz)\n");
    fprintf(fptr,"%%col 2: distortion (%%)\n");
    fprintf(fptr,"%%left side data precedes right side data\n");
    fprintf(fptr,"%%if both sides are not desired, then delete one and\n");
    fprintf(fptr,"%%change nlines= line appropriately\n");
    fprintf(fptr,"END_COMMENTS\n");
    fprintf(fptr,"PARAMS\n");
    fprintf(fptr,"END_PARAMS\n");
    fprintf(fptr,"RASTERDATA\n");
    fprintf(fptr,"nlines=%d\n", 2*SUMDATA(fd)->ndata);
    for(i=0; i<SUMDATA(fd)->ndata; i++)
      fprintf(fptr,"%e\t%e\n",xl[i],yl[i]);
    for(i=0; i<SUMDATA(fd)->ndata; i++)
      fprintf(fptr,"%e\t%e\n",xr[i],yr[i]);
    fprintf(fptr, "END_RASTERDATA\n"); }

  if(fdo->no_X) return;

  FDObj_Add_Data_All(fdo,xl,yl,SUMDATA(fd)->ndata);
  FDObj_AddLine(fdo,l,xl,yl,NULL,SUMDATA(fd)->ndata, AtFloat,
				atQuadLinePlotWidgetClass, AtTypeLINEPOINTS,
				AtMarkCIRCLE, ConvertColor(fdo->graph, "darkgreen"), 
	  (XtArgVal) "left");
 
  FDObj_Add_Data_All(fdo,xr,yr,SUMDATA(fd)->ndata);
  FDObj_AddLine(fdo,l,xr,yr,NULL,SUMDATA(fd)->ndata, AtFloat,
				atQuadLinePlotWidgetClass, AtTypeLINEPOINTS,
				AtMarkCIRCLE, ConvertColor(fdo->graph, "red"), 
	  (XtArgVal) "right");

  ylabel="distortion (%)";
  xlabel="frequency (Hz)";
  ti = "distortion";
  XtVaSetValues(fdo->xaxis, XtNlabel, xlabel, NULL);
  XtVaSetValues(fdo->yaxis, XtNlabel, ylabel, NULL);
  XtVaSetValues(fdo->graph, XtNtitle, FDObj_Title(fdo, ti, l),
        XtNshowTitle, True, NULL);
}

static void fr_differenceFN(FDO *fdo, FILEDATA *fd, int l, int mag, FILE *fptr)
{
  char *ylabel, *xlabel, *ti, *legend;
  int i;
  float *xl, *yl;
  float So,SoL,SoR;

  SoL=FD_GF(fd,"So_Left");
  SoR=FD_GF(fd,"So_Right");

  if(*FD_GV(fd,"Reference_Channel")=='L')
    So=SoL / ((SoR == 0.0) ? 1.0 : SoR);
  if(*FD_GV(fd,"Reference_Channel")=='R')
    So=SoR / ((SoL == 0.0) ? 1.0 : SoL);

  assert((xl = (float*)malloc(sizeof(float)*SUMDATA(fd)->ndata))!=NULL);
  assert((yl = (float*)malloc(sizeof(float)*SUMDATA(fd)->ndata))!=NULL);

  for(i=0; i<SUMDATA(fd)->ndata; i++) {
    xl[i]=SUMDATA(fd)->freqs[i];
    if(*FD_GV(fd,"Reference_Channel")=='L') {
      if(mag) {
        yl[i]=20.0*log10(So*(SUMDATA(fd)->r_mags[i]/SUMDATA(fd)->l_mags[i])); }
      else {
        yl[i]=(SUMDATA(fd)->r_phis[i]-SUMDATA(fd)->l_phis[i]); } }
    else if(*FD_GV(fd,"Reference_Channel")=='R') {
      if(mag) {
        yl[i]=20.0*log10(So*(SUMDATA(fd)->l_mags[i]/SUMDATA(fd)->r_mags[i])); }
      else {
        yl[i]=(SUMDATA(fd)->l_phis[i]-SUMDATA(fd)->r_phis[i]); } } }

  if(!mag) {
    unwrap(yl,SUMDATA(fd)->ndata,M_PI);
  	for(i=0; i<SUMDATA(fd)->ndata; i++)
      yl[i] *= 1.0e6/(2.0*M_PI*SUMDATA(fd)->freqs[i]); }

  if(fptr!=NULL) {
    fprintf(fptr, "COMMENTS\n");
    fprintf(fptr,"%%text dump of %s, difference view\n",
          strrchr(fd->filename,'/')==NULL ? fd->filename :
          1+(char*)strrchr(fd->filename,'/'));
    fprintf(fptr,"%%col 1: frequency (Hz)\n");
    if(mag)
      fprintf(fptr,"%%col 2: intensity (dB SPL)\n");
    else
      fprintf(fptr,"%%col 2: phase (us)\n");
    fprintf(fptr, "END_COMMENTS\n");
    fprintf(fptr, "PARAMS\n");
    fprintf(fptr, "END_PARAMS\n");
    fprintf(fptr, "RASTERDATA\n");
    fprintf(fptr, "nlines=%d\n", SUMDATA(fd)->ndata);
    for(i=0; i<SUMDATA(fd)->ndata; i++)
      fprintf(fptr,"%e\t%e\n",xl[i],yl[i]);
    fprintf(fptr, "END_RASTERDATA\n"); }

  if(fdo->no_X) return;

  FDObj_Add_Data_All(fdo,xl,yl,SUMDATA(fd)->ndata);
  if(*FD_GV(fd,"Reference_Channel")=='L')
    legend = mag ? "R/L" : "R-L";
  else
    legend = mag ? "L/R" : "L-R";
  FDObj_AddLine(fdo,l,xl,yl,NULL,SUMDATA(fd)->ndata, AtFloat,
				atQuadLinePlotWidgetClass, AtTypeLINEPOINTS,
				AtMarkCIRCLE, ConvertColor(fdo->graph, "blue"), 
	  (XtArgVal) legend);

  if(mag) {
    ylabel="intensity difference (dB)";
    ti = "difference intensity"; }
  else {
    ylabel="phase difference (us)";
    ti = "difference phase"; }
  xlabel="frequency (Hz)";
  XtVaSetValues(fdo->xaxis, XtNlabel, xlabel, NULL);
  XtVaSetValues(fdo->yaxis, XtNlabel, ylabel, NULL);
  XtVaSetValues(fdo->graph, XtNtitle, FDObj_Title(fdo, ti, l),
        XtNshowTitle, True, NULL);
}


void CombineFrFiles(void)
{
  char *file1, *file2, *fileout;
  FILEDATA *fd1, *fd2;
  FILE *fpout;
  char one_ref, two_ref;
  int i;

  if((file1=fileBox(".fr","r","First file..."))==NULL)
    return;
  if((fd1=FD_read(file1,0))==NULL) {
    alert("Can't read that file.");
    return; }
  if(strcmp(fd1->modname,"fr")) {
    alert("That file was not made by xcalibur.fr.");
    return; }
  if(FD_GI(fd1,"fr.ambient")!=0) {
    alert("That file is an ambient file.");
    return; }

  if((file2=fileBox(".fr","r","Second file..."))==NULL)
    return;
  if((fd2=FD_read(file2,0))==NULL) {
    alert("Can't read that file.");
    return; }
  if(strcmp(fd2->modname,"fr")) {
    alert("That file was not made by xcalibur.fr.");
    return; }
  if(FD_GI(fd2,"fr.ambient")!=0) {
    alert("That file is an ambient file.");
    return; }

  if(!StimArray_Equivalent(FD_GV(fd1,"fr.range"),FD_GV(fd2,"fr.range"))) {
    alert("The two files have different ranges.  Can't combine.");
    return; }

  if((fileout=fileBox(".fr","w","Output file..."))==NULL)
    return;
  if((fpout=fopen(fileout,"w"))==NULL) {
    alert("Can't write that file.");
    return; }
  
  fprintf(fpout, ";; xcalibur %s ver %s\n", fd1->modname, XDPHYS_VERSION);

  fprintf(fpout, "COMMENTS\n");
  writeComments(getComments(fileout),fpout);
  fprintf(fpout, ";com\n");
  fprintf(fpout, ";com comments from first file (%s):\n",file1);
  writeComments(fd1->comments, fpout);
  fprintf(fpout, ";com end comments from first file\n");
  fprintf(fpout, ";com\n");
  fprintf(fpout, ";com comments from second file (%s):\n",file2);
  writeComments(fd2->comments, fpout);
  fprintf(fpout, ";com end comments from second file\n");
  fprintf(fpout, ";com\n");
  fprintf(fpout, ";com note that the params below are largely those\n");
  fprintf(fpout, ";com from the first file, modified to make them work\n");
  fprintf(fpout, ";com for the combined file.  certain parameters\n");
  fprintf(fpout, ";com might be different from the second file, but it\n");
  fprintf(fpout, ";com should be inconsequential.\n");
  fprintf(fpout, "END_COMMENTS\n");

  one_ref = *FD_GV(fd1,"Reference_Channel");
  two_ref = *FD_GV(fd2,"Reference_Channel");
  
  fprintf(fpout, "PARAMS\n");
  FD_SF(fd1,"So_Left", (two_ref=='L') ?
        FD_GF(fd2,"So_Left") : FD_GF(fd2,"So_Right"));
  FD_SF(fd1,"So_Right",0.0);
  FD_SV(fd1,"Mic_Adjust_Right", (one_ref=='L') ?
				FD_GV(fd1,"Mic_Adjust_Right") : FD_GV(fd1,"Mic_Adjust_Left"));
  FD_SV(fd1,"Mic_Adjust_Left", (two_ref=='L') ?
				FD_GV(fd2,"Mic_Adjust_Left") : FD_GV(fd2,"Mic_Adjust_Right"));
  FD_SV(fd1,"Speaker_Channel","");
  FD_SV(fd1,"Reference_Channel","L");
  fp_savevars(fpout,fd1->params);
  fprintf(fpout, "END_PARAMS\n");

  fprintf(fpout, "RASTERDATA\n");
  fprintf(fpout, "nrasters=%d\n",SUMDATA(fd1)->ndata);
  for(i=0; i<SUMDATA(fd1)->ndata; i++) {
    fprintf(fpout,"%5d\t%e\t%e\t0.0\t%e\t%e\t0.0\t0.0\t0.0\t0.0\t0.0\n",
          (int)(SUMDATA(fd1)->freqs[i]),
          (one_ref=='L') ?
                (SUMDATA(fd1)->l_mags[i]/SUMDATA(fd1)->r_mags[i]) :
                (SUMDATA(fd1)->r_mags[i]/SUMDATA(fd1)->l_mags[i]),
          (one_ref=='L') ?
                (SUMDATA(fd1)->l_phis[i]-SUMDATA(fd1)->r_phis[i]) :
                (SUMDATA(fd1)->r_phis[i]-SUMDATA(fd1)->l_phis[i]),
          (two_ref=='L') ?
                (SUMDATA(fd2)->r_mags[i]/SUMDATA(fd2)->l_mags[i]) :
                (SUMDATA(fd2)->l_mags[i]/SUMDATA(fd2)->r_mags[i]),
          (two_ref=='L') ?
                (SUMDATA(fd2)->r_phis[i]-SUMDATA(fd2)->l_phis[i]) :
                (SUMDATA(fd2)->l_phis[i]-SUMDATA(fd2)->r_phis[i])); }
  fprintf(fpout, "END_RASTERDATA\n");
  fclose(fpout);
}

static int fr_reader(FILE *fptr, FILEDATA *fd, int ana)
{
  int i;
  char *p,buf[1024];

  fd->rawdata=NULL;
  fd->channels=NULL;
  assert((fd->sumdata=malloc(sizeof(SumData)))!=NULL);

  if ((p = skipTo(fptr, "nrasters=")) == NULL ||
      sscanf(p, "nrasters=%d", &(SUMDATA(fd)->ndata)) != 1) {
    if (debugflag)
      fprintf(stderr, "FD_read: %s missing nrasters= line\n", fd->filename);
    fclose(fptr);
    SUMDATA(fd)->freqs=NULL;
    SUMDATA(fd)->l_mags=SUMDATA(fd)->l_phis=SUMDATA(fd)->l_dists=NULL;
    SUMDATA(fd)->r_mags=SUMDATA(fd)->r_phis=SUMDATA(fd)->r_dists=NULL;
    FD_free(fd);
    return(0); }

  assert((SUMDATA(fd)->freqs=(float*)malloc(SUMDATA(fd)->ndata*
        sizeof(float)))!=NULL);
  assert((SUMDATA(fd)->l_mags=(float*)malloc(SUMDATA(fd)->ndata*
        sizeof(float)))!=NULL);
  assert((SUMDATA(fd)->l_phis=(float*)malloc(SUMDATA(fd)->ndata*
        sizeof(float)))!=NULL);
  assert((SUMDATA(fd)->l_dists=(float*)malloc(SUMDATA(fd)->ndata*
        sizeof(float)))!=NULL);
  assert((SUMDATA(fd)->r_mags=(float*)malloc(SUMDATA(fd)->ndata*
        sizeof(float)))!=NULL);
  assert((SUMDATA(fd)->r_phis=(float*)malloc(SUMDATA(fd)->ndata*
        sizeof(float)))!=NULL);
  assert((SUMDATA(fd)->r_dists=(float*)malloc(SUMDATA(fd)->ndata*
        sizeof(float)))!=NULL);

  for(i=0; i<SUMDATA(fd)->ndata; i++) {
    fgets(buf,sizeof(buf),fptr);
    sscanf(buf,"%f %f %f %f %f %f %f", &(SUMDATA(fd)->freqs[i]),
          &(SUMDATA(fd)->l_mags[i]), &(SUMDATA(fd)->l_phis[i]),
          &(SUMDATA(fd)->l_dists[i]),
          &(SUMDATA(fd)->r_mags[i]), &(SUMDATA(fd)->r_phis[i]),
          &(SUMDATA(fd)->r_dists[i])); }

  /* sort */
  {
  float min_val;  int i,j,min_loc;

  for(i=0; i<(SUMDATA(fd)->ndata-1); i++) {
    min_val=SUMDATA(fd)->freqs[i];  min_loc=i;
    for(j=i+1; j<SUMDATA(fd)->ndata; j++) {
      if(SUMDATA(fd)->freqs[j]<min_val) {
        min_val=SUMDATA(fd)->freqs[j]; min_loc=j; } }
    if(min_loc!=i) {
      SWAP(SUMDATA(fd)->freqs[i],SUMDATA(fd)->freqs[min_loc],float);
      SWAP(SUMDATA(fd)->l_mags[i],SUMDATA(fd)->l_mags[min_loc],float);
      SWAP(SUMDATA(fd)->l_phis[i],SUMDATA(fd)->l_phis[min_loc],float);
      SWAP(SUMDATA(fd)->l_dists[i],SUMDATA(fd)->l_dists[min_loc],float);
      SWAP(SUMDATA(fd)->r_mags[i],SUMDATA(fd)->r_mags[min_loc],float);
      SWAP(SUMDATA(fd)->r_phis[i],SUMDATA(fd)->r_phis[min_loc],float);
      SWAP(SUMDATA(fd)->r_dists[i],SUMDATA(fd)->r_dists[min_loc],float); } }
  }

  return(1);
}

static int fr_free(FILEDATA *fd)
{
  FREE(SUMDATA(fd)->freqs);

  FREE(SUMDATA(fd)->l_mags);
  FREE(SUMDATA(fd)->l_phis);
  FREE(SUMDATA(fd)->l_dists);

  FREE(SUMDATA(fd)->r_mags);
  FREE(SUMDATA(fd)->r_phis);
  FREE(SUMDATA(fd)->r_dists);

  FREE(SUMDATA(fd));

  return(1);
}

int fr_get_ndata(FILEDATA *fd)
{
  return(SUMDATA(fd)->ndata);
}

float *fr_get_freqs(FILEDATA *fd)
{
  return(SUMDATA(fd)->freqs);
}

float *fr_get_lmags(FILEDATA *fd)
{
  return(SUMDATA(fd)->l_mags);
}

float *fr_get_rmags(FILEDATA *fd)
{
  return(SUMDATA(fd)->r_mags);
}

float *fr_get_lphis(FILEDATA *fd)
{
  return(SUMDATA(fd)->l_phis);
}

float *fr_get_rphis(FILEDATA *fd)
{
  return(SUMDATA(fd)->r_phis);
}

int pm_fr_init(void)
{
  setFDOvalidviewMethod("fr", fr_valid_view);
  setFDOdoplotMethod("fr", fr_do_plot);
  setFDOplotMethod("fr", fr_plotter);
  setFDreadMethod("fr", fr_reader);
  setFDfreeMethod("fr", fr_free);
  return(1);
}
