/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: pm_cal.c,v 1.20 1999/05/18 19:17:13 bjarthur Exp $
** Program: xdphys
**  Module: pm_cal.c
**  Author: bjarthur
** Descrip: xdphys plot method -- calibration data plotting routines
**
** Revision History (most recent last)
**
** 10NOV96 bjarthur
**  copied from pm_fr.c
**
** 97.1 bjarthur
**   fixed bug in reader()
**
*******************************************************************/

#include "xdphyslib.h"
#include "xcalibur.h"
#include "plotter.h"

static void cal_separateFN(FDO*,FILEDATA*,int,int,FILE*);
static void cal_differenceFN(FDO*,FILEDATA*,int,int,FILE*);
static void cal_separate_leakFN(FDO*,FILEDATA*,int,int,FILE*);
static void cal_distortionFN(FDO*, FILEDATA*, int, FILE*);
static void cal_distortion_leakFN(FDO*, FILEDATA*, int, FILE*);
static int cal_plotter(FDO*,FILEDATA*,FDObj_ViewType*,int,FILE*);
static int cal_valid_view(FDO *, int);
static int cal_reader(FILE*,FILEDATA*,int);
static int cal_free(FILEDATA*);

typedef struct {
  int           ndata;
  float        *freqs;

  float        *l_mags,*l_phis,*l_dists;
  float        *l_lkmags,*l_lkphis,*l_lkdists;

  float        *r_mags,*r_phis,*r_dists;
  float        *r_lkmags,*r_lkphis,*r_lkdists;
} SumData;

#define SUMDATA(data) ((SumData*)(data->sumdata))

static int view_order[][1] = {
			{PM_MAG},
			{PM_PHASE},
			{PM_DIFFERENCE_MAG}, 
			{PM_DIFFERENCE_PHASE},
			{PM_LEAK_MAG},
			{PM_LEAK_PHASE}, 
			{PM_DISTORTION},
			{PM_LEAK_DISTORTION}};
static int nsubviews[] = {1,1,1,1,1,1,1,1};
static int nviews = 8;

int cal_do_plot(FDO *fdo, FILEDATA *fd, int view, int l, FILE *fptr)
{
  if (!cal_valid_view(fdo, view)) {
		pm_type_error("cal", view);
		view = PM_DEFAULT;
		(void) cal_valid_view(fdo, view); }

  switch(view) {
    case PM_DEFAULT: 
    case PM_MAG: 
      cal_separateFN(fdo,fd,l,1,fptr);
      break;
    case PM_PHASE:
      cal_separateFN(fdo,fd,l,0,fptr);
      break;
    case PM_DIFFERENCE_MAG: 
      cal_differenceFN(fdo,fd,l,1,fptr);
      break;
    case PM_DIFFERENCE_PHASE:
      cal_differenceFN(fdo,fd,l,0,fptr);
      break;
    case PM_LEAK_MAG: 
      cal_separate_leakFN(fdo,fd,l,1,fptr);
      break;
    case PM_LEAK_PHASE:
      cal_separate_leakFN(fdo,fd,l,0,fptr);
      break;
    case PM_DISTORTION: 
      cal_distortionFN(fdo,fd,l,fptr);
      break;
    case PM_LEAK_DISTORTION:  
      cal_distortion_leakFN(fdo,fd,l,fptr);
      break; 
  	default:
  		pm_type_error("cal", view);
  		break; }

  return(1);
}

static int cal_valid_view(FDO *fdo, int view)
{
	int retval = 0;
	int i,j;

	if (view == PM_DEFAULT) {
			retval = 1;
	} else {
		for (i=0; i<nviews; i++) {
			for (j=0; j<nsubviews[i]; j++) {
				if (view_order[i][j] == view) {
					retval = 1;
					fdo->view.lr = i;
					fdo->view.ud = j;
					break;
				}
			}
		}
	}

	return(retval);
}


static int cal_plotter(FDO *fdo, FILEDATA *fd, FDObj_ViewType *view,
			int l, FILE *fptr)
{
  adjust_index(view->lr,view->ud);

  cal_do_plot(fdo, fd, view_order[view->lr][view->ud], l, fptr);

  return(1);
}

static void cal_separateFN(FDO *fdo, FILEDATA *fd, int l, int mag, FILE *fptr)
{
  char *ylabel, *xlabel, *ti, *mono;
  int i;
  float *xl, *yl, *xr, *yr;

  mono = FD_GV(fd,"cal.mono");

  if((*mono)!='R') {
    COPY_ARRAY(SUMDATA(fd)->freqs,xl,SUMDATA(fd)->ndata,float);
    COPY_ARRAY(mag ? SUMDATA(fd)->l_mags : SUMDATA(fd)->l_phis,
          yl,SUMDATA(fd)->ndata,float);
 
    if(!mag)
      unwrap(yl,SUMDATA(fd)->ndata,M_PI);

    for(i=0; i<SUMDATA(fd)->ndata; i++)
      if(!mag)
        yl[i]=1.0e6*yl[i]/(2.0*M_PI*xl[i]); }

  if((*mono)!='L') {
    COPY_ARRAY(SUMDATA(fd)->freqs,xr,SUMDATA(fd)->ndata,float);
    COPY_ARRAY(mag ? SUMDATA(fd)->r_mags : SUMDATA(fd)->r_phis,
          yr,SUMDATA(fd)->ndata,float);

    if(!mag)
      unwrap(yr,SUMDATA(fd)->ndata,M_PI);

    for(i=0; i<SUMDATA(fd)->ndata; i++)
      if(!mag)
        yr[i]=1.0e6*yr[i]/(2.0*M_PI*xr[i]); }

  if(fptr!=NULL) {
    fprintf(fptr, "COMMENTS\n");
    fprintf(fptr,"%%text dump of %s, separate view\n",
          strrchr(fd->filename,'/')==NULL ? fd->filename :
          1+(char*)strrchr(fd->filename,'/'));
    fprintf(fptr,"%%col 1: frequency (Hz)\n");
    if(mag)
      fprintf(fptr,"%%col 2: intensity (dB SPL)\n");
    else
      fprintf(fptr,"%%col 2: phase (us)\n");
    fprintf(fptr, "%%left side data precedes right side data IF mono='B'\n");
    fprintf(fptr, "%%if both sides are not desired, then delete one and\n");
    fprintf(fptr, "%%change nlines= line appropriately\n");
    fprintf(fptr, "END_COMMENTS\n");
    fprintf(fptr, "PARAMS\n");
    fprintf(fptr, "END_PARAMS\n");
    fprintf(fptr, "RASTERDATA\n");
    fprintf(fptr, "nlines=%d\n",
          ((*mono)=='B') ?  2*SUMDATA(fd)->ndata : SUMDATA(fd)->ndata);
    if((*mono)!='R') {
      for(i=0; i<SUMDATA(fd)->ndata; i++)
        fprintf(fptr,"%e\t%e\n",xl[i],yl[i]); }
    if((*mono)!='L') {
      for(i=0; i<SUMDATA(fd)->ndata; i++)
        fprintf(fptr,"%e\t%e\n",xr[i],yr[i]); }
    fprintf(fptr, "END_RASTERDATA\n"); }

  if(fdo->no_X) return;

  if((*mono)!='R') {
    FDObj_Add_Data_All(fdo,xl,yl,SUMDATA(fd)->ndata);
    FDObj_AddLine(fdo, l, xl, yl, NULL, SUMDATA(fd)->ndata, AtFloat,
				  atQuadLinePlotWidgetClass, AtTypeLINEPOINTS,
				  AtMarkCIRCLE, ConvertColor(fdo->graph, "darkgreen"), 
		      (XtArgVal) "left"); }

  if((*mono)!='L') {
    FDObj_Add_Data_All(fdo,xr,yr,SUMDATA(fd)->ndata);
    FDObj_AddLine(fdo, l, xr, yr, NULL, SUMDATA(fd)->ndata, AtFloat,
				  atQuadLinePlotWidgetClass, AtTypeLINEPOINTS,
				  AtMarkCIRCLE, ConvertColor(fdo->graph, "red"), 
		      (XtArgVal) "right"); }

  if(mag) {
    ylabel="maximum achievable intensity (dB)";
    ti = "separate intensity"; }
  else {
    ylabel="phase (us)";
    ti = "separate phase"; }
  xlabel="frequency (Hz)";
  XtVaSetValues(fdo->xaxis, XtNlabel, xlabel, NULL);
  XtVaSetValues(fdo->yaxis, XtNlabel, ylabel, NULL);
  XtVaSetValues(fdo->graph, XtNtitle, FDObj_Title(fdo, ti, l),
        XtNshowTitle, True, NULL);
}

static void cal_differenceFN(FDO *fdo, FILEDATA *fd, int l, int mag,
      FILE *fptr)
{
  char *ylabel, *xlabel, *ti;
  int i;
  float *xl, *yl;

  assert((xl = (float*)malloc(sizeof(float)*SUMDATA(fd)->ndata))!=NULL);
  assert((yl = (float*)malloc(sizeof(float)*SUMDATA(fd)->ndata))!=NULL);

  for(i=0; i<SUMDATA(fd)->ndata; i++) {
    xl[i]=SUMDATA(fd)->freqs[i];
    if(mag)
      yl[i]=SUMDATA(fd)->r_mags[i]-SUMDATA(fd)->l_mags[i];
    else
      yl[i]=(SUMDATA(fd)->r_phis[i]-SUMDATA(fd)->l_phis[i]); }

  if(!mag) {
    unwrap(yl,SUMDATA(fd)->ndata,M_PI);
  	for(i=0; i<SUMDATA(fd)->ndata; i++)
      yl[i] *= 1.0e6/(2.0*M_PI*SUMDATA(fd)->freqs[i]); }

  if(fptr!=NULL) {
    fprintf(fptr, "COMMENTS\n");
    fprintf(fptr,"%%text dump of %s, difference view\n",
          strrchr(fd->filename,'/')==NULL ? fd->filename :
          1+(char*)strrchr(fd->filename,'/'));
    fprintf(fptr,"%%col 1: frequency (Hz)\n");
    if(mag)
      fprintf(fptr,"%%col 2: intensity (dB SPL)\n");
    else
      fprintf(fptr,"%%col 2: phase (us)\n");
    fprintf(fptr, "END_COMMENTS\n");
    fprintf(fptr, "PARAMS\n");
    fprintf(fptr, "END_PARAMS\n");
    fprintf(fptr, "RASTERDATA\n");
    fprintf(fptr, "nlines=%d\n",SUMDATA(fd)->ndata);
    for(i=0; i<SUMDATA(fd)->ndata; i++)
      fprintf(fptr,"%e\t%e\n",xl[i],yl[i]);
    fprintf(fptr, "END_RASTERDATA\n"); }

  if(fdo->no_X) return;

  FDObj_Add_Data_All(fdo,xl,yl,SUMDATA(fd)->ndata);
  FDObj_AddLine(fdo, l, xl, yl, NULL, SUMDATA(fd)->ndata, AtFloat,
				atQuadLinePlotWidgetClass, AtTypeLINEPOINTS,
				AtMarkCIRCLE, ConvertColor(fdo->graph, "blue"), (XtArgVal) "R-L");

  if(mag) {
    ylabel="intensity difference (dB)";
    ti = "difference intensity"; }
  else {
    ylabel="phase (us)";
    ti = "difference phase"; }
  xlabel="frequency (Hz)";
  XtVaSetValues(fdo->xaxis, XtNlabel, xlabel, NULL);
  XtVaSetValues(fdo->yaxis, XtNlabel, ylabel, NULL);
  XtVaSetValues(fdo->graph, XtNtitle, FDObj_Title(fdo, ti, l),
        XtNshowTitle, True, NULL);
}

static void cal_separate_leakFN(FDO *fdo, FILEDATA *fd, int l, int mag,
      FILE *fptr)
{
  char *ylabel, *xlabel, *ti, *mono;
  int i;
  float *xl, *yl, *xr, *yr;

  mono = FD_GV(fd,"cal.mono");

  if((*mono)!='R') {
    COPY_ARRAY(SUMDATA(fd)->freqs,xr,SUMDATA(fd)->ndata,float);
    COPY_ARRAY(mag ? SUMDATA(fd)->r_lkmags : SUMDATA(fd)->r_lkphis,
          yr,SUMDATA(fd)->ndata,float);

    if(!mag)
      unwrap(yr,SUMDATA(fd)->ndata,M_PI);

    for(i=0; i<SUMDATA(fd)->ndata; i++)
      if(!mag)
        yr[i]=1.0e6*yr[i]/(2.0*M_PI*xr[i]); }

  if((*mono)!='L') {
    COPY_ARRAY(SUMDATA(fd)->freqs,xl,SUMDATA(fd)->ndata,float);
    COPY_ARRAY(mag ? SUMDATA(fd)->l_lkmags : SUMDATA(fd)->l_lkphis,
          yl,SUMDATA(fd)->ndata,float);

    if(!mag)
      unwrap(yl,SUMDATA(fd)->ndata,M_PI);

    for(i=0; i<SUMDATA(fd)->ndata; i++)
      if(!mag)
        yl[i]=1.0e6*yl[i]/(2.0*M_PI*xl[i]); }

  if(fptr!=NULL) {
    fprintf(fptr, "COMMENTS\n");
    fprintf(fptr,"%%text dump of %s, separate leak view\n",
          strrchr(fd->filename,'/')==NULL ? fd->filename :
          1+(char*)strrchr(fd->filename,'/'));
    fprintf(fptr,"%%col 1: frequency (Hz)\n");
    if(mag)
      fprintf(fptr,"%%col 2: leak intensity (dB SPL)\n");
    else
      fprintf(fptr,"%%col 2: leak phase (us)\n");
    fprintf(fptr, "END_COMMENTS\n");
    fprintf(fptr, "PARAMS\n");
    fprintf(fptr, "END_PARAMS\n");
    fprintf(fptr, "RASTERDATA\n");
    fprintf(fptr, "nlines=%d\n",
          ((*mono)=='B') ? 2*SUMDATA(fd)->ndata : SUMDATA(fd)->ndata);
    if((*mono)!='R') {
      for(i=0; i<SUMDATA(fd)->ndata; i++)
        fprintf(fptr,"%e\t%e\n",xr[i],yr[i]); }
    if((*mono)!='L') {
      for(i=0; i<SUMDATA(fd)->ndata; i++)
        fprintf(fptr,"%e\t%e\n",xl[i],yl[i]); }
    fprintf(fptr, "END_RASTERDATA\n"); }

  if(fdo->no_X) return;

  if((*mono)!='R') {
      FDObj_Add_Data_All(fdo,xr,yr,SUMDATA(fd)->ndata);
      FDObj_AddLine(fdo, l, xr, yr, NULL, SUMDATA(fd)->ndata, AtFloat,
					  atQuadLinePlotWidgetClass, AtTypeLINEPOINTS,
					  AtMarkCIRCLE, ConvertColor(fdo->graph, "red"), 
		        (XtArgVal) "right"); }

  if((*mono)!='L') {
      FDObj_Add_Data_All(fdo,xl,yl,SUMDATA(fd)->ndata);
      FDObj_AddLine(fdo, l, xl, yl, NULL, SUMDATA(fd)->ndata, AtFloat,
					  atQuadLinePlotWidgetClass, AtTypeLINEPOINTS,
					  AtMarkCIRCLE, ConvertColor(fdo->graph, "darkgreen"), 
		        (XtArgVal) "left"); }

  if(mag) {
    ylabel="leak intensity (dB)";
    ti = "separate leak intensity"; }
  else {
    ylabel="leakage phase (us)";
    ti = "separate leak phase"; }
  xlabel="frequency (Hz)";
  XtVaSetValues(fdo->xaxis, XtNlabel, xlabel, NULL);
  XtVaSetValues(fdo->yaxis, XtNlabel, ylabel, NULL);
  XtVaSetValues(fdo->graph, XtNtitle, FDObj_Title(fdo, ti, l),
        XtNshowTitle, True, NULL);
}

static void cal_distortionFN(FDO *fdo, FILEDATA *fd, int l, FILE *fptr)
{
  char *ylabel, *xlabel, *ti, *mono;
  int i;
  float *xl, *yl, *xr, *yr;

  mono = FD_GV(fd,"cal.mono");

  if((*mono)!='R') {
  	COPY_ARRAY(SUMDATA(fd)->freqs,xl,SUMDATA(fd)->ndata,float);
  	COPY_ARRAY(SUMDATA(fd)->l_dists,yl,SUMDATA(fd)->ndata,float); }
 
  if((*mono)!='L') {
  	COPY_ARRAY(SUMDATA(fd)->freqs,xr,SUMDATA(fd)->ndata,float);
  	COPY_ARRAY(SUMDATA(fd)->r_dists,yr,SUMDATA(fd)->ndata,float); }

  if(fptr!=NULL) {
    fprintf(fptr, "COMMENTS\n");
    fprintf(fptr,"%%text dump of %s, distortion view\n",
          strrchr(fd->filename,'/')==NULL ? fd->filename :
          1+(char*)strrchr(fd->filename,'/'));
    fprintf(fptr,"%%col 1: frequency (Hz)\n");
    fprintf(fptr,"%%col 2: distortion (%%)\n");
    fprintf(fptr, "END_COMMENTS\n");
    fprintf(fptr, "PARAMS\n");
    fprintf(fptr, "END_PARAMS\n");
    fprintf(fptr, "RASTERDATA\n");
    fprintf(fptr, "nlines=%d\n",
          ((*mono)=='B') ? 2*SUMDATA(fd)->ndata : SUMDATA(fd)->ndata);
    if((*mono)!='R') {
      for(i=0; i<SUMDATA(fd)->ndata; i++)
        fprintf(fptr,"%e\t%e\n",xl[i],yl[i]); }
    if((*mono)!='L') {
      for(i=0; i<SUMDATA(fd)->ndata; i++)
        fprintf(fptr,"%e\t%e\n",xr[i],yr[i]); }
    fprintf(fptr, "END_RASTERDATA\n"); }

  if(fdo->no_X) return;

  if((*mono)!='R') {
      FDObj_Add_Data_All(fdo,xl,yl,SUMDATA(fd)->ndata);
      FDObj_AddLine(fdo, l, xl, yl, NULL, SUMDATA(fd)->ndata, AtFloat,
					  atQuadLinePlotWidgetClass, AtTypeLINEPOINTS,
					  AtMarkCIRCLE, ConvertColor(fdo->graph, "darkgreen"), 
		        (XtArgVal) "left"); }

  if((*mono)!='L') {
      FDObj_Add_Data_All(fdo,xr,yr,SUMDATA(fd)->ndata);
      FDObj_AddLine(fdo, l, xr, yr, NULL, SUMDATA(fd)->ndata, AtFloat,
					  atQuadLinePlotWidgetClass, AtTypeLINEPOINTS,
					  AtMarkCIRCLE, ConvertColor(fdo->graph, "red"), 
		        (XtArgVal) "right"); }

  ylabel="distortion (%)";
  ti = "distortion";
  xlabel="frequency (Hz)";
  XtVaSetValues(fdo->xaxis, XtNlabel, xlabel, NULL);
  XtVaSetValues(fdo->yaxis, XtNlabel, ylabel, NULL);
  XtVaSetValues(fdo->graph, XtNtitle, FDObj_Title(fdo, ti, l),
        XtNshowTitle, True, NULL);
}

static void cal_distortion_leakFN(FDO *fdo, FILEDATA *fd, int l, FILE *fptr)
{
  char *ylabel, *xlabel, *ti, *mono;
  int i,ndata;
  float *xl, *yl, *xr, *yr;

  ndata = 2*(SUMDATA(fd)->ndata-1);
  mono = FD_GV(fd,"cal.mono");

  if((*mono)!='R') {
    COPY_ARRAY(SUMDATA(fd)->freqs,xl,SUMDATA(fd)->ndata,float);
    COPY_ARRAY(SUMDATA(fd)->l_lkdists,yl,SUMDATA(fd)->ndata,float); }

  if((*mono)!='L') {
    COPY_ARRAY(SUMDATA(fd)->freqs,xr,SUMDATA(fd)->ndata,float);
    COPY_ARRAY(SUMDATA(fd)->r_lkdists,yr,SUMDATA(fd)->ndata,float); }

  if(fptr!=NULL) {
    fprintf(fptr, "COMMENTS\n");
    fprintf(fptr,"%%text dump of %s, distortion leak view\n",
          strrchr(fd->filename,'/')==NULL ? fd->filename :
          1+(char*)strrchr(fd->filename,'/'));
    fprintf(fptr,"%%col 1: frequency (Hz)\n");
    fprintf(fptr,"%%col 2: leak distortion (%%)\n");
    fprintf(fptr, "END_COMMENTS\n");
    fprintf(fptr, "PARAMS\n");
    fprintf(fptr, "END_PARAMS\n");
    fprintf(fptr, "RASTERDATA\n");
    fprintf(fptr, "nlines=%d\n",
          ((*mono)=='B') ? 2*SUMDATA(fd)->ndata : SUMDATA(fd)->ndata);
    if((*mono)!='R') {
      for(i=0; i<SUMDATA(fd)->ndata; i++)
        fprintf(fptr,"%e\t%e\n",xl[i],yl[i]); }
    if((*mono)!='L') {
      for(i=0; i<SUMDATA(fd)->ndata; i++)
        fprintf(fptr,"%e\t%e\n",xr[i],yr[i]); }
    fprintf(fptr, "END_RASTERDATA\n"); }

  if(fdo->no_X) return;

  if((*mono)!='R') {
      FDObj_Add_Data_All(fdo,xl,yl,SUMDATA(fd)->ndata);
      FDObj_AddLine(fdo, l, xl, yl, NULL, SUMDATA(fd)->ndata, AtFloat,
					  atQuadLinePlotWidgetClass, AtTypeLINEPOINTS,
					  AtMarkCIRCLE, ConvertColor(fdo->graph, "darkgreen"), 
		        (XtArgVal) "left"); }

  if((*mono)!='L') {
      FDObj_Add_Data_All(fdo,xr,yr,SUMDATA(fd)->ndata);
      FDObj_AddLine(fdo, l, xr, yr, NULL, SUMDATA(fd)->ndata, AtFloat,
					  atQuadLinePlotWidgetClass, AtTypeLINEPOINTS,
					  AtMarkCIRCLE, ConvertColor(fdo->graph, "red"), 
		        (XtArgVal) "right"); }

  ylabel="leak distortion (%)";
  xlabel="frequency (Hz)";
  ti = "distortion leak";
  XtVaSetValues(fdo->xaxis, XtNlabel, xlabel, NULL);
  XtVaSetValues(fdo->yaxis, XtNlabel, ylabel, NULL);
  XtVaSetValues(fdo->graph, XtNtitle, FDObj_Title(fdo, ti, l),
        XtNshowTitle, True, NULL);
}

#ifndef SWAP
#define SWAP(a,b,type) { type tmp; tmp=a; a=b; b=tmp; }
#endif /* SWAP */
static int cal_reader(FILE *fptr, FILEDATA *fd, int ana)
{
  int i;
  char *p,buf[1024];

  fd->rawdata=NULL;
  fd->channels=NULL;
  assert((fd->sumdata=malloc(sizeof(SumData)))!=NULL);

  if ((p = skipTo(fptr, "nrasters=")) == NULL ||
      sscanf(p, "nrasters=%d", &(SUMDATA(fd)->ndata)) != 1) {
    if (debugflag)
      fprintf(stderr, "FD_read: %s missing nrasters= line\n", fd->filename);
    fclose(fptr);
    SUMDATA(fd)->freqs=NULL;
    SUMDATA(fd)->l_mags=SUMDATA(fd)->l_phis=SUMDATA(fd)->l_dists=NULL;
    SUMDATA(fd)->r_mags=SUMDATA(fd)->r_phis=SUMDATA(fd)->r_dists=NULL;
    SUMDATA(fd)->l_lkmags=SUMDATA(fd)->l_lkphis=SUMDATA(fd)->l_lkdists=NULL;
    SUMDATA(fd)->r_lkmags=SUMDATA(fd)->r_lkphis=SUMDATA(fd)->r_lkdists=NULL;
    FD_free(fd);
    return(0); }

  assert((SUMDATA(fd)->freqs=(float*)malloc(SUMDATA(fd)->ndata*
        sizeof(float)))!=NULL);
  assert((SUMDATA(fd)->l_mags=(float*)malloc(SUMDATA(fd)->ndata*
        sizeof(float)))!=NULL);
  assert((SUMDATA(fd)->l_phis=(float*)malloc(SUMDATA(fd)->ndata*
        sizeof(float)))!=NULL);
  assert((SUMDATA(fd)->l_dists=(float*)malloc(SUMDATA(fd)->ndata*
        sizeof(float)))!=NULL);
  assert((SUMDATA(fd)->l_lkmags=(float*)malloc(SUMDATA(fd)->ndata*
        sizeof(float)))!=NULL);
  assert((SUMDATA(fd)->l_lkphis=(float*)malloc(SUMDATA(fd)->ndata*
        sizeof(float)))!=NULL);
  assert((SUMDATA(fd)->l_lkdists=(float*)malloc(SUMDATA(fd)->ndata*
        sizeof(float)))!=NULL);
  assert((SUMDATA(fd)->r_mags=(float*)malloc(SUMDATA(fd)->ndata*
        sizeof(float)))!=NULL);
  assert((SUMDATA(fd)->r_phis=(float*)malloc(SUMDATA(fd)->ndata*
        sizeof(float)))!=NULL);
  assert((SUMDATA(fd)->r_dists=(float*)malloc(SUMDATA(fd)->ndata*
        sizeof(float)))!=NULL);
  assert((SUMDATA(fd)->r_lkmags=(float*)malloc(SUMDATA(fd)->ndata*
        sizeof(float)))!=NULL);
  assert((SUMDATA(fd)->r_lkphis=(float*)malloc(SUMDATA(fd)->ndata*
        sizeof(float)))!=NULL);
  assert((SUMDATA(fd)->r_lkdists=(float*)malloc(SUMDATA(fd)->ndata*
        sizeof(float)))!=NULL);

  for(i=0; i<SUMDATA(fd)->ndata; i++) {
    fgets(buf,sizeof(buf),fptr);
    sscanf(buf,"%f %f %f %f %f %f %f %f %f %f %f %f %f",
          &(SUMDATA(fd)->freqs[i]),
          &(SUMDATA(fd)->l_mags[i]),
          &(SUMDATA(fd)->l_phis[i]),
          &(SUMDATA(fd)->l_dists[i]),
          &(SUMDATA(fd)->l_lkmags[i]),
          &(SUMDATA(fd)->l_lkphis[i]),
          &(SUMDATA(fd)->l_lkdists[i]),
          &(SUMDATA(fd)->r_mags[i]),
          &(SUMDATA(fd)->r_phis[i]),
          &(SUMDATA(fd)->r_dists[i]),
          &(SUMDATA(fd)->r_lkmags[i]),
          &(SUMDATA(fd)->r_lkphis[i]),
          &(SUMDATA(fd)->r_lkdists[i])); }

  /* sort */
  {
  float min_val;  int i,j,min_loc;

  for(i=0; i<(SUMDATA(fd)->ndata-1); i++) {
    min_val=SUMDATA(fd)->freqs[i];  min_loc=i;
    for(j=i+1; j<SUMDATA(fd)->ndata; j++) {
      if(SUMDATA(fd)->freqs[j]<min_val) {
        min_val=SUMDATA(fd)->freqs[j]; min_loc=j; } }
    if(min_loc!=i) {
      SWAP(SUMDATA(fd)->freqs[i],SUMDATA(fd)->freqs[min_loc],float);
      SWAP(SUMDATA(fd)->l_mags[i],SUMDATA(fd)->l_mags[min_loc],float);
      SWAP(SUMDATA(fd)->l_phis[i],SUMDATA(fd)->l_phis[min_loc],float);
      SWAP(SUMDATA(fd)->l_dists[i],SUMDATA(fd)->l_dists[min_loc],float);
      SWAP(SUMDATA(fd)->l_lkmags[i],SUMDATA(fd)->l_lkmags[min_loc],float);
      SWAP(SUMDATA(fd)->l_lkphis[i],SUMDATA(fd)->l_lkphis[min_loc],float);
      SWAP(SUMDATA(fd)->l_lkdists[i],SUMDATA(fd)->l_lkdists[min_loc],float);
      SWAP(SUMDATA(fd)->r_mags[i],SUMDATA(fd)->r_mags[min_loc],float);
      SWAP(SUMDATA(fd)->r_phis[i],SUMDATA(fd)->r_phis[min_loc],float);
      SWAP(SUMDATA(fd)->r_dists[i],SUMDATA(fd)->r_dists[min_loc],float);
      SWAP(SUMDATA(fd)->r_lkmags[i],SUMDATA(fd)->r_lkmags[min_loc],float);
      SWAP(SUMDATA(fd)->r_lkphis[i],SUMDATA(fd)->r_lkphis[min_loc],float);
      SWAP(SUMDATA(fd)->r_lkdists[i],SUMDATA(fd)->r_lkdists[min_loc],float); } }
  }

  return(1);
}

#undef SWAP

#ifndef FREE
#define FREE(p) if ((p)) {free((p)); (p) = NULL;}
#endif /* FREE */

static int cal_free(FILEDATA *fd)
{
  FREE(SUMDATA(fd)->freqs);

  FREE(SUMDATA(fd)->l_mags);
  FREE(SUMDATA(fd)->l_phis);
  FREE(SUMDATA(fd)->l_dists);
  FREE(SUMDATA(fd)->l_lkmags);
  FREE(SUMDATA(fd)->l_lkphis);
  FREE(SUMDATA(fd)->l_lkdists);

  FREE(SUMDATA(fd)->r_mags);
  FREE(SUMDATA(fd)->r_phis);
  FREE(SUMDATA(fd)->r_dists);
  FREE(SUMDATA(fd)->r_lkmags);
  FREE(SUMDATA(fd)->r_lkphis);
  FREE(SUMDATA(fd)->r_lkdists);

  FREE(SUMDATA(fd));

  return(1);
}
#undef FREE

float *pm_cal_get_freqs(FILEDATA *data)
{
  return(SUMDATA(data)->freqs);
}

int pm_cal_get_ndata(FILEDATA *data)
{
  return(SUMDATA(data)->ndata);
}

float *pm_cal_get_lmags(FILEDATA *data)
{
  return(SUMDATA(data)->l_mags);
}

float *pm_cal_get_rmags(FILEDATA *data)
{
  return(SUMDATA(data)->r_mags);
}

float *pm_cal_get_lphis(FILEDATA *data)
{
  return(SUMDATA(data)->l_phis);
}

float *pm_cal_get_rphis(FILEDATA *data)
{
  return(SUMDATA(data)->r_phis);
}

int pm_cal_init(void)
{
  setFDOvalidviewMethod("cal", cal_valid_view);
  setFDOdoplotMethod("cal", cal_do_plot);
  setFDOplotMethod("cal", cal_plotter);
  setFDreadMethod("cal", cal_reader);
  setFDfreeMethod("cal", cal_free);
  return(1);
}
