/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: pm_bw.c,v 1.16 1999/05/18 19:17:13 bjarthur Exp $
** Program: dowl
**  Module: pm_fr.c
**  Author: bjarthur
** Descrip: xdowl plot method -- static iid tuning curve plotter
**
** Revision History (most recent last)
**
** 10NOV96 bjarthur
**  copied from pm_iid.c
**
** NOV96 bjarthur
**   copied from pm_fr.c
**
*******************************************************************/

#include "xdphyslib.h"
#include "xcalibur.h"
#include "plotter.h"

static void bw_FN(FDO*,FILEDATA*,int,FILE*);
static int bw_plotter(FDO*,FILEDATA*,FDObj_ViewType*,int,FILE*);
static int bw_valid_view(FDO *, int);
static int bw_reader(FILE*,FILEDATA*,int);
static int bw_free(FILEDATA*);

typedef struct {
  int           ndata;
  float        *bws,*mags;
} SumData;

#define SUMDATA(data) ((SumData*)(data->sumdata))

static int view_order[][1] = {{PM_CURVE}};
static int nsubviews[] = {1};
static int nviews = 1;

int bw_do_plot(FDO *fdo, FILEDATA *fd, int view, int l, FILE *fptr)
{
  if (!bw_valid_view(fdo, view)) {
		pm_type_error("bw", view);
		view = PM_DEFAULT;
		(void) bw_valid_view(fdo, view); }

  switch(view) {
  	case PM_DEFAULT:
  	case PM_CURVE:
  		bw_FN(fdo,fd,l,fptr);
  		break; 
  	default:
  		pm_type_error("bw", view);
  		break; }

  return(1);
}

static int bw_valid_view(FDO *fdo, int view)
{
	int retval = 0;
	int i,j;

	if (view == PM_DEFAULT) {
			retval = 1;
	} else {
		for (i=0; i<nviews; i++) {
	 		for (j=0; j<nsubviews[i]; j++) {
				if (view_order[i][j] == view) {
					retval = 1;
					fdo->view.lr = i;
					fdo->view.ud = j;
					break;
				}
			}
		}
	}

	return(retval);
}

static int bw_plotter(FDO *fdo, FILEDATA *fd, FDObj_ViewType *view,
			int l, FILE *fptr)
{
  adjust_index(view->lr,view->ud);

  bw_do_plot(fdo, fd, view_order[view->lr][view->ud], l, fptr);

  return(1);
}

static void bw_FN(FDO *fdo, FILEDATA *fd, int l, FILE *fptr)
{
  char *ylabel, *xlabel, *ti;
  int i;
  float *x, *y;

  COPY_ARRAY(SUMDATA(fd)->bws,x,SUMDATA(fd)->ndata,float);
  COPY_ARRAY(SUMDATA(fd)->mags,y,SUMDATA(fd)->ndata,float);
 
  if(fptr!=NULL) {
    fprintf(fptr, "COMMENTS\n");
    fprintf(fptr,"%%text dump of %s\n",
          strrchr(fd->filename,'/')==NULL ? fd->filename :
          1+(char*)strrchr(fd->filename,'/'));
    fprintf(fptr,"%%col 1: frequency (Hz)\n");
    fprintf(fptr,"%%col 2: intensity (dB SPL)\n");
    fprintf(fptr, "END_COMMENTS\n");
    fprintf(fptr, "PARAMS\n");
    fprintf(fptr, "END_PARAMS\n");
    fprintf(fptr, "RASTERDATA\n");
    fprintf(fptr, "nlines=%d\n", SUMDATA(fd)->ndata);
    for(i=0; i<SUMDATA(fd)->ndata; i++)
      fprintf(fptr,"%e\t%e\n",x[i],y[i]);
    fprintf(fptr, "END_RASTERDATA\n"); }

  if(fdo->no_X) return;

  FDObj_Add_Data_All(fdo,x,y,SUMDATA(fd)->ndata);
  FDObj_AddLine(fdo, l, x, y, NULL, SUMDATA(fd)->ndata, AtFloat,
				atQuadLinePlotWidgetClass, AtTypeLINEPOINTS, AtMarkCIRCLE,
				ConvertColor(fdo->graph, "blue"), (XtArgVal) "foo");

  ylabel="absolute intensity (dB)";
  xlabel="frequency (Hz)";
  ti = "";
  XtVaSetValues(fdo->xaxis, XtNlabel, xlabel, NULL);
  XtVaSetValues(fdo->yaxis, XtNlabel, ylabel, NULL);
  XtVaSetValues(fdo->graph, XtNtitle, FDObj_Title(fdo, ti, l),
        XtNshowTitle, True, NULL);
}

static int bw_reader(FILE *fptr, FILEDATA *fd, int ana)
{
  int i;
  char *p,buf[1024];

  fd->rawdata=NULL;
  fd->channels=NULL;
  assert((fd->sumdata=malloc(sizeof(SumData)))!=NULL);

  if ((p = skipTo(fptr, "nrasters=")) == NULL ||
      sscanf(p, "nrasters=%d", &(SUMDATA(fd)->ndata)) != 1) {
    if (debugflag)
      fprintf(stderr, "FD_read: %s missing nrasters= line\n", fd->filename);
    fclose(fptr);
    SUMDATA(fd)->bws=SUMDATA(fd)->mags=NULL;
    FD_free(fd);
    return(0); }

  assert((SUMDATA(fd)->bws=(float*)malloc(SUMDATA(fd)->ndata*
        sizeof(float)))!=NULL);
  assert((SUMDATA(fd)->mags=(float*)malloc(SUMDATA(fd)->ndata*
        sizeof(float)))!=NULL);

  for(i=0; i<SUMDATA(fd)->ndata; i++) {
    fgets(buf,sizeof(buf),fptr);
    sscanf(buf,"%f %f",&(SUMDATA(fd)->bws[i]),&(SUMDATA(fd)->mags[i])); }

  /* sort */
  {
  float min_val;  int i,j,min_loc;

  for(i=0; i<(SUMDATA(fd)->ndata-1); i++) {
    min_val=SUMDATA(fd)->bws[i];  min_loc=i;
    for(j=i+1; j<SUMDATA(fd)->ndata; j++) {
      if(SUMDATA(fd)->bws[j]<min_val) {
        min_val=SUMDATA(fd)->bws[j]; min_loc=j; } }
    if(min_loc!=i) {
      SWAP(SUMDATA(fd)->bws[i],SUMDATA(fd)->bws[min_loc],float);
      SWAP(SUMDATA(fd)->mags[i],SUMDATA(fd)->mags[min_loc],float); } }
  }

  return(1);
}

static int bw_free(FILEDATA *fd)
{
  FREE(SUMDATA(fd)->bws);
  FREE(SUMDATA(fd)->mags);

  FREE(SUMDATA(fd));

  return(1);
}

int pm_bw_init(void)
{
  setFDOvalidviewMethod("bw", bw_valid_view);
  setFDOdoplotMethod("bw", bw_do_plot);
  setFDOplotMethod("bw", bw_plotter);
  setFDreadMethod("bw", bw_reader);
  setFDfreeMethod("bw", bw_free);
  return(1);
}
