/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: pm_bf.c,v 1.11 1999/05/18 19:17:13 bjarthur Exp $
** Program: dowl
**  Module: pm_freq.c
**  Author: mazer
** Descrip: xdowl plot method -- freq. tuning curves (1 and 2 tone)
**
** Revision History (most recent last)
**
** Sun Mar 14 20:37:11 1993 mazer
**  changed to accept the new FILEDATA structure
**
** 97.4 bjarthur
**  changed name from 'freq' to 'bf' to be distinct from the new 'cf'
**
*******************************************************************/

#include "xdphyslib.h"
#include "xdphys.h"
#include "plotter.h"

static void fit_menu(FDO*);
static int bf_valid_view(FDO *, int);
static int bf_plotter(FDO*, FILEDATA*, FDObj_ViewType*, int, FILE*);

static int view_order[][3] = {
			{PM_CURVE},
			{PM_MEANPHASE, PM_VS, PM_RAYLEIGH},
			{PM_PERHIST},
			{PM_RASTER, PM_RASTER_RAW},
			{PM_PSTH},
			{PM_ISIH}};
static int nsubviews[] = {1,3,1,2,1,1};
static int nviews = 6;

int bf_do_plot(FDO *fdo, FILEDATA *fd, int view, int l, FILE *fptr)
{
	static int last_view=PM_NONE;

  if (!bf_valid_view(fdo, view)) {
		pm_type_error("bf", view);
		view = PM_DEFAULT;
		(void) bf_valid_view(fdo, view); }

  switch(view) {
    case PM_DEFAULT:
    case PM_CURVE:
      fit_menu(fdo);
      FD1_curveFN(fdo,fd,l,fptr);
      break;
    case PM_MEANPHASE:
      FD1_perhist1FN(fdo,fd,l,fptr,0.0,PERHIST_DEF_NBINS,0);
      break;
    case PM_VS:
      FD1_perhist1FN(fdo,fd,l,fptr,0.0,PERHIST_DEF_NBINS,1);
      break;
    case PM_RAYLEIGH:
      FD1_perhist1FN(fdo,fd,l,fptr,0.0,PERHIST_DEF_NBINS,2);
      break;
    case PM_PERHIST: {
			char tmp[128],*tmp2;
			int *stimarray,nstims;
			
			if(last_view!=PM_PERHIST) {
        sprintf(tmp,"%s.range",fdo->fds[fdo->nfds-1]->modname);
        if((tmp2=FD_GV(fdo->fds[fdo->nfds-1],tmp))!=NULL) {
        	stimarray = StimArray_Gen(&nstims, 0, tmp2, 1, 1, 0);
		  		fdo->vsm_data = stimarray[0]; }
		  	else {
        	sprintf(tmp,"%s.start",fdo->fds[fdo->nfds-1]->modname);
        	fdo->vsm_data=FD_GI(fdo->fds[fdo->nfds-1],tmp); } }
 
      FD1_perhist_stats_menu(fdo);
			FD_vsm_menu(fdo,0);
      FD1_perhist2FN(fdo,fd,l,fptr,0.0,PERHIST_DEF_NBINS); }
      break;
    case PM_RASTER:
      FD1_rasterFN(fdo,fd,l,0,fptr);
      break;
    case PM_RASTER_RAW:
      FD1_rasterFN(fdo,fd,l,1,fptr);
      break;
    case PM_PSTH:
      FD1_psthFN(fdo,fd,l,fptr);
      break;
    case PM_ISIH:
      FD1_isih_stats_menu(fdo);
      FD1_isihFN(fdo,fd,l,fptr);
      break; 
		default:
			pm_type_error("bf", view);
			break; }

	last_view = view;
  return(1);
}

static int bf_valid_view(FDO *fdo, int view)
{
	int retval = 0;
	int i,j;

	if (view == PM_DEFAULT) {
			retval = 1;
	} else {
		for (i=0; i<nviews; i++) {
	 		for (j=0; j<nsubviews[i]; j++) {
				if (view_order[i][j] == view) {
					retval = 1;
					fdo->view.lr = i;
					fdo->view.ud = j;
					break; } } } }

	return(retval);
}

static int bf_plotter(FDO *fdo, FILEDATA *fd, FDObj_ViewType *view,
			int l, FILE *fptr)
{
  adjust_index(view->lr,view->ud);

  if(!fdo->no_X) {
    if(nsubviews[view->lr]>1) {
      XtSetSensitive(fdo->up, True);
      XtSetSensitive(fdo->down, True); } }

  bf_do_plot(fdo, fd, view_order[view->lr][view->ud], l, fptr);

  return(1);
}

static void fit_menu(FDO *fdobj)
{
  if(fdobj->no_X) return;

  menubutton_clear(fdobj->fit,&(fdobj->fitpsh));
  XtSetSensitive(fdobj->fit, False);

  menubutton_add(fdobj->fitpsh,"peak & width",
				FD1_peak_stats,fdobj);
  menubutton_add(fdobj->fitpsh,"combine",CombineDataFN,fdobj);
  XtSetSensitive(fdobj->fit, True);
}

int pm_bf_init(void)
{
  setFDOvalidviewMethod("bf", bf_valid_view);
  setFDOdoplotMethod("bf", bf_do_plot);
  setFDOplotMethod("bf", bf_plotter);
  setFDreadMethod("bf", FD1_reader);
  setFDfreeMethod("bf", FD1_free);

  /* this is for backwards compatibility */
  setFDOvalidviewMethod("freq", bf_valid_view);
  setFDOdoplotMethod("freq", bf_do_plot);
  setFDOplotMethod("freq", bf_plotter);
  setFDreadMethod("freq", FD1_reader);
  setFDfreeMethod("freq", FD1_free);

  return(1);
}
