/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: pm_abi.c,v 1.13 1999/07/15 15:01:54 bjarthur Exp $
** Program: dowl
**  Module: pm_abi.c
**  Author: bjarthur
** Descrip: xdowl plot method -- freq. tuning curves (1 and 2 tone)
**
** Revision History (most recent last)
**
** NOV96 bjarthur
**  copied from pm_freq.c
**
** 97.3 bjarthur
**   added FitDataToTanh courtesy YA
**
*******************************************************************/

#include "xdphyslib.h"
#include "xdphys.h"
#include "plotter.h"

static void fit_menu(FDO*);
static int abi_plotter(FDO*, FILEDATA*, FDObj_ViewType*, int, FILE*);
static int abi_valid_view(FDO *, int);
static float EstimateInflexion(float*, float*, int);
static float FitTanhPoint(float, float*);
void Logistic(float, float[], float*, float[], int);
static int FitDataToTanh(float*, float*, float*, int, float**, float*, float*,
    float*, float*, float*, float*);

static int view_order[][3] = {
			{PM_CURVE},
			{PM_MEANPHASE, PM_VS, PM_RAYLEIGH},
			{PM_PERHIST},
			{PM_RASTER, PM_RASTER_RAW},
			{PM_PSTH},
			{PM_ISIH}};
static int nsubviews[] = {1,3,1,2,1,1};
static int nviews = 6;

int abi_do_plot(FDO *fdo, FILEDATA *fd, int view, int l, FILE *fptr)
{
  float period;
  int nbins;
  static int last_view=PM_NONE;

  if (!abi_valid_view(fdo, view)) {
		pm_type_error("abi", view);
		view = PM_DEFAULT;
		(void) abi_valid_view(fdo, view); }

	FD_perhist_compute_period_nbins(fd,&period,&nbins);

  switch(view) {
    case PM_DEFAULT:
    case PM_CURVE:
      fit_menu(fdo);
      FD1_curveFN(fdo,fd,l,fptr);
      break;
    case PM_MEANPHASE:
      FD1_perhist1FN(fdo,fd,l,fptr,period,nbins,0);
      break;
    case PM_VS:
      FD1_perhist1FN(fdo,fd,l,fptr,period,nbins,1);
      break;
    case PM_RAYLEIGH:
      FD1_perhist1FN(fdo,fd,l,fptr,period,nbins,2);
      break;
    case PM_PERHIST: {
      if(last_view!=PM_PERHIST)
        fdo->vsm_data=VSM_ALL_CODE;
      FD1_perhist_stats_menu(fdo);
      FD_vsm_menu(fdo,1);
      FD1_perhist2FN(fdo,fd,l,fptr,period,nbins); }
      break;
    case PM_RASTER:
      FD1_rasterFN(fdo,fd,l,0,fptr);
      break;
    case PM_RASTER_RAW:
      FD1_rasterFN(fdo,fd,l,1,fptr);
      break;
    case PM_PSTH:
      FD1_psthFN(fdo,fd,l,fptr);
      break;
    case PM_ISIH:
      FD1_isih_stats_menu(fdo);
      FD1_isihFN(fdo,fd,l,fptr);
      break; 
		default:
			pm_type_error("abi", view);
			break; }

  last_view = view;
  return(1);
}

static int abi_valid_view(FDO *fdo, int view)
{
	int retval = 0;
	int i,j;

	if (view == PM_DEFAULT) {
			retval = 1;
	} else {
		for (i=0; i<nviews; i++) {
			for (j=0; j<nsubviews[i]; j++) {
				if (view_order[i][j] == view) {
					retval = 1;
					fdo->view.lr = i;
					fdo->view.ud = j;
					break;
				}
			}
		}
	}

	return(retval);
}

static int abi_plotter(FDO *fdo, FILEDATA *fd, FDObj_ViewType *view,
			int l, FILE *fptr)
{
  syn_spec ss;

  fd_syn_spec_parse(FD_GV(fdo->fds[l], "abi.Stim"),0,&ss);

  adjust_index(view->lr,view->ud);

  if(l==0) {
    if (!((ss.class == SC_TONE) || 
        ((ss.class == SC_STACK) && (ss.parms.stack.num_freqs == 1)))) {
      if(view->lr == 1) {
        view->lr = 3;
        view->ud = 0; }
      else if(view->lr == 2) {
        view->lr = 0;
        view->ud = 0; } } }
  else {
    if(!((ss.class == SC_TONE) ||
          ((ss.class == SC_STACK) && (ss.parms.stack.num_freqs == 1))))
      if((view->lr == 1) || (view->lr == 2)) {
        notify("some curves missing due to different stim types");
        return(0); } }

  if(nsubviews[view->lr]>1) {
    XtSetSensitive(fdo->up, True);
    XtSetSensitive(fdo->down, True); }

  abi_do_plot(fdo, fd, view_order[view->lr][view->ud], l, fptr);

  return(1);
}


void Sigmoid(Widget w, XtPointer fdoptr, XtPointer call_data)
{
  FDO *fdo = (FDO *) fdoptr;

  int i,j,count,size;
  char *buf,buf2[128],*tmp;
  float *yfit,dc,amp,slope,offset,chi;
  float big;
  float *x,*y,ndata;
  float *depvals;
  syn_spec ss;
  char *foo;
  float Min,Max,Slope,From,To;

  assert((buf=(char*)malloc(1024*sizeof(char)))!=NULL);
  size=1024;

  tmp="equation:  dc + amp * tanh(slope * x + offset)\n";
  strcpy(buf,tmp); count=strlen(tmp);
  tmp="units:  dc, amp (spikes);  slope(1/dB);  offset(none)\n";
  strcat(buf,tmp); count+=strlen(tmp);
  tmp="units:  Min, Max (spikes);  Slope(spikes/dB);  Range, Low, High (dB)\n\n";
  strcat(buf,tmp); count+=strlen(tmp);
  tmp="file            dc     amp     slope    offset  chi    |  Min    Max    Slope   Range   From   To\n";
  strcat(buf,tmp); count+=strlen(tmp);
  tmp="----            --     ---     -----    ------  ---    |  ---    ---    -----   -----   ----   --\n";
  strcat(buf,tmp); count+=strlen(tmp);

  for(i=0; i<fdo->nfds; i++) {
    ndata=FD1_get_ndata(fdo->fds[i]);
    depvals=FD1_get_depvals(fdo->fds[i]);
    foo=FD_GV(fdo->fds[i],"itd.Stim");
    fd_syn_spec_parse(foo,0,&ss);
    FitDataToTanh(depvals,
          FD1_get_means(fdo->fds[i]), FD1_get_stderrs(fdo->fds[i]),
          ndata, &yfit,&dc,&amp,&slope,&offset,&chi,&big);

    Min = dc - amp;
    Max = dc + amp;
    Slope = amp * slope;
    From = -(offset+1) / slope;
    To = -(offset-1) / slope;

#ifdef __linux__
    count += sprintf(buf2,"%-13s   %5.1f  %5.1f   %5.3f    %5.1f   %5.1f%1s |  %5.1f  %5.1f  %5.1f   %5.1f   %5.1f  %5.1f\n",
          strrchr(fdo->fds[i]->filename,'/')==NULL ?
                fdo->fds[i]->filename :
                1+(char*)strrchr(fdo->fds[i]->filename,'/'),
          dc,amp,slope,offset,chi, ((int) big) ? "*" : " ",
          Min,Max,Slope,(To-From),From,To);
#else
    count += strlen(sprintf(buf2,"%-13s   %5.1f  %5.1f   %5.3f    %5.1f   %5.1f%1s |  %5.1f  %5.1f  %5.1f   %5.1f   %5.1f  %5.1f\n",
          strrchr(fdo->fds[i]->filename,'/')==NULL ?
                fdo->fds[i]->filename :
                1+(char*)strrchr(fdo->fds[i]->filename,'/'),
          dc,amp,slope,offset,chi, ((int) big) ? "*" : " ",
          Min,Max,Slope,(To-From),From,To));
#endif
    if(fdo->to_tty) {
      printf("%s",buf2);
      continue; }
    strcat(buf,buf2);
    if((size-count)<512) {
        assert((buf=(char*)realloc(buf,(size+1024)*sizeof(char)))!=NULL);
        size += 1024; }

    fdo->num_atlines[i]++;
    assert((fdo->atlines[i] = (Widget*)realloc(fdo->atlines[i],
          fdo->num_atlines[i]*sizeof(Widget)))!=NULL);
    fdo->atlines[i][fdo->num_atlines[i]-1] = XtVaCreateWidget("line",
          atQuadLinePlotWidgetClass, fdo->graph, NULL);
    assert((x=(float*)malloc(2*(ndata-2)*sizeof(float)))!=NULL);
    assert((y=(float*)malloc(2*(ndata-2)*sizeof(float)))!=NULL);

    for(j=0; j<(ndata-2); j++) {
      x[2*j]=depvals[j+1];   /* skip over spont */
      x[2*j+1]=depvals[j+2];
      y[2*j]=yfit[j+1];      /* yfit is unit offset */
      y[2*j+1]=yfit[j+2]; }

    FDObj_Add_Data_All(fdo,x,y,2*(ndata-2));
    AtQuadLinePlotAttachData(fdo->atlines[i][fdo->num_atlines[i]-1],
                     x, AtFloat, sizeof(float), y, AtFloat, sizeof(float),
                     NULL, AtFloat, sizeof(float),
                     NULL, AtFloat, sizeof(float), 0, 2*(ndata-2));
		if(fdo->baw) {
        XtVaSetValues(fdo->atlines[i][fdo->num_atlines[i]-1],
            XtNplotType, AtTypeLINEPOINTS,
            XtNplotMark, FDObj_Mark[i%FDObj_numMark], NULL); }
		else {
        XtVaSetValues(fdo->atlines[i][fdo->num_atlines[i]-1],
            XtNplotType, AtTypeLINEPOINTS,
            XtNplotMark, AtMarkCIRCLE,
            XtNforeground, ConvertColor(fdo->graph,
            FDObj_Colors[i%FDObj_numColors]), NULL); } }

  tmp="\n* = chi is big, fitted parameters are questionable";
  strcat(buf,tmp); count+=strlen(tmp);

  if(!fdo->to_tty)
    pop_text("Sigmoid", buf, strlen(buf)+1, False);
}

static void fit_menu(FDO *fdobj)
{
  if(fdobj->no_X) return;

  menubutton_clear(fdobj->fit,&(fdobj->fitpsh));
  XtSetSensitive(fdobj->fit, False);

  menubutton_add(fdobj->fitpsh,"Sigmoid",Sigmoid,fdobj);
  XtSetSensitive(fdobj->fit, True);
}

int pm_abi_init(void)
{
  setFDOvalidviewMethod("abi", abi_valid_view);
  setFDOdoplotMethod("abi", abi_do_plot);
  setFDOplotMethod("abi", abi_plotter);
  setFDreadMethod("abi", FD1_reader);
  setFDfreeMethod("abi", FD1_free);
  return(1);
}




/****************** from yuda **************************/

#define CHI2_STOP_STEP (0.5)

int ndBRangeMin = -70;
int ndBRangeMax = 70;

static float EstimateInflexion(float* x, float* y, int ndata)
{
  float miny, maxy, medy;
  float x1, x2, y1, y2;
  int i;
  miny = maxy = y[1];
  for(i = 1; i < ndata; i++)
  {
    if(y[i] > maxy) maxy = y[i];
    if(y[i] < miny) miny = y[i];
  }
  medy = (miny + maxy) / 2;

  for(i = 1; i < ndata; i++) if(y[i] > medy) break;
  x1 = x[i-1];
  x2 = x[i];
  y1 = y[i-1];
  y2 = y[i];
#if(0)
  fprintf(stderr, "\nX1 %5.2f X2 %5.2f Y1 %5.2f Y2 %5.2f medy %5.2f\n",
    x1, x2, y1, y2, medy);
#endif
  if(y2 == y1) return (x2 + x1) /2;
  return x1 + (medy - y1) * (x2 - x1) / (y2 - y1);
}

static float FitTanhPoint(float x, float* a)
{
  return a[1] + a[2] * tanh(a[3] * x + a[4]);
}

void Logistic(float x, float a[], float *y, float dyda[], int na)
{
  float X = a[3] * x + a[4];
  float T = tanh(X);
  float C = cosh(X);
  *y = a[1] + a[2] * T;

  dyda[1] = 1;
  dyda[2] = T;
  dyda[3] = x * a[2] / C / C;
  dyda[4] = a[2] / C / C;
} 

static int FitDataToTanh(float* fAxis, float* fData, float* fSig, int ndata,
    float **yfit, float *dc, float *amp, float *slope, float *offset,
    float *chi, float *big)
{
#if(0)
  float DRMax, DRMin, DRSlope, DRdr, DRFrom, DRTo;
  float fFreq, fCD, fPeriod, fFall;
#endif
  float *x, *y, *sig;
  int  i, *ia, ma, undermin, abovemax;
  float miny, maxy, *a, **covar, **alpha, alamda = -1, chisq = 10, 
	  oldchi = 1000;
  void (*GeneralFit)(float, float[], float*, float[], int);


  GeneralFit = Logistic;
  ma = 4;

  for(undermin = abovemax = i = 0; i < ndata; i++)
  {
    if(fAxis[i] < ndBRangeMin) undermin++;
    if(fAxis[i] > ndBRangeMax) abovemax++;
#if(0)
    fprintf(stderr, "\n%d %f %d %d", i, fAxis[i], undermin, abovemax);
#endif
  }
  fAxis += undermin;
  fData += undermin;
  ndata -= (undermin + abovemax);
    
  alpha = matrix(1,ma,1,ma);
  covar = matrix(1,ma,1,ma);
  ia = ivector(1,ma);
  a = vector(1,ma);
  x = vector(1, ndata);
  y = vector(1, ndata);
  sig = vector(1, ndata);

  for(i = 1; i <= ndata; i++)
  {
    sig[i] = fSig[i-1];
    x[i] = fAxis[i-1];
    y[i] = fData[i-1];
  }

  for(i = 1; i <= ma; i++) ia[i] = 1;

  miny = maxy = y[1];
  for(i = 2; i <= ndata; i++) if(y[i] > maxy) maxy = y[i];
  for(i = 2; i <= ndata; i++) if(y[i] < miny) miny = y[i];
  a[1] = (maxy + miny) / 2;
  a[2] = (maxy - miny) / 2;
  a[3] = 1.0 / 10.0;
  a[4] = -a[3] * EstimateInflexion(x, y, ndata);

  alamda = -1;
  while(((oldchi - chisq) > CHI2_STOP_STEP) || ((oldchi - chisq) < -CHI2_STOP_STEP))
  {
    oldchi = chisq;
#if(0)
    fprintf(stderr,"\n%5.1f + %5.1f * tanh(%5.3f * x + %5.1f)"
		,a[1],a[2],a[3],a[4]);
    fprintf(stderr,"\t%5.3f\t%5.3f",alamda, oldchi-chisq);
#endif
    mrqmin(x,y,sig,ndata,a,ia,ma,covar,alpha,&chisq,GeneralFit,&alamda);
  }
  oldchi = 0;
  mrqmin(x,y,sig,ndata,a,ia,ma,covar,alpha,&chisq,GeneralFit,&oldchi);

  assert(((*yfit)=(float*)malloc((ndata+1)*sizeof(float)))!=NULL);
  for(i=1; i<=ndata; i++)
    (*yfit)[i]=FitTanhPoint(x[i],a);
  (*dc)=a[1]; (*amp)=a[2]; (*slope)=a[3]; (*offset)=a[4];
  (*chi)=chisq;
  (*big) = (chisq > ((ndata-ma) + (float)sqrt(2.0 * (ndata-ma)))) ? 1 : 0;

#if(0)
  DRParams(a, &DRMax, &DRMin, &DRSlope, &DRdr, &DRFrom, &DRTo);

  fprintf(stderr, "\n%s\t%s\t%s\n", "ABI", "Data", "Fit");
  for(i = 1; i <= ndata; i++)
    fprintf(stderr, "\n%.0f\t%.1f\t%.1f", x[i], y[i],  FitTanhPoint(x[i], a));
  fprintf(stderr, "\n");

  fprintf(stderr, "\nI think the parameters are:\n");
  fprintf(stderr, "\nEquation : %5.1f + %5.1f * tanh(%5.3f * x + %5.1f))"
		,a[1],a[2],a[3],a[4]);
  fprintf(stderr, "\nMax: %5.2f Min: %5.2f Slope %5.2f Dyn. Range: %5.2f From: %5.2f To: %5.2f\n"
		,DRMax, DRMin, DRSlope, DRdr, DRFrom, DRTo);
#endif

  free_matrix(covar,1,ma,1,ma);
  free_matrix(alpha,1,ma,1,ma);
  free_ivector(ia,1,ma);
  free_vector(a,1,ma);
  free_vector(x,1,ma);
  free_vector(y,1,ma);
  free_vector(sig,1,ma);
  fflush(stderr);
  return 1;
}
