/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: mod_rig.c,v 2.66 2000/06/09 22:49:16 cmalek Exp $
** Program: dowl
**  Module: mod_rig
**  Author: mazer
** Descrip: synthetic rig
**
** Revision History (most recent last)
**
** Sat Mar  7 16:10:48 1992 mazer
**  modified is_init for ms_da/ms_ad
**
** Fri May 29 12:38:35 1992 mazer
**  auto shut off added -- this counts number of rig-reps without
**  any changes in the parameters (trip_counts) and when it exceeds
**  rig.auto_shutoff, if shuts off the rig. Set rig.auto_shutoff
**  to a negative value to disable this feature.
**
** Mon Jun  8 17:42:24 1992 mazer
**  moved rig.XXX++ to prefs in xdowl main Prefs worksheet
**  deleted mark stuff, since it's never used, changed help screen
**
** Fri Jan 29 19:14:51 1993 mazer
**  changed the way atten-max keys operate..
**  made things shutdown after halting
**
** Wed Oct 13 16:29:37 1993 mazer
**  renamed from runrig.c to rig.c
**
** Sun Nov 13 16:02:01 1994 mazer
**  added "ana" button for jose..
**
** Tue Nov 15 15:28:14 1994 mazer
**  added calibration field: C=mv/tick and D=mv-offset
**  to XX.rig.ana records..
**
** Fri Nov 18 13:00:40 1994 mazer
**  removed C= and D= fields from analog saves and placed this
**  info in the waveheader via wavio.c and analogfns.c
**
** 97.1 bjarthur
**  fixed latt--, ratt--, latt++, and ratt++ keys 
**  fixed bug in lmono, off, and rmono (ie keypad =,/,*) buttons
**  changed help system to conform to other modules
**  changed so that bw and cf vars were saved as well
**  changes so that last stimulus was default, not BroadbandNoise
**  modified status line to display actual iid and abi, instead of att settings
**
** 97.4 bjarthur
**  changed DOWLDIRNEW to XDPHYSDIR
**
*******************************************************************/

#include "xdphyslib.h"
#include "xdphys.h"

#include <X11/keysym.h>

#define MAXITD    400
#define MAXFREQ	  14000

static Widget rigshell;
static Widget lspl_lab, rspl_lab, iid_lab, abi_lab;
static SliderObj freqbar;
static SliderObj lattbar, rattbar, iidbar, itdbar;
static SliderObj cfbar, bwbar;

static float rig_latt = 20;
static float rig_ratt = 20;

static int lmono = 0;
static int rmono = 0;

static int new_wave = 1;
static int new_att = 1;
static int resynthed = 1;

static char s1_buf[100] = "";
static char s2_buf[100] = "";

static int trip_count = 0;

#if(0)
static char *rig_ana_save_to = NULL;
static char cstim[250];		/* current stimulus for analog label.. */
static char cinten[250];	/* current spl values.. */
#endif

static void HelpCB(Widget, char*, caddr_t);
int SynthStim(int);
static void DrawRig(void);
static void BroadbandNoiseCB(Widget, caddr_t, caddr_t);
static void HighpassNoiseCB(Widget, caddr_t, caddr_t);
static void RigRunner(int);
static void RunStopCB(Widget, caddr_t, caddr_t);
static void OneShotCB(Widget, caddr_t, caddr_t);
static void SliderCallback(Slider);
static void RigControlAction(Widget, XEvent*, String*, Cardinal*);
/*static void RecTogCB(Widget, caddr_t, caddr_t);*/
Widget setupRig(DockSlot*);

syn_spec G_ss;

static void HelpCB(Widget w, char *name, caddr_t call_data)
{
  char tmp[100],*file;

  sprintf(tmp,"$(XDPHYSDIR)/doc/%s",name);
  file = envexpand(tmp);
  if (!probefile(file)) {
    alert("Can't find help file.");
    return; }
  help(file,name);
}

int SynthStim(int force)
{
	int ep, dur, delay, rise, fall;
	int delta, reinit;
	int cf, bw;
	char sspec[100];
	int outsize, da_channels;
#ifdef __tdtproc__
	int use_led;		/* use the optic tectum stimulating LED bank */
	int led_dur;		/* duration for the LED bank stimulus */
	int led_delay;		/* delay for the LED bank stimulus */
#endif				/* __tdtproc__ */
	xword *outbuf;

	static int last_epoch = -1;
	static int last_dur = -1;
	static int last_delay = -1;
	static int last_rise = -1;
	static int last_fall = -1;
	static int f = -1;
	static int itd = -1;
	static int last_cf = -1;
	static int last_bw = -1;

	ep = GI("epoch");
	dur = GI("dur");
	delay = GI("delay");
	rise = GI("rise");
	fall = GI("fall");

#ifdef __tdtproc__
	use_led = GI("use_led");
	led_dur = GI("led_dur");
	led_delay = GI("led_delay");
#endif				/* __tdtproc__ */


	if (is_inited &&
	    (ep != last_epoch || dur != last_dur || delay != last_delay))
		reinit = 1;
	else
		reinit = 0;

	delta = 0;
	if (!is_inited || reinit) {
		is_setNice(GI("nicelevel"));
		is_setSyncPulse(GI("syncPulse"));

#ifdef __tdtproc__
		if (getRaster_getmode() == GSR_ttl) 
			is_et1_enable();
		else
			is_et1_disable();

		if (use_led) {
			is_led_enable();
			is_setLedDur(led_dur);
			is_setLedDelay(led_delay);
		} else
			is_led_disable();
#endif	/* __tdtproc__ */

		is_init(GF("daFc"), 0.0, GF("evtFc"), dur + delay, ep);

		new_wave = 1;
		delta = 1;
	}
	da_channels = is_getDAnchans();
	outbuf = is_getDAbuffer(&outsize);

	if (rise != last_rise || fall != last_fall)
		new_wave = 1;

	f = (int) slider_get(freqbar);
	if (f == 0) {
		cf = (int) slider_get(cfbar);
		bw = (int) slider_get(bwbar);
		if (cf != last_cf || bw != last_bw) {
			new_wave = 1;
		}
	}
	syn_calibrate = 1;

	if (force || new_wave) {
		if (f != 0) {
			fd_syn_spec_parse(NULL, (int) slider_get(freqbar), &G_ss);
			synthesize(&G_ss, NULL, outbuf, outsize, da_channels, delay, dur, ep, itd = (int) slider_get(itdbar), 100, rise, fall, SYN_BOTH);
/*
   sprintf(cstim, "T f=%d del=%d dur=%d itd=%d", f, delay, dur, itd);
 */
		} else {
			int a, b;

			last_cf = cf;
			last_bw = bw;
			sprintf(sspec, "cf=%d,bw=%d",
				a = (int) slider_get(cfbar), b = (int) slider_get(bwbar));
			fd_syn_spec_parse(sspec, 0, &G_ss);
			synthesize(&G_ss, NULL, outbuf, outsize, da_channels, delay, dur, ep, itd = (int) slider_get(itdbar), 100, rise, fall, SYN_BOTH);
/*
   sprintf(cstim, "N cf=%d bw=%d del=%d dur=%d itd=%d",
   a, b, delay, dur, itd);
 */
		}
		if (lmono)
			is_clearOutput(SYN_RIGHT);
		if (rmono)
			is_clearOutput(SYN_LEFT);

		new_wave = 0;
		resynthed = 1;
		if (!progRunning)
			view_output();
		delta = 1;
	}
	if (force || new_att) {
		setRack(0, rig_latt, rig_ratt);
		new_att = 0;
		delta = 1;
	}
	last_epoch = ep;
	last_dur = dur;
	last_delay = delay;
	last_rise = rise;
	last_fall = fall;

	if (delta) {
		char buf[100];
		float lspl, rspl;

		if (!cal_FigureSPL(&G_ss, rig_latt, rig_ratt, &lspl, &rspl))
			notify("Can't compute dB SPL info.");

/* *cinten = 0; */

/* strcat(cinten, "L="); */
		if (rmono)
			sprintf(buf, "---");
		else
			sprintf(buf, "%.0fspl", lspl);
		label_set(lspl_lab, buf);
/* strcat(cinten, buf); */

		if (lmono || rmono)
			sprintf(buf, "---");
		else
			sprintf(buf, "%.0fspl", rspl - lspl);
		label_set(iid_lab, buf);

/*    strcat(cinten, " R="); */
		if (lmono)
			sprintf(buf, "---");
		else
			sprintf(buf, "%.0fspl", rspl);
		label_set(rspl_lab, buf);
/*    strcat(cinten, buf); */

		if (lmono || rmono)
			sprintf(buf, "---");
		else
			sprintf(buf, "%.0fspl", (rspl + lspl) / 2.0);
		label_set(abi_lab, buf);

		if (f == 0)
			sprintf(s1_buf, "NOISE     %dus", itd);
		else
			sprintf(s1_buf, "%5dHz   %dus", f, itd);
		if (lmono && rmono)
			sprintf(s2_buf, "L=OFF     R=OFF");
		else if (lmono)
			sprintf(s2_buf, "L=%3d     R=OFF", (int) (lspl + 0.5));
		else if (rmono)
			sprintf(s2_buf, "L=OFF     R=%3d", (int) (rspl + 0.5));
		else
			sprintf(s2_buf, "IID=%3d,   ABI=%3d", (int) (rspl - lspl + 0.5),
				(int) ((lspl + rspl) / 2.0));
	}
	return (resynthed);
}

static void DrawRig(void)
{
  float iid;

  iid = rig_latt - rig_ratt;

  slider_set_noCB(lattbar, rig_latt);
  slider_set_noCB(rattbar, rig_ratt);
  slider_set_noCB(iidbar, iid);
}

static void BroadbandNoiseCB(Widget w, caddr_t client_data, caddr_t call_data)
{
  int cf, bw;

  cf = (GI("noise.high") + GI("noise.low")) / 2;
  bw = GI("noise.high") - GI("noise.low");
  slider_set(cfbar, (float)cf);
  slider_set(bwbar, (float)bw);
}

static void HighpassNoiseCB(Widget w, caddr_t client_data, caddr_t call_data)
{
  int cf, bw;

  cf = (4000 + 13000) / 2;
  bw = 13000 - 4000;
  slider_set(cfbar, (float)cf);
  slider_set(bwbar, (float)bw);
}

static void RigRunner(int oneshot)
{
  int *spikelist, ep, i;
  float *sumPMk;
  void *isitime;                /* handle for isi_functions() */
  static char s1_save[100] = "";
  static char s2_save[100] = "";
  double timestamp;
  int status;

  set_progRunning(True);
  recording(/*(rig_ana_save_to != NULL) ? 1 :*/ 0);

  isitime = isi_start(0);
  while (progRunning) {
    ep = GI("epoch");
    SynthStim(0);

    cur_depvar = 0;
    if (status1 && (strcmp(s1_save, s1_buf) != 0)) {
      statline_set(status1, s1_buf);
      strcpy(s1_save, s1_buf);
    }
    if (status2 && (strcmp(s2_save, s2_buf) != 0)) {
      statline_set(status2, s2_buf);
      strcpy(s2_save, s2_buf);
    }

	set_led(3,1);

    status = is_sample(X_TRIG | X_EVT | X_DA, ep, &timestamp);
    if (!(single_step(status))) {
      alert("is_sample failed");
      set_progRunning(False);
    } else {
      MaybePause();

	set_led(3,0);

	view_input();
	view_output();

  spikelist = getRaster(ep, &sumPMk);
	update_displays(spikelist,1);
	if(spikelist != NULL) free(spikelist);

#if(0)
      if (rig_ana_save_to != NULL) {
	char *rigstr = (char *)malloc(strlen(cstim) + strlen(cinten) + 25);
	sprintf(rigstr, "V2 %s %s", cstim, cinten);
	appendAnalog(rig_ana_save_to, rigstr, NULL, 0);
	free(rigstr);
      }
#endif
    }

    isi_stop(isitime);
    isitime = isi_start(GI("isi"));

    dloop_empty();

    if ((++trip_count % 100) == 0) {
      beep(-25);
      beep(-50);
    }

    if ((i = GI("rig.auto_shutoff")) >= 0) {
			percdone((float)trip_count / ((i==0) ? 500.0 : (float)i) );
      if ((i == 0 && trip_count > 500) || (i > 0 && trip_count > i)) {
	set_progRunning(False);
        memset((void *) s1_save, 0, 100);
        memset((void *) s2_save, 0, 100);
	notify("Auto Shutoff after %d reps", trip_count);
      }
    }
    if (oneshot) {
      set_progRunning(False);
      memset((void *) s1_save, 0, 100);
      memset((void *) s2_save, 0, 100);
    }
  }
}

static void RunStopCB(Widget w, caddr_t client_data, caddr_t call_data)
{
  trip_count = 0;
  if (progRunning)
    set_progRunning(False);
  else {
    if (load_calib_data(GS("cal_file"), CAL_RELOAD_NO)) {
      new_wave = new_att = 1;
      RigRunner(0);
      is_shutdown(); } }
}

static void OneShotCB(Widget w, caddr_t client_data, caddr_t call_data)
{
  if (progRunning)
    beep(1);
  else {
    if (load_calib_data(GS("cal_file"), CAL_RELOAD_NO)) {
      trip_count = 0;
      new_wave = new_att = 1;
      RigRunner(1);
      is_shutdown(); } }
}

static void SliderCallback(Slider s)
{
  float iid, newiid;
  float latt, ratt;

  trip_count = 0;

  if (s == freqbar) {
    new_wave = 1;
    DrawRig();
  } else if (s == itdbar) {
    new_wave = 1;
    DrawRig();
  } else if (s == lattbar) {
    rig_latt = slider_get(s);
    new_att = 1;
    DrawRig();
  } else if (s == rattbar) {
    rig_ratt = slider_get(s);
    new_att = 1;
    DrawRig();
  } else if (s == iidbar) {
    iid = rig_latt - rig_ratt;
    newiid = slider_get(s);
    latt = RND2INT(rig_latt + (newiid - iid) / 2.0);
    ratt = RND2INT(rig_ratt - (newiid - iid) / 2.0);
    if ((latt < 0.0) || (ratt < 0.0) ||
          (latt > attMaxAtten) || (ratt > attMaxAtten)) {
      slider_set_noCB(s, iid);
    } else {
      rig_latt = latt;
      rig_ratt = ratt;
      new_att = 1;
      DrawRig();
    }
  }
}

static void RigControlAction(Widget w_blah, XEvent *event, String *params,
      Cardinal *num_params)
{
/* WaveWidget w = (WaveWidget)w_blah; */
  float step;
  int redraw = 0;
  int rig_freq, rig_itd, _latt, _ratt, _itd, _freq;

  trip_count = 0;

  _latt = rig_latt;
  _ratt = rig_ratt;
  _itd = rig_itd = (int)slider_get(itdbar);
  _freq = rig_freq = (int)slider_get(freqbar);

  if (event->type == KeyPress) {
    KeySym ks;
    char str[256+1];

    XLookupString((XKeyEvent*)event, str, 256, &ks, NULL);
    switch(ks)
      {
	/* ON/OFF */
#ifdef __linux__
      case XK_KP_Insert:
      case XK_KP_0:
#else /* __linux__ */ 
      case XK_Insert:
#endif /* __linux__ */ 
	RunStopCB(NULL, NULL, NULL);
	return;

	/* CHANGE FREQUENCY */
      case XK_KP_Enter:
	{
	    _freq = 0;
#if(0)

  /* 97.4 bjarthur:  changed FigureSPL to work with syn_spec.  to make
     the FigureSPL calls work here, would require resynthesizing the
     stimulus inside this function, and I'm too lazy to figure out how
     to do it.  so for now, changing the frequency will also change the
     intensity (b/c the speaker's spectrum is not flat) */

	  if (cal_FigureSPL(&G_ss, rig_latt, rig_ratt, &lspl, &rspl)) {
	    pre_abi = (lspl + rspl) / 2.0;
	    _freq = 0;
	    cal_FigureSPL(&G_ss, rig_latt, rig_ratt, &lspl, &rspl);
	    post_abi = (lspl + rspl) / 2.0;
	    lnew = RND2INT(_latt - (pre_abi - post_abi));
	    rnew = RND2INT(_ratt - (pre_abi - post_abi));
	    if (lnew >= 0 && lnew <= attMaxAtten &&
		rnew >= 0 && rnew <= attMaxAtten) {
	      _latt = lnew;
	      _ratt = rnew;
	    } else {
	      beep(0);
	    }
	  }
#endif
	}
	break;
      case XK_KP_Subtract:
	{
	    if (_freq < MAXFREQ)
	      _freq += GI("rig.freq++");
#if(0)
	  if (cal_FigureSPL(&G_ss, rig_latt, rig_ratt, &lspl, &rspl)) {
	    pre_abi = (lspl + rspl) / 2.0;
	    if (_freq < MAXFREQ) {
	      _freq += GI("rig.freq++");
	      cal_FigureSPL(&G_ss, rig_latt, rig_ratt, &lspl, &rspl);
	      post_abi = (lspl + rspl) / 2.0;
	      lnew = RND2INT(_latt - (pre_abi - post_abi));
	      rnew = RND2INT(_ratt- (pre_abi - post_abi));
	      if (lnew >= 0 && lnew <= attMaxAtten &&
		  rnew >= 0 && rnew <= attMaxAtten) {
		_latt = lnew;
		_ratt = rnew;
	      } else {
		beep(0);
	      }
	    }
	  } else {
	    if (_freq < MAXFREQ)
	      _freq += GI("rig.freq++");
	  }
#endif
	}
	break;
      case XK_KP_Add:
	{
	    if (_freq > 0)
	      _freq -= GI("rig.freq++");
#if(0)
	  if (cal_FigureSPL(&G_ss, rig_latt, rig_ratt, &lspl, &rspl)) {
	    pre_abi = (lspl + rspl) / 2.0;
	    if (_freq > 0) {
	      _freq -= GI("rig.freq++");
	      cal_FigureSPL(&G_ss, rig_latt, rig_ratt, &lspl, &rspl);
	      post_abi = (lspl + rspl) / 2.0;
	      lnew = RND2INT(_latt - (pre_abi - post_abi));
	      rnew = RND2INT(_ratt- (pre_abi - post_abi));
	      if (lnew >= 0 && lnew <= attMaxAtten &&
		  rnew >= 0 && rnew <= attMaxAtten) {
		_latt = lnew;
		_ratt = rnew;
	      } else {
		beep(0);
	      }
	    }
	  } else {
	    if (_freq > 0)
	      _freq -= GI("rig.freq++");
	  }
#endif
	}
	break;
	
	/* RESET/CENTER JUST ITD */
#ifdef __linux__
      case XK_KP_5:		/* keypad-5 */
#else /* __linux__ */ 
      case XK_F31:		/* keypad-5 */
#endif /* __linux__ */ 
	_itd = 0;
	break;

#ifndef __linux__
	/* FLIP ITD */
      case XK_KP_5:		/* keypad-5 */
	_itd = -_itd;
	break;
#endif /* __linux__ */ 

	/* RESET/CENTER MOST EVERYTHING */
#ifdef __linux__
      case XK_KP_Decimal:		/* keypad-period */
      case XK_KP_Delete:		/* keypad-period */
#else /* __linux__ */ 
      case XK_Delete:		/* keypad-period */
#endif /* __linux__ */ 
	_latt = _ratt = RND2INT((rig_latt + rig_ratt) / 2.0);
	break;
	
	/* CHANGE ITD */
#ifdef __linux__
      case XK_KP_4:
      case XK_KP_6:
	if (ks == XK_KP_4)
#else /* __linux__ */ 
      case XK_Left:
      case XK_Right:
	if (ks == XK_Left)
#endif /* __linux__ */ 
	  step = -GF("rig.itd++");
	else
	  step = GF("rig.itd++");
	if ((_itd += step) < -MAXITD)
	  _itd = -MAXITD;
	else if (_itd > MAXITD)
	  _itd = MAXITD;
	break;

	/* CHANGE IID monaurally */
#ifdef __linux__
	  case XK_KP_7:
#else /* __linux__ */ 
      case XK_R7:  /* the "Home" key on keypad */
#endif /* __linux__ */ 
	if (event->xkey.state != 0) {
	  _latt = attMaxAtten;
	  break;
	}
#ifdef __linux__
      case XK_KP_1: /* the "End" key on the keypad */
	if (ks == XK_KP_7)
#else /* __linux__ */ 
      case XK_R13: /* the "End" key on the keypad */
	if (ks == XK_F27)
#endif /* __linux__ */ 
	  step = RND2INT(GF("rig.iid++") / 2.0);
	else
	  step = -RND2INT(GF("rig.iid++") / 2.0);
	if (_latt + step <= attMaxAtten && _latt + step >= 0)
	  _latt += step;
	else
	  beep(0);
	break;
#ifdef __linux__
      case XK_KP_9: /* the "PgUp" key on the keypad */
#else /* __linux__ */ 
      case XK_R9: /* the "PgUp" key on the keypad */
#endif /* __linux__ */ 
	if (event->xkey.state != 0) {
	  _ratt = attMaxAtten;
	  break;
	}
#ifdef __linux__
      case XK_KP_3:  /* the "PgDn" key on the keypad */
	if (ks == XK_KP_9)
#else /* __linux__ */ 
      case XK_R15:  /* the "PgDn" key on the keypad */
	if (ks == XK_F29)
#endif /* __linux__ */ 
	  step = RND2INT(GF("rig.iid++") / 2.0);
	else
	  step = -RND2INT(GF("rig.iid++") / 2.0);
	if (_ratt + step <= attMaxAtten && _ratt + step >= 0)
	  _ratt += step;
	else
	  beep(0);
	break;
#ifdef __linux__
      case XK_Num_Lock:		/* the "=" key.  on/off left */
#else /* __linux__ */ 
      case XK_R4:		/* the "=" key.  on/off left */
#endif /* __linux__ */ 
	rmono = !rmono;
	ReverseColors(rspl_lab);
	redraw = new_wave = 1;
	beep(0);
	break;
#ifdef __linux__
      case XK_KP_Multiply:	/* the "*" key.  on/off right */
#else /* __linux__ */ 
      case XK_R6:	/* the "*" key.  on/off right */
#endif /* __linux__ */ 
	lmono = !lmono;
	ReverseColors(lspl_lab);
	redraw = new_wave = 1;
	beep(0);
	break;
#ifdef __linux__
      case XK_KP_Divide:	/* the "/" key.  on/off left and right */
#else /* __linux__ */ 
      case XK_R5:	/* the "/" key.  on/off left and right */
#endif /* __linux__ */ 
	if ((!lmono && !rmono) || (lmono && rmono)) {
	  lmono = !lmono;
	  rmono = !rmono;
	  redraw = new_wave = 1;
	  ReverseColors(lspl_lab);
	  ReverseColors(rspl_lab);
	  beep(0);
	}
	break;

	/* CHANGE IID binaurally */
#ifdef __linux__
      case XK_KP_2:
      case XK_KP_8:
	if (ks == XK_KP_2)
#else /* __linux__ */ 
      case XK_Down:
      case XK_Up:
	if (ks == XK_Down)
#endif /* __linux__ */ 
	  step = GF("rig.iid++") / 2.0;
	else
	  step = -GF("rig.iid++") / 2.0;
	_latt -= RND2INT(step);
	_ratt += RND2INT(step);
	if (_latt < 0 || _ratt < 0 ||
	    _latt > attMaxAtten || _ratt > attMaxAtten) {
	  beep(0);
	  _latt = rig_latt;
	  _ratt = rig_ratt;
	}
	break;

	/* CHANGE ABI binaurally */
      case XK_greater:
      case XK_period:
      case XK_less:
      case XK_comma:
	if (ks == XK_greater || ks == XK_period)
	  step = GF("rig.iid++") / 2.0;
	else
	  step = -GF("rig.iid++") / 2.0;
	_latt -= RND2INT(step);
	_ratt -= RND2INT(step);
	if (_latt < 0 || _ratt < 0 ||
	    _latt > attMaxAtten || _ratt > attMaxAtten) {
	  beep(0);
	  _latt = rig_latt;
	  _ratt = rig_ratt;
	}
	break;

	/* OOPS... */
      default:
	return;
      }
  }
  if (rig_itd != _itd) {
    slider_set(itdbar, (float)_itd);
    redraw = new_wave = 1;
  }
  if (rig_latt != _latt || rig_ratt != _ratt) {
    rig_latt = _latt;
    rig_ratt = _ratt;
    redraw = new_att = 1;
  }
  if (rig_freq != _freq) {
    redraw = new_wave = 1;
    slider_set(freqbar, (float)_freq);	/* this will cause resynth! */
  }
  if (redraw)
    DrawRig();
}

#if(0)
static void RecTogCB(Widget w, caddr_t call_data, caddr_t client_data)
{
  int state;
  char buf[45], *f;

  if ((state = toggle_get(w)) == True) {
    if (progRunning) {
      /* don't do this while running -- it's cause a timeout!! */
      toggle_set(w, False);
      beep(0);
    } else {
      sprintf(buf, "%s.%s.rig.ana", GS("animal"), GS("unit"));
      if ((f = fileBox(buf, "a", "Save/Append to..")) == NULL) {
	toggle_set(w, False);
	rig_ana_save_to = NULL;
      } else {
	rig_ana_save_to = f;
      }
    }
  } else {
    free(rig_ana_save_to);
    rig_ana_save_to = NULL;
  }
  if (progRunning)
    recording((rig_ana_save_to != NULL) ? 1 : 0);
}
#endif

Widget setupRig(DockSlot *slot)
{
  int rig_itd, rig_freq, cf, bw;
  Widget form, w, rs;
  static XtActionsRec rigformActions[] = { 
    { "rig-control-action",	RigControlAction },
  };
  static char *overrideTranslations = 
    "<Key>: rig-control-action()\n";

  rig_latt = GI("rig.L");
  if (LastVarUndefined)
    rig_latt = 20;
  rig_ratt = GI("rig.R");
  if (LastVarUndefined)
    rig_ratt = 20;
  rig_itd = GI("rig.ITD");
  if (LastVarUndefined)
    rig_itd = 0;
  rig_freq = GI("rig.Freq");
  if (LastVarUndefined)
    rig_freq = 0;

  cf = GI("rig.cf");
  if (LastVarUndefined)
    cf = (13100 + 100) / 2;
  bw = GI("rig.bw");
  if (LastVarUndefined)
    bw = (13100 - 100);

  rigshell = pop_new(TopLevel, "rig");
  
  form = XtVaCreateManagedWidget("form", formWidgetClass, rigshell, NULL);

  rs = button_new(form, "Run/Stop", "Run/Stop", (XtCallbackProc) RunStopCB, NULL, NULL, NULL);
  XtSetKeyboardFocus(rs, form);
  w = button_new(form, "One-Shot", "One-Shot", (XtCallbackProc) OneShotCB, NULL, rs, NULL);
  XtSetKeyboardFocus(w, form);
  w = button_new(form, "Help", "Help", (XtCallbackProc) HelpCB, "xdphys.rig.doc", w, NULL);
  XtSetKeyboardFocus(w, form);
  w = button_new(form, "->BB", "->BB", (XtCallbackProc) BroadbandNoiseCB, NULL, w, NULL);
  XtSetKeyboardFocus(w, form);
  w = button_new(form, "->HP", "->HP", (XtCallbackProc) HighpassNoiseCB, NULL, w, NULL);
  XtSetKeyboardFocus(w, form);

/*
  w = toggle_new(form, "ana", "ana", w, NULL, NULL, (XtCallbackProc) RecTogCB, NULL, False);
*/
  XtSetKeyboardFocus(w, form);

  lspl_lab = label_new(form, "lspl", NULL, rs, 1);
  XtSetKeyboardFocus(lspl_lab, form);
  label_set(lspl_lab, "---spl");

  lattbar = slider_new(form, "rig.L", 0.0, attMaxAtten, 1.0, (float)rig_latt,
		       SL_VERT, SL_FILL, SL_MOD, 150);
  slider_setCB(lattbar, SliderCallback);
  XtVaSetValues(slider_widget(lattbar),
		XtNfromVert, lspl_lab,
		XtNfromHoriz, NULL,
		NULL);
  XtSetKeyboardFocus(slider_widget(lattbar), form);
  
  iid_lab = label_new(form, "iid", slider_widget(lattbar), rs, 1);
  XtSetKeyboardFocus(iid_lab, form);
  label_set(iid_lab, "---spl");

  iidbar = slider_new(form, "rig.IID", -attMaxAtten, attMaxAtten, 1.0,
		      (float)(rig_latt-rig_ratt), SL_VERT,SL_TICK, SL_MOD, 150);
  slider_setCB(iidbar, SliderCallback);
  XtVaSetValues(slider_widget(iidbar),
		XtNfromVert, iid_lab,
		XtNfromHoriz, slider_widget(lattbar), NULL);
  XtSetKeyboardFocus(slider_widget(iidbar), form);

  rspl_lab = label_new(form, "rspl", slider_widget(iidbar), rs, 1);
  XtSetKeyboardFocus(rspl_lab, form);
  label_set(rspl_lab, "---spl");

  rattbar = slider_new(form, "rig.R", 0.0, attMaxAtten, 1.0, (float)rig_ratt,
		       SL_VERT, SL_FILL, SL_MOD, 150);
  slider_setCB(rattbar, SliderCallback);
  XtVaSetValues(slider_widget(rattbar),
		XtNfromVert, rspl_lab,
		XtNfromHoriz, slider_widget(iidbar), NULL);
  XtSetKeyboardFocus(slider_widget(rattbar), form);

  itdbar = slider_new(form, "rig.ITD", (float)(-MAXITD), (float)MAXITD, 1.0,
                (float)rig_itd, SL_HORIZ, SL_TICK, SL_MOD, 230);
  slider_setCB(itdbar, SliderCallback);
  XtVaSetValues(slider_widget(itdbar),
		XtNfromVert, slider_widget(lattbar),
		XtNfromHoriz, NULL, NULL);
  XtSetKeyboardFocus(slider_widget(itdbar), form);

  abi_lab = label_new(form, "abi", slider_widget(rattbar), rs, 1);
  XtSetKeyboardFocus(abi_lab, form);
  label_set(abi_lab, "---spl");

  freqbar = slider_new(form, "rig.Freq", 0.0, (float)MAXFREQ, 100.0,
                (float)rig_freq, SL_VERT, SL_TICK, SL_MOD, 150);
  slider_setCB(freqbar, SliderCallback);
  XtVaSetValues(slider_widget(freqbar),
		XtNfromVert, abi_lab,
		XtNfromHoriz, slider_widget(rattbar), NULL);
  XtSetKeyboardFocus(slider_widget(freqbar), form);

  cfbar = slider_new(form, "rig.cf", 0.0, (float)MAXFREQ, 100.0, (float)cf,
		     SL_VERT, SL_TICK, SL_MOD, 150);
  XtVaSetValues(slider_widget(cfbar),
		XtNfromVert, abi_lab,
		XtNfromHoriz, slider_widget(freqbar),
		XtNhorizDistance, 10, NULL);
  XtSetKeyboardFocus(slider_widget(cfbar), form);
  
  bwbar = slider_new(form, "rig.bw", 100.0, 14000.0, 100.0, (float)bw,
		     SL_VERT, SL_TICK, SL_MOD, 150);
  XtVaSetValues(slider_widget(bwbar),
		XtNfromVert, abi_lab,
		XtNfromHoriz, slider_widget(cfbar), NULL);
  XtSetKeyboardFocus(slider_widget(bwbar), form);

  DrawRig();

  XtAppAddActions(XtWidgetToApplicationContext(TopLevel),
		  rigformActions, XtNumber(rigformActions));
  XtAugmentTranslations(form, XtParseTranslationTable(overrideTranslations));
  XtSetKeyboardFocus(form, form);
  
/*
  BroadbandNoiseCB(NULL, NULL, NULL);
*/

  return(rigshell);
}
