/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: mod_int.c,v 1.15 2000/11/28 06:28:47 cmalek Exp $
** Program: dowl
**  Module: xbkint.c
**  Author: mazer/bjarthur
** Descrip: calculate freq dependent open-circuit sensistivity
**          for new microphones
**
** Revision History (most recent last)
**  
** July 1996 bjarthur
**   dup'd from xbk.c and added stuff to scan intensity as well
**   as freq.  perhaps it's not so wise to assume freq resp of
**   spkr doesn't change w/ intensity.
**
**   use S+ script xbkint.s to view data.  delete comments first
**
** NOV96 bjarthur
**   copied from xbkint.c
**
** 97.1 bjarthur
**   changed so that mean,stddev was calculated on dB scale.
**
** 97.4 bjarthur
**   changes stddev to stderror
**
*******************************************************************/

#include "xdphyslib.h"
#include "xcalibur.h"

static Field ws_table[] = {
  { "int.freq (Hz)",		"%d",	10, ""			},
  { "int.att_range (dB)",	"%s",	30, "30:80:10"		},
  { NULL }
};

static int runScan(FILE*, Widget);
Widget setupInt(DockSlot *slot);

static int runScan(FILE *outfile, Widget headline)
{
  int 	freq, att;		/* current freq in Hz */
  char *range_att;		/* range of intensities */
  int  *stimarray_att, nstims_att;    /* stimulus array derived from range */
  int 	reps;			/* reps to take at each freq */
  int   msdur, tdur;            /* sample parameters */
  int   msrise, trise;
  int   msfall, tfall;
  int   msdelay, tdelay;
  int   msepoch, tepoch;
  xword *ibuf;			/* analog input buffer */
  int 	count;			/* num of input samples; measure over [a,b] */
  int   ref_off;
  float gain;			/* chan a/b gains as mult. factor */
  float inv;   /* chan a/b invert flag */
  float *mags;			/* measured l/r mv-rms amplitudes */
  float *phis;			/* measured l/r phase values */
  float *dists;			/* measured l/r harmonic distortion */
  float	mag,mag_sd;		/* mean values for amplitudes -- in mv-rms */
  float phi,phi_sd;
  float dist;			/* mean harmonic distortions */
  float So,fooL,fooR;			/* B&K open circuit sens'y (OCS) in mv/Pa */
  int daFc, adFc;
  float adj_mag, adj_phi;
  syn_spec ss;
  int 	i,j;
  void *isitime;    /* handle for isi_functions() */
  int isi;      /* inter-stimulus interval */
  char *spk_chan, *ref_chan;
  double timestamp;
  int status;
  int outsize, da_channels;
  xword *outbuf;

  /* calculate parameters */
  daFc = GI("daFc");
  adFc = GI("adFc");

  msdur = GI("Dur");     tdur = (int) ((float) msdur*((float) daFc)/1000.0);
  msrise = GI("Rise");   trise = (int) ((float) msrise*((float) daFc)/1000.0);
  msfall = GI("Fall");   tfall = (int) ((float) msfall*((float) daFc)/1000.0);
  msdelay = GI("Delay"); tdelay = (int) ((float) msdelay*((float) daFc)/1000.0);
  msepoch = GI("Epoch"); tepoch = (int) ((float) msepoch*((float) daFc)/1000.0);

  freq = GI("int.freq");
  range_att  = GS("int.att_range");
  reps   = GI("Reps");
  isi    = GI("isi");

  spk_chan = GS("Speaker_Channel");
  ref_chan = GS("Reference_Channel");

  if((*ref_chan)=='L') {
    ref_off = 0;
    So  = GF("So_Left");
    gain = pow(10.0, GF("Gain_Left") / 20.0);
		inv = GI("Invert_Left"); }
  else {
    ref_off = 1;
    So  = GF("So_Right");
    gain = pow(10.0, GF("Gain_Right") / 20.0);
		inv = GI("Invert_Right"); }

  ws_lock(ws_table, False);
  unlock_worksheets();
  /* no svars should be accessed beyond this point */


	fooL = fooR = So;
  if(!load_mic_adjust_data(0,&fooL,&fooR)) return(0);
  if((((*ref_chan)=='L') && (fooL!=So)) || (((*ref_chan)=='R') && (fooR!=So))) {
    alert("So[L,R] in Prefs do not match those in .fr files\nspecified in Mic_Adjust[Left,Right].");
    return(0); }

  if((msdelay+msdur>msepoch)||(msrise+msfall>msdur)) {
    alert("Something wrong with either Dur, Delay, Epoch, Rise or Fall.");
    return(0); }

  if((*spk_chan!='L') && (*spk_chan!='R')) {
    alert("Speaker_Channel must either be 'L' or 'R'.");
    return(0); }

  if((*ref_chan!='L') && (*ref_chan!='R')) {
    alert("Reference_Channel must either be 'L' or 'R'.");
    return(0); }

  if(So==0.0) {
    alert("Reference_Channel has So of 0.0.");
    return(0); }

  if((stimarray_att = StimArray_Gen(&nstims_att, 0, range_att,
        1, 1, 0))==NULL) return(0);

  syn_calibrate = 0;
 
  mags = floatVec(reps);
  dists = floatVec(reps);
  phis = floatVec(reps);

  is_setNice(GI("nicelevel"));
  is_setSyncPulse(GI("syncPulse"));
  is_init(daFc, adFc, 0.0, msepoch, msepoch);

  da_channels = is_getDAnchans();
  outbuf = is_getDAbuffer(&outsize);
 
  set_progRunning(True);

  statline_set(headline, "Running int...");

  fprintf(outfile, ";  1: att (dB)\n");
  fprintf(outfile, ";  2: mag (dB)\n");
  fprintf(outfile, ";  3: phase (rad)\n");
  fprintf(outfile, ";  4: dist (100*f2_mag/f1_mag)\n");
  fprintf(outfile, ";  5: mag_stderror (dB)\n");
  fprintf(outfile, ";  6: phase_stderror (rad)\n");
  fprintf(outfile, "nrasters=%d\n", nstims_att);

  isitime = isi_start(0);

  for (j=0; progRunning && (j<nstims_att); j++) {
    percdone((float)j/(float)nstims_att);
    att=stimarray_att[j];
    for (i = 0; i < reps; i++) {
      mags[i] = 0.0;
      dists[i] = 0.0;
      phis[i] = 0.0; }
    is_clearOutput(SYN_BOTH);

    fd_syn_spec_parse(NULL,freq,&ss);
    if(*spk_chan=='L') {
      synthesize(&ss, NULL, outbuf, outsize, da_channels, msdelay, msdur, msepoch, 0, 100, msrise, msfall, SYN_LEFT);
      is_clearOutput(SYN_RIGHT);
      setRack(0, att, attMaxAtten); }
    else if(*spk_chan=='R') {
      synthesize(&ss, NULL, outbuf, outsize, da_channels, msdelay, msdur, msepoch, 0, 100, msrise, msfall, SYN_RIGHT);
      is_clearOutput(SYN_LEFT);
      setRack(0, attMaxAtten, att); }

    for (i = 0; progRunning && i < reps; i++) {
	set_led(3,1);

    status = is_sample(X_TRIG | X_EVT | X_DA, msepoch, &timestamp);
    if (!(single_step(status))) {
        alert("is_sample failed");
        set_progRunning(False); }

	set_led(3,0);

	view_input();
	view_output();

      ibuf = is_getADbuffer(&count);
      isi_stop(isitime);
      isitime = isi_start(isi);
      dloop_empty();

      plottrace();

      if (clipCheck(ibuf+ref_off, 0, tepoch, 0.99)) {
        set_progRunning(False);
        alert("Clip on Reference Input");
        break; }
      
      dsp_fitsin(ibuf+ref_off + 2*(tdelay+trise), tdur-trise-tfall, 2,
            is_adFc, 1.0*(float)freq, &mags[i], NULL);
      dsp_fitsin(ibuf+ref_off + 2*tdelay, tdur, 2,
            is_adFc, 1.0*(float)freq, NULL, &phis[i]);
      dsp_fitsin(ibuf+ref_off + 2*(tdelay+trise), tdur-trise-tfall, 2,
            is_adFc, 2.0*(float)freq, &dists[i], NULL);

      dists[i] /= mags[i];                    /* A(F1)/A(F0) */

      if(*ref_chan=='L') {
        if (!ad_to_mv(mags[i], mags+i, NULL, AddaRelative)) {
          alert("Can't convert adticks-rms to mv-rms");
          set_progRunning(False);
          break; } }
      else if(*ref_chan=='R') {
        if (!ad_to_mv(mags[i], NULL, mags+i, AddaRelative)) {
          alert("Can't convert adticks-rms to mv-rms");
          set_progRunning(False);
          break; } }

      mags[i] /= gain;
      query_mic_adjust((float)freq,*ref_chan,&adj_mag,&adj_phi);
      mags[i] /= adj_mag;
      phis[i] -= adj_phi;
      if(inv) phis[i]+=M_PI;

      mags[i] /= ROOT2;	                      /* Amp to RMS */
      mags[i] = 20.0*log10(mags[i]/So)+94.0;  /* to dB */      }

    unwrap(phis,reps,M_PI);
    mag  = mean(mags, reps);   mag_sd = stderror(mags, reps);
    phi  = mean(phis, reps);   phi_sd = stderror(phis, reps);
    dist = 100.0*mean(dists, reps);

    statline_set(status1, "%.1ddB att,  %.1fdB SPL,  %.1f%%hd",
          att, mag, dist);
    statline_set(status2, "");

    fprintf(outfile, "%2d\t%e\t%e\t%e\t%e\t%e\n",
          att, mag, phi, dist, mag_sd, phi_sd);

    fflush(outfile);
    dloop_empty();
    MaybePause(); }

  free(mags);
  free(phis);
  free(dists);

  is_shutdown();

  if (progRunning) {
    statline_set(headline, "Done");
    set_progRunning(False);
    return(1); }
   else {
    statline_set(headline, "Halted");
    return(0); }
}

Widget setupInt(DockSlot *slot)
{
  Widget ws, h;

  ws = pop_ws_new(TopLevel, ws_table, "int", &h, NULL, NULL);
  make_start_temp(h,
        make_module_info("int", "int", "int", NULL, runScan,
              NULL, default_display_fn, ws_table, WS_N(ws_table)),
        NULL, NULL, slot);
  return(ws);
}
