/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: mod_cal.c,v 1.17 2001/04/24 04:09:37 cmalek Exp $
** Program: dowl
**  Module: mod_cal.c
**  Author: mazer/bjarthur
** Descrip: session calibration program -- measures full range
**          of converters in dbspl
**
** Revision History (most recent last)
**  
** Thu Feb 27 01:38:50 1992 mazer
**  creation date xdbspl :: derrived from xocs.c
**
** Fri May 29 16:54:46 1992 mazer
**  create date xdbspl2:: derrived from xdbspl.c
**  - measures l,r,l,r instead of lr,lr,lr and computes leakage
**    values and ambient sound levels
**  - tries to increase or decrease attenuation to avoid clipping
**    at measuring at the noise floor.
**
** Tue Jun 16 15:42:37 1992 mazer
**  changed calls to pow(10,x) to db_to_gain() calls .. fixes
**  off by 10db problem..
**
** Thu Sep 16 19:56:30 1993 mazer
**  + minor cleanup + added tdelay/delay to avoid problems with
**    onset clicks -- dunno where this is comming from..
**  + also changed phase measurement to start at beginning of
**    waveform
**
** NOV96 bjarthur
**   copied from xdbspl2.c
**
** 97.1 bjarthur
**   changed so that mean,stddev was calculated on dB scale.
**
** 97.4 bjarthur
**   changes stddev to stderror
**
*******************************************************************/

#include "xdphyslib.h"
#include "xcalibur.h"

static int runScan(FILE*, Widget);
Widget setupCal(DockSlot *slot);

static Field ws_table[] = {
  { "cal.range (Hz)",		"%s",	30, "1000:13000:500"	},
  { "cal.mono (L,B,R)",		"%s",	1,  "B"		},
  { WS_VSEP },
  { "cal.max_spl (dB)",		"%d",	3,  "60"		},
  { "cal.min_spl (dB)",		"%d",	3,  "50"		},
  { "cal.auto_step (dB)",	"%d",	1,  "2"		},
  { "cal.atten (dB)",		"%d",	2,  "40"		},
  { WS_VSEP },
  { "cal.check (0,PT,TS,WN)",	"%s",	2,  "0"	},
  { "cal.check_file",		"%s",	30, ""	},
  { NULL }
};

static int runScan(FILE *outfile, Widget headline)
{
  int 	freq;			/* current freq in Hz */
  char *range;			/* range of frequences */
  int  *stimarray, nstims;      /* stimulus array derived from range */
  int 	reps;			/* reps to take at each freq */
  int 	msdur, tdur;		/* sample parameters */
  int   msrise, trise;
  int   msfall, tfall;
  int   msdelay, tdelay;
  int   msepoch, tepoch;
  xword *ibuf;			/* analog input buffer */
  int 	count;			/* num of input samples; measure over [a,b] */
  float latten, ratten;		/* settings for digital attenuators */
  float lgain, rgain;		/* chan a/b gains as mult. factor */
  float linv, rinv;   /* chan a/b invert flag */
  float *lmags,  *rmags;	/* measured mv-rms amplitudes */
  float *lphis,  *rphis;	/* measured phases */
  float *ldists, *rdists;	/* measured harmonic distortion */
  float *llkmags, *rlkmags;	/* measured leak mv-rms amps */
  float *llkphis,  *rlkphis;	/* measured leak phases */
  float *llkdists, *rlkdists;	/* measured leak harmonic distortion */
  float	lmag, rmag;		/* mean values for amplitudes */
  float lphi, rphi;		/* mean phase values */
  float ldist, rdist;		/* mean harmonic distortions */
  float	llkmag, rlkmag;		/* mean values for amplitudes */
  float llkphi, rlkphi;		/* mean phase values */
  float llkdist, rlkdist;	/* mean harmonic distortions */

  /* a note about leak conventions: lleaks,lleak are measured values in 
     the left ear when a sound is played in the right ear.  ldbleak is
     the amount the sound is attenuated as it travels from right to left */

  float	lmagsdv, rmagsdv;	/* standard deviations */
  float	lphisdv, rphisdv;
  float max_spl, min_spl;	/* allowable range of spl values */
  int   clip_retry;		/* retry due to clipping error */
  int   spl_retry_l,spl_retry_r;/* retry due to l or r max/min spl exceeded */
  int   spl_retry_count;	/* number of retries on this freq */
  float astep;			/* auto step size (dB) */
  int 	i,j;
  float l_calib, r_calib;
  float min_r_dbspl, min_l_dbspl, min_dbspl;
  syn_spec ss;
  int min_bw;
  int daFc,adFc;
  float SoL, SoR;               /* B&K open circuit sens'y (OCS) in mv/Pa */
  float adj_mag, adj_phi;
  char *check, *check_file, *mono;
  void *isitime;    /* handle for isi_functions() */
  int isi;      /* inter-stimulus interval */
  double timestamp;
  int status;
  int outsize, da_channels;
  xword *outbuf;

  /* calculate parameters */
  ws_lock(ws_table, True);

  check=GS("cal.check");
  check_file=GS("cal.check_file");
  mono=GS("cal.mono");

  daFc = GI("daFc");
  adFc = GI("adFc");

  msdur = GI("Dur");     tdur = (int) ((float) msdur*((float) daFc)/1000.0);
  msrise = GI("Rise");   trise = (int) ((float) msrise*((float) daFc)/1000.0);
  msfall = GI("Fall");   tfall = (int) ((float) msfall*((float) daFc)/1000.0);
  msdelay = GI("Delay"); tdelay = (int) ((float) msdelay*((float) daFc)/1000.0);
  msepoch = GI("Epoch"); tepoch = (int) ((float) msepoch*((float) daFc)/1000.0);

  range = GS("cal.range");
  reps   = GI("Reps");
  max_spl = (float)GI("cal.max_spl");
  min_spl = (float)GI("cal.min_spl");
  astep   = (float)GI("cal.auto_step");
  isi    = GI("isi");

  lgain = pow(10.0, GF("Gain_Left") / 20.0);
  rgain = pow(10.0, GF("Gain_Right") / 20.0);
  linv = GI("Invert_Left");
  rinv = GI("Invert_Right");

  latten = ratten = GF("cal.atten");
  if((*mono)=='L') ratten=0.0;
  if((*mono)=='R') latten=0.0;

  ws_lock(ws_table, False);
  unlock_worksheets();
  /* no svars should be accessed beyond this point */


  if(!load_mic_adjust_data(1,&SoL,&SoR)) return(0);

  if(strcmp(check,"0") && strcmp(check,"PT") && 
        strcmp(check,"TS") && strcmp(check,"WN")) {
    alert("check parameter must be one of 0 (zero), PT, TS, or WN.");
    return(0); }

  if(!strcmp(check,"PT") || !strcmp(check,"TS") || !strcmp(check,"WN")) {
    if (!load_calib_data(check_file, CAL_RELOAD_YES)) return(0);
    syn_calibrate = 1; }
  else
    syn_calibrate = 0;

  if(strcmp(mono,"L") && strcmp(mono,"R") && strcmp(mono,"B")) {
    alert("mono parameter must be one of L, R, or B.");
    return(0); }

  if (max_spl - min_spl < 2.0) {
    if (pop_box("max_ and min_spl are VERY close, do you really want this?",
		"No, cancel", "Yes", NULL) == 1) {
      set_progRunning(False);
      return(0); } }

  if((stimarray = StimArray_Gen(&nstims, 0, range, 1, 1, 0))==NULL)
    return(0);

  lmags = floatVec(reps);     rmags = floatVec(reps);
  lphis = floatVec(reps);     rphis = floatVec(reps);
  ldists = floatVec(reps);    rdists = floatVec(reps);
  llkmags = floatVec(reps);   rlkmags = floatVec(reps);
  llkphis = floatVec(reps);   rlkphis = floatVec(reps);
  llkdists = floatVec(reps);  rlkdists = floatVec(reps);

  is_setNice(GI("nicelevel"));
  is_setSyncPulse(GI("syncPulse"));
  is_init(daFc, adFc, 0.0, msepoch, msepoch);
  da_channels = is_getDAnchans();
  outbuf = is_getDAbuffer(&outsize);
 
  set_progRunning(True);

  fd_syn_spec_parse("hp",0,&ss);
  synthesize(&ss, NULL, outbuf, outsize, da_channels, msdelay, msdur, msepoch, 0, 100, msrise, msfall, SYN_LEFT);
  min_bw = (int)ceil(ss.parms.noise.res);

  statline_set(headline, "Running cal...");

  fprintf(outfile, "; 1: freq (Hz)\n");
  fprintf(outfile, "; 2: left_mag (dB)\n");
  fprintf(outfile, "; 3: left_phase (rad)\n");
  fprintf(outfile, "; 4: left_distortion (100*f2_mag/f1_mag)\n");
  fprintf(outfile, "; 5: left_leak_mag (dB, delta_mag for right stim)\n");
  fprintf(outfile, "; 6: left_leak_phase (rad, delta_phase for right stim)\n");
  fprintf(outfile, "; 7: left_leak_distortion (100*f2_leak_mag/f1_leak_mag)\n");
  fprintf(outfile, "; 8: right_mag (dB)\n");
  fprintf(outfile, "; 9: right_phase (rad)\n");
  fprintf(outfile, "; 10: right_distortion (100*f2_mag/f1_mag)\n");
  fprintf(outfile, "; 11: right_leak_mag (dB, delta_mag for left stim)\n");
  fprintf(outfile, "; 12: right_leak_phase (rad, delta_phase for left stim)\n");
  fprintf(outfile, "; 13: right_leak_distortion (100*f2_leak_mag/f1_leak_mag)\n");
  fprintf(outfile, "; 14: left_mag_stderr\n");
  fprintf(outfile, "; 15: left_phase_stderr\n");
  fprintf(outfile, "; 16: right_mag_stderr\n");
  fprintf(outfile, "; 17: right_phase_stderr\n");

  fprintf(outfile, "nrasters=%d\n",nstims);

  if(syn_calibrate) {  /* ie if (*check)!='0' */
    min_l_dbspl = cal_GetOverallMinDBSPL(0);
    min_r_dbspl = cal_GetOverallMinDBSPL(1);
    if((*mono)=='R')
       min_dbspl = min_r_dbspl;
    else if((*mono)=='L')
       min_dbspl = min_l_dbspl;
    else
       min_dbspl = (min_l_dbspl < min_r_dbspl) ? min_l_dbspl : min_r_dbspl;
    if((min_l_dbspl == -1000.0) || (min_r_dbspl == -1000.0))
      return(0); }

  isitime = isi_start(0);

  for (j=0; progRunning && j<nstims; j++) {
    percdone((float)j/(float)nstims);
    freq=stimarray[j];
    spl_retry_count = 0;
    spl_retry_l = spl_retry_r = 1;
    do {
      if((*mono)!='R') {
        if (spl_retry_l) {
          is_clearOutput(SYN_BOTH);
          if(!strcmp(check,"PT")) {
            fd_syn_spec_parse(NULL,freq,&ss);
            synthesize(&ss, NULL, outbuf, outsize, da_channels, msdelay, msdur, msepoch, 0, 100, msrise, msfall,
                  SYN_LEFT);
            is_clearOutput(SYN_RIGHT);
            cal_SafeFigureAtten(&ss, min_dbspl, min_dbspl, &l_calib,NULL); }
          if(!strcmp(check,"TS")) {
            char foo[128];
            sprintf(foo,"stack=%d",freq);
            fd_syn_spec_parse(foo,0,&ss);
            synthesize(&ss, NULL, outbuf, outsize, da_channels, msdelay, msdur, msepoch, 0, 100, msrise, msfall,
                  SYN_LEFT);
            is_clearOutput(SYN_RIGHT);
            cal_SafeFigureAtten(&ss, min_dbspl, min_dbspl, &l_calib,NULL); }
          if(!strcmp(check,"WN")) {
            char foo[128];
            sprintf(foo,"CF=%d,BW=%d",freq,min_bw);
            fd_syn_spec_parse(foo,0,&ss);
            synthesize(&ss, NULL, outbuf, outsize, da_channels, msdelay, msdur, msepoch, 0, 100, msrise, msfall,
                  SYN_LEFT);
            is_clearOutput(SYN_RIGHT);
            cal_SafeFigureAtten(&ss, min_dbspl, min_dbspl, &l_calib,NULL); }
          if(!strcmp(check,"0")) {
            fd_syn_spec_parse(NULL,freq,&ss);
            synthesize(&ss, NULL, outbuf, outsize, da_channels, msdelay, msdur, msepoch, 0, 100, msrise, msfall,
                  SYN_LEFT);
            is_clearOutput(SYN_RIGHT);
            l_calib = 0.0; }

          do {
			  clip_retry = 0;
			  for (i = 0; !clip_retry && progRunning && i < reps; i++) {
				  if(((latten+l_calib)>attMaxAtten) || ((latten+l_calib)<0.0))
					  notify("Can't calibrate properly in LEFT channel.");
				  setRack(0, (latten+l_calib), attMaxAtten);

				set_led(3,1);

				status = is_sample(X_TRIG | X_AD | X_DA, msepoch, &timestamp);
				if (!(single_step(status))) {
					  alert("is_sample failed");
					  set_progRunning(False);
				}

				set_led(3,0);

				view_input();
				view_output();

				  ibuf = is_getADbuffer(&count);
				  isi_stop(isitime);
				  isitime = isi_start(isi);
				  dloop_empty();
				  clip_retry = 0;
				  if (clipCheck(ibuf, 0, tepoch, 0.99)) {
					  if ((latten+l_calib) == attMaxAtten) {
						  notify("Possible clip on LEFT chn, %d hz\n", freq); }
					  else {
						  clip_retry = 1; } }
				  else if (clipCheck(ibuf + 1, 0, tepoch, 0.99)) {
					  if ((latten+l_calib) == attMaxAtten) {
						  notify("Possible clip on RIGHT chn leakage, %d hz\n", freq); }
					  else {
						  clip_retry = 1; } }
				  if (clip_retry) {
					  plottrace();
					  latten += astep;
					  if(latten>attMaxAtten) latten=attMaxAtten; }
				  else {
					  plottrace();

					  if(strcmp(check,"WN")) {
						  dsp_fitsin(ibuf + 2*(tdelay+trise), tdur-trise-tfall, 2,
							  is_adFc, 1.0*(float)freq, &lmags[i], NULL);
						  dsp_fitsin(ibuf + 2*tdelay, tdur, 2,
							  is_adFc, 1.0*(float)freq, NULL, &lphis[i]);
						  dsp_fitsin(ibuf + 2*(tdelay+trise), tdur-trise-tfall, 2,
							  is_adFc, 2.0*(float)freq, &ldists[i], NULL); }
					  else {
						  lmags[i] = ROOT2*dsp_calcrms(ibuf + 2*(tdelay+trise),
							  tdur-trise-tfall, 2, 1);
						  lphis[i] = 0.0;
						  ldists[i] = 0.0; }

					  ldists[i] /= lmags[i];     /* A(F1)/A(F0) */

					  if (!ad_to_mv(lmags[i], lmags+i, NULL, AddaRelative)) {
						  alert("Can't convert adticks-rms to mv-rms");
						  set_progRunning(False);
						  break; }

					  lmags[i] /= lgain;
					  query_mic_adjust(freq, 'L', &adj_mag, &adj_phi);
					  lmags[i] /= adj_mag;
					  lphis[i] -= adj_phi;
					  if(linv) lphis[i]+=M_PI;

					  lmags[i] /= ROOT2;         /* Amp to RMS */
					  lmags[i] = 20.0*log10(lmags[i]/SoL)+94.0;

					  if(SoR!=0.0) {
						  if(strcmp(check,"WN")) {
							  dsp_fitsin(ibuf+1 + 2*(tdelay+trise), tdur-trise-tfall, 2,
								  is_adFc, 1.0 * (float)freq, &rlkmags[i], NULL);
							  dsp_fitsin(ibuf+1 + 2*tdelay, tdur, 2,
								  is_adFc, 1.0*(float)freq, NULL, &rlkphis[i]);
							  dsp_fitsin(ibuf+1 + 2*(tdelay+trise), tdur-trise-tfall, 2,
								  is_adFc, 2.0*(float)freq, &rlkdists[i], NULL); }
						  else {
							  rlkmags[i] = ROOT2*dsp_calcrms(ibuf+1 + 2*(tdelay+trise),
								  tdur-trise-tfall, 2, 1);
							  rlkphis[i] = 0.0;
							  rlkdists[i] = 0.0; }

						  rlkdists[i] /= rlkmags[i];

						  if (!ad_to_mv(rlkmags[i], NULL, rlkmags+i, AddaRelative)) {
							  alert("Can't convert adticks-rms to mv-rms");
							  set_progRunning(False);
							  break; }

						  rlkmags[i] /= rgain;
						  query_mic_adjust(freq, 'R', &adj_mag, &adj_phi);
						  rlkmags[i] /= adj_mag;
						  rlkphis[i] -= adj_phi;
						  if(rinv) rphis[i]+=M_PI;

						  rlkmags[i] /= ROOT2;
						  rlkmags[i] = 20.0*log10(rlkmags[i]/SoR)+94.0; } }

				  dloop_empty(); } }
          while (clip_retry && progRunning); } }

      if((*mono)!='L') {
        if (spl_retry_r) {
          is_clearOutput(SYN_BOTH);
          if(!strcmp(check,"PT")) {
            fd_syn_spec_parse(NULL,freq,&ss);
            synthesize(&ss, NULL, outbuf, outsize, da_channels, msdelay, msdur, msepoch, 0, 100, msrise, msfall,
                  SYN_RIGHT);
            is_clearOutput(SYN_LEFT);
            cal_SafeFigureAtten(&ss, min_dbspl, min_dbspl, NULL, &r_calib); }
          if(!strcmp(check,"TS")) {
            char foo[128];
            sprintf(foo,"stack=%d",freq);
            fd_syn_spec_parse(foo,0,&ss);
            synthesize(&ss, NULL, outbuf, outsize, da_channels, msdelay, msdur, msepoch, 0, 100, msrise, msfall,
                  SYN_RIGHT);
            is_clearOutput(SYN_LEFT);
            cal_SafeFigureAtten(&ss, min_dbspl, min_dbspl, NULL, &r_calib); }
          if(!strcmp(check,"WN")) {
            char foo[128];
            sprintf(foo,"CF=%d,BW=%d",freq,min_bw);
            fd_syn_spec_parse(foo,0,&ss);
            synthesize(&ss, NULL, outbuf, outsize, da_channels, msdelay, msdur, msepoch, 0, 100, msrise, msfall,
                  SYN_RIGHT);
            is_clearOutput(SYN_LEFT);
            cal_SafeFigureAtten(&ss, min_dbspl, min_dbspl, NULL,&r_calib); }
          if(!strcmp(check,"0")) {
            fd_syn_spec_parse(NULL,freq,&ss);
            synthesize(&ss, NULL, outbuf, outsize, da_channels, msdelay, msdur, msepoch, 0, 100, msrise, msfall,
                  SYN_RIGHT);
            is_clearOutput(SYN_LEFT);
            r_calib = 0.0; }

          do {
			  clip_retry = 0;
			  for (i = 0; !clip_retry && progRunning && i < reps; i++) {
				  if(((ratten+r_calib)>attMaxAtten) || ((ratten+r_calib)<0.0))
					  notify("Can't calibrate properly in RIGHT channel.");
				  setRack(0, attMaxAtten, (ratten+r_calib));
				  set_led(3,1);

				  status = is_sample(X_TRIG | X_AD | X_DA, msepoch, &timestamp);
				  if (!(single_step(status))) {
						  alert("is_sample failed");
						  set_progRunning(False); 
				  }

				  set_led(3,0);

				  view_input();
				  view_output();

					  ibuf = is_getADbuffer(&count);
					  isi_stop(isitime);
					  isitime = isi_start(isi);
					  dloop_empty();
					  if (clipCheck(ibuf, 0, tepoch, 0.99)) {
						  if ((ratten+r_calib) == attMaxAtten) {
							  notify("Possible clip on LEFT chn leakage, %d hz\n", freq); }
						  else {
							  clip_retry = 1; } }
					  else if (clipCheck(ibuf + 1, 0, tepoch, 0.99)) {
						  if ((ratten+r_calib) == attMaxAtten) {
							  notify("Possible clip on RIGHT chn, %d hz\n", freq); }
						  else {
							  clip_retry = 1; } }

					  if (clip_retry) {
						  ratten += astep;
						  plottrace();
						  if(ratten>attMaxAtten) ratten=attMaxAtten; }
					  else {
						  plottrace();

						  if(strcmp(check,"WN")) {
							  dsp_fitsin(ibuf+1 + 2*(tdelay+trise), tdur-trise-tfall, 2,
								  is_adFc, 1.0 * (float)freq, &rmags[i], NULL);
							  dsp_fitsin(ibuf+1 + 2*tdelay, tdur, 2,
								  is_adFc, 1.0 * (float)freq, NULL, &rphis[i]);
							  dsp_fitsin(ibuf+1 + 2*(tdelay+trise), tdur-trise-tfall, 2,
								  is_adFc, 2.0 * (float)freq, &rdists[i], NULL); }
						  else {
							  rmags[i] = ROOT2*dsp_calcrms(ibuf+1 + 2*(tdelay+trise),
								  tdur-trise-tfall, 2, 1);
							  rphis[i] = 0.0;
							  rdists[i] = 0.0; }

						  rdists[i] /= rmags[i];     /* A(F1)/A(F0) */

						  if (!ad_to_mv(rmags[i], rmags+i, NULL, AddaRelative)) {
							  alert("Can't convert adticks-rms to mv-rms");
							  set_progRunning(False);
							  break; }

						  rmags[i] /= rgain;
						  query_mic_adjust(freq, 'R', &adj_mag, &adj_phi);
						  rmags[i] /= adj_mag;
						  rphis[i] -= adj_phi;
						  if(rinv) rphis[i]+=M_PI;

						  rmags[i] /= ROOT2;         /* Amp to RMS */
						  rmags[i] = 20.0*log10(rmags[i]/SoR)+94.0;

						  if(SoL!=0.0) {
							  if(strcmp(check,"WN")) {
								  dsp_fitsin(ibuf + 2*(tdelay+trise), tdur-trise-tfall, 2,
									  is_adFc, 1.0 * (float)freq, &llkmags[i], NULL);
								  dsp_fitsin(ibuf + 2*tdelay, tdur, 2,
									  is_adFc, 1.0 * (float)freq, NULL, &llkphis[i]);
								  dsp_fitsin(ibuf + 2*(tdelay+trise), tdur-trise-tfall, 2,
									  is_adFc, 2.0 * (float)freq, &llkdists[i], NULL); }
							  else {
								  llkmags[i] = ROOT2*dsp_calcrms(ibuf + 2*(tdelay+trise),
									  tdur-trise-tfall, 2, 1);
								  llkphis[i] = 0.0;
								  llkdists[i] = 0.0; }

							  llkdists[i] /= llkmags[i];

							  if (!ad_to_mv(llkmags[i], llkmags+i, NULL, AddaRelative)) {
								  alert("Can't convert adticks-rms to mv-rms");
								  set_progRunning(False);
								  break; }

							  llkmags[i] /= lgain;
							  query_mic_adjust(freq, 'L', &adj_mag, &adj_phi);
							  llkmags[i] /= adj_mag;
							  llkphis[i] -= adj_phi;
							  if(linv) lphis[i]+=M_PI;

							  llkmags[i] /= ROOT2;
							  llkmags[i] = 20.0*log10(llkmags[i]/SoL)+94.0; } }

					  dloop_empty(); } }
          while (clip_retry && progRunning); } }

      unwrap(lphis,reps,M_PI);
      lmag = mean(lmags, reps);		lmagsdv = stderror(lmags, reps);
      lphi = mean(lphis, reps);		lphisdv = stderror(lphis, reps);
      ldist = 100.0*mean(ldists, reps);

      unwrap(llkphis,reps,M_PI);
      llkmag = mean(llkmags, reps);
      llkphi = mean(llkphis, reps);
      llkdist = 100*mean(llkdists, reps);

      unwrap(rphis,reps,M_PI);
      rmag = mean(rmags, reps);		rmagsdv = stderror(rmags, reps);
      rphi = mean(rphis, reps);		rphisdv = stderror(rphis, reps);
      rdist = 100.0*mean(rdists, reps);

      unwrap(rlkphis,reps,M_PI);
      rlkmag = mean(rlkmags, reps);
      rlkphi = mean(rlkphis, reps);
      rlkdist = 100*mean(rlkdists, reps);

      spl_retry_l = spl_retry_r = 0;

      if((*mono)!='R') {
        statline_set(status1, "%dHz,  %.1f(%.1f)dB,  %.1f%%hd   LEFT",
              freq, lmag, lmag+latten, ldist); }
      else {
        statline_set(status1, ""); }

      if((*mono)!='L') {
        statline_set(status2, "%dHz,  %.1f(%.1f)dB,  %.1f%%hd   RIGHT",
              freq, rmag, rmag+ratten, rdist); }
      else {
        statline_set(status2, ""); }

      if((*mono)!='R') {
        if (lmag > max_spl && (latten+l_calib) < attMaxAtten) {
          spl_retry_l = 1;
          latten += ((lmag - max_spl)+(max_spl-min_spl)/2.0);
          if ((latten+l_calib) > attMaxAtten)
            latten = attMaxAtten - l_calib; }
        else if (lmag < min_spl && (latten+l_calib) > 0.0) {
          spl_retry_l = 1;
          latten -= ((min_spl - lmag)+(max_spl-min_spl)/2.0);
          if ((latten+l_calib) < 0.0)
            latten = -l_calib; }
        else if ((lmag > max_spl) || (lmag < min_spl)) {
          /* if it's not in the range, but you can't do anything */
          /* then write it to the file, and continue */
          spl_retry_l = 1;
          spl_retry_count=10; } }

      if((*mono)!='L') {
        if (rmag > max_spl && (ratten+r_calib) < attMaxAtten) {
          spl_retry_r = 1;
          ratten += ((rmag - max_spl)+(max_spl-min_spl)/2.0);
          if ((ratten+r_calib) > attMaxAtten)
            ratten = attMaxAtten - r_calib; }
        else if (rmag < min_spl && (ratten+r_calib) > 0.0) {
          spl_retry_r = 1;
          ratten -= ((min_spl - rmag)+(max_spl-min_spl)/2.0);
          if ((ratten+r_calib) < 0.0)
            ratten = -r_calib; }
        else if ((rmag > max_spl) || (rmag < min_spl)) {
          /* if it's not in the range, but you can't do anything */
          /* then write it to the file, and continue */
          spl_retry_r = 1;
          spl_retry_count=10; } }

      if (!spl_retry_l && !spl_retry_r)
        fprintf(outfile, "%5d\t%e\t%e\t%e\t%e\t%e\t%e\t%e\t%e\t%e\t%e\t%e\t%e\t%e\t%e\t%e\t%e\n", freq,
              lmag+latten, lphi, ldist, llkmag-rmag, llkphi-rphi, llkdist,
              rmag+ratten, rphi, rdist, rlkmag-lmag, rlkphi-lphi, rlkdist,
              lmagsdv, lphisdv, rmagsdv, rphisdv);

      if (++spl_retry_count > 10) {
        notify("Can't auto-range at %dHz%s%s -- continuing on...",
              freq, spl_retry_l ? " L" : "", spl_retry_r ? " R" : "");
        fprintf(outfile, "%5d\t%e\t%e\t%e\t%e\t%e\t%e\t%e\t%e\t%e\t%e\t%e\t%e\t%e\t%e\t%e\t%e\n", freq,
              lmag+latten, lphi, ldist, llkmag-rmag, llkphi-rphi, llkdist,
              rmag+ratten, rphi, rdist, rlkmag-lmag, rlkphi-lphi, rlkdist,
              lmagsdv, lphisdv, rmagsdv, rphisdv);
        spl_retry_l = spl_retry_r = 0; } }
    while ((spl_retry_l || spl_retry_r) && progRunning);

    fflush(outfile);
    dloop_empty();
    MaybePause(); }

  free(lmags);     free(rmags);
  free(lphis);     free(rphis);
  free(ldists);    free(rdists);
  free(llkmags);   free(rlkmags);
  free(llkphis);   free(rlkphis);
  free(llkdists);  free(rlkdists);

  is_shutdown();

  if (progRunning) {
    statline_set(headline, "Done");
    set_progRunning(False);
    return(1); }
  else {
    statline_set(headline, "Halted");
    return(0); }
}

Widget setupCal(DockSlot *slot)
{
  Widget ws, h;

  ws = pop_ws_new(TopLevel, ws_table, "cal", &h, NULL, NULL);
  make_start_temp(h,
        make_module_info("cal", "cal", "cal", NULL, runScan,
              NULL, default_display_fn, ws_table, WS_N(ws_table)),
        NULL, NULL, slot);
  return(ws);
}
