/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: mathfns.c,v 2.54 2000/05/25 00:08:35 bjarthur Exp $
** Program: xdphys
**  Module: mathfns.c
**  Author: mazer
** Descrip: misc useful math functions
**
** Revision History (most recent last)
**
** Fri Nov 26 13:10:28 1993 mazer
**  added splineify() interface to nr2 spline() and splinet()
**  routines; added XX_minmax() routines
**
** Thu May  5 23:19:52 1994 mazer
**  added lin_interpolate_at() -- can supply position in vectors
**  to use to avoid searching (if you know something a priori
**  about the vectors).
**
** 97.4 bjarthur
**  #ifdef'ed out a bunch of unused stuff
**  added greatest_common_factor()
**
** 97.5 bjarthur
**  changed lin_interpolate to work when data wasn't evenly spaced.
**   previously, it assumed this was the case.  should also be faster
**   now, as it only loops through the data once.
**
** 98.11 bjarthur
**  deleted bunch of stuff that isn't used
**
*******************************************************************/

#include "xdphyslib.h"

float limited_uniform(long *seedp)
{
  register float u;

  do {
    u = (2.0 * ran1(seedp)) - 1.0;
  } while (u > 1.0 || u < -1.0);
  return(u);
}

/* doesn't assume xs[] is ordered, or evenly spaced  */
float lin_interpolate(float x, int n, float *xs, float *ys)
{
	int i, pf, pc, f, c;

	for(i=0, pf=-1, pc=-1; i<n; i++) {
		if(((pf==-1) || (xs[i]>f)) && (xs[i]<=x)) {
			pf = i;
			f = xs[i]; }
		if(((pc==-1) || (xs[i]<c)) && (xs[i]>=x)) {
			pc = i;
			c = xs[i]; } }

  if((pf!=-1)&&(pc!=-1)) {
		if(pf==pc)
    	return(ys[pc]);
		else
    	return(((ys[pc]-ys[pf]) / (xs[pc]-xs[pf])) * (x-xs[pf]) + ys[pf]); }
  else if(pf==-1)
    return(ys[pc]);
  else if(pc==-1)
    return(ys[pf]);

  assert((pf==-1)&&(pc==-1));

  return(0.0);
}

/* assumes ordered and even for speed */
float lin_interpolate2(xword *buffer, int nsamps, int nskip, float u)
{
  int x;
  float r, ret_val;

  if(u<0) {
    ret_val = buffer[0]; }
  else if(u>(nsamps-1)) {
    ret_val = buffer[(nsamps-1)*nskip]; }
  else {
    x = floor(u);
    r = u-x;
    ret_val = buffer[x*nskip] + r*(buffer[(x+1)*nskip]-buffer[x*nskip]); }

  return(ret_val);
}

float cubic_interpolate(xword *buffer, int nsamps, int nskip, float u)
{
  float s,s2,s3,f;
  int ui;

  s = u - floor(u);
  ui = floor(u);

  if(((ui-1)<0) || ((ui+2)>nsamps))
    return(MAXFLOAT);

  s2 = s*s; s3 = s*s2;
  f = buffer[(ui-1)*nskip]*(-s3+2*s2-s) + buffer[ui*nskip]*(3*s3-5*s2+2) + 
    buffer[(ui+1)*nskip]*(-3*s3+4*s2+s) + buffer[(ui+2)*nskip]*(s3-s2);
  return(f/2);
}

int greatest_common_factor(int one, int two)
{
  int tmp,rem=1;

  assert((one>0)&&(two>0));

  if(one>two) {
    tmp=one; one=two; two=tmp; }

  tmp=one;
  while((tmp>0)&&(rem!=0)) {
    while((tmp>0)&&(rem!=0)) {
      rem=one%tmp;
      tmp--; }
    rem=two%(tmp+1); }

  return(tmp+1);
}

/************************* UNUSED *********************/
#if(0)


/*
** use dy0=dyn=1.0e30 for "natural splines"
 */

void splinefit(float *xin, float *yin, int nin, float **xoutptr,
      float **youtptr, int nout, float dy0, float dyn)
{
  int i;
  float min, max, step, y, *y2;
  float *xout, *yout;

  floatVec_minmax(xin, nin, &min, &max);
  step = (max - min) / (nout - 1);
  /*
   * y2 must be nin long, and passed in as y2-1, or else,
   * nin+1 long --> it's the usual 1-offset problem from NR
   */
  y2 = floatVec(nin);
  spline(xin - 1, yin - 1, nin, dy0, dyn, y2 - 1);

  xout = floatVec(nout);
  yout = floatVec(nout);
  for (i = 0; i < nout; i++, min += step) {
    splint(xin - 1, yin - 1, y2 - 1, nin, min, &y);
    xout[i] = min;
    yout[i] = y;
  }
  *xoutptr = xout;
  *youtptr = yout;
  free(y2);
}

/* in this function, ninterp refers to the window around
** a given point that will be used for polynominal
** interpolation -- 1 + (2 * ninterp) orignal data
** points will be used to interpolate any given output
** point
*/
void polyfit(float *xin, float *yin, int nin, float **xoutptr, float **youtptr,
      int nout, int ninterp)
{
  int i, k;
  float min, max, step, y, dy;
  float *xout, *yout;

  floatVec_minmax(xin, nin, &min, &max);
  step = (max - min) / (nout - 1);

  xout = floatVec(nout);
  yout = floatVec(nout);

  for (i = 0; i < nout; i++, min += step) {
    k = RND2INT((float)i * (float)nin / (float)nout);
    if (k - ninterp < 0)
      k = 0;
    else if (k + (2 * ninterp) > nin)
      k = nin - (2 * ninterp);
    else
      k = k - ninterp;

    polint(xin + k - 1, yin + k - 1, 2 * ninterp, min, &y, &dy);
    xout[i] = min;
    yout[i] = y;
  }
  *xoutptr = xout;
  *youtptr = yout;
}


#define next2(n, n2) for ((n2) = 1; (n2) < (n); (n2) <<= 1) ;

int xyfloat_load(char *fname, float **xdata, float **ydata)
{
  int n, navail;
  char lbuf[200];
  float *x, *y;
  FILE *fp;

  if ((fp = fopen2(fname, "r")) == NULL)
    return(-1);

  n = 0;
  navail = 10;
  x = (float *)calloc(sizeof(float), navail);
  y = (float *)calloc(sizeof(float), navail);

  while (fgets(lbuf, sizeof(lbuf), fp) != NULL) {
    if (lbuf[0] == '#' || sscanf(lbuf, "%f%f", &x[n], &y[n]) != 2) {
      continue;
    } else {
      n++;
      if (n == navail) {
	navail += 10;
	x = (float *)realloc(x, sizeof(float) * navail);
	y = (float *)realloc(y, sizeof(float) * navail);
      }
    }
  }
  if (n > 0) {
    x = (float *)realloc(x, sizeof(float) * n);
    y = (float *)realloc(y, sizeof(float) * n);
    *xdata = x;
    *ydata = y;
    return(n);
  } else {
    free(x);
    free(y);
    return(-1);
  }
}


float lin_interpolate_at(float x, int n, float *xs, float *ys, int p1, int p2)
{
  return(((ys[p1] - ys[p2]) / (xs[p1] - xs[p2])) * (x - xs[p1]) + ys[p1]);
}

float lin_interpolate_limited(float x, int n, float *xs, float *ys, int from,
      int to)
{
  int i;
  int p1, p2;
  float d, mind;

  if (from > n)
    from = n - 1;
  if (from < 0)
    from = 0;
  if (to > n)
    to = n;
  if (to < 0)
    to = 0;

  p1 = p2 = -1;
  for (mind = -1.0, i = from; i < to && i < n; i++) {
    if (xs[i] == x)
      return(ys[i]);
    if ((d = xs[i] - x) < 0)
      d = -d;
    if (mind < 0.0 || d < mind) {
      p1 = i;
      mind = d;
    }
  }
  for (mind = -1.0, i = from; i < to && i < n; i++) {
    if (i != p1) {
      if ((d = xs[i] - x) < 0)
	d = -d;
      if (mind < 0.0 || d < mind) {
	p2 = i;
	mind = d;
      }
    }
  }
  return(((ys[p1] - ys[p2]) / (xs[p1] - xs[p2])) * (x - xs[p1]) + ys[p1]);
}

/* float *xwordbuf_to_fv(xbuf, n, skip, power_of_2, opt_fn)
** Convert an xword buffer (i.e. signal) into a floating
** point vector.
**   If power_of_2 is specified, the buffer will be zero
**     padded to the nearest power_of_2 (this is useful for
**     fft's).
**   If opt_fn != NULL, *opt_fn will be set to the actual
**     length of the vector.
*/
float *xwordbuf_to_fv(xword *xbuf, int n, int skip, int power_of_2,
      int *opt_actualn)
{
  float *v;
  int i, actualn;

  if (power_of_2) {
    next2(n, actualn);
  } else {
    actualn = n;
  }
  v = floatVec(actualn);
  for (i = 0; i < n; i++) {
    v[i] = xbuf[i * skip];
  }
  while (i < actualn)
    v[i++] = 0;
  if (opt_actualn != NULL)
    *opt_actualn = actualn;
  return(v);
}

/* vector add */
float *fv_vadd(float *dest, float *src, int n)
{
  int i;
  for (i = 0; i < n; i++)
    dest[i] += src[i];
  return(dest);
}

/* scalar mult */
float *fv_smult(float *dest, float scalar, int n)
{
  int i;
  for (i = 0; i < n; i++)
    dest[i] *= scalar;
  return(dest);
}

/* find min, max */
void fv_minmax(float *x, float *pmin, float *pmax, int n)
{
  int i;

  *pmin = *pmax = x[0];
  for (i = 1; i < n; i++) {
    if (x[i] < *pmin)
      *pmin = x[i];
    if (x[i] > *pmax)
      *pmax = x[i];
  }
}

#ifdef USE_VSPLINE
void vsplinefit(float *xin, float *yin, int nin, float **xoutptr,
      float **youtptr, int nout, float dy0, float dyn)
{
  int i;

  float min, max, step, y;
  float *xout, *yout;

  double *ys, *dxin, *dyin, *inv_sigma;
  double *dxout, *dyout;
  double *alpha;

  floatVec_minmax(xin, nin, &min, &max);
  step = (max - min) / (nout - 1);
  ys = doubleVec(nin);
  dxin = doubleVec(nin);
  dyin = doubleVec(nin);
  alpha = doubleVec(nin);
  inv_sigma = doubleVec(nin * nin);
  for (i = 0; i < nin; i++) {
    dxin[i] = xin[i];
    dyin[i] = yin[i];
    alpha[i] = 1.0;
  }
  for (i = 0; i < nin * nin; i++) {
    inv_sigma[i] =  1.0/i;
  }
  bestcv(ys-1, alpha-1, dxin-1, dyin-1, inv_sigma-1, 1, nin, 1);
  printf("vspline()=%d\n",
	 vspline(ys-1, alpha-1, dxin-1, dyin-1, inv_sigma-1, 1, nin, 1));

  dxout = doubleVec(nout);
  dyout = doubleVec(nout);
  for (i = 0; i < nout; i++, min += step) {
    dxout[i] = min;
  }

  printf("cubic_interp()=%d\n",
	 cubic_interp(dyout-1, dxin-1, ys-1, dxout-1, nin, nout));

  for (i = 0; i < nout; i++) {
    xout[i] = dxout[i];
    yout[i] = dyout[i];
  }

  *xoutptr = xout;
  *youtptr = yout;
  free(dxin); free(dxout);
  free(dyin); free(dyout);
  free(inv_sigma);
  free(alpha);
  free(ys);
}
#endif

void doubleVec_minmax(double *v, int n, double *minp, double *maxp)
{
  int i;

  if (n > 0) {
    *minp = *maxp = v[0];
    for (i = 1; i < n; i++) {
      if (v[i] > *maxp)
	*maxp = v[i];
      if (v[i] < *minp)
	*minp = v[i];
    }
  }
}


void intVec_minmax(int *v, int n, int *minp, int *maxp)
{
  int i;

  if (n > 0) {
    *minp = *maxp = v[0];
    for (i = 1; i < n; i++) {
      if (v[i] > *maxp)
	*maxp = v[i];
      if (v[i] < *minp)
	*minp = v[i];
    }
  }
}

#endif
