/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: foo.c,v 1.1 1998/09/10 20:10:12 cmalek Exp $
** Program: dowl
**  Module: plotfile.c
**  Author: mazer
** Descrip: high level interface to graph.c -- reads dowl datafiles
**
** Revision History (most recent last)
**
** 97.2 bjarthur
**  created from fd.c
**
** 97.4 bjarthur
**   added FD1_perhist1FN(), FD1_perhist2FN()
**
** 98.2 bjarthur
**  moved struct RawData definition into fd1.h to facilitate less duplication
**  of FD1_*FN() routines.  now mod's which have the same RawData like
**  bam,beats,cf, etc... can use some of the same plot FNs, like psth, isih,
**  and perhist.
**
** 98.5 bjarthur
**  added similarity_stats and isih_stats.
**  added nspikes in SumData struct, did NOT change fd2.c, or the bam,
**    beats, cf, or rover modules accordingly
**  modified _summarize() to realloc arrays to save space
**
*******************************************************************/

#include "xdphyslib.h"

/* ------------------------------------------------------------------------
   Global Variables
   ------------------------------------------------------------------------ */
char dis[50][24];  /* used in perhist_stats */
float mps[50];
float vss[50];
float rays[50];

/* ------------------------------------------------------------------------
   Typedefs
   ------------------------------------------------------------------------ */
typedef struct {
  int           unit;           /* unit id or "mask" */
  int           ndata;          /* number of summary data points */
  char        **depstrs;        /* depvar values (int) for matching means */
                                /* note: the depstrs[n] here are actually owned
                                 * by deplines, and should NOT be free'd*/
  float        *depvals;        /* depvar values (int) for matching means */
  float        *means;          /* mean responses (nrasters/reps) */
  float        *stderrs;        /* std error of responses (nrasters/reps) */
  float        *stddevs;        /* std dev of responses (nrasters/reps) */
  int          *n;              /* # of data points means, etc are based on */
  int         **nspikes;        /* nspikes[depval][rep] = #spikes */
} SumData;

/* ------------------------------------------------------------------------
   Prototypes
   ------------------------------------------------------------------------ */
static void FD1_sort(FILEDATA*);
static int FD1_summarize(FILEDATA*, int);
static float *copyVec(float*, int, float);
static void FD1_freeSumData(FILEDATA*);
static int common_basis(float*, int, int**, int*, float*, int, int**, int*,
        float****, int*, int*, int***);
static void compute_isih_stats(int, int**, int, float, int*, int*);

/* ------------------------------------------------------------------------
                        FUNCTIONS
   ------------------------------------------------------------------------ */
/* ------------------------------------------------------------------------
   FD1_curveFN
   ------------------------------------------------------------------------ */
void FD1_curveFN(FDO *fdo, FILEDATA *fd, int l, FILE *fptr)
{
    int i;
    char *ti;
    float f1;
    char *xlabel, *ylabel;
    float *px, *py;
    int pn;


    f1 = FD1_summarize(fd, fdo->sum_code[l]);
    if (fdo->normalize == 1) {
		ylabel = "% max response";
		f1 = SUMDATA(fd)->means[0];
		for (i = 1; i < SUMDATA(fd)->ndata; i++) {
			if (SUMDATA(fd)->means[i] > f1)
				f1 = SUMDATA(fd)->means[i]; }
		f1 /= 100.0;		/* convert to percent */
    } else {
		ylabel = "nspikes";
		f1 = 1.0;		/* no scale factor.. */ }
    xlabel = FD_GV(fd, "depvar");
    ti = "curve";

    FD_plotter_hack(SUMDATA(fd)->depvals, SUMDATA(fd)->means,
		!(fdo->no_errs) ? SUMDATA(fd)->stderrs : NULL, SUMDATA(fd)->ndata, 1,
      	&px, &py, &pn);

    for(i=0; i<pn; i++)
		py[i] /= f1;

    if(fptr!=NULL) {
		fprintf(fptr, "COMMENTS\n");
		fprintf(fptr,"%%%s, %s view\n",
			strrchr(fd->filename,'/')==NULL ? fd->filename :
			1+(char*)strrchr(fd->filename,'/'), ti);
		fprintf(fptr,"%%col 1: %s\n", xlabel);
		fprintf(fptr,"%%col 2: %s\n", ylabel);
		if(!(fdo->no_errs))
			fprintf(fptr,"%%col 3: std error\n");
		fprintf(fptr, "END_COMMENTS\n");
		fprintf(fptr, "PARAMS\n");
		fprintf(fptr,"Reps=%d\n",FD_GI(fd,"Reps"));
		fprintf(fptr,"spont=%e\t%e\n",
			SUMDATA(fd)->means[0], SUMDATA(fd)->stderrs[0]);
		fprintf(fptr, "END_PARAMS\n");
		fprintf(fptr, "RASTERDATA\n");
		fprintf(fptr, "nlines=%d\n",SUMDATA(fd)->ndata-1);
		if(fdo->no_errs) {
			for(i=1; i<SUMDATA(fd)->ndata; i++)
				fprintf(fptr,"%e\t%e\n",SUMDATA(fd)->depvals[i],
					SUMDATA(fd)->means[i]); }
		else {
			for(i=1; i<SUMDATA(fd)->ndata; i++)
				fprintf(fptr,"%e\t%e\t%e\n",SUMDATA(fd)->depvals[i],
					SUMDATA(fd)->means[i], SUMDATA(fd)->stderrs[i]); }
		fprintf(fptr, "END_RASTERDATA\n"); }

    FDObj_Add_Data_All(fdo,px,py,pn);
    FDObj_AddLine(fdo,l,px,py,pn,
        AtPlotSEGMENTPOINTS, AtMarkRECTANGLE, ConvertColor(fdo->graph,
			FDObj_LineColors[l%FDObj_numLineColors]), FDObj_Legend(fdo,l));

    XtVaSetValues(fdo->xaxis, XtNautoScale, True, NULL);
    XtVaSetValues(fdo->graph, XtNtitle, FDObj_Title(fdo, ti, l),
        XtNshowTitle, True, NULL);
    XtVaSetValues(fdo->xaxis, XtNlabel, xlabel, NULL);
    XtVaSetValues(fdo->yaxis, XtNlabel, ylabel, NULL);

	fflush(stderr);
}

/* ------------------------------------------------------------------------
   FD1_rasterFN
   ------------------------------------------------------------------------ */
void FD1_rasterFN(FDO *fdo, FILEDATA *fd, int l, int raw, FILE *fptr)
{
  int i, j, k;
  char *ti;
  SA_SpikeRasterList *srl = NULL;
  char *xlabel, *ylabel;
  float *xdata, *ydata;
  int ndata, *rep;
  float low, high;
  float *ftmp,tmp2;
  int tmp;
  double xmin, xmax;
  float y;

  FD1_summarize(fd, fdo->sum_code[l]);
  srl = FD_to_SpikeRasterList(FD1_RAWDATA(fd)->nrasters,
        FD1_RAWDATA(fd)->rastlist, FD_GI(fd,"epoch"),
        NULL, fdo->sum_code[l]);
  ndata = SA_TotalSpikesInSRL(srl);
  assert((xdata = (float*)malloc(sizeof(float)*ndata))!=NULL);
  assert((ydata = (float*)malloc(sizeof(float)*ndata))!=NULL);

  xlabel = "time (ms)";
  ylabel = raw ? "raster # (pres order)" : "dep int (no SR)";
  ti = raw ? "raw_raster" : "raster";

  if(fptr!=NULL) {
    fprintf(fptr, "COMMENTS\n");
    fprintf(fptr,"%%%s, %s view\n",
          strrchr(fd->filename,'/')==NULL ? fd->filename :
          1+(char*)strrchr(fd->filename,'/'), ti);
    fprintf(fptr,"%%col 1: %s\n", xlabel);
    fprintf(fptr,"%%col 2: %s\n", ylabel);
    fprintf(fptr, "END_COMMENTS\n");
    fprintf(fptr, "PARAMS\n");
    fprintf(fptr, "END_PARAMS\n");
    fprintf(fptr, "RASTERDATA\n");
		fprintf(fptr, "nlines=%d\n",ndata); }

  assert((rep = (int*)calloc(SUMDATA(fd)->ndata,sizeof(int)))!=NULL);
  for (k=i=0; i < srl->nrasters; i++) {
    if(raw) {
      y = FD1_RAWDATA(fd)->pres_order[i]; }
    else {
      if(FD1_RAWDATA(fd)->depints[i]==SPONT) continue;
      tmp2 = (float)FD1_RAWDATA(fd)->depints[i];
      ftmp = (float*)bsearch(&tmp2,SUMDATA(fd)->depvals,SUMDATA(fd)->ndata,
            sizeof(float),floatcompare);
      high = ((ftmp+1)<(SUMDATA(fd)->depvals+SUMDATA(fd)->ndata)) ?
            (*(ftmp+1)) : 2*(*ftmp) - (*(ftmp-1));
      high = (*ftmp) + 0.5*(high-(*ftmp));
      low = ((ftmp-1)>SUMDATA(fd)->depvals) ?
            (*(ftmp-1)) : 2*(*ftmp) - (*(ftmp+1));
      low = (*ftmp) + 0.5*(low-(*ftmp));
      tmp = ftmp-SUMDATA(fd)->depvals;
      y = low + (high-low)*(float)(rep[tmp]%
            SUMDATA(fd)->n[tmp])/(float)SUMDATA(fd)->n[tmp];
      rep[tmp]++; }
    for (j=0; j < srl->sr[i]->nspikes; j++) {
      xdata[k] = srl->sr[i]->tm[j];
      ydata[k] = y;
  		if(fptr!=NULL)
      	fprintf(fptr,"%e\t%e\n", xdata[k], ydata[k]);
			k++; } }
  	if(fptr!=NULL)
    	fprintf(fptr, "END_RASTERDATA\n");

  FDObj_Add_Data_All(fdo,xdata,ydata,k);
  FDObj_AddLine(fdo, l, xdata, ydata, k,
        AtPlotPOINTS, AtMarkVBAR, ConvertColor(fdo->graph,
        FDObj_LineColors[l%FDObj_numLineColors]), FDObj_Legend(fdo,l));

  xmin = 0.0;   xmax = (double)FD_GI(fd,"epoch");
  XtVaSetValues(fdo->xaxis, XtNmin, &xmin, XtNmax, &xmax,
        XtNautoScale, False, NULL);
	if(raw) {
  	xmin = 0.0;   xmax = (double)srl->nrasters;
  	XtVaSetValues(fdo->yaxis, XtNmin, &xmin, XtNmax, &xmax,
   	     XtNautoScale, False, NULL); }

  XtVaSetValues(fdo->graph, XtNtitle, FDObj_Title(fdo, ti, l),
        XtNshowTitle, True, NULL);
  XtVaSetValues(fdo->xaxis, XtNlabel, xlabel, NULL);
  XtVaSetValues(fdo->yaxis, XtNlabel, ylabel, NULL);
  SA_FreeSpikeRasterList(srl);
  free(rep);
}

/* ------------------------------------------------------------------------
   FD1_psthFN
   ------------------------------------------------------------------------ */
void FD1_psthFN(FDO *fdo, FILEDATA *fd, int l, FILE *fptr)
{
  char *ti;
  SA_Histogram *h;
  SA_SpikeRasterList *srl = NULL;
  float f1;
  char *xlabel, *ylabel;
  float *xdata, *ydata;
  int ndata,i;
  double xmin, xmax;

  srl = FD_to_SpikeRasterList(FD1_RAWDATA(fd)->nrasters,
        FD1_RAWDATA(fd)->rastlist, FD_GI(fd,"epoch"),
        NULL, fdo->sum_code[l]);
  if ((f1 = lookupParms_float("parms.psth_bin")) <= 0.0)
    f1 = 1.0;
  h = SA_Psth(srl, 0.0, fdo->epoch, (int) (0.5 + fdo->epoch / f1),
				lookupParms_int("parms.psth_triangle"));
  ndata = h->n;
  xdata = h->x;
  ydata = h->y;
  free(h);

  ndata++;
  assert((xdata=(float*)realloc(xdata,ndata*sizeof(float)))!=NULL);
  assert((ydata=(float*)realloc(ydata,ndata*sizeof(float)))!=NULL);
  xdata[ndata-1]=2*xdata[ndata-2]-xdata[ndata-3];
  ydata[ndata-1]=ydata[ndata-2];

  ti = "psth";
  xlabel = "time (ms)";
  ylabel = "nspikes";

  if(fptr!=NULL) {
    fprintf(fptr, "COMMENTS\n");
    fprintf(fptr,"%%%s, %s view\n",
          strrchr(fd->filename,'/')==NULL ? fd->filename :
          1+(char*)strrchr(fd->filename,'/'), ti);
    fprintf(fptr,"%%col 1: %s\n", xlabel);
    fprintf(fptr,"%%col 2: %s\n", ylabel);
    fprintf(fptr, "END_COMMENTS\n");
    fprintf(fptr, "PARAMS\n");
    fprintf(fptr, "END_PARAMS\n");
    fprintf(fptr, "RASTERDATA\n");
		fprintf(fptr, "nlines=%d\n",ndata-1);
   	for(i=0; i<ndata-1; i++)
     	fprintf(fptr,"%e\t%e\n",xdata[i],ydata[i]);
    fprintf(fptr, "END_RASTERDATA\n"); }

  FDObj_Add_Data_All(fdo,xdata,ydata,ndata);
  FDObj_AddLine(fdo, l, xdata, ydata, ndata,
        AtPlotSTEPS, -1, ConvertColor(fdo->graph,
        FDObj_LineColors[l%FDObj_numLineColors]), FDObj_Legend(fdo,l));

  xmin = 0.0;   xmax = (double)FD_GI(fd,"epoch");
  XtVaSetValues(fdo->xaxis, XtNmin, &xmin, XtNmax, &xmax,
        XtNautoScale, False, NULL);
  XtVaSetValues(fdo->graph, XtNtitle, FDObj_Title(fdo, ti, l),
        XtNshowTitle, True, NULL);
  XtVaSetValues(fdo->xaxis, XtNlabel, xlabel, NULL);
  XtVaSetValues(fdo->yaxis, XtNlabel, ylabel, NULL);
  SA_FreeSpikeRasterList(srl);
}

/* ------------------------------------------------------------------------
   FD1_isihFN
   ------------------------------------------------------------------------ */
void FD1_isihFN(FDO *fdo, FILEDATA *fd, int l, FILE *fptr)
{
  char *ti;
  SA_Histogram *h;
  SA_SpikeRasterList *srl = NULL;
  float f1, f2;
  char *xlabel, *ylabel;
  float *xdata, *ydata;
  int ndata,i;
  double xmin, xmax;

  srl = FD_to_SpikeRasterList(FD1_RAWDATA(fd)->nrasters,
        FD1_RAWDATA(fd)->rastlist, FD_GI(fd,"epoch"),
        NULL, fdo->sum_code[l]);
  if ((f1 = lookupParms_float("parms.isih_max")) <= 0)
    f1 = 10.0;
  if ((f2 = lookupParms_float("parms.isih_bin")) <= 0.0)
    f2 = 1.0;
  h = SA_Isih(srl, 0.0, f1, (int) (0.5 + f1 / f2));
  ndata = h->n;
  xdata = h->x;
  ydata = h->y;
  free(h);

  ndata++;
  assert((xdata=(float*)realloc(xdata,ndata*sizeof(float)))!=NULL);
  assert((ydata=(float*)realloc(ydata,ndata*sizeof(float)))!=NULL);
  xdata[ndata-1]=2*xdata[ndata-2]-xdata[ndata-3];
  ydata[ndata-1]=ydata[ndata-2];

  ti = "isih";
  xlabel = "time (ms)";
  ylabel = "nspikes";

  if(fptr!=NULL) {
    fprintf(fptr, "COMMENTS\n");
    fprintf(fptr,"%%%s, %s view\n",
          strrchr(fd->filename,'/')==NULL ? fd->filename :
          1+(char*)strrchr(fd->filename,'/'), ti);
    fprintf(fptr,"%%col 1: %s\n", xlabel);
    fprintf(fptr,"%%col 2: %s\n", ylabel);
    fprintf(fptr, "END_COMMENTS\n");
    fprintf(fptr, "PARAMS\n");
    fprintf(fptr, "END_PARAMS\n");
    fprintf(fptr, "RASTERDATA\n");
		fprintf(fptr, "nlines=%d\n",ndata-1);
   	for(i=0; i<ndata-1; i++)
     	fprintf(fptr,"%e\t%e\n",xdata[i],ydata[i]);
    fprintf(fptr, "END_RASTERDATA\n"); }

  FDObj_Add_Data_All(fdo,xdata,ydata,ndata);
  FDObj_AddLine(fdo, l, xdata, ydata, ndata,
        AtPlotSTEPS, -1, ConvertColor(fdo->graph,
        FDObj_LineColors[l%FDObj_numLineColors]), FDObj_Legend(fdo,l));

  xmin = 0.0;   xmax = (double)lookupParms_float("parms.isih_max");
  XtVaSetValues(fdo->xaxis, XtNmin, &xmin, XtNmax, &xmax,
        XtNautoScale, False, NULL);
  XtVaSetValues(fdo->graph, XtNtitle, FDObj_Title(fdo, ti, l),
        XtNshowTitle, True, NULL);
  XtVaSetValues(fdo->xaxis, XtNlabel, xlabel, NULL);
  XtVaSetValues(fdo->yaxis, XtNlabel, ylabel, NULL);
  SA_FreeSpikeRasterList(srl);
}

/* ------------------------------------------------------------------------
   FD1_autocorrFN
   ------------------------------------------------------------------------ */
void FD1_autocorrFN(FDO *fdo, FILEDATA *fd, int l, FILE *fptr)
{
  char *ti;
  SA_Histogram *h;
  SA_SpikeRasterList *srl = NULL;
  float f1;
  char *xlabel, *ylabel;
  float *xdata, *ydata;
  int ndata;

  srl = FD_to_SpikeRasterList(FD1_RAWDATA(fd)->nrasters,
        FD1_RAWDATA(fd)->rastlist, FD_GI(fd,"epoch"),
        NULL, fdo->sum_code[l]);
  if ((f1 = lookupParms_float("parms.ach_bin")) <= 0.0)
    f1 = 1.0;
  h = SA_Correlate(srl, srl, 1, -fdo->epoch, fdo->epoch,
        (int)(0.5+(2 * fdo->epoch / f1)));
  ndata = h->n;
  xdata = h->x;
  ydata = h->y;
  free(h);

  FDObj_Add_Data_All(fdo,xdata,ydata,ndata);
  FDObj_AddLine(fdo, l, xdata, ydata, ndata,
        AtPlotLINES, -1, ConvertColor(fdo->graph,
        FDObj_LineColors[l%FDObj_numLineColors]), FDObj_Legend(fdo,l));

  ti = "autocor";
  XtVaSetValues(fdo->graph, XtNtitle, FDObj_Title(fdo, ti, l),
        XtNshowTitle, True, NULL);
  xlabel = "time (ms)";
  ylabel = "nspikes";
  XtVaSetValues(fdo->xaxis, XtNlabel, xlabel, NULL);
  XtVaSetValues(fdo->yaxis, XtNlabel, ylabel, NULL);
  SA_FreeSpikeRasterList(srl);
}

/* ------------------------------------------------------------------------
   FD1_densityFN
   ------------------------------------------------------------------------ */
void FD1_densityFN(FDO *fdo, FILEDATA *fd, int l, FILE *fptr)
{
  int i;
  char *ti;
  SA_SpikeRasterList *srl = NULL;
  SA_SpikeDensityList *sdlp = NULL;
  float f1, f2;
  char *xlabel, *ylabel;
  float *xdata, *ydata;
  int ndata;

  srl = FD_to_SpikeRasterList(FD1_RAWDATA(fd)->nrasters,
        FD1_RAWDATA(fd)->rastlist, FD_GI(fd,"epoch"),
        NULL, fdo->sum_code[l]);
  sdlp = NULL;
  if ((f1 = lookupParms_float("parms.sdens_sigma")) <= 0.0)
    f1 = 3.0;
  if ((f2 = lookupParms_float("parms.sdens_iters")) <= 0.0)
    f2 = 6.0;
  SA_CalcSpikeDensityList(srl, &sdlp, f1, (int)(0.5 + f2));
  ndata = sdlp->dur;
  assert((xdata = (float*)malloc(sizeof(float)*ndata))!=NULL);
  assert((ydata = (float*)malloc(sizeof(float)*ndata))!=NULL);
  for (i = 0; i < sdlp->dur; i++) {
    xdata[i] = i;
    ydata[i] = sdlp->mean[i]; }

  FDObj_Add_Data_All(fdo,xdata,ydata,ndata);
  FDObj_AddLine(fdo, l, xdata, ydata, ndata,
        AtPlotLINES, -1, ConvertColor(fdo->graph,
        FDObj_LineColors[l%FDObj_numLineColors]), FDObj_Legend(fdo,l));

  ti = "density";
  XtVaSetValues(fdo->graph, XtNtitle, FDObj_Title(fdo, ti, l),
        XtNshowTitle, True, NULL);
  xlabel = "time (ms)";
  ylabel = "?nspikes/sec?";
  XtVaSetValues(fdo->xaxis, XtNlabel, xlabel, NULL);
  XtVaSetValues(fdo->yaxis, XtNlabel, ylabel, NULL);
  SA_FreeSpikeDensityList(sdlp);
  SA_FreeSpikeRasterList(srl);
}

/* ------------------------------------------------------------------------
   FD1_perhist1FN
   ------------------------------------------------------------------------ */
void FD1_perhist1FN(FDO *fdo, FILEDATA *fd, int l, FILE *fptr,
	float period, int nbins, int type)
{
	int i;
	float *depvals, *means;
	float mean_phase, vector_strength, rayleigh;
	int n;
	char *ti;
	char *xlabel, *ylabel;
	float *px, *py;
	int pn;

	assert((type>=0) && (type<=2));

	FD_perhist1FN(fdo, fd, FD_GI(fd,"Spont_Stims"),
		FD1_RAWDATA(fd)->nrasters, FD1_RAWDATA(fd)->pres_order,
		FD1_RAWDATA(fd)->rastlist, FD1_RAWDATA(fd)->depints,
		&depvals, &means, &n, &mean_phase, &vector_strength, &rayleigh,
		period, nbins, type);

	switch(type) {
		case(PERHIST_MP):
			ylabel = "mean phase (cycles)";
			ti = "mean_phase";
			break;
		case(PERHIST_VS):
			ylabel = "vector strength";
			ti = "vector_strength";
			break;
		case(PERHIST_RT):
			ylabel = "rayleigh test (p-val)";
			ti = "rayleigh_test";
			break; 
	}
	xlabel = FD_GV(fd, "depvar");

	FD_plotter_hack(depvals, means, NULL, n, (period==0.0) ? 0 : 1,
		&px, &py, &pn);

	if(fptr!=NULL) {
		fprintf(fptr, "COMMENTS\n");
		fprintf(fptr,"%%%s, %s view\n",
			strrchr(fd->filename,'/')==NULL ? fd->filename :
			1+(char*)strrchr(fd->filename,'/'), ti);
		fprintf(fptr,"%%col 1: %s\n", xlabel);
		fprintf(fptr,"%%col 2: %s\n", ylabel);
		fprintf(fptr, "END_COMMENTS\n");
		fprintf(fptr, "PARAMS\n");
		fprintf(fptr, "END_PARAMS\n");
		fprintf(fptr, "RASTERDATA\n");
		fprintf(fptr, "nlines=%d\n",n);
		for(i=0; i<n; i++)
			fprintf(fptr,"%e\t%e\n",depvals[i],means[i]);
		fprintf(fptr, "END_RASTERDATA\n"); }

	FDObj_Add_Data_All(fdo,px,py,pn);
	FDObj_AddLine(fdo, l, px, py, pn,
        AtPlotSEGMENTPOINTS, AtMarkRECTANGLE, ConvertColor(fdo->graph,
			FDObj_LineColors[l%FDObj_numLineColors]), FDObj_Legend(fdo,l));

	XtVaSetValues(fdo->graph, XtNtitle, FDObj_Title(fdo, ti, l),
        XtNshowTitle, True, NULL);
	XtVaSetValues(fdo->xaxis, XtNlabel, xlabel, NULL);
	XtVaSetValues(fdo->yaxis, XtNlabel, ylabel, NULL);

	free(means); free(depvals);
}

/* ------------------------------------------------------------------------
   FD1_perhist2FN
   ------------------------------------------------------------------------ */
void FD1_perhist2FN(FDO *fdo, FILEDATA *fd, int l, FILE *fptr,
	float period, int nbins)
{
/* fdo->no_errs=1 here means it's a true histogram, ie spikes are simply summed
   and no division by # trials or # cycles occurrs.  fdo->no_errs=0 here means 
   that a division by # cycles occurrs.  note that this is different from
   the old perhist, which divides by # trials */

	int i;
	char ti[128], *xlabel, *ylabel;
	float *depvals, *means, *stderrs, *stddevs;
	int *n;
	float mean_phase, vector_strength, rayleigh;
	float *px, *py;
	int pn;

	FD_perhist2FN(fdo, fd,
		FD1_RAWDATA(fd)->nrasters, FD1_RAWDATA(fd)->pres_order,
		FD1_RAWDATA(fd)->rastlist, FD1_RAWDATA(fd)->depints,
		&depvals, &means, &stderrs, &stddevs, &n,
		&mean_phase, &vector_strength, &rayleigh,
		period, nbins);

	mps[l]=mean_phase;
	vss[l]=vector_strength;
	rays[l]=rayleigh;

	if(fdo->vsm_data==PERHIST_ALL_CODE)
		sprintf(dis[l],"all");
	else if(fdo->vsm_data==SPONT)
		sprintf(dis[l],"SPONT");
	else
		sprintf(dis[l],"%d",fdo->vsm_data);

	xlabel = "phase (cycles)";
	ylabel = "nspikes/cycle";
	sprintf(ti,"period_histogram(%s)",dis[l]);

	if(fptr!=NULL) {
		fprintf(fptr, "COMMENTS\n");
		fprintf(fptr,"%%%s, %s view\n",
			strrchr(fd->filename,'/')==NULL ? fd->filename :
			1+(char*)strrchr(fd->filename,'/'), ti);
		fprintf(fptr,"%%col 1: %s\n", xlabel);
		fprintf(fptr,"%%col 2: %s\n", ylabel);
		if(!fdo->no_errs)
			fprintf(fptr,"%%col 3: std error\n");
		fprintf(fptr, "END_COMMENTS\n");
		fprintf(fptr, "PARAMS\n");
		fprintf(fptr, "END_PARAMS\n");
		fprintf(fptr, "RASTERDATA\n");
		fprintf(fptr, "nlines=%d\n",nbins);
		if(!fdo->no_errs) {
			for(i=0; i<nbins; i++)
				fprintf(fptr,"%e\t%e\t%e\n",depvals[i],means[i],stderrs[i]); }
		else {
			for(i=0; i<nbins; i++)
				fprintf(fptr,"%e\t%e\n",depvals[i],means[i]); }
		fprintf(fptr, "END_RASTERDATA\n"); }

	FD_plotter_hack(depvals, means, (!fdo->no_errs) ? stderrs : NULL,
        nbins, 0, &px, &py, &pn);

	FDObj_Add_Data_All(fdo,px,py,pn);
	FDObj_AddLine(fdo, l, px, py, pn,
        AtPlotSEGMENTPOINTS, AtMarkRECTANGLE, ConvertColor(fdo->graph,
			FDObj_LineColors[l%FDObj_numLineColors]), FDObj_Legend(fdo,l));

	XtVaSetValues(fdo->graph, XtNtitle, FDObj_Title(fdo, ti, l),
        XtNshowTitle, True, NULL);
	XtVaSetValues(fdo->xaxis, XtNlabel, xlabel, NULL);
	XtVaSetValues(fdo->yaxis, XtNlabel, ylabel, NULL);

	free(means); free(depvals); free(stderrs); free(stddevs);
}


/* ------------------------------------------------------------------------
   FD1_similarity_stats
   ------------------------------------------------------------------------ */
static int common_basis(
        float *x1, int nx1, int **y1, int *ny1,
        float *x2, int nx2, int **y2, int *ny2,
        float ****data, int *R, int *C, int ***n)
{
  int i,j,tmpi;
  float *tmpf;
 
  /* check that #reps are same for all depvals */
  for(tmpi=ny1[0], i=1; i<nx1; i++)
    if(tmpi!=ny1[i]) return(0);
  for(tmpi=ny2[0], i=1; i<nx2; i++)
    if(tmpi!=ny2[i]) return(0);

  (*R)=2;
  assert(((*data)=(float***)malloc(2*sizeof(float**)))!=NULL);
  assert(((*data)[0]=(float**)malloc(5*sizeof(float*)))!=NULL);
  assert(((*data)[1]=(float**)malloc(5*sizeof(float*)))!=NULL);

  /* copy into data the y[1,2] vals for which a specific x val exists
     in both x1 and x2 */
  (*C)=0;
  for(i=0; i<nx1; i++) {
    if((tmpf=(float*)bsearch(x1+i,x2,nx2,sizeof(float),floatcompare))==NULL)
      continue;

    if((((*C)%5)==0) && ((*C)!=0)) {
      assert(((*data)[0]=(float**)realloc((*data)[0],
            ((*C)+5)*sizeof(float*)))!=NULL);
      assert(((*data)[1]=(float**)realloc((*data)[1],
            ((*C)+5)*sizeof(float*)))!=NULL); }
    assert(((*data)[0][(*C)]=(float*)malloc(ny1[i]*sizeof(float)))!=NULL);
    assert(((*data)[1][(*C)]=(float*)malloc(ny2[tmpf-x2]*
          sizeof(float)))!=NULL);

    for(j=0; j<ny1[i]; j++)
      (*data)[0][(*C)][j] = sqrt(0.375+y1[i][j]); /* poisson --> normal */
    for(j=0; j<ny2[tmpf-x2]; j++)
      (*data)[1][(*C)][j] = sqrt(0.375+y2[tmpf-x2][j]);
    (*C)++; }

  assert(((*n)=(int**)malloc(2*sizeof(int*)))!=NULL);
  assert(((*n)[0]=(int*)malloc((*C)*sizeof(int)))!=NULL);
  assert(((*n)[1]=(int*)malloc((*C)*sizeof(int)))!=NULL);
  for(i=0; i<(*C); i++)
    (*n)[0][i]=ny1[0];
  for(i=0; i<(*C); i++)
    (*n)[1][i]=ny2[0];

  return(1);
}

#if(0)
/* this takes the geometric mean of non-parametric pvals for each point */
static float my_metric(float ***data, int R, int C, int **n)
{
  int i,j;
  int one[50],two[50];
  double ret_val,pval,mu,ro,tmp;

  for(ret_val=1.0,ro=0.0, i=0; i<C; i++) {
    assert(n[0][i]<50);
    for(tmp=0.0, j=0; j<n[0][i]; j++) {
      one[j] = (int)(0.125+data[0][i][j]*data[0][i][j]);
      tmp += one[j]; }
    mu = tmp/n[0][i];

    assert(n[1][i]<50);
    for(tmp=0.0, j=0; j<n[1][i]; j++) {
      two[j] = (int)(0.125+data[1][i][j]*data[1][i][j]);
      tmp += two[j]; }
    mu = (mu + tmp/n[1][i])/2.0;

    pval = permutation_test_dp(one,n[0][i],two,n[1][i],0);
    ret_val *= pow(pval,mu);
    if((pval!=0.0) && (ret_val==0.0)) return(-1.0);
    ro += mu; }

  return((float)pow(ret_val,1.0/ro));
}
#endif

static void my_chisquare(
      float *x1, float *y1, float *e1, int n1,
      float *x2, float *y2, float *e2, int n2,
      float *chisq, float *pval)
{
  int i;
  float tmp, chisq1, chisq2, pval1, pval2, min;

  if(e1!=NULL) {
    min = 10000.0;
    for(i=0; i<n1; i++)
      if((e1[i]<min) && (e1[i]!=0.0)) min=e1[i];
    chisq1 = 0.0;
    for(i=0; i<n1; i++) {
      tmp = lin_interpolate(x1[i],n2,x2,y2);
      tmp -= y1[i];
      tmp /= (e1[i]==0.0) ? min : e1[i];
      tmp *= tmp;
      chisq1 += tmp; }
    pval1 = gammq(0.5*n1,0.5*chisq1); }

  if(e2!=NULL) {
    min = 10000.0;
    for(i=1; i<n2; i++)
      if((e2[i]<min) && (e2[i]!=0.0)) min=e2[i];
    chisq2 = 0.0;
    for(i=0; i<n2; i++) {
      tmp = lin_interpolate(x2[i],n1,x1,y1);
      tmp -= y2[i];
      tmp /= (e2[i]==0.0) ? min : e2[i];
      tmp *= tmp;
      chisq2 += tmp; }
    pval2 = gammq(0.5*n2,0.5*chisq2); }

  if(e1==NULL) {
    (*chisq) = chisq2;
    (*pval) = pval2; }
  else if(e2==NULL) {
    (*chisq) = chisq1;
    (*pval) = pval1; }
  else {
    (*chisq) = (chisq1+chisq2)/2.0;
    (*pval) = (pval1+pval2)/2.0; }
}

extern int txt_get_n(FILEDATA*);
extern float *txt_get_x(FILEDATA*);
extern float *txt_get_y(FILEDATA*);
extern float *txt_get_e(FILEDATA*);

void FD1_similarity_stats(Widget w, XtPointer fdoptr, XtPointer call_data)
{
  FDO *fdo = (FDO *) fdoptr;
  int i,j,count,size;
  char *buf,buf2[128],*tmp;
  float ***data,Fr,Fc,Fi,pr,pc,pi;
  int R,C,**n;
  char *name,foo[64];
  int ii,jj,txt;
  float chisq, pval;

  assert((buf=(char*)malloc(1024*sizeof(char)))!=NULL);
  size=1024;  count=0;

  for(txt=i=0; i<fdo->nfds; i++)
    if(!strcmp(fdo->fds[i]->modname,"txt"))
      txt=1;
  
  if(txt==0) {
    name = "anova";
    tmp="files                         anova, fac*int\n"; }
  else {
    name = "chi square";
    tmp="files                         pval, chisq\n"; }
  strcpy(buf,tmp); count+=strlen(tmp);
  tmp="-----                         --------------\n";
  strcat(buf,tmp); count+=strlen(tmp);

  for(i=0; i<(fdo->nfds-1); i++) {
    for(j=i+1; j<fdo->nfds; j++) {

      if(txt==0) {
        if(!common_basis(
              1+SUMDATA(fdo->fds[i])->depvals, -1+SUMDATA(fdo->fds[i])->ndata,
              1+SUMDATA(fdo->fds[i])->nspikes,  1+SUMDATA(fdo->fds[i])->n,
              1+SUMDATA(fdo->fds[j])->depvals, -1+SUMDATA(fdo->fds[j])->ndata,
              1+SUMDATA(fdo->fds[j])->nspikes,  1+SUMDATA(fdo->fds[j])->n,
              &data, &R, &C, &n))
          continue;
        anova2(data,R,C,n,&Fr,&Fc,&Fi,&pr,&pc,&pi);
        pval = pr*pi;
        chisq = -1.0;

        for(ii=0; ii<R; ii++) {
          for(jj=0; jj<C; jj++)
            free(data[ii][jj]);
          free(data[ii]); }
        free(data);
        free(n[0]); free(n[1]); free(n); }
      else {
        if((!strcmp(fdo->fds[i]->modname,"txt")) &&
              (!strcmp(fdo->fds[j]->modname,"txt")))
          continue;
        if(!strcmp(fdo->fds[i]->modname,"txt"))
          my_chisquare(txt_get_x(fdo->fds[i]),
                txt_get_y(fdo->fds[i]), NULL, txt_get_n(fdo->fds[i]),
                1+SUMDATA(fdo->fds[j])->depvals, 1+SUMDATA(fdo->fds[j])->means,
                1+SUMDATA(fdo->fds[j])->stddevs, -1+SUMDATA(fdo->fds[j])->ndata,
                &chisq, &pval);
        else if(!strcmp(fdo->fds[j]->modname,"txt"))
          my_chisquare(txt_get_x(fdo->fds[j]),
                txt_get_y(fdo->fds[j]), NULL, txt_get_n(fdo->fds[j]),
                1+SUMDATA(fdo->fds[i])->depvals, 1+SUMDATA(fdo->fds[i])->means,
                1+SUMDATA(fdo->fds[i])->stddevs, -1+SUMDATA(fdo->fds[i])->ndata,
                &chisq, &pval);
        else
          my_chisquare(
                1+SUMDATA(fdo->fds[i])->depvals, 1+SUMDATA(fdo->fds[i])->means,
                1+SUMDATA(fdo->fds[i])->stddevs, -1+SUMDATA(fdo->fds[i])->ndata,
                1+SUMDATA(fdo->fds[j])->depvals, 1+SUMDATA(fdo->fds[j])->means,
                1+SUMDATA(fdo->fds[j])->stddevs, -1+SUMDATA(fdo->fds[j])->ndata,
                &chisq, &pval); }

#ifdef __linux__
      count += sprintf(buf2,"%-13s, %-13s  %e  %e\n",
            strrchr(fdo->fds[i]->filename,'/')==NULL ? fdo->fds[i]->filename :
                  1+(char*)strrchr(fdo->fds[i]->filename,'/'),
            strrchr(fdo->fds[j]->filename,'/')==NULL ? fdo->fds[j]->filename :
                  1+(char*)strrchr(fdo->fds[j]->filename,'/'),
            pval, chisq);
#else
      count += strlen(sprintf(buf2,"%-13s, %-13s  %e  %e\n",
            strrchr(fdo->fds[i]->filename,'/')==NULL ? fdo->fds[i]->filename :
                  1+(char*)strrchr(fdo->fds[i]->filename,'/'),
            strrchr(fdo->fds[j]->filename,'/')==NULL ? fdo->fds[j]->filename :
                  1+(char*)strrchr(fdo->fds[j]->filename,'/'),
            pval, chisq));
#endif
      if(fdo->to_tty)
        printf("%s",buf2);
      strcat(buf,buf2);
      if((size-count)<512) {
          assert((buf=(char*)realloc(buf,(size+1024)*sizeof(char)))!=NULL);
          size += 1024; } } }

  if(!fdo->to_tty) {
    sprintf(foo,"%s stats",name);
    pop_text(foo, buf, strlen(buf)+1, False); }
}


#define PROB_NBINS 50
/* ------------------------------------------------------------------------
   FD1_prob_distribution
   ------------------------------------------------------------------------ */
void FD1_prob_distribution(Widget w, XtPointer fdoptr, XtPointer call_data)
{
	FDO *fdo = (FDO *) fdoptr;
	int **prob_bins = NULL;
	int i,j;
	int nreps;
	FILE *fp=NULL;

	nreps = FD_GI(fdo->fds[0], "reps");

	assert((prob_bins=(int **)calloc(SUMDATA(fdo->fds[0])->ndata,
		sizeof(int *)))!=NULL);
	for (i=0;i<SUMDATA(fdo->fds[0])->ndata;i++) 
		assert((prob_bins[i]=(int *)calloc(PROB_NBINS,sizeof(int)))!=NULL);

	for (i=0;i<SUMDATA(fdo->fds[0])->ndata;i++) 
		for (j=0;j<nreps;j++) 
			if (SUMDATA(fdo->fds[0])->nspikes[i][j]<PROB_NBINS)
				prob_bins[i][SUMDATA(fdo->fds[0])->nspikes[i][j]]++;

	/* write the MATLAB file */

	if (w==NULL)
		fp = stdout;
	else
		fp = fopen2("load_probs.m", "w");
	for (i=0;i<SUMDATA(fdo->fds[0])->ndata;i++) 
		fprintf(fp, "%%%d: %f\n",i+1,SUMDATA(fdo->fds[0])->depvals[i]);
	fprintf(fp, "global depvars\n");
	fprintf(fp, "global prob\n");
	fprintf(fp, "depvars = [ ");
	for (i=0;i<SUMDATA(fdo->fds[0])->ndata;i++) 
		fprintf(fp, "%f ",SUMDATA(fdo->fds[0])->depvals[i]);
	fprintf(fp, "];\n");
	fprintf(fp, "prob = [ ");
	for (i=0;i<SUMDATA(fdo->fds[0])->ndata;i++) {
		fprintf(fp, "[ ");
		for (j=0;j<PROB_NBINS;j++) 
			fprintf(fp, "%d ", prob_bins[i][j]);
		fprintf(fp, "]\n");
	}
	fprintf(fp,"];\n");
	if (w!=NULL)
		fclose(fp);
	for (i=0;i<SUMDATA(fdo->fds[0])->ndata;i++) 
		free(prob_bins[i]);
	free(prob_bins);
}

/* ------------------------------------------------------------------------
   FD1_linearity_stats
   ------------------------------------------------------------------------ */
void FD1_linearity_stats(FDO *fdo, char *p)
{
  int i,j,k,off;
  float min,max,min2,max2,tmpf;
  float **g_y,*g_x;
  int g_n;
	char *q;
	float qf[100];

	if(fdo->nfds < 2) return;

  for(i=1; i<fdo->nfds; i++) {
		if(strcmp(p,"")) {
			q = strtok(((i==1) ? p : NULL),",");
			sscanf(q,"%f",qf+i-1); }
		else
			qf[i-1]=0.0; }

  min = SUMDATA(fdo->fds[0])->depvals[1];
  max = SUMDATA(fdo->fds[0])->depvals[SUMDATA(fdo->fds[0])->ndata-1];
  for(i=1; i<fdo->nfds; i++) {
    min2 = -qf[i-1]+SUMDATA(fdo->fds[i])->depvals[1];
    max2 = -qf[i-1]+SUMDATA(fdo->fds[i])->depvals[
          SUMDATA(fdo->fds[i])->ndata-1];
    min = MY_MAX(min,min2);
    max = MY_MIN(max,max2); }

  assert((g_y=(float**)malloc((fdo->nfds-1)*sizeof(float*)))!=NULL);
  for(i=1; i<fdo->nfds; i++) {
    assert((g_y[i-1]=(float*)malloc(SUMDATA(fdo->fds[0])->ndata*
          sizeof(float)))!=NULL);
    for(k=0, j=1; j<SUMDATA(fdo->fds[0])->ndata; j++) {
      tmpf=SUMDATA(fdo->fds[0])->depvals[j];
      if((tmpf<min) || (tmpf>max)) continue;
      g_y[i-1][k++] = lin_interpolate(tmpf+qf[i-1], SUMDATA(fdo->fds[i])->ndata,
            SUMDATA(fdo->fds[i])->depvals, SUMDATA(fdo->fds[i])->means); } }

  assert((g_x=(float*)malloc(SUMDATA(fdo->fds[0])->ndata*sizeof(float)))!=NULL);
  for(g_n=0,off=-1,k=0, i=1; i<SUMDATA(fdo->fds[0])->ndata; i++) {
    tmpf=SUMDATA(fdo->fds[0])->depvals[i];
    if((tmpf<min) || (tmpf>max)) continue;
      if(off==-1) off=i;
      g_x[k++] = SUMDATA(fdo->fds[0])->depvals[i];
      g_n++; }

  FD_linearity_stats(
        SUMDATA(fdo->fds[0])->depvals+off, SUMDATA(fdo->fds[0])->means+off,
        SUMDATA(fdo->fds[0])->stddevs+off, g_x, g_y, g_n, fdo->nfds-1, fdo);

  for(i=0; i<fdo->nfds-1; i++)
    free(g_y[i]);
  free(g_y); free(g_x);
}


/* ------------------------------------------------------------------------
   FD1_isih_stats
   ------------------------------------------------------------------------ */
void FD1_isih_stats_menu(FDO *fdobj)
{
  menubutton_clear(fdobj->fit,&(fdobj->fitpsh));
  XtSetSensitive(fdobj->fit, False);

  menubutton_add(fdobj->fitpsh,"stats",
        FD1_isih_stats,fdobj);
  XtSetSensitive(fdobj->fit, True);
}

static void compute_isih_stats(int nrasters, int **rastlist,
        int code, float crit, int *total, int *bad)
{
  int i, j;
  float last;

  (*total) = (*bad) = 0;

  for (i=0; i<nrasters; i++) {
    if(rastlist[i]==NULL) continue;
    for (j=0; j<N_SPIKES(rastlist[i]); j++) {
      if ((code == MU_CODE) || (code == SPIKE_CHAN(rastlist[i], j))) {
        if(j==0) {
          last = SPIKE_MSEC(rastlist[i], j);
          continue; }
        (*total) += 1;
        if((-last + SPIKE_MSEC(rastlist[i], j)) < crit)
          (*bad) += 1;
        last = SPIKE_MSEC(rastlist[i], j); } } }
}

void FD1_isih_stats(Widget w, XtPointer fdoptr, XtPointer call_data)
{
  FDO *fdo = (FDO *) fdoptr;
  int i,count,size;
  char *buf,buf2[128],*tmp;
  int total, bad, sum_total, sum_bad;
  float crit;

  crit = lookupParms_float("parms.isih_percent");
  sum_total=sum_bad=0;

  assert((buf=(char*)malloc(1024*sizeof(char)))!=NULL);
  size=1024;  count=0;

  tmp="file            total    bad     %\n";
  strcpy(buf,tmp); count+=strlen(tmp);
  tmp="----            ------   ---     -\n";
  strcat(buf,tmp); count+=strlen(tmp);

  for(i=0; i<fdo->nfds; i++) {
    compute_isih_stats(FD1_RAWDATA(fdo->fds[i])->nrasters,
          FD1_RAWDATA(fdo->fds[i])->rastlist,
          fdo->sum_code[i], crit, &total, &bad);
    sum_total += total;
    sum_bad += bad;

#ifdef __linux__
    count += sprintf(buf2,"%-13s   %6d   %4d    %6.3f\n",
          strrchr(fdo->fds[i]->filename,'/')==NULL ? fdo->fds[i]->filename :
                1+(char*)strrchr(fdo->fds[i]->filename,'/'),
          total,bad,((float)bad/(float)total)*100.0);
#else
    count += strlen(sprintf(buf2,"%-13s   %6d   %4d    %6.3f\n",
          strrchr(fdo->fds[i]->filename,'/')==NULL ? fdo->fds[i]->filename :
                1+(char*)strrchr(fdo->fds[i]->filename,'/'),
          total,bad,((float)bad/(float)total)*100.0));
#endif
    if(fdo->to_tty)
      printf("%s",buf2);
    strcat(buf,buf2);
    if((size-count)<512) {
        assert((buf=(char*)realloc(buf,(size+1024)*sizeof(char)))!=NULL);
        size += 1024; } }

#ifdef __linux__
  count += sprintf(buf2,"TOTAL            %6d   %4d    %6.3f\n",
        sum_total,sum_bad,((float)sum_bad/(float)sum_total)*100.0);
#else
  count += strlen(sprintf(buf2,"TOTAL            %6d   %4d    %6.3f\n",
        sum_total,sum_bad,((float)sum_bad/(float)sum_total)*100.0));
#endif
  if(fdo->to_tty)
    printf("%s",buf2);
  strcat(buf,buf2);
  if((size-count)<512) {
      assert((buf=(char*)realloc(buf,(size+1024)*sizeof(char)))!=NULL);
      size += 1024; }

  if(!fdo->to_tty)
    pop_text("stats", buf, strlen(buf)+1, False);
}


/* ------------------------------------------------------------------------
   FD1_perhist_stats
   ------------------------------------------------------------------------ */
void FD1_perhist_stats_menu(FDO *fdobj)
{
  menubutton_clear(fdobj->fit,&(fdobj->fitpsh));
  XtSetSensitive(fdobj->fit, False);

  menubutton_add(fdobj->fitpsh,"stats",
        FD1_perhist_stats,fdobj);
  XtSetSensitive(fdobj->fit, True);
}

void FD1_perhist_stats(Widget w, XtPointer fdoptr, XtPointer call_data)
{
  FDO *fdo = (FDO *) fdoptr;
  int i,count,size;
  char *buf,buf2[128],*tmp;

  assert((buf=(char*)malloc(1024*sizeof(char)))!=NULL);
  size=1024;

  tmp="file            depvar    MP     VS     Rayleigh\n";
  strcpy(buf,tmp); count+=strlen(tmp);
  tmp="----            ------    --     --     --------\n";
  strcat(buf,tmp); count+=strlen(tmp);

  for(i=0; i<fdo->nfds; i++) {
#ifdef __linux__
    count += sprintf(buf2,"%-13s   %6s  %5.3f   %4.2f    %5.3f\n",
          strrchr(fdo->fds[i]->filename,'/')==NULL ? fdo->fds[i]->filename :
                1+(char*)strrchr(fdo->fds[i]->filename,'/'),
          dis[i],mps[i],vss[i],rays[i]);
#else
    count += strlen(sprintf(buf2,"%-13s   %6s  %5.3f   %4.2f    %5.3f\n",
          strrchr(fdo->fds[i]->filename,'/')==NULL ? fdo->fds[i]->filename :
                1+(char*)strrchr(fdo->fds[i]->filename,'/'),
          dis[i],mps[i],vss[i],rays[i]));
#endif
    strcat(buf,buf2);
    if((size-count)<512) {
        assert((buf=(char*)realloc(buf,(size+1024)*sizeof(char)))!=NULL);
        size += 1024; } }

  pop_text("stats", buf, strlen(buf)+1, False);
}

/* ------------------------------------------------------------------------
   FD1_contra_ipsi
   ------------------------------------------------------------------------ */
void FD1_contra_ipsi(FILEDATA *fd)
{
  int i;
  char *p, *q;

  if (as_bool(getpref("contraipsi", "no"))) {
    if (FD_GF(fd, "ML") < 0.0) {
      for (i=0; i<FD1_RAWDATA(fd)->nrasters; i++) {
        if (FD1_RAWDATA(fd)->depints[i] != SPONT)
          FD1_RAWDATA(fd)->depints[i] *= -1; 
	  }
      for (i=0; i<SUMDATA(fd)->ndata; i++) {
        if (SUMDATA(fd)->depvals[i] != SPONT)
          SUMDATA(fd)->depvals[i] *= -1.0; 
	  } 
	}
    p = FD_GV(fd, "depvar");
    q = (char*)malloc(strlen(p) + 200);

    sprintf(q, "C leads I        %s        I leads C", p);
    setvar("depvar", q, fd->params);

    free(q);
    fd->trashed = 1; 
  }
}

/* ------------------------------------------------------------------------
   FD1_sort

	   sorts trials by depints 
   ------------------------------------------------------------------------ */
static void FD1_sort(FILEDATA *data)
{
  int i, j, min, idx;

  for (i=0; i<(FD1_RAWDATA(data)->nrasters-1); i++) {
    idx=i;   min=FD1_RAWDATA(data)->depints[i];
    for (j=i+1; j<FD1_RAWDATA(data)->nrasters; j++) {
      if (FD1_RAWDATA(data)->depints[j] < min) {
        idx=j;   min=FD1_RAWDATA(data)->depints[j]; } }
    if(idx!=i) {
      SWAP(FD1_RAWDATA(data)->pres_order[i],
            FD1_RAWDATA(data)->pres_order[idx],int);
      SWAP(FD1_RAWDATA(data)->rastlist[i],
            FD1_RAWDATA(data)->rastlist[idx],spike_t*);
      SWAP(FD1_RAWDATA(data)->deplines[i],
            FD1_RAWDATA(data)->deplines[idx],char*);
      SWAP(FD1_RAWDATA(data)->depints[i],
            FD1_RAWDATA(data)->depints[idx],int); } }
}

/* ------------------------------------------------------------------------
   FD1_summarize

	   summarize() is a little tricky because there are two options for
	   spontaneous trials.  if Spont_Stims=1, then one spont trial is
	   present for every repetition and the spont rate is calculated
	   from those trials.  if Spont_Stims=0, then
	   independent spont trials are not present, and the spont rate is
	   calculated from the Delay portion of all the trials *
   ------------------------------------------------------------------------ */
static int FD1_summarize(FILEDATA *fd, int unitid)
{
  int nreps;					/* number of reps (ideally) */
  int t_begin;				/* driven begins at in ms (t_ in ticks) */
  int t_end;					/* driven ends at in ms (t_ in ticks) */
  int use_delay;			/* use the pre-stim delay as spontaneous */
  int t_spont_begin;	/* spont begins at in ms (t_ in ticks) */
  int t_spont_end;		/* spont ends at in ms (t_ in ticks) */
  int driven;					/* response during driven period */
  float *stattmp;			/* temp buf for means,stderrs,stddevs */
  int *stattmpi;
  int last;
  char *last_line;
  int i, j, k;
  int spont_stims;
  float *spont_resp;
  int *spont_respi;
  int tmp;
  int period;

	FD1_freeSumData(fd);

  tmp=(1+FD1_RAWDATA(fd)->nrasters);

  assert((SUMDATA(fd)          =(SumData*)malloc(sizeof(SumData)))!=NULL);
  assert((SUMDATA(fd)->n       =(int*)malloc(tmp*sizeof(float)))!=NULL);
  assert((SUMDATA(fd)->means   =(float*)malloc(tmp*sizeof(float)))!=NULL);
  assert((SUMDATA(fd)->stderrs =(float*)malloc(tmp*sizeof(float)))!=NULL);
  assert((SUMDATA(fd)->stddevs =(float*)malloc(tmp*sizeof(float)))!=NULL);
  assert((SUMDATA(fd)->depvals =(float*)malloc(tmp*sizeof(float)))!=NULL);
  assert((SUMDATA(fd)->depstrs =(char**)malloc(tmp*sizeof(char*)))!=NULL);
  assert((SUMDATA(fd)->nspikes =(int**)malloc(tmp*sizeof(int*)))!=NULL);

  /* pre-calculate all analysis parameters */

  FD_calc_params(fd, &spont_stims, &use_delay,
      &t_begin, &t_end, &t_spont_begin, &t_spont_end, &period);
  nreps = FD_GI(fd, "reps");

  /* now summarize the data */
  k = 0;
  if((!spont_stims) || use_delay) {
	  k = 1;

	  assert((spont_resp = (float*)malloc(
          (FD1_RAWDATA(fd)->nrasters)*sizeof(float)))!=NULL);
	  assert((spont_respi = (int*)malloc(
          (FD1_RAWDATA(fd)->nrasters)*sizeof(int)))!=NULL);

	  for(i=0;i<FD1_RAWDATA(fd)->nrasters;i++) {
		  spont_resp[i] = countRaster(t_spont_begin, t_spont_end,
			  FD1_RAWDATA(fd)->rastlist[i], unitid);
		  spont_resp[i] *= ((float)(t_end-t_begin)/
			  (float)(t_spont_end-t_spont_begin));
		  spont_respi[i] = (int)(spont_resp[i]+0.5); 
	  }

	  SUMDATA(fd)->n[0]       = i;
	  SUMDATA(fd)->means[0]   = mean(spont_resp, i);
	  SUMDATA(fd)->stderrs[0] = stderror(spont_resp, i);
	  SUMDATA(fd)->stddevs[0] = stddev(spont_resp, i);
	  SUMDATA(fd)->depvals[0] = (float)SPONT;
	  SUMDATA(fd)->depstrs[0] = "depvar=-6666 <SPONT>";
	  SUMDATA(fd)->nspikes[0] = spont_respi;

	  free(spont_resp); 
  }

  for (i=0; i<FD1_RAWDATA(fd)->nrasters;) {

	  assert((stattmp=(float*)malloc(nreps*sizeof(float)))!=NULL);
	  assert((stattmpi=(int*)malloc(nreps*sizeof(int)))!=NULL);

	  last = FD1_RAWDATA(fd)->depints[i];
	  last_line = FD1_RAWDATA(fd)->deplines[i];

	  if(use_delay && (last==SPONT)) {
		  i++; 
		  continue; 
	  }
    
	  for (j=0; i<FD1_RAWDATA(fd)->nrasters && 
			   FD1_RAWDATA(fd)->depints[i]==last; i++,j++) {
		  driven = countRaster(t_begin,t_end,FD1_RAWDATA(fd)->rastlist[i],
			  unitid);
		  stattmp[j] = (float)driven;
		  stattmpi[j] = driven; 
	  }

	  SUMDATA(fd)->n[k]       = j;
	  SUMDATA(fd)->means[k]   = mean(stattmp, j);
	  SUMDATA(fd)->stderrs[k] = stderror(stattmp, j);
	  SUMDATA(fd)->stddevs[k] = stddev(stattmp, j);
	  SUMDATA(fd)->depvals[k] = (float) last;
	  SUMDATA(fd)->depstrs[k] = last_line;
	  SUMDATA(fd)->nspikes[k] = stattmpi;

	  free(stattmp);
	  k++;
    
	  if (j != nreps)
		  fprintf(stderr, "; note trial %d has %d of %d samples\n",last,j,
			  nreps); 
  }

  assert((SUMDATA(fd)->n=(int*)realloc(SUMDATA(fd)->n,
        k*sizeof(float)))!=NULL);
  assert((SUMDATA(fd)->means=(float*)realloc(SUMDATA(fd)->means,
        k*sizeof(float)))!=NULL);
  assert((SUMDATA(fd)->stderrs=(float*)realloc(SUMDATA(fd)->stderrs,
        k*sizeof(float)))!=NULL);
  assert((SUMDATA(fd)->stddevs=(float*)realloc(SUMDATA(fd)->stddevs,
        k*sizeof(float)))!=NULL);
  assert((SUMDATA(fd)->depvals=(float*)realloc(SUMDATA(fd)->depvals,
        k*sizeof(float)))!=NULL);
  assert((SUMDATA(fd)->depstrs=(char**)realloc(SUMDATA(fd)->depstrs,
        k*sizeof(char*)))!=NULL);
  assert((SUMDATA(fd)->nspikes=(int**)realloc(SUMDATA(fd)->nspikes,
        k*sizeof(int*)))!=NULL);

  SUMDATA(fd)->ndata = k;
  SUMDATA(fd)->unit = unitid;
  return(period);
}

/* ------------------------------------------------------------------------
   FD1_reader
   ------------------------------------------------------------------------ */
int FD1_reader(FILE *fp, FILEDATA *data)
{
  int 		nr, act_nr, i;
  char 		*p, *q;
  spike_t	**rastlist;
  char		**strings;
  int		*ints, *pres_order;
  float         **avepmk;

  assert((data->rawdata=calloc(1,sizeof(FD1_RawData)))!=NULL);
	data->sumdata=NULL;

  if ((p = skipTo(fp, "nrasters=")) == NULL || 
      sscanf(p, "nrasters=%d", &nr) != 1) {
    fclose(fp);
    FD_free(data);
    return(0); }

  assert((rastlist=(spike_t **)calloc(nr,sizeof(spike_t *)))!=NULL);
  assert((strings=(char**)calloc(nr,sizeof(char*)))!=NULL);
  assert((ints=(int*)calloc(nr,sizeof(int)))!=NULL);
  assert((pres_order=(int*)calloc(nr,sizeof(int)))!=NULL);
  assert((avepmk=(float**)calloc(nr,sizeof(float*)))!=NULL);

  for (i=act_nr=0; i<nr; i++) {
    if ((p = skipTo(fp, "depvar=")) == NULL)
      break;
    rastlist[act_nr] = readRaster(fp, data, avepmk+act_nr);
    if ((q = rindex(p, '\n')) != NULL)
      *q = 0;
    if(rastlist[act_nr]!=NULL) {
      pres_order[act_nr] = act_nr;
      strings[act_nr] = strsave(p);
      sscanf(p, "depvar=%d", &ints[act_nr++]); } }

  if (act_nr < nr) {
    fprintf(stderr, "Warning:  %s is data light\n", data->filename);
    assert((rastlist=(spike_t**)realloc(rastlist,
          act_nr*sizeof(spike_t*)))!=NULL);
    assert((strings=(char**)realloc(strings,act_nr*sizeof(char*)))!=NULL);
    assert((ints=(int*)realloc(ints,act_nr*sizeof(int)))!=NULL);
    assert((pres_order=(int*)realloc(pres_order,act_nr*sizeof(int)))!=NULL);
    assert((avepmk=(float**)realloc(avepmk,act_nr*sizeof(float*)))!=NULL); }

  FD1_RAWDATA(data)->rastlist = rastlist;
  FD1_RAWDATA(data)->deplines = strings;
  FD1_RAWDATA(data)->depints = ints;
  FD1_RAWDATA(data)->pres_order = pres_order;
  FD1_RAWDATA(data)->avepmk = avepmk;
  FD1_RAWDATA(data)->nrasters = act_nr;

  FD1_sort(data);

  data->channels = NULL;
  FD_howmany_units(&(data->channels),FD1_RAWDATA(data)->nrasters,
        FD1_RAWDATA(data)->rastlist);

  return(1);
}

/* ------------------------------------------------------------------------
   FD1_free
   ------------------------------------------------------------------------ */

void FD1_freeRawData(FILEDATA *data)
{
  int i;

  for (i=0; i<FD1_RAWDATA(data)->nrasters; i++) {
    if (FD1_RAWDATA(data)->rastlist != NULL)
      FREE(FD1_RAWDATA(data)->rastlist[i]);
    if (FD1_RAWDATA(data)->deplines != NULL)
      FREE(FD1_RAWDATA(data)->deplines[i]);
    if (FD1_RAWDATA(data)->avepmk != NULL)
      FREE(FD1_RAWDATA(data)->avepmk[i]); }

  FREE(FD1_RAWDATA(data)->pres_order);
  FREE(FD1_RAWDATA(data)->rastlist);
  FREE(FD1_RAWDATA(data)->deplines);
  FREE(FD1_RAWDATA(data)->depints);
  FREE(FD1_RAWDATA(data)->avepmk);

  FREE(FD1_RAWDATA(data));
}

static void FD1_freeSumData(FILEDATA *data)
{
  int i;

	if(SUMDATA(data)!=NULL) {
    FREE(SUMDATA(data)->depstrs);	/*vals were freed by free'ing deplines array*/
    FREE(SUMDATA(data)->depvals);
    FREE(SUMDATA(data)->means);
    FREE(SUMDATA(data)->stderrs);
    FREE(SUMDATA(data)->stddevs);
    FREE(SUMDATA(data)->n);
    for(i=0; i<SUMDATA(data)->ndata; i++)
      FREE(SUMDATA(data)->nspikes[i]);
    FREE(SUMDATA(data)->nspikes);
  	FREE(SUMDATA(data)); }
}

int FD1_free(FILEDATA *data)
{
	FD1_freeRawData(data);
	FD1_freeSumData(data);

  return(1);
}

/* ------------------------------------------------------------------------
   FD1_get_*
   ------------------------------------------------------------------------ */
float *FD1_get_means(FILEDATA *fd)
{
  return(SUMDATA(fd)->means);
}

float *FD1_get_depvals(FILEDATA *fd)
{
  return(SUMDATA(fd)->depvals);
}

float *FD1_get_stderrs(FILEDATA *fd)
{
  return(SUMDATA(fd)->stderrs);
}

int FD1_get_ndata(FILEDATA *fd)
{
  return(SUMDATA(fd)->ndata);
}

int *FD1_get_n(FILEDATA *fd)
{
  return(SUMDATA(fd)->n);
}
int **FD1_get_nspikes(FILEDATA *fd)
{
  return(SUMDATA(fd)->nspikes);
}

int *FD1_get_depints(FILEDATA *fd)
{
  return(FD1_RAWDATA(fd)->depints);
}

int FD1_get_nrasters(FILEDATA *fd)
{
  return(FD1_RAWDATA(fd)->nrasters);
}

int *FD1_get_pres_order(FILEDATA *fd)
{
  return(FD1_RAWDATA(fd)->pres_order);
}

spike_t **FD1_get_rastlist(FILEDATA *fd)
{
  return(FD1_RAWDATA(fd)->rastlist);
}

/* ------------------------------------------------------------------------
   FD1_peak_stats
   ------------------------------------------------------------------------ */
void FD1_peak_stats(Widget w, XtPointer fdobjptr, XtPointer call_data)
{
  FDO *fdobj = (FDO *) fdobjptr;
	int i, *ndata;
  float **x, **y, *spont;
	char **file;

	assert((ndata=(int*)malloc(fdobj->nfds*sizeof(int)))!=NULL);
	assert((x=(float**)malloc(fdobj->nfds*sizeof(float*)))!=NULL);
	assert((y=(float**)malloc(fdobj->nfds*sizeof(float*)))!=NULL);
	assert((spont=(float*)malloc(fdobj->nfds*sizeof(float)))!=NULL);
	assert((file=(char**)malloc(fdobj->nfds*sizeof(char*)))!=NULL);

	for(i=0; i<fdobj->nfds; i++) {
		ndata[i] = SUMDATA(fdobj->fds[i])->ndata-1;
		x[i] = SUMDATA(fdobj->fds[i])->depvals+1;
		y[i] = SUMDATA(fdobj->fds[i])->means+1;
    spont[i] = SUMDATA(fdobj->fds[i])->means[0];
		file[i] = fdobj->fds[i]->filename; }

	FD_peak_stats(fdobj->nfds,ndata,x,y,spont,file,fdobj->to_tty,
			!strcmp(fdobj->fds[0]->modname,"itd") ? 1 : 0);

	free(ndata);
	free(x);
	free(y);
	free(spont);
	free(file);
}

static float *copyVec(float *in, int n, float shift)
{
	float *out;
	int i;

	assert((out=(float*)malloc(n*sizeof(float)))!=NULL);
	for(i=0; i<n; i++)
		out[i] = in[i]-shift;

	return(out);
}

/* ------------------------------------------------------------------------
   FD1_combine_data
   ------------------------------------------------------------------------ */
void FD1_combine_data(FDO *fdobj, char *p, FILE *fptr)
{
	int i, n, tn, t2n;
	float *x, *y, *tx, *ty, *t2x, *t2y;
	float spont;
	char *q;
	float qf;

	if(fdobj->nfds < 2) return;

  fprintf(fptr, "COMMENTS\n");
	fprintf(fptr,"%%a combine w/ %s of ",p);

  for(i=n=0, spont=0.0; i<fdobj->nfds; i++) {
		if(strcmp(p,"")) {
			q = strtok(((i==0) ? p : NULL),",");
			sscanf(q,"%f",&qf); }
		else
			qf=0.0;

		t2n = FD1_get_ndata(fdobj->fds[i])-1;
		t2y = FD1_get_means(fdobj->fds[i])+1;
		t2x = copyVec(FD1_get_depvals(fdobj->fds[i])+1,t2n,qf);

		FD_combine_data(x,y,n,t2x,t2y,t2n,&tx,&ty,&tn);
		spont += FD1_get_means(fdobj->fds[i])[0];

		free(t2x);
		if(i>0) {
			free(x); free(y); }

		x = tx; y = ty; n = tn; }

	spont /= fdobj->nfds;

  for(i=0; i<fdobj->nfds; i++)
		fprintf(fptr,"%s, ",
          strrchr(fdobj->fds[i]->filename,'/')==NULL ? fdobj->fds[i]->filename :
          1+(char*)strrchr(fdobj->fds[i]->filename,'/'));
	fprintf(fptr,"\n");
	fprintf(fptr,"%%col 1: x[i]\n");
	fprintf(fptr,"%%col 2: y[i]\n");
  fprintf(fptr, "END_COMMENTS\n");
  fprintf(fptr, "PARAMS\n");
	fprintf(fptr,"spont=%e\t%e\n",spont,0.0);
  fprintf(fptr, "END_PARAMS\n");
  fprintf(fptr, "RASTERDATA\n");
	fprintf(fptr, "nlines=%d\n",n);
	for(i=0; i<n; i++)
		fprintf(fptr,"%e\t%e\n",x[i],y[i]);
  fprintf(fptr, "END_RASTERDATA\n");
  fclose(fptr);
}
